/*
 * zcom_wlan_wds.c
 */
#ifdef _ZCOM_FOR_WDS_
#include <linux/config.h>
#include <linux/version.h>
#include <linux/module.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/init.h>

#include "net80211/if_media.h"
#include "net80211/ieee80211_var.h"
#include "net80211/if_ethersubr.h"
#include "net80211/igwsha1.h"
#include "net80211/hmacsha1.h"
#include "net80211/if_zcom_wds.h"
#include "net80211/if_zcom_wds_private.h"

////////////////////////////////////////////////////////////////////
// WDS module static defines
////////////////////////////////////////////////////////////////////
static WDS_HANDLE WdsHandle;
static ieee80211_hardstart_send_cb_t ieee80211_hard_xmit = NULL;

static __inline unsigned int HashIndexFromMacAddr(const WDS_MAC_ADDR MacAddr)
{
	return (MacAddr[3]^MacAddr[4]^MacAddr[5])&(WDS_NODE_MAX_HASH_INDEX - 1);
}

static __inline bool IsMulticastFrame(const WDS_MAC_ADDR MacAddr)
{
	return (MacAddr[0] & 0x01) == 0x01;
}

static PWDS_HASH_NODE
WDS_FindHashNode(
	IN	WDS_HANDLE WdsHandle,
	IN	const WDS_MAC_ADDR MacAddr
)
{
	bool	bFound = false;
	PWDS_HASH_NODE	pHashNode = NULL;

	WDS_ASSERT(WdsHandle);
	WDS_ASSERT(MacAddr);
	do
	{
		pHashNode = WdsHandle->pWdsHashList[HashIndexFromMacAddr(MacAddr)];

		while (NULL != pHashNode) 
		{
			if (0 == WDS_MEM_CMP(pHashNode->WdsNode.MacAddr, MacAddr, WDS_MAC_ADDR_SIZE))
			{
				bFound = true;
				break;
			}
			pHashNode = pHashNode->pNext;
		}
		
	} while(0);

	return (bFound) ? pHashNode : NULL;
}

static void
WDS_RecycleHashNode(
	IN	WDS_HANDLE WdsHandle,
	IN	PWDS_HASH_NODE pDelHashNode
)
{
	unsigned int	uiHashIndex = 0;
	PWDS_HASH_NODE	pPreviousHashNode = NULL;

	WDS_ASSERT(WdsHandle);
	WDS_ASSERT(pDelHashNode);

	uiHashIndex = HashIndexFromMacAddr(pDelHashNode->WdsNode.MacAddr);

	//Delete it from hash list
	if (pDelHashNode == WdsHandle->pWdsHashList[uiHashIndex])
	{
		//Deleted node is hash head node
		WdsHandle->pWdsHashList[uiHashIndex] = pDelHashNode->pNext;
	}
	else
	{
		//Find it's previous node
		pPreviousHashNode = WdsHandle->pWdsHashList[uiHashIndex];
		while (pPreviousHashNode != NULL)
		{
			if (pPreviousHashNode->pNext == pDelHashNode)
			{
				break;
			}
			pPreviousHashNode = pPreviousHashNode->pNext;
		}

		//Previous'next node point deleted's next node
		if (pPreviousHashNode != NULL)
			pPreviousHashNode->pNext = pDelHashNode->pNext;
	}

	//Put it back to free pool
	pDelHashNode->pNext = WdsHandle->pWdsHashFreeList;
	WdsHandle->pWdsHashFreeList = pDelHashNode;
}

static bool
WDS_DelHashNode(
	IN	WDS_HANDLE WdsHandle,
	IN	PWDS_HASH_NODE pDelHashNode
)
{
	unsigned int		i = 0;
	PDESTINATION_NODE	pDelDestNode = NULL;

	WDS_ASSERT(WdsHandle);
	WDS_ASSERT(pDelHashNode);

	if (WDS_NODE_ASSOCIATION == pDelHashNode->WdsNode.NodeType)
	{
		//Delete all destination nodes in the destination node list of this assocation node
		while (NULL != pDelHashNode->pDestNodeList) 
		{
			pDelDestNode = pDelHashNode->pDestNodeList;
			if (NULL != pDelDestNode->pDestHashNode && WDS_NODE_DESTINATION == pDelDestNode->pDestHashNode->WdsNode.NodeType)
			{
				WDS_RecycleHashNode(WdsHandle, pDelDestNode->pDestHashNode);
			}
			pDelHashNode->pDestNodeList = pDelDestNode->pNext;
			WDS_MEM_FREE(pDelDestNode);
			pDelDestNode = NULL;
		}
		
		// Refresh association node list
		for (i = 0;i < WdsHandle->AssocNodeList.ulCount;i++)
		{
			if (0 == WDS_MEM_CMP(
				WdsHandle->AssocNodeList.assoc_node_ni[i]->ni_macaddr,
				pDelHashNode->WdsNode.MacAddr,
				WDS_MAC_ADDR_SIZE
				))
			{
				WdsHandle->AssocNodeList.ulCount--;
				for (;i < (WdsHandle->AssocNodeList.ulCount);i++)
				{
					WdsHandle->AssocNodeList.assoc_node_ni[i] = WdsHandle->AssocNodeList.assoc_node_ni[i+1];
				}
				//WDS_MEM_SET(WdsHandle->AssocNodeList.AssocNodeMac[i], 0, WDS_MAC_ADDR_SIZE);
				break;
			}
		}
	}
	WDS_RecycleHashNode(WdsHandle, pDelHashNode);

	return true;
}

static PWDS_HASH_NODE
WDS_AddHashNode(
	IN	WDS_HANDLE WdsHandle,
	IN	const WDS_MAC_ADDR MacAddr
)
{
	bool			bSuccess = false;
	unsigned int	uiHashIndex = 0;
	PWDS_HASH_NODE	pHashNode = NULL;

	WDS_ASSERT(WdsHandle);
	WDS_ASSERT(MacAddr);

	do
	{
		if (NULL == WdsHandle->pWdsHashFreeList &&
			(WDS_GET_CURRENT_MS() - WdsHandle->ulLastTimeNodeKeeping > WDS_NODE_KEEPING_INTERVAL))
		{
			// No free node, do node keeping
			for (uiHashIndex = 0;uiHashIndex < WDS_NODE_MAX_HASH_INDEX;uiHashIndex++)
			{
				pHashNode = WdsHandle->pWdsHashList[uiHashIndex];

				while (NULL != pHashNode)
				{
					if (WDS_NODE_DESTINATION == pHashNode->WdsNode.NodeType &&
						(WDS_GET_CURRENT_MS() - pHashNode->ulTimeToLife > WDS_NODE_TIME_TO_LIFE))
					{
						// Now only clean timeout destination node
						WDS_DelHashNode(WdsHandle, pHashNode);
					}
					else
					{
						pHashNode = pHashNode->pNext;
					}
				}
			}
			WdsHandle->ulLastTimeNodeKeeping = WDS_GET_CURRENT_MS();
		}

		if (NULL == WdsHandle->pWdsHashFreeList)
		{
			// No free node, add fail
			break;
		}

		pHashNode = WdsHandle->pWdsHashFreeList;
		WdsHandle->pWdsHashFreeList = pHashNode->pNext;
		WDS_MEM_SET(pHashNode, 0, sizeof(WDS_HASH_NODE));	// Initialize this node
		uiHashIndex = HashIndexFromMacAddr(MacAddr);
		pHashNode->pNext = WdsHandle->pWdsHashList[uiHashIndex];
		WdsHandle->pWdsHashList[uiHashIndex] = pHashNode;
		WDS_MEM_CPY(pHashNode->WdsNode.MacAddr, MacAddr, WDS_MAC_ADDR_SIZE);

		bSuccess = true;
	} while(0);

	return bSuccess ? pHashNode : NULL;
}

static struct ieee80211_node*
WDS_DelNode(
	IN	WDS_HANDLE WdsHandle,
	IN	const WDS_MAC_ADDR MacAddr
)
{
	struct ieee80211_node *ni = NULL;
	PWDS_HASH_NODE	pHashNode = NULL;
	unsigned int	uiLockFlags = 0;

	WDS_ASSERT(WdsHandle);
	if (MacAddr == NULL)
		return ni;
	//WDS_ASSERT(MacAddr);

	WDS_SPIN_LOCK_SAVE(WdsHandle->WdsSpinLock, uiLockFlags);

	do
	{
		//Try to find node by MAC address
		pHashNode = WDS_FindHashNode(WdsHandle, MacAddr);
		if (NULL == pHashNode)
		{
			//Can not find, delete fail
			break;
		}
		else
		{
			ni = pHashNode->WdsNode.ni_wds_node;
			if (!WDS_DelHashNode(WdsHandle, pHashNode)) ni = NULL;
		}
		
	} while(0);

	WDS_SPIN_LOCK_RESTORE(WdsHandle->WdsSpinLock, uiLockFlags);
	ZComLogAppendWlan(ZCOM_LOG_WLAN_REMOTE_AP_DELETED, (unsigned char*)MacAddr);
	return ni;
}

////////////////////////////////////////////////////////////////////
// WDS module interface defines
////////////////////////////////////////////////////////////////////
bool
WDS_Reset(void)
{
	int	i = 0;
	struct ieee80211_node *ni = NULL;
	WDS_ASSERT(WdsHandle);

	while (WdsHandle->AssocNodeList.ulCount > 0)
	{
		ni = WDS_DelNode(WdsHandle,
			WdsHandle->AssocNodeList.assoc_node_ni[WdsHandle->AssocNodeList.ulCount-1]->ni_macaddr);
		if (ni != NULL) {
			//ieee80211_node_refsub(ni, 0xa);
			ni->ni_zcomwds_mode = ZCOMWDS_MODE_NONE;//free static node
			ieee80211_crypto_delkey(ni->ni_vap, &ni->ni_ucastkey, ni);
			ieee80211_node_leave(ni);
			ieee80211_free_node(ni);
		}
	}

	WDS_MEM_SET(WdsHandle, 0, sizeof(*WdsHandle));

	WDS_SPIN_LOCK_INIT(WdsHandle->WdsSpinLock);
	
	for (i = 0; i < (WDS_NODE_MAX_NUM-1); i++)
	{
		WdsHandle->WdsHashNodePool[i].pNext = &WdsHandle->WdsHashNodePool[i+1];
	}
	
	WdsHandle->pWdsHashFreeList = &WdsHandle->WdsHashNodePool[0];

	return true;
}
EXPORT_SYMBOL(WDS_Reset);

/*
 *	Basic Operations
 */
WDS_HANDLE
WDS_Initialize(void)
{
	bool bSuccess = false;

	do
	{
		//Allocate WDS module memory
		WdsHandle = (WDS_HANDLE)WDS_MEM_MALLOC(sizeof(*WdsHandle));
		if (NULL == WdsHandle)
		{
			break;
		}
		WDS_MEM_SET(WdsHandle, 0, sizeof(*WdsHandle));

		//Initialize WDS resource
		if (false == WDS_Reset())
		{
			break;
		}
		bSuccess = true;
		
	} while(0);

	if (false == bSuccess)
	{
		//Release resource when initialize fail
		WDS_Release();
	}

	return bSuccess ? WdsHandle : NULL;
}

/*
 *	Node operations
 */
PWDS_NODE
WDS_AddAssociationNode(
	IN	const WDS_MAC_ADDR MacAddr,
	IN	struct ieee80211_node* ni
)
{
	bool			bSuccess = false;
	PWDS_HASH_NODE	pHashNode = NULL;
	unsigned int	uiLockFlags = 0;

	WDS_ASSERT(WdsHandle);
	WDS_ASSERT(MacAddr);

	WDS_SPIN_LOCK_SAVE(WdsHandle->WdsSpinLock, uiLockFlags);
	do 
	{
		//Try to find node by MAC address
		pHashNode = WDS_FindHashNode(WdsHandle, MacAddr);
		if (NULL == pHashNode)
		{
			//Can not find, add it
			pHashNode = WDS_AddHashNode(WdsHandle, MacAddr);

			if (NULL == pHashNode)
			{
				break;
			}
		}

		pHashNode->ulTimeToLife = WDS_GET_CURRENT_MS();
		pHashNode->WdsNode.NodeType = WDS_NODE_ASSOCIATION;
		pHashNode->pAssocNode = NULL;
		pHashNode->WdsNode.ni_wds_node = ni;

		//Refresh association node list
		//WDS_MEM_CPY(
		//			WdsHandle->AssocNodeList.AssocNodeMac[WdsHandle->AssocNodeList.ulCount++],
		//			MacAddr,
		//			WDS_MAC_ADDR_SIZE
		//			);
		WdsHandle->AssocNodeList.assoc_node_ni[WdsHandle->AssocNodeList.ulCount++] = ni;
		zdebug(ZCOM_DEBUG_WDS, "%s, %d: %s, assocWDSnum=%d\n",
			__func__, __LINE__, ether_sprintf(MacAddr), WdsHandle->AssocNodeList.ulCount);
		ZComLogAppendWlan(ZCOM_LOG_WLAN_REMOTE_AP_JOINED, (unsigned char*)MacAddr);
		bSuccess = true;
	} while(0);

	WDS_SPIN_LOCK_RESTORE(WdsHandle->WdsSpinLock, uiLockFlags);

	return bSuccess ? &pHashNode->WdsNode : NULL;
}
EXPORT_SYMBOL(WDS_AddAssociationNode);

PWDS_NODE
WDS_AddDestinationNode(
	IN	 WDS_HANDLE WdsHandle,
	IN	const WDS_MAC_ADDR MacAddr,
	IN	const WDS_MAC_ADDR AssocMacAddr
)
{
	bool			bSuccess = false;
	bool			bFoundHashNode = false;
	bool			bItsAssocNodeIsOurs = false;
	bool			bFoundInDestNodeList = false;
	PWDS_HASH_NODE	pHashNode = NULL;
	PWDS_HASH_NODE	pAssocHashNode = NULL;
	PDESTINATION_NODE	pDestNode = NULL;
	unsigned int	uiLockFlags = 0;

	WDS_ASSERT(WdsHandle);
	WDS_ASSERT(MacAddr);
	WDS_SPIN_LOCK_SAVE(WdsHandle->WdsSpinLock, uiLockFlags);

	do
	{
		//If its association node is not one of ours, do not add it
		if (NULL != AssocMacAddr)
		{
			pAssocHashNode = WDS_FindHashNode(WdsHandle, AssocMacAddr);
			if (NULL != pAssocHashNode && WDS_NODE_ASSOCIATION == pAssocHashNode->WdsNode.NodeType)
			{
				bItsAssocNodeIsOurs = true;
			}
			else
			{
				break;
			}
		}

		//Try to find node by MAC address
		pHashNode = WDS_FindHashNode(WdsHandle, MacAddr);
		if (NULL == pHashNode)
		{
			//Can not find, try to add it
			pHashNode = WDS_AddHashNode(WdsHandle, MacAddr);
			if (NULL == pHashNode)
			{
				break;
			}
		}
		else
		{
			bFoundHashNode = true;
		}

		if (bFoundHashNode && WDS_NODE_ASSOCIATION == pHashNode->WdsNode.NodeType)
		{
			//Found node is association node, do not update
			bSuccess = true;
			break;
		}

		pHashNode->ulTimeToLife = WDS_GET_CURRENT_MS();
		pHashNode->WdsNode.NodeType = WDS_NODE_DESTINATION;
		pHashNode->WdsNode.ni_wds_node = NULL;
		
		//Specific its association node
		if (true == bItsAssocNodeIsOurs)
		{
			//TODO: try to update it according to the better WDS route path
			pHashNode->pAssocNode = &pAssocHashNode->WdsNode;

			// Store this destination node in the destination node list of its association node
			pDestNode = pAssocHashNode->pDestNodeList;
			if (NULL == pDestNode)
			{
				pDestNode = WDS_MEM_MALLOC(sizeof(DESTINATION_NODE));
				if (NULL != pDestNode)
				{
					pDestNode->pNext = NULL;
					pDestNode->pDestHashNode = pHashNode;
					pAssocHashNode->pDestNodeList = pDestNode;
				}
			}
			else
			{
				if (pHashNode == pDestNode->pDestHashNode)
				{
					bFoundInDestNodeList = true;
				}
				else
				{
					while (NULL != pDestNode->pNext) 
					{
						if (pHashNode == pDestNode->pNext->pDestHashNode)
						{
							bFoundInDestNodeList = true;
							break;
						}
						pDestNode = pDestNode->pNext;
					}
				}

				if (false == bFoundInDestNodeList)
				{
					pDestNode->pNext = WDS_MEM_MALLOC(sizeof(DESTINATION_NODE));
					if (NULL != pDestNode)
					{
						pDestNode = pDestNode->pNext;
						pDestNode->pNext = NULL;
						pDestNode->pDestHashNode = pHashNode;
					}
				}
			}
		}
		else
		{
			pHashNode->pAssocNode = NULL;
		}
		bSuccess = true;
	} while(0);

	WDS_SPIN_LOCK_RESTORE(WdsHandle->WdsSpinLock, uiLockFlags);
	return bSuccess ? &pHashNode->WdsNode : NULL;
}

static struct ieee80211_node* WDS_DelAssocNode(IN const WDS_MAC_ADDR MacAddr)
{
	zdebug(ZCOM_DEBUG_WDS, "%s, %d: %s, assocWDSnum=%d\n",
		__func__, __LINE__, ether_sprintf(MacAddr), WdsHandle->AssocNodeList.ulCount);
	return WDS_DelNode(WdsHandle, MacAddr);
}

PWDS_NODE
WDS_FindNode(
	IN	WDS_HANDLE WdsHandle,
	IN	const WDS_MAC_ADDR MacAddr
)
{
	PWDS_HASH_NODE	pHashNode = NULL;

	WDS_ASSERT(WdsHandle);
	WDS_ASSERT(MacAddr);

	pHashNode = WDS_FindHashNode(WdsHandle, MacAddr);
	if (NULL == pHashNode || WDS_NODE_UNKNOWN == pHashNode->WdsNode.NodeType)
	{
		return NULL;
	}
	else
	{
		return &pHashNode->WdsNode;
	}
}


void WDS_Release(void)
{
	int	i = 0;
	struct ieee80211_node *ni = NULL;
	
	if(WdsHandle)
	{
		for (i = 0;i < WdsHandle->AssocNodeList.ulCount;i++)
		{
			ni = WDS_DelNode(WdsHandle, 
				WdsHandle->AssocNodeList.assoc_node_ni[i]->ni_macaddr);
			if (ni) {
				//ieee80211_node_refsub(ni, 0xa);
				ni->ni_zcomwds_mode = ZCOMWDS_MODE_NONE;//free static node
				ieee80211_crypto_delkey(ni->ni_vap, &ni->ni_ucastkey, ni);
				ieee80211_node_leave(ni);
				
				ieee80211_free_node(ni);
			}
			//if (WdsHandle->AssocNodeList.assoc_node_ni[i])
			//	ieee80211_free_node(WdsHandle->AssocNodeList.assoc_node_ni[i]);
		}

		WDS_SPIN_LOCK_RELEASE(WdsHandle->WdsSpinLock);
		WDS_MEM_FREE(WdsHandle);
		WdsHandle = NULL;
	}
}

/*
 *	wds Mesh function.
 */
void
WDS_LocalPcMacEntryAge(IN WDS_HANDLE pWdsHandle)
{
	int iFirstLocalPcMacEntry = 1;
	LocalPCEntry *pNextLocalPCEntry, *pFrontLocalPCEntry = NULL, *pLocalPCEntry;
	
	WDS_ASSERT(pWdsHandle);
	
	if(pWdsHandle->wdsLocalPcEntryFlag == 1)
	{
		return;
	}
	pWdsHandle->wdsLocalPcEntryFlag = 1;
	pLocalPCEntry = pWdsHandle->localPCEntry;
	while(pLocalPCEntry)
	{
		pNextLocalPCEntry = pLocalPCEntry->pNextEntry;
		if(WDS_GET_CURRENT_MS() - pLocalPCEntry->ulLastActiveTicks > MAX_LOCAL_PC_LIFE_TICKS)
		{
			if(iFirstLocalPcMacEntry == 1)
			{
				pWdsHandle->localPCEntry = pNextLocalPCEntry;
			}
			else
			{
				pFrontLocalPCEntry->pNextEntry = pNextLocalPCEntry;
			}
			pLocalPCEntry->pNextEntry = NULL;
			pWdsHandle->iLocalPCEntryCount--;
			WDS_MEM_SET(pLocalPCEntry, 0, sizeof(pLocalPCEntry));
			WDS_MEM_FREE(pLocalPCEntry);	
		}
		else
		{
			pFrontLocalPCEntry = pLocalPCEntry;
			iFirstLocalPcMacEntry = 0;
		}
		pLocalPCEntry = pNextLocalPCEntry;
	}
	pWdsHandle->wdsLocalPcEntryFlag = 0;
}

void
WDS_LocalPcMacAddEntry(IN const WDS_MAC_ADDR LocalApPcAddr)
{
	LocalPCEntry *pLocalPCEntry, *pTmpLocalPCEntry;

	WDS_ASSERT(WdsHandle);
	
	if(NULL == LocalApPcAddr
	  || WdsHandle->wdsLocalPcEntryFlag ==1
	  || ETHER_IS_BROADCAST(LocalApPcAddr))
	{
		return;
	}
	WDS_LocalPcMacEntryAge(WdsHandle);
	WdsHandle->wdsLocalPcEntryFlag = 1;
	pTmpLocalPCEntry = WdsHandle->localPCEntry;
	while(pTmpLocalPCEntry)
	{
		if(memcmp(pTmpLocalPCEntry->localApPC, LocalApPcAddr, ETH_ALEN) == 0)
		{
			pTmpLocalPCEntry->ulLastActiveTicks = WDS_GET_CURRENT_MS();
			zdebug(ZCOM_DEBUG_RX, "%s, %d:[%s] update ticks\n", __func__, __LINE__, ether_sprintf(LocalApPcAddr));
			WdsHandle->wdsLocalPcEntryFlag = 0;
			return;
		}
		pTmpLocalPCEntry = pTmpLocalPCEntry->pNextEntry;
	}

	if(WdsHandle->iLocalPCEntryCount > MAX_LOCAL_PC_COUNT)
	{
		WdsHandle->wdsLocalPcEntryFlag = 0;
		return;
	}

	pLocalPCEntry = (LocalPCEntry*)WDS_MEM_MALLOC(sizeof(LocalPCEntry));
	if(NULL == pLocalPCEntry)
	{
		WdsHandle->wdsLocalPcEntryFlag = 0;
		return;
	}
	WDS_MEM_SET(pLocalPCEntry, 0, sizeof(pLocalPCEntry));
	pLocalPCEntry->pNextEntry = NULL;
	pLocalPCEntry->ulLastActiveTicks = WDS_GET_CURRENT_MS();
	zdebug(ZCOM_DEBUG_RX, "%s, %d:[%s] update ticks\n", __func__, __LINE__, ether_sprintf(LocalApPcAddr));
	memcpy(pLocalPCEntry->localApPC, LocalApPcAddr, ETH_ALEN);
		
	pTmpLocalPCEntry = WdsHandle->localPCEntry;
	if(NULL == pTmpLocalPCEntry)
	{
		WdsHandle->localPCEntry = pLocalPCEntry;
		WdsHandle->iLocalPCEntryCount = 1;
		WdsHandle->wdsLocalPcEntryFlag = 0;
		return;
	}
	
	while(pTmpLocalPCEntry->pNextEntry)
	{
		pTmpLocalPCEntry = pTmpLocalPCEntry->pNextEntry;
	}
	WdsHandle->iLocalPCEntryCount++;
	pTmpLocalPCEntry->pNextEntry = pLocalPCEntry;
	WdsHandle->wdsLocalPcEntryFlag = 0;
}

bool
WDS_LocalPcEntryCheck(
	IN const WDS_MAC_ADDR SrcMacAddr)
{
	LocalPCEntry *pLocalPCEntry;
	
	WDS_ASSERT(WdsHandle);
	
	if(NULL == SrcMacAddr
		|| WdsHandle->wdsLocalPcEntryFlag == 1)
	{
		return false;
	}
	
	WdsHandle->wdsLocalPcEntryFlag = 1;
	pLocalPCEntry = WdsHandle->localPCEntry;
	while(pLocalPCEntry)
	{
		if(0 == memcmp(pLocalPCEntry->localApPC, SrcMacAddr, ETH_ALEN)
			&& WDS_GET_CURRENT_MS() - pLocalPCEntry->ulLastActiveTicks < MAX_LOCAL_PC_LIFE_TICKS)
		{
			WdsHandle->wdsLocalPcEntryFlag = 0;
			return false;
		}
		pLocalPCEntry = pLocalPCEntry->pNextEntry;
	}	
	WdsHandle->wdsLocalPcEntryFlag = 0;
	return true;
}

// check whether the mac was remote pc (or assoced wds)
struct ieee80211_node* WDS_RemotePCCheck(IN const WDS_MAC_ADDR MacAddr)
{
	unsigned int	uiLockFlags = 0;
	PWDS_HASH_NODE	pHashNode = NULL;
	struct ieee80211_node *ni = NULL;

	WDS_ASSERT(WdsHandle);
	WDS_ASSERT(MacAddr);

	WDS_SPIN_LOCK_SAVE(WdsHandle->WdsSpinLock, uiLockFlags);
	pHashNode = WDS_FindHashNode(WdsHandle, MacAddr);
	if (NULL != pHashNode && WDS_GET_CURRENT_MS() - pHashNode->ulTimeToLife <= WDS_NODE_TIME_TO_LIFE)
	{
		if (WDS_NODE_ASSOCIATION == pHashNode->WdsNode.NodeType)
			ni = pHashNode->WdsNode.ni_wds_node;
		if (WDS_NODE_DESTINATION == pHashNode->WdsNode.NodeType && pHashNode->pAssocNode != NULL)
			ni = pHashNode->pAssocNode->ni_wds_node;

		if (NULL != ni)
			ni = ieee80211_ref_node(ni
								#ifdef NODE_FREE_DEBUG
									, __func__
								#endif
			);
	}
	WDS_SPIN_LOCK_RESTORE(WdsHandle->WdsSpinLock, uiLockFlags);
	return ni;
}

void
WDS_RouteMapAge(IN WDS_HANDLE pWdsHandle)
{	
	int i, iFirstRouteMapEntry;
	PWDS_NODE	pwdsHashNode = NULL;
	struct ieee80211_node *ni = NULL;
	RouteMapEntry *pNextRouteMapEntry, *pFrontRouteMapEntry = NULL, *pRouteMapEntry;

	WDS_ASSERT(pWdsHandle);

	if (pWdsHandle->AssocNodeList.ulCount < 1
		|| pWdsHandle->wdsTimerFlag == 1)
	{
		return;
	}

	pWdsHandle->wdsTimerFlag = 1;
	for (i = 0; i < pWdsHandle->AssocNodeList.ulCount; i++)
	{
		ni = pWdsHandle->AssocNodeList.assoc_node_ni[i];
		pwdsHashNode = WDS_FindNode(pWdsHandle, ni->ni_macaddr);
		if(NULL == pwdsHashNode)
		{
			continue;
		}
		pRouteMapEntry = pwdsHashNode->RouteMapNode;
		iFirstRouteMapEntry = 1;
		while(pRouteMapEntry)
		{
			pNextRouteMapEntry = pRouteMapEntry->pNextMap;
			if(WDS_GET_CURRENT_MS() - pRouteMapEntry->ulLastActiveTicks > MAX_WDS_MAP_LIFE_TICKS)
			{		
				if(iFirstRouteMapEntry == 1)
				{
					pwdsHashNode->RouteMapNode = pNextRouteMapEntry;
				}
				else
				{
					pFrontRouteMapEntry->pNextMap = pNextRouteMapEntry;
				}
				pRouteMapEntry->pNextMap = NULL;
				pwdsHashNode->iRouteMapNodeCount--;
				WDS_MEM_SET(pRouteMapEntry, 0, sizeof(pRouteMapEntry));
				WDS_MEM_FREE(pRouteMapEntry);
			}
			else
			{
				pFrontRouteMapEntry = pRouteMapEntry;
				iFirstRouteMapEntry = 0;
			}
			pRouteMapEntry = pNextRouteMapEntry;
		}
	}

	pWdsHandle->wdsTimerFlag = 0;
}

bool
WDS_AddRouteMapEntry(
	IN const WDS_MAC_ADDR RemoteAPAddr,
	IN const WDS_MAC_ADDR RemoteApPcAddr)
{
	int i;
	PWDS_NODE	pwdsHashNode = NULL;
	struct ieee80211_node *ni = NULL;
	RouteMapEntry *pTmpRouteMapEntry, *pRouteMapEntry;
		
	WDS_ASSERT(WdsHandle);

	if (WdsHandle->AssocNodeList.ulCount < 1
		|| WdsHandle->wdsTimerFlag == 1)
	{
		zdebug(ZCOM_DEBUG_RX, "%s, %d\n", __func__, __LINE__);
		return false;
	}

	WDS_RouteMapAge(WdsHandle);
	if(WDS_LocalPcEntryCheck(RemoteApPcAddr) == false)
	{
		zdebug(ZCOM_DEBUG_RX, "%s, %d\n", __func__, __LINE__);
		return false;
	}
	
	WdsHandle->wdsTimerFlag = 1;
	for (i = 0; i < WdsHandle->AssocNodeList.ulCount; i++) {
		ni = WdsHandle->AssocNodeList.assoc_node_ni[i];
		pwdsHashNode = WDS_FindNode(WdsHandle, ni->ni_macaddr);
		if(NULL == pwdsHashNode)
		{
			continue;
		}

		if(memcmp(pwdsHashNode->MacAddr, RemoteApPcAddr, ETH_ALEN) == 0)
		{
			WdsHandle->wdsTimerFlag = 0;
			return true;
		}

		pTmpRouteMapEntry = pwdsHashNode->RouteMapNode;
		while(pTmpRouteMapEntry)
		{
			if(memcmp(pTmpRouteMapEntry->remoteApPC, RemoteApPcAddr, ETH_ALEN) == 0)
			{	
				//The RemoteApPC is existed remoteap table.
				if(memcmp(pwdsHashNode->MacAddr, RemoteAPAddr, ETH_ALEN) == 0)
				{
					pTmpRouteMapEntry->ulLastActiveTicks = WDS_GET_CURRENT_MS();
					zdebug(ZCOM_DEBUG_RX, "%s, %d:[%s] update ticks\n", __func__, __LINE__, ether_sprintf(RemoteApPcAddr));
					WdsHandle->wdsTimerFlag = 0;
					return true;
				}
				else
				{
					//Invalid packet not send.
					WdsHandle->wdsTimerFlag = 0;
					zdebug(ZCOM_DEBUG_RX, "%s, %d\n", __func__, __LINE__);
					return false;
				}
			}
			pTmpRouteMapEntry = pTmpRouteMapEntry->pNextMap;
		}
	}

	pwdsHashNode = WDS_FindNode(WdsHandle, RemoteAPAddr);
	if(NULL == pwdsHashNode)
	{
		WdsHandle->wdsTimerFlag = 0;
		zdebug(ZCOM_DEBUG_RX, "%s, %d\n", __func__, __LINE__);
		return false;
	}

	if(memcmp(RemoteAPAddr, RemoteApPcAddr, ETH_ALEN) == 0)
	{
		WdsHandle->wdsTimerFlag = 0;
		return true;
	}

	if(pwdsHashNode->iRouteMapNodeCount > MAX_REMOTEMAP_COUNT)
	{
		WdsHandle->wdsTimerFlag = 0;
		return true;
	}

	pRouteMapEntry = (RouteMapEntry*)WDS_MEM_MALLOC(sizeof(RouteMapEntry));
	if(NULL == pRouteMapEntry)
	{
		WdsHandle->wdsTimerFlag = 0;
		zdebug(ZCOM_DEBUG_RX, "%s, %d\n", __func__, __LINE__);
		return false;
	}
	WDS_MEM_SET(pRouteMapEntry, 0, sizeof(pRouteMapEntry));
	pRouteMapEntry->pNextMap = NULL;
	pRouteMapEntry->ulLastActiveTicks = WDS_GET_CURRENT_MS();
	zdebug(ZCOM_DEBUG_RX, "%s, %d:[%s] update ticks\n", __func__, __LINE__, ether_sprintf(RemoteApPcAddr));
	memcpy(pRouteMapEntry->remoteApPC, RemoteApPcAddr, ETH_ALEN);

	pTmpRouteMapEntry = pwdsHashNode->RouteMapNode;
	if(NULL == pTmpRouteMapEntry)
	{
		pwdsHashNode->RouteMapNode = pRouteMapEntry;
		pwdsHashNode->iRouteMapNodeCount = 1;
		WdsHandle->wdsTimerFlag = 0;
		return true;
	}

	while(pTmpRouteMapEntry->pNextMap)
	{
		pTmpRouteMapEntry = pTmpRouteMapEntry->pNextMap;
	}
	pwdsHashNode->iRouteMapNodeCount++;
	pTmpRouteMapEntry->pNextMap = pRouteMapEntry;
	WdsHandle->wdsTimerFlag = 0;

	return true;
}

struct ieee80211_node *
WDS_RouteMapFind(IN const WDS_MAC_ADDR RemoteApPC)
{
	int i;
	PWDS_NODE	pwdsHashNode = NULL;
	struct ieee80211_node *ni = NULL;
	RouteMapEntry *pTmpRouteMapEntry;

	WDS_ASSERT(WdsHandle);

	if (WdsHandle->AssocNodeList.ulCount < 1
		|| WdsHandle->wdsTimerFlag == 1
		|| NULL == RemoteApPC)
	{
		return NULL;
	}
	WdsHandle->wdsTimerFlag = 1;

	for (i = 0; i < WdsHandle->AssocNodeList.ulCount; i++) {
#ifdef NODE_FREE_DEBUG
		ni = ieee80211_ref_node(WdsHandle->AssocNodeList.assoc_node_ni[i], __func__);
#else
		ni = ieee80211_ref_node(WdsHandle->AssocNodeList.assoc_node_ni[i]);
#endif
		pwdsHashNode = WDS_FindNode(WdsHandle, ni->ni_macaddr);
		if(NULL == pwdsHashNode)
		{
			continue;
		}
		
		pTmpRouteMapEntry = pwdsHashNode->RouteMapNode;
		while(pTmpRouteMapEntry)
		{
			if(memcmp(pTmpRouteMapEntry->remoteApPC, RemoteApPC, ETH_ALEN) == 0)
			{	
				WdsHandle->wdsTimerFlag = 0;
				return ni;
			}
			pTmpRouteMapEntry = pTmpRouteMapEntry->pNextMap;
		}
		ieee80211_free_node(ni);
	}
	
	WdsHandle->wdsTimerFlag = 0;
	return NULL;
}

bool
WDS_RouteCheck(
	IN WDS_HANDLE pWdsHandle,
	IN const WDS_MAC_ADDR RemoteAPAddr,
	IN const WDS_MAC_ADDR SrcMacAddr)
{
	PWDS_NODE	pwdsHashNode = NULL;
	RouteMapEntry *pTmpRouteMapEntry;

	if(NULL == RemoteAPAddr
		|| NULL == SrcMacAddr
		|| memcmp(RemoteAPAddr, SrcMacAddr, ETH_ALEN) == 0
		|| pWdsHandle->wdsTimerFlag == 1)
	{
		return false;
	}

	pwdsHashNode = WDS_FindNode(pWdsHandle, RemoteAPAddr);
	if(NULL == pwdsHashNode)
	{
		return false;
	}
	pWdsHandle->wdsTimerFlag = 1;
	pTmpRouteMapEntry = pwdsHashNode->RouteMapNode;
	while(pTmpRouteMapEntry)
	{
		if(0 == memcmp(pTmpRouteMapEntry->remoteApPC, SrcMacAddr, ETH_ALEN)
			&& WDS_GET_CURRENT_MS() - pTmpRouteMapEntry->ulLastActiveTicks < MAX_WDS_MAP_LIFE_TICKS)
		{
			pWdsHandle->wdsTimerFlag = 0;
			return false;
		}
		pTmpRouteMapEntry = pTmpRouteMapEntry->pNextMap;
	}
	pWdsHandle->wdsTimerFlag = 0;
	return true;
}

/*
 *	Packet operations
 */
WDS_PACKET_STATUS
WDS_SendPacket(
	IN	struct ieee80211vap *vap,
	IN	const WDS_MAC_ADDR DestMacAddr,
	IN	const WDS_MAC_ADDR SourceMacAddr,
	OUT	struct ieee80211_node **ni
)
{
	WDS_PACKET_STATUS	WdsPktStatus = WDS_PACKET_DISCARD;
	PWDS_HASH_NODE		pDestHashNode = NULL;
	
	WDS_ASSERT(WdsHandle);
	WDS_ASSERT(DestMacAddr);

	//NULL SourceMacAddr means do not learning destination node
	//WDS_ASSERT(SourceMacAddr);
	WDS_ASSERT(ni);
	do
	{
		//Learning by SA
		if (NULL != SourceMacAddr && NULL == WDS_AddDestinationNode(WdsHandle, SourceMacAddr, NULL))
		{
			break;
		}

		//Multicast frame
		if (true == IsMulticastFrame(DestMacAddr))
		{
			WdsPktStatus = WDS_PACKET_BROADCASTING;
			break;
		}

		//Look up by DA
		pDestHashNode = WDS_FindHashNode(WdsHandle, DestMacAddr);
		if (NULL == pDestHashNode)
		{
			//Can not find, broadcast sending
			*ni = WDS_RouteMapFind(DestMacAddr);
			if(NULL != *ni)
			{
				WdsPktStatus = WDS_PACKET_UNICASTING;
			}
			else
			{
				WdsPktStatus = WDS_PACKET_BROADCASTING;
			}
			break;
		}

		if (WDS_NODE_ASSOCIATION == pDestHashNode->WdsNode.NodeType)
		{
			//DA is our association node, send it to DA directly
			//WDS_MEM_CPY(ReceiverMacAddr, DestMacAddr, WDS_MAC_ADDR_SIZE);
#ifdef NODE_FREE_DEBUG
			*ni = ieee80211_ref_node(pDestHashNode->WdsNode.ni_wds_node, __func__);
#else
			*ni = ieee80211_ref_node(pDestHashNode->WdsNode.ni_wds_node);
#endif

			WdsPktStatus = WDS_PACKET_UNICASTING;
			break;
		}
		else if (WDS_NODE_DESTINATION == pDestHashNode->WdsNode.NodeType)
		{
			if (NULL == pDestHashNode->pAssocNode)
			{
				//This frame may be sent from DS, discard it
				break;
			}
			else
			{
				//Repeat it via its association node
				//WDS_MEM_CPY(ReceiverMacAddr, pDestHashNode->pAssocNode->MacAddr, WDS_MAC_ADDR_SIZE);
#ifdef NODE_FREE_DEBUG
				*ni = ieee80211_ref_node(pDestHashNode->pAssocNode->ni_wds_node, __func__);
#else
				*ni = ieee80211_ref_node(pDestHashNode->pAssocNode->ni_wds_node);
#endif
				WdsPktStatus = WDS_PACKET_UNICASTING;
				break;
			}
		}
	} while(0);
	// for multi-VAPs: if the ni was not assoced with current vap, drop it
	if (*ni != NULL && (*ni)->ni_vap != vap)
	{
		// rx the unicasting pkt to bridge(then to other vap by the routemap); drop multi pkt
		if (!IsMulticastFrame(DestMacAddr))
		{
			WdsPktStatus = WDS_PACKET_UNICASTING;
		}
		else
		{
			ieee80211_free_node(*ni);
			*ni = NULL;
			WdsPktStatus = WDS_PACKET_DISCARD;
		}
	}

	return WdsPktStatus;
}
EXPORT_SYMBOL(WDS_SendPacket);

WDS_PACKET_STATUS
WDS_ReceivePacket(
	IN	struct ieee80211vap *vap,
	IN	const WDS_MAC_ADDR DestMacAddr,
	IN	const WDS_MAC_ADDR SourceMacAddr,
	IN	const WDS_MAC_ADDR TransmitterMacAddr,
	OUT	struct ieee80211_node **ni
)
{
	WDS_PACKET_STATUS	WdsPktStatus = WDS_PACKET_DISCARD;
	PWDS_HASH_NODE		pDestHashNode = NULL;

	WDS_ASSERT(WdsHandle);
	WDS_ASSERT(DestMacAddr);
	WDS_ASSERT(SourceMacAddr);
	WDS_ASSERT(ni);

	do
	{
		//Learning by SA & TA
		if (NULL == WDS_AddDestinationNode(WdsHandle, SourceMacAddr, TransmitterMacAddr))
		{
			break;
		}

		//Multicast frame
		if (true == IsMulticastFrame(DestMacAddr))
		{
			//Multicast frame may be broadcasted to WDS and indicated to upper for DS sending
			WdsPktStatus = WDS_PACKET_BROADCASTING_INDICATE;
			break;
		}

		// Look up by DA
		pDestHashNode = WDS_FindHashNode(WdsHandle, DestMacAddr);
		if (NULL == pDestHashNode)
		{
			//Can not find, broadcast to WDS and indicate to upper
			*ni = WDS_RouteMapFind(DestMacAddr);
			if(NULL != *ni)
			{
				WdsPktStatus = WDS_PACKET_UNICASTING;
			}
			else
			{
				WdsPktStatus = WDS_PACKET_BROADCASTING_INDICATE;
			}
			break;
		}

		if (WDS_NODE_ASSOCIATION == pDestHashNode->WdsNode.NodeType)
		{
			//DA is our association node, send it to DA directly
			//WDS_MEM_CPY(ReceiverMacAddr, DestMacAddr, WDS_MAC_ADDR_SIZE);
#ifdef NODE_FREE_DEBUG
			*ni = ieee80211_ref_node(pDestHashNode->WdsNode.ni_wds_node, __func__);
#else
			*ni = ieee80211_ref_node(pDestHashNode->WdsNode.ni_wds_node);
#endif
			WdsPktStatus = WDS_PACKET_UNICASTING;
			break;
		}
		else if (WDS_NODE_DESTINATION == pDestHashNode->WdsNode.NodeType)
		{
			if (NULL == pDestHashNode->pAssocNode)
			{
				//This frame may be sent from DS, discard it
				WdsPktStatus = WDS_PACKET_INDICATE;
				break;
			}
			else
			{
				if (0 == WDS_MEM_CMP(pDestHashNode->pAssocNode->MacAddr, TransmitterMacAddr, WDS_MAC_ADDR_SIZE))
				{
					WdsPktStatus = WDS_PACKET_BROADCASTING_INDICATE;
				}
				else
				{
					//Repeat it via its association node
					//WDS_MEM_CPY(ReceiverMacAddr, pDestHashNode->pAssocNode->MacAddr, WDS_MAC_ADDR_SIZE);
#ifdef NODE_FREE_DEBUG
					*ni = ieee80211_ref_node(pDestHashNode->pAssocNode->ni_wds_node, __func__);
#else
					*ni = ieee80211_ref_node(pDestHashNode->pAssocNode->ni_wds_node);
#endif

					WdsPktStatus = WDS_PACKET_UNICASTING;
				}
				break;
			}
		}
	} while(0);
	// for multi-VAPs: if the ni was not assoced with current vap, drop it
	if (*ni != NULL && (*ni)->ni_vap != vap)
	{
		// rx the unicasting pkt to bridge(then to other vap by the routemap); drop multi pkt
		if (!IsMulticastFrame(DestMacAddr))
		{
			WdsPktStatus = WDS_PACKET_INDICATE;
		}
		else
		{
			ieee80211_free_node(*ni);
			*ni = NULL;
			WdsPktStatus = WDS_PACKET_DISCARD;
		}
	}

	return WdsPktStatus;
}
EXPORT_SYMBOL(WDS_ReceivePacket);

struct ieee80211_node**
WDS_GetAssocNodeList(
	void
)
{
	WDS_ASSERT(WdsHandle);

	return WdsHandle->AssocNodeList.assoc_node_ni;
}
u_int8_t
WDS_GetAssocNodeNum(
	void
)
{
	WDS_ASSERT(WdsHandle);

	return WdsHandle->AssocNodeList.ulCount;
}


void WDS_ForwardPacket(
		IN	struct sk_buff *skb,
		IN	struct ieee80211vap *vap,
		IN	struct ieee80211_node *ni
		)
{
	struct ether_header *eh = (struct ether_header *)skb->data;

	if(ni != NULL 
	&& (ni->ni_vap != vap || !WDS_RouteCheck(WdsHandle, ni->ni_macaddr, eh->ether_shost)))
	{
		ieee80211_free_node(ni);
		dev_kfree_skb(skb);
	}
	else
	{
		(*ieee80211_hard_xmit)(vap, ni, skb);
	}
}

static int WDS_BroadcastToWDS(
		IN	struct sk_buff *skb,
		IN	WDS_MAC_ADDR ExceptWDSMac,
		IN	struct ieee80211vap *vap
		)
{
	int i;
	struct sk_buff *skb1;
	struct ieee80211_node *ni, *ni_remote = NULL;
	struct ether_header *eh = (struct ether_header *)skb->data;

	if (WdsHandle->AssocNodeList.ulCount < 1) {
		dev_kfree_skb(skb);
		return 0;
	}
	for (i = 0; i < WdsHandle->AssocNodeList.ulCount; i++) {
#ifdef NODE_FREE_DEBUG
		ni = ieee80211_ref_node(WdsHandle->AssocNodeList.assoc_node_ni[i], __func__);
#else
		ni = ieee80211_ref_node(WdsHandle->AssocNodeList.assoc_node_ni[i]);
#endif
		ni_remote = NULL;
		if ((ExceptWDSMac != NULL && 
			memcmp(ni->ni_macaddr, ExceptWDSMac, ETH_ALEN) == 0)
			|| ! WDS_RouteCheck(WdsHandle, ni->ni_macaddr, eh->ether_shost)
			|| ni->ni_vap != vap							// for multi-VAPs: if the ni was not connected with current vap, drop it
			|| (ni_remote=WDS_RemotePCCheck(eh->ether_shost)) == ni)	// donot broadcast to the ni where the pkt recved from
		{
			ieee80211_free_node(ni);
			if (ni_remote != NULL)
				ieee80211_free_node(ni_remote);
			if (i + 1 == WdsHandle->AssocNodeList.ulCount)
			{
				// the last ni not need send, free skb
				dev_kfree_skb(skb);
				skb = NULL;
			}
			continue;
		}
		// if not last ni, do skb_copy for this send
		skb1 = (i + 1 < WdsHandle->AssocNodeList.ulCount) ? skb_copy(skb, GFP_ATOMIC) : skb;
		if (skb1 == NULL)
		{
			ieee80211_free_node(ni);
			continue;
		}
		(*ieee80211_hard_xmit)(vap, ni, skb1);
	}
	return 0;
}

static struct ieee80211_node* WDS_find_wds_rx_node(
		IN const u_int8_t *srcmac)
{
	int i;
	struct ieee80211_node* ni;
	for (i = 0; i < WdsHandle->AssocNodeList.ulCount; i++) {
		ni = WdsHandle->AssocNodeList.assoc_node_ni[i];
		if (IEEE80211_ADDR_EQ(srcmac, ni->ni_macaddr)) {
#ifdef NODE_FREE_DEBUG
			ieee80211_ref_node(ni, __func__);	/* mark referenced */
#else
			ieee80211_ref_node(ni);
#endif
			return ni;
		}
	}
	return NULL;
}

static void F(T_CHAR8 *password,
			  T_UCHAR8 *ssid,
			  T_INT32 ssidlength,
			  T_INT32 iterations,
			  T_INT32 count,
			  T_UCHAR8 *output)
{
	T_UCHAR8 digest[36], digest1[A_SHA_DIGEST_LEN];
	T_INT32 i, j;

	memcpy(digest, ssid, ssidlength);
	digest[ssidlength] = (T_UCHAR8)((count>>24) & 0xff);
	digest[ssidlength+1] = (T_UCHAR8)((count>>16) & 0xff);
	digest[ssidlength+2] = (T_UCHAR8)((count>>8) & 0xff);
	digest[ssidlength+3] = (T_UCHAR8)(count & 0xff);
	IGWHMACSHA1(
		digest, 
		ssidlength+4, 
		(T_UCHAR8*) password, 
		(T_INT32) strlen(password),
		digest1);
	/* output = U1 */ 
	memcpy(output, digest1, A_SHA_DIGEST_LEN);

	for (i = 1; i < iterations; i++) 
	{
		/* Un = PRF(P, Un-1) */
		IGWHMACSHA1( 
			digest1, A_SHA_DIGEST_LEN, 
			(T_UCHAR8*) password, (T_INT32) strlen(password),
			digest);
		memcpy(digest1, digest, A_SHA_DIGEST_LEN);

		/* output = output xor Un */
		for (j = 0; j < A_SHA_DIGEST_LEN; j++) 
		{
			output[j] ^= digest[j];
		}
	}
}

static T_INT32 IGWPasswordHash (T_CHAR8 *password,
								T_UCHAR8 *ssid,
								T_INT32 ssidlength,
								T_UCHAR8 *output)
{
	if ((strlen(password) < IGW8021X_PASS_PHRASE_MIN_LEN) || (strlen(password) > IGW8021X_PASS_PHRASE_MAX_LEN) || (ssidlength > IGW8021X_MAX_SSID_LEN-1))
		return T_FAILURE;

	F(password, ssid, ssidlength, 4096, 1, output);
	F(password, ssid, ssidlength, 4096, 2,&output[A_SHA_DIGEST_LEN]);
	return T_SUCCESS;
}

static void WDS_set_wds_psk(struct ieee80211vap *vap, u_int8_t key_type, u_int8_t key_len, u_int8_t *passPhrase)
{
	int i;
	T_UCHAR8 keytmp[40];
	T_UCHAR8 secKey[40];
	struct ieee80211_node* ni;

	for (i = 0;i < WdsHandle->AssocNodeList.ulCount; i++)
	{
		ni = WdsHandle->AssocNodeList.assoc_node_ni[i];
		if (memcmp(vap->iv_bss->ni_macaddr, ni->ni_macaddr, 6) > 0)
		{
			IGWPasswordHash(passPhrase, vap->iv_bss->ni_macaddr, 6, keytmp);
			memcpy(secKey, keytmp, key_len);
		}
		else if (memcmp(vap->iv_bss->ni_macaddr, ni->ni_macaddr, 6) < 0)
		{
			IGWPasswordHash(passPhrase, ni->ni_macaddr, 6, keytmp);
			memcpy(secKey, keytmp, 16);
			if(32 == key_len)
			{
				memcpy(&secKey[16], &keytmp[24], 8);
				memcpy(&secKey[24], &keytmp[16], 8);
			}
		}
		ieee80211_set_wds_psk(vap, secKey, key_len, key_type, ni->ni_macaddr);
	}
}

struct ieee80211_wds_handle pWdsHandle = 
{
	.iv_wds_send_packet = (ieee80211_wds_send_packet_cb_t)WDS_SendPacket,
	.iv_wds_recv_packet = (ieee80211_wds_recv_packet_cb_t)WDS_ReceivePacket,
	.iv_wds_get_assocnodelist = (ieee80211_wds_get_assocnodelist_cb_t)WDS_GetAssocNodeList,
	.iv_wds_get_assocnodenum = (ieee80211_wds_get_assocnodenum_cb_t)WDS_GetAssocNodeNum,
	.iv_wds_broad_to_wds = (ieee80211_broad_to_wds_cb_t)WDS_BroadcastToWDS,
	.iv_wds_forward_packet= (iv_wds_forward_packet_cb_t)WDS_ForwardPacket,
	.iv_wds_add_assoc_node = (iv_wds_add_assoc_node_cb_t)WDS_AddAssociationNode,
	.iv_wds_del_assoc_node = (iv_wds_del_assoc_node_cb_t)WDS_DelAssocNode,
	.iv_wds_find_rx_node = (iv_wds_find_rx_node_cb_t)WDS_find_wds_rx_node,
	.iv_wds_reset = (iv_wds_reset_cb_t)WDS_Reset,
	.wds_set_wds_psk = (wds_set_wds_psk_cb_t)WDS_set_wds_psk,
	.wds_addRouteMapEntry = (wds_addRouteMapEntry_cb_t)WDS_AddRouteMapEntry,
	.wds_localPCEntryCheck = (wds_localPCEntryCheck_cb_t)WDS_LocalPcEntryCheck,
	.wds_localPcMacAddEntry = (wds_localPcMacAddEntry_cb_t)WDS_LocalPcMacAddEntry,
	.wds_remotePCCheck = (wds_remotePCCheck_cb_t)WDS_RemotePCCheck,
};

void attach_wds_interface(struct ieee80211_wds_handle * *ppWdsHandle,
					ieee80211_hardstart_send_cb_t fc) 
{
	zdebug(ZCOM_DEBUG_WDS, "%s, %d\n", __func__, __LINE__);
	WDS_Initialize();
	*ppWdsHandle=&pWdsHandle ;
	ieee80211_hard_xmit = (ieee80211_hardstart_send_cb_t)fc;
}

void deattach_wds_interface(struct ieee80211_wds_handle * *ppWdsHandle)
{
	zdebug(ZCOM_DEBUG_WDS, "%s, %d\n", __func__, __LINE__);
	*ppWdsHandle = NULL;
	ieee80211_hard_xmit = NULL;
	WDS_Release();
}

/*----------------------------------------------------------------------------
 *
 *---------------------------------------------------------------------------*/
#define	IS_RUNNING(_dev) \
		((_dev->flags & (IFF_RUNNING|IFF_UP)) == (IFF_RUNNING|IFF_UP))

struct ieee80211_node *alloc_wds_node(struct ieee80211vap *vap, u_int8_t *macaddr, u_int8_t is_11n)
{
	struct ieee80211com *ic = vap->iv_ic;
	struct ieee80211_node_table *nt = &ic->ic_sta;
	struct ieee80211_node *ni;

	ni = ieee80211_find_node(nt, macaddr);
	if (ni)
	{
		ni->ni_zcomwds_mode = ZCOMWDS_MODE_NONE;
		ieee80211_crypto_delkey(ni->ni_vap, &ni->ni_ucastkey, ni);
		ieee80211_node_leave(ni);
		ieee80211_free_node(ni);
	}

	ni = ieee80211_dup_bss(vap, macaddr);
	if (ni == NULL)
		goto _end;

	ni->ni_zcomwds_mode = ZCOMWDS_MODE_BRIDGE;
	ni->ni_inact_reload = ni->ni_vap->iv_inact_run;
	//ni->ni_rssi = 35;
	ni->ni_assoctime = jiffies;
	ni->ni_flags |= (IEEE80211_NODE_AUTH | IEEE80211_NODE_QOS
					| IEEE80211_NODE_AREF | IEEE80211_NODE_ERP);
	ni->ni_flags &= (~IEEE80211_NODE_PWR_MGT);
	ni->ni_flags &= (~IEEE80211_NODE_UAPSD_TRIG);
	ni->ni_flags &= (~IEEE80211_NODE_HT);

	if (is_11n)
	{
		ni->ni_flags |= IEEE80211_NODE_HT;
		ni->ni_htcap &= (~IEEE80211_HTCAP_C_SM_MASK);
		ni->ni_htcap |= IEEE80211_HTCAP_C_SM_ENABLED;
		ni->ni_updaterates = IEEE80211_NODE_SM_EN;
		ni->ni_ath_flags |= IEEE80211_NODE_HT_DS;
		if (ic->ic_htflags & IEEE80211_HTF_SHORTGI)
		{
			ni->ni_htcap |= IEEE80211_HTCAP_C_SHORTGI40;
			ni->ni_htcap |= IEEE80211_HTF_SHORTGI;
		}
		else
		{
			ni->ni_htcap &= (~IEEE80211_HTCAP_C_SHORTGI40);
			ni->ni_htcap &= (~IEEE80211_HTF_SHORTGI);
		}
		if( ic->ic_cwm.cw_mode == IEEE80211_CWM_MODE20)
		{	//20HT
			ni->ni_htcap &= ~IEEE80211_HTCAP_C_CHWIDTH40;
			ni->ni_chwidth = IEEE80211_CWM_WIDTH20;
			ic->ic_cwm.cw_width = IEEE80211_CWM_WIDTH20;
			
		}
		else
		{	//40HT
			ni->ni_htcap |=IEEE80211_HTCAP_C_CHWIDTH40;
			ni->ni_chwidth = IEEE80211_CWM_WIDTH40;
			ic->ic_cwm.cw_width = IEEE80211_CWM_WIDTH40;
		}
	}
	vap->iv_sta_assoc++;
	ic->ic_sta_assoc++;
	if(ic->ic_chwidth_change)
		ic->ic_chwidth_change(ni);

	if (ic->ic_newassoc)
		ic->ic_newassoc(ni, 1);
_end:
	//printk("== %s, %d, --------------\n", __func__, __LINE__);
	//ieee80211_dump_node(nt, ni);
	return ni;
}

static void ieee80211_zcom_wds_timer(unsigned long arg)
{
	struct ieee80211vap* vap = (struct ieee80211vap *)arg;
	struct ieee80211com *ic = vap->iv_ic;
	struct ieee80211_node** ni_list = ic->wds_handle->iv_wds_get_assocnodelist();
	int addr4NodeNum = ic->wds_handle->iv_wds_get_assocnodenum(), wdsNodeNum = 0;
	u_int32_t min_fail_count = -1;
	int i, count_inc;

	if(ic->wds_handle == NULL
	  || !IS_RUNNING(vap->iv_dev)
	  || vap->iv_state != IEEE80211_S_RUN
	  || ic->ic_bsschan == IEEE80211_CHAN_ANYC)
	{
		goto _end;
	}
	for (i = 0;i < addr4NodeNum; i++)
	{
		if (ni_list[i] == NULL || !ZCOMWDS_NODE_IS_WDS(ni_list[i]))
			continue;
		if (min_fail_count > ni_list[i]->ni_nulldata_count)
			min_fail_count = ni_list[i]->ni_nulldata_count;
		// try to confirm this node
		count_inc = 0;
		if (vap->iv_nulldata_error
			|| jiffies - ni_list[i]->ni_tx_last > WDS_NULLDATA_INTVAL
			|| jiffies - ni_list[i]->ni_rx_last > WDS_NULLDATA_INTVAL)
		{
			ieee80211_wds_send_nulldata(ieee80211_ref_node(ni_list[i]
				#ifdef NODE_FREE_DEBUG
				, __func__
				#endif
				), IEEE80211_WDS_NULLDATA_REQ);
				count_inc = 1;
		}
		// if lost touch with this node long time, send error info
		if (ni_list[i]->ni_nulldata_count > MAX_WDS_NULLDATA_FAIL_COUNT / 2)
		{
			ieee80211_wds_send_nulldata(ieee80211_ref_node(ni_list[i]
				#ifdef NODE_FREE_DEBUG
				, __func__
				#endif
				), IEEE80211_WDS_NULLDATA_ERR);
				count_inc = 1;
		}
		ni_list[i]->ni_nulldata_count += count_inc;
		wdsNodeNum++;
	}

	//cannot recv nulldata_resp from all the wds nodes, do wireless reset
	if (wdsNodeNum >= ic->ic_sta_assoc
	 && min_fail_count != -1
	 && min_fail_count >= MAX_WDS_NULLDATA_FAIL_COUNT)
	{
		IEEE80211_DPRINTF(vap, IEEE80211_MSG_DEBUG | IEEE80211_MSG_IQUE,
			"%s: WDS cannot recv the response from other side, reset myself\n", __func__);

		for (i = 0;i < addr4NodeNum; i++)
		{
			if (ni_list[i] == NULL || !ZCOMWDS_NODE_IS_WDS(ni_list[i]))
				continue;
			ni_list[i]->ni_nulldata_count = 0;
			vap->iv_nulldata_error = 0;
		}
		ic->ic_reset(ic);
#ifdef _ZCOM_CHAN_HALF_
		// reset bandwidth reg for 5/10MHz
		ic->ic_set_channel(ic);
#endif
	}
_end:
	mod_timer(&ic->wds_nulldata_timer, jiffies + WDS_NULLDATA_INTVAL);
}

void del_wds_timer(struct ieee80211vap *vap)
{
	struct ieee80211com *ic = vap->iv_ic;

	del_timer(&ic->wds_nulldata_timer);
	zdebug(ZCOM_DEBUG_CFG, "%s, %d: del\n", __func__, __LINE__);
}

void add_wds_timer(struct ieee80211vap *vap)
{
	struct ieee80211com *ic = vap->iv_ic;

	vap->iv_nulldata_error = 0;
	del_wds_timer(vap);
	init_timer(&ic->wds_nulldata_timer);
	ic->wds_nulldata_timer.function = ieee80211_zcom_wds_timer;
	ic->wds_nulldata_timer.data = (unsigned long)vap;
	mod_timer(&ic->wds_nulldata_timer, jiffies + 10 * HZ);
	zdebug(ZCOM_DEBUG_CFG, "%s, %d: add\n", __func__, __LINE__);
}

int ieee80211_wds_is_old_nulldata(struct ieee80211_node *ni, u_int8_t *frm, unsigned int len)
{
	/* as the zcn-1523h v3.0.0~3.0.3 wds has 4bytes for owl, we try to identify those devies
	*/
#if 0
#define OLD_WDS_NULLDATA_TAG	"\x48\x03\x00\x00"
	return (memcmp(frm+16, OLD_WDS_NULLDATA_TAG, 4) == 0
		 && memcmp(frm+20, ra, 2) == 0
		 && memcmp(frm+24, ra+2, 4) == 0
		 && memcmp(frm+28, ta, 2) == 0);
#else
	return (frm[16]==0x48 && frm[17]==0x03 && frm[18]==0x00 && frm[19]==0x00)
		&& (frm[20]==frm[4] && frm[21]==frm[5])
		&& (frm[24]==frm[6] && frm[25]==frm[7] && frm[26]==frm[8] && frm[27]==frm[9])
		&& (frm[28]==frm[10] && frm[29]==frm[11]);
#endif
}
int ieee80211_wds_old_send_nulldata(struct ieee80211_node *ni, u_int8_t type)
{
	struct ieee80211vap *vap = ni->ni_vap;
	struct ieee80211com *ic = vap->iv_ic;
    struct net_device *parent = ic->ic_dev;
	struct sk_buff *skb;
	struct ieee80211_cb *cb;
	struct ieee80211_frame_addr4 *wh4;
	u_int8_t *frm;

	skb = ieee80211_getmgtframe(&frm, 6 + 2);
	if (skb == NULL) {
		/* XXX debug msg */
		vap->iv_stats.is_tx_nobuf++;
		ieee80211_free_node(ni);
		return -ENOMEM;
	}
	cb = (struct ieee80211_cb *)skb->cb;
	cb->ni = ni;

	wh4 = (struct ieee80211_frame_addr4 *)skb_push(skb, sizeof(struct ieee80211_frame));
#if 1
    wh4->i_fc[0] = IEEE80211_FC0_VERSION_0 | IEEE80211_FC0_TYPE_DATA | IEEE80211_FC0_SUBTYPE_NODATA;
	wh4->i_fc[1] = IEEE80211_FC1_DIR_DSTODS;
	IEEE80211_ADDR_COPY(wh4->i_addr1, ni->ni_macaddr);
	IEEE80211_ADDR_COPY(wh4->i_addr2, vap->iv_myaddr);
	IEEE80211_ADDR_COPY(wh4->i_addr3, ni->ni_macaddr);
	IEEE80211_ADDR_COPY(wh4->i_addr4, vap->iv_myaddr);
    *(u_int16_t *)&wh4->i_dur[0] = 0;
    /* NB: use non-QoS tid */
    *(u_int16_t *)&wh4->i_seq[0] =
        htole16(ni->ni_txseqs[0] << IEEE80211_SEQ_SEQ_SHIFT);
    ni->ni_txseqs[0]++;
#else
	ieee80211_send_setup(vap, ni, (struct ieee80211_frame*)wh4,
		IEEE80211_FC0_TYPE_DATA | IEEE80211_FC0_SUBTYPE_NODATA,
		vap->iv_myaddr, ni->ni_macaddr, ni->ni_bssid);
#endif
	frm = (u_int8_t *)&wh4[1];
	*frm = type;
	IEEE80211_DPRINTF(vap, IEEE80211_MSG_DEBUG | IEEE80211_MSG_DUMPPKTS,
	    "[%s] send null data 0x%x frame on channel %u for wds\n",
	    ether_sprintf(ni->ni_macaddr), type, ieee80211_chan2ieee(ic, ic->ic_curchan));

	/* XXX assign some priority; this probably is wrong */
	skb->priority = WME_AC_BE;
    skb->dev = parent;

	IEEE80211_NODE_STAT(ni, tx_data);
	ni->ni_tx_last = jiffies;
    ic->ic_lastdata = jiffies;
    vap->iv_lastdata = jiffies;

    (void) dev_queue_xmit(skb);
	return 0;
}

/*
 * NULLDATA frame format

 * |0    |2    |4    |6    |8    |10   |12   |14 15|
 * |-----------------------------------------------|
 * |i_fc |i_dur i_addr1          |i_addr2          |
 * |-----------------------------------------------|
 * |i_addr3          |i_seq|i_addr4          |pad  |
 * |-----------------------------------------------|
 * |nulldata               |N/A                    |
 * |-----------------------------------------------|
 * /                       \
 * "ALIVE"  VER  Type  OWL
 */
void ieee80211_wds_recv_nulldata(struct ieee80211_node *ni, u_int8_t *frm, unsigned int len, u_int8_t owl_cmpt)
{
	struct ieee80211vap *vap = ni->ni_vap;
	struct ieee80211com *ic = vap->iv_ic;
	u_int8_t *pNullData = frm + 8;		// skip the 4th addr and 2byte pads
	u_int8_t version = 0, type = 0, owl = owl_cmpt;

#if 0
	{
	u_int8_t *p = frm - sizeof(struct ieee80211_frame);
	int i;
	printk("____RX:len=%d___", len);
	for (i=0; i<sizeof(struct ieee80211_frame_addr4) + 64; i++)
		printk("%c%02x", i%16==0?'\n':' ', p[i]);
	printk("\n");
	}
#endif
	if (owl_cmpt)
	{
		// hack for CMPT Fw2.0.3 & 3.0.0 & 3.0.1 nulldata format
		type = *(frm + 36);
		goto nulldata_rx;
	}
	if (len < IEEE80211_WDS_NULLDATA_LEN
	 || memcmp(pNullData, IEEE80211_WDS_NULLDATA_MAGIC, IEEE80211_WDS_NULLDATA_MAGIC_LEN) != 0)
		goto _end;
	pNullData += IEEE80211_WDS_NULLDATA_MAGIC_LEN;
	version = *(pNullData++);
	if (version > IEEE80211_WDS_NULLDATA_VERSION && (version | IEEE80211_WDS_NULLDATA_UNCMPT))
		// uncompatible version, unknown format
		goto _end;
	type = *(pNullData++);
	owl = *(pNullData++);

nulldata_rx:
	if (owl)
		ni->ni_flags |= IEEE80211_NODE_OWL_WDSWAR;
	else
		ni->ni_flags &= ~IEEE80211_NODE_OWL_WDSWAR;

	switch (type)
	{
	case IEEE80211_WDS_NULLDATA_OK:
		vap->iv_nulldata_error = 0;
		break;
	case IEEE80211_WDS_NULLDATA_ERR:
		// the other side told me: I has error. so has to confirm whether the connection being ok
		vap->iv_nulldata_error = 1;
		break;
	case IEEE80211_WDS_NULLDATA_REQ:
		ieee80211_wds_send_nulldata(ieee80211_ref_node(ni
		#ifdef NODE_FREE_DEBUG
		, __func__
		#endif
		), IEEE80211_WDS_NULLDATA_RESP);
		break;
	case IEEE80211_WDS_NULLDATA_RESP:
		ni->ni_nulldata_count = 0;
		vap->iv_nulldata_error = 0;
		break;
	default:
		// not support
		break;
	}
_end:
	IEEE80211_DPRINTF(vap, IEEE80211_MSG_DEBUG | IEEE80211_MSG_DUMPPKTS,
		"[%s] recv null data len=%d, ver=0x%x, type=0x%x, owl=0x%x, frame on channel %u for wds\n",
		ether_sprintf(ni->ni_macaddr), len, version, type, owl, ieee80211_chan2ieee(ic, ic->ic_curchan));
	ni->ni_rx_last = jiffies;
}

int ieee80211_wds_send_nulldata(struct ieee80211_node *ni, u_int8_t type)
{
	struct ieee80211vap *vap = ni->ni_vap;
	struct ieee80211com *ic = vap->iv_ic;
	struct sk_buff *skb;
	struct ieee80211_cb *cb;
	struct ieee80211_frame_addr4 *wh4;
	u_int8_t *frm;

	if (ni->ni_flags & IEEE80211_NODE_OWL_WDSWAR)
		return ieee80211_wds_old_send_nulldata(ni, type);

	// total_frame_len = sizeof(struct ieee80211_frame_addr4) + 2 + IEEE80211_WDS_NULLDATA_LEN
	// = sizeof(struct ieee80211_frame) + IEEE80211_ADDR_LEN + 2pad + IEEE80211_WDS_NULLDATA_LEN
	// = 8 + IEEE80211_WDS_NULLDATA_LEN
	skb = ieee80211_getmgtframe(&frm, 8 + IEEE80211_WDS_NULLDATA_LEN);
	if (skb == NULL) {
		/* XXX debug msg */
		vap->iv_stats.is_tx_nobuf++;
		ieee80211_free_node(ni);
		return -ENOMEM;
	}
	frm += 8;
	memset(frm, 0, IEEE80211_WDS_NULLDATA_LEN);
	// "alive", version, type, owl
	memcpy(frm, IEEE80211_WDS_NULLDATA_MAGIC, IEEE80211_WDS_NULLDATA_MAGIC_LEN);
	frm += IEEE80211_WDS_NULLDATA_MAGIC_LEN;
	*(frm++) = IEEE80211_WDS_NULLDATA_VERSION;
	*(frm++) = type;
	*(frm++) = (vap->iv_bss->ni_flags & IEEE80211_NODE_OWL_WDSWAR);

	cb = (struct ieee80211_cb *)skb->cb;
	cb->ni = ni;

	wh4 = (struct ieee80211_frame_addr4 *)skb_push(skb, sizeof(struct ieee80211_frame));
#if 1
    wh4->i_fc[0] = IEEE80211_FC0_VERSION_0 | IEEE80211_FC0_TYPE_DATA | IEEE80211_FC0_SUBTYPE_NODATA;
	wh4->i_fc[1] = IEEE80211_FC1_DIR_DSTODS;
	IEEE80211_ADDR_COPY(wh4->i_addr1, ni->ni_macaddr);
	IEEE80211_ADDR_COPY(wh4->i_addr2, vap->iv_myaddr);
	IEEE80211_ADDR_COPY(wh4->i_addr3, ni->ni_macaddr);
	IEEE80211_ADDR_COPY(wh4->i_addr4, vap->iv_myaddr);
    *(u_int16_t *)&wh4->i_dur[0] = 0;
    /* NB: use non-QoS tid */
    *(u_int16_t *)&wh4->i_seq[0] =
        htole16(ni->ni_txseqs[0] << IEEE80211_SEQ_SEQ_SHIFT);
    ni->ni_txseqs[0]++;
#else
	ieee80211_send_setup(vap, ni, (struct ieee80211_frame*)wh4,
		IEEE80211_FC0_TYPE_DATA | IEEE80211_FC0_SUBTYPE_NODATA,
		vap->iv_myaddr, ni->ni_macaddr, ni->ni_bssid);
#endif
	IEEE80211_DPRINTF(vap, IEEE80211_MSG_DEBUG | IEEE80211_MSG_DUMPPKTS,
	    "[%s] send null data 0x%x frame on channel %u for wds\n",
	    ether_sprintf(ni->ni_macaddr), type, ieee80211_chan2ieee(ic, ic->ic_curchan));

	/* XXX assign some priority; this probably is wrong */
	skb->priority = WME_AC_BE;
    ieee80211_pwrsave_wakeup(vap, TRANSMIT);
	IEEE80211_NODE_STAT(ni, tx_data);
	ni->ni_tx_last = jiffies;
	ic->ic_lastdata = jiffies;
    vap->iv_lastdata = jiffies;

	(void) ic->ic_mgtstart(ic, skb);	/* cheat */
	return 0;
}

#endif
