#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/sched.h>
#include <linux/version.h>
#include <linux/module.h>
#include <linux/netdevice.h>
#include <linux/utsname.h>
#include <linux/if_arp.h>		/* XXX for ARPHRD_ETHER */
#include <net/iw_handler.h>
#include <asm/uaccess.h>
#include <asm/processor.h>

#include "if_media.h"
#include <net80211/ieee80211_var.h>
#include "ieee80211_dev_config.h"
#include "ieee80211.h"
#ifdef _ZCOM_INFO_
#include "if_zcom_info.h"
#endif

#ifdef ZCOM_TDM_COORDINATION		//add for STDM
#include "stdm_config.h"
#endif

static if_user_cfg_t 	g_user_cfg;
#define g_dev_cfg 		g_user_cfg
#define g_vap_cfg 		g_user_cfg.iu_vap_cfg

const static WlanIfElem_t g_pub_elems[] = 
{
	{IF_ELEM_INT,	"powerlevel",		FIELD_INFO(if_user_cfg_t,	iu_powerlevel)			},
	{IF_ELEM_INT,	"distance",			FIELD_INFO(if_user_cfg_t,	iu_distance)			},
	{IF_ELEM_INT,	"antenna", 			FIELD_INFO(if_user_cfg_t,	iu_antenna)				},
	{IF_ELEM_INT,	"ampdu_enable", 	FIELD_INFO(if_user_cfg_t,	iu_ampdu_enable)		},
	{IF_ELEM_INT,	"ampdu_frames", 	FIELD_INFO(if_user_cfg_t,	iu_ampdu_frames)		},
	{IF_ELEM_INT,	"ampdu_limit", 		FIELD_INFO(if_user_cfg_t,	iu_ampdu_limit)			},

	{IF_ELEM_INT,	"domain",			FIELD_INFO(if_user_cfg_t,	iu_domain)				},
	{IF_ELEM_INT,	"opmode",			FIELD_INFO(if_user_cfg_t,	iu_opmode)				},
	{IF_ELEM_INT,	"wir_mode", 		FIELD_INFO(if_user_cfg_t,	iu_wir_mode)			},
	{IF_ELEM_INT,	"rts_thres",		FIELD_INFO(if_user_cfg_t,	iu_rts_thres)			},
	{IF_ELEM_INT,	"frag_thres",		FIELD_INFO(if_user_cfg_t,	iu_frag_thres) 			},
	{IF_ELEM_INT,	"preamble", 		FIELD_INFO(if_user_cfg_t,	iu_preamble)			},
	{IF_ELEM_INT,	"dtim_intvl",		FIELD_INFO(if_user_cfg_t,	iu_dtim_intvl) 			},
	{IF_ELEM_INT,	"bcn_intvl",		FIELD_INFO(if_user_cfg_t,	iu_bcn_intvl)			},
	{IF_ELEM_INT,	"amsdu_enable", 	FIELD_INFO(if_user_cfg_t,	iu_amsdu_enable)		},
	{IF_ELEM_INT,	"htprot_enable",	FIELD_INFO(if_user_cfg_t,	iu_htprot_enable)		},
	{IF_ELEM_INT,	"cwm_mode", 		FIELD_INFO(if_user_cfg_t,	iu_cwm_mode)			},
	{IF_ELEM_INT,	"chan_offset", 		FIELD_INFO(if_user_cfg_t,	iu_chan_offset)			},
	{IF_ELEM_INT,	"rifs_enable",		FIELD_INFO(if_user_cfg_t,	iu_rifs_enable)			},
	{IF_ELEM_INT,	"txrate",			FIELD_INFO(if_user_cfg_t,	iu_txrate) 				},
	{IF_ELEM_INT,	"shortgi_enable",	FIELD_INFO(if_user_cfg_t,	iu_shortgi_enable) 		},
	{IF_ELEM_INT,	"ext_protmode", 	FIELD_INFO(if_user_cfg_t,	iu_ext_protmode)		},
	{IF_ELEM_INT,	"full_link",		FIELD_INFO(if_user_cfg_t,	iu_full_link)			},
	{IF_ELEM_INT,	"channel",			FIELD_INFO(if_user_cfg_t,	iu_channel)				},
	{IF_ELEM_INT,	"ctl_enable",		FIELD_INFO(if_user_cfg_t,	iu_ctl_enable) 			},
	{IF_ELEM_INT,	"ignore11d",		FIELD_INFO(if_user_cfg_t,	iu_ignore11d)			},
	{IF_ELEM_INT,	"Radiotest_Enable", FIELD_INFO(if_user_cfg_t,	iu_radiotest_enable)	},
	{IF_ELEM_INT,	"Radiotest_Total",	FIELD_INFO(if_user_cfg_t,	iu_radiotest_total)		},
	{IF_ELEM_MAC,	"Radiotest_Mac",	FIELD_INFO(if_user_cfg_t,	iu_radiotest_mac)		},
	{IF_ELEM_MACS,	"wds_entry",		FIELD_INFO(if_user_cfg_t,	iu_wds_cfg.iu_wds_addr)	},
	{IF_ELEM_INT,	"wds_auth", 		FIELD_INFO(if_user_cfg_t,	iu_wds_cfg.iu_wds_auth)	},
	{IF_ELEM_STR,	"wds_key",			FIELD_INFO(if_user_cfg_t,	iu_wds_cfg.iu_wds_key) 	},
	{IF_ELEM_STR,	"wep_key1",			FIELD_INFO(if_user_cfg_t,	iu_wep_key[0]) 			},
	{IF_ELEM_STR,	"wep_key2",			FIELD_INFO(if_user_cfg_t,	iu_wep_key[1]) 			},
	{IF_ELEM_STR,	"wep_key3",			FIELD_INFO(if_user_cfg_t,	iu_wep_key[2]) 			},
	{IF_ELEM_STR,	"wep_key4",			FIELD_INFO(if_user_cfg_t,	iu_wep_key[3]) 			},
#ifdef _ZCOM_INFO_
	{IF_ELEM_INT,	"ip_info", 			FIELD_INFO(if_user_cfg_t,	iu_info_cfg.iu_ip)		},
	{IF_ELEM_STR,	"name_info",		FIELD_INFO(if_user_cfg_t,	iu_info_cfg.iu_name)	},
	{IF_ELEM_STR,	"ver_info",			FIELD_INFO(if_user_cfg_t,	iu_info_cfg.iu_ver)		},
#endif
#ifdef _ZCOM_LAN2LAN_
	{IF_ELEM_INT,	"lan2lan",			FIELD_INFO(if_user_cfg_t,	iu_lan2lan)				},
#endif
#ifdef _ZCOM_TDM_WDS_
	{IF_ELEM_INT,	"wds_stdm_mode",	FIELD_INFO(if_user_cfg_t,	iu_wds_cfg.iu_wds_stdm_mode) },
#endif
};
const static WlanIfElem_t g_vap_elems[] = 
{
	{IF_ELEM_INT,	"vap_enable",		FIELD_INFO(if_vap_cfg_t,	iu_vap_enable)				},
	{IF_ELEM_STR,	"vap_name", 		FIELD_INFO(if_vap_cfg_t,	iu_vap_name)				},
	{IF_ELEM_STR,	"ssid", 			FIELD_INFO(if_vap_cfg_t,	iu_ssid)					},
	{IF_ELEM_INT,	"vlanid",			FIELD_INFO(if_vap_cfg_t,	iu_vlanid)					},
	{IF_ELEM_INT,	"wmm_enable",		FIELD_INFO(if_vap_cfg_t,	iu_wmm_enable)				},
	{IF_ELEM_INT,	"max_stanum",		FIELD_INFO(if_vap_cfg_t,	iu_max_stanum) 				},
	{IF_ELEM_INT,	"auth_type",		FIELD_INFO(if_vap_cfg_t,	iu_auth_type)				},
	{IF_ELEM_INT,	"encrypt",			FIELD_INFO(if_vap_cfg_t,	iu_encrypt) 				},
	{IF_ELEM_INT,	"wep_def_no",		FIELD_INFO(if_vap_cfg_t,	iu_wep_def_no)				},
	{IF_ELEM_INT,	"acl_policy",		FIELD_INFO(if_vap_cfg_t,	iu_acl_cfg.iu_acl_policy)	},
	{IF_ELEM_MACS,	"acl_addr", 		FIELD_INFO(if_vap_cfg_t,	iu_acl_cfg.iu_acl_addr) 	},
	{IF_ELEM_INT,	"intra_bss_enable", FIELD_INFO(if_vap_cfg_t,	iu_intra_bss_enable)		},
	{IF_ELEM_INT,	"privacy",			FIELD_INFO(if_vap_cfg_t,	iu_privacy) 				},
	{IF_ELEM_INT,	"hidden_ssid",		FIELD_INFO(if_vap_cfg_t,	iu_hidden_ssid) 			},
	{IF_ELEM_INT,	"scan11n",			FIELD_INFO(if_vap_cfg_t,	iu_scan11n_enable)			},
	{IF_ELEM_INT,	"IGMP_snooping",	FIELD_INFO(if_vap_cfg_t,	iu_igmpsnooping)			},
#ifdef ZCOM_TDM_COORDINATION			// add for STDM
	{IF_ELEM_INT,	"stdm_enable",		FIELD_INFO(if_vap_cfg_t,	iu_stdm)					},
#endif
	{IF_ELEM_INT,	"wds_isolation",	FIELD_INFO(if_vap_cfg_t,	iu_wds_isolation)			},
};

#ifdef	_ZCOM_DEBUG_
uint32_t g_zcomDebugLevel = 0;
void zcom_debug(const char *fmt, ...)
{
    char buf[128];      /* XXX */
    va_list ap;

    va_start(ap, fmt);
    vsnprintf(buf, sizeof(buf), fmt, ap);
    va_end(ap);

    printk("%s", buf);   /* NB: no \n */
}
EXPORT_SYMBOL_C(g_zcomDebugLevel);
EXPORT_SYMBOL_C(zcom_debug);
#endif

unsigned char AsciiToHex(unsigned char ucAscii)
{
    if ((ucAscii>=0x30) &&(ucAscii<=0x39))
	{
		return ucAscii-0x30;
	}
    else
	{
	    return ucAscii-0x37;
	}
}

void MacAsciiToHex(unsigned char *pAscii,unsigned char *pHex)
{
	unsigned char index;
	for(index=0;index<6;index++)
	{
		*(pHex+index) = ((AsciiToHex(*(pAscii+index*2))<<4)&0xf0) |(AsciiToHex(*(pAscii+index*2+1))&0x0f);
	}
}

static int hexkey2asckey(unsigned char *input, unsigned char* key)
{

	int			keylen = 0;
	const char	*p;
	int			dlen;		/* Digits sequence length */
	unsigned char	out[64];

	/* Third case : as hexadecimal digits */
	p = input;
	dlen = -1;

	/* Loop until we run out of chars in input or overflow the output */
	while(*p != '\0')
	{
		int temph;
		int templ;
		int count;
		char cc[2] = {0};
		/* No more chars in this sequence */
		if(dlen <= 0)
		{
			/* Skip separator */
			if(dlen == 0)
				p++;
			/* Calculate num of char to next separator */
			dlen = strcspn(p, "-:;.,");
		}
		/* Get each char separatly (and not by two) so that we don't
		* get confused by 'enc' (=> '0E'+'0C') and similar */
		count = 2;
		memcpy(cc, &p[0], 1);
		temph = strtoul(cc);
		memcpy(cc, &p[1], 1);
		templ = strtoul(cc);
		if(count < 1)
			return(-1);		/* Error -> non-hex char */
		/* Fixup odd strings such as '123' is '01'+'23' and not '12'+'03'*/
		if(dlen % 2)
			count = 1;
		/* Put back two chars as one byte and output */
		if(count == 2)
			templ |= temph << 4;
		else
			templ = temph;
		out[keylen++] = (unsigned char) (templ & 0xFF);
		/* Check overflow in output */
		if(keylen >= 64)
			break;
		/* Move on to next chars */
		p += count;
		dlen -= count;
	}
	/* We use a temporary output buffer 'out' so that if there is
	* an error, we don't overwrite the original key buffer.
	* Because of the way iwconfig loop on multiple key/enc arguments
	* until it finds an error in here, this is necessary to avoid
	* silently corrupting the encryption key... */
	memcpy(key, out, keylen);
	return keylen;
}

static int set_elem_val(void *target,
						const WlanIfElem_t *elems, u_int16_t elem_count,
						char* buf_line, u_int16_t buf_len)
{
	char *eq_tag, *elem_name, *elem_val;
	int i, val_len, offset, count;
	u_int8_t *pu8_val;
	u_int16_t *pu16_val;
	u_int32_t *pu32_val;

	eq_tag = strchr(buf_line, TAG_EQ_CHR);
	if (eq_tag == NULL)
		return -1;
	*eq_tag = '\0';
	elem_name = buf_line;
	elem_val = eq_tag + 1;
	val_len = buf_line + buf_len - elem_val;

	for (i = 0; i < elem_count; i++)
	{
		if (strcmp(elems[i].name, elem_name) != 0)
			continue;
		void *field = target + elems[i].offset;
		switch(elems[i].type)
		{
		case IF_ELEM_INT:
			switch(elems[i].len)
			{
			case sizeof(u_int8_t):
				pu8_val = field;
				*pu8_val = atoi(elem_val);
				zdebug(ZCOM_DEBUG_CFG, "--- %s: %s=%d\n", __func__, elem_name, *pu8_val);
				break;
			case sizeof(u_int16_t):
				pu16_val = field;
				*pu16_val = atoi(elem_val);
				zdebug(ZCOM_DEBUG_CFG, "--- %s: %s=%d\n", __func__, elem_name, *pu16_val);
				break;
			case sizeof(u_int32_t):
				pu32_val = field;
				*pu32_val = atoi(elem_val);
				zdebug(ZCOM_DEBUG_CFG, "--- %s: %s=%d\n", __func__, elem_name, *pu32_val);
				break;
			default:
				zdebug(ZCOM_DEBUG_CFG, "%s: Unknown len of IF_ELEM_INT!\n", __func__);
			}
			break;
		case IF_ELEM_STR:
			strncpy(field, elem_val, strlen(elem_val));
			zdebug(ZCOM_DEBUG_CFG, "--- %s: %s=%s\n", __func__, elem_name, (char*)field);
			break;
		case IF_ELEM_MAC:
			if (val_len >= IEEE80211_ADDR_LEN)
			{
				memcpy(field, elem_val, IEEE80211_ADDR_LEN);
			}
			zdebug(ZCOM_DEBUG_CFG, "--- %s: %s=%s\n", __func__, elem_name, ether_sprintf(field));
			break;
		case IF_ELEM_MACS:
			for (offset=0, count=0;
				(offset+2*IEEE80211_ADDR_LEN<=val_len) && (count*IEEE80211_ADDR_LEN<elems[i].len);
				offset+=2*IEEE80211_ADDR_LEN+1)
			{
				if (memcmp(elem_val+offset, NULL_MAC, 2*IEEE80211_ADDR_LEN) != 0)
				{
					//memcpy(field+(count*IEEE80211_ADDR_LEN), elem_val+offset, IEEE80211_ADDR_LEN);
					MacAsciiToHex(elem_val+offset, field+(count*IEEE80211_ADDR_LEN));  //huangyu modify for "0xa" 2010-04-07
					count++;
					zdebug(ZCOM_DEBUG_CFG, "--- %s: %s=%s\n", __func__, elem_name, ether_sprintf(field + ((count-1)*IEEE80211_ADDR_LEN)));
				}
			}
			
			break;
		default:
			printk("%s: Unknown IF_ELEM_TYPE!\n", __func__);
		}
		return 0;
	}
	return -1;
}

static int read_line(struct file *fp, unsigned char *line_buf, int buf_len)
{
	unsigned char c;
	int len = 0, errcode = 0;

	if (fp->f_op && fp->f_op->read)
	{
		memset(line_buf, 0, buf_len);
		do
		{
			errcode = fp->f_op->read((fp), &c, 1, &(fp)->f_pos);
			if (errcode < 0 || c == '\n')
				break;
			line_buf[len++] = c;
		} while (c != '\n' && len < buf_len - 1);
	}
	line_buf[len] = '\0';
	return len;
}

int check_channel_offset(struct ieee80211com *ic, u_int8_t wirmode, u_int8_t channel, u_int8_t bandwidth, u_int8_t offset)
{
	int i;
	u_int8_t valid = WIRELESS_CWM_OFFSET_NONE;
	u_int8_t is11n = (wirmode == ZCOM_WIRMODE_NA || wirmode == ZCOM_WIRMODE_NA_ONLYN
			  	  || wirmode == ZCOM_WIRMODE_NG || wirmode == ZCOM_WIRMODE_NG_ONLYN);

	for(i=0; i < ic->ic_nchans; i++)
	{
		struct ieee80211_channel *c = &ic->ic_channels[i];
		if(c->ic_ieee == channel || channel == 0)
		{
			if (is11n && bandwidth > 0 && valid!=WIRELESS_CWM_OFFSET_NONE)
			{
				if (IEEE80211_IS_CHAN_11N_CTL_U_CAPABLE(c))
					valid = WIRELESS_CWM_OFFSET_UPPER;
				if (IEEE80211_IS_CHAN_11N_CTL_L_CAPABLE(c))
					valid = WIRELESS_CWM_OFFSET_LOWER;
			}
			if((IEEE80211_IS_CHAN_11N_CTL_U_CAPABLE(c) && offset == WIRELESS_CWM_OFFSET_UPPER)
			 ||(IEEE80211_IS_CHAN_11N_CTL_L_CAPABLE(c) && offset == WIRELESS_CWM_OFFSET_LOWER))
			{
				return offset;
			}
		}
	}
	// if the given offset was invalid, return a valid one
	return valid;
}

#ifdef _ZCOM_TDM_WDS_
static int check_wds_stdm_config(if_user_cfg_t *user_cfg)
{
	switch (user_cfg->iu_opmode)
	{
	case ZCOM_OPMODE_P2P:
		if (user_cfg->iu_wds_cfg.iu_wds_stdm_mode != WDS_STDM_DISABLE
		 && user_cfg->iu_wds_cfg.iu_wds_stdm_mode != WDS_STDM_BRIDGE_SLAVE)
		{
			user_cfg->iu_wds_cfg.iu_wds_stdm_mode = WDS_STDM_DISABLE;
		}
		break;
	case ZCOM_OPMODE_P2P_AP:
		if (user_cfg->iu_wds_cfg.iu_wds_stdm_mode != WDS_STDM_DISABLE
//		 && user_cfg->iu_wds_cfg.iu_wds_stdm_mode != WDS_STDM_BRIDGE_MASTER
		 && user_cfg->iu_wds_cfg.iu_wds_stdm_mode != WDS_STDM_AP_REPEATER_MASTER)
		{
			user_cfg->iu_wds_cfg.iu_wds_stdm_mode = WDS_STDM_DISABLE;
		}
		break;
	default:
		// For the other cases, must set WDS_STDM mode to DISABLE.
		user_cfg->iu_wds_cfg.iu_wds_stdm_mode = WDS_STDM_DISABLE;
		break;
	}
	return 0;
}
#endif

/*
 * validate settings, check the dependences of settings
 * return value: 0-success; other-failed
 */
static int check_config(struct ieee80211com *ic, if_user_cfg_t *user_cfg)
{
	struct if_vap_cfg_t *vap_cfg = NULL;
	int is11n, i;

	is11n = (user_cfg->iu_wir_mode == ZCOM_WIRMODE_NA
	  	  || user_cfg->iu_wir_mode == ZCOM_WIRMODE_NA_ONLYN
	  	  || user_cfg->iu_wir_mode == ZCOM_WIRMODE_NG
	  	  || user_cfg->iu_wir_mode == ZCOM_WIRMODE_NG_ONLYN);

	if (user_cfg->iu_opmode != ZCOM_OPMODE_AP && user_cfg->iu_opmode != ZCOM_OPMODE_CLIENT)
		user_cfg->iu_full_link = 0;
#ifdef _ZCOM_CHAN_HALF_
	if (user_cfg->iu_cwm_mode == IEEE80211_CWM_MODEHALF)
	{
		user_cfg->iu_cwm_mode = IEEE80211_CWM_MODE20;
		user_cfg->iu_chan_bandwidth = 1;
	}
	else if (user_cfg->iu_cwm_mode == IEEE80211_CWM_MODEQUARTER)
	{
		user_cfg->iu_cwm_mode = IEEE80211_CWM_MODE20;
		user_cfg->iu_chan_bandwidth = 2;
	}
	else
		user_cfg->iu_chan_bandwidth = 0;
#endif
	if (!is11n)
	{
		user_cfg->iu_cwm_mode = IEEE80211_CWM_MODE20;
		user_cfg->iu_shortgi_enable = 0;
	}
	for (i=0; i<MAX_VAP_NUM; i++)
	{
		vap_cfg = &g_vap_cfg[i];
		if (is11n)
		{
			vap_cfg->iu_wmm_enable = 1;
		}
		if (vap_cfg->iu_max_stanum >= IEEE80211_AID_DEF)
			vap_cfg->iu_max_stanum = IEEE80211_AID_DEF - 1;
	}
	if (user_cfg->iu_powerlevel == 0)
		user_cfg->iu_powerlevel = IEEE80211_TXPOWER_MAX;
#ifdef _ZCOM_LAN2LAN_
	// only client allow lan2lan; ap was auto lan2lan
	if (user_cfg->iu_opmode != ZCOM_OPMODE_CLIENT)
		user_cfg->iu_lan2lan = 0;
#endif
	if (user_cfg->iu_opmode != ZCOM_OPMODE_CLIENT)
		user_cfg->iu_chan_offset = check_channel_offset(ic, user_cfg->iu_wir_mode, user_cfg->iu_channel,
														user_cfg->iu_cwm_mode, user_cfg->iu_chan_offset);
	else
		user_cfg->iu_chan_offset = 0;

#ifdef _ZCOM_TDM_WDS_
	check_wds_stdm_config(user_cfg);
#endif

	return 0;
}

/*
 * return value: 0-success; other-failed
 */
static int read_config(struct ieee80211com *ic, const char *path, if_user_cfg_t *user_cfg)
{
	mm_segment_t old_fs;
	struct file *fp = NULL;
	unsigned char buf_line[512];
	int buf_len, errcode = 0, vapIndex = 0;
	uint8_t ifType = ZCOM_IFTYPE_NONE;

	zdebug(ZCOM_DEBUG_CFG, "== %s, %d\n", __func__, __LINE__);
	old_fs = get_fs();
	set_fs(KERNEL_DS);	//set kernel env

	fp = filp_open(path, O_RDONLY, 0);
	if(IS_ERR(fp))
	{
		printk("%s: failed to open file %s\n", __func__, path);
		errcode = -ENOENT;
		goto _end;
	}

	memset(&g_user_cfg, 0, sizeof(g_user_cfg));
	//memset(&this_vap, 0, sizeof(this_vap));
	while((buf_len = read_line(fp, buf_line, sizeof(buf_line))))
	{
		//skip explain and empty line
		if (buf_line[0] == '#' || buf_line[0] == 0 || buf_line[0]=='\n')
			continue;

		if (buf_line[0] == TAG_SECTION_START)
		{
			if (memcmp(&buf_line[1], TAG_WIFI_START, strlen(TAG_WIFI_START)) == 0)
			{
				ifType = ZCOM_IFTYPE_WIFI;
			}
			else if (memcmp(&buf_line[1], TAG_VAP_START, strlen(TAG_VAP_START)) == 0)
			{
				ifType |= ZCOM_IFTYPE_VAP;
				vapIndex = atoi((char*)&buf_line[1 + strlen(TAG_VAP_START) + 1]);
				if (vapIndex >= MAX_VAP_NUM)
					goto _end;
			}
			else if (memcmp(&buf_line[1], TAG_VAP_END, strlen(TAG_VAP_END)) == 0)
			{
				ifType &= ~ZCOM_IFTYPE_VAP;
			}
			else if (memcmp(&buf_line[1], TAG_WIFI_END, strlen(TAG_WIFI_END)) == 0)
				ifType = ZCOM_IFTYPE_NONE;
		}
		else
		{
			if(ifType & ZCOM_IFTYPE_VAP)
				set_elem_val(&g_vap_cfg[vapIndex], g_vap_elems, TABLE_SIZE(g_vap_elems), buf_line, buf_len);
			else
				set_elem_val(&g_user_cfg, g_pub_elems, TABLE_SIZE(g_pub_elems), buf_line, buf_len);
		}
	}
	if (filp_close(fp, NULL))
		errcode = -ENOENT;
	if (check_config(ic, &g_user_cfg) < 0)
		errcode = -EINVAL;

_end:
	set_fs(old_fs);
	return errcode;
}

static void config_auth(struct ieee80211vap *vap, u_int8_t authType, u_int8_t encrypt, u_int8_t wepIndex, u_int8_t wepKey[4][WEP_KEY_LEN])
{
	u_int8_t authMode = IEEE80211_AUTH_OPEN;
	u_int32_t wpa_flags = 0;		//IEEE80211_F_WPA
	const struct ieee80211_authenticator *auth = NULL;
	struct iw_point erq = {0};

	switch (authType)
	{
	case AUTH_OPEN_SYSTEM:
		authMode = IEEE80211_AUTH_OPEN;
		break;
	case AUTH_SHARED_KEY:
		authMode = IEEE80211_AUTH_SHARED;
		break;
	case AUTH_802_1X:
		authMode = IEEE80211_AUTH_8021X;
		break;
	case AUTH_WPA_RADIUS:
	case AUTH_WPA_PSK:
		authMode = IEEE80211_AUTH_WPA;
		wpa_flags |= IEEE80211_F_WPA1;
		break;
	case AUTH_WPA2_PSK:
	case AUTH_WPA2_RADIUS:
		authMode = IEEE80211_AUTH_WPA;
		wpa_flags |= IEEE80211_F_WPA2;
		break;
	case AUTH_WPA1_2_PSK:
	case AUTH_WPA1_2_RADIUS:
		authMode = IEEE80211_AUTH_WPA;
		wpa_flags |= (IEEE80211_F_WPA|IEEE80211_F_WPA2);
		break;
	}

	switch (authMode)
	{
	case IEEE80211_AUTH_WPA:		/* WPA */
	case IEEE80211_AUTH_8021X:		/* 802.1x */
	case IEEE80211_AUTH_OPEN:		/* open */
	case IEEE80211_AUTH_SHARED:		/* shared-key */
	case IEEE80211_AUTH_AUTO:		/* auto */
		auth = ieee80211_authenticator_get(authMode);
		if (auth == NULL)
			break;
		break;
	default:
		break;
	}

	switch (authMode)
	{
	case IEEE80211_AUTH_WPA:		/* WPA w/ 802.1x */
		vap->iv_flags |= wpa_flags;
		vap->iv_flags |= IEEE80211_F_PRIVACY;
		authMode = IEEE80211_AUTH_8021X;
		break;
	case IEEE80211_AUTH_OPEN:		/* open */
		vap->iv_flags &= ~(IEEE80211_F_WPA);
		break;
	case IEEE80211_AUTH_SHARED:		/* shared-key */
	case IEEE80211_AUTH_AUTO:		/* auto */
	case IEEE80211_AUTH_8021X:		/* 802.1x */
		vap->iv_flags &= ~IEEE80211_F_WPA;
		/* both require a key so mark the PRIVACY capability */
		vap->iv_flags |= IEEE80211_F_PRIVACY;
		break;
	}
	if (auth)
	{
		/* NB: authenticator attach/detach happens on state change */
		vap->iv_bss->ni_authmode = authMode;
		/* XXX mixed/mode/usage? */
		vap->iv_auth = auth;
	}
	/* WEP key setting */
	erq.flags |= IW_ENCODE_DISABLED;
	config_wep_encode(vap, &erq, NULL);
	if(encrypt == ENCRYPT_WEP && wepIndex > 0 && wepIndex <= 4 && strlen(wepKey[wepIndex-1]) > 0)
	{
		unsigned char szKey[64];
		int i;

		for (i=1; i<=4; i++)
		{
			memset(&erq, 0, sizeof(erq));
			erq.flags &= ~IW_ENCODE_DISABLED;
			erq.flags |= i;
			erq.length = hexkey2asckey(wepKey[i-1], szKey);
			config_wep_encode(vap, &erq, szKey);
		}

		//guilent
		memset(&erq, 0, sizeof(erq));
		erq.flags |= wepIndex;
		config_wep_encode(vap, &erq, NULL);
	}
	else
	{
		if (authType == AUTH_802_1X && encrypt != 0)
		{
			vap->iv_flags |= IEEE80211_F_PRIVACY;
		}
	}
}

static void config_acl(struct ieee80211vap *vap, struct if_acl_cfg_t *acl_cfg)
{
	int i;
	
	if(vap->iv_acl == NULL)
	{
		vap->iv_acl = ieee80211_aclator_get("mac");
		if(vap->iv_acl == NULL ||! vap->iv_acl->iac_attach(vap))
			return;
	}
	// free all
	vap->iv_acl->iac_flush(vap);
	if(acl_cfg->iu_acl_policy != IEEE80211_MACCMD_POLICY_OPEN)
	{
		for (i=0; i<TABLE_SIZE(acl_cfg->iu_acl_addr); i++)
		{
			if (memcmp(acl_cfg->iu_acl_addr[i], NULL_MAC, IEEE80211_ADDR_LEN) == 0)
				break;
			zdebug(ZCOM_DEBUG_CFG, "--- %s: add_acl=%s\n", __func__, ether_sprintf(acl_cfg->iu_acl_addr[i]));
			vap->iv_acl->iac_add(vap, acl_cfg->iu_acl_addr[i]);
		}
	}
	vap->iv_acl->iac_setpolicy(vap, acl_cfg->iu_acl_policy);
}

static void config_dev(struct net_device *dev, if_user_cfg_t *user_cfg)
{
	struct ieee80211com *ic = ((struct ieee80211vap *)dev->priv)->iv_ic;
	int is11n;

	is11n = (user_cfg->iu_wir_mode == ZCOM_WIRMODE_NA
	  	  || user_cfg->iu_wir_mode == ZCOM_WIRMODE_NA_ONLYN
	  	  || user_cfg->iu_wir_mode == ZCOM_WIRMODE_NG
	  	  || user_cfg->iu_wir_mode == ZCOM_WIRMODE_NG_ONLYN);

#ifdef _ZCOM_FOR_WDS_
	/* setup WDS flag */
	ic->ic_zcomwds_mode = ZCOMWDS_MODE_NONE;
	if (user_cfg->iu_opmode == ZCOM_OPMODE_P2P_AP)
		ic->ic_zcomwds_mode = ZCOMWDS_MODE_REPEATER;
	else if (user_cfg->iu_opmode == ZCOM_OPMODE_P2P)
		ic->ic_zcomwds_mode = ZCOMWDS_MODE_BRIDGE;
#ifdef _ZCOM_LAN2LAN_
	else if (user_cfg->iu_opmode == ZCOM_OPMODE_CLIENT && user_cfg->iu_lan2lan)
		ic->ic_zcomwds_mode = ZCOMWDS_MODE_LAN2LAN;
#endif
#endif

#ifdef _ZCOM_CHAN_HALF_
	/*Channel bandwidth*/
	switch (user_cfg->iu_chan_bandwidth)
	{
	case 1:
		ic->ic_chanbwflag = IEEE80211_CHAN_HALF;
		break;
	case 2:
		ic->ic_chanbwflag = IEEE80211_CHAN_QUARTER;
		break;
	default:
		ic->ic_chanbwflag = 0;
		break;
	}
	if (ic->ic_set_param)
		ic->ic_set_param(ic, ATH_PARAM_CHANBW, &user_cfg->iu_chan_bandwidth);
#endif
	/* link intergration */
	ic->ic_denyStaWithoutCable = user_cfg->iu_full_link ? 1: 0;
	/* beacon interval */
	ic->ic_lintval = user_cfg->iu_bcn_intvl;
	ic->ic_ignore_11dbeacon = user_cfg->iu_ignore11d;
	/* ctl power */
	if (user_cfg->iu_ctl_enable)
		ic->ic_flags_priv &= ~IEEE80211_PRIV_OUTDOOR;
	else
		ic->ic_flags_priv |= IEEE80211_PRIV_OUTDOOR;
	/*
	 * short preamble
	 */
	if(user_cfg->iu_preamble)
		ic->ic_caps |= IEEE80211_C_SHPREAMBLE;
	else
		ic->ic_caps &= ~IEEE80211_C_SHPREAMBLE;
	/*
	* cwm mode: channel mode
	*/
    if (user_cfg->iu_cwm_mode == IEEE80211_CWM_MODE20)
	{
        ic->ic_cwm.cw_width = IEEE80211_CWM_WIDTH20;
        ic->ic_cwm.cw_mode = IEEE80211_CWM_MODE20;
	    ic->ic_htcap &= ~IEEE80211_HTCAP_C_CHWIDTH40;
		ic->ic_flags_ext &= ~IEEE80211_FEXT_COEXT_DISABLE;
	}
	else
	{
        ic->ic_cwm.cw_width = IEEE80211_CWM_WIDTH40;
        ic->ic_cwm.cw_mode = IEEE80211_CWM_MODE2040;
	    ic->ic_htcap |= IEEE80211_HTCAP_C_CHWIDTH40;
		ic->ic_flags_ext |= IEEE80211_FEXT_COEXT_DISABLE;
	}
	/*
	 * shortgi enable or not
	 */
	ic->ic_htflags &= ~IEEE80211_HTF_SHORTGI;
	ic->ic_htcap &= ~ IEEE80211_HTCAP_C_SHORTGI40;
	ic->ic_htcap &= ~IEEE80211_HTCAP_C_SHORTGI20;
	if(user_cfg->iu_shortgi_enable)
	{
		if (ic->ic_cwm.cw_mode != IEEE80211_CWM_MODE20)
		{
			ic->ic_htcap |= IEEE80211_HTCAP_C_SHORTGI40;
			ic->ic_htflags |= IEEE80211_HTF_SHORTGI;
		}
		else if (is11n)
		{
			ic->ic_htcap |= IEEE80211_HTCAP_C_SHORTGI20;
		}
	}
	/*
	 * ampdu attribute
	 */
	if(user_cfg->iu_ampdu_enable)
	{
		ic->ic_flags_ext |= IEEE80211_FEXT_AMPDU;
		ic->ic_ampdu_limit = IEEE80211_AMPDU_LIMIT_MAX;
		ic->ic_ampdu_subframes = IEEE80211_AMPDU_SUBFRAME_DEFAULT;
	}
	else
	{
		ic->ic_flags_ext &= ~IEEE80211_FEXT_AMPDU;
	}
	/*
	 * amsdu attribute
	 */
	if(user_cfg->iu_amsdu_enable)
	{
		ic->ic_flags_ext |= IEEE80211_FEXT_AMSDU;
		ic->ic_amsdu_limit = IEEE80211_AMSDU_LIMIT_MAX;
		ic->ic_amsdu_enable(ic, 1);
	}
	else
	{
		ic->ic_flags_ext &= ~IEEE80211_FEXT_AMSDU;
		ic->ic_amsdu_enable(ic, 0);
	}
	/*
	 * channel protect and channel offset
	 */
	ic->ic_cwm.cw_extprotmode = user_cfg->iu_ext_protmode;
	/*
	 * ht prot
	 */
	if(!user_cfg->iu_htprot_enable)
		ic->ic_flags_ext &= ~IEEE80211_FEXT_HTPROT;
	else
		ic->ic_flags_ext |= IEEE80211_FEXT_HTPROT;
	ic->ic_roaming = IEEE80211_ROAMING_AUTO;	// fixed bug: sometime station was not do scan
	ic->ic_set_txPowerLimit(ic, 2 * user_cfg->iu_powerlevel, 2 * user_cfg->iu_powerlevel);
	if (ic->ic_set_param)
	{
#if 0
		ic->ic_set_param(ic, ATH_PARAM_ATH_TPSCALE, &user_cfg->iu_powerlevel);
#endif
		ic->ic_set_param(ic, ATH_PARAM_DISTANCE, &user_cfg->iu_distance);
		ic->ic_set_param(ic, ATH_PARAM_AMPDU, &user_cfg->iu_ampdu_enable);
		ic->ic_set_param(ic, ATH_PARAM_AMPDU_SUBFRAMES, &user_cfg->iu_ampdu_frames);
		ic->ic_set_param(ic, ATH_PARAM_AMPDU_LIMIT, &user_cfg->iu_ampdu_limit);
		ic->ic_set_param(ic, ATH_PARAM_ANTENNASW, &user_cfg->iu_antenna);
	}
}

static void config_vap(struct net_device *dev, 
		if_user_cfg_t *user_cfg,
		u_int8_t idx)
{
	struct ieee80211vap *vap = dev->priv;
	struct ieee80211com *ic = vap->iv_ic;
	struct if_vap_cfg_t *vap_cfg = &g_vap_cfg[idx];
	u_int8_t dst_wirmode = IEEE80211_MODE_AUTO;
	int i;

	zdebug(ZCOM_DEBUG_CFG, "%s dev=%s, index=%d\n", __func__, dev->name, idx);
	/* max stations number */
	vap->iv_max_aid = vap_cfg->iu_max_stanum + 1;
	/*
	 * rts, frag, bcn_intval, dtim
	 */
	vap->iv_rtsthreshold = user_cfg->iu_rts_thres;
	vap->iv_fragthreshold = user_cfg->iu_frag_thres;
	vap->iv_dtim_period = user_cfg->iu_dtim_intvl;
	vap->iv_mc_snoop_enable = vap_cfg->iu_igmpsnooping;
#ifdef ZCOM_TDM_COORDINATION			// add for STDM
	if(user_cfg->iu_opmode == ZCOM_OPMODE_AP)
		vap->iv_stdm = vap_cfg->iu_stdm;
	else
		vap->iv_stdm = IEEE80211_STDM_DISABLE;
#endif
#ifdef _ZCOM_TDM_WDS_
	if (user_cfg->iu_opmode == ZCOM_OPMODE_P2P
		|| user_cfg->iu_opmode == ZCOM_OPMODE_P2P_AP)
	{
		vap->iv_wds_stdm_mode = user_cfg->iu_wds_cfg.iu_wds_stdm_mode;
	}
	else
	{
		vap->iv_wds_stdm_mode = WDS_STDM_DISABLE;
	}
#endif
	vap->iv_wds_isolation = vap_cfg->iu_wds_isolation;
	vap->iv_bss->ni_vlan = vap_cfg->iu_vlanid;
	if (user_cfg->iu_wir_mode == ZCOM_WIRMODE_G)
		vap->iv_flags |= IEEE80211_F_PUREG;
	else
		vap->iv_flags &= ~IEEE80211_F_PUREG;

	if (user_cfg->iu_wir_mode == ZCOM_WIRMODE_NA_ONLYN
	  || user_cfg->iu_wir_mode == ZCOM_WIRMODE_NG_ONLYN)
		vap->iv_flags_ext |= IEEE80211_FEXT_PUREN;
	else
		vap->iv_flags_ext &= ~IEEE80211_FEXT_PUREN;

	if (vap_cfg->iu_scan11n_enable)
		vap->iv_zcom_priv_flags |=IEEE80211_PRIV_F_SCAN11N;
	else
		vap->iv_zcom_priv_flags &= ~IEEE80211_PRIV_F_SCAN11N;
	vap->iv_ht40_intolerant = (ic->ic_cwm.cw_mode == IEEE80211_CWM_MODE20) ? 1 : 0;
	vap->iv_chwidth = (ic->ic_cwm.cw_mode == IEEE80211_CWM_MODE20) ? 1 : 2;
	ic->ic_flags_ext |= IEEE80211_FEXT_HTUPDATE;

	/*
	* wireless mode
	*/
	config_wir_freq(vap, ic, 0);
	if (user_cfg->iu_opmode == ZCOM_OPMODE_CLIENT)
	{
		dst_wirmode = IEEE80211_MODE_AUTO;
	}
	else
	{
		switch(user_cfg->iu_wir_mode)
		{
		case ZCOM_WIRMODE_A:
			dst_wirmode = IEEE80211_MODE_11A;
			break;
		case ZCOM_WIRMODE_NA:
		case ZCOM_WIRMODE_NA_ONLYN:
			if(ic->ic_cwm.cw_mode == IEEE80211_CWM_MODE20)
			{
				dst_wirmode = IEEE80211_MODE_11NA_HT20;
			}
			else
			{
				if (user_cfg->iu_chan_offset == WIRELESS_CWM_OFFSET_UPPER)
					dst_wirmode = IEEE80211_MODE_11NA_HT40PLUS;
				else if (user_cfg->iu_chan_offset == WIRELESS_CWM_OFFSET_LOWER)
					dst_wirmode = IEEE80211_MODE_11NA_HT40MINUS;
				else
					dst_wirmode = IEEE80211_MODE_11NA_HT20;
			}
			break;
		case ZCOM_WIRMODE_BG:
			dst_wirmode = IEEE80211_MODE_11G;
			break;
		case ZCOM_WIRMODE_B:
			dst_wirmode = IEEE80211_MODE_11B;
			break;
		case ZCOM_WIRMODE_G:
			dst_wirmode = IEEE80211_MODE_11G;
			break;
		case ZCOM_WIRMODE_NG:
			case ZCOM_WIRMODE_NG_ONLYN:
			if(ic->ic_cwm.cw_mode == IEEE80211_CWM_MODE20)
			{
				dst_wirmode = IEEE80211_MODE_11NG_HT20;
			}
			else
			{
				if (user_cfg->iu_chan_offset == WIRELESS_CWM_OFFSET_UPPER)
					dst_wirmode = IEEE80211_MODE_11NG_HT40PLUS;
				else if (user_cfg->iu_chan_offset == WIRELESS_CWM_OFFSET_LOWER)
					dst_wirmode = IEEE80211_MODE_11NG_HT40MINUS;
				else
					dst_wirmode = IEEE80211_MODE_11NG_HT20;
			}
			break;
		default:
			break;
		}
		ic->ic_cwm.cw_extoffset = user_cfg->iu_chan_offset;
	}
	config_wir_mode(vap, dst_wirmode);

	/*
	 channel
	*/
	if(user_cfg->iu_channel == 0)
	{
		/* config !0 channel to effect the settings */
		for (i=0; i<ic->ic_nchans; i++)
		{
#ifdef _ZCOM_CHAN_HALF_
            const struct ieee80211_channel *c = &ic->ic_channels[i];
            if ( IEEE80211_CHAN_HALF == ic->ic_chanbwflag && !IEEE80211_IS_CHAN_HALF(c) ) {
                continue;
            }
    
    		if ( IEEE80211_CHAN_QUARTER == ic->ic_chanbwflag  && !IEEE80211_IS_CHAN_QUARTER(c)) {
                continue;
            }
    
            if ( 0 == ic->ic_chanbwflag && (IEEE80211_IS_CHAN_QUARTER(c) || IEEE80211_IS_CHAN_HALF(c))) {
                continue;
            }
#endif
			if (findchannel(ic, ic->ic_channels[i].ic_ieee, vap->iv_des_mode) != NULL)
			{
				config_wir_freq(vap, ic, ic->ic_channels[i].ic_ieee);
				zdebug(ZCOM_DEBUG_CFG, "%s, %d: as set auto chan, we chan %d to effect\n",
						__func__, __LINE__, ic->ic_channels[i].ic_ieee);
				break;
			}
		}
	}
	config_wir_freq(vap, ic, user_cfg->iu_channel);
	if (user_cfg->iu_opmode == ZCOM_OPMODE_CLIENT)
	{
		config_wir_freq(vap, ic, 0);
	}

	/* set fixed data rate, this should be deal with after wireless mode */
	if (user_cfg->iu_txrate == RATE_AUTO)
	{
		config_data_rate(vap->iv_dev, 0, 0);
	}
	else
	{
		config_data_rate(vap->iv_dev, 1, user_cfg->iu_txrate);
	}
	
	/* essid */
	if(!strcmp(vap_cfg->iu_ssid, "off") || !strcmp(vap_cfg->iu_ssid, "any"))
	{
		vap->iv_des_nssid = 0;
	}
	else
	{
		vap->iv_des_nssid = 1;
		vap->iv_des_ssid[0].len = strlen(vap_cfg->iu_ssid);
		memcpy(vap->iv_des_ssid[0].ssid, vap_cfg->iu_ssid, vap->iv_des_ssid[0].len);
	}
	if (user_cfg->iu_opmode == ZCOM_OPMODE_CLIENT)
	{
		switch(user_cfg->iu_wir_mode) 
		{
		case ZCOM_WIRMODE_A:
		case ZCOM_WIRMODE_NA:
		case ZCOM_WIRMODE_NA_ONLYN:
			vap->iv_zcom_priv_flags |= IEEE80211_PRIV_F_5G;
			vap->iv_zcom_priv_flags &= ~IEEE80211_PRIV_F_2G;
			break;
		case ZCOM_WIRMODE_B:
		case ZCOM_WIRMODE_G:
		case ZCOM_WIRMODE_BG:
		case ZCOM_WIRMODE_NG:
		case ZCOM_WIRMODE_NG_ONLYN:
			vap->iv_zcom_priv_flags |= IEEE80211_PRIV_F_2G;
			vap->iv_zcom_priv_flags &= ~IEEE80211_PRIV_F_5G;
			break;
		default:
			vap->iv_zcom_priv_flags &= ~IEEE80211_PRIV_F_5G;
			vap->iv_zcom_priv_flags &= ~IEEE80211_PRIV_F_2G;
			break;
		}
	}
	else if (user_cfg->iu_opmode == ZCOM_OPMODE_AP || user_cfg->iu_opmode == ZCOM_OPMODE_P2P_AP)
	{
		/* intra-bss */
#ifndef _ZCOM_FOR_WDS_
		if (vap_cfg->iu_intra_bss_enable == 1)
			vap->iv_flags |= IEEE80211_F_NOBRIDGE;
		else
			vap->iv_flags &= ~IEEE80211_F_NOBRIDGE;
#else
		if (vap_cfg->iu_intra_bss_enable == 0) {			//OFF
			vap->iv_zcom_priv_flags &= ~IEEE80211_PRIV_UNICAST_BRIDGE;
			vap->iv_zcom_priv_flags &= ~IEEE80211_PRIV_MULTICAST_BRIDGE;
			vap->iv_zcom_priv_flags &= ~IEEE80211_PRIV_BROADCAST_BRIDGE;
		} else if (vap_cfg->iu_intra_bss_enable == 1) { 	//Unicast
			vap->iv_zcom_priv_flags |= IEEE80211_PRIV_UNICAST_BRIDGE;
			vap->iv_zcom_priv_flags &= ~IEEE80211_PRIV_MULTICAST_BRIDGE;
			vap->iv_zcom_priv_flags &= ~IEEE80211_PRIV_BROADCAST_BRIDGE;
		} else if (vap_cfg->iu_intra_bss_enable == 2) { 	//Broadcast
			vap->iv_zcom_priv_flags &= ~IEEE80211_PRIV_UNICAST_BRIDGE;
			vap->iv_zcom_priv_flags &= ~IEEE80211_PRIV_MULTICAST_BRIDGE;
			vap->iv_zcom_priv_flags |= IEEE80211_PRIV_BROADCAST_BRIDGE;
		} else if (vap_cfg->iu_intra_bss_enable == 3) { 	//All packet
			vap->iv_zcom_priv_flags |= IEEE80211_PRIV_UNICAST_BRIDGE;
			vap->iv_zcom_priv_flags |= IEEE80211_PRIV_MULTICAST_BRIDGE;
			vap->iv_zcom_priv_flags |= IEEE80211_PRIV_BROADCAST_BRIDGE;
		} else if (vap_cfg->iu_intra_bss_enable == 4) { 	//Multicast
			vap->iv_zcom_priv_flags &= ~IEEE80211_PRIV_UNICAST_BRIDGE;
			vap->iv_zcom_priv_flags |= IEEE80211_PRIV_MULTICAST_BRIDGE;
			vap->iv_zcom_priv_flags &= ~IEEE80211_PRIV_BROADCAST_BRIDGE;
		}
#endif
	}
	if (user_cfg->iu_opmode != ZCOM_OPMODE_CLIENT)
	{
		/* hidden_ssid */
		if (vap_cfg->iu_hidden_ssid
#ifdef _ZCOM_FOR_WDS_
		|| user_cfg->iu_opmode == ZCOM_OPMODE_P2P
#endif
		)
			vap->iv_flags |= IEEE80211_F_HIDESSID;
		else
			vap->iv_flags &= ~IEEE80211_F_HIDESSID;
	}
	/* wmm */
	if (ic->ic_caps & IEEE80211_C_WME)
	{
		if(vap_cfg->iu_wmm_enable)
		{
			vap->iv_flags |= IEEE80211_F_WME;
			ic->ic_flags |= IEEE80211_F_WME;
		}
		else
		{
			vap->iv_flags &= ~IEEE80211_F_WME;
			ic->ic_flags &= ~IEEE80211_F_WME;
		}
	}
	/*
	* acl policy
	*/
	config_acl(vap, &vap_cfg->iu_acl_cfg);

	/* auth_type */
	config_auth(vap, vap_cfg->iu_auth_type, vap_cfg->iu_encrypt, vap_cfg->iu_wep_def_no, user_cfg->iu_wep_key);
}
static void config_radiotest(struct ieee80211vap* vap,
		if_user_cfg_t *user_cfg)
{
	vap->iv_radiotest_enable = user_cfg->iu_radiotest_enable;
	vap->iv_radiotest_count= 0;
	memcpy(vap->iv_radiotest_myaddr ,user_cfg->iu_radiotest_mac,IEEE80211_ADDR_LEN);
	vap->iv_radiotest_total= user_cfg->iu_radiotest_total;
	vap->iv_radiotest_rssiavg = 0;
}

#ifdef _ZCOM_INFO_
static void config_info(struct net_device *dev, struct if_info_cfg_t *info_cfg)
{
	struct ieee80211com *ic = ((struct ieee80211vap *)dev->priv)->iv_ic;

	ic->zcominfo_ip = info_cfg->iu_ip;
	strcpy(ic->zcominfo_name, info_cfg->iu_name);
	strcpy(ic->zcominfo_version, info_cfg->iu_ver);
}
#endif

#ifdef _ZCOM_FOR_WDS_
static void config_wds(struct net_device *dev, if_user_cfg_t *user_cfg)
{
	struct if_wds_cfg_t *wds_cfg = &user_cfg->iu_wds_cfg;
	struct ieee80211vap *vap = dev->priv;
	struct ieee80211com *ic = vap->iv_ic;
	struct ieee80211_node *ni = NULL;
	int i = 0, is_11n;

	zdebug(ZCOM_DEBUG_CFG, "== %s, %d, enter\n", __func__, __LINE__);
	if( user_cfg->iu_opmode != ZCOM_OPMODE_P2P
	 && user_cfg->iu_opmode != ZCOM_OPMODE_P2P_AP)
	 	return;
	if (ic->wds_handle == NULL)
		return;

	is_11n = (( g_dev_cfg.iu_wir_mode == ZCOM_WIRMODE_NA
			|| g_dev_cfg.iu_wir_mode == ZCOM_WIRMODE_NG
			|| g_dev_cfg.iu_wir_mode == ZCOM_WIRMODE_NA_ONLYN
			|| g_dev_cfg.iu_wir_mode == ZCOM_WIRMODE_NG_ONLYN)
			&& wds_cfg->iu_wds_auth != ENCRYPT_WEP
			&& wds_cfg->iu_wds_auth != ENCRYPT_TKIP);
	zdebug(ZCOM_DEBUG_CFG, "== %s, %d, do\n", __func__, __LINE__);
	if (user_cfg->iu_opmode == ZCOM_OPMODE_P2P)
	{
		if( ic->ic_cwm.cw_mode == IEEE80211_CWM_MODE20)
			ic->ic_bss_to20(ic);
		else
			ic->ic_bss_to40(ic);
	}

	for (i = 0; i < MAX_WDS_NUM; i++)
	{
		if (0 == memcmp(wds_cfg->iu_wds_addr[i], NULL_MAC, IEEE80211_ADDR_LEN))
			continue;

		ni = alloc_wds_node(vap, wds_cfg->iu_wds_addr[i], is_11n);
		ic->wds_handle->iv_wds_add_assoc_node(wds_cfg->iu_wds_addr[i], ni);
		zdebug(ZCOM_DEBUG_CFG, "--- %s: add_wds=%s\n", __func__, ether_sprintf(wds_cfg->iu_wds_addr[i]));
	}

	//Group Key set
	if(wds_cfg->iu_wds_auth == ENCRYPT_TKIP || wds_cfg->iu_wds_auth == ENCRYPT_AES)
	{
		unsigned char szKey[16];
		struct iw_point erq = {0};

		erq.flags &= ~IW_ENCODE_DISABLED;
		erq.length = hexkey2asckey("0123456789", szKey);
		erq.flags |= 1;
		config_wep_encode(vap, &erq, szKey);
		//Unicast Key set
		if(wds_cfg->iu_wds_auth == ENCRYPT_TKIP)
		{
			ic->wds_handle->wds_set_wds_psk(vap, IEEE80211_CIPHER_TKIP, 32, wds_cfg->iu_wds_key);
		}
		else if(wds_cfg->iu_wds_auth == ENCRYPT_AES)
		{
			ic->wds_handle->wds_set_wds_psk(vap, IEEE80211_CIPHER_AES_CCM, 16, wds_cfg->iu_wds_key);
		}
	}
}
#endif

int if_zcom_update(struct net_device *dev, u_int8_t code)
{
	struct ieee80211vap *vap = dev->priv;
	struct ieee80211com *ic  = vap->iv_ic;

	zdebug(ZCOM_DEBUG_CFG, "== %s, %d, code=%d\n", __func__, __LINE__, code);

	if (vap->iv_unit != 0)
		return -1;
	if (read_config(ic, IF_ZCOM_FILE, &g_user_cfg) < 0)
		return -1;

	if (code == IF_UPDATE_WIRELESS_SETTINGS)
	{
		TAILQ_FOREACH(vap, &(ic)->ic_vaps, iv_next)
		{
			vap->iv_zcom_priv_flags |= IEEE80211_PRIV_F_UPDATE;
		}
	}
#ifdef _ZCOM_INFO_
	// update the CDP info which neednot sync with wireless settings
	if (code == IF_UPDATE_ZCOM_INFO)
	{
		config_info(dev, &g_dev_cfg.iu_info_cfg);
	}
#endif
	return 0;
}

void if_zcom_open(struct net_device *dev)
{
	struct ieee80211vap *vap = dev->priv;
	zdebug(ZCOM_DEBUG_CFG, "%s dev=%s, index=%d\n", __func__, dev->name, vap->iv_unit);

	if (vap->iv_unit >= MAX_VAP_NUM)
		return;
	if (vap->iv_zcom_priv_flags & IEEE80211_PRIV_F_UPDATE)
	{
		zdebug(ZCOM_DEBUG_CFG, "== %s, %d, do\n", __func__, __LINE__);
		config_dev(dev, &g_dev_cfg);
		config_vap(dev, &g_dev_cfg, vap->iv_unit);
#ifdef _ZCOM_FOR_WDS_
		if (vap->iv_unit == 0
			&&( g_dev_cfg.iu_opmode == ZCOM_OPMODE_P2P
			 || g_dev_cfg.iu_opmode == ZCOM_OPMODE_P2P_AP) )
		{
			config_wds(dev, &g_dev_cfg);
		}
#endif
#ifdef _ZCOM_INFO_
		config_info(dev, &g_dev_cfg.iu_info_cfg);
#endif
		config_radiotest(vap, &g_dev_cfg);
		vap->iv_zcom_priv_flags &= ~IEEE80211_PRIV_F_UPDATE;
	}
#ifdef _ZCOM_FOR_WDS_
	if (vap->iv_unit == 0
		&&( g_dev_cfg.iu_opmode == ZCOM_OPMODE_P2P
		 || g_dev_cfg.iu_opmode == ZCOM_OPMODE_P2P_AP) )
		add_wds_timer(vap);
#endif
#ifdef _ZCOM_INFO_
	add_zcominfo_timer(vap);
#endif
	zdebug(ZCOM_DEBUG_CFG, "== %s, %d, exit\n", __func__, __LINE__);
	dev->flags |= IFF_UP;
}

/*
 * Bridge implementation depends on existing of ath0!!!
 */
void if_zcom_stop(struct net_device *dev)
{
	struct ieee80211vap *vap = dev->priv;
#ifdef _ZCOM_FOR_WDS_
	struct ieee80211com *ic  = vap->iv_ic;
#endif

	zdebug(ZCOM_DEBUG_CFG, "%s dev=%s, index=%d\n", __func__, dev->name, vap->iv_unit);
	if (vap->iv_unit >= MAX_VAP_NUM)
		return;
#ifdef _ZCOM_INFO_
	del_zcominfo_timer(vap);
#endif
#ifdef _ZCOM_FOR_WDS_
	if (vap->iv_unit == 0)
	{
		del_wds_timer(vap);

#ifdef _ZCOM_TDM_WDS_
		extern int bTimerIRQRegistered;
		if (bTimerIRQRegistered)
		{
			StopPlatformTimer();
		}
#endif

		if (ic->wds_handle != NULL && (vap->iv_zcom_priv_flags & IEEE80211_PRIV_F_UPDATE))
		{
			int i, wdsNodeNum = 0;
			struct ieee80211_node *ni_wds = NULL;
			int wdsNum = ic->wds_handle->iv_wds_get_assocnodenum();

			// be care: del assoc assoced wds node would cause wds assoc node list refresh
			// we del from last one to avoid the refresh
			for (i = wdsNum-1; i >= 0; i--)
			{
				ni_wds = ic->wds_handle->iv_wds_get_assocnodelist()[i];
				if (ni_wds != NULL && ZCOMWDS_NODE_IS_WDS(ni_wds))
				{
					wdsNodeNum++;
					ic->wds_handle->iv_wds_del_assoc_node((u_int8_t *)ni_wds->ni_macaddr);
					ni_wds->ni_zcomwds_mode = ZCOMWDS_MODE_NONE;
					ieee80211_crypto_delkey(ni_wds->ni_vap, &ni_wds->ni_ucastkey, ni_wds);
					ieee80211_node_leave(ni_wds);
					ieee80211_free_node(ni_wds);
				}
			}
			vap->iv_sta_assoc = 0;	//-= wdsNodeNum;
			ic->ic_sta_assoc  -= wdsNodeNum;
		}
	}
#endif
#if 0	// we do destroy/create athx in configserver
	zdebug(ZCOM_DEBUG_CFG, "%s %d\n", __func__, __LINE__);
	if(!(vap->iv_zcom_priv_flags & IEEE80211_PRIV_F_UPDATE)
	  || vap->iv_opmode == IEEE80211_M_MONITOR
	  || vap->iv_unit != 0)
		return;

	struct if_vap_cfg_t *vap_cfg = &g_vap_cfg[vap->iv_unit];
	u_int16_t new_opmode = g_dev_cfg.iu_opmode;
	u_int16_t priv_flags, create_flag = 0;
	const u_int16_t opmode[] = {
		IEEE80211_M_HOSTAP, 			//hostap
		IEEE80211_M_STA,				//sta
#ifdef _ZCOM_FOR_WDS_
		IEEE80211_M_HOSTAP,				//bridge
		IEEE80211_M_HOSTAP, 			//repeater
#endif
	};

	// destroy / create athx
	if(ic->ic_country.countryCode != g_dev_cfg.iu_domain
	|| vap->iv_opmode != opmode[new_opmode])
	{
		priv_flags = vap->iv_zcom_priv_flags;

		zdebug(ZCOM_DEBUG_CFG, "%s, %d: delete vap\n", __func__, __LINE__);
		if (dev->flags & IFF_RUNNING)
			ieee80211_stop(vap->iv_dev);
		dev->flags &= ~IFF_UP;
		ic->ic_vap_delete(vap);

		/* set country */
		if(ic->ic_country.countryCode != g_dev_cfg.iu_domain)
		{
			if(!ic->ic_set_country(ic, NULL, g_dev_cfg.iu_domain))
				ic->ic_get_currentCountry(ic, &ic->ic_country);
		}

		zdebug(ZCOM_DEBUG_CFG, "%s, %d: create new vap \n", __func__, __LINE__);
		create_flag = IEEE80211_CLONE_BSSID;
		if (new_opmode == ZCOM_OPMODE_CLIENT
#ifdef _ZCOM_FOR_WDS_
		  || new_opmode == ZCOM_OPMODE_P2P
#endif
		  )
			create_flag |= IEEE80211_NO_STABEACONS;

		vap = ic->ic_vap_create(ic, vap_cfg->iu_vap_name, vap->iv_unit, opmode[new_opmode], create_flag, NULL);
		KASSERT(vap, ("%s assert: fail to create vap(%s), mode %d flag 0x%x\n",
					__func__, vap_cfg->iu_vap_name, opmode[new_opmode], create_flag));
		vap->iv_zcom_priv_flags = priv_flags;
	}
#endif
	zdebug(ZCOM_DEBUG_CFG, "== %s, %d, exit\n", __func__, __LINE__);
}

EXPORT_SYMBOL(if_zcom_update);
EXPORT_SYMBOL(if_zcom_open);
EXPORT_SYMBOL(if_zcom_stop);

