/**************************************************************************/
/*                                                                        */
/*                                                                        */
/*           Copyright 1998 - 2001,                                       */
/*             Intoto, Inc, SANTACLARA, CA, USA                           */
/*                      ALL RIGHTS RESERVED                               */
/*                                                                        */
/*   Permission is hereby granted to licensees of Intoto,                 */
/*   Inc. products to use or abstract this computer program for the       */
/*   sole purpose of implementing a product based on                      */
/*   Intoto, Inc. products.No other rights to reproduce,                  */
/*   use,or disseminate this computer program, whether in part or in      */
/*   whole, are granted.                                                  */
/*                                                                        */
/*   Intoto, Inc. makes no representation or warranties                   */
/*   with respect to the performance of this computer program, and        */
/*   specifically disclaims any responsibility for any damages,           */
/*   special or consequential, connected with the use of this program.    */
/*                                                                        */
/************************************************************************ */
/************************************************************************/
/*  File         : hmacsha1.c                                           */
/*                                                                      */
/*  Description  :                                                      */
/*                                                                      */
/*  Version      Date      Author      Change Description               */
/*  -------    --------    ------    ----------------------             */
/*                                                                      */
/************************************************************************/
//nclude <string.h>

#include "net80211/igwsha1.h"

/*
 *  Define the SHA1 circular left shift macro
 */
#define SHA1CircularShift(bits,word) \
                (((word) << (bits)) | ((word) >> (32-(bits))))

/* Local Function Prototyptes */
LOCAL T_VOID SHA1PadMessage(SHA1Context *);
LOCAL T_VOID SHA1ProcessMessageBlock(SHA1Context *);
LOCAL T_INT32 SHA1Reset(SHA1Context *context);
LOCAL T_INT32 SHA1Result(SHA1Context *context, T_UINT8 Message_Digest[IGW_SHA1HASH_SIZE]);
LOCAL T_INT32 SHA1Input(SHA1Context *context, const T_UINT8 *message_array, T_UINT32 length);
/*
 *  SHA1Reset
 *
 *  Description:
 *      This function will initialize the SHA1Context in preparation
 *      for computing a new SHA1 message digest.
 *
 *  Parameters:
 *      context: [in/out]
 *          The context to reset.
 *
 *  Returns:
 *      sha Error Code.
 *
 */
LOCAL T_INT32 
SHA1Reset(SHA1Context *context)
{
    if (!context)
    {
        return shaNull;
    }

    context->Length_Low             = 0;
    context->Length_High            = 0;
    context->Message_Block_Index    = 0;

    context->Intermediate_Hash[0]   = 0x67452301;
    context->Intermediate_Hash[1]   = 0xEFCDAB89;
    context->Intermediate_Hash[2]   = 0x98BADCFE;
    context->Intermediate_Hash[3]   = 0x10325476;
    context->Intermediate_Hash[4]   = 0xC3D2E1F0;

    context->Computed   = 0;
    context->Corrupted  = 0;

    return shaSuccess;
}

/*
 *  SHA1Result
 *
 *  Description:
 *      This function will return the 160-bit message digest into the
 *      Message_Digest array  provided by the caller.
 *      NOTE: The first octet of hash is stored in the 0th element,
 *            the last octet of hash in the 19th element.
 *
 *  Parameters:
 *      context: [in/out]
 *          The context to use to calculate the SHA-1 hash.
 *      Message_Digest: [out]
 *          Where the digest is returned.
 *
 *  Returns:
 *      sha Error Code.
 *
 */
LOCAL T_INT32
SHA1Result(SHA1Context *context, T_UINT8 Message_Digest[IGW_SHA1HASH_SIZE])
{
    T_INT32 i;

    if (!context || !Message_Digest)
    {
        return shaNull;
    }

    if (context->Corrupted)
    {
        return context->Corrupted;
    }

    if (!context->Computed)
    {
        SHA1PadMessage(context);
        for(i=0; i<64; ++i)
        {
            /* message may be sensitive, clear it out */
            context->Message_Block[i] = 0;
        }
        context->Length_Low = 0;    /* and clear length */
        context->Length_High = 0;
        context->Computed = 1;
    }

    for(i = 0; i < IGW_SHA1HASH_SIZE; ++i)
    {
        Message_Digest[i] = context->
	    Intermediate_Hash[i>>2] >> 8 * ( 3 - ( i & 0x03 ) );
    }

    return shaSuccess;
}

/*
 *  SHA1Input
 *
 *  Description:
 *      This function accepts an array of octets as the next portion
 *      of the message.
 *
 *  Parameters:
 *      context: [in/out]
 *          The SHA context to update
 *      message_array: [in]
 *          An array of characters representing the next portion of
 *          the message.
 *      length: [in]
 *          The length of the message in message_array
 *
 *  Returns:
 *      sha Error Code.
 *
 */
LOCAL T_INT32 
SHA1Input(SHA1Context *context, const T_UINT8 *message_array, T_UINT32 length)
{
    if (!length)
    {
        return shaSuccess;
    }

    if (!context || !message_array)
    {
        return shaNull;
    }

    if (context->Computed)
    {
        context->Corrupted = shaStateError;
        return shaStateError;
    }

    if (context->Corrupted)
    {
        return context->Corrupted;
    }

    while(length-- && !context->Corrupted)
    {
        context->Message_Block[context->Message_Block_Index++] =
            (*message_array & 0xFF);

        context->Length_Low += 8;
        if (context->Length_Low == 0)
        {
            context->Length_High++;
            if (context->Length_High == 0)
            {
                /* Message is too long */
                context->Corrupted = 1;
            }
        }

        if (context->Message_Block_Index == 64)
        {
            SHA1ProcessMessageBlock(context);
        }

        message_array++;
    }

    return shaSuccess;
}

/*
 *  SHA1ProcessMessageBlock
 *
 *  Description:
 *      This function will process the next 512 bits of the message
 *      stored in the Message_Block array.
 *
 *  Parameters:
 *      None.
 *
 *  Returns:
 *      Nothing.
 *
 *  Comments:
 *      Many of the variable names in this code, especially the
 *      single character names, were used because those were the
 *      names used in the publication.
 *
 *
 */
LOCAL T_VOID
SHA1ProcessMessageBlock(SHA1Context *context)
{
    const T_UINT32 K[] =    {       /* Constants defined in SHA-1   */
        0x5A827999,
        0x6ED9EBA1,
        0x8F1BBCDC,
        0xCA62C1D6
    };
    T_INT32           t;                 /* Loop counter                */
    T_UINT32      temp;              /* Temporary word value        */
    T_UINT32      W[80];             /* Word sequence               */
    T_UINT32      A, B, C, D, E;     /* Word buffers                */

    /*
     *  Initialize the first 16 words in the array W
     */
    for (t = 0; t < 16; t++)
    {
        W[t] = context->Message_Block[t * 4] << 24;
        W[t] |= context->Message_Block[t * 4 + 1] << 16;
        W[t] |= context->Message_Block[t * 4 + 2] << 8;
        W[t] |= context->Message_Block[t * 4 + 3];
    }

    for (t = 16; t < 80; t++)
    {
        W[t] = SHA1CircularShift(1,W[t-3] ^ W[t-8] ^ W[t-14] ^ W[t-16]);
    }

    A = context->Intermediate_Hash[0];
    B = context->Intermediate_Hash[1];
    C = context->Intermediate_Hash[2];
    D = context->Intermediate_Hash[3];
    E = context->Intermediate_Hash[4];

    for (t = 0; t < 20; t++)
    {
        temp =  SHA1CircularShift(5,A) +
            ((B & C) | ((~B) & D)) + E + W[t] + K[0];
        E = D;
        D = C;
        C = SHA1CircularShift(30,B);
        B = A;
        A = temp;
    }

    for (t = 20; t < 40; t++)
    {
        temp = SHA1CircularShift(5,A) + (B ^ C ^ D) + E + W[t] + K[1];
        E = D;
        D = C;
        C = SHA1CircularShift(30,B);
        B = A;
        A = temp;
    }

    for (t = 40; t < 60; t++)
    {
        temp = SHA1CircularShift(5,A) +
            ((B & C) | (B & D) | (C & D)) + E + W[t] + K[2];
        E = D;
        D = C;
        C = SHA1CircularShift(30,B);
        B = A;
        A = temp;
    }

    for (t = 60; t < 80; t++)
    {
        temp = SHA1CircularShift(5,A) + (B ^ C ^ D) + E + W[t] + K[3];
        E = D;
        D = C;
        C = SHA1CircularShift(30,B);
        B = A;
        A = temp;
    }

    context->Intermediate_Hash[0] += A;
    context->Intermediate_Hash[1] += B;
    context->Intermediate_Hash[2] += C;
    context->Intermediate_Hash[3] += D;
    context->Intermediate_Hash[4] += E;

    context->Message_Block_Index = 0;
}


/*
 *  SHA1PadMessage
 *
 *  Description:
 *      According to the standard, the message must be padded to an even
 *      512 bits.  The first padding bit must be a '1'.  The last 64
 *      bits represent the length of the original message.  All bits in
 *      between should be 0.  This function will pad the message
 *      according to those rules by filling the Message_Block array
 *      accordingly.  It will also call the ProcessMessageBlock function
 *      provided appropriately.  When it returns, it can be assumed that
 *      the message digest has been computed.
 *
 *  Parameters:
 *      context: [in/out]
 *          The context to pad
 *      ProcessMessageBlock: [in]
 *          The appropriate SHA*ProcessMessageBlock function
 *  Returns:
 *      Nothing.
 *
 */

LOCAL T_VOID 
SHA1PadMessage(SHA1Context *context)
{
    /*
     *  Check to see if the current message block is too small to hold
     *  the initial padding bits and length.  If so, we will pad the
     *  block, process it, and then continue padding into a second
     *  block.
     */
    if (context->Message_Block_Index > 55)
    {
        context->Message_Block[context->Message_Block_Index++] = 0x80;
        while (context->Message_Block_Index < 64)
        {
            context->Message_Block[context->Message_Block_Index++] = 0;
        }

        SHA1ProcessMessageBlock(context);

        while (context->Message_Block_Index < 56)
        {
            context->Message_Block[context->Message_Block_Index++] = 0;
        }
    }
    else
    {
        context->Message_Block[context->Message_Block_Index++] = 0x80;
        while (context->Message_Block_Index < 56)
        {
            context->Message_Block[context->Message_Block_Index++] = 0;
        }
    }

    /*
     *  Store the message length as the last 8 octets
     */
    context->Message_Block[56] = context->Length_High >> 24;
    context->Message_Block[57] = context->Length_High >> 16;
    context->Message_Block[58] = context->Length_High >> 8;
    context->Message_Block[59] = context->Length_High;
    context->Message_Block[60] = context->Length_Low >> 24;
    context->Message_Block[61] = context->Length_Low >> 16;
    context->Message_Block[62] = context->Length_Low >> 8;
    context->Message_Block[63] = context->Length_Low;

    SHA1ProcessMessageBlock(context);
}

LOCAL T_INT32
IGWSHAInit(SHA1Context *ctx)
{
    return SHA1Reset(ctx);
}

LOCAL T_INT32
IGWSHAUpdate(SHA1Context *ctx, const T_UINT8 *in, T_UINT32 len)
{
    return SHA1Input(ctx, in, len);
}

LOCAL T_INT32
IGWSHAFinal(SHA1Context *ctx, T_UINT8 digest[])
{
    return SHA1Result(ctx, digest);
}

T_VOID IGWHMACSHA1(T_UCHAR8 *text, T_INT32 text_len,
                   T_UCHAR8 *key, T_INT32 key_len,
                   T_UCHAR8 *digest)
{
    SHA1Context context;
    T_UCHAR8 k_ipad[65]; /* inner padding - key XORd with ipad */
    T_UCHAR8 k_opad[65]; /* outer padding - key XORd with opad */
    T_INT32 i;

    /* if key is longer than 64 bytes reset it to key=SHA1(key) */
    if (key_len > 64) {
        SHA1Context tctx;

        IGWSHAInit(&tctx);
        IGWSHAUpdate(&tctx, key, (T_UINT32)key_len);
        IGWSHAFinal(&tctx, key);

        key_len = 20;
    }

    /*
     * the HMAC_SHA1 transform looks like:
     *
     * SHA1(K XOR opad, SHA1(K XOR ipad, text))
     *
     * where K is an n byte key
     * ipad is the byte 0x36 repeated 64 times
     * opad is the byte 0x5c repeated 64 times
     * and text is the data being protected
     */

    /* start out by storing key in pads */
    memset(k_ipad, 0, sizeof k_ipad);
    memset(k_opad, 0, sizeof k_opad);
    memcpy(k_ipad, key, key_len);
    memcpy(k_opad, key, key_len);

    /* XOR key with ipad and opad values */
    for (i = 0; i < 64; i++) {
        k_ipad[i] ^= 0x36;
        k_opad[i] ^= 0x5c;
    }

    /* perform inner SHA1 */
    IGWSHAInit(&context); /* init context for 1st pass */
    IGWSHAUpdate(&context, k_ipad, 64); /* start with inner pad */
    IGWSHAUpdate(&context, text, text_len); /* then text of datagram */
    IGWSHAFinal(&context, digest); /* finish up 1st pass */

    /* perform outer SHA1 */
    IGWSHAInit(&context); /* init context for 2nd pass */
    IGWSHAUpdate(&context, k_opad, 64); /* start with outer pad */
    IGWSHAUpdate(&context, digest, 20); /* then results of 1st hash */
    IGWSHAFinal(&context, digest); /* finish up 2nd pass */
}
