/*
 * WPA Supplicant / main() function for UNIX like OSes and MinGW
 * Copyright (c) 2003-2005, Jouni Malinen <jkmaline@cc.hut.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * Alternatively, this software may be distributed under the terms of BSD
 * license.
 *
 * See README and COPYING for more details.
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>

#include "common.h"
#include "wpa_supplicant_i.h"

#define CERT_VERIFY_FILE	"/tmp/cert_verify"

extern const char *wpa_supplicant_version;
extern const char *wpa_supplicant_license;
#ifndef CONFIG_NO_STDOUT_DEBUG
extern const char *wpa_supplicant_full_license;
#endif /* CONFIG_NO_STDOUT_DEBUG */

extern struct wpa_driver_ops *wpa_supplicant_drivers[];


static void usage(void)
{
#ifndef CONFIG_NO_STDOUT_DEBUG
	int i;
	printf("%s\n\n%s\n"
	       "usage:\n"
	       "  wpa_supplicant [-BddehLqqvwW] [-P<pid file>] "
	       "[-g<global ctrl>] \\\n"
	       "        -i<ifname> -c<config file> [-C<ctrl>] [-D<driver>] "
	       "[-p<driver_param>] \\\n"
	       "        [-N -i<ifname> -c<conf> [-C<ctrl>] [-D<driver>] "
	       "[-p<driver_param>] ...]\n"
	       "\n"
	       "drivers:\n",
	       wpa_supplicant_version, wpa_supplicant_license);

	for (i = 0; wpa_supplicant_drivers[i]; i++) {
		printf("  %s = %s\n",
		       wpa_supplicant_drivers[i]->name,
		       wpa_supplicant_drivers[i]->desc);
	}

	printf("options:\n"
	       "  -B = run daemon in the background\n"
	       "  -c = Configuration file\n"
	       "  -C = ctrl_interface parameter (only used if -c is not)\n"
	       "  -i = interface name\n"
	       "  -d = increase debugging verbosity (-dd even more)\n"
	       "  -D = driver name\n"
	       "  -g = global ctrl_interace\n"
	       "  -K = include keys (passwords, etc.) in debug output\n"
	       "  -t = include timestamp in debug messages\n"
	       "  -h = show this help text\n"
	       "  -L = show license (GPL and BSD)\n"
	       "  -p = driver parameters\n"
	       "  -P = PID file\n"
	       "  -q = decrease debugging verbosity (-qq even less)\n"
	       "  -v = show version\n"
	       "  -w = wait for interface to be added, if needed\n"
	       "  -W = wait for a control interface monitor before starting\n"
	       "  -N = start describing new interface\n");

	printf("example:\n"
	       "  wpa_supplicant -Dwext -iwlan0 -c/tmp/wpa_supplicant.conf\n");
#endif /* CONFIG_NO_STDOUT_DEBUG */
}


static void license(void)
{
#ifndef CONFIG_NO_STDOUT_DEBUG
	printf("%s\n\n%s\n",
	       wpa_supplicant_version, wpa_supplicant_full_license);
#endif /* CONFIG_NO_STDOUT_DEBUG */
}


static void wpa_supplicant_fd_workaround(void)
{
	int s, i;
	/* When started from pcmcia-cs scripts, wpa_supplicant might start with
	 * fd 0, 1, and 2 closed. This will cause some issues because many
	 * places in wpa_supplicant are still printing out to stdout. As a
	 * workaround, make sure that fd's 0, 1, and 2 are not used for other
	 * sockets. */
	for (i = 0; i < 3; i++) {
		s = open("/dev/null", O_RDWR);
		if (s > 2) {
			close(s);
			break;
		}
	}
}

#ifdef _ZCOM_NANJING_

#include <openssl/ssl.h>
#include <openssl/err.h>
#include <openssl/pkcs12.h>
#include <openssl/x509v3.h>

//#include <openssl/e_os.h>
//#include <openssl/ui.h>
//#include <openssl/apps.h>
#include <openssl/ui.h>
#include <openssl/ossl_typ.h>
#include "e_os.h"
#include "apps.h"
#include "ui_locl.h"

const EVP_CIPHER *enc;
BIO *bio_out=NULL;

#ifndef CRYPTO_MDEBUG /* avoid duplicate #define */
#define CRYPTO_MDEBUG
#endif

#define FORMAT_UNDEF    0
#define FORMAT_ASN1     1
#define FORMAT_TEXT     2
#define FORMAT_PEM      3
#define FORMAT_NETSCAPE 4
#define FORMAT_PKCS12   5
#define FORMAT_SMIME    6
#define FORMAT_ENGINE   7
#define FORMAT_IISSGC	8	/* XXX this stupid macro helps us to avoid
				 * adding yet another param to load_*key() */


static UI_METHOD *ui_method = NULL;

int get_cert_chain (X509 *cert, X509_STORE *store, STACK_OF(X509) **chain);
int dump_certs_keys_p12(BIO *out, PKCS12 *p12, char *pass, int passlen, int options, char *pempass);
int dump_certs_pkeys_bags(BIO *out, STACK_OF(PKCS12_SAFEBAG) *bags, char *pass,
			  int passlen, int options, char *pempass);
int dump_certs_pkeys_bag(BIO *out, PKCS12_SAFEBAG *bags, char *pass, int passlen, int options, char *pempass);
int print_attribs(BIO *out, STACK_OF(X509_ATTRIBUTE) *attrlst,const char *name);
void hex_prin(BIO *out, unsigned char *buf, int len);
int alg_print(BIO *x, X509_ALGOR *alg);

#define NOKEYS		0x1
#define NOCERTS 	0x2
#define INFO		0x4
#define CLCERTS		0x8
#define CACERTS		0x10
int dump_cert_text (BIO *out, X509 *x)
{
	char *p;

	p=X509_NAME_oneline(X509_get_subject_name(x),NULL,0);
	BIO_puts(out,"subject=");
	BIO_puts(out,p);
	OPENSSL_free(p);

	p=X509_NAME_oneline(X509_get_issuer_name(x),NULL,0);
	BIO_puts(out,"\nissuer=");
	BIO_puts(out,p);
	BIO_puts(out,"\n");
	OPENSSL_free(p);

	return 0;
}

int dump_certs_keys_p12 (BIO *out, PKCS12 *p12, char *pass,
	     int passlen, int options, char *pempass)
{
	STACK_OF(PKCS7) *asafes = NULL;
	STACK_OF(PKCS12_SAFEBAG) *bags;
	int i, bagnid;
	int ret = 0;
	PKCS7 *p7;

	if (!( asafes = PKCS12_unpack_authsafes(p12))) 
	{
		printf ("%s %d:PKCS12_unpack_authsafes failed!\n", __FILE__, __LINE__);
		return 0;
	}

	guilent_printf("%s %d: sk_PKCS7_num (asafes) =%d \n" , __FILE__, __LINE__, sk_PKCS7_num (asafes));
	for (i = 0; i < sk_PKCS7_num (asafes); i++) {
		p7 = sk_PKCS7_value (asafes, i);
		bagnid = OBJ_obj2nid (p7->type);
		
		guilent_printf("%s %d: bagnid =%d \n" , __FILE__, __LINE__, bagnid);
		if (bagnid == NID_pkcs7_data) {
			bags = PKCS12_unpack_p7data(p7);
			BIO_printf (bio_err, "PKCS7 Data\n");
			guilent_printf ("PKCS7 Data\n");
		} 
		else if (bagnid == NID_pkcs7_encrypted) 
		{
			guilent_printf ("PKCS7 Encrypted data: \n");
			bags = PKCS12_unpack_p7encdata(p7, pass, passlen);
		} 
		else 
			continue;
		
		if (!bags) 
		{
			printf ("%s %d:bags is NULL!\n", __FILE__, __LINE__);
			goto err;
		}
		guilent_printf("%s %d: bags =%X \n" , __FILE__, __LINE__, bags);
	    	if (!dump_certs_pkeys_bags (out, bags, pass, passlen, 
						 options, pempass)) {
			sk_PKCS12_SAFEBAG_pop_free (bags, PKCS12_SAFEBAG_free);
			printf ("%s %d:dump_certs_pkeys_bags failed.\n", __FILE__, __LINE__);
			goto err;
		}
		sk_PKCS12_SAFEBAG_pop_free (bags, PKCS12_SAFEBAG_free);
		bags = NULL;
	}
	ret = 1;

	err:

	if (asafes)
		sk_PKCS7_pop_free (asafes, PKCS7_free);
	return ret;
}

int dump_certs_pkeys_bags (BIO *out, STACK_OF(PKCS12_SAFEBAG) *bags,
			   char *pass, int passlen, int options, char *pempass)
{
	int i;
	guilent_printf("%s %d: sk_PKCS12_SAFEBAG_num (bags) =%d \n" , __FILE__, __LINE__, sk_PKCS12_SAFEBAG_num (bags));
	for (i = 0; i < sk_PKCS12_SAFEBAG_num (bags); i++) {
		if (!dump_certs_pkeys_bag (out,
					   sk_PKCS12_SAFEBAG_value (bags, i),
					   pass, passlen,
					   options, pempass))
		    return 0;
	}
	return 1;
}

int dump_certs_pkeys_bag (BIO *out, PKCS12_SAFEBAG *bag, char *pass,
	     int passlen, int options, char *pempass)
{
	EVP_PKEY *pkey;
	PKCS8_PRIV_KEY_INFO *p8;
	X509 *x509;
	
	guilent_printf("%s %d: M_PKCS12_bag_type(bag) =%d \n" , __FILE__, __LINE__, M_PKCS12_bag_type(bag));
	switch (M_PKCS12_bag_type(bag))
	{
	case NID_keyBag:
		if (options & INFO) BIO_printf (bio_err, "Key bag\n");
		if (options & NOKEYS) return 1;
		print_attribs (out, bag->attrib, "Bag Attributes");
		p8 = bag->value.keybag;
		if (!(pkey = EVP_PKCS82PKEY (p8))) 
		{
			printf("%s %d: pkey is NULL\n" , __FILE__, __LINE__);
			return 0;
		}
		print_attribs (out, p8->attributes, "Key Attributes");
		PEM_write_bio_PrivateKey (out, pkey, enc, NULL, 0, NULL, pempass);
		EVP_PKEY_free(pkey);
	break;

	case NID_pkcs8ShroudedKeyBag:
		if (options & INFO) {
			BIO_printf (bio_err, "Shrouded Keybag: ");
			alg_print (bio_err, bag->value.shkeybag->algor);
		}
		if (options & NOKEYS) return 1;
		print_attribs (out, bag->attrib, "Bag Attributes");
		if (!(p8 = PKCS12_decrypt_skey(bag, pass, passlen)))
		{
			printf("%s %d: p8 is NULL\n" , __FILE__, __LINE__);
			return 0;
		}
		if (!(pkey = EVP_PKCS82PKEY (p8))) {
			printf("%s %d: pkey is NULL\n" , __FILE__, __LINE__);
			PKCS8_PRIV_KEY_INFO_free(p8);
			return 0;
		}
		print_attribs (out, p8->attributes, "Key Attributes");
		PKCS8_PRIV_KEY_INFO_free(p8);
		PEM_write_bio_PrivateKey (out, pkey, enc, NULL, 0, NULL, pempass);
		EVP_PKEY_free(pkey);
	break;

	case NID_certBag:
		if (options & INFO) BIO_printf (bio_err, "Certificate bag\n");
		if (options & NOCERTS) return 1;
                if (PKCS12_get_attr(bag, NID_localKeyID)) {
			if (options & CACERTS) return 1;
		} else if (options & CLCERTS) return 1;
		print_attribs (out, bag->attrib, "Bag Attributes");
		if (M_PKCS12_cert_bag_type(bag) != NID_x509Certificate )
								 return 1;
		if (!(x509 = PKCS12_certbag2x509(bag))) 
		{
			printf("%s %d: x509 is NULL\n" , __FILE__, __LINE__);
			return 0;
		}
		dump_cert_text (out, x509);
		PEM_write_bio_X509 (out, x509);
		X509_free(x509);
	break;

	case NID_safeContentsBag:
		{
			int res = 0;
			if (options & INFO) 
			{
				print_attribs (out, bag->attrib, "Bag Attributes");
				guilent_printf("%s %d: Safe Contents bag\n" , __FILE__, __LINE__);
				BIO_printf (bio_err, "Safe Contents bag\n");
			}
			
			res = dump_certs_pkeys_bags (out, bag->value.safes, pass,
								    passlen, options, pempass);
			guilent_printf("%s %d: res =%d \n" , __FILE__, __LINE__, res);
			return res;
		}
					
	default:
		printf("%s %d: Warning unsupported bag: %d\n" , __FILE__, __LINE__, M_PKCS12_bag_type(bag));
		BIO_printf (bio_err, "Warning unsupported bag type: ");
		i2a_ASN1_OBJECT (bio_err, bag->type);
		BIO_printf (bio_err, "\n");
		return 1;
	break;
	}
	return 1;
}

/* Given a single certificate return a verified chain or NULL if error */

/* Hope this is OK .... */

int get_cert_chain (X509 *cert, X509_STORE *store, STACK_OF(X509) **chain)
{
	X509_STORE_CTX store_ctx;
	STACK_OF(X509) *chn;
	int i;

	/* FIXME: Should really check the return status of X509_STORE_CTX_init
	 * for an error, but how that fits into the return value of this
	 * function is less obvious. */
	X509_STORE_CTX_init(&store_ctx, store, cert, NULL);
	if (X509_verify_cert(&store_ctx) <= 0) {
		i = X509_STORE_CTX_get_error (&store_ctx);
		goto err;
	}
	chn =  X509_STORE_CTX_get1_chain(&store_ctx);
	i = 0;
	*chain = chn;
err:
	X509_STORE_CTX_cleanup(&store_ctx);
	
	return i;
}	

int alg_print (BIO *x, X509_ALGOR *alg)
{
	PBEPARAM *pbe;
	const unsigned char *p;
	p = alg->parameter->value.sequence->data;
	pbe = d2i_PBEPARAM (NULL, &p, alg->parameter->value.sequence->length);
	BIO_printf (bio_err, "%s, Iteration %ld\n", 
		OBJ_nid2ln(OBJ_obj2nid(alg->algorithm)),
		ASN1_INTEGER_get(pbe->iter));
	PBEPARAM_free (pbe);
	return 0;
}

/* Generalised attribute print: handle PKCS#8 and bag attributes */

int print_attribs (BIO *out, STACK_OF(X509_ATTRIBUTE) *attrlst,const char *name)
{
	X509_ATTRIBUTE *attr;
	ASN1_TYPE *av;
	char *value;
	int i, attr_nid;
	if(!attrlst) {
		BIO_printf(out, "%s: <No Attributes>\n", name);
		return 1;
	}
	if(!sk_X509_ATTRIBUTE_num(attrlst)) {
		BIO_printf(out, "%s: <Empty Attributes>\n", name);
		return 1;
	}
	BIO_printf(out, "%s\n", name);
	for(i = 0; i < sk_X509_ATTRIBUTE_num(attrlst); i++) {
		attr = sk_X509_ATTRIBUTE_value(attrlst, i);
		attr_nid = OBJ_obj2nid(attr->object);
		BIO_printf(out, "    ");
		if(attr_nid == NID_undef) {
			i2a_ASN1_OBJECT (out, attr->object);
			BIO_printf(out, ": ");
		} else BIO_printf(out, "%s: ", OBJ_nid2ln(attr_nid));

		if(sk_ASN1_TYPE_num(attr->value.set)) {
			av = sk_ASN1_TYPE_value(attr->value.set, 0);
			switch(av->type) {
				case V_ASN1_BMPSTRING:
        			value = uni2asc(av->value.bmpstring->data,
                                	       av->value.bmpstring->length);
				BIO_printf(out, "%s\n", value);
				OPENSSL_free(value);
				break;

				case V_ASN1_OCTET_STRING:
				hex_prin(out, av->value.octet_string->data,
					av->value.octet_string->length);
				BIO_printf(out, "\n");	
				break;

				case V_ASN1_BIT_STRING:
				hex_prin(out, av->value.bit_string->data,
					av->value.bit_string->length);
				BIO_printf(out, "\n");	
				break;

				default:
					BIO_printf(out, "<Unsupported tag %d>\n", av->type);
				break;
			}
		} else BIO_printf(out, "<No Values>\n");
	}
	return 1;
}

void hex_prin(BIO *out, unsigned char *buf, int len)
{
	int i;
	for (i = 0; i < len; i++) BIO_printf (out, "%02X ", buf[i]);
}

int VerifyP12CertLife (const char * pcert, const char *passwd)
{
#ifdef PKCS12_FUNCS
	FILE *f=NULL;
	PKCS12 *p12;
	EVP_PKEY *pkey;
	X509 *cert;
	STACK_OF(X509) *certs;
	int bValid = 1;
	char buf[256];
	FILE *fp =NULL;

	guilent_printf("%s %d:pcert=%s \n", __FILE__, __LINE__, pcert);
	f = fopen(pcert, "r");
	guilent_printf("%s %d: f =%X pcert =%s\n", __FILE__, __LINE__, f, pcert);
	if (f == NULL)
	{
		printf("%s %d: Can not open cert pfx file --- %s\n", __FILE__, __LINE__, pcert);
		return 0;
	}

	guilent_printf("%s %d: pcert=%s p12 =%X p12->mac=%X\n", __FILE__, __LINE__, pcert, p12, p12->mac);
	p12 = d2i_PKCS12_fp(f, NULL);
	guilent_printf("%s %d: pcert=%s p12 =%X p12->mac=%X\n", __FILE__, __LINE__, pcert, p12, p12->mac);
	if (p12 == NULL) {
		printf("%s %d:TLS: Failed to read PKCS12 file '%s'\n", __FILE__, __LINE__, pcert);
		fclose(f);
		return 0;
	}
	fclose(f);

	pkey = NULL;
	cert = NULL;
	certs = NULL;
	guilent_printf("%s %d: pcert file =%s passwd =%s\n", __FILE__, __LINE__, pcert, passwd);
	if (!PKCS12_parse(p12, passwd, &pkey, &cert, &certs)) {
		printf("%s %d:Failed to PKCS12_parse file '%s'\n", __FILE__, __LINE__, pcert);
		return 0;
	}

	if (cert) {
		time_t current_time;
		BIO *STDout=NULL;
		ASN1_TIME *notBefore =X509_get_notBefore(cert);
		ASN1_TIME *notAfter = X509_get_notAfter(cert);
		ASN1_UTCTIME *be=ASN1_STRING_dup(notBefore);
		ASN1_UTCTIME *af=ASN1_STRING_dup(notAfter);
		//X509_NAME_oneline(X509_get_subject_name(cert), buf, sizeof(buf));
		//printf("TLS: Got certificate from PKCS12: subject='%s'\n", buf);

		time(&current_time);
		{
			BIO *STDout=NULL;
			STDout=BIO_new_fp(stdout,BIO_NOCLOSE);
			BIO_puts(STDout,"notBefore=");
			ASN1_TIME_print(STDout, notBefore);
			BIO_puts(STDout,"\n");
#if 0		
			if (ASN1_UTCTIME_cmp_time_t(be, current_time) >= 0)
			{
				printf("%s %d: TLS: notBefore Failed.\n", __FILE__, __LINE__);
			}
			else
			{
				printf("%s %d: TLS: notBefore OK.\n", __FILE__, __LINE__);
			}
#endif
			BIO_puts(STDout,"notAfter=");
			ASN1_TIME_print(STDout, notAfter);
			BIO_puts(STDout,"\n");
#if 0		
			if (ASN1_UTCTIME_cmp_time_t(af, current_time) <= 0)
			{
				cert_printf("%s %d: TLS: notAfter Failed.\n", __FILE__, __LINE__);
			}
			else
			{
				cert_printf("%s %d: TLS: notAfter OK.\n", __FILE__, __LINE__);
			}
#endif		
		}
		if (/*ASN1_UTCTIME_cmp_time_t(be, current_time) >= 0
			|| */ASN1_UTCTIME_cmp_time_t(af, current_time) <= 0)
		{
			fp = fopen(CERT_VERIFY_FILE,"wt");
			if ( fp != NULL )
			{
				fprintf(fp,"%d\n", 0);
				fclose(fp);
			}
			bValid = 0;
		}
		else
		{
			fp = fopen(CERT_VERIFY_FILE,"wt");
			if ( fp != NULL )
			{
				fprintf(fp,"%d\n", 1);
				fclose(fp);
			}
			bValid = 1;
		}
		cert_printf("%s %d: bValid=%d\n", __FILE__, __LINE__, bValid);
		
		M_ASN1_UTCTIME_free(be);
		M_ASN1_UTCTIME_free(af);
  		X509_free(cert);
	}

	PKCS12_free(p12);

	return 0;
#else /* PKCS12_FUNCS */
	printf("%s %d:TLS: PKCS12 support disabled - cannot read p12/pfx files\n", __FILE__, __LINE__);
	return 0;
#endif  /* PKCS12_FUNCS */

}

#ifndef OPENSSL_NO_ENGINE

/* Try to load an engine in a shareable library */
static ENGINE *try_load_engine(BIO *err, const char *engine, int debug)
	{
	ENGINE *e = ENGINE_by_id("dynamic");
	if (e)
		{
		if (!ENGINE_ctrl_cmd_string(e, "SO_PATH", engine, 0)
			|| !ENGINE_ctrl_cmd_string(e, "LOAD", NULL, 0))
			{
			ENGINE_free(e);
			e = NULL;
			}
		}
	return e;
	}

ENGINE *setup_engine(BIO *err, const char *engine, int debug)
        {
        ENGINE *e = NULL;

        if (engine)
                {
		if(strcmp(engine, "auto") == 0)
			{
			BIO_printf(err,"enabling auto ENGINE support\n");
			ENGINE_register_all_complete();
			return NULL;
			}
		if((e = ENGINE_by_id(engine)) == NULL
			&& (e = try_load_engine(err, engine, debug)) == NULL)
			{
			BIO_printf(err,"invalid engine \"%s\"\n", engine);
			ERR_print_errors(err);
			return NULL;
			}
		if (debug)
			{
			ENGINE_ctrl(e, ENGINE_CTRL_SET_LOGSTREAM,
				0, err, 0);
			}
                ENGINE_ctrl_cmd(e, "SET_USER_INTERFACE", 0, ui_method, 0, 1);
		if(!ENGINE_set_default(e, ENGINE_METHOD_ALL))
			{
			BIO_printf(err,"can't use that engine\n");
			ERR_print_errors(err);
			ENGINE_free(e);
			return NULL;
			}

		BIO_printf(err,"engine \"%s\" set.\n", ENGINE_get_id(e));

		/* Free our "structural" reference. */
		ENGINE_free(e);
		}
        return e;
        }
#endif

int password_callback(char *buf, int bufsiz, int verify,
	PW_CB_DATA *cb_tmp)
	{
	UI *ui = NULL;
	int res = 0;
	const char *prompt_info = NULL;
	const char *password = NULL;
	PW_CB_DATA *cb_data = (PW_CB_DATA *)cb_tmp;

	if (cb_data)
		{
		if (cb_data->password)
			password = cb_data->password;
		if (cb_data->prompt_info)
			prompt_info = cb_data->prompt_info;
		}

	if (password)
		{
		res = strlen(password);
		if (res > bufsiz)
			res = bufsiz;
		memcpy(buf, password, res);
		return res;
		}

	ui = UI_new_method(ui_method);
	if (ui)
		{
		int ok = 0;
		char *buff = NULL;
		int ui_flags = 0;
		char *prompt = NULL;

		prompt = UI_construct_prompt(ui, "pass phrase",
			prompt_info);

		ui_flags |= UI_INPUT_FLAG_DEFAULT_PWD;
		UI_ctrl(ui, UI_CTRL_PRINT_ERRORS, 1, 0, 0);

		if (ok >= 0)
			ok = UI_add_input_string(ui,prompt,ui_flags,buf,
				PW_MIN_LENGTH,BUFSIZ-1);
		if (ok >= 0 && verify)
			{
			buff = (char *)OPENSSL_malloc(bufsiz);
			ok = UI_add_verify_string(ui,prompt,ui_flags,buff,
				PW_MIN_LENGTH,BUFSIZ-1, buf);
			}
		if (ok >= 0)
			do
				{
				ok = UI_process(ui);
				}
			while (ok < 0 && UI_ctrl(ui, UI_CTRL_IS_REDOABLE, 0, 0, 0));

		if (buff)
			{
			OPENSSL_cleanse(buff,(unsigned int)bufsiz);
			OPENSSL_free(buff);
			}

		if (ok >= 0)
			res = strlen(buf);
		if (ok == -1)
			{
			BIO_printf(bio_err, "User interface error\n");
			ERR_print_errors(bio_err);
			OPENSSL_cleanse(buf,(unsigned int)bufsiz);
			res = 0;
			}
		if (ok == -2)
			{
			BIO_printf(bio_err,"aborted!\n");
			OPENSSL_cleanse(buf,(unsigned int)bufsiz);
			res = 0;
			}
		UI_free(ui);
		OPENSSL_free(prompt);
		}
	return res;
	}

X509 *load_cert(BIO *err, const char *file, int format,
	const char *pass, ENGINE *e, const char *cert_descrip)
	{
	ASN1_HEADER *ah=NULL;
	BUF_MEM *buf=NULL;
	X509 *x=NULL;
	BIO *cert;

	if ((cert=BIO_new(BIO_s_file())) == NULL)
		{
		ERR_print_errors(err);
		goto end;
		}

	if (file == NULL)
		{
		setvbuf(stdin, NULL, _IONBF, 0);
		BIO_set_fp(cert,stdin,BIO_NOCLOSE);
		}
	else
		{
		if (BIO_read_filename(cert,file) <= 0)
			{
			BIO_printf(err, "Error opening %s %s\n",
				cert_descrip, file);
			ERR_print_errors(err);
			goto end;
			}
		}

	if 	(format == FORMAT_ASN1)
		x=d2i_X509_bio(cert,NULL);
	else if (format == FORMAT_NETSCAPE)
		{
		const unsigned char *p,*op;
		int size=0,i;

		/* We sort of have to do it this way because it is sort of nice
		 * to read the header first and check it, then
		 * try to read the certificate */
		buf=BUF_MEM_new();
		for (;;)
			{
			if ((buf == NULL) || (!BUF_MEM_grow(buf,size+1024*10)))
				goto end;
			i=BIO_read(cert,&(buf->data[size]),1024*10);
			size+=i;
			if (i == 0) break;
			if (i < 0)
				{
				perror("reading certificate");
				goto end;
				}
			}
		p=(unsigned char *)buf->data;
		op=p;

		/* First load the header */
		if ((ah=d2i_ASN1_HEADER(NULL,&p,(long)size)) == NULL)
			goto end;
		if ((ah->header == NULL) || (ah->header->data == NULL) ||
			(strncmp(NETSCAPE_CERT_HDR,(char *)ah->header->data,
			ah->header->length) != 0))
			{
			BIO_printf(err,"Error reading header on certificate\n");
			goto end;
			}
		/* header is ok, so now read the object */
		p=op;
		ah->meth=X509_asn1_meth();
		if ((ah=d2i_ASN1_HEADER(&ah,&p,(long)size)) == NULL)
			goto end;
		x=(X509 *)ah->data;
		ah->data=NULL;
		}
	else if (format == FORMAT_PEM)
		x=PEM_read_bio_X509_AUX(cert,NULL,
			(pem_password_cb *)password_callback, NULL);
	else if (format == FORMAT_PKCS12)
		{
#if 0	//guilent		
		if (!load_pkcs12(err, cert,cert_descrip, NULL, NULL,
					NULL, &x, NULL))
			goto end;
#endif
		}
	else	{
		BIO_printf(err,"bad input format specified for %s\n",
			cert_descrip);
		goto end;
		}
end:
	if (x == NULL)
		{
		BIO_printf(err,"unable to load certificate\n");
		ERR_print_errors(err);
		}
	if (ah != NULL) ASN1_HEADER_free(ah);
	if (cert != NULL) BIO_free(cert);
	if (buf != NULL) BUF_MEM_free(buf);
	return(x);
	}


int VerifyPemCertLife (const char * pcert)
{
	ENGINE *e = NULL;
	X509 *x=NULL;
	int informat,outformat,keyformat,CAformat,CAkeyformat;
	char *infile=NULL,*outfile=NULL,*keyfile=NULL,*CAfile=NULL;
#ifndef OPENSSL_NO_ENGINE
	char *engine=NULL;
#endif
	int bValid = 1;
	char buf[256];
	FILE *fp =NULL;

	informat=FORMAT_PEM;
	outformat=FORMAT_PEM;
	keyformat=FORMAT_PEM;
	CAformat=FORMAT_PEM;
	CAkeyformat=FORMAT_PEM;

#ifndef OPENSSL_NO_ENGINE
        e = setup_engine(bio_err, engine, 0);
#endif

	x=load_cert(bio_err,pcert,informat,NULL,e,"Certificate");
	if (x == NULL)
	{
		printf("%s %d: load_cert failed x=NULL\n", __FILE__, __LINE__);
		return 0;
	}

	time_t current_time;
	BIO *STDout=NULL;
	ASN1_TIME *notBefore =X509_get_notBefore(x);
	ASN1_TIME *notAfter = X509_get_notAfter(x);
	ASN1_UTCTIME *be=ASN1_STRING_dup(notBefore);
	ASN1_UTCTIME *af=ASN1_STRING_dup(notAfter);
	//X509_NAME_oneline(X509_get_subject_name(x), buf, sizeof(buf));
	//printf("TLS: Got certificate %s from PKCS12: subject='%s'\n", pcert, buf);

	time(&current_time);
#if 0		
	{
		BIO *STDout=NULL;
		STDout=BIO_new_fp(stdout,BIO_NOCLOSE);
		BIO_puts(STDout,"notBefore=");
		ASN1_TIME_print(STDout, notBefore);
		BIO_puts(STDout,"\n");
		if (ASN1_UTCTIME_cmp_time_t(be, current_time) >= 0)
		{
			printf("%s %d: TLS: notBefore Failed.\n", __FILE__, __LINE__);
		}
		else
		{
			printf("%s %d: TLS: notBefore OK.\n", __FILE__, __LINE__);
		}

		BIO_puts(STDout,"notAfter=");
		ASN1_TIME_print(STDout, notAfter);
		BIO_puts(STDout,"\n");
		if (ASN1_UTCTIME_cmp_time_t(af, current_time) <= 0)
		{
			cert_printf("%s %d: TLS: notAfter Failed.\n", __FILE__, __LINE__);
		}
		else
		{
			cert_printf("%s %d: TLS: notAfter OK.\n", __FILE__, __LINE__);
		}
	}
#endif		
	if (/*ASN1_UTCTIME_cmp_time_t(be, current_time) >= 0
		|| */ASN1_UTCTIME_cmp_time_t(af, current_time) <= 0)
	{
		fp = fopen(CERT_VERIFY_FILE,"wt");
		if ( fp != NULL )
		{
			fprintf(fp,"%d\n", 0);
			fclose(fp);
		}
		bValid = 0;
	}
	else
	{
		fp = fopen(CERT_VERIFY_FILE,"wt");
		if ( fp != NULL )
		{
			fprintf(fp,"%d\n", 1);
			fclose(fp);
		}
		bValid = 1;
	}
	//printf("%s %d: bValid=%d\n", __FILE__, __LINE__, bValid);
}

int str2fmt(char *s)
	{
	if 	((*s == 'D') || (*s == 'd'))
		return(FORMAT_ASN1);
	else if ((*s == 'T') || (*s == 't'))
		return(FORMAT_TEXT);
	else if ((*s == 'P') || (*s == 'p'))
		return(FORMAT_PEM);
	else if ((*s == 'N') || (*s == 'n'))
		return(FORMAT_NETSCAPE);
	else if ((*s == 'S') || (*s == 's'))
		return(FORMAT_SMIME);
	else if ((*s == '1')
		|| (strcmp(s,"PKCS12") == 0) || (strcmp(s,"pkcs12") == 0)
		|| (strcmp(s,"P12") == 0) || (strcmp(s,"p12") == 0))
		return(FORMAT_PKCS12);
	else if ((*s == 'E') || (*s == 'e'))
		return(FORMAT_ENGINE);
	else
		return(FORMAT_UNDEF);
}

#endif

int main(int argc, char *argv[])
{
	int c, i;
	struct wpa_interface *ifaces, *iface;
	int iface_count, exitcode;
	struct wpa_params params;
	struct wpa_global *global;
	FILE *fp;

#ifdef CONFIG_NATIVE_WINDOWS
	WSADATA wsaData;
	if (WSAStartup(MAKEWORD(2, 0), &wsaData)) {
		printf("Could not find a usable WinSock.dll\n");
		return -1;
	}
#endif /* CONFIG_NATIVE_WINDOWS */

#ifdef _ZCOM_NANJING_
	if (argc > 1)
	{
		daemon(1, 1);

		//guilent
		ERR_load_crypto_strings();
		OpenSSL_add_all_ciphers();
		OpenSSL_add_all_digests();
		enc = EVP_des_ede3_cbc();

		if (argc == 3)	//for cer x509 server certificate
		{
			/* 
			wpa_supplicant -verify /mnt/jffs2/cert/sgl.cer 
			*/
			if((strncmp(argv[1],"-verify",strlen("-verify"))==0)
				||strncmp(argv[1],"-v",strlen("-v"))==0)
			{
				guilent_printf("%s %d: Verify CA cert  %s\n", __FILE__, __LINE__, argv[2]);
				VerifyPemCertLife(argv[2]);
			}
			else
			{
				printf ("Usage: wpa_supplicant verify [options]\n");
				printf ("where options are\n");
				printf ("wpa_supplicant -verify /mnt/jffs2/cert/sgl.cer\n");
			}
			return 0;
		}
		else if (argc == 4)
		{
			/* 
			wpa_supplicant -verify /mnt/jffs2/cert/*_key.pem *** 
			*/
			if((strncmp(argv[1],"-verify",strlen("-verify"))==0)
				||strncmp(argv[1],"-v",strlen("-v"))==0)
			{
				guilent_printf("%s %d: Verify cert  %s [%s]\n", __FILE__, __LINE__, argv[2], argv[3]);
				VerifyP12CertLife(argv[2], argv[3]);
			}
			else
			{
				printf ("Usage: wpa_supplicant verify [options]\n");
				printf ("where options are\n");
				printf ("wpa_supplicant -verify /mnt/jffs2/cert/*_key.pem ***\n");
			}
			return 0;
		}
		else if (argc == 8)
		{
			char *infile=NULL, *outfile=NULL, *keyname = NULL;	
			BIO *in=NULL, *out = NULL;
			PKCS12 *p12 = NULL;
			char Importpassword[50];
			char macpass[50];
			int macver = 1;
			STACK_OF(X509) *certs=NULL;
			int options = INFO;
			
			/* 
			wpa_supplicant p12 -in *.pfx -out *_key.pem -certpass ***  
			*/
			if((strncmp(argv[1],"-PKCS12",strlen("-PKCS12"))==0)
				||strncmp(argv[1],"-pkcs12",strlen("-pkcs12"))==0
				||strncmp(argv[1],"-P12",strlen("-P12"))==0
				||strncmp(argv[1],"-p12",strlen("-p12"))==0)
			{
				if ((strncmp(argv[2],"-in",strlen("-in"))==0)) 
				{
					infile = argv[3];
					if ((strncmp(argv[4],"-out",strlen("-out"))==0)) 
					{
						outfile = argv[5];
					}

					if ((strncmp(argv[6],"-certpass",strlen("-certpass"))==0)) 
					{
						memcpy(Importpassword, argv[7], strlen(argv[7]));
						guilent_printf("%s %d: Importpassword = %s \n", __FILE__, __LINE__, Importpassword);
					}
				} 
				else
				{
					printf("%s %d: Invalid supplicant pkcs12 input !\n", __FILE__, __LINE__);
					
					printf ("Usage: wpa_supplicant pkcs12 [options]\n");
					printf ("where options are?\n");
					printf ("-in  infile   		input filename\n");
					printf ("-out outfile 	output filename\n");
					printf ("-certpass p     	certificate password\n");
					return 0;
				}

				//printf("%s %d: infile = %s \n", __FILE__, __LINE__, infile);
				in = BIO_new_file(infile, "rb");
				if (!in) {
					guilent_printf("%s %d: Error opening input file %s \n", __FILE__, __LINE__, infile ? infile : "<stdout>");
					return 0;
				}

				guilent_printf("%s %d: outfile = %s \n", __FILE__, __LINE__, outfile);
				out = BIO_new_file(outfile, "wb");
				if (!out) {
					printf("%s %d: Error opening output file %s \n", __FILE__, __LINE__, outfile ? outfile : "<stdout>");
						return 0;
				}

				//load p12 structure from input pem file
				if (!(p12 = d2i_PKCS12_bio (in, NULL))) 
				{
					printf("%s %d: d2i_PKCS12_bio failed!!\n", __FILE__, __LINE__);
					return 0;
				}
				
				if(1/*macver*/) {
					guilent_printf("%s %d: verify MAC now Importpassword = %s \n", __FILE__, __LINE__, Importpassword);

					/* Try no password first */
					if(!Importpassword[0] && PKCS12_verify_mac(p12, NULL, 0)) 
					{
						printf (" %s %d: No password in.\n", __FILE__, __LINE__);
					} 
					else if (!PKCS12_verify_mac(p12, Importpassword, -1)) 
					{
						printf (" %s %d: Mac verify error: invalid password '%s' ?\n", __FILE__, __LINE__, Importpassword);
						return 0;			
					}
					//printf("%s %d: MAC verified OK\n", __FILE__, __LINE__);
				}
				
				guilent_printf ("MAC Iteration %ld\n", p12->mac->iter ? ASN1_INTEGER_get (p12->mac->iter) : 1);
				//if (!dump_certs_keys_p12 (out, p12, Importpassword, -1, 0/*options*/, NULL/*passout*/)) 
				if (!dump_certs_keys_p12 (out, p12, Importpassword, strlen(Importpassword), options, Importpassword)) 
				{
					printf (" %s %d: Error outputting keys and certificatesn", __FILE__, __LINE__);
				}

				BIO_free(in);
				BIO_free_all(out);
			}
			return 0;			
		}
		else if (argc == 10)
		{
			if(strncmp(argv[1],"-x509",strlen("-x509"))==0
				|| strncmp(argv[1],"-x",strlen("-x"))==0)
			{
				/* 
				wpa_supplicant -x509 -inform DER -in sgl.cer -outform PEM -out sgl.cer
				*/
				BIO *in=NULL, *out = NULL;
				char *infile=NULL,*outfile=NULL;
				int informat,outformat;
				X509 *x=NULL;
				ENGINE *e = NULL;
#ifndef OPENSSL_NO_ENGINE
				char *engine=NULL;
#endif
				int i=0;
				time_t current_time;
				char buf[256];
				int bValid = 1;

				informat=FORMAT_PEM;
				outformat=FORMAT_PEM;

				if ((strncmp(argv[2],"-inform",strlen("-inform"))==0)) 
				{
					informat=str2fmt(argv[3]);
					if ((strncmp(argv[4],"-in",strlen("-in"))==0)) 
					{
						infile = argv[5];
						if ((strncmp(argv[6],"-outform",strlen("-outform"))==0)) 
						{
							outformat=str2fmt(argv[7]);
							//printf("%s %d: informat=%d outformat=%d\n", __FILE__, __LINE__, informat, outformat);
							if ((strncmp(argv[8],"-out",strlen("-out"))==0)) 
							{
								outfile = argv[9];
#ifndef OPENSSL_NO_ENGINE
								e = setup_engine(bio_err, engine, 0);
#endif
								x=load_cert(bio_err,infile,informat,NULL,e,"Certificate");
								if (x == NULL)
								{
									FILE *fp =NULL;
									
									fp = fopen(CERT_VERIFY_FILE,"wt");
									if ( fp != NULL )
									{
										fprintf(fp,"%d\n", 2);	//B64
										fclose(fp);
									}
									bValid = 2;
									
									//printf("%s %d: load_cert failed bValid=%d\n", __FILE__, __LINE__, bValid);
									return 0;
								}

								ASN1_TIME *notBefore =X509_get_notBefore(x);
								ASN1_TIME *notAfter = X509_get_notAfter(x);
								ASN1_UTCTIME *be=ASN1_STRING_dup(notBefore);
								ASN1_UTCTIME *af=ASN1_STRING_dup(notAfter);
								//X509_NAME_oneline(X509_get_subject_name(x), buf, sizeof(buf));
								//printf("TLS: Got certificate %s from PKCS12: subject='%s'\n", infile, buf);
								time(&current_time);
#if 0		
								{
									BIO *STDout=NULL;
									
									STDout=BIO_new_fp(stdout,BIO_NOCLOSE);
									BIO_puts(STDout,"notBefore=");
									ASN1_TIME_print(STDout, notBefore);
									BIO_puts(STDout,"\n");
									if (ASN1_UTCTIME_cmp_time_t(be, current_time) >= 0)
									{
										printf("%s %d: TLS: notBefore Failed.\n", __FILE__, __LINE__);
									}
									else
									{
										printf("%s %d: TLS: notBefore OK.\n", __FILE__, __LINE__);
									}

									BIO_puts(STDout,"notAfter=");
									ASN1_TIME_print(STDout, notAfter);
									BIO_puts(STDout,"\n");
									if (ASN1_UTCTIME_cmp_time_t(af, current_time) <= 0)
									{
										cert_printf("%s %d: TLS: notAfter Failed.\n", __FILE__, __LINE__);
									}
									else
									{
										cert_printf("%s %d: TLS: notAfter OK.\n", __FILE__, __LINE__);
									}
								}
#endif		

								if (/*ASN1_UTCTIME_cmp_time_t(be, current_time) >= 0
									|| */ASN1_UTCTIME_cmp_time_t(af, current_time) <= 0)
								{
									fp = fopen(CERT_VERIFY_FILE,"wt");
									if ( fp != NULL )
									{
										fprintf(fp,"%d\n", 0);
										fclose(fp);
									}
									bValid = 0;
								}
								else
								{
									fp = fopen(CERT_VERIFY_FILE,"wt");
									if ( fp != NULL )
									{
										fprintf(fp,"%d\n", 1);
										fclose(fp);
									}
									bValid = 1;
								}

								//convert x509 DER to pem now
								OBJ_create("2.99999.3",
									"SET.ex3","SET x509v3 extension 3");
								out=BIO_new(BIO_s_file());
								//printf("%s %d: out=%X\n", __FILE__, __LINE__, out);
								if (out == NULL)
								{
									printf("%s %d: BIO_new failed\n", __FILE__, __LINE__);
									return 0;
								}
								
								if (BIO_write_filename(out,outfile) <= 0)
								{
									printf("%s %d: BIO_write_filename failed\n", __FILE__, __LINE__);
									return 0;
								}

								if(outformat == FORMAT_ASN1)
								{
									i=i2d_X509_bio(out,x);
								}
								else if (outformat == FORMAT_PEM)
								{
									if (0/*trustout*/) 
										i=PEM_write_bio_X509_AUX(out,x);
									else 
										i=PEM_write_bio_X509(out,x);
								}
								else if (outformat == FORMAT_NETSCAPE)
								{
									ASN1_HEADER ah;
									ASN1_OCTET_STRING os;

									os.data=(unsigned char *)NETSCAPE_CERT_HDR;
									os.length=strlen(NETSCAPE_CERT_HDR);
									ah.header= &os;
									ah.data=(char *)x;
									ah.meth=X509_asn1_meth();

									i=ASN1_i2d_bio_of(ASN1_HEADER,i2d_ASN1_HEADER,out,&ah);
								}
								else	
								{
									printf("%s %d: bad output format specified for outfile\n", __FILE__, __LINE__);
									return 0;
								}

								if (!i)
								{
									printf("%s %d: unable to write certificatee\n", __FILE__, __LINE__);
									return 0;
								}

								//printf("%s %d: outfile=%s\n", __FILE__, __LINE__, outfile);
								//VerifyPemCertLife(outfile);
							}
						}
					}
				} 
				else
				{
					printf ("Usage: wpa_supplicant x509 [options]\n");
					printf ("wpa_supplicant x509 -inform DER -in sgl.cer -outform PEM -out sgl.cer\n");
					return 0;
				}
			}
			else if(strncmp(argv[1],"-pkcs7",strlen("-pkcs7"))==0)
			{
				/* 
				wpa_supplicant pkcs7 -inform DER -in *.p7b -outform PEM -out *.pem
				*/

				//not support now
			}
			
			return 0;			
		}
		else
		{
			printf("%s %d: Invalid supplicant paramater!\n", __FILE__, __LINE__);
			return 0;
		}
	}
#endif

	memset(&params, 0, sizeof(params));
	params.wpa_debug_level = MSG_MSGDUMP;	//MSG_INFO;

	iface = ifaces = malloc(sizeof(struct wpa_interface));
	if (ifaces == NULL)
		return -1;
	memset(iface, 0, sizeof(*iface));
	iface_count = 1;

//	wpa_supplicant_fd_workaround();

		
	
	iface->confname = "/tmp/supp_conf";	
			
	iface->driver = "madwifi";
			
	params.wpa_debug_level--;
		
	iface->ifname = "ath0";
					
	guilent_printf("%s %d: iface->ifname =%s iface->driver=%s", __FILE__, __LINE__, iface->ifname, iface->driver);

	exitcode = 0;
	global = wpa_supplicant_init(&params);
	guilent_printf("%s %d: iglobal=%X ", __FILE__, __LINE__, global);
	if (global == NULL) {
		printf("Failed to initialize wpa_supplicant\n");
		exitcode = -1;
	}

	fp = fopen("/tmp/pid_hostapd","wt");
	if ( fp != NULL )
	{
		fprintf(fp,"%d\n",getpid());
		fclose(fp);
	}

	for (i = 0; exitcode == 0 && i < iface_count; i++) {
		if ((ifaces[i].confname == NULL &&
		     ifaces[i].ctrl_interface == NULL) ||
		    ifaces[i].ifname == NULL) {
			if (iface_count == 1 && params.ctrl_interface)
				break;
			usage();
			return -1;
		}
		if (wpa_supplicant_add_iface(global, &ifaces[i]) == NULL)
			exitcode = -1;
	}

	if (exitcode == 0)
		exitcode = wpa_supplicant_run(global);

	wpa_supplicant_deinit(global);

	free(ifaces);
	free(params.pid_file);

	return exitcode;
}
