/*
 * Part of Very Secure FTPd
 * Licence: GPL
 * Author: Chris Evans
 * prelogin.c
 *
 * Code to parse the FTP protocol prior to a successful login.
 */

#include "prelogin.h"
#include "ftpcmdio.h"
#include "ftpcodes.h"
#include "str.h"
#include "vsftpver.h"
#include "tunables.h"
#include "oneprocess.h"
#include "twoprocess.h"
#include "sysdeputil.h"
#include "sysutil.h"
#include "session.h"
#include "banner.h"
#include "logging.h"
#ifdef	_ZCOM_NANJING_
#include "wag302client.h"
#include "zcomlog.h"
#endif//_ZCOM_NANJING_

/* Functions used */
static void emit_greeting(struct vsf_session* p_sess);
static void parse_username_password(struct vsf_session* p_sess);
static void handle_user_command(struct vsf_session* p_sess);
static void handle_pass_command(struct vsf_session* p_sess);

#ifdef	_ZCOM_NANJING_
#define DEFAULT_LOGIN_NAME	"admin"
void ZComLogAddAuthorization(
	unsigned short	usIdentification,	// refer to ZComLogID
	unsigned char	puchFromIPv4[4],	// IP Address
	unsigned char	*szUserName			// Authorization User name
	)
{
	int	fd = open(ZCOM_LOG_FULLNAME, O_RDWR);
	int	iResult = -1;
	if (-1!=fd)
	{
		ZComLogAuthReq_t ZComLogAuthReq;
		ZComLogAuthReq.usIdentification = usIdentification;
		memcpy(ZComLogAuthReq.puchFromIPv4, puchFromIPv4, 4);
		strncpy(ZComLogAuthReq.szUserName, szUserName, sizeof(ZComLogAuthReq.szUserName));
		iResult = ioctl(fd, ZCOM_LOG_IOC_AUTH, &ZComLogAuthReq);
		if (iResult < 0)
			printf("%s: ioctl error!\n", __FUNCTION__);
		close(fd);
	}
	else
	{
		printf("%s: open error!\n", __FUNCTION__);
	}
}
#endif

void
init_connection(struct vsf_session* p_sess)
{
  if (tunable_setproctitle_enable)
  {
    vsf_sysutil_setproctitle("not logged in");
  }
  /* Before we talk to the remote, make sure an alarm is set up in case
   * writing the initial greetings should block.
   */
  vsf_cmdio_set_alarm(p_sess);
  emit_greeting(p_sess);
  parse_username_password(p_sess);
}

static void
emit_greeting(struct vsf_session* p_sess)
{
  struct mystr str_log_line = INIT_MYSTR;
  /* Check for client limits (standalone mode only) */
  if (tunable_max_clients > 0 &&
      p_sess->num_clients > tunable_max_clients)
  {
    str_alloc_text(&str_log_line, "Connection refused: too many sessions.");
    vsf_log_line(p_sess, kVSFLogEntryConnection, &str_log_line);
    vsf_cmdio_write_noblock(p_sess, FTP_TOO_MANY_USERS,
                    "There are too many connected users, please try later.");
    vsf_sysutil_exit(0);
  }
  if (tunable_max_per_ip > 0 &&
      p_sess->num_this_ip > tunable_max_per_ip)
  {
    str_alloc_text(&str_log_line,
                   "Connection refused: too many sessions for this address.");
    vsf_log_line(p_sess, kVSFLogEntryConnection, &str_log_line);
    vsf_cmdio_write_noblock(p_sess, FTP_IP_LIMIT,
        "There are too many connections from your internet address.");
    vsf_sysutil_exit(0);
  }
  if (!p_sess->tcp_wrapper_ok)
  {
    str_alloc_text(&str_log_line,
                   "Connection refused: tcp_wrappers denial.");
    vsf_log_line(p_sess, kVSFLogEntryConnection, &str_log_line);
    vsf_cmdio_write_noblock(p_sess, FTP_IP_DENY, "Service not available.");
    vsf_sysutil_exit(0);
  }
  if (!str_isempty(&p_sess->banner_str))
  {
    vsf_banner_write(p_sess, &p_sess->banner_str, FTP_GREET);
    str_free(&p_sess->banner_str);
    vsf_cmdio_write(p_sess, FTP_GREET, "");
  }
  else if (tunable_ftpd_banner == 0)
  {
    vsf_cmdio_write(p_sess, FTP_GREET, "(vsFTPd " VSF_VERSION 
                    ")");
  }
  else
  {
    vsf_cmdio_write(p_sess, FTP_GREET, tunable_ftpd_banner);
  }
}

static void
parse_username_password(struct vsf_session* p_sess)
{
  while (1)
  {
    vsf_cmdio_get_cmd_and_arg(p_sess, &p_sess->ftp_cmd_str,
                              &p_sess->ftp_arg_str, 1);
    if (str_equal_text(&p_sess->ftp_cmd_str, "USER"))
    {
      handle_user_command(p_sess);
    }
    else if (str_equal_text(&p_sess->ftp_cmd_str, "PASS"))
    {
      handle_pass_command(p_sess);
    }
    else if (str_equal_text(&p_sess->ftp_cmd_str, "QUIT"))
    {
#ifdef	_ZCOM_NANJING_
      struct vsf_sysutil_ipv4addr ip= vsf_sysutil_sockaddr_get_ipaddr(p_sess->p_remote_addr);
      ZComLogAddAuthorization(ZCOM_LOG_FTP_LOGOUT, ip.data, DEFAULT_LOGIN_NAME);
#endif
      vsf_cmdio_write(p_sess, FTP_GOODBYE, "Goodbye.");
      vsf_sysutil_exit(0);
    }
    else
    {
#ifdef	_ZCOM_NANJING_
      struct vsf_sysutil_ipv4addr ip= vsf_sysutil_sockaddr_get_ipaddr(p_sess->p_remote_addr);
      ZComLogAddAuthorization(ZCOM_LOG_FTP_UNAUTHORIZED, ip.data, DEFAULT_LOGIN_NAME);
#endif
      vsf_cmdio_write(p_sess, FTP_LOGINERR,
                      "Please login with USER and PASS.");
    }
  }
}

static void
handle_user_command(struct vsf_session* p_sess)
{
#ifdef	_ZCOM_NANJING_
	char	szUserName[40] = {0};
	
	CONF_READ_STRING0(MIB_USER_NAME, szUserName);
	if (!str_equal_text(&p_sess->ftp_arg_str, szUserName))
	{
		vsf_cmdio_write(p_sess, FTP_LOGINERR, "Permission denied.");
		return;
	}
	str_copy(&p_sess->user_str, &p_sess->ftp_arg_str);
#else
  /* SECURITY: If we're in anonymous only-mode, immediately reject
   * non-anonymous usernames in the hope we save passwords going plaintext
   * over the network
   */
  int is_anon = 1;
  str_copy(&p_sess->user_str, &p_sess->ftp_arg_str);
  str_upper(&p_sess->ftp_arg_str);
  if (!str_equal_text(&p_sess->ftp_arg_str, "admin") &&
      !str_equal_text(&p_sess->ftp_arg_str, "ANONYMOUS"))
  {
    is_anon = 0;
  }
  if (!tunable_local_enable && !is_anon)
  {
    vsf_cmdio_write(p_sess, FTP_LOGINERR,
                    "This FTP server is anonymous only.");
    str_empty(&p_sess->user_str);
    return;
  }
  if (!str_isempty(&p_sess->userlist_str))
  {
    int located = str_contains_line(&p_sess->userlist_str, &p_sess->user_str);
    if ((located && tunable_userlist_deny) ||
        (!located && !tunable_userlist_deny))
    {
      vsf_cmdio_write(p_sess, FTP_LOGINERR, "Permission denied.");
      str_empty(&p_sess->user_str);
      return;
    }
  }
  if (is_anon && tunable_no_anon_password)
  {
    /* Fake a password */
    str_alloc_text(&p_sess->ftp_arg_str, "<no password>");
    handle_pass_command(p_sess);
  }
  else
#endif//!_ZCOM_NANJING_
  {
    vsf_cmdio_write(p_sess, FTP_GIVEPWORD, "Please specify the password.");
  }
}

static void
handle_pass_command(struct vsf_session* p_sess)
{
  if (str_isempty(&p_sess->user_str))
  {
    vsf_cmdio_write(p_sess, FTP_NEEDUSER, "Login with USER first.");
    return;
  }
#ifdef _ZCOM_NANJING_
	{
		char	szPassword[40] = {0};
		
		CONF_READ_STRING0(MIB_PASSWORD, szPassword);
		if (!str_equal_text(&p_sess->ftp_arg_str, szPassword))
		{
			vsf_cmdio_write(p_sess, FTP_LOGINERR, "Login incorrect.");
			return;
		}
#ifdef	_ZCOM_NANJING_
        struct vsf_sysutil_ipv4addr ip= vsf_sysutil_sockaddr_get_ipaddr(p_sess->p_remote_addr);
        ZComLogAddAuthorization(ZCOM_LOG_FTP_AUTHORIZED, ip.data, DEFAULT_LOGIN_NAME);
#endif
	}
#endif//_ZCOM_NANJING_
  /* These login calls never return if successful */
  if (tunable_one_process_model)
  {
    vsf_one_process_login(p_sess, &p_sess->ftp_arg_str);
  }
  else
  {
    vsf_two_process_login(p_sess, &p_sess->ftp_arg_str);
  }
  vsf_cmdio_write(p_sess, FTP_LOGINERR, "Login incorrect.");
  str_empty(&p_sess->user_str);
  /* FALLTHRU if login fails */
}

