#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdlib.h>
#include <sys/sysinfo.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <ctype.h>
#include <sys/ioctl.h>

#include "zcomutil.h"
#include "zcomlog.h"
#include "compatibility.h"
#include "dot11MIB.h"

/*****************************************************************************
* misc utilitys
******************************************************************************/
int console_printf(const char *format, ...)
{
    va_list args;
    int rv;
    FILE* fp;

    fp = fopen("/dev/console", "a");
    if(!fp)
	return -1;

    va_start(args, format);
    rv = vfprintf(fp, format, args);
    va_end(args);

    fclose(fp);

    return rv;
}

int systemf(const char *format, ...)
{
	va_list args;
	char cmdBuf[256];

	va_start(args, format);
	vsnprintf(cmdBuf, sizeof(cmdBuf) - 1, format, args);
	va_end(args);
	return system(cmdBuf);
}

// get system uptime either than current time to avoid side effects rised by time change
__inline__ unsigned long GetUptime(void)
{
	static struct sysinfo info;

	sysinfo(&info);
	return info.uptime;
}

/* filesize -- returns size of a regular file.
 * Returns -1 if file does not exist or is not a regular file
 * or information cannot be obtained.
 */
int FileSize(const char *filepath)
{
	struct stat Stat;
	if (stat(filepath, &Stat)) return -1;
	if (!S_ISREG(Stat.st_mode)) return -1;
	return Stat.st_size;
}

uint16_t CheckSum(uint16_t *usBuff, int32_t iSize)
{
	uint32_t ckSum = 0;
	
	for (; iSize > 1; iSize -= sizeof(uint16_t))
	{
		ckSum += *usBuff++;
	}

	if (iSize == 1)
	{
		ckSum += *(uint8_t *)usBuff;
	}
	
	ckSum = (ckSum >> 16) + (ckSum & 0xffff);
	ckSum += (ckSum >> 16);

	return (uint16_t)(~ckSum);
}

__inline__ bool_t IsBigEndian(void)
{
	static const uint32_t ulMagic = 0x01234567;
	static const uint8_t *pMagic = (const uint8_t *)&ulMagic;
	return 0x01==pMagic[0];
}
__inline__ unsigned short EndianAdjust16(unsigned short num)
{
	return IsBigEndian() ? ((num & 0xff00) >> 8) | ((num & 0xff)<<8) : num;
}
__inline__ unsigned int EndianAdjust32(unsigned int num)
{
	return  IsBigEndian() ? ((num & 0xff000000) >> 24) | ((num & 0x00ff0000) >> 8)
			|  ((num & 0x0000ff00) << 8) | ((num & 0x000000ff) << 24) : num;
}

/*****************************************************************************
* convertion routines
******************************************************************************/
void IP2Str(char *szIP, in_addr_t ulIP)
{
#if 1
	int SLen = 0;
	char * IPString = NULL;
	struct in_addr inADR;

	inADR.s_addr = ulIP;
	if ( NULL != (IPString = inet_ntoa(inADR))
	  && 15 >= (SLen = strlen(IPString))
	  && 7  <= SLen ) {
		strcpy(szIP, IPString);
	} else {
		strcpy(szIP, "0.0.0.0");
	}
#else
	strcpy(szIP, inet_ntoa(*(struct in_addr*)&ulIP));
#endif
}

bool_t Str2IP(in_addr_t *pulIP, const char *cszIP)
{
#if 1
	int SLen = strlen(cszIP);

	*pulIP = INADDR_NONE;
	if ( 15 >= SLen && 7 <= SLen ) {
		*pulIP = inet_addr(cszIP);
	}

	return (INADDR_NONE != (*pulIP));
#else
	/* Richard: Using inet_aton() may be better, because inet_addr() is an
		obsolete interface to it. */
	*pulIP = inet_addr(cszIP);
	return INADDR_NONE!= (*pulIP);
#endif
}

void Mac2Str(char *szMacAddress, MacAddr_t MacAddr)
{
	sprintf(szMacAddress, "%02x:%02x:%02x:%02x:%02x:%02x",
		MacAddr[0], MacAddr[1], MacAddr[2],
		MacAddr[3], MacAddr[4], MacAddr[5]);
}

bool_t Str2Mac(MacAddr_t MacAddr, const char *szMacAddress)
{
	int			i;
	char		szMacAddr[20];
	char		szDoubleCheck[20];

	memcpy(szMacAddr, szMacAddress, sizeof(szMacAddr));

	for (i=0; i<sizeof(MacAddr_t); i++)
	{
		MacAddr[i] = (uint8_t)strtoul(szMacAddr+i*3, NULL, 16);
	}
	Mac2Str(szDoubleCheck, MacAddr);
#ifdef	WIN32
	return 0==strnicmp(szDoubleCheck, szMacAddr, 17);
#else
	return 0==strncasecmp(szDoubleCheck, szMacAddr, 17);
#endif
}

bool_t	IsHexString(const char *cszString)
{
	uint32_t	ulLen = 0;

	while (0!=cszString[ulLen])
	{
		if (!isxdigit(cszString[ulLen]))
		{
			return FALSE;
		}
		ulLen++;
	}
	return 0==(ulLen&0x01);
}

static __inline uint8_t	_Str2Hex(const char c)
{
	if (c>='0' && c<='9')
	{
		return c-'0';
	}
	else if (c>='a' && c<='f')
	{
		return c-'a'+0x0a;
	}
	else if (c>='A' && c<='F')
	{
		return c-'A'+0x0a;
	}
	return 0xff;
}

uint32_t Str2Hex(uint8_t *pHex, const char *cszString)
{
	uint32_t	ulLen = 0;

	while (0!=cszString[0] && 0!=cszString[1])
	{
		*pHex = _Str2Hex(*cszString++);
		*pHex <<= 4;
		*pHex |= _Str2Hex(*cszString++);
		pHex++;
		ulLen++;
	}
	return ulLen;
}

void Hex2Str(char *szString, uint8_t *pHex, uint32_t ulHexLen)
{
	while (ulHexLen-->0)
	{
		sprintf(szString, "%02x", *pHex++);
		szString += 2;
	}
	*szString = '\0';
}


unsigned char mhz2ieee(unsigned int freq)
{
	if (freq == 0)
		return 0;
	//freq = freq/100000;
	if (freq == 2484)
		return 14;
	if (freq < 2484)
		return (freq - 2407) / 5;
	if (freq < 3000)
		return 15 + ((freq - 2512) / 20);
	if (freq < 5000)
	{
		if(freq > 4000)
			return (freq - 4000) / 5;
		else
			return 15 + ((freq - 2512) / 20);
	}
	return (freq - 5000) / 5;
}

int  RSSI2dBm(int iRSSI)
{
	iRSSI += MIN_RX_DBM; // RSSI ->dBm
	if (iRSSI<MIN_RX_DBM)
	{
		iRSSI = MIN_RX_DBM;
	}
	else if (iRSSI>MAX_RX_DBM)
	{
		iRSSI = MAX_RX_DBM;
	}
	return iRSSI;
}

/*****************************************************************************
* validation routines
******************************************************************************/
bool_t IsValidIP(in_addr_t ulIP)
{
	uint8_t uchFirstOctet;
	uint8_t uchLastOctet;

	ulIP = ntohl(ulIP);

	uchFirstOctet	= (uint8_t)((ulIP>>24)&0x000000FFL);
	uchLastOctet	= (uint8_t)(ulIP&0x000000FFL);

	if (uchFirstOctet<1 || uchFirstOctet>223)
	{
		printf("First octet of IP must be between 1 and 223\n");
		return FALSE;
	}
	if (uchFirstOctet==127)
	{
		printf("First octet 127 is reserved for loop back\n");
		return FALSE;
	}
	if (uchLastOctet==255 || uchLastOctet==0)
	{
		printf("Last octet 0 and 255 are reserved for broadcast\n");
		return FALSE;
	}
	return TRUE;
}

#define	IP_BIT_NUM	32
static __inline bool_t GetIPBit(in_addr_t ulIP, int iBit)
{
	return ((ulIP>>iBit)&0x01)?TRUE:FALSE;
}

bool_t IsValidNetmask(in_addr_t ulIP, in_addr_t ulIPMask)
{
	int		i = 0;
	int		iLastSetBit = 0;
	bool_t	bBit = 0;
	int		iMin = IP_BIT_NUM;
	int		iMax = 0;

	if (!IsValidIP(ulIP))
	{
		printf("Invalid IP\n");
		return FALSE;
	}

	ulIP = ntohl(ulIP);
	ulIPMask = ntohl(ulIPMask);

	// Check IPMask 0 and 1 is continuous
	for (i=0; i<IP_BIT_NUM; i++)
	{
		if (bBit!=GetIPBit(ulIPMask,i))
		{
			bBit = bBit?0:1;
			if (bBit && 0!=i)
			{
				iLastSetBit = i;
			}
			else
			{
				printf("0 and 1 of IPMask is not continue\n");
				return FALSE;
			}
		}
	}

	// Find last 1 bit of IP
	for (i=0; i<IP_BIT_NUM; i++)
	{
		if (0!=GetIPBit(ulIP,i))
		{
			iMin = i;
			break;
		}
	}

	bBit = GetIPBit(ulIP, IP_BIT_NUM-1);
	for (i=IP_BIT_NUM-2; i>=0; i--)
	{
		if (bBit!=GetIPBit(ulIP,i))
		{
			iMax = i+1;
			break;
		}
	}
	return (iMax>iMin) && (iLastSetBit>iMin) && (iLastSetBit<iMax);
}

/*****************************************************************************
* ioctl routines
******************************************************************************/
static int g_skfd =INVALID_SOCKET_HANDLE;

static void ZComIOControlClose(void)
{
	if (INVALID_SOCKET_HANDLE!=g_skfd)
	{
		close(g_skfd);
		g_skfd = INVALID_SOCKET_HANDLE;
	}
}

int ZComIOControl(int request, void* pData)
{
	if (INVALID_SOCKET_HANDLE==g_skfd)
	{
		static int	fIsFirstOpen = 1;

		g_skfd = socket(AF_INET, SOCK_DGRAM, 0);
		if (INVALID_SOCKET_HANDLE==g_skfd)
		{
			printf("%s: socket error!\n", __FUNCTION__);
			return -1;
		}
		if(fIsFirstOpen)
		{
			fIsFirstOpen = 0;
			atexit(ZComIOControlClose);
		}
	}
	return ioctl(g_skfd, request, pData);
}

int	WlanAccess(
	int iIfIndex, int	fIsRead,
	uint32_t ulMIB, void* pData, uint32_t ulSize
	)
{
	int	iReturn = -1;
	zcomioreq_t IOReq;

	//strncpy(IOReq.sDevice.ifname, "ath0", IFNAMSIZ);
	sprintf(IOReq.sDevice.ifname, "ath%d", iIfIndex);
	IOReq.u.info.MIBCommand = ulMIB;
	IOReq.u.info.dataLength = ulSize;
	IOReq.u.info.IOReturn	= -1;
	IOReq.u.info.data		= pData;

	iReturn = ZComIOControl(fIsRead?SIOCGIWREQ:SIOCSIWREQ, &IOReq);
	if (iReturn>=0 && IOReq.u.info.IOReturn>0)
	{
		return IOReq.u.info.dataLength;
	}

	if (IOReq.u.info.IOReturn <= 0)
		return IOReq.u.info.IOReturn;
	return iReturn;
}

void ZComLogAddAuthorization(
	unsigned short	usIdentification,	// refer to ZComLogID
	unsigned char	puchFromIPv4[4],	// IP Address
	unsigned char	*szUserName			// Authorization User name
	)
{
	int	fd = open(ZCOM_LOG_FULLNAME, O_RDWR);
	int	iResult = -1;
	if (-1!=fd)
	{
		ZComLogAuthReq_t ZComLogAuthReq;
		ZComLogAuthReq.usIdentification = usIdentification;
		memcpy(ZComLogAuthReq.puchFromIPv4, puchFromIPv4, 4);
		strncpy(ZComLogAuthReq.szUserName, szUserName, sizeof(ZComLogAuthReq.szUserName));
		iResult = ioctl(fd, ZCOM_LOG_IOC_AUTH, &ZComLogAuthReq);
		if (iResult < 0)
			printf("%s: ioctl error!\n", __FUNCTION__);
		close(fd);
	}
	else
	{
		printf("%s: open error!\n", __FUNCTION__);
	}
}

