/*
	nf_sipsys.h add by sipsys yansl
*/
#include <linux/types.h>
#include <linux/netfilter.h>
#include <linux/module.h>
#include <linux/skbuff.h>
#include <linux/proc_fs.h>
#include <linux/vmalloc.h>
#include <linux/stddef.h>
#include <linux/slab.h>
#include <linux/random.h>
#include <linux/jhash.h>
#include <linux/err.h>
#include <linux/percpu.h>
#include <linux/moduleparam.h>
#include <linux/notifier.h>
#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/socket.h>
#include <linux/mm.h>

#include <net/netfilter/nf_conntrack.h>
#include <net/netfilter/nf_conntrack_l3proto.h>
#include <net/netfilter/nf_conntrack_helper.h>
#include <net/netfilter/nf_conntrack_core.h>

#include "slub_def.h"

#define	IPS_FIXED_TIMEOUT_BIT 10

LIST_HEAD(unconfirmed);

struct kmem_cache *nf_conntrack_expect_cachep;

struct nf_conntrack_expect *nf_conntrack_expect_alloc(struct nf_conn *me)
{
        struct nf_conntrack_expect *new;

        new = kmem_cache_alloc(nf_conntrack_expect_cachep, GFP_ATOMIC);
        if (!new)
                return NULL;

        new->master = me;
        atomic_set(&new->use, 1);
        return new;
}

#define IN6PTON_XDIGIT          0x00010000
#define IN6PTON_DIGIT           0x00020000
#define IN6PTON_COLON_MASK      0x00700000
#define IN6PTON_COLON_1         0x00100000      /* single : requested */
#define IN6PTON_COLON_2         0x00200000      /* second : requested */
#define IN6PTON_COLON_1_2       0x00400000      /* :: requested */
#define IN6PTON_DOT             0x00800000      /* . */
#define IN6PTON_DELIM           0x10000000
#define IN6PTON_NULL            0x20000000      /* first/tail */
#define IN6PTON_UNKNOWN         0x40000000

static inline int digit2bin(char c, int delim)
{                       
        if (c == delim || c == '\0')
                return IN6PTON_DELIM;
        if (c == '.')   
                return IN6PTON_DOT;
        if (c >= '0' && c <= '9')
                return (IN6PTON_DIGIT | (c - '0'));
        return IN6PTON_UNKNOWN;
}

static inline int xdigit2bin(char c, int delim)
{       
        if (c == delim || c == '\0')
                return IN6PTON_DELIM;
        if (c == ':')
                return IN6PTON_COLON_MASK;
        if (c == '.')
                return IN6PTON_DOT;
        if (c >= '0' && c <= '9')
                return (IN6PTON_XDIGIT | IN6PTON_DIGIT| (c - '0'));
        if (c >= 'a' && c <= 'f')
                return (IN6PTON_XDIGIT | (c - 'a' + 10));
        if (c >= 'A' && c <= 'F')
                return (IN6PTON_XDIGIT | (c - 'A' + 10));
        if (delim == -1)
                return IN6PTON_DELIM;
        return IN6PTON_UNKNOWN;
}

int in4_pton(const char *src, int srclen,
             u8 *dst,
             int delim, const char **end)
{
        const char *s;
        u8 *d;
        u8 dbuf[4];
        int ret = 0;
        int i;
        int w = 0;

        if (srclen < 0)
                srclen = strlen(src);
        s = src;
        d = dbuf;
        i = 0;
        while(1) {
                int c;
                c = xdigit2bin(srclen > 0 ? *s : '\0', delim);
                if (!(c & (IN6PTON_DIGIT | IN6PTON_DOT | IN6PTON_DELIM | IN6PTON_COLON_MASK))) {
                        goto out;
                }
                if (c & (IN6PTON_DOT | IN6PTON_DELIM | IN6PTON_COLON_MASK)) {
                        if (w == 0)
                                goto out;
                        *d++ = w & 0xff;
                        w = 0;
                        i++;
                        if (c & (IN6PTON_DELIM | IN6PTON_COLON_MASK)) {
                                if (i != 4)
                                        goto out;
                                break;
                        }
                        goto cont;
                }
                w = (w * 10) + c;
                if ((w & 0xffff) > 255) {
                        goto out;
                }
cont:
                if (i >= 4)
                        goto out;
                s++;
                srclen--;
        }
        ret = 1;
        memcpy(dst, dbuf, sizeof(dbuf));
out:
        if (end)
                *end = s;
        return ret;
}

EXPORT_SYMBOL(in4_pton);

int in6_pton(const char *src, int srclen,
             u8 *dst,
             int delim, const char **end)
{
        const char *s, *tok = NULL;
        u8 *d, *dc = NULL;
        u8 dbuf[16];
        int ret = 0;
        int i;
        int state = IN6PTON_COLON_1_2 | IN6PTON_XDIGIT | IN6PTON_NULL;
        int w = 0;
                
        memset(dbuf, 0, sizeof(dbuf));
                        
        s = src;
        d = dbuf;
        if (srclen < 0) 
                srclen = strlen(src);
                        
        while (1) {     
                int c;  
                        
                c = xdigit2bin(srclen > 0 ? *s : '\0', delim);
                if (!(c & state))       
                        goto out;
                if (c & (IN6PTON_DELIM | IN6PTON_COLON_MASK)) {
                        /* process one 16-bit word */
                        if (!(state & IN6PTON_NULL)) {
                                *d++ = (w >> 8) & 0xff;
                                *d++ = w & 0xff;
                        }
                        w = 0;
                        if (c & IN6PTON_DELIM) {
                                /* We've processed last word */
                                break;
                        }
                        /*
                         * COLON_1 => XDIGIT
                         * COLON_2 => XDIGIT|DELIM
                         * COLON_1_2 => COLON_2
                         */
                        switch (state & IN6PTON_COLON_MASK) {
                        case IN6PTON_COLON_2:
                                dc = d;
                                state = IN6PTON_XDIGIT | IN6PTON_DELIM;
                                if (dc - dbuf >= sizeof(dbuf))
                                        state |= IN6PTON_NULL;
                                break;
                        case IN6PTON_COLON_1|IN6PTON_COLON_1_2:
                                state = IN6PTON_XDIGIT | IN6PTON_COLON_2;
                                break;
                        case IN6PTON_COLON_1:
                                state = IN6PTON_XDIGIT;
                                break;
                        case IN6PTON_COLON_1_2:
                                state = IN6PTON_COLON_2;
                                break;
                        default:
                                state = 0;
                        }
                        tok = s + 1;
                        goto cont;
                }

                if (c & IN6PTON_DOT) {
                        ret = in4_pton(tok ? tok : s, srclen + (int)(s - tok), d, delim, &s);
                        if (ret > 0) {
                                d += 4;
                                break;
                        }
                        goto out;
                }

                w = (w << 4) | (0xff & c);
                state = IN6PTON_COLON_1 | IN6PTON_DELIM;
                if (!(w & 0xf000)) {
                        state |= IN6PTON_XDIGIT;
                }
                if (!dc && d + 2 < dbuf + sizeof(dbuf)) {
                        state |= IN6PTON_COLON_1_2;
                        state &= ~IN6PTON_DELIM;
                }
                if (d + 2 >= dbuf + sizeof(dbuf)) {
                        state &= ~(IN6PTON_COLON_1|IN6PTON_COLON_1_2);
                }
cont:
                if ((dc && d + 4 < dbuf + sizeof(dbuf)) ||
                    d + 4 == dbuf + sizeof(dbuf)) {
                        state |= IN6PTON_DOT;
                }
                if (d >= dbuf + sizeof(dbuf)) {
                        state &= ~(IN6PTON_XDIGIT|IN6PTON_COLON_MASK);
                }
                s++;
                srclen--;
        }

        i = 15; d--;

        if (dc) {
                while(d >= dc)
                        dst[i--] = *d--;
                while(i >= dc - dbuf)
                        dst[i--] = 0;
                while(i >= 0)
                        dst[i--] = *d--;
        } else
                memcpy(dst, dbuf, sizeof(dbuf));

        ret = 1;
out:
        if (end)
                *end = s;
        return ret;
}

EXPORT_SYMBOL(in6_pton);

static __read_mostly LIST_HEAD(helpers);

struct nf_conn_help {
        /* Helper. if any */
        struct nf_conntrack_helper *helper;

        union nf_conntrack_help help;

        /* Current number of expected connections */
        unsigned int expecting;
};

#ifdef CONFIG_NF_NAT_NEEDED
static inline struct nf_conn_nat *nfct_nat(const struct nf_conn *ct)
{
        unsigned int offset = sizeof(struct nf_conn);

        if (!(ct->features & NF_CT_F_NAT))
                return NULL;

        offset = ALIGN(offset, __alignof__(struct nf_conn_nat));
        return (struct nf_conn_nat *) ((void *)ct + offset);
}

static inline struct nf_conn_help *nfct_help(const struct nf_conn *ct)
{
        unsigned int offset = sizeof(struct nf_conn);

        if (!(ct->features & NF_CT_F_HELP))
                return NULL;
        if (ct->features & NF_CT_F_NAT) {
                offset = ALIGN(offset, __alignof__(struct nf_conn_nat));
                offset += sizeof(struct nf_conn_nat);
        }

        offset = ALIGN(offset, __alignof__(struct nf_conn_help));
        return (struct nf_conn_help *) ((void *)ct + offset);
}
#else /* No NAT */
static inline struct nf_conn_help *nfct_help(const struct nf_conn *ct)
{
        unsigned int offset = sizeof(struct nf_conn);

        if (!(ct->features & NF_CT_F_HELP))
                return NULL;

        offset = ALIGN(offset, __alignof__(struct nf_conn_help));
        return (struct nf_conn_help *) ((void *)ct + offset);
}
#endif /* CONFIG_NF_NAT_NEEDED */

static inline int unhelp(struct nf_conntrack_tuple_hash *i,
                         const struct nf_conntrack_helper *me)
{
        struct nf_conn *ct = nf_ct_tuplehash_to_ctrack(i);
        struct nf_conn_help *help = nfct_help(ct);

        if (help && help->helper == me) {
                nf_conntrack_event(IPCT_HELPER, ct);
                rcu_assign_pointer(help->helper, NULL);
        }
        return 0;
}

int nf_conntrack_helper_register(struct nf_conntrack_helper *me)
{
        int size, ret;

        BUG_ON(me->timeout == 0);

        size = ALIGN(sizeof(struct nf_conn), __alignof__(struct nf_conn_help)) +  sizeof(struct nf_conn_help);
        ret = nf_conntrack_register_cache(NF_CT_F_HELP, "nf_conntrack:help", size, NULL);
        if (ret < 0) {
                printk(KERN_ERR "nf_conntrack_helper_register: Unable to create slab cache for conntracks\n");
                return ret;
        }
        write_lock_bh(&nf_conntrack_lock);
        list_add(&me->list, &helpers);
        write_unlock_bh(&nf_conntrack_lock);

        return 0;
}

void nf_conntrack_helper_unregister(struct nf_conntrack_helper *me)
{
        unsigned int i;
        struct nf_conntrack_tuple_hash *h;
        struct nf_conntrack_expect *exp, *tmp;

        /* Need write lock here, to delete helper. */
        write_lock_bh(&nf_conntrack_lock);
        list_del(&me->list);

        /* Get rid of expectations */
        list_for_each_entry_safe(exp, tmp, &nf_conntrack_expect_list, list) {
                struct nf_conn_help *help = nfct_help(exp->master);
                if ((help->helper == me || exp->helper == me) &&
                    del_timer(&exp->timeout)) {
                        nf_ct_unlink_expect(exp);
                        nf_conntrack_expect_put(exp);
                }
        }

        /* Get rid of expecteds, set helpers to NULL. */
        list_for_each_entry(h, &unconfirmed, list)
                unhelp(h, me);
        for (i = 0; i < nf_conntrack_htable_size; i++) {
                list_for_each_entry(h, &nf_conntrack_hash[i], list)
                        unhelp(h, me);
        }
        write_unlock_bh(&nf_conntrack_lock);

        /* Someone could be still looking at the helper in a bh. */
        synchronize_net();
}

/* Refresh conntrack for this many jiffies and do accounting if do_acct is 1 */
void __nf_ct_refresh_acct(struct nf_conn *ct,
                          enum ip_conntrack_info ctinfo,
                          const struct sk_buff *skb,
                          unsigned long extra_jiffies,
                          int do_acct)
{
        int event = 0;

        NF_CT_ASSERT(ct->timeout.data == (unsigned long)ct);
        NF_CT_ASSERT(skb);

        write_lock_bh(&nf_conntrack_lock);

        /* Only update if this is not a fixed timeout */
        if (test_bit(IPS_FIXED_TIMEOUT_BIT, &ct->status)) {
                write_unlock_bh(&nf_conntrack_lock);
                return;
        }

        /* If not in hash table, timer will not be active yet */
        if (!nf_ct_is_confirmed(ct)) {
                ct->timeout.expires = extra_jiffies;
                event = IPCT_REFRESH;
        } else {
                unsigned long newtime = jiffies + extra_jiffies;

                /* Only update the timeout if the new timeout is at least
                   HZ jiffies from the old timeout. Need del_timer for race
                   avoidance (may already be dying). */
                if (newtime - ct->timeout.expires >= HZ
                    && del_timer(&ct->timeout)) {
                        ct->timeout.expires = newtime;
                        add_timer(&ct->timeout);
                        event = IPCT_REFRESH;
                }
        }

#ifdef CONFIG_NF_CT_ACCT
        if (do_acct) {
                ct->counters[CTINFO2DIR(ctinfo)].packets++;
                ct->counters[CTINFO2DIR(ctinfo)].bytes +=
                        skb->len - skb_network_offset(skb);

                if ((ct->counters[CTINFO2DIR(ctinfo)].packets & 0x80000000)
                    || (ct->counters[CTINFO2DIR(ctinfo)].bytes & 0x80000000))
                        event |= IPCT_COUNTER_FILLING;
        }
#endif

        write_unlock_bh(&nf_conntrack_lock);

        /* must be unlocked when calling event cache */
        if (event)
                nf_conntrack_event_cache(event, skb);
}

/* Would two expected things clash? */
static inline int expect_clash(const struct nf_conntrack_expect *a,
                               const struct nf_conntrack_expect *b)
{
        /* Part covered by intersection of masks must be unequal,
           otherwise they clash */
        struct nf_conntrack_tuple intersect_mask;
        int count;

        intersect_mask.src.l3num = a->mask.src.l3num & b->mask.src.l3num;
        intersect_mask.src.u.all = a->mask.src.u.all & b->mask.src.u.all;
        intersect_mask.dst.u.all = a->mask.dst.u.all & b->mask.dst.u.all;
        intersect_mask.dst.protonum = a->mask.dst.protonum
                                        & b->mask.dst.protonum;

        for (count = 0; count < NF_CT_TUPLE_L3SIZE; count++){
                intersect_mask.src.u3.all[count] =
                        a->mask.src.u3.all[count] & b->mask.src.u3.all[count];
        }

        for (count = 0; count < NF_CT_TUPLE_L3SIZE; count++){
                intersect_mask.dst.u3.all[count] =
                        a->mask.dst.u3.all[count] & b->mask.dst.u3.all[count];
        }

        return nf_ct_tuple_mask_cmp(&a->tuple, &b->tuple, &intersect_mask);
}

static inline int expect_matches(const struct nf_conntrack_expect *a,
                                 const struct nf_conntrack_expect *b)
{
        return a->master == b->master
                && nf_ct_tuple_equal(&a->tuple, &b->tuple)
                && nf_ct_tuple_equal(&a->mask, &b->mask);
}

/* Race with expectations being used means we could have none to find; OK. */

static inline int refresh_timer(struct nf_conntrack_expect *i)
{       
        struct nf_conn_help *master_help = nfct_help(i->master);

        if (!del_timer(&i->timeout))
                return 0;        

        i->timeout.expires = jiffies + master_help->helper->timeout*HZ;
        add_timer(&i->timeout);
        return 1;
}

void nf_conntrack_expect_put(struct nf_conntrack_expect *exp)
{
        if (atomic_dec_and_test(&exp->use))
                kmem_cache_free(nf_conntrack_expect_cachep, exp);
}       

void nf_conntrack_expect_init(struct nf_conntrack_expect *exp, int family,
                              union nf_conntrack_address *saddr,
                              union nf_conntrack_address *daddr,
                              u_int8_t proto, __be16 *src, __be16 *dst)
{
        int len;

        if (family == AF_INET)
                len = 4;
        else
                len = 16;

        exp->flags = 0;
        exp->expectfn = NULL;
        exp->helper = NULL;
        exp->tuple.src.l3num = family;
        exp->tuple.dst.protonum = proto;
        exp->mask.src.l3num = 0xFFFF;
        exp->mask.dst.protonum = 0xFF;

        if (saddr) {
                memcpy(&exp->tuple.src.u3, saddr, len);
                if (sizeof(exp->tuple.src.u3) > len)
                        /* address needs to be cleared for nf_ct_tuple_equal */
                        memset((void *)&exp->tuple.src.u3 + len, 0x00,
                               sizeof(exp->tuple.src.u3) - len);
                memset(&exp->mask.src.u3, 0xFF, len);
                if (sizeof(exp->mask.src.u3) > len)
                        memset((void *)&exp->mask.src.u3 + len, 0x00,
                               sizeof(exp->mask.src.u3) - len);
        } else {
                memset(&exp->tuple.src.u3, 0x00, sizeof(exp->tuple.src.u3));
                memset(&exp->mask.src.u3, 0x00, sizeof(exp->mask.src.u3));
        }
        if (daddr) {
                memcpy(&exp->tuple.dst.u3, daddr, len);
                if (sizeof(exp->tuple.dst.u3) > len)
                        /* address needs to be cleared for nf_ct_tuple_equal */
                        memset((void *)&exp->tuple.dst.u3 + len, 0x00,
                               sizeof(exp->tuple.dst.u3) - len);
                memset(&exp->mask.dst.u3, 0xFF, len);
                if (sizeof(exp->mask.dst.u3) > len)
                        memset((void *)&exp->mask.dst.u3 + len, 0x00,
                               sizeof(exp->mask.dst.u3) - len);
        } else {
                memset(&exp->tuple.dst.u3, 0x00, sizeof(exp->tuple.dst.u3));
                memset(&exp->mask.dst.u3, 0x00, sizeof(exp->mask.dst.u3));
        }

        if (src) {
                exp->tuple.src.u.all = (__force u16)*src;
                exp->mask.src.u.all = 0xFFFF;
        } else {
                exp->tuple.src.u.all = 0;
                exp->mask.src.u.all = 0;
        }

        if (dst) {
                exp->tuple.dst.u.all = (__force u16)*dst;
                exp->mask.dst.u.all = 0xFFFF;
        } else {
                exp->tuple.dst.u.all = 0;
                exp->mask.dst.u.all = 0;
        }
}

