/*
 *  Boa, an http server, users.c implements UNIX users with passwords
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include <grp.h>
#include <sys/types.h>
#include "basic_auth.h"
#ifdef GROUP_POLICY
#include "group_policy.h"
#endif
#include "users.h"
#include "realms.h"
#include "boa.h"

extern struct realm *realms;

/* Error messages. */
static char out_of_memory[] = "Out of memory.";
static char fopen_failure[] = "Failed: fopen()";
static char fclose_failure[] = "Failed: fclose()";

/*
 * List contains all defined users from UNIX files.
 */
struct user *users = NULL;

void init_users(void);
#ifndef DISABLE_DEBUG
void dump_users(void);
#endif
struct user *find_user(const char *user_name);
void parse_passwd(const char *path);
void parse_group(const char *path);
struct group *find_group_name(const char *path, const char *group_name);
static struct user *new_user(struct passwd *entry);
static struct user *insert_user(struct user *user);

/* Free all allocated memory in the list users. */
void
init_users(void)
{
     struct user *user_item;
     struct user *next_user_item;

     user_item = users;
     while (user_item != NULL) {
          free(user_item->realm_list);
	  free(user_item->passwd);
	  free(user_item->name);
	  next_user_item = user_item->next;
	  free(user_item);
	  user_item = next_user_item;
     }
     users = NULL;
}

#ifndef DISABLE_DEBUG
/* Debug, dump all user items. */
void
dump_users(void)
{
     struct user *user_item;
     int i;

     user_item = users;
     while (user_item != NULL) {
	  fprintf(stderr, "[%s] ", user_item->name);
	  fprintf(stderr, "[%s] ", user_item->passwd);
	  for (i = 0; i < user_item->no_groups; i++) {
	       fprintf(stderr, "[%d]", user_item->groups[i]);
	  }
          fprintf(stderr, " [%s]", user_item->realm_list);
#ifdef GROUP_POLICY
	  fprintf(stderr, " [%x]", user_item->policy);
#endif
	  fprintf(stderr, "\n");
	  user_item = user_item->next;
     }
}
#endif

/* Return a struct user* to a defined user_name or NULL. */
struct user*
find_user(const char *user_name)
{
     struct user *user_item;

     user_item = users;
     while (user_item != NULL) {
	  if (STR_EQ(user_item->name, user_name)) {
	       break;
	  }
	  user_item = user_item->next;
     }
     return user_item;
}

/* Add all user names, passwords and primary groups in the list users. */
void
parse_passwd(const char *path)
{
     FILE *stream;
     struct passwd *entry;
     struct user *user_item;

     if ((stream = fopen(path, "r")) == NULL) {
	  perror(fopen_failure);
	  WARN((char *)path);
	  _exit(EXIT_FAILURE);
     }

     while ((entry = fgetpwent(stream)) != NULL) {
	  user_item = new_user(entry);
	  if (user_item == NULL) {
	       perror(out_of_memory);
	       WARN(out_of_memory);
	       _exit(EXIT_FAILURE);
	  }
	  user_item = insert_user(user_item);
     }

     if (fclose(stream) != 0) {
	  perror(fclose_failure);
	  WARN((char *)path);
	  _exit(EXIT_FAILURE);
     }
}

/* Add all secondary groups required by a realm. */
void
parse_group(const char *path)
{
     FILE *stream;
     struct group *entry;
     char **member;
     int i;
     struct realm *realm_item;
     struct realm_group *group_item;
     struct user *user_item;

     if ((stream = fopen(path, "r")) == NULL) {
	  perror(fopen_failure);
	  WARN((char *)path);
	  _exit(EXIT_FAILURE);
     }

     /* for each group */
     while ((entry = fgetgrent(stream)) != NULL) {
	  realm_item = realms;
          /* for each realm */
	  while (realm_item != NULL) {
	       group_item = realm_item->groups;
               /* for each group, set the group ID */
	       while (group_item != NULL) {
		    if (STR_EQ(entry->gr_name, group_item->name)) {
			 group_item->id = entry->gr_gid;
		    }
		    group_item = group_item->next;
	       }
	       realm_item = realm_item->next;
	  }

	  i = 0;
	  member = entry->gr_mem;
          /* for each group member */
	  while (member[i] != NULL) {
	       user_item = users;
               /* for each user */
	       while (user_item != NULL) {
		    if (STR_EQ(member[i], user_item->name)) {
			 user_item->groups[user_item->no_groups++] = 
                           entry->gr_gid;
                         /* only one user matches */
                         break;
		    }
		    user_item = user_item->next;
	       }
	       i++;
	  }
     }

     user_item = users;
     while (user_item != NULL) {
          set_user_realm_list(user_item);
          user_item = user_item->next;
     } 

     if (fclose(stream) != 0) {
	  perror(fclose_failure);
	  WARN((char *)path);
	  _exit(EXIT_FAILURE);
     }
}

/* Return a struct group* for a group_name or NULL. */
struct group*
find_group_name(const char *path, const char *group_name)
{
     FILE *stream;
     struct group *entry;

     if ((stream = fopen(path, "r")) == NULL) {
	  perror(fopen_failure);
	  WARN((char *)path);
	  _exit(EXIT_FAILURE);
     }

     /* for each group */
     while ((entry = fgetgrent(stream)) != NULL) {
	  if (STR_EQ(entry->gr_name, group_name)) {
	       break;
	  }
     }

     if (fclose(stream) != 0) {
	  perror(fclose_failure);
	  WARN((char *)path);
	  _exit(EXIT_FAILURE);
     }

     return entry;
}

/* Return a struct user* to an intialised user_name or NULL. */
static struct user*
new_user(struct passwd *entry)
{
     struct user *user_item;

     user_item = (struct user *)malloc(sizeof(struct user));
     if (user_item == NULL) {
	  return NULL;
     }
     user_item->name = strdup(entry->pw_name);
     if (user_item->name == NULL) {
	  free(user_item);
	  return NULL;
     }
     user_item->passwd = strdup(entry->pw_passwd);
     if (user_item->passwd == NULL) {
	  free(user_item->name);
	  free(user_item);
	  return NULL;
     }
     user_item->no_groups = 0;
     user_item->groups[user_item->no_groups++] = entry->pw_gid;
#ifdef GROUP_POLICY
     user_item->policy = http_policy;
#endif
     user_item->next = NULL;

     return user_item;
}

/* Return a struct user* user inserted in users.*/
static struct user*
insert_user(struct user *user)
{
     struct user *user_item;

     if (users == NULL) {
	  /* Empty list, insert first. */
	  users = user;
	  user_item = users;
     } else {
	  /* Insert last. TODO sorted list scales better. */
	  user_item = users;
	  while(user_item->next != NULL) {
	       user_item = user_item->next;
	  }
	  user_item->next = user;
	  user_item = user_item->next;
     }

     return user_item;
}
