/*
 *  Boa, an http server, prot_paths.c implements protected paths
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <grp.h>
#include <sys/types.h>
#include "basic_auth.h"
#include "users.h"
#include "realms.h"
#include "prot_paths.h"
#include "boa.h"

/* Error messages. */
static char out_of_memory[] = "Out of memory.";

/*
 * List contains all protected paths from the boa configuration file.
 */
static struct prot_path *prot_paths = NULL;

void init_prot_paths(void);
#ifndef DISABLE_DEBUG
void dump_paths(void);
#endif
void add_prot_path(char *realm_name, char *path);
struct realm *is_prot_path(const char *path);
static struct prot_path *new_prot_path(const char *path, struct realm *realm);
static struct prot_path *insert_prot_path(struct prot_path *path);

/* Free all allocated memory in the list prot_paths. */
void
init_prot_paths(void)
{
     struct prot_path *path;
     struct prot_path *next_path;

     /* Free memory in the list prot_paths. */
     path = prot_paths;
     while (path != NULL) {
	  free(path->name);
	  next_path = path->next;
	  free(path);
	  path = next_path;
     }
     prot_paths = NULL;
}

#ifndef DISABLE_DEBUG
/* Debug, dump all protected path items. */
void
dump_paths(void)
{
     struct prot_path *path_item;

     path_item = prot_paths;
     while (path_item != NULL) {
	  fprintf(stderr, "[%s] ", path_item->name);
	  fprintf(stderr, "[%s]\n", path_item->realm->name);
	  path_item = path_item->next;
     }
}
#endif

/*
 * Configuration file: ProtPath <realm_name> <path>
 * Called each time the configuration file parser finds the tag.
 */
void
add_prot_path(char *realm_name, char *path)
{
     struct realm *realm_item;
     struct prot_path *path_item;

     /* Find existing realm or fail. */
     realm_item = find_realm(realm_name);
     if (realm_item == NULL) {
	  /* Log error. */
	  WARN(realm_name);
	  _exit(EXIT_FAILURE);
     }

     /* Create a new protected path. */
     path_item = new_prot_path(path, realm_item);
     if (path_item == NULL) {
	  perror(out_of_memory);
	  WARN(out_of_memory);
	  _exit(EXIT_FAILURE);
     }

     path_item = insert_prot_path(path_item);
}

/* Return the associated struct realm* to a protected path or NULL. */
struct realm*
is_prot_path(const char *path)
{
     struct prot_path *item;

     for (item = prot_paths; item != NULL; item = item->next) {
	  if (STR_PREFIX(item->name, path)) {
	       if (item->realm->category == password_prot_category) {
		    return item->realm;
	       }
	       else if (item->realm->category == anonymous_prot_category) {
		    return NULL;
	       }
	  }
     }
     return NULL;
}

/* Return a struct prot_path* to an intialised path or NULL. */
static struct prot_path*
new_prot_path(const char *path, struct realm *realm)
{
     struct prot_path *path_item;

     path_item = (struct prot_path *)malloc(sizeof(struct prot_path));
     if (path_item == NULL) {
	  return NULL;
     }
     path_item->name = strdup(path);
     if (path_item->name == NULL) {
	  free(path_item);
	  return NULL;
     }
     path_item->realm = realm;
     path_item->next = NULL;

     return path_item;
}

/* Return a struct prot_path* path inserted in prot_paths.*/
static struct prot_path*
insert_prot_path(struct prot_path *path)
{
     struct prot_path *path_item;

     /* Insert the new path element in the list of paths. */
     if (prot_paths == NULL) {
	  /* Empty list, insert first. */
	  prot_paths = path;
	  path_item = prot_paths;
     } else {
	  /* Insert last. TODO sorted list scales better. */
	  path_item = prot_paths;
	  while(path_item->next != NULL) {
	       path_item = path_item->next;
	  }
	  path_item->next = path;
	  path_item = path_item->next;
     }

     return path_item;
}
