/*
 * Atmel DataFlash driver for Atmel AT91RM9200 (Thunder)
 *
 *  Copyright (C) SAN People (Pty) Ltd
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
*/

#include <linux/module.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/pci.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/partitions.h>

#include <mach/spi.h>
#include <asm/cacheflush.h>
#include <linux/spinlock.h> // spin_lock_irqsave(), spin_lock_irqrestore()
#include <linux/semaphore.h> // sem_init(), up(), down_interruptibal()
#include <linux/interrupt.h>
#include <linux/delay.h> // msleep

static struct semaphore spiflash_mutex;
static  void haydn_write_enable(void);

#ifdef DEBUG_PRINTK
#define DEBUG(args...) printk(KERN_DEBUG args)
#else
#define DEBUG(args...)
#endif

/* ......................................................................... */
#define DATAFLASH_MAX_DEVICES	4	/* max number of dataflash devices */
#undef	DATAFLASH_ALWAYS_ADD_DEVICE	/* always add whole device when using partitions? */

/* ......................................................................... */
enum
{
	//Instruction set
	SPI_FLASH_INS_WREN		= 0x06,		// write enable
	SPI_FLASH_INS_WRDI		= 0x04,		// write disable
	SPI_FLASH_INS_RDSR		= 0x05,		// read status register
	SPI_FLASH_INS_WRSR		= 0x01,		// write status register
	SPI_FLASH_INS_READ		= 0x03,		// read data bytes
	SPI_FLASH_INS_FAST_READ	= 0x0B,		// read data bytes at higher speed
	SPI_FLASH_INS_PP			= 0x02,		// page program
	SPI_FLASH_INS_SE			= 0xD8,		// sector erase
	SPI_FLASH_INS_BE			= 0xC7		// bulk erase
};

/* ......................................................................... */
struct dataflash_local
{
	int spi;			/* SPI chip-select number */

	unsigned int page_size;		/* number of bytes per page */
	unsigned short page_offset;	/* page offset in flash address */
};


/* Detected DataFlash devices */
static struct mtd_info* mtd_devices[DATAFLASH_MAX_DEVICES];
static int nr_devices = 0;
static int spi_chip_select = 0 ;

/* ......................................................................... */

#ifdef CONFIG_MTD_PARTITIONS

#ifdef CONFIG_MTD_VATICS
static struct mtd_partition vatics_sf_default_partition[] =
{
    {
        .name       = CONFIG_MTD_VATICS_MAP_0_NAME,
        .offset     = CONFIG_MTD_VATICS_MAP_0_OFFSET,
        .size       = CONFIG_MTD_VATICS_MAP_0_SIZE,
        //.mask_flags   = MTD_WRITEABLE,
    },
    {
        .name       = CONFIG_MTD_VATICS_MAP_1_NAME,
        .offset     = CONFIG_MTD_VATICS_MAP_1_OFFSET,
        .size       = CONFIG_MTD_VATICS_MAP_1_SIZE,
        //.mask_flags   = MTD_WRITEABLE,
#if defined(CONFIG_MTD_VATICS_MAP_3_PARTITION) || defined(CONFIG_MTD_VATICS_MAP_4_PARTITION)
    }, {
        .name =     CONFIG_MTD_VATICS_MAP_2_NAME,
        .offset =   CONFIG_MTD_VATICS_MAP_2_OFFSET,
        .size =     CONFIG_MTD_VATICS_MAP_2_SIZE,
#endif

#if defined(CONFIG_MTD_VATICS_MAP_4_PARTITION)
    }, {
        .name =     CONFIG_MTD_VATICS_MAP_3_NAME,
        .offset =   CONFIG_MTD_VATICS_MAP_3_OFFSET,
        .size =     CONFIG_MTD_VATICS_MAP_3_SIZE,
#endif

    }
};

#ifdef CONFIG_MTD_VATICS_SPI_FLASH_NUM_1
#define CONFIG_MTD_VATICS_MAP_WINDOW_SIZE CONFIG_MTD_VATICS_SPI_FLASH_NO1_SIZE
#define CONFIG_MTD_VATICS_MAP_PAGE_NUM CONFIG_MTD_VATICS_NO1_PAGE_NUM
#elif defined(CONFIG_MTD_VATICS_SPI_FLASH_NUM_2)
#define CONFIG_MTD_VATICS_MAP_WINDOW_SIZE (CONFIG_MTD_VATICS_SPI_FLASH_NO1_SIZE + CONFIG_MTD_VATICS_SPI_FLASH_NO2_SIZE)
#define CONFIG_MTD_VATICS_MAP_PAGE_NUM (CONFIG_MTD_VATICS_NO1_PAGE_NUM + CONFIG_MTD_VATICS_NO2_PAGE_NUM)
#else
#error "You must choose the number of spi flashes"
#endif

/* Check if the size of partitions does not match flash size */
#if defined (CONFIG_MTD_VATICS_MAP_2_PARTITION)
#if CONFIG_MTD_VATICS_MAP_0_SIZE+CONFIG_MTD_VATICS_MAP_1_SIZE > CONFIG_MTD_VATICS_MAP_WINDOW_SIZE
#error "Total size of 2 partitions > flash size"
#endif//CONFIG_MTD_VATICS_MAP_0_SIZE+CONFIG_MTD_VATICS_MAP_1_SIZE > CONFIG_MTD_VATICS_MAP_WINDOW_SIZE

#elif defined (CONFIG_MTD_VATICS_MAP_3_PARTITION)
#if CONFIG_MTD_VATICS_MAP_0_SIZE+CONFIG_MTD_VATICS_MAP_1_SIZE+CONFIG_MTD_VATICS_MAP_2_SIZE > CONFIG_MTD_VATICS_MAP_WINDOW_SIZE
#error "Total size of 3 partitions > flash size"
#endif//CONFIG_MTD_VATICS_MAP_0_SIZE+CONFIG_MTD_VATICS_MAP_1_SIZE+CONFIG_MTD_VATICS_MAP_2_SIZE > CONFIG_MTD_VATICS_MAP_WINDOW_SIZE

#elif defined (CONFIG_MTD_VATICS_MAP_4_PARTITION)
#if CONFIG_MTD_VATICS_MAP_0_SIZE+CONFIG_MTD_VATICS_MAP_1_SIZE+CONFIG_MTD_VATICS_MAP_2_SIZE+CONFIG_MTD_VATICS_MAP_3_SIZE > CONFIG_MTD_VATICS_MAP_WINDOW_SIZE
#error "Total size of 4 partitions > flash size"
#endif//CONFIG_MTD_VATICS_MAP_0_SIZE+CONFIG_MTD_VATICS_MAP_1_SIZE+CONFIG_MTD_VATICS_MAP_2_SIZE+CONFIG_MTD_VATICS_MAP_3_SIZE > CONFIG_MTD_VATICS_MAP_WINDOW_SIZE
#endif

#else /* CONFIG_MTD_VATICS */
static struct mtd_partition static_partitions_8M[] =
{
    {
        .name       = "read-only code section",
        .offset     = 0,
        .size           = 0x400000,
        .mask_flags = MTD_WRITEABLE,
    },
    {
        .name       = "jffs2 image section",
        .offset     = 0x400000,
        .size           = 0x400000,
        //.mask_flags   = MTD_WRITEABLE,
    }

};
#endif /* CONFIG_MTD_VATICS */

#endif

/* ......................................................................... */

/* Allocate a single SPI transfer descriptor.  We're assuming that if multiple
   SPI transfers occur at the same time, spi_access_bus() will serialize them.
   If this is not valid, then either (i) each dataflash 'priv' structure
   needs it's own transfer descriptor, (ii) we lock this one, or (iii) use
   another mechanism.   */
static struct spi_transfer_list* spi_transfer_desc;

/* ......................................................................... */
//#define SPIFLASH_CPUMODE_OPERATION_ONLY

static DECLARE_COMPLETION(spiflash_rx_complete);
static DECLARE_COMPLETION(spiflash_tx_complete);

int haydn_spi_interrupt(int irq, void *dev_id)
{
	unsigned long dma_ctrl_data, interrupt_chn;

	interrupt_chn = SNPS_APBC_READ(APBC_DMA_CHN_MONITOR);
	// chn3 interrupts
	if (interrupt_chn & (0x1<<APBC_SPITX_DMA_CHN))
	{
		dma_ctrl_data = SNPS_APBC_READ(APBC_SPITX_DMA_CTRL);
		SNPS_APBC_WRITE(APBC_SPITX_DMA_CTRL, dma_ctrl_data & ~0x2);
		SNPS_APBC_WRITE(APBC_DMA_CHN_MONITOR, ~(0x1<<APBC_SPITX_DMA_CHN));
		complete(&spiflash_tx_complete);
		return IRQ_HANDLED;
	}
	// chn2 interrupts
	if (interrupt_chn & (0x1<<APBC_SPIRX_DMA_CHN))
	{
		dma_ctrl_data = SNPS_APBC_READ(APBC_SPIRX_DMA_CTRL);
		SNPS_APBC_WRITE(APBC_SPIRX_DMA_CTRL, dma_ctrl_data & ~0x2);
		SNPS_APBC_WRITE(APBC_DMA_CHN_MONITOR, ~(0x1<<APBC_SPIRX_DMA_CHN));
		complete(&spiflash_rx_complete);
		return IRQ_HANDLED;
	}
	return IRQ_NONE;
}

/* ......................................................................... */
/*
 * Return the status of the DataFlash device.
 */
static unsigned short haydn_spiflash_status(void)
{
	volatile unsigned long status;
	volatile unsigned char ret;
	
	SNPS_SPI_WRITE(SNPS_SPI_SSIENR, 0x0);		//disable SSI
	SNPS_SPI_WRITE(SNPS_SPI_SER, 0x0);			//disable slave	
	SNPS_SPI_WRITE(SNPS_SPI_IMR, 0x0);			// mask all interrupts	
	SNPS_SPI_WRITE(SNPS_SPI_BAUDR, 0x0004);	// set BaudR
	SNPS_SPI_WRITE(SNPS_SPI_CTRLR0, 0x73c7);
	SNPS_SPI_WRITE(SNPS_SPI_CTRLR1,  0x0); 

	SNPS_SPI_WRITE(SNPS_SPI_SSIENR, 0x1);	
	SNPS_SPI_WRITE(SNPS_SPI_DR, SPI_FLASH_INS_RDSR);
	SNPS_SPI_WRITE(SNPS_SPI_SER, 1 << spi_chip_select);			//enable slave	
	
	// wait for SPI TX FIFO empty
	do
	{
		status = SNPS_SPI_READ(SNPS_SPI_SR);
	}while (!(status & SNPS_SPI_SR_TFE));

	while ((SNPS_SPI_READ(SNPS_SPI_SR) & 0x8) == 0);	// wait untill RX FIFO Not Empty
	ret = SNPS_SPI_READ(SNPS_SPI_DR);	
	return ret;
}


/* ......................................................................... */

/*
 * Erase blocks of flash.
 */
static int haydn_spiflash_erase(struct mtd_info *mtd, struct erase_info *instr)
{
	struct dataflash_local *priv = (struct dataflash_local *) mtd->priv;
	volatile unsigned long status;
    unsigned long erase_addr ;

	DEBUG("$sfeb$\n") ;//Serial Flash Erase Begin
	
	/* Sanity checks */
	if (instr->addr + instr->len > mtd->size)
	{
		DEBUG("$sfee1$\n") ;//Serial Flash Erase End
		return -EINVAL;
	}
	if ((instr->len % (mtd->erasesize) != 0) || (instr->len % (priv->page_size) != 0))
	{
		DEBUG("$sfee2$\n") ;//Serial Flash Erase End
		return -EINVAL;
	}
	if ((instr->addr % (priv->page_size)) != 0)
	{
		DEBUG("$sfee3$\n") ;//Serial Flash Erase End
		return -EINVAL;
	}

	if (down_interruptible(&spiflash_mutex)) {
		DEBUG("$sfee4$\n") ;//Serial Flash Erase End
		return -ERESTARTSYS;
	}
   
	while (instr->len > 0) 
	{
        erase_addr = instr->addr ;
        
#ifdef CONFIG_MTD_VATICS_SPI_FLASH_NUM_1        
        spi_chip_select = 0 ;        
#elif defined(CONFIG_MTD_VATICS_SPI_FLASH_NUM_2)
        if(erase_addr >= CONFIG_MTD_VATICS_SPI_FLASH_NO1_SIZE) {
            spi_chip_select = 1 ;
            erase_addr -= CONFIG_MTD_VATICS_SPI_FLASH_NO1_SIZE ;            
        }
        else {
            spi_chip_select = 0 ;
        }
#endif
		haydn_write_enable();
		SNPS_SPI_WRITE(SNPS_SPI_SSIENR, 0x0);		//disable SSI
		SNPS_SPI_WRITE(SNPS_SPI_SER, 0x0);			// disable slave
		SNPS_SPI_WRITE(SNPS_SPI_CTRLR0, 0x71c7);
		SNPS_SPI_WRITE(SNPS_SPI_IMR, 0x0);			// mask all interrupts	
		SNPS_SPI_WRITE(SNPS_SPI_BAUDR, 0x0004);	// set BaudR
		SNPS_SPI_WRITE(SNPS_SPI_SSIENR, 0x1);
		SNPS_SPI_WRITE(SNPS_SPI_DR, SPI_FLASH_INS_SE);
		SNPS_SPI_WRITE(SNPS_SPI_DR, (erase_addr & 0x00FF0000) >> 16);
		SNPS_SPI_WRITE(SNPS_SPI_DR, (erase_addr & 0x0000FF00) >> 8);
		SNPS_SPI_WRITE(SNPS_SPI_DR, erase_addr & 0x000000FF);
		
		SNPS_SPI_WRITE(SNPS_SPI_SER, 1 << spi_chip_select);		

		do
		{
			status = SNPS_SPI_READ(SNPS_SPI_SR);
		}while (!(status & SNPS_SPI_SR_TFE));
		do
		{
			status = SNPS_SPI_READ(SNPS_SPI_SR);
		}while (status & SNPS_SPI_SR_BUSY);
		SNPS_SPI_WRITE(SNPS_SPI_SER, 0x0);				
		
		while(haydn_spiflash_status() & 0x1)
		{
			msleep(25);
		} // wait while WriteInProgress

		SNPS_SPI_WRITE(SNPS_SPI_SER, 0x0);                      // disable slave
		SNPS_SPI_WRITE(SNPS_SPI_SSIENR, 0x0);           //disable SSI
	
		instr->addr += priv->page_size;		/* next page */
		instr->len -= priv->page_size;
	}
	up(&spiflash_mutex);
	/* Inform MTD subsystem that erase is complete */
	instr->state = MTD_ERASE_DONE;
	if (instr->callback)
	{
		instr->callback(instr);
	}

	DEBUG("$sfee5$\n") ;//Serial Flash Erase End
	return 0;

}


static int haydn_spiflash_read_CPUMODE(loff_t from, size_t len, size_t *retlen, u_char *buf) 
{
	volatile unsigned long status;
	unsigned long nSize, length, data_num;
	unsigned long i;

	unsigned short tmpdata;
    unsigned short* tmpbuf;
    unsigned long odd_len = 0;

    unsigned long read_addr ;

	*retlen = 0;

	tmpbuf = (unsigned short *)buf;	
	do{	
		nSize = (len > 16*2) ? 16*2 : len; // FIFO depth is 16. FIFO width is 16 bits (2 byte).

        read_addr = (unsigned long)from ;
        
#ifdef CONFIG_MTD_VATICS_SPI_FLASH_NUM_1
        spi_chip_select = 0 ;        
#elif defined(CONFIG_MTD_VATICS_SPI_FLASH_NUM_2)
        if(read_addr >= CONFIG_MTD_VATICS_SPI_FLASH_NO1_SIZE) {
            spi_chip_select = 1 ;
            read_addr -= CONFIG_MTD_VATICS_SPI_FLASH_NO1_SIZE ;
        }
        else if((read_addr + nSize) >= CONFIG_MTD_VATICS_SPI_FLASH_NO1_SIZE) {
            //[jam] : If this READ will cross the boundary in the process,
            //we force it to read only 1st chip in this turn.
            //i.e. In this case, from is the addr of chip "1st"
            nSize = CONFIG_MTD_VATICS_SPI_FLASH_NO1_SIZE - read_addr ;            
            spi_chip_select = 0 ;            
        }
        else {
            spi_chip_select = 0 ;
        }
#endif  

		if (nSize & 0x1)
			odd_len = 1;
		else
			odd_len = 0;

		// config spi
		SNPS_SPI_WRITE(SNPS_SPI_SSIENR, 0x0);		//disable SSI
		SNPS_SPI_WRITE(SNPS_SPI_DMACR, 0x0);		// disable DMA
		SNPS_SPI_WRITE(SNPS_SPI_SER, 0x0);			// disable slave
		SNPS_SPI_WRITE(SNPS_SPI_IMR, 0x0);			// mask all interrupts
		
		// In FAST_READ case, we should transmit 5 bytes (one byte FAST_READ command, 3 bytes address, and one dummy byte)
		// to TX FIFO. Because the FIFO width is 2 bytes, there's one more garbage byte being sent into FIFO in this case.
		// That would cause the first byte read from serial flash to be dropped by SPI (in EEPROM read mode).
		// In the event, the read data will shift one byte. So we workaround this case by substracting the "from" address by one byte.
		// If the "from" address is zero, we would take this as a special case and handle this by READ command rather than FAST_READ.
		SNPS_SPI_WRITE(SNPS_SPI_BAUDR, (read_addr) ? 0x0004 : 0x0004);	// set BaudR (4 for READ command, 2 for FAST_READ command)
		SNPS_SPI_WRITE(SNPS_SPI_CTRLR0, 0x73cF);	// DFS = 16 bit
		// Because the DFS is 16 bits (2 byte), data number should be an even number.
		// If the data number is an odd, we would take this as a special case and handle this case by reading one more data.
		if (!odd_len)
			SNPS_SPI_WRITE(SNPS_SPI_CTRLR1,  (nSize>>1)-1); 
		else //if (!odd_len)
			SNPS_SPI_WRITE(SNPS_SPI_CTRLR1,  ((nSize+1)>>1)-1); 			
		SNPS_SPI_WRITE(SNPS_SPI_RXFTLR, 0x3);

		// enable SSIC and send read command, address
		SNPS_SPI_WRITE(SNPS_SPI_SSIENR, 0x1);
		if (read_addr)
		{
			tmpdata = (SPI_FLASH_INS_FAST_READ<<8) | (((read_addr-1) & 0x00FF0000)>>16);
			SNPS_SPI_WRITE(SNPS_SPI_DR, tmpdata);
			tmpdata = (read_addr-1) & 0x0000FFFF;			
			SNPS_SPI_WRITE(SNPS_SPI_DR, tmpdata);
			SNPS_SPI_WRITE(SNPS_SPI_DR, tmpdata);		// dummy data
		}
		else //if (from)
		{
			tmpdata = (SPI_FLASH_INS_READ<<8) | ((read_addr & 0x00FF0000)>>16);
			SNPS_SPI_WRITE(SNPS_SPI_DR, tmpdata);
			tmpdata = read_addr & 0x0000FFFF;			
			SNPS_SPI_WRITE(SNPS_SPI_DR, tmpdata);
		}		
		// enable SER
		SNPS_SPI_WRITE(SNPS_SPI_SER, 1 << spi_chip_select);
		
		// wait for SPI TX FIFO empty
		do
		{
			status = SNPS_SPI_READ(SNPS_SPI_SR);
		}while (!(status & SNPS_SPI_SR_TFE));

		//  RX
		length = (odd_len) ? (nSize-1) : nSize;
		while(length > 0)
		{
			data_num = SNPS_SPI_READ(SNPS_SPI_RXFLR);
			for (i = 0; (i<data_num) && (length>0); i++)
			{
				tmpdata = SNPS_SPI_READ(SNPS_SPI_DR);
				// The 16 bit data stored in FIFO is arranged by Big-Endian.
				// We swapped it to be Little-Endian.
				*(tmpbuf++) = ((tmpdata&0xFF)<<8) | ((tmpdata&0xFF00)>>8);
				length = length - 2;
			}
		}
		if (odd_len)
		{
			while(!(SNPS_SPI_READ(SNPS_SPI_SR) & SNPS_SPI_SR_RFNE));
			// If the data num is an odd number, we will read one more data for the 16 bits FIFO width.
			// So we drop the redundant data here.
			*((u_char *)tmpbuf++) = (u_char)(SNPS_SPI_READ(SNPS_SPI_DR)>>8);
		}
		len -= nSize;
		from += nSize;
		*retlen += nSize;
	}while (len > 0);
	SNPS_SPI_WRITE(SNPS_SPI_SER, 0x0);			// disable slave
	
	return 0 ;
}

/*
 * Read from the DataFlash device.
 *   from   : Start offset in flash device
 *   len    : Amount to read
 *   retlen : About of data actually read
 *   buf    : Buffer containing the data
 */
static unsigned long descAddrArray[50] ; 
static int haydn_spiflash_read(struct mtd_info *mtd, loff_t from, size_t len, size_t *retlen, u_char *buf)
{
	volatile unsigned long status;
	unsigned long nSize, data_num;
	unsigned long i, alignment;

	unsigned short tmpdata;
	unsigned long odd_len = 0;

	unsigned long DMATransfer, CPUTransfer = 0, openRXDMA = 0, DMADestAddr, dmaSize, dmaTurn = 0;     
	unsigned long *descAddr = NULL;
	unsigned long descLen = 0;
	volatile unsigned long flags;

    unsigned long read_addr ;

	DEBUG("#sfrb#\n") ;//Serial Flash Read Begin
	
	*retlen = 0;
	// Sanity checks 
	if (!len) {
		DEBUG("$sfre1$\n") ;//Serial Flash Read End
		return 0;
	}
	if (from + len > mtd->size) {
		DEBUG("$sfre2$\n") ;//Serial Flash Read End
		return -EINVAL;
	}
	if (down_interruptible(&spiflash_mutex)) {
		DEBUG("$sfre3$\n") ;//Serial Flash Read End
		return -ERESTARTSYS;
	}

#ifdef SPIFLASH_CPUMODE_OPERATION_ONLY
    haydn_spiflash_read_CPUMODE(from, len, retlen, buf) ;
#else
	dmac_flush_range((unsigned long)(buf), (unsigned long)(buf+len-1));

	// [note]: DMA is 2-bytes aligned now.
	// Because address of "buf" may not be 2 byte aligned, we should take care of the non-2byte-aligned case.
	alignment = 4-(((unsigned long)buf)&0x3);
	if((alignment != 4) && (len >= 16*2)) {
		haydn_spiflash_read_CPUMODE( from, alignment, retlen, buf);
		// Update the related variables.
		buf += alignment;
		len -= alignment;
		from += alignment;
	}

	do {			
		nSize = (len > (64*1024*2)) ? (64*1024*2) : len; // SNPS SPI CTRL1 : maximum value is 64K(data number). FIFO width is 16 bits (data size).

        read_addr = (unsigned long)from ;

#ifdef CONFIG_MTD_VATICS_SPI_FLASH_NUM_1
        spi_chip_select = 0 ;        
#elif defined(CONFIG_MTD_VATICS_SPI_FLASH_NUM_2)
        if(read_addr >= CONFIG_MTD_VATICS_SPI_FLASH_NO1_SIZE) {
            spi_chip_select = 1 ;
            read_addr -= CONFIG_MTD_VATICS_SPI_FLASH_NO1_SIZE ;
        }
        else if((read_addr + nSize) >= CONFIG_MTD_VATICS_SPI_FLASH_NO1_SIZE) {
            //[jam] : If this READ will cross the boundary in the process,
            //we force it to read only 1st chip in this turn.
            //i.e. In this case, from is the addr of chip "1st"
            nSize = CONFIG_MTD_VATICS_SPI_FLASH_NO1_SIZE - read_addr ;            
            spi_chip_select = 0 ;            
        }
        else {
            spi_chip_select = 0 ;
        }
#endif        
        
		odd_len = nSize & 0x1;
	
		local_irq_save(flags);

		// config spi
		SNPS_SPI_WRITE(SNPS_SPI_SSIENR, 0x0);		//disable SSI
		SNPS_SPI_WRITE(SNPS_SPI_DMACR, 0x0);		// disable DMA
		SNPS_SPI_WRITE(SNPS_SPI_SER, 0x0);			// disable slave
		SNPS_SPI_WRITE(SNPS_SPI_IMR, 0x0);			// mask all interrupts	
		// In FAST_READ case, we should transmit 5 bytes (one byte FAST_READ command, 3 bytes address, and one dummy byte)
		// to TX FIFO. Because the FIFO width is 2 bytes, there's one more garbage byte being sent into FIFO in this case.
		// That would cause the first byte read from serial flash to be dropped by SPI (in EEPROM read mode).
		// In the event, the read data will shift one byte. So we workaround this case by substracting the "from" address by one byte.
		// If the "from" address is zero, we would take this as a special case and handle this by READ command rather than FAST_READ.
		SNPS_SPI_WRITE(SNPS_SPI_BAUDR, (read_addr) ? 0x0004 : 0x0004);       // set BaudR (4 for READ command, 2 for FAST_READ command)
		SNPS_SPI_WRITE(SNPS_SPI_CTRLR0, 0x73cF); // DFS = 16 bit
		// Because the DFS is 16 bits (2 byte), data number should be an even number.
		// If the data number is an odd, we would take this as a special case and handle this case by reading one more data.
		if (!odd_len)
			SNPS_SPI_WRITE(SNPS_SPI_CTRLR1, (nSize>>1)-1);
		else //if (!odd_len)
			SNPS_SPI_WRITE(SNPS_SPI_CTRLR1, ((nSize+1)>>1)-1);
		SNPS_SPI_WRITE(SNPS_SPI_RXFTLR, 31); //RxThreshold

		DMATransfer = nSize ;
		CPUTransfer = DMATransfer % (16*2); // CPU transmit, burst: INCR16
		DMATransfer -= CPUTransfer;	// DMA transmit
		openRXDMA = (DMATransfer == 0) ? 0 : 1;
		DMADestAddr = virt_to_bus(buf) ;
		dmaTurn = 0;
		if (openRXDMA) {
			descLen = 16 * (DMATransfer / (0xFFF*16*2)); // burst: INCR16
			if (descLen > 0) {
				//descAddr = kmalloc(descLen, GFP_ATOMIC);
				descAddr = descAddrArray ;
			}
			while (DMATransfer > 0) {			
				dmaSize = (DMATransfer > (0xFFF*16*2)) ? (0xFFF*16*2) : DMATransfer; // APB DMA maximum transfer unit, burst: INCR16
				DMATransfer -= dmaSize;
				if (dmaTurn == 0) {
					SNPS_APBC_WRITE(APBC_SPIRX_DMA_SRC_ADDR, MOZART_SPIC_BASE + SNPS_SPI_DR);
					SNPS_APBC_WRITE(APBC_SPIRX_DMA_DES_ADDR, DMADestAddr);
					SNPS_APBC_WRITE(APBC_SPIRX_DMA_LLP, (DMATransfer) ? virt_to_bus(descAddr) : 0x0);			
					SNPS_APBC_WRITE(APBC_SPIRX_DMA_CTRL, 0x714B5 | (dmaSize/(16*2))<<20);//handle 16 bits FIFO width, burst: INCR16
				}
				else { // prepare LLP				
					descAddr[4*(dmaTurn-1)+3] = 0x714B5 | (dmaSize/(16*2))<<20; //CTRL, burst: INCR16
					descAddr[4*(dmaTurn-1)+2] = (DMATransfer) ? (virt_to_bus(descAddr)+16*dmaTurn) : 0x0; //LLP
					descAddr[4*(dmaTurn-1)+1] = DMADestAddr; //DEST
					descAddr[4*(dmaTurn-1)] = MOZART_SPIC_BASE + SNPS_SPI_DR; //SRC
				}
				DMADestAddr += dmaSize;
				dmaTurn++;
			}
			SNPS_SPI_WRITE(SNPS_SPI_DMARDLR, 0xF); //burst:INCR16
			if (descLen > 0) {
				dmac_flush_range((unsigned long)descAddr, (unsigned long)(descAddr + descLen));
			}
		}

		// enable SSIC and send read command, address
		SNPS_SPI_WRITE(SNPS_SPI_SSIENR, 0x1);	
		if (read_addr) {
			tmpdata = (SPI_FLASH_INS_FAST_READ<<8) | (((read_addr-1) & 0x00FF0000)>>16);
			SNPS_SPI_WRITE(SNPS_SPI_DR, tmpdata);
			tmpdata = (read_addr-1) & 0x0000FFFF;
			SNPS_SPI_WRITE(SNPS_SPI_DR, tmpdata);
			SNPS_SPI_WRITE(SNPS_SPI_DR, tmpdata);           // dummy data
		}
		else { //if (from) read from address 0x0, we use READ instruction              
			tmpdata = (SPI_FLASH_INS_READ<<8) | ((read_addr & 0x00FF0000)>>16);
			SNPS_SPI_WRITE(SNPS_SPI_DR, tmpdata);
			tmpdata = read_addr & 0x0000FFFF;
			SNPS_SPI_WRITE(SNPS_SPI_DR, tmpdata);
		}
		
		// enable SSIC RX DMA
		SNPS_SPI_WRITE(SNPS_SPI_DMACR, 0x1);
		// enable SSIC RX FIFO Full Interrupt, evelyn added at 05.16.09
		SNPS_SPI_WRITE(SNPS_SPI_IMR, 0x10);		
		// enable slave		
		SNPS_SPI_WRITE(SNPS_SPI_SER, 1 << spi_chip_select);
		
		// wait for SPI TX FIFO empty
		do {
			status = SNPS_SPI_READ(SNPS_SPI_SR);
		}while (!(status & SNPS_SPI_SR_TFE));

		local_irq_restore(flags);
		//  RX
		if (openRXDMA) {	
			wait_for_completion(&spiflash_rx_complete);
			//kfree(descAddr);
			descAddr = NULL ;
		}
		SNPS_SPI_WRITE(SNPS_SPI_DMACR, 0x0);		// disable dma
		SNPS_SPI_WRITE(SNPS_SPI_IMR, 0x0);			// disable SSIC RX FIFO Full Interrupt, evelyn added at 05.16.09	
		//handle 16 bits and odd data number	
		//if (openRXDMA)
		//	dmac_inv_range((unsigned long)buf, (unsigned long)(buf + nSize - CPUTransfer-1));

		buf = buf + nSize - CPUTransfer;
		CPUTransfer = (odd_len) ? (CPUTransfer-1) : CPUTransfer;

		while(CPUTransfer > 0) {
			data_num = SNPS_SPI_READ(SNPS_SPI_RXFLR);
			for (i = 0; (i < data_num) && (CPUTransfer > 0); i++) {	
				tmpdata = SNPS_SPI_READ(SNPS_SPI_DR);
				// The 16 bit data stored in FIFO is arranged by Big-Endian.
				// We swapped it to be Little-Endian.
				*(buf++) = (tmpdata&0xFF00)>>8;
				*(buf++) = tmpdata&0xFF;
				CPUTransfer = CPUTransfer - 2;
			}
		}		
		
		if (odd_len) {
			while(!(SNPS_SPI_READ(SNPS_SPI_SR) & SNPS_SPI_SR_RFNE));
			// If the data num is an odd number, we will read one more data for the 16 bits FIFO width.
			// So we drop the redundant data here.
			*(buf++) = (u_char)(SNPS_SPI_READ(SNPS_SPI_DR)>>8);
		}
		SNPS_SPI_WRITE(SNPS_SPI_SER, 0x0);			// disable slave		
		SNPS_SPI_WRITE(SNPS_SPI_SSIENR, 0x0);		//disable SSI		

		len -= nSize;
		from += nSize;
		*retlen += nSize;	
	}while (len > 0);
#endif    
	up(&spiflash_mutex);

	DEBUG("$sfre4$\n") ;//Serial Flash Read End

	return 0 ;
}

static  void haydn_write_enable()
{
	volatile unsigned long status;
	
	SNPS_SPI_WRITE(SNPS_SPI_SSIENR, 0x0);		//disable SSI
	SNPS_SPI_WRITE(SNPS_SPI_SER, 0x0);			// disable slave
	SNPS_SPI_WRITE(SNPS_SPI_CTRLR0, 0x71c7);
	SNPS_SPI_WRITE(SNPS_SPI_IMR, 0x0);			// mask all interrupts	
	SNPS_SPI_WRITE(SNPS_SPI_BAUDR, 0x0004);	// set BaudR

	SNPS_SPI_WRITE(SNPS_SPI_SSIENR, 0x1);//0x08
	SNPS_SPI_WRITE(SNPS_SPI_DR, SPI_FLASH_INS_WREN);
	SNPS_SPI_WRITE(SNPS_SPI_SER, 1 << spi_chip_select);//0x10

	do
	{
		status = SNPS_SPI_READ(SNPS_SPI_SR);
	}while (!(status & SNPS_SPI_SR_TFE));
	do
	{
		status = SNPS_SPI_READ(SNPS_SPI_SR);
	}while (status & SNPS_SPI_SR_BUSY);

	SNPS_SPI_WRITE(SNPS_SPI_SER, 0x0);			// disable slave
}


static int haydn_spiflash_write_CPUMODE(loff_t to, size_t len, size_t *retlen, const u_char *buf)
{
	unsigned long nSize, i;
	volatile unsigned long status;
	unsigned short tmpdata;
	unsigned short* tmpbuf;
	unsigned long odd_len = 0;    

#ifdef CONFIG_MTD_VATICS_SPI_FLASH_NUM_1
        spi_chip_select = 0 ;        
#elif defined(CONFIG_MTD_VATICS_SPI_FLASH_NUM_2)
        if(to >= CONFIG_MTD_VATICS_SPI_FLASH_NO1_SIZE) {
            spi_chip_select = 1 ;
            to -= CONFIG_MTD_VATICS_SPI_FLASH_NO1_SIZE ;
        }
        else {
            spi_chip_select = 0 ;
        }
#endif        

    //printk("cpu() to = 0x%08lx, len = %d, retlen = %d\n", (unsigned long)to, len, *retlen) ;

	tmpbuf = (unsigned short*)buf;
	do {
#ifdef SPIFLASH_CPUMODE_OPERATION_ONLY     
        // We need to send 4 bytes into FIFO for PP command.
		// For the FIFO depth is 16, and FIFO width is 2 bytes, there are only 14 spaces to fill data in FIFO.
		nSize = (len > 14*2) ? 14*2 : len;
		// Page_Program command only supports one page inside programming.
		// If we want to write between pages, we have to send PP command in each page program.		
		if ((nSize+(to&255)) > 256)
			nSize = 256-(to&255);

        odd_len = (nSize&0x1) ? 1 : 0;
#else
        nSize = len;

		odd_len = nSize & 0x1;
#endif  				
		haydn_write_enable();

		SNPS_SPI_WRITE(SNPS_SPI_SSIENR, 0x0);		//disable SSI
		SNPS_SPI_WRITE(SNPS_SPI_SER, 0x0);			// disable slave
		SNPS_SPI_WRITE(SNPS_SPI_CTRLR0, (odd_len) ? 0x71c7 : 0x71cF);
		SNPS_SPI_WRITE(SNPS_SPI_IMR, 0x0);			// mask all interrupts	
		SNPS_SPI_WRITE(SNPS_SPI_BAUDR, 0x0004);	// set BaudR
		SNPS_SPI_WRITE(SNPS_SPI_SSIENR, 0x1);

		if (odd_len) {
			// We coudn't write odd number of data to 2 byte width FIFO. 
			// Therefore, we write one byte to 8bit width configuration FIFO in this case. 
			nSize = 1;	// only write one byte
			// send PP command and address
			SNPS_SPI_WRITE(SNPS_SPI_DR, SPI_FLASH_INS_PP);
			SNPS_SPI_WRITE(SNPS_SPI_DR, (to & 0x00FF0000) >>16);
			SNPS_SPI_WRITE(SNPS_SPI_DR, (to & 0x0000FF00) >>8);
			SNPS_SPI_WRITE(SNPS_SPI_DR, to & 0x000000FF);
			// send data
			SNPS_SPI_WRITE(SNPS_SPI_DR, *((u_char*)tmpbuf));
			tmpbuf = (unsigned short*)((u_char*)tmpbuf+1);			
		}
		else {
			// send PP command and address
			tmpdata = (SPI_FLASH_INS_PP<<8) | ((to & 0x00FF0000) >>16);
			SNPS_SPI_WRITE(SNPS_SPI_DR, tmpdata);
			tmpdata = to & 0x0000FFFF;			
			SNPS_SPI_WRITE(SNPS_SPI_DR, tmpdata);
			// send data
			for (i = 0; i < nSize; i = i + 2) {
				tmpdata = *tmpbuf++;
				SNPS_SPI_WRITE(SNPS_SPI_DR, ((tmpdata&0xFF)<<8) | ((tmpdata&0xFF00)>>8));
			}
		}
		SNPS_SPI_WRITE(SNPS_SPI_SER, 1 << spi_chip_select);
		do {
			status = SNPS_SPI_READ(SNPS_SPI_SR);
		}while (!(status & SNPS_SPI_SR_TFE));
		do {
			status = SNPS_SPI_READ(SNPS_SPI_SR);
		}while (status & SNPS_SPI_SR_BUSY);
		while(haydn_spiflash_status() & 0x1);  			// wait while WriteInProgress
		SNPS_SPI_WRITE(SNPS_SPI_SER, 0x0);			// disable slave		
		SNPS_SPI_WRITE(SNPS_SPI_SSIENR, 0x0);		//disable SSI

		to += nSize;
		len -= nSize;
		*retlen += nSize;
        //printk(" len = %d, retlen = %d\n", len, *retlen) ;
	} while (len > 0);
	SNPS_SPI_WRITE(SNPS_SPI_SER, 0x0);			// disable slave

	return 0 ;
}

/*
 * Write to the DataFlash device.
 *   to     : Start offset in flash device
 *   len    : Amount to write
 *   retlen : Amount of data actually written
 *   buf    : Buffer containing the data
 */
static int haydn_spiflash_write(struct mtd_info *mtd, loff_t to, size_t len, size_t *retlen, const u_char *buf)
{
	unsigned long nSize;
	volatile unsigned long status;
	unsigned long flags;

	unsigned short tmpdata;
	unsigned long DMATransfer = 0, CPUTransfer = 0 ;
	unsigned long i;

    unsigned long write_addr ;

	DEBUG("$sfwb$\n") ;//Serial Flash Write Begin
	
	if (!len) {
		DEBUG("$sfwe1$\n") ;//Serial Flash Write End
		return 0;
	}
	if (to + len > mtd->size) {
		DEBUG("$sfwe2$\n") ;//Serial Flash Write End
		return -EINVAL;
	}
	if ((unsigned long)buf & 0x1)
		DEBUG("buf address is not 2 bytes aligned..\n");
	
	if (down_interruptible(&spiflash_mutex)) {
		DEBUG("$sfwe3$\n") ;//Serial Flash Write End
		return -ERESTARTSYS;
	}
	*retlen = 0;	

#ifdef SPIFLASH_CPUMODE_OPERATION_ONLY    
    haydn_spiflash_write_CPUMODE(to, len, retlen, buf);
#else
	dmac_flush_range( (unsigned long)(buf), (unsigned long)(buf + len -1));
	do{
        write_addr = (unsigned long)to ;
        
#ifdef CONFIG_MTD_VATICS_SPI_FLASH_NUM_1
        spi_chip_select = 0 ;        
#elif defined(CONFIG_MTD_VATICS_SPI_FLASH_NUM_2)
        if(write_addr >= CONFIG_MTD_VATICS_SPI_FLASH_NO1_SIZE) {
            spi_chip_select = 1 ;
            write_addr -= CONFIG_MTD_VATICS_SPI_FLASH_NO1_SIZE ;
        }
        else {
            spi_chip_select = 0 ;
        }
#endif     

		// Page_Program command only supports one page inside programming.
		// If we want to write between pages, we have to send PP command in each page program.	
		nSize = len;
		if ((nSize+(write_addr&255)) > 256) {
			nSize = 256 - (write_addr&255);
		}

		/* alignment of source address */
		if (virt_to_bus(buf) & 0x1)
		{
			haydn_spiflash_write_CPUMODE(write_addr, 1, retlen, buf);
			buf++;
			to++;
			write_addr++;
			len--;
			nSize--;
		}

		DMATransfer = nSize ;
		CPUTransfer = nSize%(16*2);	// data_size = 2 bte, INCR=16
		DMATransfer -= CPUTransfer;
		if ( DMATransfer != 0 ) {
			local_irq_save(flags);
			
			// prepare APBC DMA
			SNPS_APBC_WRITE(APBC_SPITX_DMA_SRC_ADDR, virt_to_bus(buf));
			SNPS_APBC_WRITE(APBC_SPITX_DMA_DES_ADDR, MOZART_SPIC_BASE + SNPS_SPI_DR);
			SNPS_APBC_WRITE(APBC_SPITX_DMA_CTRL, 0x71175 | (DMATransfer/(16*2))<<20);						
			SNPS_APBC_WRITE(APBC_SPITX_DMA_LLP, 0x0); // if use linked list description
			SNPS_SPI_WRITE(SNPS_SPI_DMATDLR, 0x10); 
			haydn_write_enable();

			SNPS_SPI_WRITE(SNPS_SPI_SSIENR, 0x0);		//disable SSI
			SNPS_SPI_WRITE(SNPS_SPI_SER, 0x0);			// disable slave
			SNPS_SPI_WRITE(SNPS_SPI_CTRLR0, 0x71cF);	// data_size = 16bit
			SNPS_SPI_WRITE(SNPS_SPI_IMR, 0x0);			// mask all interrupts	
			SNPS_SPI_WRITE(SNPS_SPI_TXFTLR, 0x1);		// TFE is asserted when TXFLR=1, evelyn added at 05.16.09			
			SNPS_SPI_WRITE(SNPS_SPI_BAUDR, 0x0004);	// set BaudR
			SNPS_SPI_WRITE(SNPS_SPI_SSIENR, 0x1);

			// send PP command and address
			tmpdata = (SPI_FLASH_INS_PP<<8) | ((write_addr & 0x00FF0000) >>16);
			SNPS_SPI_WRITE(SNPS_SPI_DR, tmpdata);
			tmpdata = write_addr & 0x0000FFFF;			
			SNPS_SPI_WRITE(SNPS_SPI_DR, tmpdata);
			SNPS_SPI_WRITE(SNPS_SPI_IMR, 0x1);			// TFE interrupt enable, evelyn added at 05.16.09
			
			SNPS_SPI_WRITE(SNPS_SPI_DMACR, 0x2);
			SNPS_SPI_WRITE(SNPS_SPI_SER, 1 << spi_chip_select);

			local_irq_restore(flags);
			wait_for_completion(&spiflash_tx_complete);
			SNPS_SPI_WRITE(SNPS_SPI_IMR, 0x0);			// TFE interrupt disable, evelyn added at 05.16.09

			do {
				status = SNPS_SPI_READ(SNPS_SPI_SR);
			}while (!(status & SNPS_SPI_SR_TFE));
			do {
				status = SNPS_SPI_READ(SNPS_SPI_SR);
			}while (status & SNPS_SPI_SR_BUSY);	

			while(haydn_spiflash_status() & 0x1);			// wait while WriteInProgress

			SNPS_SPI_WRITE(SNPS_SPI_DMACR, 0x0);		// disable dma
			SNPS_SPI_WRITE(SNPS_SPI_SER, 0x0);			// disable slave		
			SNPS_SPI_WRITE(SNPS_SPI_SSIENR, 0x0);		//disable SSI

			//update the related settings
			buf += DMATransfer ;
			len -= DMATransfer ;
			*retlen += DMATransfer ;
			to += DMATransfer ;
		}	

		//handle odd length
		if ( CPUTransfer != 0 ) {
			haydn_spiflash_write_CPUMODE( to, CPUTransfer, retlen, buf);
			buf += CPUTransfer ;
			len -= CPUTransfer;
			to += CPUTransfer ;
		}

	} while (len > 0);
#endif    
	up(&spiflash_mutex);

	DEBUG("$sfwe4$\n") ;//Serial Flash Write End

	return 0 ;
}

/* ......................................................................... */

/*
 * Initialize and register DataFlash device with MTD subsystem.
 */
static int __init add_dataflash(int channel, char *name, int IDsize,
		int nr_pages, int pagesize, int pageoffset)
{
	struct mtd_info *device;
	struct dataflash_local *priv;
#ifdef CONFIG_MTD_PARTITIONS
	struct mtd_partition *mtd_parts = 0;
	int mtd_parts_nr = 0;
#endif

	if (nr_devices >= DATAFLASH_MAX_DEVICES) {
		//printk(KERN_ERR "at91_dataflash: Too many devices detected\n");
		return 0;
	}

	device = kmalloc(sizeof(struct mtd_info) + strlen(name) + 8, GFP_KERNEL);
	if (!device)
		return -ENOMEM;
	memset(device, 0, sizeof(struct mtd_info));

	device->name = (char *)&device[1];
	sprintf(device->name, "%s.spi%d", name, channel);
	device->size = nr_pages * pagesize;
	device->erasesize = pagesize;
	device->writesize = pagesize;

	device->owner = THIS_MODULE;
	device->type = MTD_NORFLASH;
	device->flags = MTD_CAP_NORFLASH;
	device->erase = haydn_spiflash_erase;
	device->read = haydn_spiflash_read;
	device->write = haydn_spiflash_write;

	priv = (struct dataflash_local *) kmalloc(sizeof(struct dataflash_local), GFP_KERNEL);
	if (!priv) {
		kfree(device);
		return -ENOMEM;
	}
	memset(priv, 0, sizeof(struct dataflash_local));

	priv->spi = channel;
	priv->page_size = pagesize;
	priv->page_offset = pageoffset;
	device->priv = priv;

	mtd_devices[nr_devices] = device;
	nr_devices++;
	//printk("haydn_spiflash: %s detected [spi%i] (%i bytes)\n", name, channel, device->size);

#ifdef CONFIG_MTD_VATICS_SPI_FLASH_NUM_1
	printk("We have 1 Serial Flash!\n") ;
#elif defined(CONFIG_MTD_VATICS_SPI_FLASH_NUM_2)
	printk("We have 2 Serial Flashes!\n") ;
#endif

#ifdef CONFIG_MTD_PARTITIONS
#if 0
	mtd_parts_nr = parse_mtd_partitions(device, part_probes, &mtd_parts, 0);
#endif
	if (mtd_parts_nr <= 0) {
#ifndef CONFIG_MTD_VATICS
		switch (IDsize) {
/*
			case SZ_2M:
				mtd_parts = static_partitions_2M;
				mtd_parts_nr = ARRAY_SIZE(static_partitions_2M);
				break;
			case SZ_4M:
				mtd_parts = static_partitions_4M;
				mtd_parts_nr = ARRAY_SIZE(static_partitions_4M);
				break;
*/
			case SZ_8M:
				mtd_parts = static_partitions_8M;
				mtd_parts_nr = ARRAY_SIZE(static_partitions_8M);
				break;
		}
#else
		printk(KERN_DEBUG "Using the default nand partition..\n") ;
        mtd_parts = vatics_sf_default_partition;
        mtd_parts_nr = ARRAY_SIZE(vatics_sf_default_partition);
#endif /* CONFIG_MTD_VATICS */
	}

	if (mtd_parts_nr > 0) {
#ifdef DATAFLASH_ALWAYS_ADD_DEVICE
		add_mtd_device(device);
#endif
		return add_mtd_partitions(device, mtd_parts, mtd_parts_nr);
	}
#endif
	return add_mtd_device(device);		/* add whole device */
}

/*
 * Detect and initialize DataFlash device connected to specified SPI channel.
 *
 *   Device            Density         ID code                 Nr Pages        Page Size       Page offset
 *   AT45DB011B        1Mbit   (128K)  xx0011xx (0x0c)         512             264             9
 *   AT45DB021B        2Mbit   (256K)  xx0101xx (0x14)         1025            264             9
 *   AT45DB041B        4Mbit   (512K)  xx0111xx (0x1c)         2048            264             9
 *   AT45DB081B        8Mbit   (1M)    xx1001xx (0x24)         4096            264             9
 *   AT45DB0161B       16Mbit  (2M)    xx1011xx (0x2c)         4096            528             10
 *   AT45DB0321B       32Mbit  (4M)    xx1101xx (0x34)         8192            528             10
 *   AT45DB0642        64Mbit  (8M)    xx1111xx (0x3c)         8192            1056            11
 *   AT45DB1282        128Mbit (16M)   xx0100xx (0x10)         16384           1056            11
 */
static int __init haydn_spiflash_detect(int channel)
{
	int res;

#ifdef CONFIG_MTD_VATICS
    res = add_dataflash(channel,
                        "VATICS SPI Flash",
                        CONFIG_MTD_VATICS_MAP_WINDOW_SIZE,
                        CONFIG_MTD_VATICS_MAP_PAGE_NUM,
                        CONFIG_MTD_VATICS_MAP_PAGE_SIZE,
                        0);
#else
    res = add_dataflash(channel, "VATICS SPI Flash", SZ_8M, 128, 65536, 0);
#endif

	return res;
}

static int __init haydn_spiflash_init(void)
{
	/* DataFlash (SPI chip select 0) */
	haydn_spiflash_detect(0);
	sema_init(&spiflash_mutex, 1);

    INIT_COMPLETION(spiflash_tx_complete) ;
    INIT_COMPLETION(spiflash_rx_complete) ;
	
	return 0;
}

static void __exit haydn_spiflash_exit(void)
{
	int i;

	for (i = 0; i < DATAFLASH_MAX_DEVICES; i++) {
		if (mtd_devices[i]) {
#ifdef CONFIG_MTD_PARTITIONS
			del_mtd_partitions(mtd_devices[i]);
#else
			del_mtd_device(mtd_devices[i]);
#endif
			kfree(mtd_devices[i]->priv);
			kfree(mtd_devices[i]);
		}
	}
	nr_devices = 0;
	kfree(spi_transfer_desc);
}


module_init(haydn_spiflash_init);
module_exit(haydn_spiflash_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("evelyn");
MODULE_DESCRIPTION("SPIFlash driver for VIATICS Mozart");

