//#include <linux/config.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>

#include <linux/kernel.h>	/* printk() */
#include <linux/slab.h>		/* kmalloc() */
#include <linux/fs.h>		/* file ops */
#include <linux/errno.h>	/* error codes */
#include <linux/types.h>	/* size_t */
#include <linux/proc_fs.h>
#include <linux/fcntl.h>	/* O_ACCMODE */
#include <linux/seq_file.h>
#include <linux/cdev.h>
#include <linux/miscdevice.h>

#include <asm/system.h>		/* cli(), *_flags */
#include <asm/uaccess.h>	/* copy_*_user */

#include <twsi/mvTwsi.h>

#define EEPROM_I2C_CHANNEL	0x0
#define EEPROM_DEBUG
//#include <asm/arch/eeprom.h>
//#include <asm/eeprom.h>
#include <eeprom.h>

/*
 * parameters which can be set at load time.
 */
int eeprom_major =   EEPROM_MAJOR;
int eeprom_minor =   EEPROM_MINOR;
int eeprom_nr_devs = EEPROM_NR_DEVS;
int eeprom_slave_addr = EEPROM_SLAVE_ADDR;

module_param(eeprom_major, int, S_IRUGO);
module_param(eeprom_minor, int, S_IRUGO);
module_param(eeprom_nr_devs, int, S_IRUGO);
module_param(eeprom_slave_addr, int, S_IRUGO);

int eeprom_size = 256;
unsigned char eeprom_scratch[EEPROM_MAX_SIZE];

/*
 * eeprom_mutex protects eeprom_inuse
 */
static DEFINE_MUTEX(eeprom_mutex);
static unsigned long eeprom_inuse;

/* actual write to eeprom */
int eeprom_twsi_write(unsigned offset, unsigned char *buf)
{
    int cnt = eeprom_size;
    MV_TWSI_SLAVE   twsiSlave;

    twsiSlave.slaveAddr.type = ADDR7_BIT;
    twsiSlave.slaveAddr.address = eeprom_slave_addr;
    twsiSlave.validOffset = MV_TRUE;
    twsiSlave.offset = offset;
    twsiSlave.moreThen256 = MV_FALSE;

    while(cnt > 8) {
		mvTwsiWrite (EEPROM_I2C_CHANNEL, &twsiSlave, buf+offset, 8);
        cnt -= 8;
        offset += 8;
        twsiSlave.offset = offset;
        //udelay(500); /* Wait a short while for the write to succeed */
        mdelay(3); /* Wait a short while for the write to succeed. For SVR-516 */
    }
    mvTwsiWrite (EEPROM_I2C_CHANNEL, &twsiSlave, buf+offset, cnt);
    return 0;
}

/* actual read from eeprom */
int eeprom_twsi_read(unsigned offset, unsigned char *buf)
{
    MV_TWSI_SLAVE   twsiSlave;

    twsiSlave.slaveAddr.type = ADDR7_BIT;
    twsiSlave.slaveAddr.address = eeprom_slave_addr;
    twsiSlave.validOffset = MV_TRUE;
    twsiSlave.offset = offset;
    if (eeprom_size > 256)
        twsiSlave.moreThen256 = MV_TRUE;
    else
        twsiSlave.moreThen256 = MV_FALSE;
    mvTwsiRead (EEPROM_I2C_CHANNEL, &twsiSlave, buf, eeprom_size);
    return 0;
}

/* blk write to eeprom */
int eeprom_twsi_write_blk(unsigned offset, unsigned count, unsigned char *buf)
{
    int cnt = (count>eeprom_size)?eeprom_size:count;
    MV_TWSI_SLAVE   twsiSlave;

    twsiSlave.slaveAddr.type = ADDR7_BIT;
    twsiSlave.slaveAddr.address = eeprom_slave_addr;
    twsiSlave.validOffset = MV_TRUE;
    twsiSlave.offset = offset;
    twsiSlave.moreThen256 = MV_FALSE;

    while(cnt > 8) {
		mvTwsiWrite (EEPROM_I2C_CHANNEL, &twsiSlave, buf+offset, 8);
        cnt -= 8;
        offset += 8;
        twsiSlave.offset = offset;
        //udelay(500); /* Wait a short while for the write to succeed/ For SVR-116 */
        mdelay(3); /* Wait a short while for the write to succeed. For SVR-516 */
		//mdelay(1); will cause write failed.
    }
    mvTwsiWrite (EEPROM_I2C_CHANNEL, &twsiSlave, buf+offset, cnt);
    return 0;
}

int eeprom_twsi_read_blk(unsigned offset, unsigned count, unsigned char *buf)
{
    int cnt = (count>eeprom_size)?eeprom_size:count;
    MV_TWSI_SLAVE   twsiSlave;

    twsiSlave.slaveAddr.type = ADDR7_BIT;
    twsiSlave.slaveAddr.address = eeprom_slave_addr;
    twsiSlave.validOffset = MV_TRUE;
    twsiSlave.offset = offset;
    if (cnt > 256)
        twsiSlave.moreThen256 = MV_TRUE;
    else
        twsiSlave.moreThen256 = MV_FALSE;
    mvTwsiRead (EEPROM_I2C_CHANNEL, &twsiSlave, buf+offset, cnt);
    return 0;
}

static ssize_t eeprom_write(struct file *file, const char __user *buf, size_t count,
        loff_t *ppos)
{
    int retval = 0;

    if (*ppos+count > eeprom_size)
        return -EINVAL;

    if (copy_from_user(&eeprom_scratch[*ppos], buf, count)) {
        retval = -EFAULT;
        goto out;
    }
    *ppos += count;
    retval = count;

out:
    return retval;
}

static ssize_t eeprom_read(struct file *file, char __user *buf, size_t count,
        loff_t *ppos)
{
    int retval = 0;

    if (*ppos > eeprom_size)
        return -EINVAL;

    if (copy_to_user(buf, &eeprom_scratch[*ppos], count)) {
        retval = -EFAULT;
        goto out;
    }
    *ppos += count;
    retval = count;

out:
    return retval;
}

static int eeprom_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
        unsigned long arg)
{
    int err = 0;
    int retval = 0;
    ioblock_t iob;

    if (_IOC_TYPE(cmd) != EEPROM_IOC_MAGIC) return -ENOTTY;
    if (_IOC_NR(cmd) > EEPROM_IOC_MAXNR) return -ENOTTY;

    if (_IOC_DIR(cmd) & _IOC_READ)
        err = !access_ok(VERIFY_WRITE, (void __user *)arg, _IOC_SIZE(cmd));
    else if (_IOC_DIR(cmd) & _IOC_WRITE)
        err =  !access_ok(VERIFY_READ, (void __user *)arg, _IOC_SIZE(cmd));
    if (err) return -EFAULT;

    switch(cmd) {
        case EEPROM_IOCGETSIZE:
            retval = __put_user(eeprom_size, (int __user *)arg);
            break;
        case EEPROM_IOCSETSIZE:
            {
                int new_size;
                retval = __get_user(new_size, (int __user *)arg);
                if (new_size > EEPROM_MAX_SIZE)
                    return -EINVAL;
                eeprom_size = new_size;
            }
            break;
        case EEPROM_IOCGETBUFF:
            eeprom_twsi_read(0, eeprom_scratch);
            break;
        case EEPROM_IOCSETBUFF:
            eeprom_twsi_write(0, eeprom_scratch);
            break;
        case EEPROM_IOCSETBUFFBLK:
            if (copy_from_user(&iob, (void __user *)arg, sizeof(iob)))
                return -EFAULT;
            eeprom_twsi_write_blk(iob.offset, iob.count, eeprom_scratch);
            break;
        case EEPROM_IOCGETBUFFBLK:
            if (copy_from_user(&iob, (void __user *)arg, sizeof(iob)))
                return -EFAULT;
            eeprom_twsi_read_blk(iob.offset, iob.count, eeprom_scratch);
            break;
        case EEPROM_IOCSETADDR:
            retval = __get_user(eeprom_slave_addr, (int __user *)arg);
            break;
    }
    return retval;
}

static int eeprom_open(struct inode *inode, struct file *file)
{
    int ret;

    mutex_lock(&eeprom_mutex);

    if (eeprom_inuse) {
        ret = -EBUSY;
    } else {
        eeprom_inuse = 1;
        ret = 0;
    }
    mutex_unlock(&eeprom_mutex);

    return ret;
}

static int eeprom_release(struct inode *inode, struct file *file)
{
    mutex_lock(&eeprom_mutex);
    eeprom_inuse = 0;
    mutex_unlock(&eeprom_mutex);
    return 0;
}

loff_t eeprom_llseek(struct file *filp, loff_t off, int whence)
{
    loff_t newpos;

    switch(whence) {
        case 0: /* SEEK_SET */
            newpos = off;
            break;

        case 1: /* SEEK_CUR */
            newpos = filp->f_pos + off;
            break;

        case 2: /* SEEK_END */
            newpos = eeprom_size-1;
            break;

        default: /* can't happen */
            return -EINVAL;
    }
    if (newpos < 0) return -EINVAL;
    filp->f_pos = newpos;
    return newpos;
}

static struct file_operations eeprom_fops = {
    .owner		= THIS_MODULE,
    .llseek		= eeprom_llseek,
    .read		= eeprom_read,
    .write		= eeprom_write,
    .ioctl		= eeprom_ioctl,
    .open		= eeprom_open,
    .release	= eeprom_release,
};

static struct miscdevice eeprom_miscdev = {
    .minor		= MISC_DYNAMIC_MINOR,
    .name		= "eeprom",
    .fops		= &eeprom_fops,
};


static int eeprom_read_proc(char *page, char **start, off_t off, int count, int *eof, void *data)
{
    int i = 0;
    char *p = page;
    char tmp[32];

    p += sprintf(p, "eeprom in use = %lu\n",eeprom_inuse);
    p += sprintf(p, "eeprom size = %d\n",eeprom_size);
    while (i < eeprom_size) 
    {
        p += sprintf(p, "%02x ",eeprom_scratch[i++]);
        if (i && !(i%32)) {
            strncpy(tmp, &eeprom_scratch[i-32], 32);
            p += sprintf(p, "\t%s\n", tmp);
        }
    }
    p += sprintf(p, "\n");
    return p - page;
}

void eeprom_cleanup_module(void)
{
    mutex_lock(&eeprom_mutex);
    remove_proc_entry("driver/eeprom", NULL);
    misc_deregister(&eeprom_miscdev);
    mutex_unlock(&eeprom_mutex);
}

int eeprom_init_module(void)
{
    int ret = -EBUSY;

    mutex_lock(&eeprom_mutex);
    ret = misc_register(&eeprom_miscdev);
    if (ret == 0)
        create_proc_read_entry("driver/eeprom", 0, NULL,
                eeprom_read_proc, NULL);
    mutex_unlock(&eeprom_mutex);

    eeprom_twsi_read(0, eeprom_scratch);
    return ret;
}

module_init(eeprom_init_module);
module_exit(eeprom_cleanup_module);


MODULE_AUTHOR("Ajith B");
MODULE_LICENSE("Seenergy Corp.");



