/* rwsem.h: R/W semaphores, public interface
 *
 * Written by David Howells (dhowells@redhat.com).
 * Derived from asm-i386/semaphore.h
 */

#ifndef _LINUX_RWSEM_H
#define _LINUX_RWSEM_H

#include <linux/linkage.h>

#ifdef CONFIG_PREEMPT_RT
# include <linux/rt_lock.h>
#endif

#ifdef __KERNEL__

#include <linux/types.h>
#include <linux/kernel.h>
#include <asm/system.h>
#include <asm/atomic.h>

#ifndef CONFIG_PREEMPT_RT
/*
 * On !PREEMPT_RT all rw-semaphores are compat:
 */
#define compat_rw_semaphore rw_semaphore
#endif

struct compat_rw_semaphore;

#ifdef CONFIG_RWSEM_GENERIC_SPINLOCK
# include <linux/rwsem-spinlock.h> /* use a generic implementation */
#  ifndef CONFIG_PREEMPT_RT
#  define __RWSEM_INITIALIZER __COMPAT_RWSEM_INITIALIZER
#  define DECLARE_RWSEM COMPAT_DECLARE_RWSEM
# endif
#else
# include <asm/rwsem.h> /* use an arch-specific implementation */
#endif

/*
 * lock for reading
 */
extern void compat_down_read(struct compat_rw_semaphore *sem);

/*
 * trylock for reading -- returns 1 if successful, 0 if contention
 */
extern int compat_down_read_trylock(struct compat_rw_semaphore *sem);

/*
 * lock for writing
 */
extern void compat_down_write(struct compat_rw_semaphore *sem);

/*
 * trylock for writing -- returns 1 if successful, 0 if contention
 */
extern int compat_down_write_trylock(struct compat_rw_semaphore *sem);

/*
 * release a read lock
 */
extern void compat_up_read(struct compat_rw_semaphore *sem);

/*
 * release a write lock
 */
extern void compat_up_write(struct compat_rw_semaphore *sem);

/*
 * downgrade write lock to read lock
 */
extern void compat_downgrade_write(struct compat_rw_semaphore *sem);

#ifdef CONFIG_DEBUG_LOCK_ALLOC
/*
 * nested locking. NOTE: rwsems are not allowed to recurse
 * (which occurs if the same task tries to acquire the same
 * lock instance multiple times), but multiple locks of the
 * same lock class might be taken, if the order of the locks
 * is always the same. This ordering rule can be expressed
 * to lockdep via the _nested() APIs, but enumerating the
 * subclasses that are used. (If the nesting relationship is
 * static then another method for expressing nested locking is
 * the explicit definition of lock class keys and the use of
 * lockdep_set_class() at lock initialization time.
 * See Documentation/lockdep-design.txt for more details.)
 */
extern void
compat_down_write_nested(struct compat_rw_semaphore *sem, int subclass);
/*
 * Take/release a lock when not the owner will release it.
 *
 * [ This API should be avoided as much as possible - the
 *   proper abstraction for this case is completions. ]
 */
extern void
compat_down_read_non_owner(struct compat_rw_semaphore *sem);
extern void
compat_up_read_non_owner(struct compat_rw_semaphore *sem);
#else
# define compat_down_write_nested(sem, subclass)	compat_down_write(sem)
# define compat_down_read_non_owner(sem)		compat_down_read(sem)
# define compat_up_read_non_owner(sem)			compat_up_read(sem)
#endif

#ifndef CONFIG_PREEMPT_RT

#define DECLARE_RWSEM COMPAT_DECLARE_RWSEM

/*
 * NOTE, lockdep: this has to be a macro, so that separate class-keys
 * get generated by the compiler, if the same function does multiple
 * init_rwsem() calls to different rwsems.
 */
#define init_rwsem(rwsem)	compat_init_rwsem(rwsem)

static inline void down_read(struct compat_rw_semaphore *rwsem)
{
	compat_down_read(rwsem);
}
static inline int down_read_trylock(struct compat_rw_semaphore *rwsem)
{
	return compat_down_read_trylock(rwsem);
}
static inline void down_write(struct compat_rw_semaphore *rwsem)
{
	compat_down_write(rwsem);
}
static inline int down_write_trylock(struct compat_rw_semaphore *rwsem)
{
	return compat_down_write_trylock(rwsem);
}
static inline void up_read(struct compat_rw_semaphore *rwsem)
{
	compat_up_read(rwsem);
}
static inline void up_write(struct compat_rw_semaphore *rwsem)
{
	compat_up_write(rwsem);
}
static inline void downgrade_write(struct compat_rw_semaphore *rwsem)
{
	compat_downgrade_write(rwsem);
}
static inline int rwsem_is_locked(struct compat_rw_semaphore *sem)
{
	return compat_rwsem_is_locked(sem);
}
# define down_write_nested(sem, subclass) \
		compat_down_write_nested(sem, subclass)
# define down_read_non_owner(sem) \
		compat_down_read_non_owner(sem)
# define up_read_non_owner(sem) \
		compat_up_read_non_owner(sem)
#endif /* !CONFIG_PREEMPT_RT */

#endif /* __KERNEL__ */
#endif /* _LINUX_RWSEM_H */
