/*
 *	Wireless Tools
 *
 *		Jean II - HPLB '99 - HPL 99->01
 *
 * This tool can access various piece of information on the card
 * not part of iwconfig...
 * You need to link this code against "iwlist.c" and "-lm".
 *
 * This file is released under the GPL license.
 *     Copyright (c) 1997-2002 Jean Tourrilhes <jt@hpl.hp.com>
 */

#include "iwlib.h"		/* Header */
#include <sys/time.h>

#define CONFIG_RESULT           "/tmp/config_result.inc"
#define SITESURVEY_RESULT       "/tmp/sitesurvey_result.inc"

/***************************** SCANNING *****************************/
/*
 * This one behave quite differently from the others
 */
#if WIRELESS_EXT > 13
static int print_wireless_info(int skfd, char* ifname, char args[], int count)
{
	unsigned char	buffer[IW_SCAN_MAX_DATA];
	struct wireless_info info;
	struct _wlanstruct wlaninfo;
	struct iwreq	wrq;
      	struct ifreq ifr;
	FILE *fp;

	args = args;
	count = count;

        if ((fp = fopen (CONFIG_RESULT, "wb")) == NULL){
		printf ("Can't write the file(sitesurvey_result.inc)\n");
		exit(0);
	}

	memset(&info, 0x00, sizeof(wireless_info));

	/* Get wireless name */
	
	if(iw_get_ext(skfd, ifname, SIOCGIWNAME, &wrq) < 0){
		/* If no wireless name : no wireless extensions */
		/* But let's check if the interface exists at all */
		strcpy(ifr.ifr_name, ifname);
		if(ioctl(skfd, SIOCGIFFLAGS, &ifr) < 0)
			return(-ENODEV);
		else
			return(-ENOTSUP);
	}
	else{
		strncpy(info.name, wrq.u.name, IFNAMSIZ);
		info.name[IFNAMSIZ] = '\0';
	}

	/* Get operation mode */
	if(iw_get_ext(skfd, ifname, SIOCGIWMODE, &wrq) >= 0){
		info.mode = wrq.u.mode;
		if((info.mode < IW_NUM_OPER_MODE) && (info.mode >= 0))
			info.has_mode = 1;
		if(info.has_mode == 1){
			memcpy(wlaninfo.mode, "Infrastructure", 14);
			printf("MODE=Infrastructure\n");
		}
		else{
			memcpy(wlaninfo.mode, "Ad-hoc", 6);
			printf("MODE=Ad-hoc\n");
		}
		sprintf(wlaninfo.state, "%s", "Connect");
	}
	else{
                fputs ("MODE=Unknown\n", fp);
                fputs ("ESSID=Unknown\n", fp);
                fputs ("CHANNEL=Unknown\n", fp);
                fputs ("WEP=Unknown\n", fp);
                fputs ("STATE=Disconnected\n", fp);
                fputs ("BSSID=Unknown\n", fp);
		fclose(fp);
		return(0);
	}

	/* Get ESSID */
	wrq.u.essid.pointer = (caddr_t) info.essid;
	wrq.u.essid.length = IW_ESSID_MAX_SIZE + 1;
	wrq.u.essid.flags = 0;
	if(iw_get_ext(skfd, ifname, SIOCGIWESSID, &wrq) >= 0){
		info.has_essid = 1;
		info.essid_on = wrq.u.data.flags;
		printf("SSID=%s\n", info.essid);
		sprintf(wlaninfo.ssid, "%s", info.essid);
	}

        /* Get frequency / channel */
        if(iw_get_ext(skfd, ifname, SIOCGIWFREQ, &wrq) >= 0){
                info.has_freq = 1;
        //      info->freq = iw_freq2float(&(wrq.u.freq));
        //      frequency = (((unsigned int) wrq.u.freq.m / 100000) - 2407);
        //      printf ("Frequency=%d\n",frequency);
        //      info->freq = (double)(frequency / 5);
                info.freq = (double) wrq.u.freq.m;
                printf("CHANNEL=%d\n", (int)info.freq);
		sprintf(wlaninfo.channel, "%d", (int)info.freq);
        }

        /* Get encryption information */
	
        wrq.u.data.pointer = (caddr_t) info.key;
        wrq.u.data.length = IW_ENCODING_TOKEN_MAX;
        wrq.u.data.flags = 0;
        if(iw_get_ext(skfd, ifname, SIOCGIWENCODE, &wrq) >= 0){
                info.has_key = 1;
                info.key_size = wrq.u.data.length;
                info.key_flags = wrq.u.data.flags;
		printf("WEP=on");
		sprintf(wlaninfo.encryption, "on");
        }
	else{
		sprintf(wlaninfo.encryption, "off");
		printf("WEP=off");
	}
	
	/* Get AP address */
	
	if(iw_get_ext(skfd, ifname, SIOCGIWAP, &wrq) >= 0){
		info.has_ap_addr = 1;
		memcpy(&(info.ap_addr), &(wrq.u.ap_addr), sizeof (sockaddr));
		printf("BSSID=%s", iw_pr_ether(buffer, info.ap_addr.sa_data));
		sprintf(wlaninfo.bssid, "%s", iw_pr_ether(buffer, info.ap_addr.sa_data));
		if(!memcmp(wlaninfo.bssid, "FF:FF:FF:FF:FF:FF", 17)){
	                fputs ("MODE=Unknown\n", fp);
        	        fputs ("ESSID=Unknown\n", fp);
                	fputs ("CHANNEL=Unknown\n", fp);
	                fputs ("WEP=Unknown\n", fp);
        	        fputs ("STATE=Disconnected\n", fp);
                	fputs ("BSSID=Unknown\n", fp);
	                fclose(fp);
        	        return(0);			
		}
			
	}

	sprintf (buffer, "MODE=%s\n", wlaninfo.mode);
        fputs (buffer, fp);
        sprintf (buffer, "ESSID=%s\n", wlaninfo.ssid);
        fputs (buffer, fp);
        sprintf (buffer, "CHANNEL=%s\n", wlaninfo.channel);
        fputs (buffer, fp);
        sprintf (buffer, "WEP=%s\n", wlaninfo.encryption);
        fputs (buffer, fp);
        sprintf (buffer, "STATE=%s\n", wlaninfo.state);
        fputs (buffer, fp);
        sprintf (buffer, "BSSID=%s\n", wlaninfo.bssid);
        fputs (buffer, fp);
	

	fclose(fp);
	return (0);	
}
			

/*------------------------------------------------------------------*/
/*
 * Perform a scanning on one device
 */
static int print_scanning_info(int skfd, char *ifname, char *args[], int count)
{
	struct iwreq wrq;
	unsigned char buffer[IW_SCAN_MAX_DATA];	/* Results */
	struct timeval	tv;			/* Select timeout */
	int timeout = 5000000;			/* 5s */
	struct	_wlanstruct info;
	FILE	*fp;
			  
	/* Avoid "Unused parameter" warning */
	args = args; count = count;

	/* Init timeout value -> 250ms*/
	tv.tv_sec = 0;
	tv.tv_usec = 250000;

        if ((fp = fopen (SITESURVEY_RESULT, "wb")) == NULL)
        {
                printf ("Can't write the file(sitesurvey_result.inc)\n");
                exit(0);
        }
	

  	/*
	   * Here we should look at the command line args and set the IW_SCAN_ flags
	   * properly
	   */
	wrq.u.param.flags = IW_SCAN_DEFAULT;
	wrq.u.param.value = 0;		/* Later */

	/* Initiate Scanning */
	if(iw_set_ext(skfd, ifname, SIOCSIWSCAN, &wrq) < 0){
		if(errno != EPERM){
			fprintf(stderr, "%-8.8s Interface doesn't support scanning : %s\n\n", 
					ifname, strerror(errno));
			return(-1);
		}
      /* If we don't have the permission to initiate the scan, we may
       * still have permission to read left-over results.
       * But, don't wait !!! */
		tv.tv_usec = 0;
	}
	
	timeout -= tv.tv_usec;
  /* Forever */
	
	while(1){
		fd_set rfds;		/* File descriptors for select */
		int last_fd;	/* Last fd */
		int ret;
		
		/* Guess what ? We must re-generate rfds each time */
		FD_ZERO(&rfds);
		last_fd = -1;
		
		/* In here, add the rtnetlink fd in the list */
		/* Wait until something happens */
		ret = select(last_fd + 1, &rfds, NULL, NULL, &tv);
		
		/* Check if there was an error */
		if(ret < 0){
			if(errno == EAGAIN || errno == EINTR)
				continue;
			fprintf(stderr, "Unhandled signal - exiting...\n");
			return(-1);
		}
		
		/* Check if there was a timeout */
		if(ret == 0){
			/* Try to read the results */
			wrq.u.data.pointer = buffer;
			wrq.u.data.flags = 0;
			wrq.u.data.length = sizeof(buffer);
			if(iw_get_ext(skfd, ifname, SIOCGIWSCAN, &wrq) < 0){
				/* Check if results not available yet */
				if(errno == EAGAIN){
					/* Restart timer for only 100ms*/
					tv.tv_sec = 0;
					tv.tv_usec = 100000;
					timeout -= tv.tv_usec;
					if(timeout > 0)
						continue;	/* Try again later */
				}
				/* Bad error */
				fprintf(stderr, "%-8.8s  Failed to read scan data : %s\n\n",
						ifname, strerror(errno));
				return(-2);
			}
			else
				/* We have the results, go to process them */
				break;
		}
		
		/* In here, check if event and event type
		 * if scan event, read results. All errors bad & no reset timeout */
	}
	if(wrq.u.data.length){
		struct iw_event	iwe;
		struct stream_descr stream;
		int ret;
		struct iw_range range;
		int has_range;
		unsigned int frequency;
		unsigned char   key[IW_ENCODING_TOKEN_MAX];
		
		has_range = (iw_get_range_info(skfd, ifname, &range) >= 0);
		printf("Scan completed !!\r\n");
		iw_init_event_stream(&stream, buffer, wrq.u.data.length);
		do{
			/* Extract an event and print it */
			ret = iw_extract_event_stream(&stream, &iwe);
			if(ret > 0){
				switch(iwe.cmd){
					case SIOCGIWAP:				//BSSID	
						memset(&info, 0x00, sizeof(wlanstruct));
						iw_pr_ether(buffer, iwe.u.ap_addr.sa_data);
						sprintf(info.bssid, "%s", buffer);
					break;

					case SIOCGIWFREQ:			//Channel
						frequency = (unsigned int) iwe.u.freq.m / 100000;
						frequency = (frequency - 2407) / 5;
						if (frequency > 13)
							frequency = 14;
						sprintf(info.channel, "%d", frequency);
						break;
						
					case IWEVQUAL:				//SIGNAL
						iwe.u.qual.updated = 0x0;
						iw_print_stats(buffer,&iwe.u.qual,&range,has_range);
						sprintf(info.state, "%s", buffer);
						break;

					case SIOCGIWMODE:
//memcpy(info.mode, iw_operation_mode[iwe.u.mode], sizeof(iw_operation_mode[iwe.u.mode]));
					sprintf(info.mode, "%s", iw_operation_mode[iwe.u.mode]);
						break;				//Mode
					
					case SIOCGIWESSID:			//SSID
					if((iwe.u.essid.pointer) && (iwe.u.essid.length))
						memcpy(info.ssid, iwe.u.essid.pointer, iwe.u.essid.length);
					info.ssid[iwe.u.essid.length] = '\0';
					if(iwe.u.essid.flags){
						/* Does it have an ESSID index ? */
					}
						break;

				case SIOCGIWENCODE:			//WEP
					if(iwe.u.data.pointer)
						memcpy(key,iwe.u.essid.pointer,iwe.u.data.length);
					else
						iwe.u.data.flags |= IW_ENCODE_NOKEY;
					if(iwe.u.data.flags & IW_ENCODE_DISABLED){
						memcpy(info.encryption, "off", 3);
					}
					else{
						memcpy(info.encryption, "on", 2);
					}
					fwrite (&info, sizeof(wlanstruct), 1, fp);
					break;
					default:
						break;
				}
			}
		}
		while(ret > 0);
	}
	else
	       	printf("%-8.8s  No scan results\n", ifname);
	fclose(fp);
	return(0);
}
#endif	/* WIRELESS_EXT > 13 */

/************************* COMMON UTILITIES *************************/
/*
 * This section was written by Michael Tokarev <mjt@tls.msk.ru>
 * But modified by me ;-)
 */

/* command list */
typedef struct iwlist_entry {
	const char *cmd;
	iw_enum_handler fn;
	int min_count;
	int max_count;
}iwlist_cmd;

static const struct iwlist_entry iwlist_cmds[] = {
#if WIRELESS_EXT > 13
	{ "config",		print_wireless_info,	0, 5 },
	{ "sitesurvey",		print_scanning_info,	0, 5 },
#endif
	{ NULL, NULL, 0, 0 },
};

/*------------------------------------------------------------------*/
/*
 * Find the most appropriate command matching the command line
 */
static inline const iwlist_cmd * find_command(const char *cmd)
{
	const iwlist_cmd *found = NULL;
	int ambig = 0;
	unsigned int len = strlen(cmd);
	int i;
	
	/* Go through all commands */
	
	for(i = 0; iwlist_cmds[i].cmd != NULL; ++i){
		/* No match -> next one */
		if(strncasecmp(iwlist_cmds[i].cmd, cmd, len) != 0)
			continue;
		
		/* Exact match -> perfect */
		if(len == strlen(iwlist_cmds[i].cmd))
			return &iwlist_cmds[i];
		
		/* Partial match */
		if(found == NULL)
			/* First time */
			found = &iwlist_cmds[i];
		else
			/* Another time */
			if (iwlist_cmds[i].fn != found->fn)
				ambig = 1;
	}
	
	if(found == NULL){
		fprintf(stderr, "iwlist: unknown command `%s'\n", cmd);
		return NULL;
	}
	
	if(ambig){
		fprintf(stderr, "iwlist: command `%s' is ambiguous\n", cmd);
		return NULL;
	}
	
	return found;
}

/*------------------------------------------------------------------*/
/*
 * Display help
 */
static void iw_usage(int status)
{
	FILE* f = status ? stderr : stdout;
	int i;
	
	fprintf(f,   "Usage: iwlist [interface] %s\n", iwlist_cmds[0].cmd);
	
	for(i = 1; iwlist_cmds[i].cmd != NULL; ++i)
		fprintf(f, "              [interface] %s\n", iwlist_cmds[i].cmd);
	
	exit(status);
}

/******************************* MAIN ********************************/

/*------------------------------------------------------------------*/
/*
 * The main !
 */
int main(int	argc, char **	argv)
{
	int skfd;			/* generic raw socket desc.	*/
	char *dev;			/* device name			*/
	char *cmd;			/* command			*/
	char **args;			/* Command arguments */
	int count;			/* Number of arguments */
	const iwlist_cmd *iwcmd;

	if(argc == 1 || argc > 3)
		iw_usage(1);

	if (!strcmp(argv[1], "-h") || !strcmp(argv[1], "--help"))
		iw_usage(0);
	
	/* This is also handled slightly differently */
	if (!strcmp(argv[1], "-v") || !strcmp(argv[1], "--version"))
		return(iw_print_version_info("iwlist"));
	
	if (argc == 2){
		cmd = argv[1];
		dev = NULL;
		args = NULL;
		count = 0;
	}
	else{
		cmd = argv[2];
		dev = argv[1];
		args = argv + 3;
		count = argc - 3;
	}
	
	/* find a command */
	iwcmd = find_command(cmd);
	
	if(iwcmd == NULL)
		return 1;
	
	/* Check arg numbers */
	if(count < iwcmd->min_count){
		fprintf(stderr, "iwlist: command `%s' needs more arguments\n", cmd);
		return 1;
	}
	
	if(count > iwcmd->max_count){
		fprintf(stderr, "iwlist: command `%s' needs fewer arguments\n", cmd);
		return 1;
	}
	
	/* Create a channel to the NET kernel. */
	if((skfd = iw_sockets_open()) < 0){
		perror("socket");
		return -1;
	}
	
	/* do the actual work */
	if (dev){
		(*iwcmd->fn)(skfd, dev, args, count);
	}
	else{
		iw_enum_devices(skfd, iwcmd->fn, args, count);
	}
	
	/* Close the socket. */
	close(skfd);
	
	return 0;
}
