#!/bin/bash

BONDING_UP=1
DHCPCD_SH=/etc/init.d/dhcpcd.sh

#==================================================================
# The functions of starting up the wireless LAN
#

#------------------------------------------------------------------
# dot11_mibset(dev,mibattribute,value)
#	Arguments:	
#		dev		wlan device
#		mibattribute	attribute name
#		value		value to set
#	Returns:   
#		0 - success
#		1 - wlanctl failed (ioctl or something)
#		2 - mibset failed (bad message, unsupported, not r/w etc.)
#  Sets a mib item for a wlan device.
#------------------------------------------------------------------
dot11_mibset()
{
	result=`/sbin/wlanctl-ng $1 dot11req_mibset "mibattribute=$2=$3"`
	if [ $? = 0 ] ; then
		eval $result
		if [ $resultcode != "success" ] ; then 
			echo "Failed to set $2."
			return 2
		fi
	else
		echo "wlanctl-ng dot11req_mibset Failed to set $2."
		return 1
	fi
	return 0
}

#------------------------------------------------------------------
# dot11_mibget(dev,mibattribute)
#	Arguments:
#		dev		wlan device
#		mibattribute	attribute name
#	Returns:
#		0 - success
#		1 - wlanctl failed (ioctl or something)
#		2 - mibset failed (bad message, unsupported, not r/w etc.)
#  Collects the named mib item.  Upon success, the value can be
#  found in the a variable named for the mibattribute.
#------------------------------------------------------------------
dot11_mibget()
{
	result=`/sbin/wlanctl-ng $1 dot11req_mibget "mibattribute=$2"`
	if [ $? = 0 ] ; then
		eval $result
		if [ $resultcode != "success" ] ; then
			echo "Failed to get $2."
			return 2
		fi
		eval $mibattribute
	else
		echo "wlanctl-ng dot11req_mibget Failed to get $2."
		return 1
	fi
	return 0
}


#------------------------------------------------------------------
# dot11_apstartbss(ssid, bcnintm, dtimint, cfpollable, cfpollreq,
#                   cfpperiod, cfpmaxduration, probedelay, channel,
#                   basicrates, oprates)
#	Arguments:
#		dev		wlan device
#		ssid
#		bcnintm
#		dtimint
#		cfpollable
#		cfpollreq
#		cfpperiod
#		cfpmaxduration
#		probedelay
#		channel
#		basicrates
#		oprates
#	Returns:
#		0 - success
#		1 - wlanctl failed (ioctl or something)
#		2 - start failed (bad message, unsupported etc.)
#
#  Instructs the MAC to create a new infrastructure bss.
#------------------------------------------------------------------
dot11_apstartbss()
{
	startcmd="/sbin/wlanctl-ng $1 dot11req_start " ; shift 1
	startcmd="$startcmd ssid=$1 " ; shift 1
	startcmd="$startcmd bsstype=infrastructure "
	startcmd="$startcmd beaconperiod=$1 " ; shift 1
	startcmd="$startcmd dtimperiod=$1 " ; shift 1
	startcmd="$startcmd cfpollable=$1 " ; shift 1
	startcmd="$startcmd cfpollreq=$1 " ; shift 1
	startcmd="$startcmd cfpperiod=$1 " ; shift 1
	startcmd="$startcmd cfpmaxduration=$1 " ; shift 1
	startcmd="$startcmd probedelay=$1 " ; shift 1
	startcmd="$startcmd dschannel=$1 " ; shift 1

	j=1
	for i in `echo $1 | tr ":" " "` ; do
		startcmd="$startcmd basicrate$j=$i "
		j=$[j + 1]
	done
	shift 1

	j=1
	for i in `echo $1 | tr ":" " "` ; do
		startcmd="$startcmd operationalrate$j=$i "
		j=$[j + 1]
	done

	results=`$startcmd`	# Here's where it runs

	if [ $? = 0 ]; then
		eval $results
		if [ $resultcode != "success" ] ; then
			echo "dot11req_start failed: resultcode=$resultcode"
			return 2
		else
			return 0
		fi
	else
		return 1
	fi
}


#------------------------------------------------------------------
# bridge_create(bridgedev, if0, if1, [if2, ...])
#	Arguments:
#		bridgedev	bridge device to create
#		if0		first device to bind to bridge
#		if1		second device to bind to bridge
#		if2 ...		(optional) third device to bind to bridge
#	Returns:
#		0 - success
#		1 - creation of bridge device failed
#		2 - addition of a device to the bridge failed
#
# Creates a bridge device binding the given interfaces.
#------------------------------------------------------------------
bridge_create()
{
	brdev=$1
	/sbin/brctl addbr $brdev || return 1
	shift 1
	while [ $# -gt 0 ]; do
		/sbin/brctl addif $brdev $1 || return 2
		shift 1
	done
	return 0
}

#------------------------------------------------------------------
# bridge_delete(bridgedev, if0, if1, [if2, ...])
#	Arguments:
#		bridgedev	bridge device to delete
#		if0		first device to remove from bridge
#		if1		second device to remove from bridge
#		if2 ...		(optional) third device to remove from bridge
#	Returns:
#		0 - success
#		1 - deletion of bridge device failed
#		2 - deletion of a device to the bridge failed
#
# Deletes a bridge device removing the given interfaces.
#------------------------------------------------------------------
bridge_delete()
{
	brdev=$1
	shift 1
	while [ $# -gt 0 ]; do
		/sbin/brctl delif $brdev $1 || return 2
		shift 1
	done
	/sbin/brctl delbr $brdev || return 1
	return 0
}

#------------------------------------------------------------------
# wlan0_set_mibs
#	Arguments:
#		none
#	Returns:
#		0 - success
#
# Sets up the common mibs for the wlan0 interface.  Note, their
# are no arguments to this function.  It picks up the values
# from the 'global' configuration variables.
#------------------------------------------------------------------
wlan0_set_mibs()
{
	# Set wlan0 settings
    dot11_mibset $WLAN_DEV dot11RTSThreshold \
		"$WLAN_dot11RTSThreshold"
    dot11_mibset $WLAN_DEV dot11FragmentationThreshold \
		"$WLAN_dot11FragmentationThreshold"


    if [ $WLAN_dot11PrivacyInvoked = "true" ]; then
		# Check that we can enable privacy
		dot11_mibget $WLAN_DEV dot11PrivacyOptionImplemented
		if [ $dot11PrivacyOptionImplemented != "true" ]; then
	    	echo "Cannot enable privacy, dot11PrivacyOptionImplemented=false"
	    	echo "  Privacy settings skipped."
            WLAN_AUTH_OPENSYSTEM="true"
            WLAN_AUTH_SHAREDKEY="false"
	    	dot11_mibset $WLAN_DEV dot11AuthenticationAlgorithmsEnable1 \
	        	"$WLAN_AUTH_OPENSYSTEM"
	    	dot11_mibset $WLAN_DEV dot11AuthenticationAlgorithmsEnable2 \
	        	"$WLAN_AUTH_SHAREDKEY"
		else
            WLAN_AUTH_OPENSYSTEM="true"
            WLAN_AUTH_SHAREDKEY="true"
            dot11_mibset $WLAN_DEV dot11AuthenticationAlgorithmsEnable1 \
	        		"$WLAN_AUTH_OPENSYSTEM"
	    	dot11_mibset $WLAN_DEV dot11AuthenticationAlgorithmsEnable2 \
	        	"$WLAN_AUTH_SHAREDKEY"
		dot11_mibset $WLAN_DEV dot11WEPDefaultKeyID \
				"$WLAN_dot11WEPDefaultKeyID"

	    	dot11_mibset $WLAN_DEV dot11WEPDefaultKey0 \
				"$WLAN_dot11WEPDefaultKey"
		dot11_mibset $WLAN_DEV dot11WEPDefaultKey1 \
				"$WLAN_dot11WEPDefaultKey"
	 	dot11_mibset $WLAN_DEV dot11WEPDefaultKey2 \
				"$WLAN_dot11WEPDefaultKey"
		dot11_mibset $WLAN_DEV dot11WEPDefaultKey3 \
				"$WLAN_dot11WEPDefaultKey"

	    	dot11_mibset $WLAN_DEV dot11ExcludeUnencrypted \
				"$WLAN_dot11ExcludeUnencrypted"
	    	dot11_mibset $WLAN_DEV dot11PrivacyInvoked \
				"$WLAN_dot11PrivacyInvoked"
		fi
    fi
    return 0
}

#----------------------------------------------------------------------------
#	Get parameter from /etc/config/uLinux.conf
#----------------------------------------------------------------------------
get_param()
{
    WLAN_AP_SSID="`/sbin/getcfg "WLAN" SSID -d "NAS-wlan"`"
    WLAN_AP_CHANNEL="`/sbin/getcfg "WLAN" Channel -d 6`"
    WLAN_dot11RTSThreshold="`/sbin/getcfg "WLAN" RTSThreshold -d 2347`"
    WLAN_dot11FragmentationThreshold="`/sbin/getcfg "WLAN" FragmentThreshold -d 2346`"
    if [ `/sbin/getcfg "WLAN" WEPEncyption -u -d 0` = 0 ]; then
        WLAN_dot11PrivacyInvoked="false"
        WLAN_dot11ExcludeUnencrypted="false"
    else
        WLAN_dot11PrivacyInvoked="true"
        WLAN_dot11ExcludeUnencrypted="true"
    fi
    WLAN_dot11WEPDefaultKeyID="`/sbin/getcfg "WLAN" WEPDefaultKeyID -d 0`"
    WLAN_dot11WEPDefaultKey0="`/sbin/getcfg "WLAN" Key0 -d ""`"
    WLAN_dot11WEPDefaultKey1="`/sbin/getcfg "WLAN" Key1 -d ""`"
    WLAN_dot11WEPDefaultKey2="`/sbin/getcfg "WLAN" Key2 -d ""`"
    WLAN_dot11WEPDefaultKey3="`/sbin/getcfg "WLAN" Key3 -d ""`"
    if [ $WLAN_dot11WEPDefaultKeyID = 0 ]; then
    	WLAN_dot11WEPDefaultKey=$WLAN_dot11WEPDefaultKey0
    elif [ $WLAN_dot11WEPDefaultKeyID = 1 ]; then
    	WLAN_dot11WEPDefaultKey=$WLAN_dot11WEPDefaultKey1
    elif [ $WLAN_dot11WEPDefaultKeyID = 2 ]; then
    	WLAN_dot11WEPDefaultKey=$WLAN_dot11WEPDefaultKey2
    elif [ $WLAN_dot11WEPDefaultKeyID = 3 ]; then
    	WLAN_dot11WEPDefaultKey=$WLAN_dot11WEPDefaultKey3
    fi
}
#----------------------------------------------------------------------------
# Load device driver and it's firmware
#----------------------------------------------------------------------------
load_driver()
{
    /sbin/insmod /lib/modules/misc/p80211.o
    /sbin/insmod /lib/modules/misc/prism2_pci.o
    /sbin/prism2dl -r /sbin/wlan_ap.hex $WLAN_DEV
}

#----------------------------------------------------------------------------
# Unload the driver
#----------------------------------------------------------------------------
unload_driver()
{
    /sbin/rmmod prism2_pci
    /sbin/rmmod p80211
}

#----------------------------------------------------------------------------
#	Start up the wlan
#	1. load the parameters
#	2. set mibs
#	3. start up the AP
#	4. shut dowm lan's interfaces. eg: eth1 and wlan0
#	5. create a bridge and add both eth1 and wlan0 interfaces into group
#	6. start up bridge and both interfaces
#----------------------------------------------------------------------------
start_wlan()
{
    #Load driver
#    load_driver

	# Get parameter
    get_param

	# Set wlan0 settings
	wlan0_set_mibs

	# Start BSS
	dot11_apstartbss \
		$WLAN_DEV \
		$WLAN_AP_SSID \
		"100" \
		"3" \
		"false" \
		"false" \
		"3" \
		"100" \
		"100" \
		$WLAN_AP_CHANNEL \
		"2 4" \
		"2 4 11 22"

	# Down wlan0 and eth0
	/sbin/ifconfig $DEV down
	/sbin/ifconfig $WLAN_DEV down

	# Create bridge
	bridge_create $BRIDGE_DEV $DEV $WLAN_DEV

	# Up the bridge and bridged devices
	/sbin/ifconfig $BRIDGE_DEV $IPADDR netmask $NETMASK
	/sbin/ifconfig $DEV up 0.0.0.0
	/sbin/ifconfig $WLAN_DEV up 0.0.0.0
	# need sleep ???
#	/bin/sleep 3
}
#----------------------------------------------------------------------------
#	Stop wlan
#	Step:
#	1. shut down both eth1 and wlan0 interfaces and bridge
#	2. remove both interfaces from bridge br0 group
#	3. delete the bridge
#	4. shut down the AP via dot11req_reset command
#	5. unload the driver modules
#----------------------------------------------------------------------------
stop_wlan()
{
	# Step 1
	/sbin/ifconfig $DEV down
	/sbin/ifconfig $WLAN_DEV down
    /sbin/ifconfig $BRIDGE_DEV down
    # Step 2, 3
	bridge_delete $BRIDGE_DEV $DEV $WLAN_DEV
    # Step 4
   # /sbin/wlanctl-ng $WLAN_DEV dot11req_reset setdefaultmib=true
# need sleep ???
#    /bin/sleep 3

    #Unload driver
    #unload_driver
}
#
#----------------------------------------------------------------------------

restart_dhcpd()
{
	if [ `/sbin/getcfg "DHCP Server" Enable -u -d FALSE` = TRUE ]; then
		/etc/init.d/dhcpd.sh restart 2>/dev/null 1>/dev/null
	fi
}

#====================================================
#	find correct nic conf file
#	NIC is a global variable
#====================================================
find_nic()
{
	for i in /tmp/nic*.tmp
	do
	#### not support this format under our NAS bash ####
#		if [ -e $i ]; then
			. $i
			if [ "$DEV" = "$1" ]; then
				NIC=$i
			fi
#		fi
	done
}

#=====================================================
#	up one nic, parameter1 is the conf file
#=====================================================
ifup_nic()
{
	. $1
	/bin/echo "start $DEV by using $USAGE"

	if [ "$2" = "WAN" ]; then
		/sbin/ifconfig $DEV hw ether `/sbin/nic_getinfo $DEV hwaddr` 2>/dev/null 1>/dev/null
	fi

	if [ "$USAGE" = "STATIC" ]; then
		if [ `/sbin/getcfg Network "BONDING Support" -u -d FALSE` != TRUE ]; then
			/sbin/ifconfig $DEV $IPADDR netmask $NETMASK broadcast $BROADCAST up
			/sbin/route add -host $IPADDR $DEV
		fi
		if [ "$2" = "WAN" ]; then
			/sbin/route add default gw $GATEWAY metric 1 $DEV
		fi
		if [ "$2" = "LAN" ] ; then
			# check bonding interface
			if [ `/sbin/getcfg Network "BONDING Support" -u -d FALSE` = TRUE ]; then
				bonding_type=`/sbin/getcfg $DEV Bonding -u -d "FAIL OVER"`
				if [ "$bonding_type" != "STANDALONE" ]; then
					if [ $BONDING_UP = 1 ]; then
						BONDING_UP=`/usr/bin/expr $BONDING_UP + 1`
						/sbin/ifconfig bond0 $IPADDR netmask $NETMASK broadcast $BROADCAST up
						/sbin/route add -host $IPADDR bond0
						/sbin/route add default gw $GATEWAY metric 1 bond0
					# *** Catherine ==>
					else
						BONDING_UP=`/usr/bin/expr $BONDING_UP + 1`
					# *** <== Catherine
					fi
					/sbin/ifenslave bond0 $DEV
					return
				else
					# *** Catherine ==>
					/sbin/ifconfig $DEV $IPADDR netmask $NETMASK broadcast $BROADCAST up
					/sbin/route add -host $IPADDR $DEV
					/sbin/route add default gw $GATEWAY metric 1 $DEV
					# *** <== Catherine
					return
				fi
			fi

	             	if [ `/sbin/getcfg Network "Interface Number"` = 1 ]; then
				/sbin/route add default gw $GATEWAY metric 1 $DEV
			fi
        	     	if [ `/sbin/getcfg Network "Interface Number"` = 2 ]; then
				if [ `/sbin/wlan_chk` = TRUE -a `/sbin/getcfg "WLAN" Support -u -d FALSE` = TRUE -a `/sbin/getcfg "WLAN" Enable -u -d FALSE` = TRUE ]; then
            				start_wlan
				fi
				if [ `/sbin/getcfg NAT Support -u -d TRUE` = FALSE ]; then
					# not enable NAT, so as a default router
					if [ `/sbin/wlan_chk` = TRUE -a `/sbin/getcfg "WLAN" Support -u -d FALSE` = TRUE -a `/sbin/getcfg "WLAN" Enable -u -d FALSE` = TRUE ]; then
						/sbin/route add default gw $GATEWAY metric 1 $BRIDGE_DEV
					else
						/sbin/route add default gw $GATEWAY metric 1 $DEV
					fi
				fi
			fi
		fi
	        NAS_DNS=`/sbin/getcfg "Network" DNS -d 0.0.0.0`
	        if [ "$NAS_DNS" != "0.0.0.0" ]; then
			/bin/rm /etc/resolv.* -f 2>/dev/null 1>/dev/null
        	        /bin/echo "nameserver $NAS_DNS" >> /etc/resolv.conf
	        fi
	fi
	if [ "$USAGE" = "DHCP" ]; then
		/sbin/lcd_disp -2 "$2 REQUEST DHCP IP"
		# check if we need to change nic mac address
		if [ "$2" = "WAN" ]; then
			if [ `/sbin/getcfg $DEV "VHWADDR Enable" -d "FALSE"` = "TRUE" ]; then
				if [ `/sbin/nic_getinfo $DEV vhwaddr` != "00:00:00:00:00:00" ]; then
					/sbin/ifconfig $DEV hw ether `/sbin/nic_getinfo $DEV vhwaddr` 2>/dev/null 1>/dev/null
				fi
			fi
		fi

		# check bonding interface
		if [ `/sbin/getcfg Network "BONDING Support" -u -d FALSE` = TRUE ]; then
			bonding_type=`/sbin/getcfg $DEV Bonding -u -d "FAIL OVER"`
			nic_cnt=`/sbin/getcfg Network "Interface Number" -d 2`
			bonding_cnt=`/sbin/getcfg Network "Bonding Interface Number" -d $nic_cnt`
			if [ "$bonding_type" != "STANDALONE" ]; then
				if [ $BONDING_UP = 1 ]; then
					BONDING_UP=`/usr/bin/expr $BONDING_UP + 1`
					/sbin/ifconfig bond0 $IPADDR netmask $NETMASK broadcast $BROADCAST up
					/sbin/ifenslave bond0 $DEV
					# wait for bonding linking ok
					/bin/sleep 7
				elif [ $BONDING_UP = 2 ]; then
					BONDING_UP=`/usr/bin/expr $BONDING_UP + 1`
					/sbin/ifenslave bond0 $DEV
					# wait for bonding linking ok
					/bin/sleep 7
					if [ $bonding_cnt -le 2 ]; then
						$DHCPCD_SH start "" bond0 &
					fi
				elif [ $BONDING_UP = 3 ]; then
					BONDING_UP=`/usr/bin/expr $BONDING_UP + 1`
					/sbin/ifenslave bond0 $DEV
					#wait for bonding linking ok
					/bin/sleep 7
					if [ $bonding_cnt -le 3 ]; then
						$DHCPCD_SH start "" bond0 &
					fi
				fi
				return
			else
				# *** Catherine modified ==>
				/sbin/ifconfig $DEV up
				$DHCPCD_SH start "" $DEV &
				# *** <== Catherine modified
				return
			fi
		else
			/sbin/ifconfig $DEV $IPADDR netmask $NETMASK broadcast $BROADCAST up
			$DHCPCD_SH start $3 &
		fi
	fi
	if [ "$USAGE" = "PPPOE" ]; then
		#========================================
		# must clear wan device ip address
		# or pppoe can't work correctly
		#=========================================
#		if [ -e /var/pppoe.status ]; then
			/bin/rm -f /var/pppoe.status
#		fi
		/sbin/ifconfig $DEV 0.0.0.0
		/usr/sbin/adsl-start
	fi
	if [ "$USAGE" = "PPTP" ]; then
#		if [ -e /var/pppoe.status ]; then
			/bin/rm -f /var/pppoe.status
#		fi
		/etc/init.d/pptp.sh start &
	fi
	#========================================
	# do we need to restart NAT and firewall?
	#========================================
	if [ "$USAGE" != "PPPOE" -a "$USAGE" != "PPTP" -a "$2" = "WAN" ]; then
		restart_dhcpd
		if [ `/sbin/getcfg NAT Support -u -d FALSE` = TRUE ]; then
			/sbin/init_fw
		fi
	fi
}

#=======================================================
#	begin to up all nic
#=======================================================
start_nic()
{
	if [ `/sbin/getcfg Network "BONDING Support" -u -d FALSE` = "TRUE" ]; then
		/bin/echo "insert bonding module..."
		interface=`/sbin/getcfg Network BONDING01 -u -d Interface1`
		dev1=`/sbin/getcfg Network $interface -d eth0`
		bonding=`/sbin/getcfg $dev1 Bonding -u -d "FAIL OVER"`
		if [ "$bonding" = "FAIL OVER" ]; then
			/sbin/insmod /lib/modules/misc/bonding.o mode=1 miimon=100
		elif [ "$bonding" = "LOAD BALANCE" ]; then
			/sbin/insmod /lib/modules/misc/bonding.o mode=0 miimon=100
		fi
	fi
	while read LINE
	do
		tmp=`/bin/echo $LINE | /sbin/cut -c-3`
		if [ "$tmp" = "WAN" ]; then
			DEV=`/bin/echo $LINE | /sbin/cut -c 5-8`
		fi
		if [ "$tmp" = "LAN" ]; then
			DEV=`/bin/echo $LINE | /sbin/cut -c 7-10`
		fi

		if [ "$1" = "" -o "$1" = "lcdrefresh" ]; then
			/sbin/lcd_disp -2 "Start Network $tmp"
			find_nic $DEV
			ifup_nic $NIC $tmp $1
		elif [ "$1" = "wan" -a "$tmp" = "WAN" ]; then
			/sbin/lcd_disp -2 "Start Network $tmp"
			find_nic $DEV
			ifup_nic $NIC $tmp
		elif [ "$1" = "lan" -a "$tmp" = "LAN" ]; then
			/sbin/lcd_disp -2 "Start Network $tmp"
			find_nic $DEV
			ifup_nic $NIC $tmp
		fi

#		/sbin/ifconfig $DEV up
#		/sbin/ifconfig $DEV down
#		NIC=`find_nic $DEV`
	done < /tmp/nic.dev
}

#==========================================================
#	stop one nic
#===========================================================
ifdown_nic()
{
	. $1
	/bin/echo "shutdown $DEV by using $USAGE"

	/etc/init.d/dhcpcd.sh stop
	if [ `/sbin/getcfg WLAN Support -u -d FALSE` = TRUE -a "$2" = "LAN" ]; then
		stop_wlan
	fi
	/sbin/ifconfig $DEV 0.0.0.0 down 2>/dev/null 1>/dev/null

	# del dhcp client pid file
#	if [ -e /var/run/dhcpcd*.pid ]; then
		/bin/rm /var/run/dhcpcd*.pid -f 2>/dev/null 1>/dev/null
		/bin/rm /etc/dhcpc/* -f 2>/dev/null 1>/dev/null
#	fi
	/bin/kill -9 `/sbin/pidof /sbin/dhcpcd` 2>/dev/null 1>/dev/null
	/usr/bin/setsid /usr/sbin/adsl-stop
	/etc/init.d/pptp.sh stop

	# need to del /etc/resolv.* ???
	if [ "$2" = "WAN" ]; then
		/bin/rm /etc/resolv.* -f 2>/dev/null 1>/dev/null
	fi
}

#===========================================================
#	begin to stop all nic
#===========================================================
stop_nic()
{
	while read LINE
	do
		tmp=`/bin/echo $LINE | /sbin/cut -c-3`
		if [ "$tmp" = "WAN" ]; then
			DEV=`/bin/echo $LINE | /sbin/cut -c 5-8`
		fi
		if [ "$tmp" = "LAN" ]; then
			DEV=`/bin/echo $LINE | /sbin/cut -c 7-10`
		fi
		if [ "$1" = "" -o "$1" = "lcdrefresh" ]; then
			find_nic $DEV
			ifdown_nic $NIC $tmp 
		elif [ "$1" = "wan" -a "$tmp" = "WAN" ]; then
			find_nic $DEV
			ifdown_nic $NIC $tmp
		elif [ "$1" = "lan" -a "$tmp" = "LAN" ]; then
			find_nic $DEV
			ifdown_nic $NIC $tmp
		fi
	done < /tmp/nic.dev
	if [ `/sbin/getcfg Network "BONDING Support" -u -d FALSE` = TRUE ]; then
		/bin/echo "remove bonding module..."
		/sbin/ifconfig bond0 down
		/sbin/rmmod bonding
	fi
}

#===============================================
#	write to /etc/hosts
#===============================================
write_hosts()
{
	/bin/echo "127.0.0.1		localhost		localhost" > /etc/hosts
	while read LINE
	do
		tmp=`/bin/echo $LINE | /sbin/cut -c-3`
		if [ "$tmp" = "WAN" ]; then
			DEV=`/bin/echo $LINE | /sbin/cut -c 5-8`
		fi
		if [ "$tmp" = "LAN" ]; then
			DEV=`/bin/echo $LINE | /sbin/cut -c 7-10`
		fi
		/bin/echo "`/sbin/nic_getinfo $DEV ip`		`/bin/hostname`		`/bin/hostname`" >> /etc/hosts
#		find_nic $DEV
#		NIC=`find_nic $DEV`
#		. $NIC
#		/bin/echo "$IPADDR		`/bin/hostname`		`/bin/hostname`" >> /etc/hosts
	done < /tmp/nic.dev
}

#====================================================
#	find correct nic conf file
#====================================================
find_dev()
{
	for i in /tmp/nic*.tmp; 
	do
		. $i
		if [ "$DEV" = "$1" ]; then
			return
		fi
	done
}

#=====================================================
#	up one nic, parameter1 is the conf file
#=====================================================
ifup_dev()
{
	if [ `/sbin/getcfg Network "BONDING Support" -u -d FALSE` = TRUE -a "$1" = "bond0" ]; then
		bondnic=`/sbin/getcfg Network Bonding01 -d Interface1`
		device=`/sbin/getcfg Network $bondnic -d eth0`

		/bin/echo "slave device = $device"
		find_dev $device

		DEV=$1
	else
		find_dev $1
	fi

	/bin/echo "start $DEV by using $USAGE"
	/bin/echo "default IP = $IPADDR"
	/bin/echo "netmask = $NETMASK"
	/bin/echo "broadcast = $BROADCAST"

	if [ "$USAGE" = "STATIC" ]; then
		if [ `/sbin/getcfg Network "BONDING Support" -u -d FALSE` != TRUE ]; then
			/sbin/ifconfig $DEV $IPADDR netmask $NETMASK broadcast $BROADCAST up
			/sbin/route add -host $IPADDR $DEV
			return
		fi

		/sbin/ifconfig $DEV $IPADDR netmask $NETMASK broadcast $BROADCAST up
		/sbin/route add -host $IPADDR $DEV
		/sbin/route add default gw $GATEWAY metric 1 $DEV
		if [ "$DEV" = "bond0" ]; then
			/bin/kill -SIGUSR1 `/sbin/pidof hwmond`
			bondcnt=`/sbin/getcfg Network "Bonding Interface Number" -d 2`
			for (( i = 1; i <= $bondcnt; i++ ));
			do
				bondnic=`/sbin/getcfg Network Bonding0$i -d Interface1`
				device=`/sbin/getcfg Network $bondnic -d eth0`
				/bin/echo "$DEV slave $device up"
				/sbin/ifenslave bond0 $device
			done
			/bin/sleep 3
			/bin/kill -SIGUSR1 `/sbin/pidof hwmond`
		fi
		return
	fi
	if [ "$USAGE" = "DHCP" ]; then
		/sbin/lcd_disp -2 "LAN REQUEST DHCP IP"
		# check if we need to change nic mac address
		# check bonding interface
		if [ "$DEV" = "bond0" ]; then
			/sbin/ifconfig bond0 $IPADDR netmask $NETMASK broadcast $BROADCAST up
			# wait for bonding linking ok
			/bin/sleep 7
			$DHCPCD_SH start "" bond0 &
		else
			/sbin/ifconfig $DEV $IPADDR netmask $NETMASK broadcast $BROADCAST up
			$DHCPCD_SH start "" $DEV &
		fi
	fi
}

# change a 8 byte-long IP to xxx.xxx.xxx.xxx format
ip_base_change()
{
	let x=0x`/bin/echo $1 | /sbin/cut -c 7-8`
	byte1=`/bin/echo $x`
	let x=0x`/bin/echo $1 | /sbin/cut -c 5-6`
	byte2=`/bin/echo $x`
	let x=0x`/bin/echo $1 | /sbin/cut -c 3-4`
	byte3=`/bin/echo $x`
	let x=0x`/bin/echo $1 | /sbin/cut -c 1-2`
	byte4=`/bin/echo $x`
	OUT_IP=$byte1.$byte2.$byte3.$byte4
}
	
reorder_router_table()
{
	if [ `/sbin/getcfg Network "BONDING Support" -u -d FALSE` = FALSE -o `/sbin/getcfg Network "Switch Support" -u -d FALSE` = FALSE ]; then
		return
	fi
#Shone added codes below, when in Load balance or fail over, it will appear [: too many arguments , because it return two words
	if [ `/sbin/getcfg eth0 Bonding -u -d FALSE` = STANDALONE ]; then
        	return
	fi


	# read original routing table to "/tmp/nasroute.tmp"
	/bin/cat /proc/net/route | /bin/sed '/^Iface/d' > /tmp/nasroute.org

	# bonding routing entries only
	/bin/sed '/eth2/d' /tmp/nasroute.org > /tmp/bond0.route

	# eth2 routing entries only
	/bin/sed '/bond0/d' /tmp/nasroute.org > /tmp/eth2.route

	while read LINE
	do
		interface=`/bin/echo $LINE | /sbin/cut -f 1 -d " "`
		ip_base_change `/bin/echo $LINE | /sbin/cut -f 2 -d " "`
		dest=$OUT_IP
		ip_base_change `/bin/echo $LINE | /sbin/cut -f 3 -d " "`
		gateway=$OUT_IP
		flag=`/bin/echo $LINE | /sbin/cut -f 4 -d " "`
		ip_base_change `/bin/echo $LINE | /sbin/cut -f 8 -d " "`
		netmask=$OUT_IP

		# flag is net (1) or gateway (3) or host (5) ?
		if [ $flag = 0001 ]; then
			/sbin/route del -net $dest netmask $netmask $interface
		elif [ $flag = 0003 ]; then
			/sbin/route del default gw $gateway $interface
		else 
			/sbin/route del -host $dest $interface
		fi
	done < /tmp/nasroute.org
	/bin/rm /tmp/nasroute.org

	# add new routing entries
	# switch (eth2) first
	while read LINE
	do
		ip_base_change `/bin/echo $LINE | /sbin/cut -f 2 -d " "`
		dest=$OUT_IP
		ip_base_change `/bin/echo $LINE | /sbin/cut -f 3 -d " "`
		gateway=$OUT_IP
		flag=`/bin/echo $LINE | /sbin/cut -f 4 -d " "`
		ip_base_change `/bin/echo $LINE | /sbin/cut -f 8 -d " "`
		netmask=$OUT_IP

		if [ $flag = 0001 ]; then
			/sbin/route add -net $dest netmask $netmask eth2
		elif [ $flag = 0003 ]; then
			/sbin/route add default gw $gateway netmask $netmask eth2
		else 
			/sbin/route add -host $dest eth2
		fi
	done < /tmp/eth2.route
	/bin/rm /tmp/eth2.route

	# bond0 last
	while read LINE
	do
		ip_base_change `/bin/echo $LINE | /sbin/cut -f 2 -d " "`
		dest=$OUT_IP
		ip_base_change `/bin/echo $LINE | /sbin/cut -f 3 -d " "`
		gateway=$OUT_IP
		flag=`/bin/echo $LINE | /sbin/cut -f 4 -d " "`
		ip_base_change `/bin/echo $LINE | /sbin/cut -f 8 -d " "`
		netmask=$OUT_IP

		if [ $flag = 0001 ]; then
			/sbin/route add -net $dest netmask $netmask bond0
		elif [ $flag = 0003 ]; then
			/sbin/route add default gw $gateway netmask $netmask bond0
		else 
			/sbin/route add -host $dest bond0
		fi
	done < /tmp/bond0.route
	/bin/rm /tmp/bond0.route
}

#============================================
#	shell script main function
#============================================

BRIDGE_DEV="br0"
WLAN_DEV="wlan0"
TARGET_DEV=$3

case "$1" in
	start)
		/sbin/ifconfig lo 127.0.0.1 up
		# generate nic files in /tmp directory
		if [ "$TARGET_DEV" = "" ]; then
			/sbin/init_nic

			/bin/echo "Configuring and Starting Network:"
			$0 stop $2
			start_nic $2
#			/sbin/route add -host 255.255.255.255 eth0
		else
			ifup_dev $TARGET_DEV
		fi
		write_hosts $2
		/bin/kill -HUP `/sbin/pidof sysmond` 2>/dev/null 1>/dev/null
		/sbin/lcd_disp -C -R
		reorder_router_table
		;;
	restart)
		$0 start $2
		;;
	stop)
		stop_nic $2
		;;
	*)
		/bin/echo "Usage: network.sh {start|stop|restart} [wan|lan]"
		/bin/echo "       default of parameter 2 is all network device"
		exit 1
		;;
esac

