#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <sys/file.h>
#include "shm_conf.h"
#include "uLinux.h"
#include "config.h"
#include "Util.h"

typedef unsigned char uint8;
typedef struct shm_header
{
	uint8 dirty;
	pid_t pid;
	struct tm timestamp;
} SHM_HEADER;

extern int Open_File(char* , int, FILE** );
extern int Close_File(char* , int, FILE*);
extern char *get_one_line_of_string(FILE *fp);
extern int get_section_name(char *str, char *section_name, int buf_size);
extern int get_field_name(char *str, char *field_name, int buf_size);
extern int conf_get_field(char *conf_file, char *section, char *field, char *result, int buf_size);

int conf_set_field(char *conf_file, char *section_name, char *field_name, char *field_data);
static int shm_init(int key,int size,void **shm_addr_ptr,int *shm_id_ptr,BOOL *isnew);
static int shm_close(void *shm_addr);
static int shm_destroy(int shmid);

static int shm_lock(pid_t pid, void *shm_addr, BOOL force_lock);
static BOOL is_shm_locked(void *shm_addr, pid_t *pid, struct tm* tp);
static int shm_unlock(pid_t pid, void *shm_addr);

//------------------ Private Functions ----------------------
int conf_set_field(char *conf_file, char *section_name, char *field_name,
		char *field_data)
{
	FILE *fp = NULL, *tmpfp = NULL;
	char buf[BUF_SIZE];
	char *p;
	int i, dirty = 0, result = ERROR_FAIL;

	if ((i = Open_File(conf_file,O_RDONLY, &fp)) != SUCCESS)
		return i;
	
	rewind(fp);
	tmpfp = tmpfile();		// temp file for buffer

	if (NULL != fp) {
		int section_found = 0;
		if (EMPTY_STRING(section_name)) section_found = 1;
		while ((p = get_one_line_of_string(fp)) != NULL) {
			if (get_section_name(p, buf, sizeof(buf)) == SUCCESS) {
				if (strcasecmp(section_name, buf) == 0)
					section_found = 1;
				else {
					if (section_found && (!dirty)) {
						// no existing field in section
						dirty++;
						fprintf(tmpfp, "%s = %s\n",
							field_name, field_data);
					}
					section_found = 0;
				}
			}
			else
			if (section_found) {
				if (get_field_name(p,buf,sizeof(buf))==SUCCESS){
					if (strcasecmp(field_name,buf) == 0) {
						// field name match...
						dirty++;
						fprintf(tmpfp, "%s = %s\n",
							field_name, field_data);
						continue;
						// don't copy this line...
					}
				}
			}
			if(fputs(p, tmpfp) < 0) { // Catherine 2002/03/20 close all files before return
				Close_File(conf_file,O_RDONLY,fp);
				fclose(tmpfp);
				return ERROR_WRITE_FILE;
			}
			//end
			fputs("\n", tmpfp);
		}
		Close_File(conf_file,O_RDONLY,fp);

		if (!dirty) {
			if (!section_found)
				fprintf(tmpfp, "[%s]\n", section_name);
			fprintf(tmpfp, "%s = %s\n", field_name, field_data);
			dirty++;
		}

		if (dirty) {
			if ((i=Open_File(conf_file,O_WRONLY, &fp)) == SUCCESS) {
				rewind(tmpfp);
				while ((i=fread(buf,1,sizeof(buf),tmpfp))>0) {
					fwrite(buf, 1, i, fp);
				}
				Close_File(conf_file,O_WRONLY,fp);
				result = SUCCESS;
			}
			else
				result = i;
		}
		fclose(tmpfp);
	}

	return result;
}

// lock shared memory by setting the dirty flag, process id & time stamp
static int shm_lock(pid_t pid, void *shm_addr, BOOL force_lock)
{
	SHM_HEADER *sh;
	time_t t_now;
	
	sh = (SHM_HEADER*) shm_addr;
	
	if ((!force_lock) && (sh->dirty==1)) // already locked
		return ERROR_FAIL;

	sh->dirty = 1;
	sh->pid = pid;
	time(&t_now);
	memcpy((void*)&(sh->timestamp),(void*)localtime(&t_now),
		sizeof(struct tm));
#ifdef DEBUG
	printf("shm_lock: dirty=%u, time stamp=[%d/%d/%d %d:%d:%d]\n",
			sh->dirty, sh->timestamp.tm_year + 1900,
			sh->timestamp.tm_mon + 1, sh->timestamp.tm_mday,
			sh->timestamp.tm_hour, sh->timestamp.tm_min,
			sh->timestamp.tm_sec);
#endif
	return (SUCCESS);
}

// is the dirty flag set?
// if so, also return the shm lock owner & the last lock time stamp
static BOOL is_shm_locked(void *shm_addr, pid_t *pid, struct tm* tp)
{
	SHM_HEADER *sh;
	
	sh = (SHM_HEADER*) shm_addr;
	if (sh->dirty==0)
		return FALSE;
	else {
		if (pid!=NULL)
			*pid = sh->pid;

		if (tp!=NULL)
			memcpy((void*)tp, (void*)&(sh->timestamp),
				sizeof(struct tm));
#ifdef DEBUG
		printf("is_shm_locked: last lock time=[%d/%d/%d %d:%d:%d]\n",
			sh->timestamp.tm_year + 1900,
			sh->timestamp.tm_mon + 1, sh->timestamp.tm_mday,
			sh->timestamp.tm_hour, sh->timestamp.tm_min,
			sh->timestamp.tm_sec);
		printf("is_shm_locked: pid = %d\n", sh->pid);
#endif
		return TRUE;
	}
}

// unlock the shared memory by resetting the dirty flag,
// the process id & the time stamp
static int shm_unlock(pid_t pid, void *shm_addr)
{
	SHM_HEADER *sh;
	
	sh = (SHM_HEADER*) shm_addr;

	if (pid!=sh->pid) // not the process that lockes the shared memory
		return ERROR_FAIL;

	sh->pid = 0;
	memset((void*)&(sh->timestamp), 0, sizeof(struct tm));
	sh->dirty = 0;
#ifdef DEBUG
	printf("shm_unlock\n");
#endif
	return (SUCCESS);
}

// create & attach a shared memory (if not existed before, clean the memory)
static int shm_init(int key, int size, void **shm_addr_ptr, int *shm_id_ptr,
		BOOL *isnew)
{
	int shm_id;
	void* shm_addr=(void*)0;
	BOOL dirty=FALSE;
	
	if (key<0 || size<=0)
		return (ERROR_INVALID_PARAM);
		
	shm_id = shmget(key, size, 0666);
	if (shm_id==-1)  {
		// fail to get the existed shared memory
#ifdef DEBUG
		printf("Fail to get the shared memory for key=%d, size=%d!\n",
			key, size);
#endif		
		dirty = TRUE;
		shm_id = shmget(key, size, 0666 | IPC_CREAT);
		if (shm_id==-1) {
#ifdef DEBUG
			printf("Fail to create the shared memory, too!\n");
#endif
			return (ERROR_EXEC_FAIL);
		}
#ifdef DEBUG
		printf("Success to create the shared memory!\n");
#endif
	}
		
#ifdef DEBUG
	printf("Shared memory open succesfully!\n");
#endif

	shm_addr = shmat(shm_id, (void*) 0, 0);
	if (shm_addr==(void *)-1) {
#ifdef DEBUG
		printf("Fail to attach the shared memory!\n");
#endif
		return (ERROR_EXEC_FAIL);
	}
	
	if (dirty) {
#ifdef DEBUG
		printf("shm_init: clear unused memory %d!\n", (int)shm_addr);
#endif
		memset(shm_addr, 0, size);
	}
	
	*shm_addr_ptr = shm_addr;
	*shm_id_ptr = shm_id;
	*isnew=dirty;
	return SUCCESS;
}

// detach the shared memory
static int shm_close(void *shm_addr)
{
	if (shmdt(shm_addr)==-1) {
#ifdef DEBUG
		printf("Fail to detach the shared memory!\n");
#endif
		return (ERROR_EXEC_FAIL);
	}
	
	return (SUCCESS);
}

// destroy the shared memory
static int shm_destroy(int shmid)
{
	if (shmid>=0) { 
		if (shmctl(shmid, IPC_RMID, 0)==-1) {
#ifdef DEBUG
			printf("Fail to destroy the shared memory id = %d!\n" , shmid);
#endif
			return (ERROR_EXEC_FAIL);
		}
	}
	else return (ERROR_INVALID_PARAM);
	
	return (SUCCESS);
}

//--------------------------- Public Functions --------------------------------
//-----------------------------------------------------------------------------
// SHM_Get_Config : get the value of a config item from shared memory,
//		if exists; otherwise, create a shared memory and get the item
//		from the config file
// Output: value_ptr -- pointer to the output config data
// Return: SUCCESS -- the returned value is put into shmd->value
//	ERROR_BUFFER_TOO_SMALL -- the input size is not enough
//	ERROR_INVALID_PARAM -- the key or size is not valid
//	ERROR_EXEC_FAIL -- fail to create/open/attach the shared memory
//	ERROR_BUSY -- the data is being modified by others
//	ERROR_NOT_FOUND -- the item does not exist in either the shared
//				memory or the config file
//	ERROR_OPEN_FILEGfail to open the config file
//-----------------------------------------------------------------------------
int SHM_Get_Config(int key, int size, char* conf_file, char* section,
		char* field, char **value_ptr)
{
	int shmid;
	void* shm_addr;
	int ret;
	BOOL is_fresh=FALSE;
	char *result;
	char *data_ptr;
	struct tm tstamp;
	time_t now;
	double wait_time;
	pid_t act_pid=0, mypid = getpid();

#ifdef DEBUG
	SHM_HEADER *sh;
	printf("SHM_Get_Config(%d, %d, %s, %s, %s)\n",
		key, size, conf_file, section, field);
#endif

	ret = shm_init(key, size+sizeof(SHM_HEADER),&shm_addr,&shmid,&is_fresh);
	if (ret!=SUCCESS)
		return (ret);

#ifdef DEBUG
	printf("shm_init OK : shm_addr=%d,shm_id=%d,is_fresh=%d,content=%s\n",
		(int)shm_addr, shmid, (int)is_fresh, (char*)shm_addr);
	sh = (SHM_HEADER*) shm_addr;
#endif

	data_ptr = (char*)(shm_addr + sizeof(SHM_HEADER));
	
	if (is_fresh) {
		// no one has used the shared memory
		if (SUCCESS==shm_lock(mypid, shm_addr, TRUE)) {
			result = (char*) calloc(size, sizeof(char));
			if ((ret = conf_get_field(conf_file, section, field,
				result, size))!=SUCCESS) {
#ifdef DEBUG
				printf("Fail to get field(%s, %s, %s),error code=%d!\n",
					conf_file, section, field, ret);
#endif
				free(result);
				shm_unlock(mypid, shm_addr); // clear dirty flag
				shm_close(shm_addr);
				shm_destroy(shmid);
				return (ret);
			}
#ifdef DEBUG
			printf("Conf_Get_Field OK : %s\n", result);
#endif
			strncpy(data_ptr, result, size);
			shm_unlock(mypid, shm_addr); // clear dirty flag
			free(result);
#ifdef DEBUG
			printf("Finish reading: %s\n", data_ptr);
#endif
		}
		else {
			shm_close(shm_addr);
			shm_destroy(shmid);
			return (ret);
		}
	}

	time(&now);
	while ((is_shm_locked(shm_addr, &act_pid, &tstamp)) &&
		(act_pid!=mypid) &&
		((wait_time=difftime(now,mktime(&tstamp)))
			<(double)MAX_WAIT_TIME)) {
#ifdef DEBUG
		printf("SHM_Get_Config:I'm waiting.....\n");
#endif		
		sleep(1);
		time(&now);
	}
#ifdef DEBUG
	printf("Time Stamp = %f, Now = %f\n",
		(double)mktime(&tstamp), (double)now);
	printf("Wait Time = %f\n", wait_time);
#endif
	
	if  ((is_shm_locked(shm_addr, &act_pid, &tstamp)) &&
		(act_pid!=mypid) &&
		((wait_time=difftime(now,mktime(&tstamp)))
			>=(double)MAX_WAIT_TIME)) {
		// some process might be dead, re-read from the config file
#ifdef DEBUG
		printf("The shared memory is busy since [%d/%d/%d %d:%d:%d]\n",
			sh->timestamp.tm_year + 1900,
			sh->timestamp.tm_mon + 1, sh->timestamp.tm_mday,
			sh->timestamp.tm_hour, sh->timestamp.tm_min,
			sh->timestamp.tm_sec);
#endif

		if (SUCCESS==shm_lock(mypid, shm_addr, TRUE)) {
			result = (char*) calloc(size, sizeof(char));
			if ((ret = conf_get_field(conf_file, section, field,
				result, size))!=SUCCESS) {
#ifdef DEBUG
				printf("Fail to get field(%s, %s, %s),
					error code=%d!\n",
					conf_file, section, field, ret);
#endif
				free(result);
				shm_unlock(mypid, shm_addr);
				shm_close(shm_addr);
				shm_destroy(shmid);
				return (ret);
			}
#ifdef DEBUG
			printf("Conf_Get_Field OK : %s\n", result);
#endif
			strncpy(data_ptr, result, size);
			shm_unlock(mypid, shm_addr);
			free(result);
#ifdef DEBUG
			printf("Finish reading: %s\n", data_ptr);
#endif
			ret = SUCCESS;
		}
		else ret = ERROR_BUSY;
	}
	else {
		*value_ptr = (char*)calloc(size, sizeof(char));
		strncpy((*value_ptr), data_ptr, size);
#ifdef DEBUG
		printf("The data to return = %d, %s\n", (int)(*value_ptr),
			*value_ptr);
#endif
		ret = SUCCESS;	
	}

	shm_close(shm_addr);
	return (ret);
}

//-----------------------------------------------------------------------------
// SHM_Set_Config : set the value of a config item to both the shared memory
// and the config file
// Required input fields: key, size, path, section, field, value
// Return: SUCCESS -- OK
//	ERROR_INVALID_PARAM -- the key or size is not valid
//	ERROR_EXEC_FAIL -- fail to create/open/attach the shared memory
//	ERROR_OPEN_FILEGfail to open the config file
//	ERROR_WRITE_FILE -- fail to write to the config file
//------------------------------------------------------------------------------
int SHM_Set_Config(int key, int size, char* path, char* section, char* field,
		char* value)
{
	int shmid;
	void* shm_addr;
	int ret;
	BOOL is_fresh;
	char *data_ptr;
	struct tm tstamp;
	int shm_header_size = sizeof(SHM_HEADER);
	time_t now;
	double wait_time=0;
	pid_t act_pid=0, mypid=getpid();
	int	count;
	
#ifdef DEBUG
	printf("SHM_Set_Config(%d, %d, %s, %s, %s, %s)\n",
		key, size, path, section, field, value);
#endif
	ret = shm_init(key,size + shm_header_size,&shm_addr,&shmid,&is_fresh);
	if (ret!=SUCCESS)
		return (ret);
		
	data_ptr = (char*) (shm_addr + shm_header_size);

	if (!is_fresh) {
		// check if anyone is writing
		time(&now);
		while ((is_shm_locked(shm_addr, &act_pid, &tstamp)) &&
			(act_pid!=mypid) &&
			((wait_time=difftime(now, mktime(&tstamp)))
				<(double)MAX_WAIT_TIME)) {
			sleep(1);
			time(&now);
#ifdef DEBUG
			printf("SHM_Set_Config:I've been waiting for %f seconds\n", wait_time);
#endif
		}
#ifdef DEBUG
		// the shared mem has been locked for too long?
		if ((is_shm_locked(shm_addr, &act_pid, &tstamp)) &&
			(act_pid!=mypid) &&
			((wait_time=difftime(now, mktime(&tstamp)))
				>=(double)MAX_WAIT_TIME))
			printf("SHM_Set_Config: time out!\n");
#endif
	}

	if (SUCCESS==shm_lock(mypid, shm_addr, TRUE)) {
		// lock the shared memory
		count=0;
		do
		{
			ret=conf_set_field(path, section, field, value);
			if (ret==SUCCESS)
			{
				strncpy(data_ptr, value, size);
				shm_unlock(mypid, shm_addr);
				shm_close(shm_addr);
				return SUCCESS;
			}
			else
			{
				sleep(1);
				count++;
			}
		} while (ret!=SUCCESS && count<30);

		if ((ret = conf_set_field(path, section, field, value))!=0) {
#ifdef DEBUG
			printf("Failed to set field(%s, %s, %s, %s), error code=%d!\n",
				path, section, field, value, ret);
#endif
			shm_unlock(mypid, shm_addr);
			shm_close(shm_addr);
			return (ret);
		}
	
		strncpy(data_ptr, value, size);
		shm_unlock(mypid, shm_addr); //unlock the shared memory
		ret = SUCCESS;
	}
	else ret = ERROR_BUSY;
	
	shm_close(shm_addr);
	return (ret);
}

//------------------------------------------------------------------------------
// SHM_Free : destroy (free) the shared memory
// Return: SUCCESS -- OK
//	ERROR_NOT_EXISTED -- the item does not exist in the shared memory
//	ERROR_EXEC_FAIL -- fail to destroy the shared memory
//------------------------------------------------------------------------------
int SHM_Free(int key, int size)
{
	int shm_id;
	
	if ((shm_id = shmget(key, size+sizeof(SHM_HEADER), 0666))==-1) {
#ifdef DEBUG
		printf("Fail to get the shared memory for the key = %d!\n" , key);
#endif		
		return (ERROR_NOT_EXISTED);
	}
		
	if (shmctl(shm_id, IPC_RMID, 0)==-1) {
#ifdef DEBUG
		printf("Fail to destroy the shared memory id = %d!\n" , shm_id);
#endif
		return (ERROR_EXEC_FAIL);
	}
	
	return (SUCCESS);
}

//-----------------------------------------------------------------------------
// SHM_Refresh : refresh the shared memory data from the config file
// Return: SUCCESS -- the returned value is put into shmd->value
//	ERROR_BUFFER_TOO_SMALL -- the input size is not enough
//	ERROR_INVALID_PARAM -- the key or size is not valid
//	ERROR_EXEC_FAIL -- fail to create/open/attach the shared memory
//	ERROR_BUSY -- the data is being modified by others
//	ERROR_NOT_FOUND -- the item does not exist in either the shared
//				memory or the config file
//	ERROR_OPEN_FILE -- fail to open the config file
//-----------------------------------------------------------------------------
int SHM_Refresh(int key, int size, char* conf_file, char* section, char* field)
{
	int shmid;
	void* shm_addr;
	int ret;
	BOOL is_fresh=FALSE;
	char *result;
	char *data_ptr;
	struct tm tstamp;
	time_t now;
	pid_t act_pid=0, mypid = getpid();

#ifdef DEBUG
	SHM_HEADER *sh;
	printf("SHM_Refresh(%d, %d, %s, %s, %s)\n",
		key, size, conf_file, section, field);
#endif

	ret = shm_init(key, size+sizeof(SHM_HEADER),&shm_addr,&shmid,&is_fresh);
	if (ret!=SUCCESS)
		return (ret);

#ifdef DEBUG
	printf("shm_init OK : shm_addr=%d,shm_id=%d,is_fresh=%d,content=%s\n",
		(int)shm_addr, shmid, (int)is_fresh, (char*)shm_addr);
	sh = (SHM_HEADER*) shm_addr;
#endif

	data_ptr = (char*)(shm_addr + sizeof(SHM_HEADER));
	time(&now);
	while ((is_shm_locked(shm_addr, &act_pid, &tstamp)) &&
		(act_pid!=mypid) &&
		(difftime(now,mktime(&tstamp))
			<(double)MAX_WAIT_TIME)) {
#ifdef DEBUG
		printf("SHM_Get_Config:I'm waiting.....\n");
#endif		
		sleep(1);
		time(&now);
	}

	if (SUCCESS==shm_lock(mypid, shm_addr, TRUE)) {
		result = (char*) calloc(size, sizeof(char));
		if ((ret = conf_get_field(conf_file, section, field,
			result, size))!=SUCCESS) {
#ifdef DEBUG
			printf("Fail to get field(%s, %s, %s),
				error code=%d!\n",
				conf_file, section, field, ret);
#endif
			free(result);
			shm_unlock(mypid, shm_addr);
			shm_close(shm_addr);
			shm_destroy(shmid);
			return (ret);
		}
#ifdef DEBUG
		printf("Conf_Get_Field OK : %s\n", result);
#endif
		strncpy(data_ptr, result, size);
		shm_unlock(mypid, shm_addr);
		free(result);
#ifdef DEBUG
		printf("Finish reading: %s\n", data_ptr);
#endif
		ret = SUCCESS;
	}
	else ret = ERROR_BUSY;

	shm_close(shm_addr);
	return (ret);
}

#ifdef DEBUG
void main() {
	int run=1;
	int func, ret, key, size;
	int shm_id;
	char *value, conf_file[BUF_SIZE], section[20], field[20];
	void *shm_addr;
	SHM_HEADER *sh;
	BOOL isnew;
	pid_t mypid=getpid();

	while(run==1) {
		printf("Current Process ID = %d\n", mypid);
		printf("Max Wait Time = %d\n", (int)MAX_WAIT_TIME);
		printf("What do you want?\n");
		printf("0. Quit\n");
		printf("1. int SHM_Get_Config(int key, int size, char* conf_file, char* section, char* field)\n");
		printf("2. int SHM_Set_Config(int key, int size, char* path, char* section, char* field, char* value)\n");
		printf("3. int SHM_Free(int key, int size)\n");
		printf("4. Real Free SHM (int key, int realsize)\n");
		printf("5. Lock SHM only (int key, int size)\n");
		printf("6. Unlock SHM (int key, int size)\n");
		printf("7. Dump the SHM (int key, int size)\n");
		printf("8. int SHM_Refresh(int key, int size, char* conf_file, char* section, char* field)\n");
		scanf("%d", &func);
		
		switch (func) {
			case 0:
				run = 0;
				break;
			case 1:
				printf("Input parameters: ");
				scanf("%d %d %s %s %s", &key, &size,
					conf_file, section, field);
				printf("You key in [%d][%d][%s][%s][%s]\n",
					key, size, conf_file, section, field);
				ret = SHM_Get_Config(key, size,
					conf_file, section, field, &value);
				if (ret==SUCCESS) {
					printf("Result = %d, return string = %s\n", ret, value);
					if (value!=NULL) {
						printf("Before Free %d....\n",
							(int)value);
						free(value);
					}
				}
				else printf("Result = %d\n", ret);
				break;
			case 2:
				printf("Input parameters: ");
				value = (char*)calloc(BUF_SIZE, sizeof(char));
				scanf("%d %d %s %s %s %s",
					&key, &size, conf_file, section,
					field, value);
				printf("You key in [%d][%d][%s][%s][%s][%s]\n",
					key,size,conf_file,section,field,value);
				ret = SHM_Set_Config(key, size, conf_file,
					section, field, value);
				printf("Result = %d\n", ret);
				free(value);
				break;
			case 3:
				printf("Input parameters: ");
				scanf("%d %d", &key, &size);
				ret = SHM_Free(key, size);
				printf("Result = %d\n", ret);
				break;
			case 4:
				printf("Input parameters: ");
				scanf("%d %d", &key, &size);
	
				if ((shm_id = shmget(key, size, 0666))==-1) {
					printf("Fail to get the shared memory for the key = %d!\n" , key);
					break;
				}
				if (shmctl(shm_id, IPC_RMID, 0)==-1) {
					printf("Fail to destroy the shared memory id = %d!\n" , shm_id);
					break;
				}
				printf("Success!\n");
				break;
			case 5:
				printf("Input parameters: ");
				scanf("%d %d", &key, &size);
				size+=sizeof(SHM_HEADER);

				if (SUCCESS!=shm_init(key,size,&shm_addr,
					&shm_id, &isnew)) {
					printf("Fail to get the shared memory for the key = %d!\n" , key);
					break;
				}
				printf("Is Newly Allocated? %s\n",
					(isnew) ? "TRUE" : "FALSE");
				mypid = getpid();
				if (SUCCESS==shm_lock(mypid, shm_addr, TRUE)) {
					sh = (SHM_HEADER*)shm_addr;
					printf("Lock OK!\n");
					printf("SHM Dump.....\n");
					printf("Dirty: %u\n", sh->dirty);
					printf("PID: %d\n", sh->pid);
					printf("TimeStamp: %d/%d/%d %d:%d:%d\n",
						sh->timestamp.tm_year + 1900,
						sh->timestamp.tm_mon + 1,
						sh->timestamp.tm_mday,
						sh->timestamp.tm_hour,
						sh->timestamp.tm_min,
						sh->timestamp.tm_sec);
					printf("Data: %s\n",(char*)(shm_addr+sizeof(SHM_HEADER)));
				}
				else
					printf("Fail to lock!\n");

				shm_close(shm_addr);
					
				break;
			case 6:
				printf("Input parameters: ");
				scanf("%d %d", &key, &size);
				size+=sizeof(SHM_HEADER);
				if (SUCCESS!=shm_init(key,size,&shm_addr,
					&shm_id, &isnew)) {
					printf("Fail to get the shared memory for the key = %d!\n" , key);
					break;
				}
				printf("Is Newly Allocated? %s\n",
					(isnew) ? "TRUE" : "FALSE");
				mypid = getpid();
				if (SUCCESS==shm_unlock(mypid, shm_addr)) {
					sh = (SHM_HEADER*)shm_addr;
					printf("Unlock OK!\n");
					printf("Dirty: %u\n", sh->dirty);
					printf("Data: %s\n",(char*)(shm_addr+sizeof(SHM_HEADER)));
				}
				else
					printf("Fail to unlock!\n");

				shm_close(shm_addr);
					
				break;	
			case 7:
				printf("Input parameters: ");
				scanf("%d %d", &key, &size);
				size+=sizeof(SHM_HEADER);

				if (SUCCESS!=shm_init(key,size,&shm_addr,
					&shm_id, &isnew)) {
					printf("Fail to get the shared memory for the key = %d!\n" , key);
					break;
				}
				sh = (SHM_HEADER*)shm_addr;
				printf("SHM Dump.....\n");
				printf("Dirty: %u\n", sh->dirty);
				printf("PID: %d\n", sh->pid);
				printf("TimeStamp: %d/%d/%d %d:%d:%d\n",
					sh->timestamp.tm_year + 1900,
					sh->timestamp.tm_mon + 1,
					sh->timestamp.tm_mday,
					sh->timestamp.tm_hour,
					sh->timestamp.tm_min,
					sh->timestamp.tm_sec);
				printf("Data: %s\n",(char*)(shm_addr+sizeof(SHM_HEADER)));
				shm_close(shm_addr);
				break;
			case 8:
				printf("Input parameters: ");
				scanf("%d %d %s %s %s", &key, &size,
					conf_file, section, field);
				printf("You key in [%d][%d][%s][%s][%s]\n",
					key, size, conf_file, section, field);
				ret = SHM_Refresh(key, size,
					conf_file, section, field);
				printf("Result = %d\n", ret);
				break;
			default:
		}
	}
}
#endif
