//**************************************************************************
//
//	Copyright (c) 2002  ICP Electronics Inc.  All Rights Reserved.
//
//	FILE:
//		lvm_config.c
//
//	Abstract: 
//		provide LVM config file interface fuctions for cgi
//
//	HISTORY:
//		2002/08/07 Catherine Shen -- make Get_LV_Mt_Info public
//					(replace local function "get_lv_mt_info")
//		2002/04/17 Catherine Shen -- created for a new feature (LVM)
//				handling snapshot volumes config file issues
//
//**************************************************************************
#include <ctype.h>
#include <sys/vfs.h>
#include <dirent.h>
#include <sys/stat.h>
#include "naslvm.h"

extern void sort_drive_no_for_show(int *drive_list, int list_cnt);
extern long long get_lv_free_size(char* lv_name, BOOL is_snap);
extern int get_lv_info(char* lv_name, LV_INFO* lv_info);
extern int get_pv_info(char* pv_name, PV_INFO* pv_info);
extern int create_directory(char *full_path);
extern int convert_md_name(char* md_path, char* mdname, int buf_size, int out_format);
extern long long get_approx_size(long long org_size, int method);
extern long long get_lv_total_size(char* lv_name);
extern int create_snap_share(char* sn_lv);
extern void get_drive_alphabetic_order(int hd_no, char start_char, char* out_buf, int buf_size);
extern int get_drive_no_by_alphabetic_order(char start_char, char* in_buf);

time_t get_build_time_from_instance_name(char* snap_inst_name);

int Get_LV_Mt_Info(LV_MOUNT_INFO *lv_minfo)
{
	struct statfs fsd;
  	double tmp_blocks, tmp_bfree, tmp_bsize, tmp_bavail;

	if ( statfs(lv_minfo->c_LV_Mount_Point,&fsd) != 0) {
#ifdef DEBUG
		printf("Get_LV_Mt_Info: statfs(%s) fails\n",
			lv_minfo->c_LV_Mount_Point);
#endif
		return ERROR_FAIL;
	}

	tmp_blocks = fsd.f_blocks;
  	tmp_bfree  = fsd.f_bfree;     /** free blocks in fs **/
  	tmp_bavail = fsd.f_bavail;    /** free blocks for non-superuser **/
  	tmp_bsize  = fsd.f_bsize;

	lv_minfo->ll_LV_Total_Size = (tmp_blocks * tmp_bsize) / 1024 ;
  	lv_minfo->ll_LV_Free_Size = (tmp_bavail *  tmp_bsize) / 1024;
  	lv_minfo->ll_LV_Used_Size = lv_minfo->ll_LV_Total_Size - lv_minfo->ll_LV_Free_Size;

	return SUCCESS;
}

int m_lvm_cmp(const void *m1,const void *m2)
{

	LVM_VOLUME_CONF *a,*b;
	
	a = (LVM_VOLUME_CONF*) m1;
	b = (LVM_VOLUME_CONF*) m2;

	if(a->vol_no > b->vol_no ) return 1;
	if(a->vol_no < b->vol_no ) return -1;
	
	return 0;

}

int m_sn_cmp(const void *m1,const void *m2)
{

	SNAPSHOT_VOLUME_CONF *a,*b;
	
	a = (SNAPSHOT_VOLUME_CONF*) m1;
	b = (SNAPSHOT_VOLUME_CONF*) m2;

	if(a->i_Snapshot_No > b->i_Snapshot_No ) return 1;
	if(a->i_Snapshot_No < b->i_Snapshot_No ) return -1;
	
	return 0;

}

int m_sn_info_cmp(const void *m1,const void *m2)
{

	SNAPSHOT_VOLUME_INFO *a,*b;
	
	a = (SNAPSHOT_VOLUME_INFO*) m1;
	b = (SNAPSHOT_VOLUME_INFO*) m2;

	if(a->snap_conf.i_Snapshot_No > b->snap_conf.i_Snapshot_No ) return 1;
	if(a->snap_conf.i_Snapshot_No < b->snap_conf.i_Snapshot_No ) return -1;
	
	return 0;

}

int m_lvm_cmp_for_show(const void *m1,const void *m2)
{

	LVM_VOLUME_INFO *a,*b;
	
	a = (LVM_VOLUME_INFO*) m1;
	b = (LVM_VOLUME_INFO*) m2;

	if(a->vol_data.raid_level > b->vol_data.raid_level ) return 1;
	if(a->vol_data.raid_level < b->vol_data.raid_level ) return -1;
	if(a->vol_data.drive_no_list[0] > b->vol_data.drive_no_list[0] ) return 1;
	if(a->vol_data.drive_no_list[0] < b->vol_data.drive_no_list[0] ) return -1;
	if(a->vol_data.drive_no_list[0] == b->vol_data.drive_no_list[0] ) return 0;
	
	return 0;

}

int m_sn_cmp_for_show(const void *m1,const void *m2)
{

	SNAPSHOT_VOLUME_CONF *a,*b;
	
	a = (SNAPSHOT_VOLUME_CONF*) m1;
	b = (SNAPSHOT_VOLUME_CONF*) m2;

	if(strcmp(a->c_Src_LV, b->c_Src_LV)>0) return 1;
	if(strcmp(a->c_Src_LV, b->c_Src_LV)<0) return -1;
	if(a->t_Build_Time > b->t_Build_Time ) return 1;
	if(a->t_Build_Time < b->t_Build_Time ) return -1;
	if(a->t_Build_Time == b->t_Build_Time ) return 0;
	
	return 0;

}

int m_sn_info_cmp_for_show(const void *m1,const void *m2)
{

	SNAPSHOT_VOLUME_INFO *a,*b;
	
	a = (SNAPSHOT_VOLUME_INFO*) m1;
	b = (SNAPSHOT_VOLUME_INFO*) m2;

	if(strcmp(a->snap_conf.c_Src_LV, b->snap_conf.c_Src_LV)>0) return 1;
	if(strcmp(a->snap_conf.c_Src_LV, b->snap_conf.c_Src_LV)<0) return -1;
	if(a->snap_conf.t_Build_Time > b->snap_conf.t_Build_Time ) return 1;
	if(a->snap_conf.t_Build_Time < b->snap_conf.t_Build_Time ) return -1;
	if(a->snap_conf.t_Build_Time == b->snap_conf.t_Build_Time ) return 0;
	
	return 0;

}

int m_snapd_cmp(const void *m1,const void *m2)
{
	SNAPD_CONF_ENTRY *a,*b;
	
	a = (SNAPD_CONF_ENTRY*) m1;
	b = (SNAPD_CONF_ENTRY*) m2;

	if(a->start_time > b->start_time) return 1;
	if(a->start_time < b->start_time) return -1;
	if(a->snap_idx > b->snap_idx) return 1;
	if(a->snap_idx < b->snap_idx) return -1;

	return 0;
}

int m_snap_inst_cmp(const void *m1,const void *m2)
{
	SNAPSHOT_VOLUME_INSTANCE_INFO *a,*b;
	
	a = (SNAPSHOT_VOLUME_INSTANCE_INFO*) m1;
	b = (SNAPSHOT_VOLUME_INSTANCE_INFO*) m2;

	if(a->t_Build_Time > b->t_Build_Time) return 1;		// a : new
	if(a->t_Build_Time < b->t_Build_Time) return -1;	// b : new

	return 0;
}

int m_snap_inst_name_cmp(const void *m1,const void *m2)
{
	char *a, *b;
	int sa, sb;
	time_t ta, tb;

	a = (char*) m1;
	b = (char*) m2;

	sa = strlen(a);
	sb = strlen(b);
	if (sa == 0 && sb == 0) return 0;
	if (sa == 0) return 1;
	if (sb == 0) return -1;

	ta = get_build_time_from_instance_name(a);
	tb = get_build_time_from_instance_name(b);
	if (ta > tb) return 1;
	if (ta < tb) return -1;

	return 0;
}

void sort_snapshot_volume_data(SNAPSHOT_VOLUME_CONF *vol_data_list,int list_cnt)
{
	qsort((void*)vol_data_list,list_cnt,sizeof(SNAPSHOT_VOLUME_CONF),m_sn_cmp);	
}

void sort_snapshot_volume_info_data(SNAPSHOT_VOLUME_INFO *vol_data_list,int list_cnt)
{
	qsort((void*)vol_data_list,list_cnt,sizeof(SNAPSHOT_VOLUME_INFO),m_sn_info_cmp);	
}

void sort_lvm_volume_data(LVM_VOLUME_CONF *vol_data_list,int list_cnt)
{
	qsort((void*)vol_data_list,list_cnt,sizeof(LVM_VOLUME_CONF),m_lvm_cmp);	
}

void sort_snapshot_volume_data_for_show(SNAPSHOT_VOLUME_CONF *vol_data_list,int list_cnt)
{
	qsort((void*)vol_data_list,list_cnt,sizeof(SNAPSHOT_VOLUME_CONF),m_sn_cmp_for_show);	
}

void sort_lvm_volume_data_for_show(LVM_VOLUME_INFO *vol_data_list,int list_cnt)
{
	qsort((void*)vol_data_list,list_cnt,sizeof(LVM_VOLUME_INFO),m_lvm_cmp_for_show);	
}

void sort_snapshot_volume_info_for_show(SNAPSHOT_VOLUME_INFO *vol_data_list,int list_cnt)
{
	qsort((void*)vol_data_list,list_cnt,sizeof(SNAPSHOT_VOLUME_INFO),m_sn_info_cmp_for_show);	
}

void sort_snapd_conf_records(SNAPD_CONF_ENTRY *snapd_conf_list, int list_cnt)
{
	qsort((void*)snapd_conf_list,list_cnt,sizeof(SNAPD_CONF_ENTRY),m_snapd_cmp);	
}

void sort_instances_in_a_snap_volume(SNAPSHOT_VOLUME_INSTANCE_INFO* inst_list, int list_cnt)
{
	qsort((void*)inst_list,list_cnt,sizeof(SNAPSHOT_VOLUME_INSTANCE_INFO),m_snap_inst_cmp);	
}

void sort_instance_names_in_a_snap_volume(char (*inst_list)[SHARE_NAME_LENGTH], int list_cnt)
{
	qsort((void*)inst_list,list_cnt,sizeof(char) * (SHARE_NAME_LENGTH), m_snap_inst_name_cmp);	
}

BOOL is_digit_string(char* str)
{
	char *p=str;
	int cnt=0;

	if (p==NULL || *p=='\0') return FALSE;

	while (p && *p) {
		if (!isdigit((int)(*p++)))
			return FALSE;
		cnt++;
	}
	if (cnt) return TRUE;

	return FALSE;
}

BOOL is_valid_snapshot_name(char* snapname)
{
	char *ptr, tmpname[LV_PATH_MAX_LEN];
	int strlength;

	if ((ptr = strrchr(snapname, '/')) == NULL)
		strcpy(tmpname, snapname);
	else
		strcpy(tmpname, ptr+1);

	Remove_Blank_From_String(tmpname);
	strlength = strlen(tmpname);
	if (strlength<=0 || strlength>=SHARE_NAME_LENGTH
		|| strncmp(tmpname, SNAPSHOT_NAME_PREFIX, strlen(SNAPSHOT_NAME_PREFIX)))
		return FALSE;
	return TRUE;
}

#ifdef	STORAGE_FILE_LOCK
int repair_lvm_storage_conf_if_necessary()
{
	FILE *fp = NULL;
	int ret = SUCCESS;

#ifdef	DEBUG
printf("repair_lvm_storage_conf_if_necessary starts!\n");
#endif
	if (!NAS_File_Lock(STORAGE_CONF, 'r')) {
#ifdef	DEBUG
		printf("\tNAS_File_Lock(%s) fails!\n", STORAGE_CONF);
#endif
		return ERROR_LOCK_FILE;
	}

	if((fp = fopen(STORAGE_CONF,"r")) == NULL) {
#ifdef	DEBUG
		printf("\t%s not exists!\n", STORAGE_CONF);
#endif
		NAS_File_Unlock(STORAGE_CONF);
		ret = Create_LVM_Storage_Conf();
	}
	else {
		char tmp_line[80];
		strcpy(tmp_line,"");
		fgets(tmp_line,80,fp);
		fclose(fp);
		NAS_File_Unlock(STORAGE_CONF);
		if(strlen(tmp_line)== 0) {
#ifdef	DEBUG
			printf("\t%s is imcomplete!\n", STORAGE_CONF);
#endif
			ret = Restore_LVM_Storage_Conf();
		}
	}
#ifdef	DEBUG
printf("repair_lvm_storage_conf_if_necessary finishes!\n");
#endif
	return ret;
}
#endif

int do_add_snapshot_record(SNAPSHOT_VOLUME_CONF *sn_vol_conf)
{
	int vol_cnt=0,i=0;
	SNAPSHOT_VOLUME_CONF *vol_data_list=NULL;
	FILE *fp;
#ifndef	STORAGE_FILE_LOCK
	int fd=-1;
#endif
	int max_str_len = (MAX_SNAP_NO_PER_VOLUME*(SHARE_NAME_LENGTH + 2) > BUF_SIZE) ? 
		MAX_SNAP_NO_PER_VOLUME*(SHARE_NAME_LENGTH + 2) : BUF_SIZE;
	char buf[2*max_str_len], tmpbuf[max_str_len];
	void *ptr=NULL;

	if (!is_valid_snapshot_name(
		sn_vol_conf->c_Snapshot_Path)) {
#ifdef DEBUG
		printf("do_add_snapshot_record : !is_valid_snapshot_name(%s)\n",
			sn_vol_conf->c_Snapshot_Path);
#endif
		return ERROR_INVALID_PARAM;
	}

	if (sn_vol_conf->b_Is_Scheduled_Snap) {
		if (sn_vol_conf->Cycle < Hour_1 || sn_vol_conf->Cycle > Week_2 ||
			sn_vol_conf->i_Max_Instance_Size < 0 ||
			sn_vol_conf->i_Max_Instance_Size > MAX_SNAP_INSTANCE)
			return ERROR_INVALID_PARAM;
	}

	vol_cnt = Get_All_Snapshot_Volumes_Conf(&vol_data_list);
	
	if(vol_cnt >=MAX_SNAP_NO) {
		free(vol_data_list);
		return ERROR_OUT_OF_RANGE;
	}

	for (i=0; i<vol_cnt; i++) {
		if (strcmp(vol_data_list[i].c_Snapshot_Path, sn_vol_conf->c_Snapshot_Path)==0) {
			free(vol_data_list);
#ifdef DEBUG
			printf("do_add_snapshot_record : %s already exist!\n",
				vol_data_list[i].c_Snapshot_Path);
#endif
			return ERROR_ALREADY_EXISTED;
		}
	}

	if(vol_cnt >=0) {
		// get the minimum snapshot volume number
		{
			int j,vol_no,flag;
			
			for (i=1; i<MAX_SNAP_NO; i++) {
				flag=0;
				for (j=0; j<vol_cnt; j++) {
					if (i==vol_data_list[j].i_Snapshot_No) {
						flag=1;
						break;
					}
				}
				if (flag==0) {
					vol_no=i;
					break;
				}
			}
			sn_vol_conf->i_Snapshot_No = vol_no;
		}
		memset(sn_vol_conf->Restore_Status, 0, MAX_SNAP_INSTANCE*sizeof(SNAP_RESTORE_STATUS));

		ptr = realloc((void*)vol_data_list,
			(vol_cnt+1)*sizeof(SNAPSHOT_VOLUME_CONF));
		if (ptr==NULL) {
			if (vol_data_list)
				free(vol_data_list);
			return ERROR_OUT_OF_MEMORY;
		}
		vol_data_list = (SNAPSHOT_VOLUME_CONF *)ptr;

		memcpy((void*)(&vol_data_list[vol_cnt]),
			(void*)sn_vol_conf,
			sizeof(SNAPSHOT_VOLUME_CONF));

		vol_cnt++;
		sort_snapshot_volume_data(vol_data_list,vol_cnt);

#ifdef	STORAGE_FILE_LOCK
		if (NAS_File_Lock(SNAPSHOT_CONF, 'w') == 0) {
			free(vol_data_list);
			return ERROR_LOCK_FILE;
		}
		if((fp = fopen(SNAPSHOT_CONF,"w+")) == NULL) {
			free(vol_data_list);
			NAS_File_Unlock(SNAPSHOT_CONF);
			return ERROR_OPEN_FILE;
		}
#else
		if((fp = fopen(SNAPSHOT_CONF,"w+")) != NULL)
			fclose(fp);
		else {
			free(vol_data_list);
			return ERROR_FAIL;
		}
		
		// open /etc/config/storage.conf for write
		if ((fd=open(SNAPSHOT_CONF, O_CREAT | O_RDWR))<0) {
			free(vol_data_list);
			return ERROR_OPEN_FILE;
		}
		// exclusively lock file
		if (flock(fd, LOCK_EX)<0) {
			close(fd);
			free(vol_data_list);
			return ERROR_LOCK_FILE;
		}
#endif
		// write the new content into the config file
		for(i=0;i<vol_cnt;i++){
			memset(buf,0,BUF_SIZE*2);
			// volume header
			sprintf(tmpbuf, "\n[SNAPSHOT %d]\n",
				vol_data_list[i].i_Snapshot_No);
			strcpy(buf, tmpbuf);
	
			// device name
			sprintf(tmpbuf, "device name = %s\n",
				vol_data_list[i].c_Snapshot_Path);
			strcat(buf, tmpbuf);
	
			// source volume
			sprintf(tmpbuf, "source volume = %s\n",
				vol_data_list[i].c_Src_LV);
			strcat(buf, tmpbuf);

			// time
			sprintf(tmpbuf, "build time = %u\n",
				(unsigned int)vol_data_list[i].t_Build_Time);
			strcat(buf, tmpbuf);

			// priority
			sprintf(tmpbuf, "priority = %u\n",
				(unsigned int)vol_data_list[i].Priority);
			strcat(buf, tmpbuf);

			// status
			sprintf(tmpbuf, "status = %d\n",
				vol_data_list[i].i_Status);
			strcat(buf, tmpbuf);

			// is scheduled?
			sprintf(tmpbuf, "is scheduled = %s\n",
				(vol_data_list[i].b_Is_Scheduled_Snap) ? "True" : "False");
			strcat(buf, tmpbuf);

			if (vol_data_list[i].b_Is_Scheduled_Snap) {
				char tmpbuf2[max_str_len];
				int j=0;

				// start time
				sprintf(tmpbuf, "start time = %u\n",
					(unsigned int)vol_data_list[i].t_Start_Time);
				strcat(buf, tmpbuf);

				// priority
				sprintf(tmpbuf, "priority = %d\n", (int)vol_data_list[i].Priority);
				strcat(buf, tmpbuf);

				// cycle
				sprintf(tmpbuf, "cycle = %d\n",	(int)vol_data_list[i].Cycle);
				strcat(buf, tmpbuf);

				// max instance count
				sprintf(tmpbuf, "max instance size = %d\n",
					vol_data_list[i].i_Max_Instance_Size);
				strcat(buf, tmpbuf);

				// snap instances && snap restore status
				strcpy(tmpbuf, "snap instances = ");
				strcpy(tmpbuf2, "snap restore status = ");
				while (j<vol_data_list[i].i_Max_Instance_Size &&
					strlen(vol_data_list[i].c_Instances[j])>0)
				{
					sprintf(tmpbuf, "%s%s,", tmpbuf, vol_data_list[i].c_Instances[j]);
					sprintf(tmpbuf2, "%s%d,", tmpbuf2, vol_data_list[i].Restore_Status[j]);
					j++;
				}
				j = strlen(tmpbuf);
				tmpbuf[j] = '\n';
				tmpbuf[j+1] = '\0';
				strcat(buf, tmpbuf);

				j = strlen(tmpbuf2);
				tmpbuf2[j] = '\n';
				tmpbuf2[j+1] = '\0';
				strcat(buf, tmpbuf2);
			}
			else {
				// snap restore status
				sprintf(tmpbuf, "snap restore status = %d\n",
					vol_data_list[i].Restore_Status[0]);
				strcat(buf, tmpbuf);
			}
#ifdef	STORAGE_FILE_LOCK
			if (fputs(buf, fp)<0) {
				fclose(fp);
				free(vol_data_list);
				NAS_File_Unlock(SNAPSHOT_CONF);
				return ERROR_WRITE_FILE;
			}
#else
			if (write(fd, buf, strlen(buf))<0) {
				flock(fd, LOCK_UN);
				close(fd);
				free(vol_data_list);
				return ERROR_WRITE_FILE;
			}
#endif
		}
		// unlock the file & close it
#ifdef	STORAGE_FILE_LOCK
		fclose(fp);
		NAS_File_Unlock(SNAPSHOT_CONF);
#else
		flock(fd, LOCK_UN);
		close(fd);
#endif
		free(vol_data_list);
	}
	return vol_cnt;
}

time_t get_build_time_by_snap_instance_name(char* snap_name, char* snap_inst_name)
{
	char timestr[TIMESTAMP_STRING_LENGTH+1], tmp[5];
	struct tm mytime;
	int len1, len2;

	len1 = strlen(snap_name);
	len2 = strlen(snap_inst_name);

	if (len2-len1 != TIMESTAMP_STRING_LENGTH) return ERROR_INVALID_PARAM;

	strncpy(timestr, snap_inst_name+len1, TIMESTAMP_STRING_LENGTH);
	timestr[TIMESTAMP_STRING_LENGTH] = '\0';

	if ((!strncmp(snap_name, snap_inst_name, len1)) &&
		(is_digit_string(timestr)))
	{
		timestr[TIMESTAMP_STRING_LENGTH] = '\0';

		strncpy(tmp, timestr, 4);
		tmp[4] = '\0';
		mytime.tm_year = atoi(tmp) - 1900;

		strncpy(tmp, timestr+4, 2);
		tmp[2] = '\0';
		mytime.tm_mon = atoi(tmp) - 1;

		strncpy(tmp, timestr+6, 2);
		tmp[2] = '\0';
		mytime.tm_mday = atoi(tmp);

		strncpy(tmp, timestr+8, 2);
		tmp[2] = '\0';
		mytime.tm_hour = atoi(tmp);

		strncpy(tmp, timestr+10, 2);
		tmp[2] = '\0';
		mytime.tm_min = atoi(tmp);

		mytime.tm_sec = 0;

		return mktime(&mytime);
	}
	else
		return ERROR_INVALID_PARAM;
}

time_t get_build_time_from_instance_name(char* snap_inst_name)
{
	char timestr[TIMESTAMP_STRING_LENGTH+1], tmp[5];
	struct tm mytime;
	int len;

	len = strlen(snap_inst_name);

	if (len <= strlen(SNAPSHOT_NAME_PREFIX) + TIMESTAMP_STRING_LENGTH)
		return ERROR_INVALID_PARAM;

	strncpy(timestr, snap_inst_name + len - TIMESTAMP_STRING_LENGTH,
		TIMESTAMP_STRING_LENGTH);
	timestr[TIMESTAMP_STRING_LENGTH] = '\0';

	if (is_digit_string(timestr))
	{
		timestr[TIMESTAMP_STRING_LENGTH] = '\0';

		strncpy(tmp, timestr, 4);
		tmp[4] = '\0';
		mytime.tm_year = atoi(tmp) - 1900;

		strncpy(tmp, timestr+4, 2);
		tmp[2] = '\0';
		mytime.tm_mon = atoi(tmp) - 1;

		strncpy(tmp, timestr+6, 2);
		tmp[2] = '\0';
		mytime.tm_mday = atoi(tmp);

		strncpy(tmp, timestr+8, 2);
		tmp[2] = '\0';
		mytime.tm_hour = atoi(tmp);

		strncpy(tmp, timestr+10, 2);
		tmp[2] = '\0';
		mytime.tm_min = atoi(tmp);

		mytime.tm_sec = 0;

		return mktime(&mytime);
	}
	else
		return ERROR_INVALID_PARAM;
}


int get_restore_progress(char *snaplv)
{
	char* p;
	char buf[BUF_SIZE], tmp[BUF_SIZE];
	FILE *fp;
	int result = -1;

	if ((p = strrchr(snaplv, '/'))==NULL) p=snaplv;
	else p++;

	// get snapshot status file (/proc/lvm/...)
	Get_VG_Name_By_LV(snaplv, tmp, sizeof(tmp));
	sprintf(buf, LV_STATE_FILE_FORMAT, tmp, p);

	if ((fp = fopen(buf, "r")) == NULL)
		return ERROR_OPEN_FILE;

	while(fgets(buf, BUF_SIZE, fp)) {
		Get_String_Field(buf, 1, ':', tmp, BUF_SIZE);
		if (strcmp(tmp, "restore"))
			continue;

		Get_String_Field(buf, 2, ':', tmp, BUF_SIZE);
		p = strchr(tmp, '%');
		if (p==NULL) break;

		strncpy(buf, tmp, (int)(p-tmp));
		buf[(int)(p-tmp)] = '\0';
		result = atoi(buf);
		break;
	}

	fclose(fp);
	return result;
}

// Cat 2002/12/01

int get_snap_instance_name_by_source_lv(char* srclv, char* instance_short_name,
	char* outbuf, int bufsize)
{
	char vgname[LV_PATH_MAX_LEN], snpath[LV_PATH_MAX_LEN];

	if (srclv==NULL || instance_short_name==NULL || outbuf==NULL || bufsize<=0)
		return ERROR_INVALID_PARAM;

	// get VG name first
	Get_VG_Name_By_LV(srclv, vgname, sizeof(vgname));

	// attach the snap instance's short name to the VG name --> snap path under '/dev/'
	sprintf(snpath, "%s/%s",
		vgname, instance_short_name);

	if (strlen(snpath)>=bufsize)
		return ERROR_BUFFER_TOO_SMALL;
	strcpy(outbuf, snpath);
	return SUCCESS;
}

// Cat : for new snap repository size issue ==>
//------------------------------------------------------------------------
// cmp_fn_asc_1
// description : the function to compare 2 snapshot instances ascedantly by urgency 
//		mainly used for giving free space for new comers
//------------------------------------------------------------------------
int cmp_fn_asc_1(const void *m1, const void *m2)
{
	SNAPSHOT_BASIC_INFO *a,*b;
	
	a = (SNAPSHOT_BASIC_INFO*) m1;
	b = (SNAPSHOT_BASIC_INFO*) m2;

	// criteria 1 : priority lower first
	if (a->Priority > b->Priority) return 1;
	if (a->Priority < b->Priority) return -1;

	// criteria 2 : older first
	if (a->t_Build_Time > b->t_Build_Time) return 1;
	if (a->t_Build_Time < b->t_Build_Time) return -1;

	// criteria 3 : free space higher first
	if (a->ll_Free_Size > b->ll_Free_Size) return -1;
	if (a->ll_Free_Size < b->ll_Free_Size) return 1;
	
	return 0;
}

//------------------------------------------------------------------------
// cmp_fn_asc_2
// description : the function to compare 2 snapshot instances ascedantly by urgency 
//		mainly used for killing snapshots
//------------------------------------------------------------------------
int cmp_fn_asc_2(const void *m1, const void *m2)
{
	SNAPSHOT_BASIC_INFO *a,*b;
	
	a = (SNAPSHOT_BASIC_INFO*) m1;
	b = (SNAPSHOT_BASIC_INFO*) m2;

	// criteria 1 : priority lower first
	if (a->Priority > b->Priority) return 1;
	if (a->Priority < b->Priority) return -1;

	// criteria 2 : older first
	if (a->t_Build_Time > b->t_Build_Time) return 1;
	if (a->t_Build_Time < b->t_Build_Time) return -1;
#ifndef __EXT3_SNAP__
	// criteria 3 : free space lower first
	if (a->ll_Free_Size > b->ll_Free_Size) return 1;
	if (a->ll_Free_Size < b->ll_Free_Size) return -1;
#endif
	return 0;
}

//------------------------------------------------------------------------
// cmp_fn_desc_1
// description : the function to compare 2 snapshot instances descendantly by urgency 
//		mainly used for asking for more space
//------------------------------------------------------------------------
int cmp_fn_desc_1(const void *m1, const void *m2)
{
	SNAPSHOT_BASIC_INFO *a,*b;
	
	a = (SNAPSHOT_BASIC_INFO*) m1;
	b = (SNAPSHOT_BASIC_INFO*) m2;

	// criteria 1 : priority higher first
	if (a->Priority > b->Priority) return -1;
	if (a->Priority < b->Priority) return 1;

	// criteria 2 : younger first
	if (a->t_Build_Time > b->t_Build_Time) return -1;
	if (a->t_Build_Time < b->t_Build_Time) return 1;

	// criteria 3 : free space lower first
	if (a->ll_Free_Size > b->ll_Free_Size) return 1;
	if (a->ll_Free_Size < b->ll_Free_Size) return -1;
	
	return 0;
}

//------------------------------------------------------------------------
// sort_snap_basic_info_list_by_order
// description : get the list in a specific order
//------------------------------------------------------------------------
void sort_snap_basic_info_list_by_order(SNAPSHOT_BASIC_INFO* list, int cnt, uint order_flag)
{
	if (order_flag & FLAG_IMPORTANCE_ASC) {
#ifndef __EXT3_SNAP__
		if (order_flag & FLAG_SPACE_USAGE_ASC)
			qsort((void*)list, cnt, sizeof(SNAPSHOT_BASIC_INFO), cmp_fn_asc_1);
		else
#endif
			qsort((void*)list, cnt, sizeof(SNAPSHOT_BASIC_INFO), cmp_fn_asc_2);
	}
	else
		qsort((void*)list, cnt, sizeof(SNAPSHOT_BASIC_INFO), cmp_fn_desc_1);
}
// <== Cat : for new snap repository size issue

#ifdef __EXT3_SNAP__
int get_possible_max_snap_count(char* lvname, long long snap_rep_size)
{
	int result = MAX_SNAP_NO_PER_VOLUME;
	int snapcnt;
	SNAPSHOT_VOLUME_CONF *snaplist = NULL;

	snapcnt = Get_Snapshot_Volumes_Conf_On_One_LV(lvname, &snaplist);
	if (snapcnt>0) {
		long long min=snap_rep_size, max=0, size;
		int i, j;
		char instname[LV_PATH_MAX_LEN];

		for (i=0; i<snapcnt; i++) {
			if (snaplist[i].b_Is_Scheduled_Snap) {
				for (j=0; j<snaplist[i].i_Max_Instance_Size; j++)
				{
					if (strlen(snaplist[i].c_Instances[j])==0) continue;

					get_snap_instance_name_by_source_lv(
						lvname, snaplist[i].c_Instances[j],
						instname, sizeof(instname));
					size = get_lv_total_size(instname);
#ifdef DEBUG
					printf("get_possible_max_snap_count(%s) : "
						"snap %s size = %lld KB\n",
						lvname, instname, size);
#endif
					if (size>0 && size<min)
						min = size;
					if (size>max)
						max = size;
				}
			}
			else {
				size = get_lv_total_size(snaplist[i].c_Snapshot_Path);
#ifdef DEBUG
				printf("get_possible_max_snap_count(%s) : "
					"snap %s size = %lld KB\n",
					lvname, snaplist[i].c_Snapshot_Path, size);
#endif
				if (size>0 && size<min)
					min = size;
				if (size>max)
					max = size;
			}
		}
#ifdef DEBUG
		printf("get_possible_max_snap_count (%s) : min = %lld , max = %lld\n",
			lvname, min, max);
#endif
		if (min>0 && max>0) {
			long long avgnum = (snap_rep_size * 2) / (min+max);
			long long remainder = (snap_rep_size * 2) % (min+max);

			if ( ( 2 * remainder ) < ( min + max ))
				result = avgnum;
			else
				result = avgnum + 1;
		}
	}
#ifdef DEBUG
	else {
		printf("get_possible_max_snap_count (%s) : no snapshots found!\n", lvname);
	}
	printf("get_possible_max_snap_count (%s) : output = %d\n", lvname, result);
#endif

	return result;
}
#endif

int get_all_subitems(char *dir, char *items, int item_max_cnt, int item_size)
{
	DIR *dp;
	struct dirent *entry;
	struct stat statbuf;
	int cnt=0;

	if((dp = opendir(dir)) == NULL) {
#ifdef DEBUG
		printf("get_all_subitems : cannot open directory: %s\n", dir);
#endif
		return ERROR_NOT_FOUND;
	}

	while((entry = readdir(dp)) != NULL) {
		lstat(entry->d_name, &statbuf);

		if(strcmp(".",entry->d_name)==0) continue;

		if(strcmp("..",entry->d_name) == 0)
			continue;

		if (cnt >= item_max_cnt) {
			cnt = ERROR_BUFFER_TOO_SMALL;
			goto get_all_subitems_out;
		}

		strncpy((items+cnt*item_size), entry->d_name, item_size);
		*(items+(cnt+1)*item_size-1) = '\0';
		cnt++;
	}
get_all_subitems_out:
	closedir(dp);
	return cnt;
}

// get all existent snapshot names from driver status files	
int get_all_existent_snapshots(SNAPSHOT_VOLUME_CONF **slist)
{
	int sncnt = 0, vgcnt, i;
	char vgs[MAX_VOLUME_NO][LV_PATH_MAX_LEN];
	char snaps[2*MAX_SNAP_NO_PER_VOLUME][LV_PATH_MAX_LEN];
	SNAPSHOT_VOLUME_CONF *snaplist = NULL;
	void *tmp;
#ifdef DEBUG
	int ret;
#endif

	vgcnt = get_all_subitems(LVM_STATE_PATH, (char*)vgs, MAX_VOLUME_NO, LV_PATH_MAX_LEN);
	if (vgcnt<=0) {
#ifdef DEBUG
		printf("get_all_existent_snapshots : "
			"get_all_subitems returns %d\n",
			vgcnt);
#endif
		sncnt = vgcnt;
		goto get_all_existent_snapshots_out;
	}
#ifdef DEBUG
	printf("get_all_existent_snapshots : "
		"get_all_subitems returns %d\n",
		vgcnt);
#endif
	for (i=0; i<vgcnt; i++) {
		char path[BUF_SIZE];
		int scnt, j;

		sprintf(path, "%s%s/LVs/", LVM_STATE_PATH, vgs[i]);
		scnt = get_all_subitems(path, (char*) snaps, 2*MAX_SNAP_NO_PER_VOLUME, LV_PATH_MAX_LEN);
		if (scnt<0) {
#ifdef DEBUG
			printf("get_all_existent_snapshots : "
				"get_all_subitems returns %d\n",
				vgcnt);
#endif
			continue;
		}
		for (j=0; j<scnt; j++) {
#ifndef DEBUG
			if (strncmp(snaps[j], SNAPSHOT_NAME_PREFIX, strlen(SNAPSHOT_NAME_PREFIX))) {
#else
			if ((ret = strncmp(snaps[j], SNAPSHOT_NAME_PREFIX, strlen(SNAPSHOT_NAME_PREFIX)))!=0) {
				printf("\t\t%s, %s, %d, %d is not a snapshot name!\n",
					snaps[j], SNAPSHOT_NAME_PREFIX, strlen(SNAPSHOT_NAME_PREFIX), ret);
#endif
				continue;
			}
			tmp = realloc(snaplist, (sncnt+1)* sizeof(SNAPSHOT_VOLUME_CONF));
			if (tmp==NULL) {
#ifdef DEBUG
				printf("get_all_existent_snapshots : "
					"fails to realloc memory!\n");
#endif
				if (sncnt>0) free(snaplist);
				snaplist = NULL;
				sncnt = ERROR_OUT_OF_MEMORY;
				goto get_all_existent_snapshots_out;
			}
			snaplist = (SNAPSHOT_VOLUME_CONF *) tmp;

			memset(&(snaplist[sncnt]), 0, sizeof(SNAPSHOT_VOLUME_CONF));
			sprintf(snaplist[sncnt].c_Snapshot_Path,
				"/dev/%s/%s", vgs[i], snaps[j]);
			Get_LV_Name_By_VG(vgs[i], snaplist[sncnt].c_Src_LV, 
					sizeof(snaplist[sncnt].c_Src_LV));
			sncnt++;
#ifdef DEBUG
			printf("get_all_existent_snapshots : "
				"%d, %s, %s\n",
				sncnt, snaplist[sncnt-1].c_Snapshot_Path,
				snaplist[sncnt].c_Src_LV);
#endif

		}
	}
get_all_existent_snapshots_out:
	*slist = snaplist;
	return sncnt;
}
// ------------- functions to export ---------------------------------

// Cat : for new snap repository size issue ==>
/**********************************************************************
** Get_Snapshots_On_One_LV_In_Order
** Description : get the snapshot list in a specific order
**		ex. FLAG_IMPORTANCE_ASC -- the ones 
**					(1) w/ lower priority
**					(2) older
**					first
**		FLAG_SPACE_USAGE_ASC -- the ones w/ more free size first
** Input :
**	srclv --	the source LV path
**	order_flag --	the flag that indicates the order
**			possible masks : FLAG_IMPORTANCE_ASC, FLAG_SPACE_USAGE_ASC
**	outlistptr --	the pointer to the output snapshot list,
**			initially must points to NULL
** Output : the size of the output list, or error code
**
***********************************************************************/
int Get_Snapshots_On_One_LV_In_Order(char* srclv, uint order_flag,
	SNAPSHOT_BASIC_INFO **outlistptr)
{
	int snvolcnt, sninstcnt=0, i, j=0;
	SNAPSHOT_VOLUME_CONF *snaplist = NULL;
	SNAPSHOT_BASIC_INFO *outlist = NULL, *p;
	LV_INFO lvinfo;

	if (srclv==NULL || outlistptr==NULL)
		return ERROR_INVALID_PARAM;

	snvolcnt = Get_Snapshot_Volumes_Conf_On_One_LV(srclv, &snaplist);
	if (snvolcnt<0) {
#ifdef DEBUG
		printf("Get_Snapshots_On_One_LV_In_Order(%s, %x) : "
			"Get_Snapshot_Volumes_Conf_On_One_LV fails, err=%d\n",
			srclv, order_flag, snvolcnt);
#endif
		return snvolcnt;
	}

	if (snvolcnt>0) {
		for (i=0; i<snvolcnt; i++) {
			if (snaplist[i].b_Is_Scheduled_Snap) {
				char instname[LV_PATH_MAX_LEN];

				j=0;
				while (j<snaplist[i].i_Max_Instance_Size &&
					strlen(snaplist[i].c_Instances[j])>0)
				{
					p = (SNAPSHOT_BASIC_INFO *)realloc(outlist,
						(sninstcnt+1) * sizeof(SNAPSHOT_BASIC_INFO));
					if (p==NULL) {
						sninstcnt = ERROR_OUT_OF_MEMORY;
						goto get_snapshots_on_one_lv_in_order_fail;
					}
					outlist = p;

					outlist[sninstcnt].i_Snapshot_No = snaplist[i].i_Snapshot_No;
					get_snap_instance_name_by_source_lv(
						snaplist[i].c_Src_LV,
						snaplist[i].c_Instances[j],
						instname, sizeof(instname));
					outlist[sninstcnt].t_Build_Time =
						get_build_time_by_snap_instance_name(
							snaplist[i].c_Snapshot_Path,
							instname);
					outlist[sninstcnt].Restore_Status = snaplist[i].Restore_Status[j];
					outlist[sninstcnt].Priority = snaplist[i].Priority;
					get_snap_instance_name_by_source_lv(
						snaplist[i].c_Src_LV,
						snaplist[i].c_Instances[j],
						outlist[sninstcnt].c_Path,
						sizeof(outlist[sninstcnt].c_Path));
					strcpy(outlist[sninstcnt].c_Src_LV, snaplist[i].c_Src_LV);

					if (get_lv_info(outlist[sninstcnt].c_Path, &lvinfo)==SUCCESS) {
						outlist[sninstcnt].ll_Snap_Size = lvinfo.ll_LV_Size;
						outlist[sninstcnt].ll_Free_Size = lvinfo.ll_SnapLV_Free_Size;
					}
					else {
						outlist[sninstcnt].ll_Snap_Size = 0;
						outlist[sninstcnt].ll_Free_Size = 0;
					}

					sninstcnt++;
					j++;
				}
			}
			else {
				p = (SNAPSHOT_BASIC_INFO *)realloc(outlist,
					(sninstcnt+1) * sizeof(SNAPSHOT_BASIC_INFO));
				if (p==NULL) {
					sninstcnt = ERROR_OUT_OF_MEMORY;
					goto get_snapshots_on_one_lv_in_order_fail;
				}
				outlist = p;

				outlist[sninstcnt].i_Snapshot_No = snaplist[i].i_Snapshot_No;
				outlist[sninstcnt].t_Build_Time = snaplist[i].t_Build_Time;
				outlist[sninstcnt].Restore_Status = snaplist[i].Restore_Status[0];
				outlist[sninstcnt].Priority = snaplist[i].Priority;
				strcpy(outlist[sninstcnt].c_Path, snaplist[i].c_Snapshot_Path);
				strcpy(outlist[sninstcnt].c_Src_LV, snaplist[i].c_Src_LV);

				if (get_lv_info(outlist[sninstcnt].c_Path, &lvinfo)==SUCCESS) {
					outlist[sninstcnt].ll_Snap_Size = lvinfo.ll_LV_Size;
					outlist[sninstcnt].ll_Free_Size = lvinfo.ll_SnapLV_Free_Size;
				}
				else {
					outlist[sninstcnt].ll_Snap_Size = 0;
					outlist[sninstcnt].ll_Free_Size = 0;
				}
				sninstcnt++;
			}
		}

		free(snaplist);
	}

	sort_snap_basic_info_list_by_order(outlist, sninstcnt, order_flag);

get_snapshots_on_one_lv_in_order_out:
	*outlistptr = outlist;
	return sninstcnt;

get_snapshots_on_one_lv_in_order_fail:
	free(outlist);
	outlist = NULL;
	goto get_snapshots_on_one_lv_in_order_out;
}
// <== Cat : for new snap repository size issue

// 2002/12/01
/**********************************************************************
** Get_VG_Name_By_LV
** Description:	get the VG name for a LV
** Input:
**		lvname :	the LV (not snapshot) of the volume group
**		vg_name :	the output volume group name
**		vg_buf_size :	the output name buffer length (including '\0')
** Output:	error code --
**		SUCCESS
**		ERROR_INVALID_PARAM
**		ERROR_BUFFER_TOO_SMALL
***********************************************************************/
int Get_VG_Name_By_LV(char* lvname, char* vgname, int buf_size)
{
	char *ptr;
	char buf[BUF_SIZE];

	if (lvname == NULL || vgname == NULL)
		return ERROR_INVALID_PARAM;

	if ((ptr = strrchr(lvname, '/')) != NULL) {
		strncpy(buf, lvname, (int)(ptr-lvname));
		buf[(int)(ptr-lvname)] = '\0';
	}
	else if ((ptr = strchr(lvname, '-')) != NULL)
		sprintf(buf, VG_PATH_FORMAT, ptr+1);
	else
		return ERROR_INVALID_PARAM;

	if (strlen(buf)>=buf_size)
		return ERROR_BUFFER_TOO_SMALL;

	strcpy(vgname, buf);
	return SUCCESS;
}

/**********************************************************************
** Get_VG_Name_By_Device
** Description:	get the VG name for a device
** Input:
**		pv_path :	the device of the volume group
**		vg_name :	the output volume group name
**		vg_buf_size :	the output name buffer length (including '\0')
** Output:	error code --
**		SUCCESS
**		ERROR_INVALID_PARAM
**		ERROR_BUFFER_TOO_SMALL
***********************************************************************/
int Get_VG_Name_By_Device(char* pv_path, char* vg_name, int vg_buf_size)
{
	char* ptr;
	char pvname[LV_PATH_MAX_LEN], tmp[BUF_SIZE], buf[BUF_SIZE];

	if (pv_path==NULL || vg_name==NULL)
		return ERROR_INVALID_PARAM;

	convert_md_name(pv_path, pvname, sizeof(pvname), 1);
	if (strncasecmp(pvname, MD_DEVICE_PREFIX, strlen(MD_DEVICE_PREFIX))==0) {
		// PV is a RAID device
		ptr = strrchr(pvname, '/');
		if (ptr==NULL)
			strcpy(tmp, pvname);
		else
			strcpy(tmp, ptr+1);
		String_To_Upper_Case(tmp);
		sprintf(buf, VG_PATH_FORMAT, tmp);
	}
	else {
		int drive_no;

		strcpy(tmp, pvname);
		// input device is a single disk device
		drive_no = Get_Drive_No(tmp);
		get_drive_alphabetic_order(drive_no, 'A', tmp, sizeof(tmp));
		sprintf(buf, VG_PATH_FORMAT, tmp);
	}

	if (strlen(buf) >= vg_buf_size)
		return ERROR_BUFFER_TOO_SMALL;

	strcpy(vg_name, buf);
	return SUCCESS;
}

/**********************************************************************
** Get_LV_Name_By_VG
** Description:	get the LV name for a VG
** Input:
**		vgname :	the the volume group name
**		lvname :	the output LV name
**		buf_size :	the output name buffer length (including '\0')
** Output:	error code --
**		SUCCESS
**		ERROR_INVALID_PARAM
**		ERROR_BUFFER_TOO_SMALL
***********************************************************************/
int Get_LV_Name_By_VG(char* vgname, char* lvname, int buf_size)
{
	char *ptr, *ptr2;
	char buf[BUF_SIZE];

	if (lvname == NULL || vgname == NULL || buf_size <=0)
		return ERROR_INVALID_PARAM;

	if ((ptr = strrchr(vgname, '/')) == NULL)
		ptr = vgname;
	else
		ptr++;

	if ((ptr2 = strchr(ptr, '-')) != NULL)
		sprintf(buf, LV_PATH_FORMAT, ptr2+1, ptr2+1);
	else
		return ERROR_INVALID_PARAM;

	if (strlen(buf)>=buf_size)
		return ERROR_BUFFER_TOO_SMALL;

	strcpy(lvname, buf);
	return SUCCESS;
}

/**********************************************************************
** Get_LV_Name_For_Device
** Description:	get the correspondent logical volume name for the device
** Input:
**		device_name :	the device of the logical volume
**		lv_name :	the output logical volume name
**		buf_len :	the output name buffer length (including '\0')
** Output:	error code --
**		SUCCESS
**		ERROR_BUFFER_TOO_SMALL
***********************************************************************/
int Get_LV_Name_For_Device(char* device_name, char* lv_name, int buf_len)
{
	char buf[BUF_SIZE];
	char tmp[HD_DEVICE_NAME_LENGTH];
	char *ptr=NULL;
	int drive_no;

	// device_name is a RAID device name
	if (Is_RaidDev_Exist(device_name)) {
		ptr = strrchr(device_name, '/');
		if (ptr==NULL)
			strcpy(tmp, device_name);
		else
			strcpy(tmp, ptr+1);
		String_To_Upper_Case(tmp);
		sprintf(buf, LV_PATH_FORMAT, tmp, tmp);
	}
	else {

		strcpy(tmp, device_name);
		// input device is a single disk device
		drive_no = Get_Drive_No(tmp);
		get_drive_alphabetic_order(drive_no, 'A', tmp, sizeof(tmp));

		sprintf(buf, LV_PATH_FORMAT, tmp, tmp);
	}
	if (strlen(buf)>=buf_len)
		return ERROR_BUFFER_TOO_SMALL;

	strcpy(lv_name, buf);
	return SUCCESS;
}

/**********************************************************************
** Get_PV_Path_For_LV
** Description:	get the correspondent PV name for the LV
** Input:
**		lv_name :	the logical volume
**		pv_name :	the output PV name
**		buf_len :	the output name buffer length (including '\0')
** Output:	error code --
**		SUCCESS
**		ERROR_BUFFER_TOO_SMALL
***********************************************************************/
int Get_PV_Path_For_LV(char* lv_name, char* pv_name, int buf_len)
{
	char buf[BUF_SIZE];
	char tmp[HD_DEVICE_NAME_LENGTH];
	char *ptr, *s;
#ifdef DEBUG
printf("Get_PV_Path_For_LV(%s)\n", lv_name);
#endif
	if ((ptr = strrchr(lv_name, '/')) == NULL ||
		(s = strchr(ptr+1, '-')) == NULL)
	{
#ifdef DEBUG
		printf("Get_PV_Path_For_LV : LV %s not valid\n", lv_name);
#endif
		return ERROR_INVALID_PARAM;
	}
#ifdef DEBUG
printf("Get_PV_Path_For_LV step2\n");
#endif
	s++;
	// device_name is a RAID device name
	if (strncmp(s, "MD", 2)==0)
		sprintf(buf, "%s%s", MD_DEVICE_PREFIX, s+2);
	else {
		int drive_no;

		drive_no = get_drive_no_by_alphabetic_order('A', s);
#ifdef DEBUG
printf("\tDrive No = %d", drive_no);
#endif
		sprintf(buf, "Drive %d", drive_no);
		if (Get_Profile_String(NASCONF_STORAGE_SECTION, buf, "", tmp, sizeof(tmp))>0)
			sprintf(buf, "/dev/%s%d", tmp, DATA_PART);
		else
			return ERROR_FAIL;
	}
#ifdef DEBUG
printf("\t Device = %s\n", buf);
#endif
	if (strlen(buf)>=buf_len)
		return ERROR_BUFFER_TOO_SMALL;
#ifdef DEBUG
printf("Get_PV_Path_For_LV final : %s\n", buf);
#endif
	strcpy(pv_name, buf);
	return SUCCESS;
}

/**********************************************************************
** Get_LV_Volume_No
** Description:	get the volume number of the logical volume
** Input :
**		lv_name :	the logical volume name
** Output:	the volume number or error code if fails --
**		ERROR_NOT_FOUND
***********************************************************************/
int Get_LV_Volume_No(char* lv_name)
{
	int vol_cnt=0,i, vol_no;
	LVM_VOLUME_CONF *vol_data_list=NULL;

	vol_cnt = Get_All_LVM_Volumes_Conf(&vol_data_list);

	if (vol_cnt<=0) return ERROR_NOT_FOUND;

	for(i=0;i<vol_cnt;i++) {//fixme
		if (!strcmp(vol_data_list[i].lv_name, lv_name) ||
			Is_Instance_Name_Of_Scheduled_Snapshot(vol_data_list[i].lv_name, lv_name))
		{
			vol_no = vol_data_list[i].vol_no;
			free(vol_data_list);
			return vol_no;
		}
	}
	free(vol_data_list);
	return ERROR_NOT_FOUND;
}

/**********************************************************************
** Get_LV_Path
** Description:	get the logical volume path of the volume
** Input :
**		volno :	the volume
**		lv_path : the output logical volume name
**		buf_size : the size of the output buffer
** Output:	lv path length
***********************************************************************/
int Get_LV_Path(int volno, char* lv_path, int buf_size)
{
	int ret, drv_no;
	char tmp[BUF_SIZE], pv_path[buf_size], tmp_lv[buf_size];

	ret = Get_PV_Path(volno, pv_path, buf_size);

	drv_no = Get_Drive_No(pv_path);

	if (Get_LV_Name_For_Device(pv_path, tmp_lv, buf_size) == FALSE)
		strcpy(tmp_lv, "");

	sprintf(tmp, "VOLUME %d", volno);

	Get_Private_Profile_String(tmp, "lv name", tmp_lv, lv_path, buf_size, STORAGE_CONF);

	return strlen(lv_path);
}

/**********************************************************************
** Get_PV_Path
** Description:	get the physical volume path of the volume
** Input :
**		volno :	the volume
**		pv_path : the output logical volume name
**		buf_size : the size of the output buffer
** Output:	pv path length
***********************************************************************/
int Get_PV_Path(int volno, char* pv_path, int buf_size)
{
	char tmp[BUF_SIZE];

	sprintf(tmp, "VOLUME %d", volno);

	Get_Private_Profile_String(tmp, "device name", "", pv_path, buf_size, STORAGE_CONF);

	return strlen(pv_path);
}

/**********************************************************************
** Create_LVM_Storage_Conf
** Description:	create a brand new config file
** Output:	error code --
**		SUCCESS
**
***********************************************************************/
int Create_LVM_Storage_Conf()
{
	int i = 1,j,ret = 0,raid_cnt=0;
	RAID_DEVICE rd_list[MAX_DRIVE_NO];
	char device_name[HD_DEVICE_NAME_LENGTH];
	LVM_VOLUME_CONF vol_data;
	FILE *fp;
	LV_INFO lvinfo;
	PV_INFO pvinfo;
	
	int drive_no = Get_Profile_Integer(NASCONF_STORAGE_SECTION,NASCONF_DRIVE_NO_FIELD,0);

#ifdef	STORAGE_FILE_LOCK
	if (NAS_File_Lock(STORAGE_CONF, 'w')==0)
		return ERROR_OPEN_FILE;
#endif
	//remove original storage.conf & create new	
	unlink(STORAGE_CONF);
	if((fp = fopen(STORAGE_CONF,"w+")) != NULL)
		fclose(fp);
	else {
#ifdef	STORAGE_FILE_LOCK
		NAS_File_Unlock(STORAGE_CONF);
#endif
		return ERROR_FAIL;
	}

	while((ret = Get_Partition_Name(i,DATA_PART,device_name))>=0) {
		if(i>drive_no) break;
		if(Is_Raid_HD(i,NULL,0)) {
			i++;
			continue;
		}
		Make_Single_Volume_Conf(&vol_data, i, 0);

		Get_LV_Name_For_Device(device_name, lvinfo.c_LV_Name, sizeof(lvinfo.c_LV_Name));
		if (Is_LV_Exist(lvinfo.c_LV_Name)) {
			int ratio;
			long long pvsize;

			ret = get_lv_info(lvinfo.c_LV_Name, &lvinfo);
			if (ret<0) {
				vol_data.snapshot_enable = FALSE;
				vol_data.snap_ratio = 0;
#ifdef __EXT3_SNAP__
				vol_data.max_snap_count = MAX_SNAP_NO_PER_VOLUME;
#endif
				goto create_lvm_storage_conf_single_status;
			}
			ret = get_pv_info(device_name, &pvinfo);
			if (ret<0) {
				vol_data.snapshot_enable = FALSE;
				vol_data.snap_ratio = 0;
#ifdef __EXT3_SNAP__
				vol_data.max_snap_count = MAX_SNAP_NO_PER_VOLUME;
#endif
				goto create_lvm_storage_conf_single_status;
			}
			pvsize = pvinfo.ll_PE_Size * pvinfo.i_PE_Count;
			if (pvsize <= 0) {
				vol_data.snapshot_enable = FALSE;
				vol_data.snap_ratio = 0;
#ifdef __EXT3_SNAP__
				vol_data.max_snap_count = MAX_SNAP_NO_PER_VOLUME;
#endif
				goto create_lvm_storage_conf_single_status;
			}
			ratio = lvinfo.ll_LV_Size*100 / pvsize;
			if (ratio == 100) {
				vol_data.snapshot_enable = FALSE;
				vol_data.snap_ratio = 0;
#ifdef __EXT3_SNAP__
				vol_data.max_snap_count = MAX_SNAP_NO_PER_VOLUME;
#endif
			}
			else {
				vol_data.snapshot_enable = TRUE;
				vol_data.snap_ratio = 100-ratio;
#ifdef __EXT3_SNAP__
				vol_data.max_snap_count = get_possible_max_snap_count(
					lvinfo.c_LV_Name,
					pvsize-lvinfo.ll_LV_Size);
#endif
			}
		}
create_lvm_storage_conf_single_status:
		if (!Is_HD_ReadWrite_Safe(i))
			vol_data.status = S_NOT_EXIST;
		else
			vol_data.status = S_UNINITIALIZE;
		i++;
#ifdef	STORAGE_FILE_LOCK
		ret = Add_LVM_Volume_Record_Ex(&vol_data, FALSE);
#else
		ret = Add_LVM_Volume_Record(&vol_data);
#endif

		if(ret<0) {
#ifdef	STORAGE_FILE_LOCK
			NAS_File_Unlock(STORAGE_CONF);
#endif
			return ret;
		}
	}
	raid_cnt = Get_All_RaidDev_Conf(rd_list,MAX_DRIVE_NO);
	if(raid_cnt>0) {
		for(i=0;i<raid_cnt;i++) {
			strcpy(vol_data.device_name,rd_list[i].name);
			Get_LV_Name_For_Device(vol_data.device_name, vol_data.lv_name, LV_PATH_MAX_LEN);
			vol_data.list_cnt = rd_list[i].data_drive_cnt;
			vol_data.spare_list_cnt = rd_list[i].spare_drive_cnt;
			if(strcmp(rd_list[i].raid_level,"linear")==0)
				vol_data.raid_level = LINEAR;
			else
				vol_data.raid_level = atoi(rd_list[i].raid_level);
				
			for(j=0;j<rd_list[i].data_drive_cnt;j++) {
				vol_data.drive_no_list[j] = rd_list[i].data_drive_no[j];
			}
			for(j=0;j<rd_list[i].spare_drive_cnt;j++) {
				vol_data.spare_drive_list[j] = rd_list[i].spare_drive_no[j];
			}
// Cat : ==>
			Get_LV_Name_For_Device(vol_data.device_name, lvinfo.c_LV_Name, sizeof(lvinfo.c_LV_Name));
			if (Is_LV_Exist(lvinfo.c_LV_Name)) {
				int ratio;
				long long pvsize;

				ret = get_lv_info(lvinfo.c_LV_Name, &lvinfo);
				if (ret<0) {
					vol_data.snapshot_enable = FALSE;
					vol_data.snap_ratio = 0;
#ifdef __EXT3_SNAP__
					vol_data.max_snap_count = MAX_SNAP_NO_PER_VOLUME;
#endif
					goto create_lvm_storage_conf_raid_status;
				}
				ret = get_pv_info(vol_data.device_name, &pvinfo);
				if (ret<0) {
					vol_data.snapshot_enable = FALSE;
					vol_data.snap_ratio = 0;
#ifdef __EXT3_SNAP__
					vol_data.max_snap_count = MAX_SNAP_NO_PER_VOLUME;
#endif
					goto create_lvm_storage_conf_raid_status;
				}
				pvsize = pvinfo.ll_PE_Size * pvinfo.i_PE_Count;
				if (pvsize <= 0) {
					vol_data.snapshot_enable = FALSE;
					vol_data.snap_ratio = 0;
#ifdef __EXT3_SNAP__
					vol_data.max_snap_count = MAX_SNAP_NO_PER_VOLUME;
#endif
					goto create_lvm_storage_conf_raid_status;
				}

				ratio = lvinfo.ll_LV_Size*100 / pvsize;
				if (ratio == 100) {
					vol_data.snapshot_enable = FALSE;
					vol_data.snap_ratio = 0;
#ifdef __EXT3_SNAP__
					vol_data.max_snap_count = MAX_SNAP_NO_PER_VOLUME;
#endif
				}
				else {
					vol_data.snapshot_enable = TRUE;
					vol_data.snap_ratio = 100-ratio;
#ifdef __EXT3_SNAP__
					vol_data.max_snap_count = get_possible_max_snap_count(
						lvinfo.c_LV_Name,
						pvsize-lvinfo.ll_LV_Size);
#endif
				}
			}
create_lvm_storage_conf_raid_status:
			vol_data.status = S_READY;
#ifdef	STORAGE_FILE_LOCK
			ret = Add_LVM_Volume_Record_Ex(&vol_data, FALSE);
#else
			ret = Add_LVM_Volume_Record(&vol_data);
#endif
			if(ret<0) {
#ifdef	STORAGE_FILE_LOCK
				NAS_File_Unlock(STORAGE_CONF);
#endif
				return ret;
			}
		}
	}

#ifdef	STORAGE_FILE_LOCK
	NAS_File_Unlock(STORAGE_CONF);
#endif
	return SUCCESS;
}

/**********************************************************************
** Restore_LVM_Storage_Conf
** Description:	create a new config file to replace the incomplete one
** Output:	error code --
**		SUCCESS
**
***********************************************************************/
int Restore_LVM_Storage_Conf()
{
	int i = 1,j,ret = 0,raid_cnt=0;
	RAID_DEVICE rd_list[MAX_DRIVE_NO];
	char device_name[HD_DEVICE_NAME_LENGTH];
	LVM_VOLUME_CONF vol_data;
	FILE *fp;
	LV_INFO lvinfo;
	PV_INFO pvinfo;
	int drive_no = Get_Profile_Integer(NASCONF_STORAGE_SECTION,NASCONF_DRIVE_NO_FIELD,0);

#ifdef	DEBUG
printf("%s %s -- line %d\n", __FILE__, __FUNCTION__, __LINE__);
#endif
#ifdef	STORAGE_FILE_LOCK
	if (NAS_File_Lock(STORAGE_CONF, 'w') == 0)
		return ERROR_LOCK_FILE;
#endif
	//remove original storage.conf & create new	
#ifdef	DEBUG
printf("%s %s -- line %d\n", __FILE__, __FUNCTION__, __LINE__);
#endif
	unlink(STORAGE_CONF);
	if((fp = fopen(STORAGE_CONF,"w+")) != NULL)
		fclose(fp);
	else {
#ifdef	STORAGE_FILE_LOCK
		NAS_File_Unlock(STORAGE_CONF);
#endif
		return ERROR_FAIL;
	}
#ifdef	DEBUG
printf("%s %s -- line %d\n", __FILE__, __FUNCTION__, __LINE__);
#endif
		
	while((ret = Get_Partition_Name(i,DATA_PART,device_name))>=0) {
		if(i>drive_no) break;
		if(Is_Raid_HD(i,NULL,0)) {
			i++;
			continue;
		}

		Make_Single_Volume_Conf(&vol_data, i, 0);
#ifdef	DEBUG
printf("%s %s -- line %d\n", __FILE__, __FUNCTION__, __LINE__);
#endif
// Cat : ==>
		Get_LV_Name_For_Device(device_name, lvinfo.c_LV_Name, sizeof(lvinfo.c_LV_Name));
#ifdef	DEBUG
printf("%s %s -- line %d; lv %s\n", __FILE__, __FUNCTION__, __LINE__, lvinfo.c_LV_Name);
#endif
		if (Is_LV_Exist(lvinfo.c_LV_Name)) {
			int ratio;
			long long pvsize;
#ifdef	DEBUG
printf("%s %s -- line %d\n", __FILE__, __FUNCTION__, __LINE__);
#endif

			ret = get_lv_info(lvinfo.c_LV_Name, &lvinfo);
#ifdef	DEBUG
printf("%s %s -- line %d\n", __FILE__, __FUNCTION__, __LINE__);
#endif
			if (ret<0) {
#ifdef DEBUG
				printf("Restore_LVM_Storage_Conf : "
					"get_lv_info(%s) fails, err=%d\n",
					lvinfo.c_LV_Name, ret);
#endif
				vol_data.snapshot_enable = FALSE;
				vol_data.snap_ratio = 0;
#ifdef __EXT3_SNAP__
				vol_data.max_snap_count = MAX_SNAP_NO_PER_VOLUME;
#endif
				goto restore_lvm_storage_conf_single_status;
			}
#ifdef	DEBUG
printf("%s %s -- line %d\n", __FILE__, __FUNCTION__, __LINE__);
#endif
			ret = get_pv_info(device_name, &pvinfo);
#ifdef	DEBUG
printf("%s %s -- line %d\n", __FILE__, __FUNCTION__, __LINE__);
#endif
			if (ret<0) {
#ifdef DEBUG
				printf("Restore_LVM_Storage_Conf : "
					"get_pv_info(%s) fails, err=%d\n",
					device_name, ret);
#endif
				vol_data.snapshot_enable = FALSE;
				vol_data.snap_ratio = 0;
#ifdef __EXT3_SNAP__
				vol_data.max_snap_count = MAX_SNAP_NO_PER_VOLUME;
#endif
				goto restore_lvm_storage_conf_single_status;
			}
#ifdef	DEBUG
printf("%s %s -- line %d\n", __FILE__, __FUNCTION__, __LINE__);
#endif
			pvsize = pvinfo.ll_PE_Size * pvinfo.i_PE_Count;
			if (pvsize<=0) {
				vol_data.snapshot_enable = FALSE;
				vol_data.snap_ratio = 0;
#ifdef __EXT3_SNAP__
				vol_data.max_snap_count = MAX_SNAP_NO_PER_VOLUME;
#endif
				goto restore_lvm_storage_conf_single_status;
			}
#ifdef	DEBUG
printf("%s %s -- line %d\n", __FILE__, __FUNCTION__, __LINE__);
#endif
			ratio = lvinfo.ll_LV_Size*100 / pvsize;
#ifdef DEBUG
			printf("Restore_LVM_Storage_Conf : "
				"pvsize = %lld KB, lv size = %lld KB, ratio = %d\n",
				pvsize, lvinfo.ll_LV_Size, ratio);
#endif

			if (ratio == 100) {
				vol_data.snapshot_enable = FALSE;
				vol_data.snap_ratio = 0;
#ifdef __EXT3_SNAP__
				vol_data.max_snap_count = MAX_SNAP_NO_PER_VOLUME;
#endif
			}
			else {
				vol_data.snapshot_enable = TRUE;
				vol_data.snap_ratio = 100-ratio;
#ifdef __EXT3_SNAP__
				vol_data.max_snap_count = get_possible_max_snap_count(
					lvinfo.c_LV_Name,
					pvsize-lvinfo.ll_LV_Size);
#endif
			}
		}
#ifdef	DEBUG
printf("%s %s -- line %d\n", __FILE__, __FUNCTION__, __LINE__);
#endif
restore_lvm_storage_conf_single_status:
// <== Cat
		if (!Is_HD_ReadWrite_Safe(i))
			vol_data.status=S_NOT_EXIST;
		else if (!Is_HD_Valid(i))
			vol_data.status = S_UNINITIALIZE;
		else
			vol_data.status = S_READY;
#ifdef	DEBUG
printf("%s %s -- line %d\n", __FILE__, __FUNCTION__, __LINE__);
#endif

		i++;
#ifdef	STORAGE_FILE_LOCK
		ret = Add_LVM_Volume_Record_Ex(&vol_data, FALSE);
#else
		ret = Add_LVM_Volume_Record(&vol_data);
#endif

#ifdef	DEBUG
printf("%s %s -- line %d\n", __FILE__, __FUNCTION__, __LINE__);
#endif
		if(ret<0) {
#ifdef	STORAGE_FILE_LOCK
			NAS_File_Unlock(STORAGE_CONF);	
#endif
			return ret;
		}
	}
#ifdef	DEBUG
printf("%s %s -- line %d\n", __FILE__, __FUNCTION__, __LINE__);
#endif

	raid_cnt = Get_All_RaidDev_Conf(rd_list,MAX_DRIVE_NO);
	if(raid_cnt>0) {
		for(i=0;i<raid_cnt;i++) {
#ifdef	DEBUG
printf("%s %s -- line %d\n", __FILE__, __FUNCTION__, __LINE__);
#endif
			strcpy(vol_data.device_name,rd_list[i].name);
			Get_LV_Name_For_Device(vol_data.device_name, vol_data.lv_name, LV_PATH_MAX_LEN);
			vol_data.list_cnt = rd_list[i].data_drive_cnt;
			vol_data.spare_list_cnt = rd_list[i].spare_drive_cnt;
			if(strcmp(rd_list[i].raid_level,"linear")==0)
				vol_data.raid_level = LINEAR;
			else
				vol_data.raid_level = atoi(rd_list[i].raid_level);
				
			for(j=0;j<rd_list[i].data_drive_cnt;j++) {
				vol_data.drive_no_list[j] = rd_list[i].data_drive_no[j];
			}
			for(j=0;j<rd_list[i].spare_drive_cnt;j++) {
				vol_data.spare_drive_list[j] = rd_list[i].spare_drive_no[j];
			}
			vol_data.status = S_READY;
// Cat : ==>
			Get_LV_Name_For_Device(vol_data.device_name, lvinfo.c_LV_Name, sizeof(lvinfo.c_LV_Name));
#ifdef	DEBUG
printf("%s %s -- line %d\n", __FILE__, __FUNCTION__, __LINE__);
#endif
			if (Is_LV_Exist(lvinfo.c_LV_Name)) {
				int ratio;
				long long pvsize;

				ret = get_lv_info(lvinfo.c_LV_Name, &lvinfo);
				if (ret<0) {
					vol_data.snapshot_enable = FALSE;
					vol_data.snap_ratio = 0;
#ifdef __EXT3_SNAP__
					vol_data.max_snap_count = MAX_SNAP_NO_PER_VOLUME;
#endif
					goto restore_lvm_storage_conf_raid_status;
				}
				ret = get_pv_info(vol_data.device_name, &pvinfo);
				if (ret<0) {
					vol_data.snapshot_enable = FALSE;
					vol_data.snap_ratio = 0;
#ifdef __EXT3_SNAP__
					vol_data.max_snap_count = MAX_SNAP_NO_PER_VOLUME;
#endif
					goto restore_lvm_storage_conf_raid_status;
				}
				pvsize = pvinfo.ll_PE_Size * pvinfo.i_PE_Count;
				if (pvsize <= 0) {
					vol_data.snapshot_enable = FALSE;
					vol_data.snap_ratio = 0;
#ifdef __EXT3_SNAP__
					vol_data.max_snap_count = MAX_SNAP_NO_PER_VOLUME;
#endif
					goto restore_lvm_storage_conf_raid_status;
				}

				ratio = lvinfo.ll_LV_Size*100 / pvsize;
				if (ratio == 100) {
					vol_data.snapshot_enable = FALSE;
					vol_data.snap_ratio = 0;
#ifdef __EXT3_SNAP__
					vol_data.max_snap_count = MAX_SNAP_NO_PER_VOLUME;
#endif
				}
				else {
					vol_data.snapshot_enable = TRUE;
					vol_data.snap_ratio = 100-ratio;
#ifdef __EXT3_SNAP__
					vol_data.max_snap_count = get_possible_max_snap_count(
						lvinfo.c_LV_Name,
						pvsize-lvinfo.ll_LV_Size);
#endif
				}
			}
#ifdef	DEBUG
printf("%s %s -- line %d\n", __FILE__, __FUNCTION__, __LINE__);
#endif

restore_lvm_storage_conf_raid_status:
#ifdef	DEBUG
printf("%s %s -- line %d\n", __FILE__, __FUNCTION__, __LINE__);
#endif
#ifdef	STORAGE_FILE_LOCK
			ret = Add_LVM_Volume_Record_Ex(&vol_data, FALSE);
#else
			ret = Add_LVM_Volume_Record(&vol_data);
#endif
#ifdef	DEBUG
printf("%s %s -- line %d\n", __FILE__, __FUNCTION__, __LINE__);
#endif
			if(ret<0) {
#ifdef	STORAGE_FILE_LOCK
				NAS_File_Unlock(STORAGE_CONF);	
#endif
				return ret;
			}
		}
	}
#ifdef	DEBUG
printf("%s %s -- line %d\n", __FILE__, __FUNCTION__, __LINE__);
#endif
#ifdef	STORAGE_FILE_LOCK
	NAS_File_Unlock(STORAGE_CONF);
#endif

	return SUCCESS;
}

/**********************************************************************
** Get_Snapshot_Volume_No
** Description:	get the volume number of the logical volume
** Input :
**		snapshot_volume_name :	the snapshot logical volume name
** Output:	the volume number or error code if fails --
**		ERROR_NOT_FOUND
***********************************************************************/
int Get_Snapshot_Volume_No(char* snapshot_volume_name)
{
	int vol_cnt=0, i, vol_no, slen1, slen2;
	SNAPSHOT_VOLUME_CONF *vol_data_list=NULL;
	char tmpbuf[LV_PATH_MAX_LEN];
	int buildpath_flag=0;

	vol_cnt = Get_All_Snapshot_Volumes_Conf(&vol_data_list);
	if (vol_cnt<=0) return ERROR_NOT_FOUND;

	if (strchr(snapshot_volume_name, '/')==NULL)
		buildpath_flag++; // must guess the full path of the snapshot in every loop
	else
		strcpy(tmpbuf, snapshot_volume_name);

	for(i=0;i<vol_cnt;i++) {
		if (buildpath_flag)
			get_snap_instance_name_by_source_lv(vol_data_list[i].c_Src_LV,
				snapshot_volume_name,
				tmpbuf, sizeof(tmpbuf));

		slen1 = strlen(vol_data_list[i].c_Snapshot_Path);
		slen2 = strlen(tmpbuf);
		if (slen1 > slen2) continue;
		// either the same or snapshot_volume_name is an instance of a scheduled
		if ((slen1==slen2 &&
			!strcmp(tmpbuf, vol_data_list[i].c_Snapshot_Path)) ||
			Is_Instance_Name_Of_Scheduled_Snapshot(vol_data_list[i].c_Snapshot_Path, tmpbuf))
		{
			vol_no = vol_data_list[i].i_Snapshot_No;
			free(vol_data_list);
			return vol_no;
		}
	}
	free(vol_data_list);
	return ERROR_NOT_FOUND;
}

/**********************************************************************
** Get_Snapshot_Volume_Path
** Description:	get the logical volume path of the volume
** Input :
**		sn_volno:	the volume
**		sn_path	: 	the output logical volume name
**		buf_size: 	the size of the output buffer
** Output:	snapshot volume path length or error code
***********************************************************************/
int Get_Snapshot_Volume_Path(int sn_volno, char* sn_path, int buf_size)
{
	char tmp[BUF_SIZE];

	sprintf(tmp, "SNAPSHOT %d", sn_volno);

	Get_Private_Profile_String(
		tmp, "DEVICE NAME", "",
		sn_path, buf_size, SNAPSHOT_CONF);

	return strlen(sn_path);
}

/**********************************************************************
** Is_LVM_Volume
** Description:	Check if the volume is a LVM volume or an old one
** Input:
**		vol_no:		the snapshot volume number
** Output:	TRUE or FALSE
***********************************************************************/
BOOL Is_LVM_Volume(int vol_no)
{
	char buf[BUF_SIZE];

	if (Get_System_Version(buf, sizeof(buf)) == SUCCESS) {
		char *tmp;
		int major_ver;

		tmp = strtok(buf, ".");
		major_ver = atoi(buf);
		//Shone marked for TS-101 was ported from NasWare4 while its version had been set to 1.0.0 2005,09,12
		//if (major_ver > 2) // NasWare 3.x or later
			return TRUE;
	}
	return FALSE;
}
// -------------------normal logical volumes --------------------------

/**********************************************************************
** Get_All_LVM_Volumes_Conf
** Description:	Get all LVM volumes' configurations
** Output:	the count of all volumes
***********************************************************************/
int Get_All_LVM_Volumes_Conf(LVM_VOLUME_CONF **vol_data_list_ptr)
{
	int i,cnt=0,ret;
	LVM_VOLUME_CONF *vol_data_list=NULL;
	LVM_VOLUME_CONF vol_data;
	void *ptr=NULL;

	for(i=1;i<MAX_VOLUME_NO+1;i++) {
		ret = Get_One_LVM_Volume_Conf(i,&vol_data);
		if (ret < 0)
			continue;
		else {
			ptr = realloc((void*)vol_data_list,
				(cnt+1)*sizeof(LVM_VOLUME_CONF));
			if (ptr==NULL) {
				if (vol_data_list) free(vol_data_list);
				return ERROR_OUT_OF_MEMORY;
			}

			vol_data_list = (LVM_VOLUME_CONF *)ptr;

			memcpy((void*)(&vol_data_list[cnt]),(void*)(&vol_data),
				sizeof(LVM_VOLUME_CONF));
			cnt++;
		}
	}
	*vol_data_list_ptr = vol_data_list;
	return cnt;
}

/**********************************************************************
** Get_One_LVM_Volume_Conf
** Description:	Get one LVM volume configurations
** Output:	Error Code --
**		SUCCESS
***********************************************************************/
int Get_One_LVM_Volume_Conf(int vol_no, LVM_VOLUME_CONF *vol_data)
{
	char vol_section[20];
	char tmp_buf[BUF_SIZE],drv_no_str[4];
	int drive_no_cnt=1,ret;

	sprintf(vol_section,"VOLUME %d",vol_no);
	//set vol no
	vol_data->vol_no = vol_no;

	// get device name
	ret = Conf_Get_Field(STORAGE_CONF,vol_section,"device name",vol_data->device_name,HD_DEVICE_NAME_LENGTH);
	if(ret<0) return ret;
	// get raid level
	ret = Conf_Get_Field(STORAGE_CONF,vol_section,"raid level",tmp_buf,BUF_SIZE);
	if(ret<0) return ret;
	vol_data->raid_level = atoi(tmp_buf);

	//get drive no list & list count
	ret = Conf_Get_Field(STORAGE_CONF,vol_section,"drive no",tmp_buf,BUF_SIZE);
	if(ret<0) return ret;

	while(Get_String_Field(tmp_buf,drive_no_cnt,',',drv_no_str,4)>=0) {
		if ((strlen(drv_no_str)==0) || (atoi(drv_no_str)<=0))
			break;

		vol_data->drive_no_list[drive_no_cnt-1] = atoi(drv_no_str);

		drive_no_cnt++;
	}
	vol_data->list_cnt = drive_no_cnt-1;

	//get spare drive no list & list count
	ret = Conf_Get_Field(STORAGE_CONF,vol_section,"spare drive no",tmp_buf,BUF_SIZE);
	if(ret==0) {
		drive_no_cnt=1;
		while(Get_String_Field(tmp_buf,drive_no_cnt,',',drv_no_str,4)>=0) {
			if ((strlen(drv_no_str)==0) || (atoi(drv_no_str)<=0))
				break;

			vol_data->spare_drive_list[drive_no_cnt-1] = atoi(drv_no_str);
			drive_no_cnt++;
		}
		vol_data->spare_list_cnt = drive_no_cnt-1;
	}
	else vol_data->spare_list_cnt = 0;

	//get volume status
	ret = Conf_Get_Field(STORAGE_CONF,vol_section,"status",tmp_buf,BUF_SIZE);
	if(ret<0) return ret;
	vol_data->status = atoi(tmp_buf);

	// get lv path
	ret = Get_LV_Path(vol_no, vol_data->lv_name, LV_PATH_MAX_LEN);
	if (ret<0) return ret;

	//get snapshot enable or not
	strcpy(tmp_buf, "FALSE");
	ret = Conf_Get_Field(STORAGE_CONF, vol_section, "enable snapshot",
		tmp_buf, BUF_SIZE);
	vol_data->snapshot_enable = (strcasecmp(tmp_buf, "TRUE")) ? FALSE : TRUE;

	if (vol_data->snapshot_enable) {
#ifdef	__EXT3_SNAP__
		//get snap max count
		ret = Conf_Get_Field(STORAGE_CONF,vol_section,"max snap count",tmp_buf,BUF_SIZE);
		if(ret<0) return ret;
		vol_data->max_snap_count = atoi(tmp_buf);
#endif
		//get snap repository ratio
		ret = Conf_Get_Field(STORAGE_CONF,vol_section,"snap repository ratio",tmp_buf,BUF_SIZE);
		if(ret==0) {
			PV_INFO pvinfo;

			vol_data->snap_ratio = atoi(tmp_buf);

			if (vol_data->snap_ratio>0 &&
				get_pv_info(vol_data->device_name, &pvinfo)==SUCCESS)
			{
				vol_data->snap_repository_size = 
					get_approx_size(pvinfo.ll_PE_Size * pvinfo.i_PE_Count * \
						vol_data->snap_ratio /100, -1);
			}
			else
				vol_data->snap_repository_size = 0;
		}
		else {
			vol_data->snap_ratio = 0;
			vol_data->snap_repository_size = 0;
		}
	}
	else {
		vol_data->snap_ratio = 0;
		vol_data->snap_repository_size = 0;
#ifdef	__EXT3_SNAP__
		vol_data->max_snap_count = 0;
#endif
	}

	return SUCCESS;
}
/**********************************************************************
** Get_Uninit_Disk_LVM_Volumes_Conf
** Description:	Get uninitialized volumes' configurations
** Output:	the number of all uninit volumes or the error Code --
**		SUCCESS
***********************************************************************/
int Get_Uninit_Disk_LVM_Volumes_Conf(LVM_VOLUME_CONF **vol_data_list_ptr)
{
	LVM_VOLUME_CONF *vol_data_list=NULL;
	LVM_VOLUME_INFO *vol_confs=NULL;
	int i, cnt=0, vol_cnt=0;
	void *ptr=NULL;

#ifdef	STORAGE_FILE_LOCK
	repair_lvm_storage_conf_if_necessary();
#else
	FILE *fp;

	if((fp = fopen(STORAGE_CONF,"r")) == NULL) 
		Create_LVM_Storage_Conf();
	else{
		char tmp_line[80];
		strcpy(tmp_line,"");
		fgets(tmp_line,80,fp);
		fclose(fp);
		if(strlen(tmp_line)== 0)
			Restore_LVM_Storage_Conf(); 	
	}
#endif
	vol_cnt = Get_All_LVM_Volumes_Info(&vol_confs);

	if (vol_cnt<=0) return vol_cnt;

	for(i=0;i<vol_cnt;i++) {
		//just get single-disk volumes
		if(vol_confs[i].vol_data.raid_level == SINGLE &&
			(vol_confs[i].vol_data.status == S_UNINITIALIZE ||
			vol_confs[i].vol_data.status == S_INVALID)) {

			if(!Is_Volume_HD_Exist(vol_confs[i].vol_data.vol_no))
				continue;

			ptr = realloc((void*)vol_data_list,
				(cnt+1)*sizeof(LVM_VOLUME_CONF));

			if (ptr==NULL) {
				Release_LVM_Info_List(vol_confs, vol_cnt);
				Release_List(vol_data_list);
				return ERROR_OUT_OF_MEMORY;
			}

			vol_data_list = (LVM_VOLUME_CONF *)ptr;

			memcpy((void*)&vol_data_list[cnt],
				(void*)&(vol_confs[i].vol_data),
				sizeof(LVM_VOLUME_CONF));
			cnt++;
		}
	}
	Release_LVM_Info_List(vol_confs, vol_cnt);
	*vol_data_list_ptr = vol_data_list;
	return cnt;
}

/**********************************************************************
** Get_Single_Disk_LVM_Volumes_Conf
** Description:	Get single-disk volumes' configurations
** Output:	Error Code --
**		SUCCESS
***********************************************************************/
int Get_Single_Disk_LVM_Volumes_Conf(LVM_VOLUME_CONF **vol_data_list_ptr)
{	
	int i, cnt=0, vol_cnt;
	LVM_VOLUME_INFO *tmp_vol_list=NULL;
	LVM_VOLUME_CONF *vol_data_list=NULL;
	void *ptr=NULL;
	
#ifdef	STORAGE_FILE_LOCK
	repair_lvm_storage_conf_if_necessary();
#else
	FILE *fp;

	if((fp = fopen(STORAGE_CONF,"r")) == NULL) 
		Create_LVM_Storage_Conf();
	else {
		char tmp_line[80];
		strcpy(tmp_line,"");
		fgets(tmp_line,80,fp);
		fclose(fp);
		if(strlen(tmp_line)== 0) Restore_LVM_Storage_Conf(); 	
	}
#endif
	vol_cnt = Get_All_LVM_Volumes_Info(&tmp_vol_list);

	if (vol_cnt<=0) return vol_cnt;
	
	for(i=0;i<vol_cnt;i++) {
		//just get single-disk volumes
		if(tmp_vol_list[i].vol_data.raid_level == SINGLE) {

			ptr = realloc((void*)vol_data_list,
				(cnt+1)*sizeof(LVM_VOLUME_CONF));

			if (ptr==NULL) {
				Release_List(vol_data_list);
				Release_LVM_Info_List(tmp_vol_list, vol_cnt);
				return ERROR_OUT_OF_MEMORY;
			}
			vol_data_list = (LVM_VOLUME_CONF *)ptr;

			memcpy((void*)&vol_data_list[cnt],
				(void*)&(tmp_vol_list[i].vol_data),
				sizeof(LVM_VOLUME_CONF));

			if(vol_data_list[cnt].status == S_UNINITIALIZE) {
				if(!Is_Volume_HD_Exist(vol_data_list[cnt].vol_no)) continue;
			}
					
			cnt++;
		}
	}
	Release_LVM_Info_List(tmp_vol_list, vol_cnt);
	*vol_data_list_ptr=vol_data_list;
	return cnt;
}

/**********************************************************************
** Get_Raid_Disk_LVM_Volumes_Info
** Description:	Get one RAID volume status
** Output:	RAID volume count
***********************************************************************/
int Get_Raid_Disk_LVM_Volumes_Info(LVM_VOLUME_INFO **vol_info_list_ptr)
{
	int i,cnt=0, vol_cnt;
	LVM_VOLUME_INFO *vol_info_list=NULL;
	LVM_VOLUME_INFO *all_vol_list=NULL;
	void *ptr=NULL;

	vol_cnt = Get_All_LVM_Volumes_Info(&all_vol_list);

	for(i=0; i<vol_cnt; i++) {
		if(all_vol_list[i].vol_data.raid_level > SINGLE) {
			ptr = realloc((void*)vol_info_list,
				(cnt+1)*sizeof(LVM_VOLUME_INFO));
			if (ptr==NULL) {
				Release_LVM_Info_List(vol_info_list, cnt);
				Release_LVM_Info_List(all_vol_list, vol_cnt);
				return ERROR_OUT_OF_MEMORY;
			}
			vol_info_list = (LVM_VOLUME_INFO *)ptr;

			memcpy((void*)&(vol_info_list[cnt]),
				(void*)&(all_vol_list[i]),
				sizeof(LVM_VOLUME_INFO));

			// rebuild the snapshot list
			if (all_vol_list[i].snapshot_cnt>0) {
				ptr = calloc(all_vol_list[i].snapshot_cnt,
					sizeof(SNAPSHOT_VOLUME_CONF));
				if (ptr==NULL) {
					free(vol_info_list + cnt);

					Release_LVM_Info_List(vol_info_list, cnt);
					Release_LVM_Info_List(all_vol_list, vol_cnt);
					return ERROR_OUT_OF_MEMORY;
				}
				vol_info_list[cnt].snap_list = ptr;

				memcpy(vol_info_list[cnt].snap_list,
					all_vol_list[i].snap_list,
					all_vol_list[i].snapshot_cnt*sizeof(SNAPSHOT_VOLUME_CONF));
			}
			else vol_info_list[cnt].snap_list = NULL;

			cnt++;
		}
	}
	Release_LVM_Info_List(all_vol_list, vol_cnt);

	*vol_info_list_ptr = vol_info_list;
	return cnt;
}
/**********************************************************************
** Get_All_LVM_Configured_Vol_For_Share
** Description:	Get all ready volumes
** Output:	total volumes found
***********************************************************************/
int Get_All_LVM_Configured_Vol_For_Share(LVM_VOLUME_INFO **mount_vol_ptr)
{
	LVM_VOLUME_INFO *vol_info=NULL,*mount_vol=NULL;
	int vol_cnt,i,ret_cnt=0;
	void *ptr=NULL;

	vol_cnt = Get_All_LVM_Volumes_Info(&vol_info);

	if (vol_cnt<=0)
		return 0;

	for(i=0;i<vol_cnt;i++) {
		if(vol_info[i].vol_data.status == S_READY ||
			vol_info[i].vol_data.status == S_DEGRADED ||
			vol_info[i].vol_data.status == S_REBUILDING_RAID) {

			ret_cnt++;

			ptr = realloc((void*)mount_vol,
				ret_cnt * sizeof(LVM_VOLUME_INFO));
			if (ptr==NULL) {
				Release_LVM_Info_List(mount_vol, ret_cnt-1);
				Release_LVM_Info_List(vol_info, vol_cnt);
				return ERROR_OUT_OF_MEMORY;
			}
			mount_vol = (LVM_VOLUME_INFO*)ptr;

			memcpy(&(mount_vol[ret_cnt-1]), &(vol_info[i]), sizeof(LVM_VOLUME_INFO));

			// rebuild the snapshot list
			if (vol_info[i].snapshot_cnt>0) {
				ptr = calloc(vol_info[i].snapshot_cnt,
					sizeof(SNAPSHOT_VOLUME_CONF));
				if (ptr==NULL) {
					free(mount_vol + ret_cnt - 1);

					Release_LVM_Info_List(mount_vol, ret_cnt-1);
					Release_LVM_Info_List(vol_info, vol_cnt);
					return ERROR_OUT_OF_MEMORY;
				}
				mount_vol[ret_cnt-1].snap_list = ptr;

				memcpy(mount_vol[ret_cnt-1].snap_list,
					vol_info[i].snap_list,
					vol_info[i].snapshot_cnt*sizeof(SNAPSHOT_VOLUME_CONF));
			}
			else mount_vol[ret_cnt-1].snap_list = NULL;
		}
	}

	Release_LVM_Info_List(vol_info, vol_cnt);
	*mount_vol_ptr = mount_vol;
	return ret_cnt;
}

/**********************************************************************
** Get_LVM_Configured_Vol_No_For_Share
** Description:	Get the LVM volume number where the share resides in
** Output:	the volume number or error code
***********************************************************************/
int Get_LVM_Configured_Vol_No_For_Share(char *share_name)
{
	NAS_SHARE_INFO	share_info;
	LVM_VOLUME_INFO	*vol_info=NULL;
	int vol_cnt,i,ret;
	char mp_str[128];

	strcpy(share_info.share_name,share_name);
	share_info.read_list_cnt = 0;       
	share_info.read_list_ptr = NULL;       
	share_info.write_list_cnt = 0;     
	share_info.read_list_ptr = NULL;
	ret = Get_NAS_Share_Info_Ex(&share_info);

	if (ret<0) {
#ifdef DEBUG
		printf("\tGet_NAS_Share_Info_Ex fails, err=%d\n", ret);
#endif
		return ret;
	}

	vol_cnt = Get_All_LVM_Configured_Vol_For_Share(&vol_info);

	if (vol_cnt<=0) {
#ifdef DEBUG
		printf("\tGet_All_LVM_Configured_Vol_For_Share "
			"fails or not configured volume found, err=%d", ret);
#endif
		return ERROR_NOT_FOUND;
	}

	for(i=0;i<vol_cnt;i++) {
		Get_MP_String(vol_info[i].vol_data.drive_no_list[0],DATA_PART,mp_str);
		if( strstr(share_info.path,mp_str) != NULL ) {
			if (share_info.read_list_cnt>0)
				free(share_info.read_list_ptr);
			if (share_info.write_list_cnt>0)
				free(share_info.write_list_ptr);
			Release_LVM_Info_List(vol_info, vol_cnt);
			return i+1;
		}
	}	
	if (share_info.read_list_cnt>0)
		free(share_info.read_list_ptr);
	if (share_info.write_list_cnt>0)
		free(share_info.write_list_ptr);

	Release_LVM_Info_List(vol_info, vol_cnt);
	return ERROR_NOT_FOUND;
}

/**********************************************************************
** Get_One_LVM_Volume_Info
** Description:	Get one volume status
** Output:	Error Code --
**		SUCCESS
***********************************************************************/
int Get_One_LVM_Volume_Info(int vol_no,LVM_VOLUME_INFO *vol_info)
{
	LVM_VOLUME_CONF vol_data;
	PARTITION_INFO part_info;
	SNAPSHOT_VOLUME_CONF *sn_list=NULL;
	int ret;
	BOOL is_lvm_volume;

#ifdef	STORAGE_FILE_LOCK
	repair_lvm_storage_conf_if_necessary();
#else
	FILE *fp;

	if((fp = fopen(STORAGE_CONF,"r")) == NULL) 
		Create_LVM_Storage_Conf();
	else {
		char tmp_line[80];
		strcpy(tmp_line,"");
		fgets(tmp_line,80,fp);
		fclose(fp);
		if(strlen(tmp_line)== 0) Restore_LVM_Storage_Conf(); 	
	}
#endif

	ret = Get_One_LVM_Volume_Conf(vol_no,&vol_data);
	if(ret<0) {
#ifdef DEBUG
		printf("\tGet_One_LVM_Volume_Conf fails,"
			" err=%d!\n", ret);
#endif
		return ret;
	}

	memcpy((void*)&(vol_info->vol_data), (void*)&vol_data,
		sizeof(LVM_VOLUME_CONF));
	vol_info->total_size = -1;
	vol_info->free_size  = -1;
	vol_info->resync_per = -1;

	if(((vol_info->vol_data.status > S_READY &&
		vol_info->vol_data.status < S_REBUILDING_RAID) &&
	    (vol_info->vol_data.raid_level == SINGLE ||
		(vol_info->vol_data.status != S_FORMATTING &&
		 vol_info->vol_data.status != S_SCANNING))) ||
	    vol_info->vol_data.status >= S_RESTORING_SNAPSHOT)
	{
		vol_info->snapshot_cnt = 0;
		vol_info->snap_list = NULL;
		return SUCCESS;	
	}

	is_lvm_volume = Is_LVM_Volume(vol_no);

	if(vol_info->vol_data.raid_level != SINGLE &&
		!Is_RaidDev_Active(vol_data.device_name))
	{
		vol_info->vol_data.status = S_NOT_ACTIVE;
	}
	else if (vol_info->vol_data.raid_level == SINGLE &&
		!Is_HD_ReadWrite_Safe(vol_data.drive_no_list[0]))
	{
		vol_info->vol_data.status = S_NOT_EXIST;
	}
	else if (vol_info->vol_data.raid_level == SINGLE &&
		Verify_Hidden_Conf_File(vol_data.drive_no_list[0])<0)
	{
		vol_info->vol_data.status = S_UNINITIALIZE;
	}
	else if(vol_info->vol_data.status <= S_READY ||
		vol_info->vol_data.status == S_REBUILDING_RAID ||
		(vol_info->vol_data.raid_level > SINGLE &&
		 (vol_info->vol_data.status == S_FORMATTING ||
		  vol_info->vol_data.status == S_SCANNING)))
	{
		if (vol_info->vol_data.raid_level > SINGLE &&
			(vol_info->vol_data.status == S_FORMATTING ||
			 vol_info->vol_data.status == S_SCANNING))
		{
			 // do nothing --> already assigned above
		}
		else
		if (is_lvm_volume) {
			PV_INFO pvinfo;
			LV_MOUNT_INFO lv_minfo;

			Get_MP_String(vol_info->vol_data.drive_no_list[0],
				DATA_PART,lv_minfo.c_LV_Mount_Point);

			ret = Get_LV_Mt_Info(&lv_minfo);
			if (ret) {
#ifdef DEBUG
				printf("\tGet_LV_Mt_Info(%s) fails, err=%d\n",
					lv_minfo.c_LV_Mount_Point, ret);
#endif
				return ret;
			}

			get_pv_info(vol_info->vol_data.device_name, &pvinfo);
			vol_info->total_size = pvinfo.ll_PV_Size;
			vol_info->free_size  = lv_minfo.ll_LV_Free_Size;
		}
		else {
			Get_Partition_Info(vol_data.device_name, &part_info);

			vol_info->total_size = part_info.total_size;
			vol_info->free_size  = part_info.free_size;
		}
		vol_info->resync_per = -1;

		if(vol_info->vol_data.raid_level == SINGLE) {// single volume
			int drvno = vol_info->vol_data.drive_no_list[0];

			if(!Is_Volume_HD_Exist(vol_no))
				vol_info->vol_data.status = S_NOT_EXIST;
			else if ((is_lvm_volume && !Is_Mounted(vol_info->vol_data.lv_name))
				|| (!is_lvm_volume && !Is_Mounted(vol_info->vol_data.device_name)))
			{
				if (vol_info->vol_data.status != S_UNINITIALIZE) {
					if (!Is_HD_Valid(drvno))
						vol_info->vol_data.status = S_INVALID;
					else if (Get_HD_RW_Error(drvno))
						vol_info->vol_data.status = S_RW_ERROR;
					else
						vol_info->vol_data.status = S_NOT_MOUNTED;
				}
			}
			else if (!Is_HD_Valid(drvno))
				vol_info->vol_data.status = S_INVALID;
			else if (Get_HD_RW_Error(drvno))
				vol_info->vol_data.status = S_RW_ERROR;
			else
				vol_info->vol_data.status = S_READY;
		}
		else {// raid volume
			RAID_STATUS sts_rd;

			ret = Get_One_RaidDev_Status(vol_data.device_name,&sts_rd);
			if(ret == SUCCESS) {
				// Catherine 2002/05/20 ==>
				int intsize=sizeof(int);
				int i, j;

				if(sts_rd.active_drive_cnt<vol_info->vol_data.list_cnt) {
					// no disk is spare
					memset(vol_info->vol_data.drive_no_list,
						0, MAX_DRIVE_NO*intsize);
					memcpy(vol_info->vol_data.drive_no_list,
						sts_rd.active_drive_no,
						sts_rd.active_drive_cnt*intsize);

					sort_drive_no_for_show(vol_info->vol_data.drive_no_list,
						sts_rd.active_drive_cnt);

					memset(vol_info->vol_data.spare_drive_list,
						0, MAX_DRIVE_NO*intsize);
				}
				else {
					memset(vol_info->vol_data.drive_no_list,
						0, MAX_DRIVE_NO*intsize);
					memcpy(vol_info->vol_data.drive_no_list,
						sts_rd.active_drive_no,
						vol_info->vol_data.list_cnt*intsize);
					sort_drive_no_for_show(vol_info->vol_data.drive_no_list,
						vol_info->vol_data.list_cnt);

					memset(vol_info->vol_data.spare_drive_list,
						0, MAX_DRIVE_NO*intsize);
					if (sts_rd.active_drive_cnt>vol_info->vol_data.list_cnt) {
						for(i=vol_info->vol_data.list_cnt, j=0;
							i<sts_rd.active_drive_cnt; i++, j++)
						{
							vol_info->vol_data.spare_drive_list[j]=sts_rd.active_drive_no[i];
						}
					}
				}

				if (vol_info->vol_data.status == S_FORMATTING ||
					vol_info->vol_data.status == S_SCANNING)
				{
					vol_info->snapshot_cnt = 0;
					vol_info->snap_list = NULL;
					return SUCCESS;
				}

				if(sts_rd.resync_per >= RAID_RESYNC_PER_BASE) {
					vol_info->resync_per = sts_rd.resync_per-RAID_RESYNC_PER_BASE;

					if ((is_lvm_volume && !Is_Mounted(vol_info->vol_data.lv_name))
						|| (!is_lvm_volume && !Is_Mounted(vol_info->vol_data.device_name)))
					{
						vol_info->vol_data.status=S_NOT_MOUNTED;
					}
					else
						vol_info->vol_data.status = S_READY;
				} else // <== Catherine 2002/06/11 for resync only
						
				if(sts_rd.resync_per > -1){
					vol_info->resync_per = sts_rd.resync_per;
					vol_info->vol_data.status=S_REBUILDING_RAID;
				}
				else if(sts_rd.active_drive_cnt==vol_info->vol_data.list_cnt-1){
					if(vol_info->vol_data.raid_level==MIRROR || vol_info->vol_data.raid_level==RAID5)
						vol_info->vol_data.status=S_DEGRADED;
					else if(vol_info->vol_data.raid_level==STRIPING)
						vol_info->vol_data.status=S_NOT_ACTIVE;
				}
				else if(sts_rd.active_drive_cnt<vol_info->vol_data.list_cnt-1)
					vol_info->vol_data.status=S_NOT_ACTIVE;
				else if((is_lvm_volume && !Is_Mounted(vol_info->vol_data.lv_name))
					|| (!is_lvm_volume && !Is_Mounted(vol_info->vol_data.device_name)))
					vol_info->vol_data.status=S_NOT_MOUNTED;
				else
					vol_info->vol_data.status = S_READY;
			}
			else if(ret == ERROR_NOT_FOUND)
				vol_info->vol_data.status = S_NOT_ACTIVE;
			else if (!Is_Mounted(vol_info->vol_data.lv_name))
				vol_info->vol_data.status = S_NOT_MOUNTED;
			else
				vol_info->vol_data.status = S_UNKNOW_ERROR;
		}
	}
	Set_Volume_Status(vol_no,vol_info->vol_data.status);

	if (is_lvm_volume && vol_info->vol_data.snapshot_enable) {
		ret = Get_Snapshot_Volumes_Conf_For_One_Volume(vol_no, &sn_list);
		if (ret<0) {
#ifdef DEBUG
			printf("\tGet_Snapshot_Volumes_Conf_For_One_Volume"
			"  fails, err=%d!\n", ret);
#endif
			vol_info->snapshot_cnt = 0;
			vol_info->snap_list = NULL;
			vol_info->snap_repository_use_rate = 0;
		}
		else {
			LV_INFO snapinfo;
			int i, j;
			long long used_size = 0;

			vol_info->snapshot_cnt = ret;
			vol_info->snap_list = sn_list;

			for (i=0; i<vol_info->snapshot_cnt; i++) {

				if (sn_list[i].b_Is_Scheduled_Snap) {
					char snaplv[LV_PATH_MAX_LEN];

					j=0;
					while(j < sn_list[i].i_Max_Instance_Size)
					{
						if (strlen(sn_list[i].c_Instances[j])==0) {
							j++;
							continue;
						}

						get_snap_instance_name_by_source_lv(
							vol_info->vol_data.lv_name,
							sn_list[i].c_Instances[j],
							snaplv, sizeof(snaplv));
						j++;

						ret = get_lv_info(snaplv, &snapinfo);
						if (ret<0) {
#ifdef DEBUG
							printf("\tget_lv_info(%s) fails, err=%d\n",
								snaplv, ret);
#endif
							continue;
						}
						used_size += snapinfo.ll_SnapLV_Used_Size;
					}
				}
				else {
					ret = get_lv_info(sn_list[i].c_Snapshot_Path,
						&snapinfo);
					if (ret<0) {
#ifdef DEBUG
						printf("\tget_lv_info(%s) fails, err=%d\n",
							sn_list[i].c_Snapshot_Path, ret);
#endif
						continue;
					}
					used_size += snapinfo.ll_SnapLV_Used_Size;
				}
			}
			if (vol_info->vol_data.snap_repository_size>0)
				vol_info->snap_repository_use_rate =
					used_size * 100 / vol_info->vol_data.snap_repository_size;
			else {
#ifdef DEBUG
				printf("\tsnap_repository_size == 0 ???\n");
#endif
				vol_info->snap_repository_use_rate = 0;
			}
		}
	}
	else {
		vol_info->snapshot_cnt = 0;
		vol_info->snap_list = NULL;
		vol_info->snap_repository_use_rate = 0;
	}
	return SUCCESS;
}

/**********************************************************************
** Get_All_LVM_Volumes_Info
** Description:	Get all volumes' status
** Output:	the count of all volumes
***********************************************************************/
int Get_All_LVM_Volumes_Info(LVM_VOLUME_INFO **vol_info_list_ptr)
{
	LVM_VOLUME_INFO vol_info;
	LVM_VOLUME_INFO *vol_info_list=NULL;
	void *ptr=NULL;
	int vol_cnt=0,i,ret;

#ifdef	STORAGE_FILE_LOCK
	repair_lvm_storage_conf_if_necessary();
#else
	FILE *fp;

	if((fp = fopen(STORAGE_CONF,"r")) == NULL) 
		Create_LVM_Storage_Conf();
	else {
		char tmp_line[80];
		strcpy(tmp_line,"");
		fgets(tmp_line,80,fp);
		fclose(fp);
		if(strlen(tmp_line)== 0)
			Restore_LVM_Storage_Conf(); 	
	}
#endif
	for (i=1; i<MAX_VOLUME_NO+1; i++) {
		ret = Get_One_LVM_Volume_Info(i, &vol_info);
		if (ret<0) {
#ifdef DEBUG
			printf("\tVolume %d not exist!\n", i);
#endif
			continue;
		}
		ptr = realloc((void*)vol_info_list,
			(vol_cnt+1)*sizeof(LVM_VOLUME_INFO));

		if (ptr==NULL) {
#ifdef DEBUG
			printf("\tOut of memory!\n");
#endif
			Release_LVM_Info_List(vol_info_list, vol_cnt);
			return ERROR_OUT_OF_MEMORY;
		}
		vol_info_list=(LVM_VOLUME_INFO*)ptr;

		memcpy((void*) &vol_info_list[vol_cnt],
			(void*) &vol_info, sizeof(LVM_VOLUME_INFO));
		vol_cnt++;
	}
	sort_lvm_volume_data_for_show(vol_info_list, vol_cnt);

	*vol_info_list_ptr=vol_info_list;
	return vol_cnt;
}

/**********************************************************************
** Add_LVM_Volume_Record
** Description:	Create a snapshot volume record
** Input:
**		vol_conf:	the volume's information
**				only raid_level, list_cnt,
**				drive_no_list, spare_list_cnt,
**				spare_drive_list are reliable
** Output:	total volume count or error code if fails......
**		ERROR_FAIL
***********************************************************************/
int Add_LVM_Volume_Record(LVM_VOLUME_CONF *vol_conf)
{
	int vol_cnt=0,i=0,j=0;
	LVM_VOLUME_CONF *vol_data_list=NULL;
	FILE *fp;
	time_t the_time;
#ifndef	STORAGE_FILE_LOCK
	int fd=-1;
#endif
	char buf[2*BUF_SIZE], tmpbuf[BUF_SIZE], tmp_str[10];
	void *ptr=NULL;

	vol_cnt = Get_All_LVM_Volumes_Conf(&vol_data_list);
	if(vol_cnt >=MAX_VOLUME_NO) {
		free(vol_data_list);
		return ERROR_OUT_OF_RANGE;
	}
	if(vol_cnt >=0) {
		ptr = realloc((void*)vol_data_list,
			(vol_cnt+1)*sizeof(LVM_VOLUME_CONF));
		if (ptr==NULL) {
			if (vol_data_list) free(vol_data_list);
			return ERROR_OUT_OF_MEMORY;
		}
		vol_data_list = (LVM_VOLUME_CONF *)ptr;

		memcpy((void*)(&vol_data_list[vol_cnt]),
			(void*)vol_conf,
			sizeof(LVM_VOLUME_CONF));

		// get the minimum volume number
		{
			int i,j,vol_no,flag;
			
			for (i=1; i<MAX_VOLUME_NO+1; i++) {
				flag=0;
				for (j=0; j<vol_cnt; j++) {
					if (i==vol_data_list[j].vol_no) {
						flag=1;
						break;
					}
				}
				if (flag==0) {
					vol_no=i;
					break;
				}
			}
			vol_data_list[vol_cnt].vol_no=vol_no;
			vol_conf->vol_no = vol_no; // update back
		}

		// get device name
		if (vol_conf->raid_level==SINGLE) {
			Get_Partition_Name(vol_conf->drive_no_list[0],
				DATA_PART, vol_conf->device_name);
		}

		// get logical volume name
		Get_LV_Name_For_Device(vol_conf->device_name,
			vol_conf->lv_name,
			LV_PATH_MAX_LEN);
		strcpy(vol_data_list[vol_cnt].lv_name, vol_conf->lv_name);

		vol_cnt++;
		sort_lvm_volume_data(vol_data_list,vol_cnt);

#ifdef	STORAGE_FILE_LOCK
		if (NAS_File_Lock(STORAGE_CONF, 'w') == 0) {
			free(vol_data_list);
			return ERROR_LOCK_FILE;
		}

		if((fp = fopen(STORAGE_CONF,"w+")) == NULL) {
			free(vol_data_list);
			NAS_File_Unlock(STORAGE_CONF);
			return ERROR_OPEN_FILE;
		}
#else
		if((fp = fopen(STORAGE_CONF,"w+")) != NULL)
			fclose(fp);
		else {
			free(vol_data_list);
			return ERROR_FAIL;
		}
		
		// open /etc/config/storage.conf for write
		if ((fd=open(STORAGE_CONF, O_CREAT | O_RDWR))<0) {
			free(vol_data_list);
			return ERROR_OPEN_FILE;
		}
		// exclusively lock file
		if (flock(fd, LOCK_EX)<0) {
			close(fd);
			free(vol_data_list);
			return ERROR_LOCK_FILE;
		}
#endif
		// write the new content into the config file
		for(i=0;i<vol_cnt;i++){
			memset(buf,0,BUF_SIZE*2);
			// volume header
			sprintf(tmpbuf, "[VOLUME %d]\n", vol_data_list[i].vol_no);
			strcpy(buf, tmpbuf);

			// logical volume name
			sprintf(tmpbuf, "lv name = %s\n", vol_data_list[i].lv_name);
			strcat(buf, tmpbuf);

			// device name
			sprintf(tmpbuf, "device name = %s\n", vol_data_list[i].device_name);
			strcat(buf, tmpbuf);

			// raid level
			sprintf(tmpbuf, "raid level = %d\n", vol_data_list[i].raid_level);
			strcat(buf, tmpbuf);

			// drive no
			strcpy(tmpbuf,"drive no = ");
			for(j=0;j<vol_data_list[i].list_cnt;j++) {
				sprintf(tmp_str,"%d,",vol_data_list[i].drive_no_list[j]);
				strcat(tmpbuf,tmp_str);
			}
			tmpbuf[strlen(tmpbuf)-1] = '\0';
			strcat(tmpbuf, "\n");
			strcat(buf, tmpbuf);

			// spare drive no
			strcpy(tmpbuf,"spare drive no = ");
			for(j=0;j<vol_data_list[i].spare_list_cnt;j++) {
				sprintf(tmp_str,"%d,",vol_data_list[i].spare_drive_list[j]);
				strcat(tmpbuf,tmp_str);
			}
			tmpbuf[strlen(tmpbuf)-1] = '\0';
			strcat(tmpbuf, "\n");
			strcat(buf, tmpbuf);

			// status
			sprintf(tmpbuf, "status = %d\n", vol_data_list[i].status);
			strcat(buf, tmpbuf);

			// record time
			(void) time(&the_time);
			sprintf(tmpbuf, "record_time = %s\n", ctime(&the_time));
			strcat(buf, tmpbuf);

			// enable snapshot
			sprintf(tmpbuf, "enable snapshot = %s\n",
				(vol_data_list[i].snapshot_enable) ? "TRUE" : "FALSE");
			strcat(buf, tmpbuf);

			// snap repository ratio
			sprintf(tmpbuf, "snap repository ratio = %d\n", vol_data_list[i].snap_ratio);
			strcat(buf, tmpbuf);
#ifdef	__EXT3_SNAP__
			// max snap count
			sprintf(tmpbuf, "max snap count = %d\n", vol_data_list[i].max_snap_count);
			strcat(buf, tmpbuf);
#endif
#ifdef	STORAGE_FILE_LOCK
			if (fputs(buf, fp)<0) {
				fclose(fp);
				free(vol_data_list);
				NAS_File_Unlock(STORAGE_CONF);
				return ERROR_WRITE_FILE;
			}
#else
			if (write(fd, buf, strlen(buf))<0) {
				flock(fd, LOCK_UN);
				close(fd);
				free(vol_data_list);
				return ERROR_WRITE_FILE;
			}
#endif
		}
		// unlock the file & close it
#ifdef	STORAGE_FILE_LOCK
		fclose(fp);
		NAS_File_Unlock(STORAGE_CONF);
#else
		flock(fd, LOCK_UN);
		close(fd);
#endif
		free(vol_data_list);
	}
	return vol_cnt;
}

/**********************************************************************
** Remove_LVM_Volume_Record
** Description:	remove a volume record from config file "storage.conf"
** Input:
**		vol_no:		the volume number
** Output:	Error code......
**		SUCCESS
**		ERROR_FAIL
***********************************************************************/
int Remove_LVM_Volume_Record(int vol_no)
{
	int ret=0,vol_cnt=0,i=0,j=0;
	char vol_section[20],tmp_str[10];	
	LVM_VOLUME_CONF *vol_data_list;
	FILE *fp;
#ifndef	STORAGE_FILE_LOCK
	int fd=-1;
#endif
	time_t the_time;
	char buf[2*BUF_SIZE], tmpbuf[BUF_SIZE];
	
	//remove volume record
	sprintf(vol_section,"VOLUME %d",vol_no);
	ret = Conf_Remove_Section(STORAGE_CONF,vol_section);
	if(ret<0) return ret;

	//sort volume record
	vol_cnt = Get_All_LVM_Volumes_Conf(&vol_data_list);
	if (vol_cnt<0)
		return ERROR_NOT_FOUND;
	if (vol_cnt==0)
		return SUCCESS;

	sort_lvm_volume_data(vol_data_list,vol_cnt);

#ifdef	STORAGE_FILE_LOCK
	if (NAS_File_Lock(STORAGE_CONF, 'w')==0) {
		free(vol_data_list);
		return ERROR_LOCK_FILE;
	}

	if ((fp = fopen(STORAGE_CONF, "w+")) == NULL) {
		free(vol_data_list);
		NAS_File_Unlock(STORAGE_CONF);
		return ERROR_OPEN_FILE;
	}
#else
	if((fp = fopen(STORAGE_CONF,"w+")) != NULL)
		fclose(fp);
	else {
		free(vol_data_list);
		return ERROR_FAIL;
	}

	// open /etc/config/storage.conf for write
	if ((fd=open(STORAGE_CONF, O_CREAT | O_RDWR))<0) {
		free(vol_data_list);
		return ERROR_OPEN_FILE;
	}
	// exclusively lock file
	if (flock(fd, LOCK_EX)<0) {
		close(fd);
		free(vol_data_list);
		return ERROR_LOCK_FILE;
	}
#endif
	// write the new content into the config file
	for(i=0;i<vol_cnt;i++){
		// volume header
		sprintf(tmpbuf, "[VOLUME %d]\n", vol_data_list[i].vol_no);
		strcpy(buf, tmpbuf);

		// logical volume name
		sprintf(tmpbuf, "lv name = %s\n", vol_data_list[i].lv_name);
		strcat(buf, tmpbuf);

		// device name
		sprintf(tmpbuf, "device name = %s\n", vol_data_list[i].device_name);
		strcat(buf, tmpbuf);

		// raid level
		sprintf(tmpbuf, "raid level = %d\n", vol_data_list[i].raid_level);
		strcat(buf, tmpbuf);
		// drive no
		strcpy(tmpbuf,"drive no = ");
		for(j=0;j<vol_data_list[i].list_cnt;j++) {
			sprintf(tmp_str,"%d,",vol_data_list[i].drive_no_list[j]);
			strcat(tmpbuf,tmp_str);
		}
		tmpbuf[strlen(tmpbuf)-1] = '\0';
		strcat(tmpbuf, "\n");
		strcat(buf, tmpbuf);

		// spare drive no
		strcpy(tmpbuf,"spare drive no = ");
		for(j=0;j<vol_data_list[i].spare_list_cnt;j++) {
			sprintf(tmp_str,"%d,",vol_data_list[i].spare_drive_list[j]);
			strcat(tmpbuf,tmp_str);
		}
		tmpbuf[strlen(tmpbuf)-1] = '\0';
		strcat(tmpbuf, "\n");
		strcat(buf, tmpbuf);

		// status
		sprintf(tmpbuf, "status = %d\n", vol_data_list[i].status);
		strcat(buf, tmpbuf);

		// record time
		(void) time(&the_time);
		sprintf(tmpbuf, "record_time = %s\n", ctime(&the_time));
		strcat(buf, tmpbuf);

		// enable snapshot
		sprintf(tmpbuf, "enable snapshot = %s\n",
			(vol_data_list[i].snapshot_enable) ? "TRUE" : "FALSE");
		strcat(buf, tmpbuf);

		// snap repository ratio
		sprintf(tmpbuf, "snap repository ratio = %d\n", vol_data_list[i].snap_ratio);
		strcat(buf, tmpbuf);
#ifdef	__EXT3_SNAP__
		// max snap count
		sprintf(tmpbuf, "max snap count = %d\n", vol_data_list[i].max_snap_count);
		strcat(buf, tmpbuf);
#endif
#ifdef	STORAGE_FILE_LOCK
		if (fputs(buf, fp)<0) {
			fclose(fp);
			free(vol_data_list);
			NAS_File_Unlock(STORAGE_CONF);
			return ERROR_WRITE_FILE;
		}
#else
		if (write(fd, buf, strlen(buf))<0) {
			flock(fd, LOCK_UN);
			close(fd);
			free(vol_data_list);
			return ERROR_WRITE_FILE;
		}
#endif
	}
	// unlock the file & close it
#ifdef	STORAGE_FILE_LOCK
	fclose(fp);
	NAS_File_Unlock(STORAGE_CONF);
#else
	flock(fd, LOCK_UN);
	close(fd);
#endif
	free(vol_data_list);
	return SUCCESS;
}

/**********************************************************************
** Release_LVM_Info_List
** Description:	Create a snapshot volume record
** Input:
**		vol_list:	the volume info list's pointer
**		vol_cnt:	the number of the list members
** Output:	Error code
**		SUCCESS
**		ERROR_FAIL
***********************************************************************/
int Release_LVM_Info_List(LVM_VOLUME_INFO* vol_list, int vol_cnt)
{
	int i;

	if (vol_list==NULL)
		return SUCCESS;

	for (i=0; i<vol_cnt; i++) {
		if (vol_list[i].snapshot_cnt>0 && vol_list[i].snap_list)
			free(vol_list[i].snap_list);
	}
	free(vol_list);
	return SUCCESS;
}
// --------------------- snapshot volumes ---------------------------
/**********************************************************************
** Add_Snapshot_Volume_Record
** Description:	Create a snapshot volume record
** Input:
**		src_vol:	the volume to map to
**		size:		the snapshot volume's size (in KB)
** Output:	total snapshot volume numbers or error code if fails......
**		ERROR_FAIL
***********************************************************************/
int Add_Snapshot_Volume_Record(SNAPSHOT_VOLUME_CONF *sn_vol_conf)
{
	int volcnt, ret;

	time(&(sn_vol_conf->t_Build_Time));
	sn_vol_conf->i_Status = S_UNINITIALIZE;
	memset(&sn_vol_conf->c_Instances, 0, MAX_SNAP_INSTANCE*SHARE_NAME_LENGTH*sizeof(char));
// Cat : for new snap repository size issue
//	sn_vol_conf->b_Is_Snap_PV_Allocated = FALSE;

	volcnt = do_add_snapshot_record(sn_vol_conf);
	if (volcnt<=0) return volcnt;

	if (sn_vol_conf->b_Is_Scheduled_Snap) {
		SNAPD_CONF_ENTRY snapdconf;

		snapdconf.snap_idx = sn_vol_conf->i_Snapshot_No;
		snapdconf.built_time = sn_vol_conf->t_Build_Time;
		snapdconf.start_time = sn_vol_conf->t_Start_Time;
		snapdconf.cycle = sn_vol_conf->Cycle;
		snapdconf.is_ok = FALSE;
		ret = Add_Snap_Daemon_Conf_Record(&snapdconf);
		if (ret<0) {
#ifdef DEBUG
			printf("Add_Snapshot_Volume_Record : "
				"Add_Snap_Daemon_Conf_Record failed, err=%d\n",
				ret);
#endif
			Remove_Snapshot_Volume_Record(sn_vol_conf->i_Snapshot_No);
			return ret;
		}
	}

	return volcnt;
}

/**********************************************************************
** Get_All_Snapshot_Volumes_Conf
** Description:	Get all snapshot volumes' information from config file "snapshot.conf"
** Input:
**		sn_vol_confs_ptr:	the output ptr to an new array of SNAPSHOT_VOLUME_CONF
** Output:	total snapshot volume numbers or error code if fails......
**		ERROR_FAIL
***********************************************************************/
int Get_All_Snapshot_Volumes_Conf(SNAPSHOT_VOLUME_CONF** sn_vol_confs_ptr)
{
	FILE* fp;
	SNAPSHOT_VOLUME_CONF *vol_data_list=NULL;
	SNAPSHOT_VOLUME_CONF *volptr;
	int cnt=0;
	int max_str_len = (MAX_SNAP_NO_PER_VOLUME*(SHARE_NAME_LENGTH + 2) > BUF_SIZE) ? 
		MAX_SNAP_NO_PER_VOLUME*(SHARE_NAME_LENGTH + 2) : BUF_SIZE;
	char line[max_str_len], value[max_str_len];

	if (NAS_File_Lock(SNAPSHOT_CONF, 'r')==0) {
#ifdef DEBUG
		printf("Get_All_Snapshot_Volumes_Conf : fails to NAS_File_Lock %s\n",
				SNAPSHOT_CONF);
#endif
		return ERROR_LOCK_FILE;
	}

    	fp = fopen(SNAPSHOT_CONF,"r");
    	if (fp==NULL)
	{
#ifdef	STORAGE_FILE_LOCK
		NAS_File_Unlock(SNAPSHOT_CONF);
#endif
		Restore_Snapshot_Conf();
#ifdef	STORAGE_FILE_LOCK
		if (NAS_File_Lock(SNAPSHOT_CONF, 'r') == 0)
			return ERROR_LOCK_FILE;
		if((fp = fopen(SNAPSHOT_CONF,"r")) == NULL) {
			NAS_File_Unlock(SNAPSHOT_CONF);
			return ERROR_OPEN_FILE;
		}
#else
		fp = fopen(SNAPSHOT_CONF,"r");
#endif
	}

    	while(fgets(line,sizeof(line),fp)!=NULL) {
		Get_String_Field(line,1,SPACE,value, sizeof(value));
		if(strcasecmp(value,"[snapshot") == 0 ) {
			char *p;

			Get_String_Field(line,2,SPACE,value,sizeof(value));

			p = value;
			while (*p!=']' && *p!='\0') p++;
			*p = '\0';

			volptr = realloc(vol_data_list, (cnt+1)*sizeof(SNAPSHOT_VOLUME_CONF));
			if (volptr==NULL) {
#ifdef DEBUG
				printf("Get_All_Snapshot_Volumes_Conf : "
					"fails to reallocate memory!\n");
#endif
				if (cnt>0 && vol_data_list)
					free(vol_data_list);
				fclose(fp);
#ifdef	STORAGE_FILE_LOCK
				NAS_File_Unlock(SNAPSHOT_CONF);
#endif
				return ERROR_OUT_OF_MEMORY;
			}
			vol_data_list = volptr;
			memset(&vol_data_list[cnt], 0, sizeof(SNAPSHOT_VOLUME_CONF));
			vol_data_list[cnt].i_Snapshot_No = atoi(value);
			cnt++;
			continue;	
		}

		Get_String_Field(line,1,'=',value,sizeof(value));

		if(strcasecmp(value,"device name") == 0 ) {
			Get_String_Field(line,2,'=',
				vol_data_list[cnt-1].c_Snapshot_Path,LV_PATH_MAX_LEN);
			Remove_Blank_From_String(vol_data_list[cnt-1].c_Snapshot_Path);
			continue;
		}

		if(strcasecmp(value,"source volume") == 0) {
			Get_String_Field(line,2,'=',
				vol_data_list[cnt-1].c_Src_LV,LV_PATH_MAX_LEN);
			Remove_Blank_From_String(vol_data_list[cnt-1].c_Src_LV);
			continue;
		}

		if(strcasecmp(value,"build time") == 0){
			Get_String_Field(line,2, '=', value, sizeof(value));
			vol_data_list[cnt-1].t_Build_Time = (time_t)atoi(value);
			continue;
		}
		if(strcasecmp(value,"priority") == 0){
			Get_String_Field(line,2, '=', value, sizeof(value));
			vol_data_list[cnt-1].Priority = (SNAP_PRIORITY)atoi(value);
			continue;
		}
		if(strcasecmp(value,"status") == 0){
			Get_String_Field(line,2, '=', value, sizeof(value));
			vol_data_list[cnt-1].i_Status = atoi(value);
			continue;
		}
		if(strcasecmp(value,"is scheduled") == 0){
			Get_String_Field(line,2, '=', value, sizeof(value));
			Remove_Blank_From_String(value);
			if (!strcasecmp(value, "TRUE"))
				vol_data_list[cnt-1].b_Is_Scheduled_Snap = TRUE;
			else
				vol_data_list[cnt-1].b_Is_Scheduled_Snap = FALSE;
			continue;
		}
		if(strcasecmp(value,"start time") == 0){
			Get_String_Field(line,2, '=', value, sizeof(value));
			vol_data_list[cnt-1].t_Start_Time = (time_t)atoi(value);
			continue;
		}

		if(strcasecmp(value,"priority") == 0){
			Get_String_Field(line,2, '=', value, sizeof(value));
			vol_data_list[cnt-1].Priority = (SNAP_PRIORITY) atoi(value);
			continue;
		}
		if(strcasecmp(value,"cycle") == 0){
			Get_String_Field(line,2, '=', value, sizeof(value));
			vol_data_list[cnt-1].Cycle = (SNAP_SCHEDULE_CYCLE)atoi(value);
			continue;
		}
		if(strcasecmp(value,"max instance size") == 0){
			Get_String_Field(line,2, '=', value, sizeof(value));
			vol_data_list[cnt-1].i_Max_Instance_Size = atoi(value);
			continue;
		}
		if(strcasecmp(value,"snap instances") == 0){
			int i=1;
			Get_String_Field(line, 2, '=', value, sizeof(value));

			while (i<=MAX_SNAP_INSTANCE &&
				Get_String_Field(value, i, ',', vol_data_list[cnt-1].c_Instances[i-1], SHARE_NAME_LENGTH)==SUCCESS)
			{
				Remove_Blank_From_String(vol_data_list[cnt-1].c_Instances[i-1]);
				i++;
			}
			continue;
		}

		if(strcasecmp(value,"snap restore status") == 0){
			int i=1;
			char tmp[10];
			Get_String_Field(line, 2, '=', value, sizeof(value));

			while (i<=MAX_SNAP_INSTANCE &&
				Get_String_Field(value, i, ',', tmp, 10)==SUCCESS)
			{
				vol_data_list[cnt-1].Restore_Status[i-1] = (SNAP_RESTORE_STATUS)atoi(tmp);
				i++;
			}
			continue;
		}

      	}
      	fclose(fp);
	NAS_File_Unlock(SNAPSHOT_CONF);

	sort_snapshot_volume_data_for_show(vol_data_list, cnt);
	*sn_vol_confs_ptr = vol_data_list;
	return cnt;	
}

/**********************************************************************
** Get_One_Snapshot_Volumes_Conf
** Description:	Get all snapshot volumes' information from config file "snapshot.conf"
** Input:
**		sn_vol_no:		the snapshot volume number
**		sn_vol_conf_ptr:	the ptr to an a SNAPSHOT_VOLUME_CONF
** Output:	Error code......
**		SUCCESS
**		ERROR_FAIL
***********************************************************************/
int Get_One_Snapshot_Volume_Conf(int sn_vol_no, SNAPSHOT_VOLUME_CONF* sn_vol_conf_ptr)
{
	char vol_section[20];
	int max_str_len = (MAX_SNAP_NO_PER_VOLUME*(SHARE_NAME_LENGTH + 2) > BUF_SIZE) ? 
		MAX_SNAP_NO_PER_VOLUME*(SHARE_NAME_LENGTH + 2) : BUF_SIZE;
	char tmp_buf[max_str_len];
	int ret;

	memset(sn_vol_conf_ptr, 0, sizeof(SNAPSHOT_VOLUME_CONF));
	sprintf(vol_section,"SNAPSHOT %d",sn_vol_no);

	//set vol no
	sn_vol_conf_ptr->i_Snapshot_No = sn_vol_no;
	// get device name
	ret = Conf_Get_Field(SNAPSHOT_CONF,
		vol_section,"device name",
		sn_vol_conf_ptr->c_Snapshot_Path,
		LV_PATH_MAX_LEN);
	if(ret<0) return ret;

	ret = Conf_Get_Field(SNAPSHOT_CONF,
		vol_section,"source volume",
		sn_vol_conf_ptr->c_Src_LV,
		LV_PATH_MAX_LEN);	
	if(ret<0) return ret;

	ret = Conf_Get_Field(SNAPSHOT_CONF,
		vol_section,"build time",
		tmp_buf,
		sizeof(tmp_buf));
	if(ret<0) return ret;
	sn_vol_conf_ptr->t_Build_Time = (time_t)atoi(tmp_buf);

	// priority
	ret = Conf_Get_Field(SNAPSHOT_CONF,
		vol_section,"priority",
		tmp_buf,
		sizeof(tmp_buf));
	if(ret<0) return ret;
	sn_vol_conf_ptr->Priority = (SNAP_PRIORITY)atoi(tmp_buf);

	ret = Conf_Get_Field(SNAPSHOT_CONF,
		vol_section,"status",
		tmp_buf,
		sizeof(tmp_buf));
	if(ret<0) return ret;
	sn_vol_conf_ptr->i_Status = atoi(tmp_buf);

	ret = Conf_Get_Field(SNAPSHOT_CONF,
		vol_section,"is scheduled",
		tmp_buf,
		sizeof(tmp_buf));
	if(ret<0) return ret;
	sn_vol_conf_ptr->b_Is_Scheduled_Snap = (strcasecmp(tmp_buf, "FALSE")) ? TRUE : FALSE;

	if (sn_vol_conf_ptr->b_Is_Scheduled_Snap == FALSE) {
		ret = Conf_Get_Field(SNAPSHOT_CONF,
			vol_section, "snap restore status",
			tmp_buf, 
			sizeof(tmp_buf));
		if (ret<0) return ret;
		sn_vol_conf_ptr->Restore_Status[0] = (SNAP_RESTORE_STATUS)atoi(tmp_buf);

		sn_vol_conf_ptr->t_Start_Time = (time_t)0;
		sn_vol_conf_ptr->Cycle = 0;
		sn_vol_conf_ptr->i_Max_Instance_Size = 0;
		memset(sn_vol_conf_ptr->c_Instances, 0, MAX_SNAP_INSTANCE*SHARE_NAME_LENGTH*sizeof(char));

		return SUCCESS;
	}

	ret = Conf_Get_Field(SNAPSHOT_CONF,
		vol_section,"start time",
		tmp_buf,
		sizeof(tmp_buf));
	if(ret<0) return ret;
	sn_vol_conf_ptr->t_Start_Time = (time_t)strtoul(tmp_buf, NULL, 10);

	ret = Conf_Get_Field(SNAPSHOT_CONF,
		vol_section,"priority",
		tmp_buf,
		sizeof(tmp_buf));
	if(ret<0) return ret;
	sn_vol_conf_ptr->Priority = (SNAP_PRIORITY) atoi(tmp_buf);

	ret = Conf_Get_Field(SNAPSHOT_CONF,
		vol_section,"cycle",
		tmp_buf,
		sizeof(tmp_buf));
	if(ret<0) return ret;
	sn_vol_conf_ptr->Cycle = (SNAP_SCHEDULE_CYCLE)atoi(tmp_buf);

	ret = Conf_Get_Field(SNAPSHOT_CONF,
		vol_section,"max instance size",
		tmp_buf,
		sizeof(tmp_buf));
	if(ret<0) return ret;
	sn_vol_conf_ptr->i_Max_Instance_Size = atoi(tmp_buf);

	
	ret = Conf_Get_Field(SNAPSHOT_CONF,
		vol_section, "snap instances",
		tmp_buf,
		sizeof(tmp_buf));
	if (ret==SUCCESS) {
		int i=1;
		while (i<=MAX_SNAP_INSTANCE &&
			Get_String_Field(tmp_buf, i, ',', sn_vol_conf_ptr->c_Instances[i-1],
			SHARE_NAME_LENGTH)==SUCCESS)
		{
			i++;
		}
	}

	ret = Conf_Get_Field(SNAPSHOT_CONF,
		vol_section, "snap restore status",
		tmp_buf, 
		sizeof(tmp_buf));
	if (ret==SUCCESS) {
		char tmp[10];
		int i=1;

		while (i<=MAX_SNAP_INSTANCE &&
			Get_String_Field(tmp_buf, i, ',', tmp,10)==SUCCESS)
		{
			sn_vol_conf_ptr->Restore_Status[i-1] = (SNAP_RESTORE_STATUS)atoi(tmp);
			i++;
		}
	}

	return SUCCESS;	
}

/**********************************************************************
** Remove_Snapshot_Volume_Record
** Description:	remove a snapshot volume record from config file "snapshot.conf"
** Input:
**		sn_vol_no:		the snapshot volume number
** Output:	Error code......
**		SUCCESS
**		ERROR_FAIL
***********************************************************************/
int Remove_Snapshot_Volume_Record(int sn_vol_no)
{
	int ret=0,vol_cnt=0,i=0;
	char vol_section[20];	
	SNAPSHOT_VOLUME_CONF *vol_data_list;
	FILE *fp;
#ifndef	STORAGE_FILE_LOCK
	int fd=-1;
#endif
	int max_str_len = (MAX_SNAP_NO_PER_VOLUME*(SHARE_NAME_LENGTH + 2) > BUF_SIZE) ? 
		MAX_SNAP_NO_PER_VOLUME*(SHARE_NAME_LENGTH + 2) : BUF_SIZE;
	char buf[2*max_str_len + BUF_SIZE], tmpbuf[max_str_len];

	//remove volume record
	sprintf(vol_section,"SNAPSHOT %d",sn_vol_no);
	ret = Conf_Remove_Section(SNAPSHOT_CONF,vol_section);
	if(ret<0) return ret;

	//sort volume record
	vol_cnt = Get_All_Snapshot_Volumes_Conf(&vol_data_list);
	if (vol_cnt<0)
		return ERROR_NOT_FOUND;
	if (vol_cnt==0)
		return SUCCESS;

	sort_snapshot_volume_data(vol_data_list,vol_cnt);

#ifdef	STORAGE_FILE_LOCK
	if (NAS_File_Lock(SNAPSHOT_CONF, 'w') == 0) {
		free(vol_data_list);
		return ERROR_LOCK_FILE;
	}
	if((fp = fopen(SNAPSHOT_CONF,"w+")) == NULL) {
		free(vol_data_list);
		NAS_File_Unlock(SNAPSHOT_CONF);
		return ERROR_OPEN_FILE;
	}
#else
	if((fp = fopen(SNAPSHOT_CONF,"w+")) != NULL)
		fclose(fp);
	else {
		free(vol_data_list);
		return ERROR_FAIL;
	}

	// open /etc/config/snapshot.conf for write
	if ((fd=open(SNAPSHOT_CONF, O_CREAT | O_RDWR))<0) {
		free(vol_data_list);
		return ERROR_OPEN_FILE;
	}
	// exclusively lock file
	if (flock(fd, LOCK_EX)<0) {
		close(fd);
		free(vol_data_list);
		return ERROR_LOCK_FILE;
	}
#endif
	// write the new content into the config file
	for(i=0;i<vol_cnt;i++){
		// volume header
		sprintf(tmpbuf, "[SNAPSHOT %d]\n", vol_data_list[i].i_Snapshot_No);
		strcpy(buf, tmpbuf);

		// device name
		sprintf(tmpbuf, "device name = %s\n", vol_data_list[i].c_Snapshot_Path);
		strcat(buf, tmpbuf);

		// source volume
		sprintf(tmpbuf, "source volume = %s\n", vol_data_list[i].c_Src_LV);
		strcat(buf, tmpbuf);

		// build time
		sprintf(tmpbuf, "build time = %u\n", (unsigned int)vol_data_list[i].t_Build_Time);
		strcat(buf, tmpbuf);

		// priority
		sprintf(tmpbuf, "priority = %u\n",
			(unsigned int)vol_data_list[i].Priority);
		strcat(buf, tmpbuf);

		// status
		sprintf(tmpbuf, "status = %d\n", vol_data_list[i].i_Status);
		strcat(buf, tmpbuf);

		// is scheduled?
		sprintf(tmpbuf, "is scheduled = %s\n",
			(vol_data_list[i].b_Is_Scheduled_Snap) ? "True" : "False");
		strcat(buf, tmpbuf);

		if (vol_data_list[i].b_Is_Scheduled_Snap==TRUE) {
			char tmpbuf2[max_str_len];
			int j=0;

			// start time
			sprintf(tmpbuf, "start time = %u\n",
				(unsigned int)vol_data_list[i].t_Start_Time);
			strcat(buf, tmpbuf);

			// priority
			sprintf(tmpbuf, "priority = %d\n", (int)vol_data_list[i].Priority);
			strcat(buf, tmpbuf);

			// cycle
			sprintf(tmpbuf, "cycle = %d\n",	(int)vol_data_list[i].Cycle);
			strcat(buf, tmpbuf);

			// max instance count
			sprintf(tmpbuf, "max instance size = %d\n",
				vol_data_list[i].i_Max_Instance_Size);
			strcat(buf, tmpbuf);

			// snap instances && snap restore status
			strcpy(tmpbuf, "snap instances = ");
			strcpy(tmpbuf2, "snap restore status = ");
			while (j<vol_data_list[i].i_Max_Instance_Size)
			{
				if (strlen(vol_data_list[i].c_Instances[j])==0) {
					j++;
					continue;
				}
				sprintf(tmpbuf, "%s%s,", tmpbuf, vol_data_list[i].c_Instances[j]);
				sprintf(tmpbuf2, "%s%d,", tmpbuf2, vol_data_list[i].Restore_Status[j]);
				j++;
			}
			j = strlen(tmpbuf);
			tmpbuf[j] = '\n';
			tmpbuf[j+1] = '\0';
			strcat(buf, tmpbuf);

			j = strlen(tmpbuf2);
			tmpbuf2[j] = '\n';
			tmpbuf2[j+1] = '\0';
			strcat(buf, tmpbuf2);
		}
		else {
			// snap restore status
			sprintf(tmpbuf, "snap restore status = %d\n",
				vol_data_list[i].Restore_Status[0]);
			strcat(buf, tmpbuf);
		}
#ifdef	STORAGE_FILE_LOCK
		if (fputs(buf, fp)<0) {
			fclose(fp);
			free(vol_data_list);
			NAS_File_Unlock(SNAPSHOT_CONF);
			return ERROR_WRITE_FILE;
		}
#else
		if (write(fd, buf, strlen(buf))<0) {
			flock(fd, LOCK_UN);
			close(fd);
			free(vol_data_list);
			return ERROR_WRITE_FILE;
		}
#endif
	}
	// unlock the file & close it
#ifdef	STORAGE_FILE_LOCK
	fclose(fp);
	NAS_File_Unlock(SNAPSHOT_CONF);
#else
	flock(fd, LOCK_UN);
	close(fd);
#endif
	free(vol_data_list);
	return SUCCESS;
}

/**********************************************************************
** Get_Snapshot_Volumes_Conf_For_One_Volume
** Description:	Get all snapshot volumes' for a normal volume
** Input:
**		vol_no:		the snapshot volume number
**		sn_vol_conf_ptr:	the ptr to an a SNAPSHOT_VOLUME_CONF
** Output:	# of mapping snapshot volumes or error code if fails......
**		ERROR_FAIL
**		ERROR_NOT_EXISTED
**		ERROR_OUT_OF_MEMORY
***********************************************************************/
int Get_Snapshot_Volumes_Conf_For_One_Volume(int vol_no,
		SNAPSHOT_VOLUME_CONF** sn_conf_list_ptr)
{
	int ret, cnt=0, i;
	char lv_name[LV_PATH_MAX_LEN];
	SNAPSHOT_VOLUME_CONF *all_sn_confs=NULL, *sn_confs=NULL, *ptr=NULL;

	Get_LV_Path(vol_no, lv_name, LV_PATH_MAX_LEN);

	ret = Get_All_Snapshot_Volumes_Conf(&all_sn_confs);
	if (ret<0)
		return ret;

	for (i=0; i<ret; i++) {
		if (!strcmp(lv_name,all_sn_confs[i].c_Src_LV)) {
			ptr=realloc((void*)sn_confs,
				(cnt+1)*sizeof(SNAPSHOT_VOLUME_CONF));
			if (ptr==NULL) {
				free(all_sn_confs);
				free(sn_confs);
				return ERROR_OUT_OF_MEMORY;
			}
			sn_confs=ptr;

			memcpy((void*)(&sn_confs[cnt]),
				(void*)(&all_sn_confs[i]),
				sizeof(SNAPSHOT_VOLUME_CONF));
			cnt++;
		}
	}
	sort_snapshot_volume_data_for_show(sn_confs, cnt);
	if (ret>0) free(all_sn_confs);
	*sn_conf_list_ptr=sn_confs;
	return cnt;
}

/**********************************************************************
** Get_Snapshot_Volumes_Conf_On_One_LV
** Description:	Get all snapshot volumes' config that use the space of a normal volume
** Input:
**		lv_name:		the normal volume LV path
**		sn_vol_conf_ptr:	the ptr to an a SNAPSHOT_VOLUME_CONF
** Output:	# of mapping snapshot volumes or error code if fails......
**		ERROR_FAIL
**		ERROR_NOT_EXISTED
**		ERROR_OUT_OF_MEMORY
***********************************************************************/
int Get_Snapshot_Volumes_Conf_On_One_LV(char *lv_name,
		SNAPSHOT_VOLUME_CONF** sn_conf_list_ptr)
{
	int ret, cnt=0, i;
	SNAPSHOT_VOLUME_CONF *all_sn_confs=NULL, *sn_confs=NULL, *ptr=NULL;

	ret = Get_All_Snapshot_Volumes_Conf(&all_sn_confs);
	if (ret<0) {
#ifdef DEBUG
		printf("Get_Snapshot_Volumes_Conf_On_One_LV(%s) : "
			"Get_All_Snapshot_Volumes_Conf fails, err=%d\n", lv_name, ret);
#endif
		return ret;
	}

	for (i=0; i<ret; i++) {
		if (!strcmp(lv_name,all_sn_confs[i].c_Src_LV)) {
			ptr=realloc((void*)sn_confs,
				(cnt+1)*sizeof(SNAPSHOT_VOLUME_CONF));
			if (ptr==NULL) {
#ifdef DEBUG
				printf("Get_Snapshot_Volumes_Conf_On_One_LV(%s) : "
					"fails to reallocate memory!\n", lv_name);
#endif
				free(all_sn_confs);
				free(sn_confs);
				return ERROR_OUT_OF_MEMORY;
			}
			sn_confs=ptr;

			memcpy((void*)(&sn_confs[cnt]),
				(void*)(&all_sn_confs[i]),
				sizeof(SNAPSHOT_VOLUME_CONF));
			cnt++;
		}
	}
	sort_snapshot_volume_data_for_show(sn_confs, cnt);
	if (ret>0) free(all_sn_confs);
	*sn_conf_list_ptr=sn_confs;
	return cnt;
}

/**********************************************************************
** Set_Snapshot_Status
** Description:	Set the status field of a snapshot volume config entry
** Input:
**		sn_vol_no:		the snapshot volume number
**		status:			the status code
** Output:	Error code if fails......
***********************************************************************/
int Set_Snapshot_Status(int sn_vol_no, int status) {
	char vol_section[20],tmp_str[10];
	int ret, retstatus, i=0;

	sprintf(vol_section,"SNAPSHOT %d",sn_vol_no);
	sprintf(tmp_str,"%d",status);

	ret = Conf_Set_Field(SNAPSHOT_CONF,vol_section,"status",tmp_str);
	retstatus = Get_Snapshot_Status(sn_vol_no);
	while (retstatus!=status && i++<100) {
		sleep(1);
		ret = Conf_Set_Field(SNAPSHOT_CONF,vol_section,"status",tmp_str);
		retstatus = Get_Snapshot_Status(sn_vol_no);
	}
	return ret;
}

/**********************************************************************
** Get_Snapshot_Status
** Description:	Set the status field of a snapshot volume config entry
** Input:
**		sn_vol_no:		the snapshot volume number
** Output:	Status code or error code if fails......
***********************************************************************/
int Get_Snapshot_Status(int sn_vol_no) {
	char vol_section[20],tmp_str[10],ret;

	sprintf(vol_section,"SNAPSHOT %d",sn_vol_no);
	
	ret = Conf_Get_Field(SNAPSHOT_CONF,vol_section,"status",tmp_str,10);

//	if(ret < 0) return ret;
	if(ret != 0)
		return ret;
	else
		return (atoi(tmp_str));
}

/**********************************************************************
** Set_Snapshot_Volume_Size
** Description:	Set the size field of a snapshot volume config entry
** Input:
**		sn_vol_no:		the snapshot volume number
**		size:			the size in KB
** Output:	Error code if fails......
***********************************************************************/
int Set_Snapshot_Volume_Size(int sn_vol_no, long long size)
{
	char vol_section[20],tmp_str[20], tmp_str2[20];
	int ret, i=0;
	long long retsize;

	sprintf(vol_section,"SNAPSHOT %d",sn_vol_no);
	sprintf(tmp_str,"%lld",size);
	ret = Conf_Set_Field(SNAPSHOT_CONF,vol_section,"size",tmp_str);

	memset(tmp_str2, 0, 20*sizeof(char));
	Conf_Get_Field(SNAPSHOT_CONF,vol_section,"size",tmp_str2,20);
	retsize = atoll(tmp_str2);

	while (retsize!=size && i++<100) {
		sleep(1);
		ret = Conf_Set_Field(SNAPSHOT_CONF,vol_section,"size",tmp_str);
		memset(tmp_str2, 0, 20*sizeof(char));
		Conf_Get_Field(SNAPSHOT_CONF,vol_section,"size",tmp_str2,20);
		retsize = atoll(tmp_str2);
	}
	return ret;
}

/**********************************************************************
** Get_All_Snapshot_Volumes_Info
** Description:	Get all snapshot volumes' information from config file "snapshot.conf"
** Input:
**		sn_vol_confs_ptr:	the output ptr to an new array of SNAPSHOT_VOLUME_CONF
** Output:	total snapshot volume numbers or error code if fails......
**		ERROR_FAIL
***********************************************************************/
int Get_All_Snapshot_Volumes_Info(SNAPSHOT_VOLUME_INFO** sn_vol_confs_ptr)
{
	int i,cnt=0;
	SNAPSHOT_VOLUME_INFO *vol_data_list=NULL;
	SNAPSHOT_VOLUME_CONF *snapconfs=NULL;

	cnt = Get_All_Snapshot_Volumes_Conf(&snapconfs);
	if (cnt<=0) goto get_all_snapshot_volumes_info_out;

	vol_data_list = calloc(cnt, sizeof(SNAPSHOT_VOLUME_INFO));
	for (i=0; i< cnt; i++) {
		Get_One_Snapshot_Volume_Info(
			snapconfs[i].i_Snapshot_No,
			&vol_data_list[i]);
	}

get_all_snapshot_volumes_info_out:
	sort_snapshot_volume_info_for_show(vol_data_list, cnt);
	if (cnt>0) free(snapconfs);

	*sn_vol_confs_ptr = vol_data_list;
	return cnt;	
}

/**********************************************************************
** Get_One_Snapshot_Volumes_Info
** Description:	Get all snapshot volumes' information from config file "snapshot.conf"
** Input:
**		sn_vol_no:		the snapshot volume number
**		sn_vol_conf_ptr:	the ptr to an a SNAPSHOT_VOLUME_CONF
** Output:	Error code......
**		SUCCESS
**		ERROR_FAIL
***********************************************************************/
int Get_One_Snapshot_Volume_Info(int sn_vol_no, SNAPSHOT_VOLUME_INFO* sn_vol_conf_ptr)
{
	int ret;
	float total_use_rate=0;
	LV_INFO lvinfo;

	if (sn_vol_conf_ptr==NULL)
		return ERROR_INVALID_PARAM;

	memset(sn_vol_conf_ptr, 0, sizeof(SNAPSHOT_VOLUME_INFO));

	ret = Get_One_Snapshot_Volume_Conf(sn_vol_no, &(sn_vol_conf_ptr->snap_conf));
	if (ret<0) return ret;

	if (sn_vol_conf_ptr->snap_conf.b_Is_Scheduled_Snap) {
		int i=0, instance_cnt=0;
		SNAPSHOT_VOLUME_INSTANCE_INFO *snap_inst_list=NULL, *p;
		SNAPSHOT_VOLUME_INSTANCE_INFO snapinfo;

		sn_vol_conf_ptr->ll_Snapshot_Size = 0;

		while(i<MAX_SNAP_INSTANCE) {
			if (strlen(sn_vol_conf_ptr->snap_conf.c_Instances[i])==0) {
				i++;
				continue;
			}

			get_snap_instance_name_by_source_lv(sn_vol_conf_ptr->snap_conf.c_Src_LV,
				sn_vol_conf_ptr->snap_conf.c_Instances[i++],
				snapinfo.c_Snap_Name, sizeof(snapinfo.c_Snap_Name));

			snapinfo.i_Snap_Use_Rate = Get_Snapshot_Volume_Use_Rate(snapinfo.c_Snap_Name);

			if (snapinfo.i_Snap_Use_Rate == ERROR_NOT_EXISTED) {
#ifdef DEBUG
				printf("\tSnapshot %s use rate not exist??\n", snapinfo.c_Snap_Name);
#endif
				continue;
			}
			snapinfo.t_Build_Time = get_build_time_by_snap_instance_name(
							sn_vol_conf_ptr->snap_conf.c_Snapshot_Path,
							snapinfo.c_Snap_Name);

			sn_vol_conf_ptr->i_Restore_Percent = -1;
			if (snapinfo.i_Snap_Use_Rate>=100) {
				Umount_Snapshot_Volume(snapinfo.c_Snap_Name);
				snapinfo.i_Status = S_INVALID;
			}
			else if (sn_vol_conf_ptr->snap_conf.Restore_Status[i]==s_BUSY) {
				snapinfo.i_Status = S_RESTORING_SNAPSHOT;
				sn_vol_conf_ptr->i_Restore_Percent = get_restore_progress(snapinfo.c_Snap_Name);
			}
			else if (!Is_Mounted(snapinfo.c_Snap_Name) &&
				Mount_Snapshot_Volume(snapinfo.c_Snap_Name))
				snapinfo.i_Status = S_ABORTED;	// Catherine 2003/04/04
			else
				snapinfo.i_Status = S_READY;

			ret = get_lv_info(snapinfo.c_Snap_Name, &lvinfo);
			if (ret<0) {
#ifdef DEBUG
				printf("\ttrouble get_lv_info(%s) : %d\n", snapinfo.c_Snap_Name, ret);
#endif
				continue;
			}
			snapinfo.ll_Snap_Size = lvinfo.ll_LV_Size;

			p = realloc(snap_inst_list, (instance_cnt+1)*sizeof(SNAPSHOT_VOLUME_INSTANCE_INFO));
			if (p==NULL) {
				if (snap_inst_list) free(snap_inst_list);

				return ERROR_OUT_OF_MEMORY;
			}
			snap_inst_list = p;
			if (snapinfo.i_Snap_Use_Rate>=0) total_use_rate += snapinfo.i_Snap_Use_Rate;

			memcpy(&snap_inst_list[instance_cnt], &snapinfo, sizeof(SNAPSHOT_VOLUME_INSTANCE_INFO));
			instance_cnt++;
			sn_vol_conf_ptr->ll_Snapshot_Size += snapinfo.ll_Snap_Size;
		}

		sort_instances_in_a_snap_volume(snap_inst_list, instance_cnt);

		sn_vol_conf_ptr->i_Current_Instance_Count = instance_cnt;
		sn_vol_conf_ptr->snap_instances = snap_inst_list;
		if (instance_cnt==0) {
			sn_vol_conf_ptr->i_Average_Snap_Use_Rate = 0;
			sn_vol_conf_ptr->snap_conf.i_Status = S_UNINITIALIZE;
		}
		else {
			sn_vol_conf_ptr->i_Average_Snap_Use_Rate = (int)(total_use_rate/instance_cnt);
			sn_vol_conf_ptr->snap_conf.i_Status = S_READY;
		}
	}
	else { // not schedule
		ret = get_lv_info(sn_vol_conf_ptr->snap_conf.c_Snapshot_Path, &lvinfo);
		if (ret==ERROR_NOT_EXISTED) {// not initialized
			sn_vol_conf_ptr->ll_Snapshot_Size = 0;
			sn_vol_conf_ptr->i_Average_Snap_Use_Rate = 0;
			sn_vol_conf_ptr->i_Restore_Percent = -1;
			sn_vol_conf_ptr->snap_conf.i_Status = S_UNINITIALIZE;
			goto get_one_snapshot_volume_info_out;
		}
		if (ret<0) {
#ifdef DEBUG
			printf("\ttrouble get_lv_info(%s) : %d\n",
				sn_vol_conf_ptr->snap_conf.c_Snapshot_Path, ret);
#endif
			return ERROR_FAIL;
		}
		sn_vol_conf_ptr->ll_Snapshot_Size = lvinfo.ll_LV_Size;

		ret = Get_Snapshot_Volume_Use_Rate(sn_vol_conf_ptr->snap_conf.c_Snapshot_Path);
		sn_vol_conf_ptr->i_Average_Snap_Use_Rate = ret;

		sn_vol_conf_ptr->i_Restore_Percent = -1;
		if (ret>=100) {
			Umount_Snapshot_Volume(sn_vol_conf_ptr->snap_conf.c_Snapshot_Path);
			sn_vol_conf_ptr->snap_conf.i_Status = S_INVALID;
			goto get_one_snapshot_volume_info_out;
		}
		else if (sn_vol_conf_ptr->snap_conf.Restore_Status[0]==s_BUSY) {
			sn_vol_conf_ptr->snap_conf.i_Status = S_RESTORING_SNAPSHOT;
			sn_vol_conf_ptr->i_Restore_Percent = get_restore_progress(sn_vol_conf_ptr->snap_conf.c_Snapshot_Path);
			goto get_one_snapshot_volume_info_out;
		}

		switch (sn_vol_conf_ptr->snap_conf.i_Status) {
			case S_READY:
				if (!Is_Mounted(sn_vol_conf_ptr->snap_conf.c_Snapshot_Path)) {
					sn_vol_conf_ptr->snap_conf.i_Status = S_ABORTED;	// Catherine 2003/04/04
				}
				break;
			case S_NOT_MOUNTED:
			case S_ABORTED:	// Catherine 2003/04/04
				if (Is_Mounted(sn_vol_conf_ptr->snap_conf.c_Snapshot_Path)) {
					sn_vol_conf_ptr->snap_conf.i_Status = S_READY;
				}
				else if (Mount_Snapshot_Volume(sn_vol_conf_ptr->snap_conf.c_Snapshot_Path)==SUCCESS) {
					sn_vol_conf_ptr->snap_conf.i_Status = S_READY;
				}
				sn_vol_conf_ptr->snap_conf.i_Status = S_ABORTED;	// Catherine 2003/04/04
				break;
			default:
				;// do nothing
		}
	}

get_one_snapshot_volume_info_out:	
	Set_Snapshot_Status(sn_vol_no, sn_vol_conf_ptr->snap_conf.i_Status);

	return SUCCESS;	
}

/**********************************************************************
** Add_Snap_Daemon_Conf_Record
** Description:	Add a new record to SNAPD_CONFIG
** Input:
**		entry:	the ptr to an a SNAPD_CONF_ENTRY
** Output:	Error code......
**		SUCCESS
**		ERROR_FAIL
***********************************************************************/
int Add_Snap_Daemon_Conf_Record(SNAPD_CONF_ENTRY* entry)
{
	int snapd_entry_cnt=0,i=0;
	SNAPD_CONF_ENTRY *snapd_data_list=NULL;
	FILE *fp;
#ifndef	STORAGE_FILE_LOCK
	int fd=-1;
#endif
	char buf[2*BUF_SIZE], tmpbuf[BUF_SIZE];
	void *ptr=NULL;

	snapd_entry_cnt = Get_All_Snap_Daemon_Conf_Records(&snapd_data_list);
	
	if(snapd_entry_cnt >=MAX_SNAP_NO) {
		free(snapd_data_list);
		return ERROR_OUT_OF_RANGE;
	}

	if(snapd_entry_cnt >=0) {
		ptr = realloc((void*)snapd_data_list,
			(snapd_entry_cnt+1)*sizeof(SNAPD_CONF_ENTRY));

		if (ptr==NULL) {
			if (snapd_data_list) free(snapd_data_list);
			return ERROR_OUT_OF_MEMORY;
		}
		snapd_data_list = (SNAPD_CONF_ENTRY *)ptr;

		memcpy((void*)(&snapd_data_list[snapd_entry_cnt]),
			(void*)entry,
			sizeof(SNAPD_CONF_ENTRY));

		snapd_entry_cnt++;
		sort_snapd_conf_records(snapd_data_list,snapd_entry_cnt);

#ifdef	STORAGE_FILE_LOCK
		if (NAS_File_Lock(SNAPD_CONFIG, 'w') == 0) {
			free(snapd_data_list);
			return ERROR_LOCK_FILE;
		}
		if((fp = fopen(SNAPD_CONFIG,"w+")) == NULL) {
			free(snapd_data_list);
			NAS_File_Unlock(SNAPD_CONFIG);
			return ERROR_OPEN_FILE;
		}
#else
		if((fp = fopen(SNAPD_CONFIG, "w+")) != NULL)
			fclose(fp);
		else {
			free(snapd_data_list);
			return ERROR_FAIL;
		}
		
		// open /etc/config/storage.conf for write
		if ((fd=open(SNAPD_CONFIG, O_CREAT | O_RDWR))<0) {
			free(snapd_data_list);
			return ERROR_OPEN_FILE;
		}
		// exclusively lock file
		if (flock(fd, LOCK_EX)<0) {
			close(fd);
			free(snapd_data_list);
			return ERROR_LOCK_FILE;
		}
#endif
		// write the new content into the config file
		for(i=0;i<snapd_entry_cnt;i++){
			strcpy(buf, "");

			// snapshot number
			sprintf(tmpbuf, "%d\t",
				snapd_data_list[i].snap_idx);
			strcpy(buf, tmpbuf);

			// start time
			sprintf(tmpbuf, "%u\t",
				(unsigned int)snapd_data_list[i].start_time);
			strcat(buf, tmpbuf);

			// cycle
			sprintf(tmpbuf, "%u\t",
				(unsigned int)snapd_data_list[i].cycle);
			strcat(buf, tmpbuf);

			// is OK
			sprintf(tmpbuf, "%s\t",
				(snapd_data_list[i].is_ok) ? "True" : "False");
			strcat(buf, tmpbuf);

			// built time
			sprintf(tmpbuf, "%u\n",
				(unsigned int)snapd_data_list[i].built_time);
			strcat(buf, tmpbuf);
#ifdef	STORAGE_FILE_LOCK
			if (fputs(buf, fp)<0) {
				fclose(fp);
				free(snapd_data_list);
				NAS_File_Unlock(SNAPD_CONFIG);
				return ERROR_WRITE_FILE;
			}
#else
			if (write(fd, buf, strlen(buf))<0) {
				flock(fd, LOCK_UN);
				close(fd);
				free(snapd_data_list);
				return ERROR_WRITE_FILE;
			}
#endif
		}
		// unlock the file & close it
#ifdef	STORAGE_FILE_LOCK
		fclose(fp);
		NAS_File_Unlock(SNAPD_CONFIG);
#else
		flock(fd, LOCK_UN);
		close(fd);
#endif
		free(snapd_data_list);
		return SUCCESS;
	}

	return snapd_entry_cnt;
}

/**********************************************************************
** Get_All_Snap_Daemon_Conf_Records
** Description:	Retrieve all scheduled daemon config records from SNAPD_CONF
** Input:
**		record_list:	the ptr to an array of an a SNAPD_CONF_ENTRY
** Output:	Error code......
**		SUCCESS
**		ERROR_FAIL
***********************************************************************/
int Get_All_Snap_Daemon_Conf_Records(SNAPD_CONF_ENTRY** record_list)
{
	int count=0;
	SNAPD_CONF_ENTRY *snapd_list=NULL, *p;
	FILE* fp=NULL;
	char linebuf[BUF_SIZE], fieldbuf[BUF_SIZE];

	if (record_list==NULL)
		return ERROR_BUFFER_TOO_SMALL;

#ifdef	STORAGE_FILE_LOCK
	if (NAS_File_Lock(SNAPD_CONFIG, 'r') == 0)
		return ERROR_LOCK_FILE;

	if((fp = fopen(SNAPD_CONFIG,"r")) == NULL) {
		if ((fp = fopen(SNAPD_CONFIG, "a+")) == NULL) {
			NAS_File_Unlock(SNAPD_CONFIG);
			return ERROR_OPEN_FILE;
		}
		else {
			*record_list = NULL;
			fclose(fp);
			return 0;
		}
	}
#else
	if ((fp = fopen(SNAPD_CONFIG, "r")) == NULL) {
		// config file doesn't exist
		if ((fp = fopen(SNAPD_CONFIG, "a+")) == NULL) {
#ifdef DEBUG
			printf("Get_All_Snap_Daemon_Conf_Records : fail to open config file (%s)\n",
				SNAPD_CONFIG);
#endif
			return ERROR_OPEN_FILE;
		}
		else {
			*record_list = NULL;
			fclose(fp);
			return 0;
		}
	}
#endif
	while (fgets(linebuf, BUF_SIZE, fp)!=NULL) {
		int snap_index;

		Get_String_Field(linebuf, 1, '\t', fieldbuf, BUF_SIZE);
		if ((snap_index = atoi(fieldbuf))<0) continue;

		p = realloc(snapd_list, (count+1)*sizeof(SNAPD_CONF_ENTRY));
		if (p==NULL) {
			if (snapd_list) free(snapd_list);
			snapd_list = NULL;
			count = ERROR_OUT_OF_MEMORY;

			goto get_all_snap_daemon_conf_records_out;
		}
		snapd_list = p;
		snapd_list[count].snap_idx = snap_index;

		Get_String_Field(linebuf, 2, '\t', fieldbuf, BUF_SIZE);
		snapd_list[count].start_time = (time_t) strtoul(fieldbuf, NULL, 10);

		Get_String_Field(linebuf, 3, '\t', fieldbuf, BUF_SIZE);
		snapd_list[count].cycle = (SNAP_SCHEDULE_CYCLE) atoi(fieldbuf);

		Get_String_Field(linebuf, 4, '\t', fieldbuf, BUF_SIZE);
		snapd_list[count].is_ok = (strcasecmp(fieldbuf, "True")==0) ? TRUE : FALSE;

		Get_String_Field(linebuf, 5, '\t', fieldbuf, BUF_SIZE);
		snapd_list[count].built_time = (time_t) strtoul(fieldbuf, NULL, 10);

		count++;
	}

get_all_snap_daemon_conf_records_out:
	fclose(fp);
#ifdef	STORAGE_FILE_LOCK
	NAS_File_Unlock(SNAPD_CONFIG);
#endif
	*record_list = snapd_list;
	return count;
}

/**********************************************************************
** 
** Description:	
** Input:
**		entry:	the ptr to an a SNAPD_CONF_ENTRY
** Output:	Error code......
**		SUCCESS
**		ERROR_FAIL
***********************************************************************/
//int Set_One_Snap_Daemon_Conf_Status(int snap_index, BOOL Is_OK, int try_count)
int Set_One_Snap_Daemon_Conf_Status(int snap_index, BOOL Is_OK)
{
	int snapd_entry_cnt=0,i=0;
	SNAPD_CONF_ENTRY *snapd_data_list=NULL;
	FILE *fp;
#ifndef	STORAGE_FILE_LOCK
	int fd=-1;
#endif
	char buf[2*BUF_SIZE], tmpbuf[BUF_SIZE];

	snapd_entry_cnt = Get_All_Snap_Daemon_Conf_Records(&snapd_data_list);
	
	if(snapd_entry_cnt >=0) {
#ifdef	STORAGE_FILE_LOCK
		if (NAS_File_Lock(SNAPD_CONFIG, 'w') == 0) {
			free(snapd_data_list);
			return ERROR_LOCK_FILE;
		}
		if((fp = fopen(SNAPD_CONFIG,"w+")) == NULL) {
			free(snapd_data_list);
			NAS_File_Unlock(SNAPD_CONFIG);
			return ERROR_OPEN_FILE;
		}
#else
		if((fp = fopen(SNAPD_CONFIG, "w+")) != NULL)
			fclose(fp);
		else {
			free(snapd_data_list);
			return ERROR_FAIL;
		}
		
		// open SNAPD_CONFIG for write
		if ((fd=open(SNAPD_CONFIG, O_CREAT | O_RDWR))<0) {
			free(snapd_data_list);
			return ERROR_OPEN_FILE;
		}
		// exclusively lock file
		if (flock(fd, LOCK_EX)<0) {
			close(fd);
			free(snapd_data_list);
			return ERROR_LOCK_FILE;
		}
#endif
		// write the new content into the config file
		for(i=0;i<snapd_entry_cnt;i++){
			strcpy(buf, "");

			// snapshot number
			sprintf(tmpbuf, "%d\t",
				snapd_data_list[i].snap_idx);
			strcpy(buf, tmpbuf);
			
			// start time
			sprintf(tmpbuf, "%u\t",
				(unsigned int)snapd_data_list[i].start_time);
			strcat(buf, tmpbuf);

			// cycle
			sprintf(tmpbuf, "%u\t",
				(unsigned int)snapd_data_list[i].cycle);
			strcat(buf, tmpbuf);

			// is OK && try count
			if (snap_index == snapd_data_list[i].snap_idx) {
				sprintf(tmpbuf, "%s\t",
					(Is_OK) ? "True" : "False");
				strcat(buf, tmpbuf);

			}
			else {
				sprintf(tmpbuf, "%s\t",
					(snapd_data_list[i].is_ok) ? "True" : "False");
				strcat(buf, tmpbuf);

			}

			// built time
			sprintf(tmpbuf, "%u\n",
				(unsigned int)snapd_data_list[i].built_time);
			strcat(buf, tmpbuf);

#ifdef	STORAGE_FILE_LOCK
			if (fputs(buf, fp)<0) {
				fclose(fp);
				free(snapd_data_list);
				NAS_File_Unlock(SNAPD_CONFIG);
				return ERROR_WRITE_FILE;
			}
#else
			if (write(fd, buf, strlen(buf))<0) {
				flock(fd, LOCK_UN);
				close(fd);
				free(snapd_data_list);
				return ERROR_WRITE_FILE;
			}
#endif
		}
		// unlock the file & close it
#ifdef	STORAGE_FILE_LOCK
		fclose(fp);
		NAS_File_Unlock(SNAPD_CONFIG);
#else
		flock(fd, LOCK_UN);
		close(fd);
#endif
		free(snapd_data_list);
		return SUCCESS;
	}

	return snapd_entry_cnt;
}

/**********************************************************************
** Get_One_Snap_Daemon_Conf_Record
** Description:	Retrieve the snapshot entry in SNAPD_CONFIG
** Input:
**		snap_index: the snapshot's volume number
**		entry:	the ptr of the ptr to an a SNAPD_CONF_ENTRY
** Output:	Error code......
**		SUCCESS
**		ERROR_FAIL
***********************************************************************/
int Get_One_Snap_Daemon_Conf_Record(int snap_index, SNAPD_CONF_ENTRY* record_ptr)
{
	FILE* fp=NULL;
	int ret = ERROR_NOT_FOUND, found_flag=0;
	char linebuf[BUF_SIZE], fieldbuf[BUF_SIZE];

	if (record_ptr==NULL)
		return ERROR_BUFFER_TOO_SMALL;

#ifdef	STORAGE_FILE_LOCK
	if (NAS_File_Lock(SNAPD_CONFIG, 'r') == 0)
		return ERROR_LOCK_FILE;

	if((fp = fopen(SNAPD_CONFIG,"r+")) == NULL) {
		NAS_File_Unlock(SNAPD_CONFIG);
		return ERROR_OPEN_FILE;
	}
#else
	if ((fp = fopen(SNAPD_CONFIG, "r+")) == NULL) {
#ifdef DEBUG
		fprintf(stderr, "snapd: fail to open config file (%s)\n",
			SNAPD_CONFIG);
#endif
		return ERROR_OPEN_FILE;
	}
#endif
	while (fgets(linebuf, BUF_SIZE, fp)!=NULL) {
		if (Get_String_Field(linebuf, 1, '\t', fieldbuf, BUF_SIZE)==SUCCESS &&
			snap_index == atoi(fieldbuf))
		{
			found_flag++;
			break;
		}
	}

	if (found_flag) {
		record_ptr->snap_idx = snap_index;

		Get_String_Field(linebuf, 2, '\t', fieldbuf, BUF_SIZE);
		record_ptr->start_time = (time_t) strtoul(fieldbuf, NULL, 10);

		Get_String_Field(linebuf, 3, '\t', fieldbuf, BUF_SIZE);
		record_ptr->cycle = (SNAP_SCHEDULE_CYCLE) atoi(fieldbuf);

		Get_String_Field(linebuf, 4, '\t', fieldbuf, BUF_SIZE);
		record_ptr->is_ok = (strcasecmp(fieldbuf, "True")==0) ? TRUE : FALSE;

		Get_String_Field(linebuf, 5, '\t', fieldbuf, BUF_SIZE);
		record_ptr->built_time = (time_t) strtoul(fieldbuf, NULL, 10);

		ret = SUCCESS;
	}

	fclose(fp);
#ifdef	STORAGE_FILE_LOCK
	NAS_File_Unlock(SNAPD_CONFIG);
#endif
	return ret;
}

/**********************************************************************
** Remove_One_Snap_Daemon_Conf_Record
** Description:	Remove the snapshot entry in SNAPD_CONFIG
** Input:
**		entry:	the ptr to an a SNAPD_CONF_ENTRY
** Output:	Error code......
**		SUCCESS
**		ERROR_FAIL
***********************************************************************/
int Remove_One_Snap_Daemon_Conf_Record(int snap_index)
{
	int snapd_entry_cnt=0,i=0, found_flag = 0;
	SNAPD_CONF_ENTRY *snapd_data_list=NULL;
	FILE *fp;
#ifndef	STORAGE_FILE_LOCK
	int fd=-1;
#endif
	char buf[2*BUF_SIZE], tmpbuf[BUF_SIZE];

	snapd_entry_cnt = Get_All_Snap_Daemon_Conf_Records(&snapd_data_list);

	if(snapd_entry_cnt >=0) {
#ifdef	STORAGE_FILE_LOCK
		if (NAS_File_Lock(SNAPD_CONFIG, 'w') == 0) {
			free(snapd_data_list);
			return ERROR_LOCK_FILE;
		}
		if((fp = fopen(SNAPD_CONFIG,"w+")) == NULL) {
			free(snapd_data_list);
			NAS_File_Unlock(SNAPD_CONFIG);
			return ERROR_OPEN_FILE;
		}
#else
		if((fp = fopen(SNAPD_CONFIG, "w+")) != NULL)
			fclose(fp);
		else {
			free(snapd_data_list);
			return ERROR_FAIL;
		}
		
		// open /etc/config/storage.conf for write
		if ((fd=open(SNAPD_CONFIG, O_CREAT | O_RDWR))<0) {
			free(snapd_data_list);
			return ERROR_OPEN_FILE;
		}
		// exclusively lock file
		if (flock(fd, LOCK_EX)<0) {
			close(fd);
			free(snapd_data_list);
			return ERROR_LOCK_FILE;
		}
#endif
		// write the new content into the config file
		for(i=0;i<snapd_entry_cnt;i++){
			if (snap_index == snapd_data_list[i].snap_idx) {
				found_flag++;
				continue;
			}
			strcpy(buf, "");

			// snapshot number
			sprintf(tmpbuf, "%d\t",
				snapd_data_list[i].snap_idx);
			strcpy(buf, tmpbuf);
			
			// start time
			sprintf(tmpbuf, "%u\t",
				(unsigned int)snapd_data_list[i].start_time);
			strcat(buf, tmpbuf);

			// cycle
			sprintf(tmpbuf, "%u\t",
				(unsigned int)snapd_data_list[i].cycle);
			strcat(buf, tmpbuf);

			// is OK
			sprintf(tmpbuf, "%s\t",
				(snapd_data_list[i].is_ok) ? "True" : "False");
			strcat(buf, tmpbuf);

			// built time
			sprintf(tmpbuf, "%u\n",
				(unsigned int)snapd_data_list[i].built_time);
			strcat(buf, tmpbuf);
#ifdef	STORAGE_FILE_LOCK
			if (fputs(buf, fp)<0) {
				free(snapd_data_list);
				fclose(fp);
				NAS_File_Unlock(SNAPD_CONFIG);
				return ERROR_WRITE_FILE;
			}
#else
			if (write(fd, buf, strlen(buf))<0) {
				flock(fd, LOCK_UN);
				close(fd);
				free(snapd_data_list);
				return ERROR_WRITE_FILE;
			}
#endif
		}
		// unlock the file & close it
#ifdef	STORAGE_FILE_LOCK
		fclose(fp);
		NAS_File_Unlock(SNAPD_CONFIG);
#else
		flock(fd, LOCK_UN);
		close(fd);
#endif
		free(snapd_data_list);

		if (found_flag) return SUCCESS;
		else return ERROR_NOT_FOUND;
	}

	return snapd_entry_cnt;
}

/**********************************************************************
** Get_All_Snapshot_Schedules_Conf
** Description:	Get all scheduled snapshot configs
** Input:
**		snap_vol_list:	the ptr to an array of SNAPSHOT_VOLUME_CONF
** Output:	Error code......
**		SUCCESS
**		ERROR_FAIL
***********************************************************************/
int Get_All_Snapshot_Schedules_Conf(SNAPSHOT_VOLUME_CONF** snap_vol_list)
{
	int count=0, i, total;
	SNAPSHOT_VOLUME_CONF* snaplist=NULL;
	SNAPSHOT_VOLUME_CONF *outlist=NULL, *p;

	total = Get_All_Snapshot_Volumes_Conf(&snaplist);
	if (total<0) {
#ifdef DEBUG
		printf("Get_All_Snapshot_Schedules_Conf: "
			"Get_All_Snapshot_Volumes_Conf failed, err=%d\n", total);
#endif
		return total;
	}

	for (i=0; i< total; i++) {
		if (snaplist[i].b_Is_Scheduled_Snap == FALSE)
			continue;

		p = realloc(outlist, (count+1)*sizeof(SNAPSHOT_VOLUME_CONF));
		if (p==NULL) {
			if (outlist) free(outlist);
			outlist = NULL;
			count = ERROR_OUT_OF_MEMORY;

			goto get_all_snapshot_schedules_conf_out;
		}
		outlist = p;

		memcpy(&outlist[count], &snaplist[i], sizeof(SNAPSHOT_VOLUME_CONF));
		count++;
	}

get_all_snapshot_schedules_conf_out:
	if (snaplist) free(snaplist);

	*snap_vol_list = outlist;
	return count;
}

/**********************************************************************
** Get_All_Snapshot_Schedules_Info
** Description:	Retrieve all sheduled snapshot volumes' info
** Input:
**		entry:	the ptr to an a SNAPD_CONF_ENTRY
** Output:	Error code......
**		SUCCESS
**		ERROR_FAIL
***********************************************************************/
int Get_All_Snapshot_Schedules_Info(SNAPSHOT_VOLUME_INFO** snap_info_list)
{
	int count=0, i, total;
	SNAPSHOT_VOLUME_INFO* snaplist=NULL;
	SNAPSHOT_VOLUME_INFO *outlist=NULL, *p;

	total = Get_All_Snapshot_Volumes_Info(&snaplist);
	if (total<0) {
#ifdef DEBUG
		printf("Get_All_Snapshot_Schedules_Info: "
			"Get_All_Snapshot_Volumes_Info failed, err=%d\n", total);
#endif
		return total;
	}

	for (i=0; i< total; i++) {
		if (snaplist[i].snap_conf.b_Is_Scheduled_Snap == FALSE)
			continue;

		p = realloc(outlist, (count+1)*sizeof(SNAPSHOT_VOLUME_INFO));
		if (p==NULL) {
			if (outlist)
				free(outlist);
			outlist = NULL;
			count = ERROR_OUT_OF_MEMORY;

			goto get_all_snapshot_schedules_info_out;
		}
		outlist = p;

		memcpy(&outlist[count], &snaplist[i], sizeof(SNAPSHOT_VOLUME_INFO));
		count++;
	}

get_all_snapshot_schedules_info_out:
	// snap_instances left to the output list
	if (count<=0)
		Release_Snapshot_Info_List(snaplist, total);
	else if (total>0 && snaplist) free(snaplist);

	*snap_info_list = outlist;
	return count;
}

/**********************************************************************
** Is_A_Scheduled_Snapshot_Volume
** Description:	Retrieve all sheduled snapshot volumes' info
** Input:
**		snap_vol_no:	the snapshot volume's number
** Output:	TRUE or FALSE
***********************************************************************/
BOOL Is_A_Scheduled_Snapshot_Volume(int snap_vol_no)
{
	SNAPSHOT_VOLUME_CONF snapconf;

	if (Get_One_Snapshot_Volume_Conf(snap_vol_no, &snapconf)==SUCCESS
		&& snapconf.b_Is_Scheduled_Snap)
		return TRUE;

	return FALSE;
}

/**********************************************************************
** Release_Snapshot_Info_List
** Description:	Free all snapshot volumes' info list
** Input:
**		snap_list:	the snapshot info list's ptr
**		snap_cnt:	the number of list members
** Output:	error code, SUCCESS....
***********************************************************************/
int Release_Snapshot_Info_List(SNAPSHOT_VOLUME_INFO* snap_list, int snap_cnt)
{
	int i;

	if (snap_list == NULL)
		return SUCCESS;

	for (i=0; i<snap_cnt; i++) {
		if (snap_list[i].snap_conf.b_Is_Scheduled_Snap &&
			snap_list[i].i_Current_Instance_Count>0 &&
			snap_list[i].snap_instances)
			free(snap_list[i].snap_instances);
	}

	free(snap_list);
	return SUCCESS;
}

/**********************************************************************
** Get_Next_Snapshot_Volume_Instance_Path
** Description:	Get the next snapshot volume LV name
** Input:
**		sn_volno:	the snapshot volume number
**		sn_path:	the output string buffer
**		buf_size:	the output string size
** Output:	error code, SUCCESS....
***********************************************************************/
int Get_Next_Snapshot_Volume_Instance_Path(int sn_volno, char* sn_path, int buf_size)
{
	int ret;
	struct tm *ptm;
	time_t thistime;
	SNAPSHOT_VOLUME_CONF snapconf;

	ret = Get_One_Snapshot_Volume_Conf(sn_volno, &snapconf);
	if (ret<0) return ret;

	strcpy(sn_path, "");
	time(&thistime);
	ptm = localtime(&thistime);
	snprintf(sn_path, buf_size, "%s%04d%02d%02d%02d%02d",
		snapconf.c_Snapshot_Path,
		ptm->tm_year + 1900, ptm->tm_mon + 1, ptm->tm_mday,
		ptm->tm_hour, ptm->tm_min);

	return SUCCESS;
}

/**********************************************************************
** Is_Instance_Name_Of_Scheduled_Snapshot
** Description:	check if the snapshot volume format matches
** Input:
**		snap_lv:	the LV path of the 
** Output:	TRUE or FALSE
***********************************************************************/
BOOL Is_Instance_Name_Of_Scheduled_Snapshot(char* snap_conf_lv, char* snap_lv)
{
	int len1, len2;

	len1 = strlen(snap_conf_lv);
	len2 = strlen(snap_lv);

	if ((len2-len1 == TIMESTAMP_STRING_LENGTH) &&
		(!strncmp(snap_conf_lv, snap_lv, len1)) &&
		(is_digit_string(snap_lv+len1)))
		return TRUE;

	return FALSE;
}

/**********************************************************************
** Add_Instance_To_Snapshot_Record
** Description:	Add an instance from the "snap instances" field in SNAPSHOT_CONF
** Input:
**		snap_no:		the snapshot volume number
**		snap_instance_lv:	the LV name of the instance to add
** Output:	error code
**		SUCCESS...
***********************************************************************/
int Add_Instance_To_Snapshot_Record(int snap_no, char* snap_instance_lv)
{
	BOOL success;
	char app_name[BUF_SIZE];
	char *p, *s;
	int max_str_len = (MAX_SNAP_NO_PER_VOLUME*(SHARE_NAME_LENGTH + 1) > BUF_SIZE) ? 
		MAX_SNAP_NO_PER_VOLUME*(SHARE_NAME_LENGTH + 1) : BUF_SIZE;
	char buf[max_str_len], outbuf[max_str_len];

	if ((p = strrchr(snap_instance_lv, '/'))==NULL)
		s = snap_instance_lv;
	else
		s = p+1;

	if (strlen(s)==0) return ERROR_INVALID_PARAM;

	sprintf(app_name, "snapshot %d", snap_no);

	Get_Private_Profile_String(app_name, "snap instances", "",
		buf, max_str_len, SNAPSHOT_CONF);

	Remove_Blank_From_String(buf);

	if (strlen(buf)==0)
		strcpy(outbuf, s);
	else
		sprintf(outbuf, "%s,%s", buf, s);

	success = Set_Private_Profile_String(app_name, "snap instances",
		outbuf, SNAPSHOT_CONF);
	if (success == FALSE) return ERROR_FAIL;

	Get_Private_Profile_String(app_name, "snap restore status", "",
		buf, max_str_len, SNAPSHOT_CONF);

	if (strlen(buf)==0)
		sprintf(outbuf, "%d", s_NONE);
	else {
		if (buf[strlen(buf)-1] == ',')
			buf[strlen(buf)-1] = 0x0;
		sprintf(outbuf, "%s,%d", buf, s_NONE);
	}

	success = Set_Private_Profile_String(app_name, "snap restore status",
		outbuf, SNAPSHOT_CONF);

	if (success == FALSE) return ERROR_FAIL;
	return SUCCESS;
}

/**********************************************************************
** Remove_Instance_From_Snapshot_Record
** Description:	Remove an instance from the "snap instances" field in SNAPSHOT_CONF
** Input:
**		snap_no:		the snapshot volume number
**		snap_instance_lv:	the LV name of the instance to remove
** Output:	error code
**		SUCCESS...
***********************************************************************/
int Remove_Instance_From_Snapshot_Record(int snap_no, char* snap_instance_lv)
{
	int ret = SUCCESS, i = 1, found_flag = 0, index=0;
	char app_name[BUF_SIZE], tmp[SHARE_NAME_LENGTH];
	char *p, *s;
	int max_str_len = (MAX_SNAP_NO_PER_VOLUME*(SHARE_NAME_LENGTH + 1) > BUF_SIZE) ? 
		MAX_SNAP_NO_PER_VOLUME*(SHARE_NAME_LENGTH + 1) : BUF_SIZE;
	char buf[max_str_len], outbuf[max_str_len];

	if ((p = strrchr(snap_instance_lv, '/'))==NULL)
		s = snap_instance_lv;
	else
		s = p+1;

	sprintf(app_name, "snapshot %d", snap_no);

	Get_Private_Profile_String(app_name, "snap instances", "",
		buf, max_str_len, SNAPSHOT_CONF);

	strcpy(outbuf, "");
	while (Get_String_Field(buf, i++, ',', tmp, SHARE_NAME_LENGTH) == SUCCESS) {
		if (!strcmp(tmp, s)) {
			found_flag++;
			index = i-1;
		}
		else {
			if (strlen(outbuf)==0)
				strcpy(outbuf, tmp);
			else if (strlen(tmp)>0)
				sprintf(outbuf, "%s,%s", outbuf, tmp);
		}
	}

	if (found_flag) {
		char *p=NULL, *s;

		Set_Private_Profile_String(app_name, "snap instances", outbuf, SNAPSHOT_CONF);

		strcpy(buf, "");
		Get_Private_Profile_String(app_name, "snap restore status", "",
			buf, max_str_len, SNAPSHOT_CONF);
		if (strlen(buf)>0) {
			i = 1;
			s = buf;
			p = strchr(buf, ',');

			while (p && *p && i<index) {
				i++;
				s = p+1;
				p = strchr(s, ',');
			}

			if (i == index) {
				strncpy(outbuf, buf, (int)(s-buf));
				outbuf[(int)(s-buf)] = '\0';
				if (p && *++p)
					strcat(outbuf, p);
				if (Set_Private_Profile_String(app_name, "snap restore status",
					outbuf, SNAPSHOT_CONF) == FALSE)
					ret = ERROR_FAIL;
				else
					ret = SUCCESS;
			}
		}
	}
	else
		ret = ERROR_NOT_FOUND;
	return ret;
}

/**********************************************************************
** Get_Scheduled_Snapshot_Volume_Free_Space
** Description:	Get the total free space of all the snapshot volume's instances
** Input:
**		sn_no:		the snapshot volume number
** Output:	size (>=0) in KB or error code (<0)
***********************************************************************/
long long Get_Scheduled_Snapshot_Volume_Free_Space(int sn_no)
{
	int ret, i;
	long long size = 0;
	SNAPSHOT_VOLUME_INFO	sninfo;

	ret = Get_One_Snapshot_Volume_Info(sn_no, &sninfo);
	if (ret<0) return ret;

	if (sninfo.snap_conf.b_Is_Scheduled_Snap) {
		for (i=0; i<sninfo.i_Current_Instance_Count; i++) {
			size+=get_lv_free_size(sninfo.snap_instances[i].c_Snap_Name, TRUE);
		}
		if (sninfo.i_Current_Instance_Count>0)
			free(sninfo.snap_instances);
	}
	else if (sninfo.snap_conf.i_Status == S_READY)
		size = get_lv_free_size(sninfo.snap_conf.c_Snapshot_Path, TRUE);
	else
		size = ERROR_NOT_EXISTED;

	return size;
}

/**********************************************************************
** Get_Snapshot_Name
** Description:	Get the name out of a snapshot instance's path
** Input:
**		snap_lv :	the snapshot volume number
**		sn_name :	snapshot instance name
**		buf_size :	the buffer size of the output share_name
** Output:	name length
***********************************************************************/
int Get_Snapshot_Name(char* snap_lv, char* sn_name, int buf_size)
{
	char *p;
	int ret;

	if ((p = strrchr(snap_lv, '/')) == NULL) {
		ret = strlen(snap_lv);
		if (ret>buf_size-1)
			ret = ERROR_BUFFER_TOO_SMALL;
		else
			strcpy(sn_name, snap_lv);
	}
	else {
		p++;
		ret = strlen(p);
		if (ret>buf_size-1)
			ret = ERROR_BUFFER_TOO_SMALL;
		else
			strcpy(sn_name, p);
	}
	return ret;
}

/**********************************************************************
** Get_Snapshot_Share_Name
** Description:	Get the share name of a snapshot instance's path
**		Right now, it's the same as the snapshot's name
** Input:
**		snap_lv :	the snapshot volume number
**		share_name :	snapshot instance's share name
**		buf_size :	the buffer size of the output share_name
** Output:	name length
***********************************************************************/
int Get_Snapshot_Share_Name(char* snap_lv, char* share_name, int buf_size)
{
	return Get_Snapshot_Name(snap_lv, share_name, buf_size);
}

/**********************************************************************
** Get_Snapshot_Build_Time
** Description:	Get the approximate time the snapshot instances was built
** Input:
**		snap_lv :	the snapshot volume number
** Output:	error code or SUCCESS
***********************************************************************/
int Get_Snapshot_Build_Time(char* snap_lv, time_t *btime)
{
	SNAPSHOT_VOLUME_CONF snapconf;
	int ret, sn_no;

	sn_no = Get_Snapshot_Volume_No(snap_lv);
	if (sn_no<=0) return sn_no;

	ret = Get_One_Snapshot_Volume_Conf(sn_no, &snapconf);
	if (ret<0) return ret;

	if (snapconf.b_Is_Scheduled_Snap) { // a snapshot instance
		char snap_name[SHARE_NAME_LENGTH], snap_inst_name[SHARE_NAME_LENGTH];
		time_t buildtime;

		Get_Snapshot_Name(snapconf.c_Snapshot_Path, snap_name, SHARE_NAME_LENGTH);
		Get_Snapshot_Name(snap_lv, snap_inst_name, SHARE_NAME_LENGTH);

		buildtime = get_build_time_by_snap_instance_name(snap_name, snap_inst_name);
		if (buildtime<0) return buildtime;
		*btime = buildtime;
	}
	else
		*btime = snapconf.t_Build_Time;

	return SUCCESS;
}

//------------------------------------------------------------
// Get_Snapshot_Mount_Point_String
// input :
//	snap_path--	snapshot logical volume path
//	mp_str--	the output mount-point string
// output : error code--
//	SUCCESS
//	ERROR_FAIL
//------------------------------------------------------------
int Get_Snapshot_Mount_Point_String(char* snap_path, char *mp_str)
{
	int f_handle, ret;
	char snap_name[SHARE_NAME_LENGTH];

	ret = Get_Snapshot_Share_Name(snap_path, snap_name, SHARE_NAME_LENGTH);
	if (ret<0) return ret;

	sprintf(mp_str,"/share/SNAPSHOTS/%s", snap_name);

	ret = SUCCESS;
	if ((f_handle=open(mp_str, O_RDONLY))==-1) {
		ret = create_directory(mp_str);
		return ret;
	}
	else
		ret = ERROR_ALREADY_EXISTED;

	close(f_handle);
	return ret;
}

/**********************************************************************
** Get_Snapshot_PV_Path
** Description:	Get the PV name where the snapshot lives
** Input:
**		sn_vol_no :	the snapshot volume number
**		pv_path	:	output string buffer
**		buf_size :	the size of the output buffer
** Output:	error code or SUCCESS
***********************************************************************/
int Get_Snapshot_PV_Path(int sn_vol_no, char* pv_path, int buf_size)
{
	SNAPSHOT_VOLUME_CONF snapconf;
	LVM_VOLUME_CONF lvmconf;
	int srcvono, ret;

	ret = Get_One_Snapshot_Volume_Conf(sn_vol_no, &snapconf);
	if (ret<0) return ret;

	srcvono = Get_LV_Volume_No(snapconf.c_Src_LV);
	if (srcvono<0) return srcvono;

	ret = Get_One_LVM_Volume_Conf(srcvono, &lvmconf);
	if (ret<0) return ret;

	if (strlen(lvmconf.device_name) > buf_size-1)
		return ERROR_BUFFER_TOO_SMALL;

	strcpy(pv_path, lvmconf.device_name);
	return (strlen(lvmconf.device_name));
}

/**********************************************************************
** Get_Snapshot_PV_From_LV
** Description:	Get the PV name where the snapshot lives
** Input:
**		snlv :		the snapshot LV path
**		pv_path	:	output string buffer
**		buf_size :	the size of the output buffer
** Output:	error code or SUCCESS
***********************************************************************/
int Get_Snapshot_PV_From_LV(char* snlv, char* snpv, int buf_size)
{
	int sn_no = Get_Snapshot_Volume_No(snlv);

	return (Get_Snapshot_PV_Path(sn_no, snpv, buf_size));
}

/**********************************************************************
** Modify_Snapshot_Scheduled_Volume_Conf
** Description:	modify the snapshot schedule property
** Input:
** Output:	error code or SUCCESS
** Note:	only cycle, start time, and max instance size can be modified now
***********************************************************************/
int Modify_Snapshot_Scheduled_Volume_Conf(int sn_vol_no, SNAPSHOT_VOLUME_CONF* snconf)
{
	int ret;
	SNAPSHOT_VOLUME_CONF myconf;
	SNAPD_CONF_ENTRY snapdconf;
	char vol_section[BUF_SIZE], tmp_str[10];

	ret = Get_One_Snapshot_Volume_Conf(sn_vol_no, &myconf);
	if (ret<0) return ret;

	if (myconf.i_Max_Instance_Size > myconf.i_Max_Instance_Size
		&& Is_Allocatable_Snapshot_Volume(myconf.c_Src_LV,
			myconf.i_Max_Instance_Size - myconf.i_Max_Instance_Size)==FALSE)
		return ERROR_INVALID_PARAM;

	sprintf(vol_section, "snapshot %d", sn_vol_no);

	myconf.Cycle = snconf->Cycle;
	sprintf(tmp_str, "%d", myconf.Cycle);
	ret = Conf_Set_Field(SNAPSHOT_CONF,vol_section,"cycle",tmp_str);
	if (ret<0) return ret;

	myconf.t_Start_Time = snconf->t_Start_Time;
	sprintf(tmp_str, "%u", (unsigned int)myconf.t_Start_Time);
	ret = Conf_Set_Field(SNAPSHOT_CONF,vol_section,"start time",tmp_str);
	if (ret<0) return ret;

	myconf.Priority = snconf->Priority;
	sprintf(tmp_str, "%u", (unsigned int)myconf.Priority);
	ret = Conf_Set_Field(SNAPSHOT_CONF,vol_section,"priority",tmp_str);
	if (ret<0) return ret;

	sprintf(tmp_str, "%d", snconf->i_Max_Instance_Size);
	ret = Conf_Set_Field(SNAPSHOT_CONF,vol_section,"max instance size",tmp_str);
	if (ret<0) return ret;

	ret = Remove_One_Snap_Daemon_Conf_Record(sn_vol_no);
	if (ret<0) return ret;

	snapdconf.snap_idx = myconf.i_Snapshot_No;
	time(&snapdconf.built_time);
	snapdconf.start_time = myconf.t_Start_Time;
	snapdconf.cycle = myconf.Cycle;
	snapdconf.is_ok = FALSE;
	ret = Add_Snap_Daemon_Conf_Record(&snapdconf);

	return ret;
}

/**********************************************************************
** Set_Snapshot_Restore_Status
** Description:	set the restore status of a specific snapshot volume/instance
** Input:
** Output:	error code or SUCCESS
***********************************************************************/
int Set_Snapshot_Restore_Status(char* snlv, int status)
{
	int ret = SUCCESS, snap_no, i, index=-1;
	SNAPSHOT_VOLUME_CONF snapconf;
	char *sname;
	char outbuf[BUF_SIZE], tmp[20];

	snap_no = Get_Snapshot_Volume_No(snlv);
	if (snap_no<=0) return ERROR_NOT_FOUND;

	ret = Get_One_Snapshot_Volume_Conf(snap_no, &snapconf);
	if (ret<0) return ret;

	if (snapconf.b_Is_Scheduled_Snap) {
		sname = strrchr(snlv, '/');
		if (sname == NULL)
			sname = snlv;
		else
			sname++;

		strcpy(outbuf, "");
		for (i=0; i<MAX_SNAP_INSTANCE; i++) {
			if (strlen(snapconf.c_Instances[i])==0) continue;

			if (!strcmp(snapconf.c_Instances[i], sname)) {
				index = i;
				if (strlen(outbuf)==0)
					sprintf(outbuf, "%d", status);
				else
					sprintf(outbuf, "%s,%d", outbuf, status);
			}
			else if (strlen(outbuf)==0)
				sprintf(outbuf, "%d", snapconf.Restore_Status[i]);
			else
				sprintf(outbuf, "%s,%d", outbuf, snapconf.Restore_Status[i]);
		}
		if (index<0) return ERROR_NOT_FOUND;
	}
	else
		sprintf(outbuf, "%d", status);

	sprintf(tmp, "snapshot %d", snap_no);
	ret = Conf_Set_Field(SNAPSHOT_CONF, tmp, "snap restore status", outbuf);

	return ret;
}

/**********************************************************************
** Get_Snapshot_Restore_Status
** Description:	get the latest restore status of a specific snapshot volume/instance
** Input:
** Output:	error code or SUCCESS
***********************************************************************/
int Get_Snapshot_Restore_Status(char* snlv)
{
	int ret = SUCCESS, snap_no, i, index=-1;
	SNAPSHOT_VOLUME_CONF snapconf;
	char *sname;

	snap_no = Get_Snapshot_Volume_No(snlv);
	if (snap_no<=0) return ERROR_NOT_FOUND;

	ret = Get_One_Snapshot_Volume_Conf(snap_no, &snapconf);
	if (ret<0) return ret;

	if (snapconf.b_Is_Scheduled_Snap) {
		sname = strrchr(snlv, '/');
		if (sname == NULL)
			sname = snlv;
		else
			sname++;

		for (i=0; i<MAX_SNAP_INSTANCE; i++) {
			if (strlen(snapconf.c_Instances[i])==0) continue;

			if (!strcmp(snapconf.c_Instances[i], sname)) {
				index = i;
				break;
			}
		}
		if (index<0) return ERROR_NOT_FOUND;
		return snapconf.Restore_Status[index];
	}
	else
		return snapconf.Restore_Status[0];
}

/**********************************************************************
** Get_Snapshot_Restore_Status
** Description:	recover the snapshot.conf and remove the snapd.conf
** Input:
** Output:	error code or SUCCESS
***********************************************************************/
int Restore_Snapshot_Conf()
{
	int ret, realcnt, recordcnt, i, j;
	FILE *fp;
	char linebuf[BUF_SIZE];
	char sharename[SHARE_NAME_LENGTH];
	SNAPSHOT_VOLUME_CONF *real_snap_list=NULL;
	SNAPSHOT_VOLUME_CONF *record_snap_list=NULL;

	// step 0 : if the snapshot.conf does not exist, create it

	if (NAS_File_Lock(SNAPSHOT_CONF, 'r')==0) {
#ifdef DEBUG
		printf("Restore_Snapshot_Conf : fails to NAS_File_Lock %s\n",
				SNAPSHOT_CONF);
#endif
		return ERROR_LOCK_FILE;
	}

    	fp = fopen(SNAPSHOT_CONF,"r");
    	if (fp==NULL || fgets(linebuf, BUF_SIZE, fp)==NULL)
	{
		if (fp!=NULL) fclose(fp);

		// create a new, empty snapshot.conf
		fp = fopen(SNAPSHOT_CONF,"w+");
		if (fp==NULL) {
#ifdef DEBUG
			printf("Restore_Snapshot_Conf : fails to open %s\n",
				SNAPSHOT_CONF);
#endif
			
			ret = ERROR_OPEN_FILE;
			NAS_File_Unlock(SNAPSHOT_CONF);
			goto restore_snapshot_conf_out;
		}
	}
	fclose(fp);
	fp = NULL;
	NAS_File_Unlock(SNAPSHOT_CONF);

	// step 1 : get all existent or recorded snapshot LVs
	realcnt = get_all_existent_snapshots(&real_snap_list);
#ifdef DEBUG
	for (i=0; i<realcnt; i++) {
		printf("real snap %d -- %s\n", i+1, real_snap_list[i].c_Snapshot_Path);
	}
#endif
	recordcnt = Get_All_Snapshot_Volumes_Conf(&record_snap_list);

	// step 2 : compare the 2 lists
	//	step 2.1 : if a schedule record exists, check if any of the existing
	//		snapshot LVs belongs to this record;
	//		also check if the corresponsive snapd record exists
	// 	step 2.2 : if a non-scheduled record exists, find the matching snapshot;
	//		if not found, remove the record
	for (j=0; j< recordcnt; j++) {
		int found;

		if (record_snap_list[j].b_Is_Scheduled_Snap == TRUE) {
			int k;
			char* ptr;
			SNAPD_CONF_ENTRY drec;
			int inst_flags[MAX_SNAP_NO_PER_VOLUME];

			// Catherine 2003/04/10 -- create only when the record is gone
			ret = Get_One_Snap_Daemon_Conf_Record(
				record_snap_list[j].i_Snapshot_No, &drec);
			if (ret<0) {
				drec.snap_idx = record_snap_list[j].i_Snapshot_No;
				drec.built_time = record_snap_list[j].t_Build_Time;
				drec.start_time = record_snap_list[j].t_Start_Time;
				drec.cycle = record_snap_list[j].Cycle;
				drec.is_ok = FALSE;
				ret = Add_Snap_Daemon_Conf_Record(&drec);

#ifdef DEBUG
				if (ret<0)
					printf("Restore_Snapshot_Conf() : "
						"Add_Snap_Daemon_Conf_Record(snap_no = %d) fails, err=%d\n",
						record_snap_list[j].i_Snapshot_No, ret);
#endif
			}

			// assume no such instances
			memset((void*)inst_flags, 0, sizeof(int) * MAX_SNAP_NO_PER_VOLUME);

			for (i=0; i<realcnt; i++) {
				if (Is_Instance_Name_Of_Scheduled_Snapshot(
					record_snap_list[j].c_Snapshot_Path,
					real_snap_list[i].c_Snapshot_Path))
				{
#ifdef DEBUG
printf("\t%s is an instance of %s\n", real_snap_list[i].c_Snapshot_Path, record_snap_list[j].c_Snapshot_Path);
#endif
					found = 0;
					k = 0;
					while (k<record_snap_list[j].i_Max_Instance_Size)
					{
						if (strlen(record_snap_list[j].c_Instances[k])== 0) {
							k++;
							continue;
						}

						if ((ptr = strrchr(real_snap_list[i].c_Snapshot_Path, '/'))==NULL)
							ptr = real_snap_list[i].c_Snapshot_Path;
						else
							ptr++;
						if (!strcmp(record_snap_list[j].c_Instances[k], ptr))
						{
							found++;
							inst_flags[k]++;
							break;
						}
						k++;
					}

					if (found==0) {
						ret = Add_Instance_To_Snapshot_Record(
							record_snap_list[j].i_Snapshot_No,
							real_snap_list[i].c_Snapshot_Path);
#ifdef DEBUG
						printf("\t%s not found in instance records!\n",
							real_snap_list[i].c_Snapshot_Path);
						if (ret<0) {
							printf("Restore_Snapshot_Conf() : "
								"Add_Instance_To_Snapshot_Record("
								"%d, %s) fails, err=%d\n",
								record_snap_list[j].i_Snapshot_No,
								real_snap_list[i].c_Snapshot_Path, ret);
						}
#endif
						create_snap_share(real_snap_list[i].c_Snapshot_Path);
					}
				}
			}

			k = 0;
			while (k<record_snap_list[j].i_Max_Instance_Size &&
				strlen(record_snap_list[j].c_Instances[k])>0)
			{
				if (inst_flags[k] == 0) {
					char snaplv[LV_PATH_MAX_LEN];

					get_snap_instance_name_by_source_lv(
						real_snap_list[i].c_Src_LV,
						record_snap_list[j].c_Instances[k],
						snaplv, sizeof(snaplv));
#ifdef DEBUG
					printf("Restore_Snapshot_Conf : "
						"before remove instance record %s for %d\n",
						snaplv, record_snap_list[j].i_Snapshot_No);
#endif
					ret = Remove_Instance_From_Snapshot_Record(
						record_snap_list[j].i_Snapshot_No,
						snaplv);
					Get_Snapshot_Share_Name(snaplv, sharename, SHARE_NAME_LENGTH);
					Remove_NAS_Share(sharename);
				}
				k++;
			}
		}
		else {
			found = 0;
			for (i=0; i<realcnt; i++) {
				if (!strcmp(real_snap_list[i].c_Src_LV, record_snap_list[j].c_Src_LV)
					&& !strcmp(record_snap_list[j].c_Snapshot_Path,
						real_snap_list[i].c_Snapshot_Path))
				{
					found++;
					break;
				}
			}
			if (found==0) {	// no existent snapshot found
				ret = Remove_Snapshot_Volume_Record(
					record_snap_list[j].i_Snapshot_No);
#ifdef DEBUG
printf("remove redundant snapshot record %d\n", record_snap_list[j].i_Snapshot_No);
				if (ret<0)
					printf("Remove_Snapshot_Volume_Record() : "
						"Remove_Snapshot_Volume_Record(snap_no = %d) fails, err=%d\n",
						record_snap_list[j].i_Snapshot_No, ret);
#endif
				Get_Snapshot_Share_Name(record_snap_list[j].c_Snapshot_Path, sharename, SHARE_NAME_LENGTH);
				Remove_NAS_Share(sharename);
			}
		}
	}

	// step 3 : if any of the existent snapshots doesn't find its record,
	//	create a non-scheduled record 
	for (i=0; i<realcnt; i++) {
		for (j=0; j< recordcnt; j++) {
			if (!strcmp(real_snap_list[i].c_Src_LV, record_snap_list[j].c_Src_LV)
				&& ((record_snap_list[j].b_Is_Scheduled_Snap == FALSE
					&& !strcmp(record_snap_list[j].c_Snapshot_Path,
						real_snap_list[i].c_Snapshot_Path))
				|| (record_snap_list[j].b_Is_Scheduled_Snap == TRUE
					&& Is_Instance_Name_Of_Scheduled_Snapshot(
						record_snap_list[j].c_Snapshot_Path,
						real_snap_list[i].c_Snapshot_Path))))
			{
				real_snap_list[i].i_Snapshot_No = record_snap_list[j].i_Snapshot_No;
				break;
			}
		}

		if (real_snap_list[i].i_Snapshot_No == 0) { // no record found
#ifdef DEBUG
			printf("\t\t%s has no matched record!\n",
				real_snap_list[i].c_Snapshot_Path);
#endif
			// add a new record
			real_snap_list[i].b_Is_Scheduled_Snap = FALSE;
			time(&real_snap_list[i].t_Build_Time);
			real_snap_list[i].Priority = PRIORITY_NORMAL;
			real_snap_list[i].Restore_Status[0] = s_NONE;
			real_snap_list[i].i_Status = S_READY;
			do_add_snapshot_record(&real_snap_list[i]);
			create_snap_share(real_snap_list[i].c_Snapshot_Path);
		}
	}

	// step 4 : sort all instnaces of schedules
	if (record_snap_list) free(record_snap_list);
	record_snap_list = NULL;

	recordcnt = Get_All_Snapshot_Volumes_Conf(&record_snap_list);
	for (i=0; i<recordcnt; i++) {
		char tmpbuf[2*MAX_SNAP_INSTANCE*SHARE_NAME_LENGTH], snstr[20];

		if (record_snap_list[i].b_Is_Scheduled_Snap == FALSE)
			continue;

		sort_instance_names_in_a_snap_volume(record_snap_list[i].c_Instances, MAX_SNAP_INSTANCE);
		strcpy(tmpbuf, "");
		for (j=0; j<record_snap_list[i].i_Max_Instance_Size; j++) {
			if (strlen(record_snap_list[i].c_Instances[j])==0)
				continue;

			if (strlen(tmpbuf)==0)
				strcpy(tmpbuf, record_snap_list[i].c_Instances[j]);
			else
				sprintf(tmpbuf, "%s,%s", tmpbuf, record_snap_list[i].c_Instances[j]);
		}
#ifdef DEBUG
printf("snap %d instances = [%s]\n", record_snap_list[i].i_Snapshot_No, tmpbuf);
#endif
		sprintf(snstr, "snapshot %d", record_snap_list[i].i_Snapshot_No);
		Set_Private_Profile_String(snstr, "snap instances", tmpbuf, SNAPSHOT_CONF);
	}

restore_snapshot_conf_out:
	if (record_snap_list) free(record_snap_list);
	if (real_snap_list) free(real_snap_list);

	return ret;
}

// Catherine 2003/03/25
/**********************************************************************
** Modify_Snap_Conf_Time_Zone
** Description:	change the build time & start time of the snapshot
** Input:
** Output:	error code or SUCCESS
***********************************************************************/
int Modify_Snap_Conf_Time_Zone(int orgTZ, int newTZ)
{
	int ret, i, cnt;
	long hrdiff = orgTZ - newTZ;
//	long timediff = hrdiff * 3600;	// 60 min/hr * 60 sec/min
	// modify by Kent 2003/03/26
	long	timediff=hrdiff*36;
	// end
	SNAPSHOT_VOLUME_CONF *snaplist = NULL;

	if (labs(orgTZ) > 1200 || labs(newTZ) > 1200) {
#ifdef DEBUG
		printf("Modify_Snap_Conf_Time_Zone (%d, %d) : "
			"incorrect input parameter!",
			orgTZ, newTZ);
#endif
		return ERROR_INVALID_PARAM;
	}
		
	if (hrdiff == 0) {
#ifdef DEBUG
		printf("Modify_Snap_Conf_Time_Zone (%d, %d) : no need to change!",
			orgTZ, newTZ);
#endif
		return SUCCESS;
	}

	ret = Get_All_Snapshot_Volumes_Conf(&snaplist);
	if (ret<0) {
#ifdef DEBUG
		printf("Modify_Snap_Conf_Time_Zone (%d, %d) : "
			"Get_All_Snapshot_Volumes_Conf fails, err=%d\n",
			orgTZ, newTZ, ret);
#endif
		return ret;
	}

	for (cnt=ret, i=0; i<cnt; i++) {
		char volstr[BUF_SIZE], tmp_str[BUF_SIZE];
		
		sprintf(volstr, "snapshot %d", snaplist[i].i_Snapshot_No);

		snaplist[i].t_Build_Time+=timediff;
		sprintf(tmp_str, "%u", (unsigned int)snaplist[i].t_Build_Time);

		ret = Conf_Set_Field(SNAPSHOT_CONF, volstr, "build time", tmp_str);
		if (ret<0) {
#ifdef DEBUG
			printf("Modify_Snap_Conf_Time_Zone (%d, %d) : "
				"Conf_Set_Field(\"%s\", \"%s\", \"build time\", \"%s\","
				" fails, err=%d\n",
				orgTZ, newTZ, SNAPSHOT_CONF, volstr, tmp_str, ret);
#endif
			goto modify_snap_conf_time_zone_out;
		}

		if (snaplist[i].b_Is_Scheduled_Snap == TRUE) {
			SNAPD_CONF_ENTRY snapdconf;

			snaplist[i].t_Start_Time+=timediff;
			sprintf(tmp_str, "%u", (unsigned int)snaplist[i].t_Start_Time);

			ret = Conf_Set_Field(SNAPSHOT_CONF, volstr, "start time", tmp_str);
			if (ret<0) {
#ifdef DEBUG
				printf("Modify_Snap_Conf_Time_Zone (%d, %d) : "
					"Conf_Set_Field(\"%s\", \"%s\", \"start time\", \"%s\","
					" fails, err=%d\n",
					orgTZ, newTZ, SNAPSHOT_CONF, volstr, tmp_str, ret);
#endif
				goto modify_snap_conf_time_zone_out;
			}

			// Catherine 2003/04/10 -- create only when the record is gone
			ret = Get_One_Snap_Daemon_Conf_Record(
				snaplist[i].i_Snapshot_No, &snapdconf);

			if (ret==SUCCESS) {
				ret = Remove_One_Snap_Daemon_Conf_Record(snaplist[i].i_Snapshot_No);
				if (ret<0) {
#ifdef DEBUG
					printf("Modify_Snap_Conf_Time_Zone (%d, %d) : "
						"Remove_One_Snap_Daemon_Conf_Record(%d)"
						" fails, err=%d\n",
						orgTZ, newTZ, snaplist[i].i_Snapshot_No, ret);
#endif
					goto modify_snap_conf_time_zone_out;
				}
			}
			else { // the record is gone
				snapdconf.snap_idx = snaplist[i].i_Snapshot_No;
				snapdconf.cycle = snaplist[i].Cycle;
				snapdconf.is_ok = FALSE;
			}

			time(&snapdconf.built_time);
			snapdconf.start_time = snaplist[i].t_Start_Time;
			ret = Add_Snap_Daemon_Conf_Record(&snapdconf);
		}
	}

modify_snap_conf_time_zone_out:
	if (cnt>0) free(snaplist);
	return ret;
}

/**********************************************************************
** Get_LVM_Volume_LV_From_Path
** Description:	get the path of a directory to see which volume this
**		network share is on
** Input:
** Output:	error code or the volume number
***********************************************************************/
int Get_LVM_Volume_LV_From_Path(char* path, char* out_lv)
{
	int ret, i, volcnt;
	LVM_VOLUME_CONF *vol_list = NULL;
	char mp[BUF_SIZE];

	if (path == NULL || out_lv == NULL)
		return ERROR_INVALID_PARAM;

	ret = Get_All_LVM_Volumes_Conf(&vol_list);
	if (ret<0) return ret;

	for (i = 0, volcnt = ret; i<volcnt; i++) {
		// this volume is not mounted??
		if (Get_MP_String(vol_list[i].drive_no_list[0], DATA_PART, mp) < 0)
			continue;
		if ((strncmp(mp, path, strlen(mp)) == 0) &&
			(strlen(path)==strlen(mp) || path[strlen(mp)] == '/'))
		{
			strcpy(out_lv, vol_list[i].lv_name);
			ret = vol_list[i].vol_no;
			goto get_src_vol_lv_from_path_out;
		}
	}
	ret = ERROR_NOT_FOUND;

get_src_vol_lv_from_path_out:
	if (volcnt > 0 && vol_list != NULL)
		free(vol_list);
	return ret;
}

#ifdef	STORAGE_FILE_LOCK
int Add_LVM_Volume_Record_Ex(LVM_VOLUME_CONF *vol_conf, BOOL file_lock)
{
	int vol_cnt=0,i=0,j=0;
	LVM_VOLUME_CONF *vol_data_list=NULL;
	FILE *fp;
	time_t the_time;
	char buf[2*BUF_SIZE], tmpbuf[BUF_SIZE], tmp_str[10];
	void *ptr=NULL;

	vol_cnt = Get_All_LVM_Volumes_Conf(&vol_data_list);
	
	if(vol_cnt >=MAX_VOLUME_NO) {
		free(vol_data_list);
		return ERROR_OUT_OF_RANGE;
	}
	if(vol_cnt >=0) {
		ptr = realloc((void*)vol_data_list,
			(vol_cnt+1)*sizeof(LVM_VOLUME_CONF));
		if (ptr==NULL) {
			if (vol_data_list) free(vol_data_list);
			return ERROR_OUT_OF_MEMORY;
		}
		vol_data_list = (LVM_VOLUME_CONF *)ptr;

		memcpy((void*)(&vol_data_list[vol_cnt]),
			(void*)vol_conf,
			sizeof(LVM_VOLUME_CONF));

		// get the minimum volume number
		{
			int i,j,vol_no,flag;
			
			for (i=1; i<=MAX_VOLUME_NO; i++) {
				flag=0;
				for (j=0; j<vol_cnt; j++) {
					if (i==vol_data_list[j].vol_no) {
						flag=1;
						break;
					}
				}
				if (flag==0) {
					vol_no=i;
					break;
				}
			}
			vol_data_list[vol_cnt].vol_no=vol_no;
			vol_conf->vol_no = vol_no; // update back
		}

		// get device name
		if (vol_conf->raid_level==SINGLE) {
			Get_Partition_Name(vol_conf->drive_no_list[0],
				DATA_PART, vol_conf->device_name);
		}

		// get logical volume name
		Get_LV_Name_For_Device(vol_conf->device_name,
			vol_conf->lv_name,
			LV_PATH_MAX_LEN);
		strcpy(vol_data_list[vol_cnt].lv_name, vol_conf->lv_name);

		vol_cnt++;
		sort_lvm_volume_data(vol_data_list,vol_cnt);

		if (file_lock && (NAS_File_Lock(STORAGE_CONF, 'w')==0)) {
			free(vol_data_list);
			return ERROR_LOCK_FILE;
		}
		if((fp = fopen(STORAGE_CONF,"w+")) == NULL) {
			free(vol_data_list);
			if (file_lock) NAS_File_Unlock(STORAGE_CONF);
			return ERROR_OPEN_FILE;
		}
		// write the new content into the config file
		for(i=0;i<vol_cnt;i++){
			memset(buf,0,BUF_SIZE*2);
			// volume header
			sprintf(tmpbuf, "[VOLUME %d]\n", vol_data_list[i].vol_no);
			strcpy(buf, tmpbuf);

			// logical volume name
			sprintf(tmpbuf, "lv name = %s\n", vol_data_list[i].lv_name);
			strcat(buf, tmpbuf);

			// device name
			sprintf(tmpbuf, "device name = %s\n", vol_data_list[i].device_name);
			strcat(buf, tmpbuf);

			// raid level
			sprintf(tmpbuf, "raid level = %d\n", vol_data_list[i].raid_level);
			strcat(buf, tmpbuf);

			// drive no
			strcpy(tmpbuf,"drive no = ");
			for(j=0;j<vol_data_list[i].list_cnt;j++) {
				sprintf(tmp_str,"%d,",vol_data_list[i].drive_no_list[j]);
				strcat(tmpbuf,tmp_str);
			}
			tmpbuf[strlen(tmpbuf)-1] = '\0';
			strcat(tmpbuf, "\n");
			strcat(buf, tmpbuf);

			// spare drive no
			strcpy(tmpbuf,"spare drive no = ");
			for(j=0;j<vol_data_list[i].spare_list_cnt;j++) {
				sprintf(tmp_str,"%d,",vol_data_list[i].spare_drive_list[j]);
				strcat(tmpbuf,tmp_str);
			}
			tmpbuf[strlen(tmpbuf)-1] = '\0';
			strcat(tmpbuf, "\n");
			strcat(buf, tmpbuf);

			// status
			sprintf(tmpbuf, "status = %d\n", vol_data_list[i].status);
			strcat(buf, tmpbuf);

			// record time
			(void) time(&the_time);
			sprintf(tmpbuf, "record_time = %s\n", ctime(&the_time));
			strcat(buf, tmpbuf);

			// enable snapshot
			sprintf(tmpbuf, "enable snapshot = %s\n",
				(vol_data_list[i].snapshot_enable) ? "TRUE" : "FALSE");
			strcat(buf, tmpbuf);

			// snap repository ratio
			sprintf(tmpbuf, "snap repository ratio = %d\n", vol_data_list[i].snap_ratio);
			strcat(buf, tmpbuf);
#ifdef	__EXT3_SNAP__
			// max snap count
			sprintf(tmpbuf, "max snap count = %d\n", vol_data_list[i].max_snap_count);
			strcat(buf, tmpbuf);
#endif
			if (fputs(buf, fp)<0) {
				fclose(fp);
				free(vol_data_list);
				if (file_lock) NAS_File_Unlock(STORAGE_CONF);
				return ERROR_WRITE_FILE;
			}
		}
		// unlock the file & close it
		fclose(fp);
		free(vol_data_list);
		if (file_lock) NAS_File_Unlock(STORAGE_CONF);
	}
	return vol_cnt;
}
#endif
