//**************************************************************************
//
//	Copyright (c) 2002  ICP Electronics Inc.  All Rights Reserved.
//
//	FILE:
//		lvm.c
//
//	Abstract: 
//		provide LVM volume operation fuction for cgi
//
//	HISTORY:
//		2002/08/15 Catherine Shen -- remove the code that assumes
//					drive number never exceeds 4
//		2002/08/07 Catherine Shen -- make Get_LV_Mt_Info public
//					(replace local function "get_lv_mt_info")
//		2002/04/15 Catherine Shen -- created for a new feature (LVM)
//
//**************************************************************************
#include "naslvm.h"
#include <sys/stat.h>
#include "cfg_system.h"
#include <dirent.h>

long long get_approx_size(long long org_size, int method);
int do_remove_snap_lv(char* snaplv, char* pvname);
// Cat 2002/12/01
int remove_snapshot_volume_completely(int snapno);
extern time_t get_build_time_by_snap_instance_name(char* snap_name, char* snap_inst_name);
extern void clear_volume_related_messages(VOLUME_CONF *vol_conf);
extern int get_vg_name_by_lv(char* lvname, char* vgname, int buf_size);
extern int get_snap_instance_name_by_source_lv(char* srclv, char* instance_short_name,
	char* outbuf, int bufsize);
extern int create_snap_share(char* sn_lv);
extern void get_drive_alphabetic_order(int hd_no, char start_char, char* out_buf, int buf_size);
extern int get_drive_no_by_alphabetic_order(char start_char, char* in_buf);

long long get_avail_snap_vol_size(char *src_lv, BOOL create_flag);

//-----------------------------------------------------------------------------
// get_raid_number
// description : get the RAID device's number
//-----------------------------------------------------------------------------
int get_raid_number(char* device)
{
	if (device==NULL) return -1;

	if (strstr(device, "/dev/md")!=NULL) {
		int raidno;

		raidno = atoi(device+7);
		return raidno;
	}
	else return -1;
}
//-----------------------------------------------------------------------------
// convert_md_name
// description : get required formated raid device name, if not raid device name
//		just keep it
// input :
//	md_path : original device path
//	mdname : output string
//	buf_size : the size of the output buffer
//	out_format : 1 -- long format (ex. /dev/md00), 0 -- short (ex. /dev/md0)
//-----------------------------------------------------------------------------
int convert_md_name(char* md_path, char* mdname, int buf_size, int out_format)
{
	int raidno;

	if (md_path==NULL || mdname==NULL || strlen(md_path) > buf_size-1)
		return -1;

	raidno = get_raid_number(md_path);
	if (raidno>=0) {
		if (out_format) // long format
			sprintf(mdname, "/dev/md%02d", raidno);
		else
			sprintf(mdname, "/dev/md%d", raidno);
	}	
	else if (md_path!=mdname)
		strcpy(mdname, md_path);
	return SUCCESS;
}

//------------------------------------------------------------
// is_vg_exist
// input :
//	vg_path :	volume group name
// output : TRUE or FALSE
//------------------------------------------------------------
BOOL is_vg_exist(char* vg_path)
{
	char vg_info_path[LV_PATH_MAX_LEN];
	char *ptr, *start;
	FILE *fp;

	if (vg_path==NULL) return FALSE;

	ptr = strrchr(vg_path, '/');
	if (ptr==NULL)
		start = vg_path;
	else
		start = ptr+1;
		
	sprintf(vg_info_path, "/etc/lvmtab.d/%s", start);

	if ((fp=fopen(vg_info_path, "r"))==NULL)
		return FALSE;

	fclose(fp);

	sprintf(vg_info_path, "/dev/%s/group", start);

	if ((fp=fopen(vg_info_path, "r"))==NULL)
		return FALSE;
	fclose(fp);

	return TRUE;
}

//------------------------------------------------------------
// get_vg_info
// input :
//	vgname	:	VG name
//	vg_info :	output data structure
// output : Error code--
//	SUCCESS
//	ERROR_INVALID_PARAM
//	ERROR_OPEN_FILE
//	ERROR_FAIL
//------------------------------------------------------------
int get_vg_info(char* vgname, VG_INFO* vg_info)
{
	FILE *fp;
	char* ptr;
	char buf[BUF_SIZE], title[BUF_SIZE], value[BUF_SIZE];

	if (vgname==NULL || vg_info==NULL)
		return ERROR_INVALID_PARAM;

	if ((ptr = strrchr(vgname, '/')) == NULL)
		sprintf(buf, VG_STATE_FILE_FORMAT, vgname);
	else
		sprintf(buf, VG_STATE_FILE_FORMAT, ptr+1);

	if ((fp=fopen(buf, "r"))==NULL)
		return ERROR_OPEN_FILE;

	memset(buf, 0, BUF_SIZE);
	memset(vg_info, 0, sizeof(VG_INFO));

	while (fgets(buf, BUF_SIZE, fp)) {
		if (strlen(buf)<=0)
			continue;

		Get_String_Field(buf, 1, ':', title, BUF_SIZE);
		Get_String_Field(buf, 2, ':', value, BUF_SIZE);

		if (!strcasecmp(title, "name")) {
			strncpy(vg_info->c_VG_Name, value,
				LV_PATH_MAX_LEN);
			continue;
		}
		if (!strcasecmp(title, "size")) {
			vg_info->ll_VG_Size=atoll(value);
			continue;
		}
		if (!strcasecmp(title, "access")) {
			vg_info->i_VG_Access=atoi(value);
			continue;
		}
		if (!strcasecmp(title, "status")) {
			vg_info->i_VG_Status=atoi(value);
			continue;
		}
		if (!strcasecmp(title, "number")) {
			vg_info->i_VG_No=atoi(value);
			continue;
		}
		if (!strcasecmp(title, "LV max")) {
			vg_info->i_Max_LV_Count=atoi(value);
			continue;
		}
		if (!strcasecmp(title, "LV current")) {
			vg_info->i_LV_Count=atoi(value);
			continue;
		}
		if (!strcasecmp(title, "LV open")) {
			vg_info->i_Open_LV_Count=atoi(value);
			continue;
		}
		if (!strcasecmp(title, "PV max")) {
			vg_info->i_Max_PV_Count=atoi(value);
			continue;
		}
		if (!strcasecmp(title, "PV current")) {
			vg_info->i_PV_Count=atoi(value);
			continue;
		}
		if (!strcasecmp(title, "PV active")) {
			vg_info->i_Active_PV_Count=atoi(value);
			continue;
		}
		if (!strcasecmp(title, "PE size")) {
			vg_info->ll_PE_Size=atoll(value);
			continue;
		}
		if (!strcasecmp(title, "PE total")) {
			vg_info->i_PE_Count=atoi(value);
			continue;
		}
		if (!strcasecmp(title, "PE allocated")) {
			vg_info->i_Allocated_PE_Count=atoi(value);
			continue;
		}
		if (!strcasecmp(title, "uuid")) {
			strncpy(vg_info->c_VG_UUID, value,
				BUF_SIZE);
			continue;
		}
	}
	fclose(fp);
	return SUCCESS;
}

long long convert_size_string_to_KB(char* size_str, char* unit_str)
{
	long long size;
	double tmp_size;
	long long kb_size = 1, mb_size = 1024, gb_size = 1024*1024;

	if (size_str == NULL || unit_str==NULL) return -1;

	tmp_size = atof(size_str);
	Remove_Blank_From_String(unit_str);

	if (strcasecmp(unit_str, "KB")==0)
		size = (long long)(tmp_size * kb_size);
	else if (strcasecmp(unit_str, "MB")==0)
		size = (long long)(tmp_size * mb_size);
	else if (strcasecmp(unit_str, "GB")==0)
		size = (long long)(tmp_size * gb_size);
	else
		size = -1;
	return size;
}

//------------------------------------------------------------
// get_lv_info
// input :
//	lv_name :	LV path
//	lv_info :	output data structure
// output : Error code--
//	SUCCESS
//	ERROR_NOT_EXISTED
//	ERROR_INVALID_PARAM
//	ERROR_OPEN_FILE
//	ERROR_FAIL
//------------------------------------------------------------
int get_lv_info(char* lv_name, LV_INFO* lv_info)
{
	char tmpstr[255]={"tmpXXXXXX"}, tmp_file_name[BUF_SIZE];
	char cmd[BUF_SIZE], buf[BUF_SIZE];
	char value[BUF_SIZE], tmp[20];
	int ret;
	FILE *fp;

	if (lv_name==NULL || lv_info==NULL)
		return ERROR_INVALID_PARAM;

	mktemp(tmpstr);
	sprintf(tmp_file_name,"/tmp/%s",tmpstr);
	sprintf(cmd, "%s %s -c > %s 2>&1",
		LV_STATUS_CMD, lv_name, tmp_file_name);

	if (system(cmd)) {
		unlink(tmp_file_name);
		return ERROR_FAIL;
	}

	if ((fp=fopen(tmp_file_name, "r"))==NULL) {
		unlink(tmp_file_name);
		return ERROR_OPEN_FILE;
	}

	memset(buf, 0, BUF_SIZE);
	memset(lv_info, 0, sizeof(LV_INFO));

	ret = SUCCESS;

	if (fgets(buf, BUF_SIZE, fp)) {
		if (strlen(buf)<=0 || strstr(buf, "doesn't exist") != NULL) {
#ifdef DEBUG
			printf("get_lv_info(%s) : result (%s)!\n",
				lv_name, buf);
#endif
			ret = ERROR_NOT_EXISTED;
			goto get_lv_info_out;
		}
	
		Get_String_Field(buf, 1, ':', value, BUF_SIZE);
		strncpy(lv_info->c_LV_Name, value, LV_PATH_MAX_LEN);

		Get_String_Field(buf, 2, ':', value, BUF_SIZE);
		snprintf(lv_info->c_VG_Name, LV_PATH_MAX_LEN, "/dev/%s", value);

		Get_String_Field(buf, 3, ':', value, BUF_SIZE);
		lv_info->i_LV_Access=atoi(value);

		Get_String_Field(buf, 4, ':', value, BUF_SIZE);
		lv_info->i_LV_Status=atoi(value);

		Get_String_Field(buf, 5, ':', value, BUF_SIZE);
		lv_info->i_LV_No=atoi(value);

		Get_String_Field(buf, 6, ':', value, BUF_SIZE);
		lv_info->i_Open_Count=atoi(value);

		Get_String_Field(buf, 7, ':', value, BUF_SIZE);
		lv_info->ll_LV_Size=atoll(value)/2;

		Get_String_Field(buf, 10, ':', value, BUF_SIZE);
		lv_info->i_Allocation=atoi(value);

		Get_String_Field(buf, 12, ':', lv_info->c_Block_Device, BUF_SIZE);
		Get_String_Field(buf, 13, ':', tmp, 20);
		strcat(lv_info->c_Block_Device, ":");
		strcat(lv_info->c_Block_Device, tmp);
	}

	lv_info->ll_SnapLV_Used_Size = 0;
	lv_info->ll_SnapLV_Free_Size = 0;

	if (lv_info->i_LV_Access & LV_SNAPSHOT_MASK) { // is a snapshot
		long long exp_table_size=0, cow_table_size=0;
		char *p, *s, *t;

		fclose(fp);
		unlink(tmp_file_name);

		strcpy(tmpstr, "tmpXXXXXX");
		mktemp(tmpstr);
		sprintf(tmp_file_name,"/tmp/%s",tmpstr);
		sprintf(cmd, "%s %s > %s",
			LV_STATUS_CMD, lv_name, tmp_file_name);
	
		if (system(cmd)) {
			unlink(tmp_file_name);
			return ERROR_FAIL;
		}

		if ((fp=fopen(tmp_file_name, "r"))==NULL) {
			unlink(tmp_file_name);
			return ERROR_OPEN_FILE;
		}
		while (fgets(buf, BUF_SIZE, fp)) {
			if (!strncmp(buf, "Allocated to snapshot", strlen("Allocated to snapshot"))) {
				// get used size
				p = strchr(buf, '[');
				if (p==NULL) continue;

				s = strchr(++p, '/');
				if (s==NULL) continue;

				t=s+1;
				strncpy(tmp, p, (int)(s-p));
				tmp[(int)(s-p)]='\0';
				p = strrchr(tmp, ' ');
				if (p==NULL) {
					if (!strcmp(tmp, "0"))
						lv_info->ll_SnapLV_Used_Size = 0;
				}
				else {
					*p = '\0';
					s=p+1;
					lv_info->ll_SnapLV_Used_Size = convert_size_string_to_KB(tmp, s);
				}

				// get all exception table size
				p = t;	// a character after '/'
				s = strrchr(p, ']');
				if (s==NULL) continue;

				strncpy(tmp, p, (int)(s-p));
				tmp[(int)(s-p)]='\0';
				p = strrchr(tmp, ' ');
				if (p==NULL) continue;

				*p = '\0';
				s = p+1;
				exp_table_size = convert_size_string_to_KB(tmp, s);
			}

			if (!strncmp(buf, "Allocated to COW-table", strlen("Allocated to COW-table"))) {
				p = buf + sizeof("Allocated to COW-table");
				while (*p==' ') p++;

				if (*p) s = strchr(p, ' ');
				if (*p && *(s+1)) {
					strncpy(tmp, p, (int)(s-p));
					tmp[(int)(s-p)] = '\0';
					p = buf + strlen(buf) -1; // find end-of-line
					if (*p=='\n') *p = '\0';

					cow_table_size = convert_size_string_to_KB(tmp, s);
				}
			}
		}
		lv_info->ll_LV_Size= get_approx_size(exp_table_size+cow_table_size, 0);
		lv_info->ll_SnapLV_Free_Size = lv_info->ll_LV_Size - lv_info->ll_SnapLV_Used_Size;
	}
get_lv_info_out:
	fclose(fp);
	unlink(tmp_file_name);
	return ret;
}

//------------------------------------------------------------
// get_pv_info
// input :
//	pv_name :	PV path
//	pv_info :	output data structure
// output : Error code--
//	SUCCESS
//	ERROR_NOT_EXISTED
//	ERROR_INVALID_PARAM
//	ERROR_OPEN_FILE
//	ERROR_FAIL
//------------------------------------------------------------
int get_pv_info(char* pv_name, PV_INFO* pv_info)
{
	char tmp_file_name[BUF_SIZE], buf[BUF_SIZE];
	char value[BUF_SIZE];
	FILE *fp;
	int drvno;
	char vg_name[LV_PATH_MAX_LEN], pvname[LV_PATH_MAX_LEN];
	char *pvn, *vgn;

	if (pv_name==NULL || pv_info==NULL)
		return ERROR_INVALID_PARAM;

	// Catherine 2003/04/09
	if (!Is_RaidDev_Active(pv_name) &&
		((drvno = Get_Drive_No(pv_name)) < 1 || 
		!Is_HD_ReadWrite_Safe(drvno)))
	{
#ifdef DEBUG
		printf("get_pv_info(%s) : not an active RAID device nor exist as a drive!\n",
			pv_name);
#endif
		return ERROR_NOT_EXISTED;
	}

	Get_VG_Name_By_Device(pv_name, vg_name, sizeof(vg_name));
	if ((vgn = strrchr(vg_name, '/')) != NULL)
		vgn++;
	convert_md_name(pv_name, pvname, sizeof(pvname), 0);
	if ((pvn = strrchr(pvname, '/')) != NULL)
		pvn++;
	else
		pvn = pv_name;
	
	sprintf(tmp_file_name, "/proc/lvm/VGs/%s/PVs/%s", vgn, pvn);
	if ((fp = fopen(tmp_file_name, "r")) == NULL)
		return ERROR_OPEN_FILE;

	memset(pv_info, 0, sizeof(PV_INFO));
	strcpy(pv_info->c_VG_Name, vg_name);

	// the first line is the status line
	while (fgets(buf, BUF_SIZE, fp)) {
#ifdef DEBUG
		printf("%s(%s) : %s\n", __FUNCTION__, pv_name, buf);
#endif
		if (Get_String_Field(buf, 1, ':', value, BUF_SIZE) < 0) continue;

		if (strcasecmp(value, "name") == 0) {
			Get_String_Field(buf, 2, ':', value, BUF_SIZE);
			strncpy(pv_info->c_PV_Name, value, LV_PATH_MAX_LEN);
			continue;
		}

		if (strcasecmp(value, "size") == 0) {
			Get_String_Field(buf, 2, ':', value, BUF_SIZE);
			pv_info->ll_PV_Size=atoll(value)/2;
			continue;
		}

		if (strcasecmp(value, "number") == 0) {
			Get_String_Field(buf, 2, ':', value, BUF_SIZE);
			pv_info->i_PV_No=atoi(value);
			continue;
		}

		if (strcasecmp(value, "status") == 0) {
			Get_String_Field(buf, 2, ':', value, BUF_SIZE);
			pv_info->i_PV_Status=atoi(value);
			continue;
		}

		if (strcasecmp(value, "allocatable") == 0) {
			Get_String_Field(buf, 2, ':', value, BUF_SIZE);
			pv_info->i_Allocatable=atoi(value);
			continue;
		}

		if (strcasecmp(value, "LV current") == 0) {
			Get_String_Field(buf, 2, ':', value, BUF_SIZE);
			pv_info->i_LV_Count=atoi(value);
			continue;
		}

		if (strcasecmp(value, "PE size") == 0) {
			Get_String_Field(buf, 2, ':', value, BUF_SIZE);
			pv_info->ll_PE_Size=atoll(value);
			continue;
		}

		if (strcasecmp(value, "PE total") == 0) {
			Get_String_Field(buf, 2, ':', value, BUF_SIZE);
			pv_info->i_PE_Count=atoi(value);
			continue;
		}

		if (strcasecmp(value, "PE allocated") == 0) {
			Get_String_Field(buf, 2, ':', value, BUF_SIZE);
			pv_info->i_Allocated_PE_Count=atoi(value);
			continue;
		}

		if (strcasecmp(value, "uuid:") == 0) {
			Get_String_Field(buf, 2, ':', value, BUF_SIZE);
			strncpy(pv_info->c_PV_UUID, value, BUF_SIZE);
		}
	}
	fclose(fp);

	return SUCCESS;
}

//------------------------------------------------------------
// get_lv_total_size -- get total size of a mounted logical volume
// input :
//	lv_name :	LV path
//	is_snap :	Is a snapshot volume?
// output : the available size of LV or error code if fails--
//	ERROR_OPEN_FILE
//	ERROR_FAIL
//------------------------------------------------------------
long long get_lv_total_size(char* lv_name)
{
	LV_INFO lvinfo;
	int ret;

	if (lv_name==NULL) return -1;

	ret = get_lv_info(lv_name, &lvinfo);
	if (ret<0) return ret;
	return lvinfo.ll_LV_Size;
}

//------------------------------------------------------------
// get_lv_free_size -- get availabe size in a mounted logical volume
// input :
//	lv_name :	LV path
//	is_snap :	Is a snapshot volume?
// output : the available size of LV or error code if fails--
//	ERROR_OPEN_FILE
//	ERROR_INVALID_PARAM
//	ERROR_FAIL
//------------------------------------------------------------
long long get_lv_free_size(char* lv_name, BOOL is_snap)
{
	int lv_no, ret;

	if (lv_name == NULL) return ERROR_INVALID_PARAM;

	if (!is_snap) {
		LVM_VOLUME_CONF lvconf;
		LV_MOUNT_INFO lv_minfo;

		lv_no = Get_LV_Volume_No(lv_name);
		Get_One_LVM_Volume_Conf(lv_no, &lvconf);
		Get_MP_String(lvconf.drive_no_list[0], DATA_PART,
			lv_minfo.c_LV_Mount_Point);
		ret = Get_LV_Mt_Info(&lv_minfo);
		if (ret<0) return ret;

		return lv_minfo.ll_LV_Free_Size;
	}
	else {
		LV_INFO lvinfo;
		ret = get_lv_info(lv_name, &lvinfo);
		if (ret<0) return ret;

		return lvinfo.ll_SnapLV_Free_Size;
	}

}

//------------------------------------------------------------
// get_pv_free_size -- get availabe size in a physical volume
// input :
//	pv_path :	PV path
// output : the available size of PV or error code if fails--
//	ERROR_OPEN_FILE
//	ERROR_INVALID_PARAM
//	ERROR_FAIL
//------------------------------------------------------------
long long get_pv_free_size(char* pv_path)
{
	PV_INFO pvinfo;
	long long size;
	int ret;
#ifdef DEBUG
printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif
	if (pv_path==NULL) return ERROR_INVALID_PARAM;
#ifdef DEBUG
printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif
	ret = get_pv_info(pv_path, &pvinfo);
#ifdef DEBUG
printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif
	if (ret<0) return ret;

	size = pvinfo.ll_PE_Size*(pvinfo.i_PE_Count-pvinfo.i_Allocated_PE_Count);

	return size;
}
//------------------------------------------------------------
// create_vg
// input :
//	pv_name:	the devices' names
// output : error code--
//	SUCCESS
//	ERROR_INVALID_PARAM
//	PARTITION_NOT_EXIST
//	ERROR_FAIL
//------------------------------------------------------------
int create_vg(char *pv_name)
{
	int ret;
	char vgname[LV_PATH_MAX_LEN], pvname[LV_PATH_MAX_LEN];
	char cmd[BUF_SIZE], tmp[BUF_SIZE];

	if (pv_name==NULL)
		return ERROR_INVALID_PARAM;

	Get_VG_Name_By_Device(pv_name, vgname, sizeof(vgname));
#ifdef DEBUG
	printf("create_vg(%s) : new VG name = %s\n", pv_name, vgname);
#endif
	// clear old VG device
	sprintf(cmd, "/bin/rm %s -rf > /dev/null 1>/dev/null 2>/dev/null", vgname);
	system(cmd);

	sprintf(tmp, "%s -A n -s %dK %s",
		VG_CREATE_CMD, (int)PE_SIZE, vgname);

// 2005.04.26, Johnson Cheng
// For SCSI drives w/o larger IDs would not be included in partition table...
    {
	int maxdiskcnt = Get_Profile_Integer("Storage", "Disk Drive Number", 0);
	char device[16];
	Conf_Get_Field(DEFAULT_CFG_FILE, "Storage", "Drive 1", device, 16);
	if((maxdiskcnt > 16) && !strncmp(device, "sd", 2))
	{
		if (!Is_PV_Exist(pv_name))
			return PARTITION_NOT_EXIST;
	}
	else
	{
		if (!Is_Partition_Exist(pv_name))
			return PARTITION_NOT_EXIST;
	}
    }

	convert_md_name(pv_name, pvname, LV_PATH_MAX_LEN, 0);
	sprintf(tmp, "%s %s", tmp, pvname);
#ifndef DEBUG
	sprintf(cmd, "%s>/dev/null 1>/dev/null 2>/dev/null", tmp);
#else
	strcpy(cmd, tmp);
	printf("%s -- line %d; cmd=%s\n", __FUNCTION__, __LINE__, cmd);
#endif
	ret = system(cmd);

	// rescan PV (2003/12/18)
	sprintf(cmd, "%s -l %s > /dev/null 1>/dev/null 2>/dev/null",
		LVM_INIT_CMD, pv_name);
	system(cmd);

	return ret;
}

//------------------------------------------------------------
// remove_vg
// input :
//	vg_name	: the VG to remove
// output : error code--
//	SUCCESS
//	ERROR_FAIL
//------------------------------------------------------------
int remove_vg(char* vg_name)
{
	char cmd[BUF_SIZE];

	// deactivate VG
	sprintf(cmd, "%s -a n %s>/dev/null 1>/dev/null 2>/dev/null",
		VG_ACTIVATE_CMD, vg_name);
	if (system(cmd))
		return ERROR_FAIL;

	// remove VG
	sprintf(cmd, "%s %s>/dev/null 1>/dev/null 2>/dev/null",
		VG_REMOVE_CMD, vg_name);
	if (system(cmd))
		return ERROR_FAIL;

	// remove VG's status file
	sprintf(cmd, "/bin/rm %s -rf > /dev/null 1>/dev/null 2>/dev/null", vg_name);
	system(cmd);

	return SUCCESS;
}

//------------------------------------------------------------
// create_physical_volume
// input :
//	pv_path--	the physical device for this new PV
// output : error code--
//	SUCCESS
//	ERROR_INVALID_PARAM
//	ERROR_FAIL
//------------------------------------------------------------
int create_physical_volume(char* pv_path)
{
	char cmd[BUF_SIZE], pvname[BUF_SIZE];
	int ret=SUCCESS;

	if (pv_path==NULL) return ERROR_INVALID_PARAM;
#ifdef DEBUG
	printf("create_physical_volume(%s)\n", pv_path);
#endif
	convert_md_name(pv_path, pvname, LV_PATH_MAX_LEN, 0);
#ifdef DEBUG
	printf("\tafter convert_md_name (%s)\n", pvname);
#endif
	if (Is_PV_Exist(pvname))
		return SUCCESS;

	// clear PV header (2003/12/18)
	sprintf(cmd, "/bin/dd if=/dev/zero of=%s bs=4k count=1", pvname);
	strcat(cmd, ">/dev/null 1>/dev/null 2>/dev/null");
	system(cmd);

	// create PV
#ifdef DEBUG
	sprintf(cmd, "%s -ffy %s", PV_CREATE_CMD, pvname);
	printf("\tcommand = [%s]\n", cmd);
#else
	sprintf(cmd, "%s -ffy %s>/dev/null 1>/dev/null 2>/dev/null", PV_CREATE_CMD, pvname);
#endif
	if (system(cmd)) {
#ifdef DEBUG
		printf("create_physical_volume(%s) : fails to create PV??\n", pvname);
#endif
		return ERROR_FAIL;
	}
	// create a new VG for this PV
	ret = create_vg(pv_path);
	return ret;
}

//------------------------------------------------------------
// remove_physical_volume
// input :
//	pv_path--	the physical volume to remove
// output : error code--
//	SUCCESS
//	ERROR_INVALID_PARAM
//	ERROR_BUSY
//	ERROR_NOT_EXISTED
//	ERROR_FAIL
//------------------------------------------------------------
int remove_physical_volume(char* pv_path)
{
	char vgname[LV_PATH_MAX_LEN];
	PV_INFO pvinfo;
	int ret;

	if (pv_path == NULL) return ERROR_INVALID_PARAM;
#ifdef DEBUG
printf("remove_physical_volume(%s)\n", pv_path);
#endif
	if (!Is_PV_Exist(pv_path))
		return ERROR_NOT_EXISTED;
#ifdef DEBUG
printf("remove_physical_volume : get_pv_info(%s)\n", pv_path);
#endif
	get_pv_info(pv_path, &pvinfo);
	if (pvinfo.i_Allocated_PE_Count>0)
		return ERROR_BUSY;

	Get_VG_Name_By_Device(pv_path, vgname, sizeof(vgname));
#ifdef DEBUG
printf("remove_physical_volume remove_vg(%s)\n", vgname);
#endif
	ret = remove_vg(vgname);
	return ret;
}

//------------------------------------------------------------
// create_logical_volume_directly
// input :
//	lv_name--	logical volume path
//	pv_path--	the physical volume for this new LV
//	size--		the size of the LV (or snapshot) to be created,
//			0 means the whole PV size
//	is_snapshot--	whether the LV is a snapshot
//	src_lv--	if snapshot, the source volume's LV path
// output : LV size or error code--
//	ERROR_INVALID_PARAM
//	ERROR_ALREADY_EXISTED
//	ERROR_FAIL
//------------------------------------------------------------
int create_logical_volume_directly(char* lv_name, char* pv_path, long long size,
		BOOL is_snapshot, char* src_lv)
{
	char cmd[BUF_SIZE], *ptr=NULL;
	char lv_sname[LV_PATH_MAX_LEN], pvname[LV_PATH_MAX_LEN], vgname[LV_PATH_MAX_LEN];
	long long pv_free_size=0, aprox_size;
	int ret;
#ifdef DEBUG
printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif
	if (lv_name==NULL || pv_path == NULL || size<0 || (is_snapshot && src_lv==NULL))
		return ERROR_INVALID_PARAM;
#ifdef DEBUG
printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif
	ptr = strrchr(lv_name, '/');
	if (ptr!=NULL)
		strcpy(lv_sname, ptr+1);
	else {
#ifdef DEBUG
		printf("create_logical_volume_directly : %s is a short name!\n", lv_name);
#endif
		return ERROR_INVALID_PARAM;
	}
#ifdef DEBUG
printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif

	convert_md_name(pv_path, pvname, LV_PATH_MAX_LEN, 0);
#ifdef DEBUG
printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif
	// is logical volume already existed?
	if (Is_LV_Exist(lv_name)) {
#ifdef DEBUG
		printf("create_logical_volume_directly : LV (%s) already exists!\n",
			lv_name);
#endif
		return ERROR_ALREADY_EXISTED;
	}
#ifdef DEBUG
printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif
	ret = create_physical_volume(pvname);
#ifdef DEBUG
printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif
	if (ret<0) {
#ifdef DEBUG
		printf("create_logical_volume_directly : create_physical_volume(%s) fails, err=%d\n",
			pvname, ret);
#endif
		return ret;
	}
#ifdef DEBUG
printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif
	// get available disk space (KB)
	pv_free_size = get_pv_free_size(pvname);

	if (pv_free_size<0) {
#ifdef DEBUG
		printf("create_logical_volume_directly :get_pv_free_size(%s) fails, err=%d\n",
			pvname, (int)pv_free_size);
#endif
		return (int)pv_free_size;
	}
#ifdef DEBUG
printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif
	if (size ==0)
		aprox_size = pv_free_size;
	else {
		aprox_size = get_approx_size(size, 0);
		if (pv_free_size==0 || pv_free_size < aprox_size) {
#ifdef DEBUG
			printf("create_logical_volume_directly : "
				"pv_free_size = %lld, aprox_size = %lld ==> Not Enough!!\n",
				pv_free_size, aprox_size);
#endif
			return ERROR_OUT_OF_MEMORY;
		}
	}
#ifdef DEBUG
printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif
	if (is_snapshot)
#ifdef	__SUPPORT_SNAP_RW__
		sprintf(cmd, "%s -A n -p rw -s -L %lldK -n %s %s %s>/dev/null 1>/dev/null 2>/dev/null",
			LV_CREATE_CMD, aprox_size,
			lv_sname, src_lv, pvname);
#else
		sprintf(cmd, "%s -A n -s -L %lldK -p r -n %s %s %s>/dev/null 1>/dev/null 2>/dev/null",
			LV_CREATE_CMD, aprox_size,
			lv_sname, src_lv, pvname);
#endif
	else {
		Get_VG_Name_By_Device(pvname, vgname, sizeof(vgname));

		sprintf(cmd, "%s -A n -L %lldK -n %s %s %s",
			LV_CREATE_CMD, aprox_size,
			lv_sname, vgname, pvname);
#ifndef DEBUG
		strcat(cmd, ">/dev/null 1>/dev/null 2>/dev/null");
#endif
	}
#ifdef DEBUG
printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif
	if ((ret = system(cmd)) == 0) return SUCCESS;
	else {
		PV_INFO pvinfo;
#ifdef DEBUG
		printf("\tFails to create logical volume: cmd(%s) , err=%d\n"
			"\t\tremove PV...\n",
			cmd, ret);
#endif
		ret = get_pv_info(pv_path, &pvinfo);
		if (ret==0 && pvinfo.i_LV_Count==0)
			remove_physical_volume(pvname);

		return ERROR_EXEC_FAIL;
	}
}

#ifdef __EXT3_SNAP__
//------------------------------------------------------------
// create_partial_logical_volume_directly
// description :	ordinary LV only, NOT for snapshot
// input :
//	lv_name--	logical volume path
//	pv_path--	the physical volume for this new LV
//	ratio--		the ratio that snapshots will use
// output : error code--
//	ERROR_INVALID_PARAM
//	ERROR_ALREADY_EXISTED
//	ERROR_FAIL
//------------------------------------------------------------
int create_partial_logical_volume_directly(char* lv_name, char* pv_path, int ratio)
{
	char cmd[BUF_SIZE], *ptr=NULL;
	char lv_sname[LV_PATH_MAX_LEN], pvname[LV_PATH_MAX_LEN], vgname[LV_PATH_MAX_LEN];
	long long pv_free_size=0, aprox_size;
	int ret;
#ifdef DEBUG
	printf("create_partial_logical_volume_directly(%s, %s, %d) starts!\n",
		lv_name, pv_path, ratio);
#endif
	if (lv_name==NULL || pv_path == NULL || ratio<0 || ratio>100)
		return ERROR_INVALID_PARAM;

	ptr = strrchr(lv_name, '/');
	if (ptr!=NULL)
		strcpy(lv_sname, ptr+1);
	else {
#ifdef DEBUG
		printf("create_partial_logical_volume_directly : "
			"%s is a short name!\n", lv_name);
#endif
		return ERROR_INVALID_PARAM;
	}

	convert_md_name(pv_path, pvname, LV_PATH_MAX_LEN, 0);

	// is logical volume already existed?
	if (Is_LV_Exist(lv_name)) {
#ifdef DEBUG
		printf("create_partial_logical_volume_directly : LV (%s) already exists!\n",
			lv_name);
#endif
		return ERROR_ALREADY_EXISTED;
	}

	ret = create_physical_volume(pvname);
	if (ret<0) {
#ifdef DEBUG
		printf("create_partial_logical_volume_directly : create_physical_volume(%s) fails, err=%d\n",
			pvname, ret);
#endif
		return ret;
	}

	// get available disk space (KB)
	pv_free_size = get_pv_free_size(pvname);
	if (pv_free_size<0) {
#ifdef DEBUG
		printf("create_logical_volume_directly :get_pv_free_size(%s) fails, err=%d\n",
			pvname, (int)pv_free_size);
#endif
		return (int)pv_free_size;
	}

	aprox_size = get_approx_size(pv_free_size*(100-ratio)/100, 1);

	Get_VG_Name_By_Device(pvname, vgname, sizeof(vgname));
	sprintf(cmd, "%s -A n -L %lldK -n %s %s %s>/dev/null 1>/dev/null 2>/dev/null",
		LV_CREATE_CMD, aprox_size,
		lv_sname, vgname, pvname);

	if (!system(cmd)) {
		return SUCCESS;
	}
	else {
		PV_INFO pvinfo;
#ifdef DEBUG
		printf("\tFails to create logical volume: remove PV...\n");
#endif
		ret = get_pv_info(pv_path, &pvinfo);
		if (ret==0 && pvinfo.i_LV_Count==0)
			remove_physical_volume(pvname);

		return ERROR_EXEC_FAIL;
	}
	
	return ret;
}
#endif

BOOL is_device_big_enough_for_snapshot(char* pv_path, int snap_ratio, int max_snap_count)
{
	BOOL result = FALSE;
	long long devsize = 0;
	char buf[BUF_SIZE], tmp[BUF_SIZE];
	char newpartname[HD_DEVICE_NAME_LENGTH];
	char *p = strrchr(pv_path, '/');
	FILE *fp;

	if (p == NULL) p = pv_path;
	else p++;

	if (strncmp(p, "md", 2) == 0) {
		int raidno = atoi(p+2);
		sprintf(newpartname, "md%d", raidno);
		p = newpartname;
	}

	if ((fp = fopen(PARTITIONS_STAT_FILE, "r")) == NULL)
		return FALSE;

	while ( fgets(buf,BUF_SIZE,fp) != NULL) {
		// partition name
		Get_String_Field(buf, 4, SPACE, tmp, sizeof(tmp));
		if (strcmp(tmp, p) != 0) continue;

		// raw partition size
		Get_String_Field(buf, 3, SPACE, tmp, sizeof(tmp));
		devsize = atoll(tmp); // in KB

		if (devsize * snap_ratio / 100 > max_snap_count * PE_SIZE)
			result = TRUE;
		break;
	}

	fclose(fp);
	return result;
}

//------------------------------------------------------------
// create_logical_volume
// description : used for normal LVM volume & non-scheduled snapshot
// input :
//	vol_no--	logical volume number
//	pv_path--	the physical volume for this new LV
//	is_snapshot--	whether the LV is a snapshot
// output : LV size or error code--
//	ERROR_INVALID_PARAM
//	ERROR_ALREADY_EXISTED
//	ERROR_FAIL
//------------------------------------------------------------
int create_logical_volume(int vol_no, char* pv_path, BOOL is_snapshot)
{
	long long size=0;
	char lv_name[LV_PATH_MAX_LEN];
	int ret;
	SNAPSHOT_VOLUME_CONF snapconf;

	if (vol_no<0 || pv_path==NULL) return ERROR_INVALID_PARAM;

	if (is_snapshot) {
		ret = Get_One_Snapshot_Volume_Conf(
			vol_no, &snapconf);
		if (ret<0)
			return ret;

		strcpy(lv_name, snapconf.c_Snapshot_Path);
		size = get_avail_snap_vol_size(snapconf.c_Src_LV, TRUE);
	}
	else {
#ifdef __EXT3_SNAP__
		LVM_VOLUME_CONF volconf;

		ret = Get_One_LVM_Volume_Conf(vol_no, &volconf);
		if (ret<0) return ret;

		if (volconf.snapshot_enable) {
			if (is_device_big_enough_for_snapshot(pv_path,
				volconf.snap_ratio, volconf.max_snap_count) == FALSE)
			{
				char volstr[BUF_SIZE];

				volconf.snapshot_enable = FALSE;
				sprintf(volstr, "volume %d", volconf.vol_no);
				ret = Conf_Set_Field(STORAGE_CONF, volstr, "enable snapshot", "FALSE");
				if (ret<0) {
#ifdef DEBUG
					printf("create_logical_volume(%d, %s, %s) : \n"
						"\tConf_Set_Field(%s, %s, \"enable snapshot\", \"FALSE\") "
						"fails, err=%d\n",
						vol_no, pv_path, (is_snapshot) ? "TRUE" : "FALSE",
						STORAGE_CONF, volstr, ret);
#endif
					return ret;
				}
				strcpy(lv_name, volconf.lv_name);
			}
			else
			return (create_partial_logical_volume_directly(
				volconf.lv_name, pv_path, volconf.snap_ratio));
		}
		else {
			size = 0;
			strcpy(lv_name, volconf.lv_name);
		}
#else
		size = 0;
		Get_LV_Path(vol_no, lv_name, LV_PATH_MAX_LEN);
#endif
	}

	return (create_logical_volume_directly(lv_name, pv_path, 
		size, is_snapshot, snapconf.c_Src_LV));
}

//------------------------------------------------------------
// remove_logical_volume_directly
// input :
//	lv_name--	LV path
//	pv_path--	PV path
// output : error code--
//	SUCCESS
//	ERROR_INVALID_PARAM
//	ERROR_FAIL
//------------------------------------------------------------
int remove_logical_volume_directly(char* lv_name, char* pv_path)
{
	char cmd[BUF_SIZE];
	int ret;

	if (lv_name==NULL || pv_path==NULL) return ERROR_INVALID_PARAM;
#ifdef DEBUG
printf("remove_logical_volume_directly(%s, %s)\n", lv_name, pv_path);
#endif
	if (Is_LV_Exist(lv_name)) {
		if (Is_Mounted(lv_name))
			Unmount_LV(lv_name);

		// remove logical volume
		sprintf(cmd, "%s -A n -f %s > /dev/null 1>/dev/null 2>/dev/null", LV_REMOVE_CMD, lv_name);
#ifdef DEBUG
printf("remove_logical_volume_directly step 1 command=%s\n", cmd);
#endif
		if (system(cmd))
			return ERROR_FAIL;
	}
#ifdef DEBUG
printf("remove_logical_volume_directly step 2 remove_physical_volume\n");
#endif
	// remove related physical volume
	ret = remove_physical_volume(pv_path);
	if (ret==ERROR_NOT_EXISTED || ret==ERROR_BUSY)
		ret=SUCCESS;
	return ret;
}

//------------------------------------------------------------
// remove_logical_volume_2
// description : only for normal LVM volume && non-scheduled snapshot volume
// input :
//	lvname--	logical volume
// output : error code--
//	SUCCESS
//	ERROR_INVALID_PARAM
//	ERROR_FAIL
//------------------------------------------------------------
int remove_logical_volume_2(char *lv_name, char* pv_path, BOOL is_snapshot)
{
	int ret;

	if (lv_name == NULL || pv_path==NULL)
		return ERROR_INVALID_PARAM;
#ifdef DEBUG
	printf("remove_logical_volume_2(%s, %s, %d) starts!\n",
		lv_name, pv_path, is_snapshot);
#endif
	if (!is_snapshot) {
		int sn_cnt, i;
		SNAPSHOT_VOLUME_CONF *sn_vols=NULL;

		// remove all related snapshot volumes
		sn_cnt = Get_Snapshot_Volumes_Conf_On_One_LV(lv_name, &sn_vols);
		if (sn_cnt<0 && sn_cnt!=ERROR_NOT_EXISTED) return sn_cnt;
	
		for (i=0; i<sn_cnt; i++) {
			ret = remove_snapshot_volume_completely(sn_vols[i].i_Snapshot_No);
			if (ret<0) {
#ifdef DEBUG
				printf("remove_logical_volume : "
					"remove_snapshot_volume_completely(%d) fails, err=%d\n",
					sn_vols[i].i_Snapshot_No, ret);
#endif
				free(sn_vols);
				return ret;
			}
		}
		if (sn_vols) free(sn_vols);
	}

	return (remove_logical_volume_directly(lv_name, pv_path));
}

//------------------------------------------------------------
// remove_logical_volume
// description : only for normal LVM volume && non-scheduled snapshot volume
// input :
//	vol_no--	logical volume number
// output : error code--
//	SUCCESS
//	ERROR_INVALID_PARAM
//	ERROR_FAIL
//------------------------------------------------------------
int remove_logical_volume(int vol_no, char* pv_path, BOOL is_snapshot)
{
	char lv_name[LV_PATH_MAX_LEN];
	int ret;

	if (vol_no<=0 || pv_path==NULL) return ERROR_INVALID_PARAM;
#ifdef DEBUG
	printf("remove_logical_volume(%d, %s, %d) starts!\n",
		vol_no, pv_path, is_snapshot);
#endif
	if (!is_snapshot)
		ret = Get_LV_Path(vol_no, lv_name, LV_PATH_MAX_LEN);
	else ret = Get_Snapshot_Volume_Path(vol_no, lv_name, LV_PATH_MAX_LEN);

	if (ret<0) {
#ifdef DEBUG
		printf("remove_logical_volume(%d, %s, %d) :"
			" fails to get LV Path..., err=%d\n",
			vol_no, pv_path, is_snapshot, ret);
#endif
		return ret;
	}
	return (remove_logical_volume_2(lv_name, pv_path, is_snapshot));
}

//------------------------------------------------------------
// reduce_space_from_lv
// description : works only for normal LV
// input :
//	lv_no--	logical volume number
//	size--	the size to reduce from lv_no
// output : error code--
//	SUCCESS
//	ERROR_INVALID_PARAM
//	ERROR_FAIL
//------------------------------------------------------------
int reduce_space_from_lv(int lv_no, long long size)
{
	char cmd[BUF_SIZE], lv_name[LV_PATH_MAX_LEN];
	int ret, fstype;
	long long old_filesystem_size, aprox_size;

	if (lv_no<=0) return ERROR_INVALID_PARAM;

	if (size<=0) return SUCCESS;

	Get_LV_Path(lv_no, lv_name, LV_PATH_MAX_LEN);

	fstype = Get_Device_FileSystem_Type(lv_name); // ReiserFS Support

	aprox_size = get_approx_size(size, 0);

	if (aprox_size < 0) {
#ifdef DEBUG
		printf("reduce_space_from_lv, get_approx_size(%lld)==> %lld\n", size, aprox_size);
#endif
		return ERROR_FAIL;
	}

	if (fstype==EXT3_FS || fstype==EXT2_FS) {
		long long total_size;
		int drvno;
		char pv_path[HD_DEVICE_NAME_LENGTH];

		Get_PV_Path(lv_no,pv_path,HD_DEVICE_NAME_LENGTH);
		drvno = Get_Drive_No(pv_path);
		old_filesystem_size = get_lv_total_size(lv_name);//lvminfo.ll_LV_Total_Size;
		total_size = old_filesystem_size - aprox_size;
#ifdef __EXT3_SNAP__
// modify by Kent 2003/04/19
//		sprintf(cmd, "%s -f %s %lldk > /dev/null 1>/dev/null 2>/dev/null",
//			EXT2_RESIZE_CMD2, lv_name, total_size);
		sprintf(cmd, "%s -f -p %s %lld 2>/dev/null 1>/dev/null",
			EXT2_RESIZE_CMD, lv_name, total_size/4);
// end
#else
		sprintf(cmd, "%s -f -p %s %lldk > /dev/null 1>/dev/null 2>/dev/null",
			EXT2_RESIZE_CMD, lv_name, total_size);
#endif
	}
	else if (fstype == REISER_FS)
		sprintf(cmd, "%s -s -%lldK -f %s > /dev/null 1>/dev/null 2>/dev/null",
			REISERFS_RESIZE_CMD, aprox_size, lv_name);

	Set_Volume_Status(lv_no, S_VOLUME_BUSY);

	if (Is_Mounted(lv_name)) {
		Stop_Services(); // stop network services
		ret = Umount_LVM_Volume(lv_no);
		Start_Services(); // restart network services
	}

	ret = system(cmd);

	if (ret!=0) {
#ifdef DEBUG
		printf("reduce_space_from_lv : "
			"resize filesystem(cmd=%s) : %d!\n",cmd, ret);
#endif
		goto reduce_space_from_lv_fail;
	}

	sprintf(cmd, "%s -A n -L -%lldK -f %s>/dev/null 1>/dev/null 2>/dev/null",
		LV_REDUCE_CMD, aprox_size, lv_name);

	ret = system(cmd);

	if (ret!=0) {
#ifdef DEBUG
		printf("reduce_space_from_lv : "
			"reduce LV size(cmd=%s) : %d!\n",cmd, ret);
#endif
		// resize the LV filesystem again!
		if (fstype==EXT2_FS || fstype == EXT3_FS)
// modify by Kent 2003/04/19
//			sprintf(cmd, "%s -f %s %lldk > /dev/null 1>/dev/null 2>/dev/null",
			sprintf(cmd, "%s -f -p %s %lld 2>/dev/null 1>/dev/null",
// end
				EXT2_RESIZE_CMD, lv_name,
				old_filesystem_size);
		
		else // ReiserFS support
			sprintf(cmd, "%s -s +%lldK -f %s > /dev/null 1>/dev/null 2>/dev/null",
				REISERFS_RESIZE_CMD, aprox_size, lv_name);

		system(cmd);

		goto reduce_space_from_lv_fail;
	}
	ret = SUCCESS;

reduce_space_from_lv_out:
	Mount_LVM_Volume(lv_no);
	Set_Volume_Status(lv_no, S_READY);
	return ret;

reduce_space_from_lv_fail:
	ret = ERROR_FAIL;
	goto reduce_space_from_lv_out;
}

//------------------------------------------------------------
// reduce_space_from_snap_lv
// input :
//	lv_path--	logical volume path
//	size--	the size to reduce from lv_no
// output : error code--
//	SUCCESS
//	ERROR_INVALID_PARAM
//	ERROR_FAIL
//------------------------------------------------------------
int reduce_space_from_snap_lv(char* lv_name, long long size)
{
	char cmd[BUF_SIZE];
	int ret;

	if (lv_name==NULL) return ERROR_INVALID_PARAM;

	if (size==0) return SUCCESS;

	ret = Umount_Snapshot_Volume(lv_name);

	if (ret!=0) {
#ifdef DEBUG
		printf("reduce_space_from_snap_lv step 1: "
			"Umount_Snapshot_Volume : %d!\n", ret);
#endif
		Mount_Snapshot_Volume(lv_name);
		return ret;
	}

	sprintf(cmd, "%s -A n -L -%lldK -f %s>/dev/null 1>/dev/null 2>/dev/null",
		LV_REDUCE_CMD, size, lv_name);

	ret = system(cmd);

	Mount_Snapshot_Volume(lv_name);
	if (ret!=0) {
#ifdef DEBUG
		printf("reduce_space_from_snap_lv : "
			"reduce LV size(cmd=%s) : %d!\n",cmd, ret);
#endif
		return ERROR_EXEC_FAIL;
	}

	return SUCCESS;
}

//------------------------------------------------------------
// extend_space_to_lv
// input :
//	lv_no--	logical volume number
//	size--	the size to reduce from lv_no
// output : error code--
//	SUCCESS
//	ERROR_INVALID_PARAM
//	ERROR_FAIL
//------------------------------------------------------------
int extend_space_to_lv(int lv_no, long long size)
{
	char cmd[BUF_SIZE];
	char lv_name[LV_PATH_MAX_LEN], pv_name[HD_DEVICE_NAME_LENGTH];
	int fstype, ret;

	if (lv_no<=0) return ERROR_INVALID_PARAM;
	if (size<=0) return SUCCESS;
	Get_LV_Path(lv_no, lv_name, LV_PATH_MAX_LEN);
	Get_PV_Path(lv_no, pv_name, HD_DEVICE_NAME_LENGTH);
	convert_md_name(pv_name, pv_name, HD_DEVICE_NAME_LENGTH, 0);

	fstype = Get_Volume_FileSystem_Type(lv_no); // ReiserFS support
	if (fstype == EXT2_FS || fstype==EXT3_FS) {
		Set_Volume_Status(lv_no, S_VOLUME_BUSY);
		if (Is_Mounted(lv_name)) {
			Stop_Services();
			Umount_LVM_Volume(lv_no);
			Start_Services();
		}
	}

	sprintf(cmd, "%s -A n -L +%lldK %s %s>/dev/null 1>/dev/null 2>/dev/null",
		LV_EXTEND_CMD, size, lv_name, pv_name);

	ret = system(cmd);

	if (ret<0) {
#ifdef DEBUG
		printf("extend_space_to_lv : "
			"extend LV (cmd=%s) : %d\n",cmd, ret);
#endif
		ret = ERROR_FAIL;
		goto extend_space_to_lv_out;
	}
	if (fstype==EXT2_FS || fstype == EXT3_FS) {
		sprintf(cmd, "%s -f -p %s > /dev/null 1>/dev/null 2>/dev/null",
			EXT2_RESIZE_CMD, lv_name);
	}
	else // ReiserFS support , but not sure ??
		sprintf(cmd, "%s -f %s > /dev/null 1>/dev/null 2>/dev/null",
			REISERFS_RESIZE_CMD, lv_name);

	ret = system(cmd);
	if (ret!=0) {
#ifdef DEBUG
		printf("extend_space_to_lv : "
			"extend filesystem (cmd=%s) : %d\n",cmd, ret);
#endif
		ret = ERROR_FAIL;
		goto extend_space_to_lv_out;
	}

	ret = SUCCESS;

extend_space_to_lv_out:
	if (fstype == EXT2_FS || fstype==EXT3_FS) {
		Mount_LVM_Volume(lv_no);
		Set_Volume_Status(lv_no, S_READY);
	}

	return ret;
}

//------------------------------------------------------------
// extend_space_to_snap_lv
// input :
//	lv_name--	logical volume path
//	size--	the size to reduce from lv_no
// output : error code--
//	SUCCESS
//	ERROR_INVALID_PARAM
//	ERROR_FAIL
//------------------------------------------------------------
int extend_space_to_snap_lv(char *lv_name, long long size)
{
	char cmd[BUF_SIZE];
	char pv_name[HD_DEVICE_NAME_LENGTH];
	int ret;

	if (lv_name==NULL) return ERROR_INVALID_PARAM;
	if (size==0) return SUCCESS;

	Get_Snapshot_PV_From_LV(lv_name, pv_name, HD_DEVICE_NAME_LENGTH);
#ifdef DEBUG
printf("extend_space_to_snap_lv(%s, %lld) : pv = %s\n",
	lv_name, size, pv_name);
#endif
	convert_md_name(pv_name, pv_name, sizeof(pv_name), 0);
	ret = Umount_Snapshot_Volume(lv_name);

	sprintf(cmd, "%s -A n -L +%lldK %s %s>/dev/null 1>/dev/null 2>/dev/null",
		LV_EXTEND_CMD, size, lv_name, pv_name);

	ret = system(cmd);

	Mount_Snapshot_Volume(lv_name);
	if (ret!=0) {
		return ERROR_FAIL;
	}

	return SUCCESS;
}

extern void start_vol_quota(int vol, int check);
extern int set_exist_flag(int drive_no, char *value);
extern int do_add_snapshot_record(SNAPSHOT_VOLUME_CONF* snap_conf);

//------------------------------------------------------------
// common_init_volume
// Description:	create PV/VG/LV/filesystem on a ready physical device
// Output:	SUCCESS
//		ERROR_INVALID_PARAM
//		ERROR_FAIL
//		ERROR_FORMAT
//		ERROR_MOUNT_FAIL
//------------------------------------------------------------
int common_init_volume(int vol_no, char* device)
{
	int ret;
#ifdef DEBUG
printf("%s -- line %d, vol_no = %d, device = %s\n", __FUNCTION__, __LINE__, ret, device);
#endif
	// create LV
	ret = create_logical_volume(vol_no, device, FALSE);
#ifdef DEBUG
printf("%s -- line %d, ret = %d\n", __FUNCTION__, __LINE__, ret);
#endif
	if (ret<0) {
#ifdef DEBUG
		printf("common_init_volume : create_logical_volume(%d, %s, FALSE):%d\n",
			vol_no, device, ret);
#endif
		Set_Volume_Status(vol_no,S_UNINITIALIZE);
		return ret;
	}
	// format LV
	Set_Volume_Status(vol_no,S_FORMATTING);
	if((ret=Format_LVM_Volume(vol_no))<0){
#ifdef DEBUG
		printf("common_init_volume : Format_LVM_Volume(%d):%d\n",
			vol_no, ret);
#endif
		Set_Volume_Status(vol_no,S_NOT_MOUNTED);
		return ret;
	}
	if((ret=Mount_LVM_Volume(vol_no))<0){
#ifdef DEBUG
		printf("common_init_volume : Mount_LVM_Volume(%d):%d\n",
			vol_no, ret);
#endif
		Set_Volume_Status(vol_no,S_NOT_MOUNTED);
		return ret;
	}

	start_vol_quota(vol_no, 1);
	
	Set_Volume_Status(vol_no,S_READY);

	return SUCCESS;
}

// cram the fields of LVM_VOLUME_CONF to VOLUME_CONF for clear_volume_related_messages only
void convert_lvm_to_volume_conf(LVM_VOLUME_CONF* org_volconf, VOLUME_CONF *newvolconf)
{
	int i, j;

	if (org_volconf==NULL || newvolconf==NULL) return;

	newvolconf->vol_no = org_volconf->vol_no;
	strcpy(newvolconf->device_name, org_volconf->device_name);
	newvolconf->raid_level = org_volconf->raid_level;
	for (i=0; i<org_volconf->list_cnt; i++)
		newvolconf->drive_no_list[i] = org_volconf->drive_no_list[i];

	for (j=0; j<org_volconf->spare_list_cnt; j++, i++)
		newvolconf->drive_no_list[i] = org_volconf->spare_drive_list[j];

	newvolconf->list_cnt = org_volconf->list_cnt;
	newvolconf->status = org_volconf->status;
}

//------------------------------------------------------------
// clear_lvm_volume_related_messages
// Description:	clear the old messages/statuses for the old volume
//------------------------------------------------------------
void clear_lvm_volume_related_messages(LVM_VOLUME_CONF *vol_conf)
{
	VOLUME_CONF tmpvol;

	convert_lvm_to_volume_conf(vol_conf, &tmpvol);
	clear_volume_related_messages(&tmpvol);
}

//------------------------------------------------------------
// init_lvm_single_disk_volume
// Description:	Create a LVM volume w/ only one disk
// Output:	SUCCESS
//		ERROR_INVALID_PARAM
//		ERROR_FAIL
//		ERROR_FORMAT
//		ERROR_MOUNT_FAIL
//------------------------------------------------------------
int init_lvm_single_disk_volume(LVM_VOLUME_CONF* vol_data)
{
	int ret, vol_no;

	if (vol_data==NULL) return ERROR_INVALID_PARAM;

	if(vol_data->raid_level != SINGLE) return ERROR_FAIL;

	vol_no = vol_data->vol_no;

	Stop_User_Quota(vol_no);
	
	Set_Volume_Status(vol_no,S_INITIALIZING);

	// initialize disk
	if((ret=Init_NAS_LVM_Disk(vol_data->drive_no_list[0]))<0){
#ifdef DEBUG
		printf("init_lvm_single_disk_volume : Init_NAS_LVM_Disk(%d):%d\n",
			vol_data->drive_no_list[0], ret);
#endif
		Set_Volume_Status(vol_no,S_UNINITIALIZE);
		return ret;
	}
	ret = common_init_volume(vol_no, vol_data->device_name);
	return ret;
}

//------------------------------------------------------------
// common_init_raid_volume(
// Output:	SUCCESS -- create successfully
//		ERROR_INVALID_PARAM
//		ERROR_FAIL
//		ERROR_FUNCTION_PARAMETER
//------------------------------------------------------------
int common_init_raid_volume(LVM_VOLUME_CONF* new_vol, BOOL init_disk_flag)
{
	int ret, vol_no, i;
#ifdef DEBUG
printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif
	if (init_disk_flag) {
#ifdef DEBUG
printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif
		// remove all single volume records
		for (i=0; i<new_vol->list_cnt; i++) {
			LVM_VOLUME_CONF tmpvol;

#ifdef DEBUG
printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif
			vol_no = Get_Volume_No_For_Drive(new_vol->drive_no_list[i]);
#ifdef DEBUG
printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif
			Stop_User_Quota(vol_no);
			set_exist_flag(new_vol->drive_no_list[i], "1");
#ifdef DEBUG
printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif
			Get_One_LVM_Volume_Conf(vol_no, &tmpvol);
			clear_lvm_volume_related_messages(&tmpvol);
#ifdef DEBUG
printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif
			ret = Remove_LVM_Volume_Record(vol_no);
#ifdef DEBUG
printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif
			if (ret<0) return ret;
		}

		for (i=0; i<new_vol->spare_list_cnt; i++) {
			LVM_VOLUME_CONF tmpvol;

#ifdef DEBUG
printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif
			vol_no = Get_Volume_No_For_Drive(new_vol->spare_drive_list[i]);
			Stop_User_Quota(vol_no);
			set_exist_flag(new_vol->spare_drive_list[i], "1");

			Get_One_LVM_Volume_Conf(vol_no, &tmpvol);
			clear_lvm_volume_related_messages(&tmpvol);

			ret = Remove_LVM_Volume_Record(vol_no);
#ifdef DEBUG
printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif
			if (ret<0) return ret;
		}
	}
#ifdef DEBUG
printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif
	Get_First_Available_RaidName(new_vol->device_name,
		HD_DEVICE_NAME_LENGTH);
	Get_LV_Name_For_Device(new_vol->device_name,
		new_vol->lv_name, LV_PATH_MAX_LEN);

	ret = Create_NAS_RaidDev_Ex(
		new_vol->device_name,
		new_vol->drive_no_list,
		new_vol->list_cnt,
		new_vol->spare_drive_list,
		new_vol->spare_list_cnt,
		new_vol->raid_level);
#ifdef DEBUG
printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif
	if(ret < 0) {
#ifdef DEBUG
		printf("common_init_raid_volume : "
			"Create_NAS_RaidDev_Ex(%s,%d) fails:%d\n",
			new_vol->device_name, new_vol->raid_level, ret);
#endif
		Restore_LVM_Storage_Conf();
		return ret;
	}
#ifdef DEBUG
printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif
	new_vol->status = S_INITIALIZING;
	ret = Add_LVM_Volume_Record(new_vol);
	if (ret<0) {
#ifdef DEBUG
		printf("common_init_raid_volume : "
			"Add_LVM_Volume_Record(%s, %s,%d) fails:%d\n",
			new_vol->lv_name,
			new_vol->device_name,
			new_vol->raid_level, ret);
#endif
		goto common_init_raid_volume_fail_out;
	}
#ifdef DEBUG
printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif
	if (init_disk_flag) {
		// re-initialize each disk
		for(i=0;i<new_vol->list_cnt;i++){
			if((ret=Init_NAS_LVM_Disk(new_vol->drive_no_list[i]))<0) {
#ifdef DEBUG
printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif
				goto common_init_raid_volume_fail_out;
			}
		}

		if ((new_vol->raid_level==MIRROR) || (new_vol->raid_level==RAID5)) {
			for(i=0;i<new_vol->spare_list_cnt;i++){
				if((ret=Init_NAS_LVM_Disk(new_vol->spare_drive_list[i]))<0) {
#ifdef DEBUG
printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif
					goto common_init_raid_volume_fail_out;
				}
			}
		}
		else new_vol->spare_list_cnt = 0;

		if (new_vol->raid_level==RAID5)
			sleep(1);
	}

	Set_Volume_Status(new_vol->vol_no,S_CREATING_RAID);

	Make_Raid(new_vol->device_name);
	ret = common_init_volume(new_vol->vol_no, new_vol->device_name);
#ifdef DEBUG
printf("%s -- line %d, ret = %d\n", __FUNCTION__, __LINE__, ret);
#endif
	return ret;

common_init_raid_volume_fail_out:
#ifdef DEBUG
printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif
	// restore config records
	Remove_LVM_Volume_Record(new_vol->vol_no);
	Delete_RaidDev(new_vol->device_name);
	for (i=0; i<new_vol->list_cnt; i++) {
		LVM_VOLUME_CONF lvmconf;

		Make_Single_Volume_Conf(&lvmconf, new_vol->drive_no_list[i], S_READY);
		Add_LVM_Volume_Record(&lvmconf);
	}

	for (i=0; i<new_vol->spare_list_cnt; i++) {
		LVM_VOLUME_CONF lvmconf;

		Make_Single_Volume_Conf(&lvmconf, new_vol->spare_drive_list[i], S_READY);
		Add_LVM_Volume_Record(&lvmconf);
	}
#ifdef DEBUG
printf("%s -- line %d, ret = %d\n", __FUNCTION__, __LINE__, ret);
#endif
	return ret;
}

//------------------------------------------------------------
// create_lvm_raid_volume
// Description:	Create a LVM+RAID volume
// Output:	SUCCESS -- create successfully
//		ERROR_INVALID_PARAM
//		ERROR_FAIL
//		ERROR_FUNCTION_PARAMETER
//------------------------------------------------------------
int create_lvm_raid_volume(LVM_VOLUME_CONF *vol_data)
{
	int i=0,ret,vol_no;
	LVM_VOLUME_CONF tmp_vol_data;

	if (vol_data==NULL) return ERROR_INVALID_PARAM;

	if((vol_data->list_cnt < 1) || (vol_data->raid_level<-1))
		return ERROR_FUNCTION_PARAMETER;

	switch(vol_data->raid_level) {
		case LINEAR://JBOD
		case STRIPING: //RAID 0
		case MIRROR: //RAID 1
		case RAID5: //RAID 5
			for(i=0;i<vol_data->list_cnt;i++){
				vol_no = Get_Volume_No_For_Drive(vol_data->drive_no_list[i]);

				ret = Get_One_LVM_Volume_Conf(
					vol_no,
					&tmp_vol_data);
				if(ret<0) return ret;

				if(tmp_vol_data.raid_level != SINGLE )
					return ERROR_FUNCTION_PARAMETER; //not signal volume

				if(tmp_vol_data.status != S_UNINITIALIZE &&
					tmp_vol_data.status != S_INVALID &&
					tmp_vol_data.status != S_READY &&
					tmp_vol_data.status != S_NOT_MOUNTED)
					return ERROR_FAIL;
			}
			for(i=0;i<vol_data->spare_list_cnt;i++){
				vol_no = Get_Volume_No_For_Drive(vol_data->spare_drive_list[i]);

				ret = Get_One_LVM_Volume_Conf(vol_no,
					&tmp_vol_data);
				if(ret<0) return ret; // get volume data fail
			
				if(tmp_vol_data.raid_level != SINGLE )
					return ERROR_FUNCTION_PARAMETER;
					//not signal volume
		
				if(tmp_vol_data.status != S_UNINITIALIZE &&
					tmp_vol_data.status != S_INVALID &&
					tmp_vol_data.status != S_READY &&
					tmp_vol_data.status != S_NOT_MOUNTED) 
					return ERROR_FAIL;
			}

			ret = common_init_raid_volume(vol_data, TRUE);
			break;
		default:
			ret = ERROR_FUNCTION_PARAMETER;
			break;
	}

	return ret;
}

//------------------------------------------------------------
// reinit_lvm_single_disk_volume
// Description:	Create a LVM volume w/ only one disk after a snapshot
//		gives it back
// Output:	SUCCESS
//		ERROR_INVALID_PARAM
//		ERROR_FAIL
//		ERROR_FORMAT
//		ERROR_MOUNT_FAIL
//------------------------------------------------------------
int reinit_lvm_single_disk_volume(LVM_VOLUME_CONF* vol_data)
{
	int ret, vol_no;

	if (vol_data == NULL) return ERROR_INVALID_PARAM;

	if(vol_data->raid_level != SINGLE) return ERROR_FAIL;

	vol_no = vol_data->vol_no;

	Set_Volume_Status(vol_no,S_INITIALIZING);

	ret = common_init_volume(vol_no, vol_data->device_name);
	return ret;
}

//------------------------------------------------------------
// recreate_lvm_raid_volume
// Description:	Create a LVM+RAID volume after a snapshot gives it back
// Output:	SUCCESS -- create successfully
//		ERROR_INVALID_PARAM
//		ERROR_FAIL
//		ERROR_FUNCTION_PARAMETER
//------------------------------------------------------------
int recreate_lvm_raid_volume(LVM_VOLUME_CONF *vol_data)
{
	int ret,vol_no;

	if (vol_data == NULL) return ERROR_INVALID_PARAM;
	if (!Is_RaidDev_Exist(vol_data->device_name)) {
#ifdef DEBUG
		printf("recreate_lvm_raid_volume: %s not exist!\n",
			vol_data->device_name);
#endif
		return ERROR_FUNCTION_PARAMETER;
	}

	if((vol_data->list_cnt < 1) || (vol_data->raid_level<-1))
		return ERROR_FUNCTION_PARAMETER;

	vol_no = vol_data->vol_no;

	switch(vol_data->raid_level) {
		case LINEAR://JBOD
		case STRIPING: //RAID 0
		case MIRROR: //RAID 1
		case RAID5: //RAID 5
			ret = common_init_volume(vol_no, vol_data->device_name);

			break;
		default:
			ret = ERROR_FUNCTION_PARAMETER;
			break;
	}
	return ret;
}
/*
//------------------------------------------------------------
// borrow_space_from_lv
// Description:	borrow the space from a LV or
//		claim the PV if no LV owns it
// Output:	SUCCESS
//		ERROR_INVALID_PARAM
//		ERROR_OPEN_FILE
//		ERROR_FAIL
//		ERROR_NOT_EXISTED
//------------------------------------------------------------
int borrow_space_from_lv(char* lv_path, long long size, char* pv_name, int name_len)
{
	int vol_no, ret;
	char src_pv_name[LV_PATH_MAX_LEN];
	long long pv_free_size;

	if (lv_path == NULL || size<0 || (name_len>0 && pv_name==NULL))
		return ERROR_INVALID_PARAM;
	if (size ==0) return SUCCESS;

	vol_no = Get_LV_Volume_No(lv_path);

	if (vol_no<0) {
#ifdef DEBUG
		printf("\tGet_LV_Volume_No(%s) fails, err=%d\n",
			lv_path, vol_no);
#endif
		return ERROR_NOT_EXISTED;
	}

	Get_PV_Path(vol_no, src_pv_name, LV_PATH_MAX_LEN);

	if ((pv_name!=NULL) && (name_len<strlen(src_pv_name))) {
#ifdef DEBUG
		printf("\tPV path buffer is too small\n");
#endif
		return ERROR_BUFFER_TOO_SMALL;
	}
	if (pv_name!=NULL)
		strcpy(pv_name, src_pv_name);

	if (Is_LV_Exist(lv_path)) {
		// is enough space for snapshot volume?
		pv_free_size = get_lv_free_size(lv_path, FALSE);
		if (pv_free_size<0) {
#ifdef DEBUG
			printf("\tLV is full? return code=%lld\n",
				pv_free_size);
#endif
			return (int)pv_free_size;
		}

		if (size > pv_free_size) {
#ifdef DEBUG
			printf("\tLV free size = %lldKB, too small!\n",
				pv_free_size);
#endif
			return ERROR_BUFFER_TOO_SMALL;
		}

		// reduce some space from source LV
		ret=reduce_space_from_lv(vol_no, size);
		if (ret<0) {
#ifdef DEBUG
			printf("\treduce_space_from_lv fails, error=%d\n", ret);
#endif
			return ret;
		}
	}
	else { // no one claims the PV
		if (!Is_PV_Exist(src_pv_name)) {
			ret = create_physical_volume(src_pv_name);
			if (ret<0) {
#ifdef DEBUG
				printf("\tfails to create PV, err=%d\n",ret);
#endif
				return ret;
			}
		}
		pv_free_size = get_pv_free_size(src_pv_name);
		if (size > pv_free_size)  {
#ifdef DEBUG
			printf("\tPV free size = %lldKB, too small!\n",
				pv_free_size);
#endif
			return ERROR_BUFFER_TOO_SMALL;
		}
	} 
	return SUCCESS;
}

//------------------------------------------------------------
// give_back_space_to_lv
// Description:	give back the space if it's borrowed or 
//		remove the PV if no LV claims it
// Output:	SUCCESS
//		ERROR_INVALID_PARAM
//		ERROR_NOT_EXISTED
//		ERROR_FAIL
//------------------------------------------------------------
int give_back_space_to_lv(char* lv_path, long long size)
{
	int vol_no, ret;

	if (lv_path==NULL || size<0) return ERROR_INVALID_PARAM;
	if (size == 0) return SUCCESS;

	vol_no = Get_LV_Volume_No(lv_path);

	if (vol_no<0) return ERROR_NOT_EXISTED;

	if (!Is_LV_Exist(lv_path)) {
		LVM_VOLUME_CONF lv_conf;

		if ((ret=Get_One_LVM_Volume_Conf(vol_no, &lv_conf))<0)
			return ret;
		ret = Create_LVM_Volume(&lv_conf);

		return ret;
	}

	// give it back
	ret = extend_space_to_lv(vol_no, size);
	return ret;
}
*/
//------------------------------------------------------------
// is_volume_as_big_as
// Description:	check if the targetvol has room bigger or equal
//		to srcvol's
// Output:	TRUE or FALSE
//------------------------------------------------------------
BOOL is_volume_as_big_as(int targetvol, int srcvol)
{
	BOOL is_tpv_created=FALSE, is_as_big=FALSE;
	int ret;
	PV_INFO tpv_info, spv_info;
	char tpv[LV_PATH_MAX_LEN], spv[LV_PATH_MAX_LEN];

	Get_PV_Path(srcvol, spv, LV_PATH_MAX_LEN);
	if (!Is_PV_Exist(spv)) {
#ifdef DEBUG
		printf("is_volume_as_big_as : Volume %d's PV %s not exists!\n",
			srcvol, spv);
#endif
		return FALSE;
	}

	Get_PV_Path(targetvol, tpv, LV_PATH_MAX_LEN);
	if (!Is_PV_Exist(tpv)) {
		is_tpv_created=TRUE;
		ret = create_physical_volume(tpv);
		if (ret<0) {
#ifdef DEBUG
			printf("is_volume_as_big_as : Volume %d's PV %s "
				"create fails, err=%d\n",
				targetvol, tpv, ret);
#endif
			return FALSE;
		}
	}

	ret = get_pv_info(spv, &spv_info);
	if (ret<0) {
#ifdef DEBUG
		printf("is_volume_as_big_as : get_pv_info fails, err=%d\n", ret);
#endif
		return FALSE;
	}

	ret = get_pv_info(tpv, &tpv_info);
	if (ret<0) {
#ifdef DEBUG
		printf("is_volume_as_big_as : get_pv_info fails, err=%d\n", ret);
#endif
		return FALSE;
	}

	if (tpv_info.i_PE_Count >= spv_info.i_PE_Count)
		is_as_big = TRUE;

	if (is_tpv_created)
		remove_physical_volume(tpv);

	return is_as_big;	
}

//------------------------------------------------------------
// get_approx_size
// Description:	get the acceptable LV size close to org_size
// input :
//	org_size	:	what user wants
//	method		:	(-1)--lower bound, 0--round, 1--upper bound
//------------------------------------------------------------
long long get_approx_size(long long org_size, int method)
{
	long long unitsize;
	long long quotient, remainder;

	unitsize = PE_SIZE;
	remainder = org_size % unitsize;
	quotient = org_size / unitsize;
	switch (method) {
		case -1 : {
				return quotient*unitsize;
			}
			break;
		case 0 : {
				if ((quotient>0) && (remainder<unitsize/2))
					return quotient*unitsize;
				return (quotient+1)*unitsize;
			}
			break;
		case 1 : {
				if (remainder==0)
					return org_size;
				return (quotient+1)*unitsize;
			}
			break;
		default : return ERROR_INVALID_PARAM;
	}
}
#ifndef __EXT3_SNAP__
//------------------------------------------------------------
// resize_all_instances_evenly
// Description:	resize the instances
// Output:	Error code...
//		SUCCESS
//------------------------------------------------------------
int resize_all_instances_evenly(SNAPSHOT_VOLUME_INSTANCE_INFO* instlist, int count,
	long long more_size, long long total_size, long long max_allowed_size)
{
	int i, ret = SUCCESS;
	long long free_space_table[count], total_space_table[count];
	long long share_size, extra_space=total_size;
	long long total_free_space = 0, ave_free_space;

	if (instlist==NULL || count<=0) return ERROR_INVALID_PARAM;

	for (i=0; i<count; i++) {
		total_space_table[i] = get_lv_total_size(instlist[i].c_Snap_Name);
		extra_space -= total_space_table[i];
		free_space_table[i] = get_lv_free_size(instlist[i].c_Snap_Name, TRUE);
		total_free_space += free_space_table[i];
	}

	total_free_space += extra_space;
	if (total_free_space + more_size <0)
		return ERROR_FAIL;

	ave_free_space = get_approx_size(total_free_space/count, -1);

	for (i=0; i<count; i++) {
		share_size = ave_free_space;

		// leave the rest of space to the last one
		if (i == count-1)
			share_size += total_free_space + more_size - ave_free_space*count;

		if (share_size > free_space_table[i]) {
			if (total_space_table[i] + share_size-free_space_table[i] <= max_allowed_size)
			{
				ret = extend_space_to_snap_lv(instlist[i].c_Snap_Name,
					share_size-free_space_table[i]);
			}
		}
		else if (free_space_table[i] > share_size)
			ret = reduce_space_from_snap_lv(instlist[i].c_Snap_Name,
				free_space_table[i]-share_size);
#ifdef DEBUG
		if (ret)
			printf("resize_all_instances_evenly: "
				"resize %s (%lld) failed, ret = %d\n",
				instlist[i].c_Snap_Name, share_size, ret);
#endif
	}
	return ret;
}
#endif
/* Cat : for new snap repository size issue
//------------------------------------------------------------
// prepare_snap_for_new_instance
// Description:	Prepare for a new member in a snapshot volume
//		Jobs : 
//		1. if no previous existent instances, the new instance
//			would take the whole of the volume
//		2. else if pool limit is reached,
//			kill oldest and evenly share the
//			space among the living instances & the new one
//		3. else, each instance must release some space for
//			the new comer
// Output:	Error code...
//		SUCCESS
//		ERROR_BUSY
//------------------------------------------------------------
int prepare_snap_for_new_instance(int snap_no, long long *new_instance_size)
{
	int ret;
	SNAPSHOT_VOLUME_INFO snapinfo;
	int srcvol;
	char pv_name[LV_PATH_MAX_LEN];

	ret = Get_One_Snapshot_Volume_Info(snap_no, &snapinfo);
	if (ret<0) return ret;

	srcvol = Get_LV_Volume_No(snapinfo.snap_conf.c_Src_LV);
	ret = Get_PV_Path(srcvol, pv_name, LV_PATH_MAX_LEN);
	if (ret<0) return ret;

	if (snapinfo.snap_conf.b_Is_Snap_PV_Allocated==FALSE) {
#ifdef DEBUG
		printf("prepare_snap_for_new_instance : "
			"The space was not allocated before??\n");
#endif		
		ret = ERROR_FAIL;
		goto prepare_snap_for_new_instance_out;
	}

	// the first one
	if (snapinfo.i_Current_Instance_Count==0) {
		long long maxsize;
		maxsize = get_lv_total_size(snapinfo.snap_conf.c_Src_LV) * MAX_SNAP_SIZE_ALLOWED;
		
		if (snapinfo.snap_conf.ll_Snapshot_Size > maxsize)
			*new_instance_size = maxsize;
		else
			*new_instance_size = snapinfo.snap_conf.ll_Snapshot_Size;
#ifdef DEBUG
		printf("prepare_snap_for_new_instance : "
			"new snapshot volume size = %lld KB\n", *new_instance_size);
#endif
		goto prepare_snap_for_new_instance_out;
	}
	// old instances exist
	// already full, someone has to die!!
	if (snapinfo.i_Current_Instance_Count==snapinfo.snap_conf.i_Max_Instance_Size)
	{
		LV_INFO lvinfo;
		SNAP_RESTORE_STATUS status = Get_Snapshot_Restore_Status(snapinfo.snap_instances[0].c_Snap_Name);

		if (status==s_BUSY) {
#ifdef DEBUG
			printf("prepare_snap_for_new_instance : "
				"snap pool full but the oldest is busy restoring!\n");
#endif			
			ret = ERROR_BUSY;
			goto prepare_snap_for_new_instance_out;
		}
		get_lv_info(snapinfo.snap_instances[0].c_Snap_Name, &lvinfo);
		*new_instance_size = lvinfo.ll_LV_Size;

		ret = do_remove_snap_lv(snapinfo.snap_instances[0].c_Snap_Name, pv_name);
		if (ret<0) {
#ifdef DEBUG
			printf("prepare_snap_for_new_instance : "
				"snap pool full but fails to remove the oldest instance(%s), err=%d!\n",
				snapinfo.snap_instances[0].c_Snap_Name, ret);
#endif
			goto prepare_snap_for_new_instance_out;
		}
		Remove_Instance_From_Snapshot_Record(snap_no, snapinfo.snap_instances[0].c_Snap_Name);
	}
	else {
		int i, count = snapinfo.i_Current_Instance_Count;
		long long free_space_table[count];
		long long real_free_size, total_free_space = 0, ave_free_space;

		real_free_size = snapinfo.snap_conf.ll_Snapshot_Size;
		for (i=0; i<count; i++) {
			real_free_size -= get_lv_total_size(snapinfo.snap_instances[i].c_Snap_Name);
			free_space_table[i] = get_approx_size(get_lv_free_size(
				snapinfo.snap_instances[i].c_Snap_Name, TRUE), -1);
			total_free_space += free_space_table[i];
		}

		if (real_free_size>0) { // no need to resize others
			long long srclvsize = get_lv_total_size(snapinfo.snap_conf.c_Src_LV);

			if (srclvsize<0) {
				ret = (int)srclvsize;
#ifdef DEBUG
				printf("prepare_snap_for_new_instance : "
					"get_lv_total_size=%d!\n",
					ret);
#endif
				goto prepare_snap_for_new_instance_out;
			}

			*new_instance_size = MIN(real_free_size, srclvsize * MAX_SNAP_SIZE_ALLOWED);
		}
		else {
			// old instances share free space with the new onw
			ave_free_space = get_approx_size(total_free_space/(count+1), -1);

			for (i=0; i<count; i++) {
				if (ave_free_space > free_space_table[i])
					ret = extend_space_to_snap_lv(
						snapinfo.snap_instances[i].c_Snap_Name,
						ave_free_space-free_space_table[i]);
				else
					ret = reduce_space_from_snap_lv(
						snapinfo.snap_instances[i].c_Snap_Name,
						free_space_table[i]-ave_free_space);
#ifdef DEBUG
				if (ret)
					printf("prepare_snap_for_new_instance : "
						"resize %s (%lld KB) failed, ret = %d\n",
						snapinfo.snap_instances[i].c_Snap_Name,
						ave_free_space, ret);
#endif
			}

			// leave the rest of space to the new one
			*new_instance_size = total_free_space - ave_free_space*count;
		}
		ret = SUCCESS;
	}
	
prepare_snap_for_new_instance_out:
	if (snapinfo.i_Current_Instance_Count>0)
		free(snapinfo.snap_instances);
	return ret;
}
*/

//------------------------------------------------------------
// create_snap_share
// Description : create a read-only share for a snapshot instance
// Output:	Error code...
//		SUCCESS
//------------------------------------------------------------
int create_snap_share(char* sn_lv)
{
	NAS_SHARE_INFO shareinfo;
	int ret;
	char read_list[1][USER_GROUP_NAME_LENGTH];

	Get_Snapshot_Name(sn_lv, shareinfo.share_name, SHARE_NAME_LENGTH);

	sprintf(shareinfo.comment, "Snapshot : %s", shareinfo.share_name);
	Get_Snapshot_Mount_Point_String(sn_lv, shareinfo.path);

	shareinfo.read_list_cnt = 1;
	strcpy(read_list[0], "@administrators");
	shareinfo.read_list_ptr = read_list;
	shareinfo.write_list_cnt = 0;
	shareinfo.write_list_ptr = NULL;
	
	ret = Create_NAS_Share(&shareinfo);
	return ret;
}

//------------------------------------------------------------
// do_remove_snap_lv
// Description : delete a snapshot instance
// Output:	Error code...
//		SUCCESS
//------------------------------------------------------------
int do_remove_snap_lv(char* snaplv, char* pvname)
{
	int ret;
	char sharename[SHARE_NAME_LENGTH], devname[HD_DEVICE_NAME_LENGTH];
	char cmd[BUF_SIZE];
#ifdef DEBUG
printf("do_remove_snap_lv : step 1\n");
#endif
	if (snaplv==NULL || pvname==NULL)
		return ERROR_INVALID_PARAM;

	if (!Is_LV_Exist(snaplv)) {
		ret = SUCCESS;
		goto do_remove_snap_lv_out;
	}

	// unmount snapshot
	ret = Umount_Snapshot_Volume(snaplv);
	if (ret<0) {
#ifdef DEBUG
		printf("do_remove_snap_lv : Umount_Snapshot_Volume fails!\n");
#endif
		return ERROR_UMOUNT_FAIL;
	}
#ifdef DEBUG
printf("do_remove_snap_lv : step 2\n");
#endif
	convert_md_name(pvname, devname, HD_DEVICE_NAME_LENGTH, 0);

#ifdef DEBUG
printf("do_remove_snap_lv : step 3\n");
#endif
	// remove snapshot LV
	ret = remove_logical_volume_directly(snaplv, devname);

	if (ret<0) {
#ifdef DEBUG
		printf("do_remove_snap_lv : Fails to remove snapshot volume!\n");
#endif
		//mount snapshot
		Mount_Snapshot_Volume(snaplv);
		return ERROR_FAIL;
	}
do_remove_snap_lv_out:
#ifdef DEBUG
printf("do_remove_snap_lv : step 4\n");
#endif
	Get_Snapshot_Share_Name(snaplv, sharename, SHARE_NAME_LENGTH);
	Remove_NAS_Share(sharename);
	sprintf(cmd, "/bin/rm /share/SNAPSHOT/%s -rf > /dev/null", sharename);
	system(cmd);

	return ret;
}

//------------------------------------------------------------
// remove_snapshot_schedule_and_instances(
// Description : delete a snapshot volume (if scheduled, include instances)
// Output:	Error code...
//		SUCCESS
//------------------------------------------------------------
int remove_snapshot_volume_completely(int snapno)
{
	int ret;
	SNAPSHOT_VOLUME_CONF snapconf;
	char pv_path[LV_PATH_MAX_LEN], snaplv[LV_PATH_MAX_LEN];
#ifdef DEBUG
printf("remove_snapshot_volume_completely : step 1\n");
#endif
	ret = Get_One_Snapshot_Volume_Conf(snapno, &snapconf);
	if (ret<0) return ret;
#ifdef DEBUG
printf("remove_snapshot_volume_completely : step 2\n");
#endif
	Get_PV_Path_For_LV(snapconf.c_Src_LV, pv_path, LV_PATH_MAX_LEN);

	if (snapconf.b_Is_Scheduled_Snap) {
		int j;
#ifdef DEBUG
printf("remove_snapshot_volume_completely : step 3\n");
#endif
		for (j=0; strlen(snapconf.c_Instances[j])>0; j++) {
			get_snap_instance_name_by_source_lv(snapconf.c_Src_LV, snapconf.c_Instances[j],
				snaplv, sizeof(snaplv));
#ifdef DEBUG
printf("remove_snapshot_volume_completely : step 3.5\n");
#endif
			ret = do_remove_snap_lv(snaplv, pv_path);
			if (ret<0) {
#ifdef DEBUG
				printf("remove_snapshot_volume_completely : "
					"do_remove_snap_lv fails, ret=%d!\n", ret);
#endif
				//mount snapshot
				Mount_Snapshot_Volume(snaplv);
				ret = ERROR_FAIL;
			}
			else
				Remove_Instance_From_Snapshot_Record(snapconf.i_Snapshot_No,
					snaplv);
			if (ret<0) {
#ifdef DEBUG
				printf("remove_snapshot_volume_completely : "
					"Fails to delete snapshot instance(%s) : %d\n",
					snaplv, ret);
#endif
				return ret;
			}
		}
		Remove_One_Snap_Daemon_Conf_Record(snapconf.i_Snapshot_No);
	}
	else {
#ifdef DEBUG
printf("remove_snapshot_volume_completely : step 3.9\n");
#endif
		ret = do_remove_snap_lv(snapconf.c_Snapshot_Path, pv_path);
		if (ret<0) {
#ifdef DEBUG
			printf("remove_snapshot_volume_completely : do_remove_snap_lv(%s,%s):%d!\n",
				snapconf.c_Snapshot_Path, pv_path, ret);
#endif
			Mount_Snapshot_Volume(snapconf.c_Snapshot_Path);
			Set_Snapshot_Status(snapconf.i_Snapshot_No, S_READY);
			return ret;
		}
	}
#ifdef DEBUG
printf("remove_snapshot_volume_completely : step 4\n");
#endif
	ret = Remove_Snapshot_Volume_Record(snapconf.i_Snapshot_No);

	return ret;
}
/* Cat : for new snap repository size issue ==>
//------------------------------------------------------------
// get_allowed_snap_vol_size
// Description : get the range of the allowed snapshot size
// Output:	Error code...
//		SUCCESS
//------------------------------------------------------------
int get_allowed_snap_vol_size(char *src_lv, int max_inst_cnt,
	long long *max_allowed_size, long long *min_allowed_size)
{
	long long srcsize, freesize;
	long long max_inst_size, min_inst_size;
	char pv_name[LV_PATH_MAX_LEN];
	int volno;

	srcsize = get_lv_total_size(src_lv);
	if (srcsize<0)
		return srcsize;
	volno = Get_LV_Volume_No(src_lv);
	Get_PV_Path(volno, pv_name, LV_PATH_MAX_LEN);

	if (!Is_LV_Exist(src_lv))
		return ERROR_INVALID_PARAM;

	freesize = get_lv_free_size(src_lv, FALSE);
	if (freesize<0) return freesize;

	max_inst_size = get_approx_size(MIN(MAX_SNAP_SIZE_ALLOWED * srcsize * max_inst_cnt, freesize), -1);
	min_inst_size = PE_SIZE * max_inst_cnt;

	if (max_inst_size < min_inst_size)
		return ERROR_FAIL;

	*max_allowed_size = max_inst_size;
	*min_allowed_size = min_inst_size;
	return SUCCESS;
}
*/

//------------------------------------------------------------------------
// get_total_existent_snap_size_on_one_vol
// description : get the total snapshot volume size on a volume
//------------------------------------------------------------------------
long long get_total_existent_snap_size_on_one_vol(char *src_lv)
{
	long long total_size=0, size;
	int cnt, i, j;
	SNAPSHOT_VOLUME_CONF *snaplist = NULL;
#ifdef DEBUG
printf("get_total_existent_snap_size_on_one_vol(%s) starts!\n", src_lv);
#endif
	cnt = Get_Snapshot_Volumes_Conf_On_One_LV(src_lv, &snaplist);
	if (cnt<0) {
#ifdef DEBUG
		printf("get_total_existent_snap_size_on_one_vol(%s) : "
			"Get_Snapshot_Volumes_Conf_On_One_LV fails, err=%d\n",
			src_lv, cnt);
#endif
		return (long long) cnt;
	}
#ifdef DEBUG
printf("get_total_existent_snap_size_on_one_vol step 1\n");
#endif
	for (i=0; i<cnt; i++) {
		if (snaplist[i].b_Is_Scheduled_Snap==FALSE) {
			size = get_lv_total_size(snaplist[i].c_Snapshot_Path);
			if (size>0)
				total_size += size;
		}
		else {
			char snaplv[LV_PATH_MAX_LEN];

			j=0;
			while (j<snaplist[i].i_Max_Instance_Size &&
				strlen(snaplist[i].c_Instances[j])>0)
			{
				get_snap_instance_name_by_source_lv(
					src_lv,
					snaplist[i].c_Instances[j],
					snaplv, sizeof(snaplv));

				size = get_lv_total_size(snaplv);
				if (size>0)
					total_size += size;
				j++;
			}
		}
	}
#ifdef DEBUG
printf("get_total_existent_snap_size_on_one_vol step 2 : total_used_size=%lld\n", total_size);
#endif

	if (cnt>0 && snaplist) free(snaplist);
#ifdef DEBUG
printf("get_total_existent_snap_size_on_one_vol step 3\n");
#endif

	return total_size;
}

//------------------------------------------------------------------------
// get_avail_snap_vol_size
// description : get the available new snapshot volume size
//		free size first;
//		if no free size is available,
//		get the available size from another existent snapshot
//		with the lowest priority and the most free space
//------------------------------------------------------------------------
long long get_avail_snap_vol_size(char *src_lv, BOOL create_flag)
{
	long long used_size, free_size;
	LVM_VOLUME_CONF srcvol;
	int volno, ret, snap_cnt;
#ifdef __EXT3_SNAP__
	long long unitsize=0;
#else
	long long size;
	int i;
	SNAPSHOT_BASIC_INFO *snaplist=NULL;
#endif

#ifdef DEBUG
printf("get_avail_snap_vol_size(%s, %d)\n", src_lv, create_flag);
#endif
	used_size = get_total_existent_snap_size_on_one_vol(src_lv);
	if (used_size<0) {
#ifdef DEBUG
		printf("get_avail_snap_vol_size(%s) : \n"
			"\tget_total_existent_snap_size_on_one_vol fails, err=%d\n",
			src_lv, (int)used_size);
#endif
		return used_size;
	}
#ifdef DEBUG
printf("get_avail_snap_vol_size step 1 : used_size=%lld\n", used_size);
#endif
	if ((volno = Get_LV_Volume_No(src_lv)) < 0 ) {
#ifdef DEBUG
		printf("get_avail_snap_vol_size(%s) : "
			"Get_LV_Volume_No fails, err=%d\n",
			src_lv, volno);
#endif
		return volno;
	}
#ifdef DEBUG
printf("get_avail_snap_vol_size step 2 : Get_LV_Volume_No=%d\n", volno);
#endif
	if ((ret = Get_One_LVM_Volume_Conf(volno, &srcvol)) < 0 ) {
#ifdef DEBUG
		printf("get_avail_snap_vol_size(%s) : "
			"Get_One_LVM_Volume_Conf(%d) fails, err=%d\n",
			src_lv, volno, ret);
#endif
		return ret;
	}
#ifdef DEBUG
printf("get_avail_snap_vol_size step 3\n");
#endif
	free_size = srcvol.snap_repository_size - used_size;

#ifdef __EXT3_SNAP__
	if (srcvol.max_snap_count>0)
		unitsize = get_approx_size(srcvol.snap_repository_size / srcvol.max_snap_count, -1);
	if (unitsize>0)
		snap_cnt = used_size / unitsize;
	else
		return -1;
#ifdef DEBUG
printf("get_avail_snap_vol_size step 4 :"
"snap_cnt = %d, unitsize = %lld KB, free_size = %lld KB\n",
snap_cnt, unitsize, free_size);
#endif
	if (free_size > (srcvol.max_snap_count - snap_cnt) * unitsize)
		return (unitsize + PE_SIZE);
	else
		return unitsize;

#else // not __EXT3_SNAP__
	if (srcvol.snap_repository_size > 0
		&& free_size*100/srcvol.snap_repository_size  > SNAP_MIN_SIZE_RATIO)
		return free_size;
	else if (free_size<0) {
#ifdef DEBUG
		printf("get_avail_snap_vol_size(%s) : invalid sizes -- "
			"snap repository size=%lld KB, used size = %lld KB\n",
			src_lv, srcvol.snap_repository_size, used_size);
#endif
		return ERROR_FAIL;
	}
	// no free size anymore, get the free size from existent snapshots
	size = free_size;
	snap_cnt = Get_Snapshots_On_One_LV_In_Order(src_lv,
		FLAG_IMPORTANCE_ASC | FLAG_SPACE_USAGE_ASC, &snaplist);
#ifdef DEBUG
printf("get_avail_snap_vol_size step 5\n");
#endif
	for (i=0; i< snap_cnt; i++) {
#ifdef DEBUG
printf("get_avail_snap_vol_size step 5.%d\n", i+1);
#endif
		free_size = get_approx_size(snaplist[i].ll_Free_Size * SNAP_DONATE_RATIO / 100, -1);
		if (free_size>0) {
			if (create_flag) {
				Set_Snapshot_Status(snaplist[i].i_Snapshot_No, S_VOLUME_BUSY);
				ret = reduce_space_from_snap_lv(snaplist[i].c_Path, free_size);
				Set_Snapshot_Status(snaplist[i].i_Snapshot_No, S_READY);
				if (ret<0) {
#ifdef DEBUG
					printf("get_avail_snap_vol_size(%s) : "
						"reduce_space_from_snap_lv(%s, %lld KB) fails, err=%d\n",
						src_lv, snaplist[i].c_Path, free_size, ret);
#endif
					continue;
				}
			}

			size += free_size;
			if (size*100/srcvol.snap_repository_size >= SNAP_MIN_SIZE_RATIO)
				break;
		}
	}
	return size;
#endif
}
#ifndef __EXT3_SNAP__
//------------------------------------------------------------------------
// share_free_space_among_existent_snapshots
// description : share the free space among all snapshots for the same LV
//		each priority level can share according to the ratios
//		PHIGH_SPACE_RATIO, PNORMAL_SPACE_RATIO, PLOW_SPACE_RATIO
//------------------------------------------------------------------------
int share_free_space_among_existent_snapshots(char *src_lv, long long size)
{
	// ??? not finished
	int ret = SUCCESS;
	int totalpe, cnt, realcnt, i, avgpe, pcnt, peno;
	SNAPSHOT_BASIC_INFO *snaplist = NULL;
#ifdef DEBUG
int loopcnt=0;
#endif
	if (src_lv == NULL || size<=0)
		return ERROR_INVALID_PARAM;

	totalpe = get_approx_size(size, -1) / PE_SIZE;

	// get the existent snapshots in order
	cnt = Get_Snapshots_On_One_LV_In_Order(src_lv,
		~(FLAG_IMPORTANCE_ASC & FLAG_SPACE_USAGE_ASC),
		&snaplist);
#ifdef DEBUG
printf("share_free_space_among_existent_snapshots(%s, %lld KB) : %d existent snapshots!\n",
src_lv, size, cnt);
#endif
	for (i=0, realcnt=0; i<cnt; i++) {
		if (snaplist[i].ll_Snap_Size<=0) // LV not exist
			continue;
		realcnt++;
	}
	if (realcnt>0)
		avgpe = totalpe / realcnt;
	pcnt = totalpe - avgpe * realcnt;

	for (i=0; i<cnt; i++) {
		if (i<pcnt) peno = avgpe+1;
		else peno = avgpe;
#ifdef DEBUG
loopcnt++;
printf("share_free_space_among_existent_snapshots : loop %d -- %d free PE\n",
loopcnt, totalpe);
#endif
		if (peno<=0) // from now on, no more free PEs to share
			goto share_free_space_among_existent_snapshots_out;

		if (snaplist[i].ll_Snap_Size<=0) // LV not exist
			continue;
#ifdef DEBUG
printf("\t%s gets one free PE!\n", snaplist[i].c_Path);
#endif
		Set_Snapshot_Status(snaplist[i].i_Snapshot_No, S_VOLUME_BUSY);
		ret = extend_space_to_snap_lv(
			snaplist[i].c_Path,
			PE_SIZE * peno);
		Set_Snapshot_Status(snaplist[i].i_Snapshot_No, S_READY);
		if (ret<0) {
#ifdef DEBUG
			printf("share_free_space_among_existent_snapshots"
				"(%s, %lld) : fails to extend_space_to_snap_lv"
				"(%s, %lld), err=%d\n",
				src_lv, size, snaplist[i].c_Path, (long long)PE_SIZE,
				ret);
#endif
//			goto share_free_space_among_existent_snapshots_out;
		}
	}

share_free_space_among_existent_snapshots_out:
	return ret;
}

// <== Cat : for new snap repository size issue
#endif

//------------------------------------------------------------------------
// is_valid_snap_conf
// description : check if the input config matches the record in config file
//------------------------------------------------------------------------
BOOL is_valid_snap_conf(SNAPSHOT_VOLUME_CONF* snapconf)
{
	int ret;
	SNAPSHOT_VOLUME_CONF mysnap;
	ret = Get_One_Snapshot_Volume_Conf(snapconf->i_Snapshot_No, &mysnap);
	if (ret<0) return FALSE;

	if (strcmp(mysnap.c_Snapshot_Path, snapconf->c_Snapshot_Path))
		return FALSE;
	if (strcmp(mysnap.c_Src_LV, snapconf->c_Src_LV))
		return FALSE;
/* Cat : for new snap repository size issue
	if (mysnap.ll_Snapshot_Size!=snapconf->ll_Snapshot_Size)
		return FALSE;
*/
	return TRUE;
}

//------------------------------------------------------------------------
// quick_check_finish
// Output :
//	SUCCESS
//	ERROR_INVALID_PARAM
//	ERROR_ALREADY_EXISTED
//------------------------------------------------------------------------
int quick_check_finish(LVM_VOLUME_INFO* vol_list, int vol_cnt, int raid_level,
	int disk_no, int* min_no, int* real_no)
{
	int ret = SUCCESS;
	int i, min_drive_no, real_drive_no, flag;
	LVM_VOLUME_CONF *vol_data=NULL;

	if (vol_list == NULL || vol_cnt <0 || raid_level < SINGLE
		|| raid_level > RAID5 || min_no == NULL || real_no == NULL)
		return ERROR_INVALID_PARAM;

	switch (raid_level) {
		case SINGLE:
			min_drive_no = 1;

			flag = 0;
			for (i=0; i<vol_cnt; i++) {
				vol_data = &(vol_list[i].vol_data);
				if ((vol_data->raid_level>SINGLE) ||
					(vol_data->status<S_READY))
				{
					flag = 1;
					break;
				}
			}

			if (flag==0) { // all volumes are single
#ifdef DEBUG
				printf("quick_check_finish : All volumes are ready, stop!\n");
#endif
				ret = ERROR_ALREADY_EXISTED;
				goto quick_check_finish_out;
			}
			else
				ret = SUCCESS;
			break;
		case MIRROR:
			min_drive_no = 2;

			for(i=1, real_drive_no=0; i<=disk_no; i++)
			{
				if (Is_HD_ReadWrite_Safe(i))
					real_drive_no++;
			}

			if (real_drive_no<min_drive_no) {
				ret = ERROR_FAIL;
				goto quick_check_finish_out;
			}

			flag = 0;

			for (i=0; i<vol_cnt; i++) {
				vol_data = &(vol_list[i].vol_data);
				if ((vol_data->raid_level!=MIRROR) ||
					((vol_data->raid_level==MIRROR) &&
					(vol_data->status<S_READY)))
				{
					flag = 1;
					break;
				}
			}

			if (flag==0) { // all volumes are RAID-1
#ifdef DEBUG
				printf("quick_check_finish : All volumes are ready, stop!\n");
#endif
				ret = ERROR_ALREADY_EXISTED;
				goto quick_check_finish_out;		
			}
			else
				ret = SUCCESS;
			break;
		case LINEAR:
		case STRIPING:
		case RAID5:
			vol_data = &(vol_list[0].vol_data);

			if ((vol_cnt==1) && (vol_data->raid_level==raid_level) &&
				(vol_data->status>=S_READY) &&
				(vol_data->status!=S_REMOVING_RAID))
			{
#ifdef DEBUG
				printf("quick_check_finish : All volumes are ready, stop!\n");
#endif
				ret = ERROR_ALREADY_EXISTED;
				goto quick_check_finish_out;
			}

			for(i=1, real_drive_no=0; i<=disk_no; i++)
			{
				if (Is_HD_ReadWrite_Safe(i))
					real_drive_no++;
			}
			if (raid_level==RAID5)
				min_drive_no=3;
			else
				min_drive_no=2;

			if (min_drive_no>real_drive_no) {
#ifdef DEBUG
				printf("quick_check_finish : Only %d disks exist, "
					"not enough disks to make RAID-%d\n",
					real_drive_no, raid_level);
#endif
				ret = ERROR_FAIL;
				goto quick_check_finish_out;
			}
			ret = SUCCESS;
			break;
		default: 
			;
	}
quick_check_finish_out:
	*min_no = min_drive_no;
	*real_no = real_drive_no;
	return ret;
}

//------------------------------------------------------------------
// quick_reformat_volume
// Output : error code
//	SUCCESS
//	ERROR_EXEC_FAIL
//------------------------------------------------------------------
int quick_reformat_volume(LVM_VOLUME_INFO *volinfo)
{
	int ret;
	long long sizediff;
	LV_INFO lvinfo;
	PV_INFO pvinfo;

	Set_Volume_Status(volinfo->vol_data.vol_no, S_FORMATTING);

	if (Is_Mounted(volinfo->vol_data.lv_name)
		|| Is_Mounted(volinfo->vol_data.device_name))
	{
		Stop_Services(); // stop network services
		Umount_LVM_Volume(volinfo->vol_data.vol_no);
		Start_Services(); // start network services
	}

	get_lv_info(volinfo->vol_data.lv_name, &lvinfo);
	get_pv_info(volinfo->vol_data.device_name, &pvinfo);

	sizediff = pvinfo.i_PE_Count*pvinfo.ll_PE_Size - lvinfo.ll_LV_Size;
	if (sizediff>0) { // snapshot was enabled, but no more snap instance now (removed)
		char cmd[BUF_SIZE], pvname[HD_DEVICE_NAME_LENGTH];
		char section[BUF_SIZE];

		sprintf(section, "Volume %d", volinfo->vol_data.vol_no);
		Conf_Set_Field(STORAGE_CONF, section, "enable snapshot", "FALSE");
		Conf_Set_Field(STORAGE_CONF, section, "max snap count", "0");
		Conf_Set_Field(STORAGE_CONF, section, "snap repository ratio", "0");

		convert_md_name(volinfo->vol_data.device_name, pvname,
			HD_DEVICE_NAME_LENGTH, 0);
		sprintf(cmd, "%s -A n -L +%lldK %s %s>/dev/null 1>/dev/null 2>/dev/null",
				LV_EXTEND_CMD, sizediff,
				volinfo->vol_data.lv_name, pvname);
		ret = system(cmd);
		if (ret<0) {
#ifdef DEBUG
			printf("quick_reformat_volume(%s) : %d\n\tcmd=%s\n",
				volinfo->vol_data.lv_name, ret, cmd);
#endif
			ret = ERROR_EXEC_FAIL;
			goto quick_reformat_volume_out;
		}
	}
	ret = Format_LVM_Volume(volinfo->vol_data.vol_no);

quick_reformat_volume_out:
	Mount_LVM_Volume(volinfo->vol_data.vol_no);
	Set_Volume_Status(volinfo->vol_data.vol_no, S_READY);
	return ret;
}

//------------------------------------------------------------------
// quick_remove_all_snapshot_volumes
// Output : error code
//	SUCCESS
//	...
//------------------------------------------------------------------
int quick_remove_all_snapshot_volumes()
{
	int snap_cnt=0, i, ret = SUCCESS;
	SNAPSHOT_VOLUME_CONF *snaplist = NULL;

	snap_cnt = Get_All_Snapshot_Volumes_Conf(&snaplist);
	if (snap_cnt<0) {
		ret = snap_cnt;
		goto quick_remove_all_snapshot_volumes_out;
	}

	for (i=0; i<snap_cnt; i++) {
		ret = remove_snapshot_volume_completely(snaplist[i].i_Snapshot_No);
		if (ret<0) {
#ifdef DEBUG
			printf("quick_remove_all_snapshot_volumes : "
				"remove_snapshot_volume_completely(%d) : %d\n",
				snaplist[i].i_Snapshot_No, ret);
#endif
			goto quick_remove_all_snapshot_volumes_out;
		}
	}
quick_remove_all_snapshot_volumes_out:
	if (snap_cnt>0) free(snaplist);

	return ret;
}

//------------------------------------------------------------------
// quick_remove_all_shares
// Output : error code
//	SUCCESS
//	ERROR_INVALID_PARAM
//------------------------------------------------------------------
int quick_remove_all_shares()
{
	SECTION_INFO* sharelist=NULL;
	int ret, sharecnt, i;

	// remove all NAS shares
	sharecnt = Get_NAS_Shares(&sharelist, TRUE);
	for(i=0; i<sharecnt; i++) {
		ret = Remove_NAS_Share(sharelist[i].section_name);
		if (ret<0) {
#ifdef DEBUG
			printf("quick_remove_all_shares : fails to remove (%s)--error=%d\n",
				sharelist[i].section_name, ret);
#endif
			goto quick_remove_all_shares_out;
		}
	}

quick_remove_all_shares_out:
	if (sharecnt>0) free(sharelist);

	return ret;
}

//------------------------------------------------------------------
// quick_remove_old_stuff
// Output : error code
//	SUCCESS
//	ERROR_INVALID_PARAM
//------------------------------------------------------------------
int quick_remove_old_stuff(LVM_VOLUME_INFO* vol_list, int vol_cnt)
{
	int ret = SUCCESS, i, j;
	LVM_VOLUME_CONF *vol_data = NULL;
	LVM_VOLUME_CONF new_vol_data;

	if (vol_list == NULL || vol_cnt<0)
		return ERROR_INVALID_PARAM;

	// remove all old volumes
	for(i=0; i<vol_cnt; i++) {
		vol_data = &(vol_list[i].vol_data);

//		Set_Volume_Status(vol_data->vol_no, S_INITIALIZING);
		Stop_User_Quota(vol_data->vol_no);

		if (Is_Mounted(vol_data->lv_name) || Is_Mounted(vol_data->device_name)) {
			Stop_Services(); // stop network services
			Umount_LVM_Volume(vol_data->vol_no);
			Start_Services(); // start network services
		}

		ret = remove_logical_volume(vol_data->vol_no, vol_data->device_name, FALSE);
		if (ret<0) {
#ifdef DEBUG
			printf("quick_remove_old_stuff : "
				"remove_logical_volume(%d, %s, FALSE): %d\n",
				vol_data->vol_no, vol_data->device_name, ret);
#endif
			Set_Volume_Status(vol_data->vol_no,
				S_READY);
			Mount_LVM_Volume(vol_data->vol_no);
			goto quick_remove_old_stuff_out;
		}

		if ((vol_data->raid_level>SINGLE) &&
			(vol_data->status!=S_NOT_ACTIVE) &&
			(vol_data->status!=S_INITIALIZING))
		{
			Set_Volume_Status(vol_data->vol_no,
				S_REMOVING_RAID);
			ret = Stop_Raid(vol_data->device_name);
			if (ret<0 && ret != ERROR_NO_RAIDDEV) {
#ifdef DEBUG
				printf("quick_remove_old_stuff : "
					"Fails to Stop_Raid(%s):%d\n",
					vol_data->device_name, ret);
#endif
				Set_Volume_Status(vol_data->vol_no,
					S_READY);
				Mount_LVM_Volume(vol_data->vol_no);
				goto quick_remove_old_stuff_out;
			}
		}

		if (vol_data->raid_level>SINGLE) {
			Set_Volume_Status(vol_data->vol_no,
				S_REMOVING_RAID);

			if((ret=Delete_RaidDev(vol_data->device_name))<0 &&
				ret != RAID_DEVICE_NO)
			{
#ifdef DEBUG
				printf("quick_remove_old_stuff : "
					"Fails to Delete_RaidDev(%s)\n",
					vol_data->device_name);
#endif
				Set_Volume_Status(vol_data->vol_no,
					S_RAID_DIED);
				goto quick_remove_old_stuff_out;
			}

			clear_lvm_volume_related_messages(vol_data);

			Remove_LVM_Volume_Record(vol_data->vol_no);

			for(j=0; j<vol_data->list_cnt; j++){
				memset(&new_vol_data, 0, sizeof(LVM_VOLUME_CONF));

				Make_Single_Volume_Conf(&new_vol_data, vol_data->drive_no_list[j], S_INITIALIZING);

				if (Get_HD_RW_Error(new_vol_data.drive_no_list[0])) // Catherine Aug. 2004
					new_vol_data.status = S_RW_ERROR;
				else
				if (Is_HD_ReadWrite_Safe(new_vol_data.drive_no_list[0]))
					new_vol_data.status = S_INITIALIZING;
				else
					new_vol_data.status = S_NOT_EXIST;

				ret = Add_LVM_Volume_Record(&new_vol_data);
				if(ret<0) {
#ifdef DEBUG
					printf("quick_remove_old_stuff : "
						"Fails to Add_LVM_Volume_Record"
						" for drive %d\n",
						vol_data->drive_no_list[j]);
#endif
					goto quick_remove_old_stuff_out;
				}
			}

			for(j=0; j<vol_data->spare_list_cnt; j++){
				memset(&new_vol_data, 0, sizeof(LVM_VOLUME_CONF));

				Make_Single_Volume_Conf(&new_vol_data, vol_data->spare_drive_list[j], S_INITIALIZING);

				if (Get_HD_RW_Error(new_vol_data.drive_no_list[0])) // Catherine Aug. 2004
					new_vol_data.status = S_RW_ERROR;
				else
				if (Is_HD_ReadWrite_Safe(new_vol_data.drive_no_list[0]))
					new_vol_data.status = S_INITIALIZING;
				else
					new_vol_data.status = S_NOT_EXIST;

				ret = Add_LVM_Volume_Record(&new_vol_data);
				if(ret<0) {
#ifdef DEBUG
					printf("quick_remove_old_stuff : "
						"Fails to Add_Volume_Record_Ex"
						" for drive %d\n",
						vol_data->spare_drive_list[j]);
#endif
					goto quick_remove_old_stuff_out;
				}
			}
		}
	}
quick_remove_old_stuff_out:
	return ret;
}

//------------------------------------------------------------------
// quick_init_disks
// Output : error code
//	SUCCESS
//	ERROR_INVALID_PARAM
//------------------------------------------------------------------
int quick_init_disks(LVM_VOLUME_INFO* vol_list, int vol_cnt)
{
	int ret = SUCCESS, i;
	LVM_VOLUME_CONF* vol_data=NULL;

	if (vol_list == NULL || vol_cnt<0)
		return ERROR_INVALID_PARAM;

	for (i=0; i<vol_cnt; i++) {
		vol_data = &(vol_list[i].vol_data);

		if (!Is_HD_ReadWrite_Safe(vol_data->drive_no_list[0])) {
			vol_data->status = S_NOT_EXIST;
			Set_Volume_Status(vol_data->vol_no,
				S_NOT_EXIST);
			continue;
		}
		// Catherine Aug. 2004
		if (Get_HD_RW_Error(vol_data->drive_no_list[0])) {
			vol_data->status = S_RW_ERROR;
			Set_Volume_Status(vol_data->vol_no,
				S_NOT_EXIST);
			continue;
		}

		ret=Verify_Hidden_Conf_File(vol_data->drive_no_list[0]);

		if (ret<0) {
			Set_Volume_Status(vol_data->vol_no,S_INITIALIZING);

			if ((ret=Init_NAS_LVM_Disk(vol_data->drive_no_list[0]))<0){
#ifdef DEBUG
				printf("quick_init_disks : "
					"Init_NAS_LVM_Disk(%d) fails : %d\n",
					vol_data->drive_no_list[0], ret);
#endif
				Set_Volume_Status(vol_data->vol_no,S_UNINITIALIZE);
				goto quick_init_disks_out;
			}
		}
	}
quick_init_disks_out:
	return ret;	
}

//------------------------------------------------------------------
// quick_create_new_volumes
// Output : error code
//	SUCCESS
//	ERROR_INVALID_PARAM
//------------------------------------------------------------------
int quick_create_new_volumes(LVM_VOLUME_INFO* vol_list, int vol_cnt, int raid_level,
		int disk_no, int min_drive_no, int real_drive_no)
{
	int ret = SUCCESS;
	int new_vol_cnt, list_disk_cnt, i, j, prev_vol_last_drive_no, k;
	LVM_VOLUME_CONF* vol_data = NULL;
	LVM_VOLUME_CONF new_vol_data;

	if (vol_list == NULL || vol_cnt<0 || raid_level<SINGLE || raid_level>RAID5)
		return ERROR_INVALID_PARAM;

	switch (raid_level) {
		case SINGLE:
			for (i=0; i<vol_cnt; i++) {
				vol_data = &(vol_list[i].vol_data);

				if (vol_data->status==S_READY || vol_data->status==S_NOT_EXIST
					 || vol_data->status==S_RW_ERROR) // Catherine Aug. 2004
					continue;
				ret = common_init_volume(vol_data->vol_no, vol_data->device_name);
			}
			break;
		case MIRROR:
		case LINEAR:
		case STRIPING:
		case RAID5:
			if (raid_level==MIRROR) {
				new_vol_cnt = disk_no/2;
				list_disk_cnt = 2;
			}
			else {
				new_vol_cnt=1;
				list_disk_cnt = real_drive_no;
			}

			for (i=0, prev_vol_last_drive_no=0; i<new_vol_cnt; i++)
			{
				memset(&new_vol_data, 0, sizeof(LVM_VOLUME_CONF));

				if (disk_no-prev_vol_last_drive_no<min_drive_no) {
					for (j=prev_vol_last_drive_no+1;j<=disk_no;j++) {
						int vol_no = Get_Volume_No_For_Drive(j);

						if (vol_no<0) continue;

						if (S_NOT_EXIST!=Get_Volume_Status(vol_no))
							Set_Volume_Status(vol_no, S_UNINITIALIZE);
					}
					ret = ERROR_FAIL;
					break;
				}

				for (j=prev_vol_last_drive_no+1, k=0, real_drive_no=0;
					k<list_disk_cnt && j<=disk_no; j++, k++)
				{
					int vol_no;
					LVM_VOLUME_CONF tmpvol;

					// Catherine Aug. 2004
					if (!Is_HD_ReadWrite_Safe(j) || Get_HD_RW_Error(j)) {
						k--;
						continue;
					}
					real_drive_no++;
					new_vol_data.drive_no_list[k]=j;

					vol_no = Get_Volume_No_For_Drive(j);

					Get_One_LVM_Volume_Conf(vol_no, &tmpvol);
					clear_lvm_volume_related_messages(&tmpvol);

					Remove_LVM_Volume_Record(vol_no);
				}

				if (real_drive_no<min_drive_no) {
					int status; // Catherine Aug. 2004

					for (k=prev_vol_last_drive_no+1;k<=disk_no;k++) {
						int vol_no = Get_Volume_No_For_Drive(k);

						if (vol_no<0) { // the record has been removed
							memset(&new_vol_data, 0, sizeof(VOLUME_CONF_EX));

							Make_Single_Volume_Conf(&new_vol_data, k, S_UNINITIALIZE);
							if (Get_HD_RW_Error(k)) // Catherine Aug. 2004
								new_vol_data.status = S_RW_ERROR;
							else
							if (Is_HD_ReadWrite_Safe(k))
								new_vol_data.status = S_UNINITIALIZE;
							else
								new_vol_data.status = S_NOT_EXIST;

							ret = Add_LVM_Volume_Record(&new_vol_data);
							if(ret<0) {
#ifdef DEBUG
								printf("quick_create_new_volumes : "
									"Fails to Add_Volume_Record_Ex"
									" for drive %d\n", k);
#endif
							}
							continue;
						}
						else
						if ((status=Get_Volume_Status(vol_no)) != S_NOT_EXIST
							 && status != S_RW_ERROR) // Catherine Aug. 2004
							Set_Volume_Status(vol_no, S_UNINITIALIZE);
					}

					ret = ERROR_FAIL;
					goto quick_create_new_volumes_out;
				}
				prev_vol_last_drive_no = j-1;

				new_vol_data.spare_list_cnt = 0;
				new_vol_data.raid_level = raid_level;
				new_vol_data.list_cnt = real_drive_no;

				ret = common_init_raid_volume(&new_vol_data, FALSE);
			}
			break;
		default: 
			;
	}
quick_create_new_volumes_out:
	return ret;
}

//------------------------------------------------------------------
// create_directory
// Output : error code
//	SUCCESS
//	ERROR_FAIL
//------------------------------------------------------------------
int create_directory(char *full_path)
{
	int NumberOfSlash = 0, i, j, k;
	char szText[512];

	// Do we need to check path name???
	for (i = 0; i < strlen(full_path); i ++) {
		if (full_path[i] == '/')
			NumberOfSlash++;
	}
	if (full_path[strlen(full_path) - 1] == '/') {
		NumberOfSlash--;
		full_path[strlen(full_path) - 1] = 0;
	}
	for (i = 1; i <= NumberOfSlash; i ++) {
		k = 0;
		for (j = 0; j < strlen(full_path); j++) {
			szText[j] = full_path[j];
			if (full_path[j] == '/') {
				k++;
				if (k > i)
					break;
			}
		}
		szText[j] = '\0';
		if (!Is_Existing_Directory(szText)) {
			umask(0000);  // set umask
			if (mkdir(szText, S_IRWXU | S_IRWXG | S_IRWXO) != 0) {
				return ERROR_FAIL;
			}
			else if (Change_File_Attribute(szText, "admin", "everyone") != 0) {
				// return -1; // Do we need to return here???
			}
		}
	}
	return SUCCESS;
}


//------------------------------------------------------------------
// clear_pv_header
// Description : clear the beginning of the PV, and do vgscan again
//		to remove the VG info on the device
// Output : none
//------------------------------------------------------------------
void clear_pv_header(char* pvname)
{
	int fd, ret;
	unsigned char buf[LVM_VGDA_SIZE];

	if ((fd = open(pvname, O_RDWR)) < 0) {
#ifdef DEBUG
		printf("clear_pv_header(%s) fails to open device!\n", pvname);
#endif
		return;
	}
	memset(buf, 0, sizeof(buf));
	if ((ret = write(fd, buf, sizeof(buf))) < sizeof(buf)) {
#ifdef DEBUG
		printf("clear_pv_header(%s) fails to write %d bytes, ret=%d\n",
			pvname, sizeof(buf), ret);
#endif
	}
	close(fd);
}

//------------------------------------------------------------------
// recreate_disk_partitions
// Description : fully initialize a disk
// Output : error code
//------------------------------------------------------------------
int recreate_disk_partitions(int drive_no)
{
	char part_name[HD_DEVICE_NAME_LENGTH];
	char root_name[HD_DEVICE_NAME_LENGTH];
	int ret;
	
	//Umount all partition on disk drive
	Get_Partition_Name(drive_no, ROOT_PART, root_name);//ROOT 
	if(Is_Mounted(root_name)) {
		ret = Unmount_Partition(root_name);
	}

	Get_Partition_Name(drive_no, DATA_PART, part_name);//DATA
	if (Is_Mounted(part_name))
		Unmount_Partition(part_name);

	Stop_Swapping(drive_no);

	Clear_MBR(drive_no);
	Create_NAS_LVM_Partitions(drive_no);
	clear_pv_header(part_name);
	Format_NAS_Partition(drive_no,ROOT_PART, EXT2_FS);
	Mount_NAS_Partition(drive_no,ROOT_PART, EXT2_FS);
	if((ret = Create_Hidden_Conf_File(drive_no))<0) {
#ifdef DEBUG
		printf("recreate_disk_partitions : "
			"Create_Hidden_Conf_File(%d) fails, err=%d\n",
			drive_no, ret);
#endif
		return ret;
	}

	if((ret = Start_Swapping(drive_no))<0) {
#ifdef DEBUG
		printf("recreate_disk_partitions : "
			"Start_Swapping(%d) fails, err=%d\n",
			drive_no, ret);
#endif
		return ret;
	}

	return SUCCESS;
}

int lvm_remove_recursive ( const char *dir)
{
   int ret = 0;
   int size;
   char *file_name = NULL;
   struct stat sb;
   struct dirent *dir_ent = NULL;
   DIR *this_dir = NULL;

   if ( dir != NULL) {
      if ( lstat ( dir, &sb) == 0) {
         /* check beeing called with a file name */
         if (  !S_ISDIR ( sb.st_mode)) unlink ( dir);
         else {
            /* open and read the directory */
            if ( ( this_dir = opendir ( dir)) != NULL) {
               while ( ( dir_ent = readdir ( this_dir)) != NULL && ret == 0) {
                  if ( strcmp ( dir_ent->d_name,".") == 0 ||
                       strcmp ( dir_ent->d_name,"..") == 0) continue;
                  size = strlen (dir) + strlen (dir_ent->d_name) + 3;
                  if ( ( file_name = malloc ( size)) == NULL) {
                     ret = ERROR_OUT_OF_MEMORY;
                     goto lvm_remove_recursive_end;
                  }
                  memset ( file_name, 0, size);
                  snprintf ( file_name, size - 1,
                                        "%s/%s", dir, dir_ent->d_name);
                  if ( lstat ( file_name, &sb) == 0) {
                     if ( S_ISDIR ( sb.st_mode)) {
                         if ( ( ret = lvm_remove_recursive ( file_name)) == 0)
                            rmdir ( file_name);
                     } else unlink ( file_name);
                  }
                  free ( file_name);
               }
               closedir ( this_dir);
               rmdir ( dir);
            } else ret = ERROR_OPEN_FILE;
         }
      }
   }

lvm_remove_recursive_end:

   return ret;
}

void init_lvm_tab()
{
	char *dir = "/etc/lvmtab.d";
	char *lvmtab = "/etc/lvmtab";
	char init_data = 0;
	int ret, fd=-1;

#ifdef DEBUG
	printf("init_lvm_tab : starts!\n"
		"\tstep 1 : remove lvmtab.d and its contents\n");
#endif
	if ((ret = lvm_remove_recursive(dir)) < 0) {
#ifdef DEBUG
		printf("\t\tlvm_remove_recursive (%s) fails, ret = %d\n",
			dir, ret);
#endif
		goto init_lvm_tab_out;
	}
#ifdef DEBUG
	printf("\tstep 2 : remove lvmtab\n");
#endif
	unlink (lvmtab);
#ifdef DEBUG
	printf("\tstep 3 : generate new and empty LVM\n");
#endif
	if ((fd=open(lvmtab, O_CREAT | O_WRONLY | O_TRUNC, 0640))<0) {
#ifdef DEBUG
		printf("\t\t%s create failed!\n", lvmtab);
#endif
	}
	else {
		write (fd, &init_data, 1);
		close(fd);
	}

	mkdir(dir, 0755);

init_lvm_tab_out:
	return;
}

/* ----------------- functions to export ---------------------------- */

/**********************************************************************
** Is_LV_Exist
** Description:	check if the logical volume exists?
** Input:
**		lv_path : the logical volume's path
***********************************************************************/
BOOL Is_LV_Exist(char* lv_path)
{
	LV_INFO lvinfo;

	if (get_lv_info(lv_path, &lvinfo)==SUCCESS)
		return TRUE;
	else
		return FALSE;
}

/**********************************************************************
** Is_PV_Exist
** Description:	check if the physical volume exists?
** Input:
**		pv_path : the physical volume's path
***********************************************************************/
// 2005.04.26, Johnson Cheng
// For SCSI drives w/o larger IDs would not be included in partition table...
/*BOOL Is_PV_Exist(char* pv_path)
{
	char pvname[LV_PATH_MAX_LEN];
	PV_INFO pvinfo;

	convert_md_name(pv_path, pvname, LV_PATH_MAX_LEN, 0);
	if (get_pv_info(pvname, &pvinfo)==SUCCESS)
		return TRUE;
	else
		return FALSE;
}*/

BOOL Is_PV_Exist(char* pv_path)
{
	char pvname[LV_PATH_MAX_LEN], buf[BUF_SIZE];
	char tmpfile[BUF_SIZE], tmp[BUF_SIZE]={"PVstatus.XXXXXX"};
	PV_INFO pvinfo;
	FILE *fp;
	BOOL result = FALSE;
	int drvno;
	BOOL bFixSCSIBug = FALSE;
	int maxdiskcnt = Get_Profile_Integer("Storage", "Disk Drive Number", 0);
	char device[16];
	Conf_Get_Field(DEFAULT_CFG_FILE, "Storage", "Drive 1", device, 16);
	if((maxdiskcnt > 16) && !strncmp(device, "sd" ,2))
		bFixSCSIBug = TRUE;

	convert_md_name(pv_path, pvname, LV_PATH_MAX_LEN, 0);

	if (bFixSCSIBug && !Is_RaidDev_Active(pvname) &&
		((drvno = Get_Drive_No(pv_path)) < 1 || 
		!Is_HD_ReadWrite_Safe(drvno)))
	{
#ifdef DEBUG
		printf("%s -- %s is not an existing device!\n",
			__FUNCTION__, pv_path);
#endif
		return FALSE;
	}

	// is a PV that already belongs to some VG
	if (get_pv_info(pvname, &pvinfo)==SUCCESS)
		return TRUE;
	if(!bFixSCSIBug)
		return FALSE;

	// maybe a new PV w/o VG ?
	mktemp(tmp);
	sprintf(tmpfile, "/tmp/%s", tmp);
	sprintf(buf, "%s %s > %s", PV_STATUS_CMD, pvname, tmpfile);
	system(buf);

	if ((fp=fopen(tmpfile, "r"))==NULL)
		goto is_pv_exist_over;

	memset(buf, 0, BUF_SIZE);
	while (fgets(buf, BUF_SIZE, fp)) {
		if (strstr(buf, "a new physical volume") != NULL) {
			result = TRUE;
			break;
		}
	}

is_pv_exist_over:
	if (fp) fclose(fp);
	unlink(tmpfile);
	return result;
}


/**********************************************************************
** Format_LVM_Volume
** Description:	Format a LVM volume
** Output:	SUCCESS
**		ERROR_FORMAT
***********************************************************************/
int Format_LVM_Volume(int vol_no)
{
	LVM_VOLUME_CONF vol_data;
	int fs_no,ret;
	char vol_path[LV_PATH_MAX_LEN+1];

	if((ret = Get_One_LVM_Volume_Conf(vol_no,&vol_data))== SUCCESS) {
		fs_no = Get_FileSystem_Type();

		if (Is_LVM_Volume(vol_no))
			Get_LV_Path(vol_no, vol_path, LV_PATH_MAX_LEN);
		else
			strcpy(vol_path, vol_data.device_name);

		//chek raid5 ?
		if(vol_data.raid_level == RAID5)
			ret = Format_Partition(vol_path,TRUE,fs_no);
		else
			ret = Format_Partition(vol_path,FALSE,fs_no);
	}
	else
		ret = ERROR_FORMAT;

	return ret;
}

/**********************************************************************
** Scan_LVM_Volume
** Description:	Scan a LVM volume
** Output:	SUCCESS
**		PARTITION_NOT_EXIST
**		ERROR_ALREADY_MOUNTED
**		SCAN_REBOOT
**		SCAN_ERROR
**		SCAN_USAGE
**		SCAN_LIBRARY
**		ERROR_UNKNOWN
***********************************************************************/
int Scan_LVM_Volume(int vol_no,int action)
{
	int fs_no,ret;
	char vol_path[LV_PATH_MAX_LEN+1];

	fs_no = Get_Volume_FileSystem_Type(vol_no); // ReiserFS support

	if (Is_LVM_Volume(vol_no))
		Get_LV_Path(vol_no, vol_path, LV_PATH_MAX_LEN);
	else {
		VOLUME_CONF vol_data;

		ret = Get_One_Volume_Conf(vol_no, &vol_data);
		if (ret<0)
			return ret;

		strcpy(vol_path, vol_data.device_name);
	}

	ret = Scan_Partition(vol_path,action,fs_no);

	return ret;
}

/**********************************************************************
** Mount_LVM_Volume
** Description:	Mount a LVM volume or an old one based on if the LV exists
** Output:	SUCCESS
**		ERROR_MOUNT_FAIL
***********************************************************************/
int Mount_LVM_Volume(int vol_no)
{
	int fs_no,ret;
	char mp_str[BUF_SIZE];
	char vol_path[LV_PATH_MAX_LEN+1];
	LVM_VOLUME_CONF vol_data;

	if((ret = Get_One_LVM_Volume_Conf(vol_no,&vol_data))== SUCCESS) {
		Get_MP_String(vol_data.drive_no_list[0],DATA_PART,mp_str);
		fs_no = Get_Volume_FileSystem_Type(vol_no); // ReiserFS Support
		if (Is_LVM_Volume(vol_no)) {
			Get_LV_Path(vol_no, vol_path, LV_PATH_MAX_LEN);
			ret = Mount_Partition(vol_path,mp_str,fs_no);
		}
		else
			ret = Mount_Partition(vol_data.device_name, mp_str, fs_no);
		if (ret<0) ret = ERROR_MOUNT_FAIL;
	}
	else
		ret = ERROR_MOUNT_FAIL;

	return ret;
}

/**********************************************************************
** Umount_LVM_Volume
** Description:	Unmount a volume
** Output:	SUCCESS
**		ERROR_UMOUNT_FAIL
***********************************************************************/
int Umount_LVM_Volume(int vol_no)
{	
	int ret;
	char vol_path[LV_PATH_MAX_LEN];

	if (Is_LVM_Volume(vol_no)) {
		Get_LV_Path(vol_no, vol_path, LV_PATH_MAX_LEN);
		ret = Unmount_LV(vol_path);
	}
	else {
		VOLUME_CONF vol_data;

		if((ret = Get_One_Volume_Conf(vol_no,&vol_data))== SUCCESS) {
			ret = Unmount_Partition(vol_data.device_name);
		}
		else return ERROR_UMOUNT_FAIL;
	}

	if (ret<0) ret=ERROR_UMOUNT_FAIL;

	return ret;
}

/**********************************************************************
** Mount_Snapshot_Volume
** Description:	Mount a snapshot volume
** Output:	SUCCESS
**		ERROR_MOUNT_FAIL
***********************************************************************/
int Mount_Snapshot_Volume(char* vol_path)
{
	int fs_no,ret;
	char mp_str[BUF_SIZE];

	Get_Snapshot_Mount_Point_String(vol_path,mp_str);

	fs_no = Get_Device_FileSystem_Type(vol_path); // ReiserFS Support

	ret = Mount_Partition(vol_path,mp_str,fs_no);
	if (ret<0) ret = ERROR_MOUNT_FAIL;

	return ret;
}

/**********************************************************************
** Umount_Snapshot_Volume
** Description:	Unmount a snapshot volume
** Output:	SUCCESS
**		ERROR_UMOUNT_FAIL
***********************************************************************/
int Umount_Snapshot_Volume(char *sn_lv)
{	
	int ret=SUCCESS;
	char mp_str[BUF_SIZE], cmd[BUF_SIZE];

	if (!Is_Mounted(sn_lv))
		return SUCCESS;
	ret = Unmount_LV(sn_lv);
	if (ret<0) ret=ERROR_UMOUNT_FAIL;

	Get_Snapshot_Mount_Point_String(sn_lv,mp_str);

	sprintf(cmd, "/bin/rmdir %s > /dev/null 1>/dev/null 2>/dev/null", mp_str);
	system(cmd);

	return ret;
}

/**********************************************************************
** Delete_LVM_Raid_Volume
** Description:	Remove a RAID volume
** Output:	Error code
**		SUCCESS
**		ERROR_FAIL
***********************************************************************/
int Delete_LVM_Raid_Volume(int vol_no)
{
	LVM_VOLUME_CONF raid_vol_data,single_vol_data;
	int ret,i;
	BOOL is_lvm_vol=FALSE, init_error=FALSE;

	ret = Get_One_LVM_Volume_Conf(vol_no,&raid_vol_data);
	if(ret<0) {
#ifdef DEBUG
		printf("Delete_LVM_Raid_Volume : "
			"Get_One_LVM_Volume_Conf(%d) : %d\n", vol_no, ret);
#endif
		return ret;
	}

	if(raid_vol_data.raid_level < LINEAR) {
#ifdef DEBUG
		printf("Delete_LVM_Raid_Volume : "
			"Not a RAID volume!\n");
#endif
		return ERROR_FAIL;
	}

	//removing raid
	Set_Volume_Status(vol_no,S_REMOVING_RAID);

	Stop_User_Quota(vol_no);

	is_lvm_vol = Is_LVM_Volume(vol_no);

	if(raid_vol_data.status != S_NOT_ACTIVE) {
		if (Is_Mounted(raid_vol_data.lv_name)
			|| Is_Mounted(raid_vol_data.device_name))
		{
			Stop_Services(); // stop network services
			Umount_LVM_Volume(vol_no);
			Start_Services(); // restart network services
		}

		if (is_lvm_vol) {
			// remove LV
			ret=remove_logical_volume(
				vol_no,
				raid_vol_data.device_name,
				FALSE);
			if (ret!=SUCCESS) {
#ifdef DEBUG
				printf("Delete_LVM_Raid_Volume : "
					"remove_logical_volume fails,err=%d\n",
					ret);
#endif
				Mount_LVM_Volume(vol_no);
				start_vol_quota(vol_no, 0);
				Set_Volume_Status(vol_no,S_READY);
				return ret;
			}
		}

		sleep(1);

		if((ret=Stop_Raid(raid_vol_data.device_name))<0){
#ifdef DEBUG
			printf("Delete_LVM_Raid_Volume : "
				"Stop_Raid(%s) fails, err=%d\n",
				raid_vol_data.device_name, ret);
#endif
			Mount_LVM_Volume(vol_no);
			start_vol_quota(vol_no, 0);
			Set_Volume_Status(vol_no,S_READY);
			return ret;
		}
	}
	else { // maybe because the VG is down
		int snapcnt;
		SNAPSHOT_VOLUME_CONF *snaplist=NULL;

		snapcnt = Get_Snapshot_Volumes_Conf_For_One_Volume(vol_no, &snaplist);
		if (snapcnt>0) {
			for (i=0; i<snapcnt; i++) {
				// kill all snapshot volume records on this PV
				Remove_Snapshot_Volume_Record(snaplist[i].i_Snapshot_No);
				Remove_One_Snap_Daemon_Conf_Record(snaplist[i].i_Snapshot_No);
			}
			free(snaplist);
		}
	}

	if((ret=Delete_RaidDev(raid_vol_data.device_name))<0) {
#ifdef DEBUG
		printf("Delete_LVM_Raid_Volume : "
			"Delete_RaidDev fails, err=%d\n", ret);
#endif
		return ret;
	}

	clear_lvm_volume_related_messages(&raid_vol_data);

	//remove volume record from storage.conf
	Remove_LVM_Volume_Record(vol_no);

	for(i=0;i<raid_vol_data.list_cnt;i++){
		Make_Single_Volume_Conf(&single_vol_data, raid_vol_data.drive_no_list[i], S_INITIALIZING);
		ret = Add_LVM_Volume_Record(&single_vol_data);
		if(ret<0)
			return ret;
	}

	for(i=0;i<raid_vol_data.spare_list_cnt;i++){
		Make_Single_Volume_Conf(&single_vol_data, raid_vol_data.spare_drive_list[i], S_INITIALIZING);
		ret = Add_LVM_Volume_Record(&single_vol_data);
		if(ret<0)
			return ret;
	}

	//reinitialize single disk volume belong to this volume
	for(i=0;i<raid_vol_data.list_cnt;i++){
		Get_Partition_Name(raid_vol_data.drive_no_list[i],DATA_PART,single_vol_data.device_name);
		single_vol_data.list_cnt = 1;
		single_vol_data.drive_no_list[0] = raid_vol_data.drive_no_list[i];
		single_vol_data.spare_list_cnt = 0;
		single_vol_data.raid_level = SINGLE;
		ret = Create_LVM_Volume(&single_vol_data);
		if(ret<0) {
#ifdef DEBUG
			printf("Delete_LVM_Raid_Volume : "
				"fails to Create_LVM_Volume for drive %d, err=%d\n",
				raid_vol_data.drive_no_list[i], ret);
#endif
			// Catherine Aug. 2004
			//goto delete_lvm_raid_volume_init_single_fail;
			init_error=TRUE;
		}
		sleep(1);
	}

	for(i=0;i<raid_vol_data.spare_list_cnt;i++){
		Get_Partition_Name(raid_vol_data.spare_drive_list[i],
			DATA_PART,single_vol_data.device_name);
		single_vol_data.list_cnt = 1;
		single_vol_data.drive_no_list[0] = raid_vol_data.spare_drive_list[i];
		single_vol_data.spare_list_cnt = 0;
		single_vol_data.raid_level = SINGLE;
		ret = Create_LVM_Volume(&single_vol_data);
		if(ret<0) {
#ifdef DEBUG
			printf("Delete_LVM_Raid_Volume : "
				"fails to Create_LVM_Volume for drive %d, err=%d\n",
				raid_vol_data.spare_drive_list[i], ret);
#endif
			// Catherine Aug. 2004
			//goto delete_lvm_raid_volume_init_single_fail;
			init_error=TRUE;
		}
		sleep(1);
	}
	if (init_error) { // Catherine Aug. 2004
		ret = SUCCESS;
		goto delete_lvm_raid_volume_init_single_fail;
	}

	return SUCCESS;

delete_lvm_raid_volume_init_single_fail:
	for(i=0;i<raid_vol_data.list_cnt;i++)
	{
		int volno, status;

		volno = Get_Volume_No_For_Drive(raid_vol_data.drive_no_list[i]);
		status = Get_Volume_Status(volno);
		if (status>S_READY)
			Set_Volume_Status(volno, S_UNINITIALIZE);
	}
	
	for(i=0;i<raid_vol_data.spare_list_cnt;i++)
	{
		int volno, status;

		volno = Get_Volume_No_For_Drive(raid_vol_data.spare_drive_list[i]);
		status = Get_Volume_Status(volno);
		if (status>S_READY)
			Set_Volume_Status(volno, S_UNINITIALIZE);
	}
	return ret;	
}

/**********************************************************************
** Create_Snapshot_Volume
** Description:	Create a snapshot volume
** Input:
**		sn_vol :	snapshot volume info
** Output:	Error code
**		SUCCESS
**		ERROR_FAIL
***********************************************************************/
int Create_Snapshot_Volume(SNAPSHOT_VOLUME_CONF *sn_vol)
{
	char pv_path[HD_DEVICE_NAME_LENGTH];
	int ret;//, src_vol_no;
	long long apsize;//, maxsize, minsize;

	Set_Snapshot_Status(sn_vol->i_Snapshot_No, S_INITIALIZING);
/* Cat : for new snap repository size issue
	// invalid inputs
	if (sn_vol->ll_Snapshot_Size<=0 || !is_valid_snap_conf(sn_vol))
	{
		Remove_Snapshot_Volume_Record(sn_vol->i_Snapshot_No);
		return ERROR_INVALID_PARAM;
	}
*/
	// is source a valid volume?
	if (!Is_LV_Exist(sn_vol->c_Src_LV)) {
		Remove_Snapshot_Volume_Record(sn_vol->i_Snapshot_No);
		return ERROR_NOT_EXISTED;
	}
	else {
		int volno = Get_LV_Volume_No(sn_vol->c_Src_LV);
		LVM_VOLUME_INFO lvinfo;

		ret = Get_One_LVM_Volume_Info(volno, &lvinfo);
		if (ret<0) return ret;

		if (lvinfo.vol_data.status != S_READY
			&& lvinfo.vol_data.status != S_DEGRADED
			&& lvinfo.vol_data.status != S_REBUILDING_RAID)
		{
#ifdef DEBUG
			printf("Create_Snapshot_Volume : "
				"source volume is either not READY, DEGRADED, nor REBUILDING\n");
#endif
			Remove_Snapshot_Volume_Record(sn_vol->i_Snapshot_No);
			return ERROR_INVALID_PARAM;
		}
		strcpy(pv_path, lvinfo.vol_data.device_name);
	}

	apsize = get_avail_snap_vol_size(sn_vol->c_Src_LV, TRUE);
	if (apsize<0) {
#ifdef DEBUG
		printf("Create_Snapshot_Volume : "
			"get_avail_snap_vol_size(%s, TRUE) fails, err=%d\n",
			sn_vol->c_Src_LV, (int)apsize);
#endif
		Remove_Snapshot_Volume_Record(sn_vol->i_Snapshot_No);
		return apsize;
	}
	else if (apsize==0) {
#ifdef DEBUG
		printf("Create_Snapshot_Volume : "
			"not enough free space (%d%%) for the new snap instance!\n",
			SNAP_MIN_SIZE_RATIO);
#endif
		Remove_Snapshot_Volume_Record(sn_vol->i_Snapshot_No);
		return ERROR_OUT_OF_SNAP_REPOSITORY_SPACE;
	}
// <== Cat : for new snap repository size issue

	Set_Snapshot_Status(sn_vol->i_Snapshot_No, S_CREATING_SNAPSHOT);

	ret = create_logical_volume_directly(sn_vol->c_Snapshot_Path, pv_path,
			apsize, TRUE, sn_vol->c_Src_LV);
	if (ret<0) {
		// undo what has been done....
#ifdef DEBUG
		printf("Create_Snapshot_Volume : "
			"create_logical_volume_directly fails, err=%d\n", ret);
#endif
		Remove_Snapshot_Volume_Record(sn_vol->i_Snapshot_No);
		return ret;
	}

	ret = Mount_Snapshot_Volume(sn_vol->c_Snapshot_Path);

	if (ret<0)
		Set_Snapshot_Status(sn_vol->i_Snapshot_No, S_NOT_MOUNTED);
	else
		Set_Snapshot_Status(sn_vol->i_Snapshot_No, S_READY);

	ret = create_snap_share(sn_vol->c_Snapshot_Path);
	return ret;
}

/**********************************************************************
** Create_NAS_LVM_Partitions
** Description:	Remove a RAID volume
** Output:	Error code
**		SUCCESS
**		ERROR_FAIL
***********************************************************************/
int Create_NAS_LVM_Partitions(int drive_no)
{
	PARTITION_DATA size[3];
	char device_name[HD_DEVICE_NAME_LENGTH];
	int ret = 1,retry = 3;

	//Check HD is exist
	ret = Get_HD_DevName(drive_no,device_name);
	if (ret <=0) return ERROR_Get_HDev_Name;

	//prepare partition data
	size[0].part_no = 1;
	size[0].is_boot = 1;
	size[0].sys_id = 0x83;//ROOT PART
	size[0].capacity_val = 200;
	size[0].capacity_format = SIZE_M;
	
	size[1].part_no = 2;
	size[1].is_boot = 0;
	size[1].sys_id = 0x82;//SWAP PART
	size[1].capacity_val = 64;
	size[1].capacity_format = SIZE_M;
	
	size[2].part_no = 3; //DATA PART
	size[2].is_boot = 0;
	size[2].sys_id = 0x8e; // Linux LVM
	size[2].capacity_val = 0;
	size[2].capacity_format = SIZE_M;

	if(Get_Profile_Boolean(NASCONF_STORAGE_SECTION,NASCONF_DEBUG,FALSE)) {
		int disk_size;
		disk_size = Get_Profile_Integer(NASCONF_STORAGE_SECTION, "NAS DEBUG DISK SIZE", 100);
		size[2].capacity_val = disk_size;
	}

	while(retry--) {
		ret = Delete_NAS_Partitions(drive_no);
		if(ret == SUCCESS ) break;
		sleep(drive_no);
	}

	retry = 3;
	while(retry--) {
		ret = Create_Partitions(device_name,size,3);
		if(ret == SUCCESS ) break;
		sleep(drive_no);
	}

	return ret;
}

/**********************************************************************
** Init_NAS_LVM_Disk
** Description:	Initialize a disk
** Output:	Error code
**		SUCCESS
**		ERROR_FAIL
**		ERROR_UMOUNT_FAIL
**		ERROR_MOUNT_FAIL
** Remarks: Init NAS disk include following step:
** 1. unmount all partitions on disk
** 2. Stop swapping
** 3. remove related PV
** 4. create partitions(root,swap,data)
** 5. format the root partition
** 6. init swap(Init_Swapping(),Start_Swapping())
** 7. mount root part and create .conf on it
***********************************************************************/
int Init_NAS_LVM_Disk(int drive_no)
{
	char part_name[HD_DEVICE_NAME_LENGTH];
	char lv_name[LV_PATH_MAX_LEN];
	char root_name[HD_DEVICE_NAME_LENGTH];
	char sharename[SHARE_NAME_LENGTH];
	char cmd[BUF_SIZE];
	int ret, snapcnt, k;
	SNAPSHOT_VOLUME_CONF *snaplist=NULL;

	//Check HD is exist
	if (!Is_HD_ReadWrite_Safe(drive_no)) {
#ifdef DEBUG
		printf("Init_NAS_LVM_Disk : HD %d not exist!\n",
			drive_no);
#endif
		ret = ERROR_NOT_EXISTED;
		goto init_nas_lvm_disk_out;
	}

	Get_Partition_Name(drive_no,DATA_PART,part_name);//data 
	if (!Is_PV_Exist(part_name)) {
		// may be really uninitialized or VG not active

		// remove potential snapshots
		Get_LV_Name_For_Device(part_name, lv_name, LV_PATH_MAX_LEN);

		snapcnt = Get_Snapshot_Volumes_Conf_On_One_LV(lv_name, &snaplist);
		if (snapcnt<0) {
#ifdef DEBUG
			printf("Init_NAS_LVM_Disk : "
				"Get_Snapshot_Volumes_Conf_On_One_LV(%s) fails, ret=%d\n",
				lv_name, snapcnt);
#endif
			ret = snapcnt;
			goto init_nas_lvm_disk_out;
		}
		for (k=0; k<snapcnt; k++) {
			// kill all snapshot volume records on this PV

			Remove_Snapshot_Volume_Record(snaplist[k].i_Snapshot_No);
			Remove_One_Snap_Daemon_Conf_Record(snaplist[k].i_Snapshot_No);
			Get_Snapshot_Share_Name(snaplist[k].c_Snapshot_Path, sharename, SHARE_NAME_LENGTH);
			Remove_NAS_Share(sharename);
		}
		if (snapcnt>0) free(snaplist);

		ret = recreate_disk_partitions(drive_no);
	}
	else { // PV exists, already initialized once
		PV_INFO pvinfo;
		char vgname[LV_PATH_MAX_LEN];
		char *p;

		get_pv_info(part_name, &pvinfo);
#ifdef DEBUG
		printf("Init_NAS_LVM_Disk : "
			"PV %s exists with VG = %s\n",
			part_name, pvinfo.c_VG_Name);
#endif
		// is this PV's VG valid?  -- Catherine 2003/04/04 ==>
		Get_VG_Name_By_Device(part_name, vgname, sizeof(vgname));

		if ((p=strrchr(vgname, '/')) == NULL)
			p = vgname;
		else
			p++;

		if (is_vg_exist(vgname) &&
			((strcmp(pvinfo.c_VG_Name, vgname) == 0) ||
			(strcmp(pvinfo.c_VG_Name, p) == 0)))
		{

			if (pvinfo.i_Allocated_PE_Count > 0) {
				// somebody (ex. my LV or snapshots) might be using this PV
				Get_LV_Name_By_VG(pvinfo.c_VG_Name, lv_name, sizeof(lv_name));
				ret = remove_logical_volume_2(lv_name, part_name, FALSE);
				if (ret<0) {
#ifdef DEBUG
					printf("\tremove_logical_volume_2(%s, %s, FALSE) fails, err=%d\n",
						lv_name, part_name, ret);
#endif
					goto init_nas_lvm_disk_pv_exist_fail;
				}
			}

			ret = remove_physical_volume(part_name);
			if (ret<0 && ret!=ERROR_NOT_EXISTED) {
#ifdef DEBUG
				printf("Init_NAS_LVM_Disk : "
					"remove_physical_volume(%s) fails, err=%d\n",
					part_name, ret);
#endif
				goto init_nas_lvm_disk_pv_exist_fail;
			}
		}
		else // not valid PV data -- erase it
			clear_pv_header(part_name);
		// <== Catherine 2003/04/04

		if (Verify_Hidden_Conf_File(drive_no)!=SUCCESS) {
			Get_Partition_Name(drive_no,ROOT_PART,root_name);//ROOT 
			if (!Is_Mounted(root_name)) {
				ret = Mount_NAS_Partition(drive_no, ROOT_PART, EXT2_FS);

				if (ret<0) {
					Format_NAS_Partition(drive_no,ROOT_PART, EXT2_FS);
					Mount_NAS_Partition(drive_no, ROOT_PART, EXT2_FS);
				}
			}

			if((ret = Create_Hidden_Conf_File(drive_no))<0) {
#ifdef DEBUG
				printf("Init_NAS_LVM_Disk : "
					"Create_Hidden_Conf_File(%d) fails, err=%d\n",
					drive_no, ret);
#endif
				goto init_nas_lvm_disk_out;
			}
		}
		ret = Start_Swapping(drive_no);
		goto init_nas_lvm_disk_out;

init_nas_lvm_disk_pv_exist_fail:
		// remove in violent ways -> inactivate VG & fdisk
		sprintf(cmd, "%s -An -an %s>/dev/null 1>/dev/null 2>/dev/null",
			VG_ACTIVATE_CMD, pvinfo.c_VG_Name);
		system(cmd);
		ret = recreate_disk_partitions(drive_no);
#ifdef DEBUG
		printf("\tafter recreate_disk_partitions(%d) : %d\n",
			drive_no, ret);
#endif
		sprintf(cmd, "%s>/dev/null 1>/dev/null 2>/dev/null",
			LVM_INIT_CMD);
		system(cmd);
	}
init_nas_lvm_disk_out:
	return ret;
}

/**********************************************************************
** Clear_NAS_Disk_Content
** Description:	Clear all contents (partitions, PV, VG...) of a disk violently
** Output:	Error code
**		SUCCESS
**		ERROR_FAIL
***********************************************************************/
int Clear_NAS_Disk_Content(int drive_no)
{
	char part_name[HD_DEVICE_NAME_LENGTH];
	char root_name[HD_DEVICE_NAME_LENGTH];
	int ret, snapcnt, k;
	SNAPSHOT_VOLUME_CONF *snaplist=NULL;

	//Check HD is exist
	if (!Is_HD_ReadWrite_Safe(drive_no)) {
#ifdef DEBUG
		printf("Clear_NAS_Disk_Content : HD %d not exist!\n",
			drive_no);
#endif
		ret = ERROR_NOT_EXISTED;
		goto clear_nas_disk_content_out;
	}

	Get_Partition_Name(drive_no,DATA_PART,part_name);//data 
	if (!Is_PV_Exist(part_name)) {
		// may be really uninitialized or VG not active
		char lv_name[LV_PATH_MAX_LEN];

		// remove potential snapshots
		Get_LV_Name_For_Device(part_name, lv_name, LV_PATH_MAX_LEN);

		snapcnt = Get_Snapshot_Volumes_Conf_On_One_LV(lv_name, &snaplist);
		if (snapcnt<0) {
#ifdef DEBUG
			printf("Clear_NAS_Disk_Content : "
				"Get_Snapshot_Volumes_Conf_On_One_LV(%s) fails, ret=%d\n",
				lv_name, snapcnt);
#endif
			ret = snapcnt;
			goto clear_nas_disk_content_out;
		}
		for (k=0; k<snapcnt; k++) {
			// kill all snapshot volume records on this PV
			Remove_Snapshot_Volume_Record(snaplist[k].i_Snapshot_No);
			Remove_One_Snap_Daemon_Conf_Record(snaplist[k].i_Snapshot_No);
		}
		if (snapcnt>0) free(snaplist);

		ret = recreate_disk_partitions(drive_no);
	}
	else { // PV exists, already initialized once
		clear_pv_header(part_name);

		if (Verify_Hidden_Conf_File(drive_no)!=SUCCESS) {
			Get_Partition_Name(drive_no,ROOT_PART,root_name);//ROOT 
			if (!Is_Mounted(root_name)) {
				ret = Mount_NAS_Partition(drive_no, ROOT_PART, EXT2_FS);

				if (ret<0) {
					Format_NAS_Partition(drive_no,ROOT_PART, EXT2_FS);
					Mount_NAS_Partition(drive_no, ROOT_PART, EXT2_FS);
				}
			}

			if((ret = Create_Hidden_Conf_File(drive_no))<0) {
#ifdef DEBUG
				printf("Clear_NAS_Disk_Content : "
					"Create_Hidden_Conf_File(%d) fails, err=%d\n",
					drive_no, ret);
#endif
				goto clear_nas_disk_content_out;
			}
		}
		ret = Start_Swapping(drive_no);
	}

clear_nas_disk_content_out:
	return ret;
}


/**********************************************************************
** Activate_LVM
** Description:	Activate all VGs
** Output:	Error code--
**		SUCCESS
**		ERROR_FAIL
**********************************************************************/
int Activate_LVM()
{
	char cmd[BUF_SIZE];
	int ret;

	sprintf(cmd, "%s -ay > /dev/null 1>/dev/null 2>/dev/null", VG_ACTIVATE_CMD);
	ret=system(cmd);
	if (!ret)
		return SUCCESS;

	return ERROR_FAIL;
}

/**********************************************************************
** Deactivate_LVM
** Description:	Deactivate all VGs
** Output:	Error code--
**		SUCCESS
**		ERROR_FAIL
**********************************************************************/
int Deactivate_LVM()
{
	char cmd[BUF_SIZE];
	int ret;

	sprintf(cmd, "%s -an > /dev/null 1>/dev/null 2>/dev/null", VG_ACTIVATE_CMD);
	ret=system(cmd);
	if (!ret)
		return SUCCESS;

	return ERROR_FAIL;
}

/**********************************************************************
** LVM_Disk_Hot_Unplug
** Description:	doing jobs before hot unplug a disk
** Output:	Error code--
**		SUCCESS
**		ERROR_FAIL
**********************************************************************/
int LVM_Disk_Hot_Unplug(int drive_no)
{
	FILE *fp;
	int ret;
	char tmpfile[BUF_SIZE], tmp[BUF_SIZE];

	if (drive_no > MAX_DRIVE_NO)
		return ERROR_INVALID_PARAM;

	get_drive_alphabetic_order(drive_no, 'A', tmp, sizeof(tmp));
	sprintf(tmpfile, "/tmp/HD%s", tmp);
	fp = fopen(tmpfile, "w+");

	if (fp==NULL)
		return ERROR_OPEN_FILE;
	fclose(fp);

	ret = Disable_HD_DMA(drive_no);
	return ret;
}

/**********************************************************************
** LVM_Disk_Hot_Plug
** Description:	doing jobs before hot unplug a disk
** Output:	Error code--
**		SUCCESS
**		ERROR_FAIL
**********************************************************************/
int LVM_Disk_Hot_Plug(int drive_no)
{
	int ret;
	char tmpfile[BUF_SIZE], tmp[BUF_SIZE];

	if (drive_no > MAX_DRIVE_NO)
		return ERROR_INVALID_PARAM;

	get_drive_alphabetic_order(drive_no, 'A', tmp, sizeof(tmp));
	sprintf(tmpfile, "/tmp/HD%s", tmp);
	unlink(tmpfile);

	ret = Enable_HD_DMA(drive_no);

	return ret;
}

/**********************************************************************
** Unmount_LV
** Description:	unmount a logical volume
** Output:	Error code--
**		SUCCESS
**		ERROR_FAIL
**		ERROR_NOT_EXISTED
**********************************************************************/
int Unmount_LV(char *lv_name)
{
	char  cmd_line[BUF_SIZE];
    	int   ret;
    	
   	if(!Is_LV_Exist(lv_name)) return ERROR_NOT_EXISTED;
   	if(!Is_Mounted(lv_name)) return SUCCESS;
    	
    	sprintf(cmd_line,"%s %s >/dev/null 2>/dev/null 1>/dev/null",CMD_UMOUNT,lv_name);
	
	ret = system(cmd_line);
	if (ret != 0) return ERROR_FAIL;
	return SUCCESS;
}
/**********************************************************************
** Init_LVM
** Description:	initialize LVM
** Output:	Error code--
**		SUCCESS
**		ERROR_FAIL
**********************************************************************/
int Init_LVM()
{
	char cmd[BUF_SIZE];
	LVM_VOLUME_CONF *volcnf = NULL;
	int i, cnt;
	int ret;
#ifdef DEBUG
	printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif
	init_lvm_tab();
#ifdef DEBUG
	printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif
	if ((cnt = Get_All_LVM_Volumes_Conf(&volcnf)) < 0) return cnt;
#ifdef DEBUG
	printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif
	for (i=0; i<cnt; i++) {
		if ((volcnf[i].raid_level == SINGLE &&
			Verify_Hidden_Conf_File(volcnf[i].drive_no_list[0]) == SUCCESS) ||
		    Is_RaidDev_Active(volcnf[i].device_name))
		{
			sprintf(cmd, "%s -l %s ",
				LVM_INIT_CMD, volcnf[i].device_name);
#ifdef DEBUG
			printf("%s -- init command = %s\n", __FUNCTION__, cmd);
#else
			strcat(cmd, "> /dev/null 1>/dev/null 2>/dev/null");
#endif
			if ((ret = system(cmd)) != 0) {
				ret = ERROR_FAIL;
#ifdef DEBUG
				printf("%s -- line %d\n", __FUNCTION__, __LINE__);
#endif
				goto init_lvm_out;
			}
		}
	}			
init_lvm_out:
#ifdef DEBUG
	printf("%s -- line %d with ret = %d\n", __FUNCTION__, __LINE__, ret);
#endif
	if (volcnf) free(volcnf);
	return ret;
}
/**********************************************************************
** Create_LVM_Volume
** Description:	Create a volume
** Input:
**		lv :	LVM volume config data;
**			only "raid_level", "list_cnt",
**			"drive_no_list", "spare_list_cnt",
**			"spare_drive_list" are needed
** Output:	Error code
**		SUCCESS
**		ERROR_FAIL
***********************************************************************/
int Create_LVM_Volume(LVM_VOLUME_CONF *lv)
{
	int ret;

	if (lv->raid_level==SINGLE) {
		int vol_no = Get_Volume_No_For_Drive(lv->drive_no_list[0]);
		BOOL rwerror=Get_HD_RW_Error(lv->drive_no_list[0]);

		if (vol_no>0) {
			ret = Remove_LVM_Volume_Record(vol_no);
			if (ret<0) {
#ifdef DEBUG
				printf("Create_LVM_Volume : "
					"remove old volume record %d faild, "
					"err=%d\n",
					vol_no, ret);
#endif
				return ret;
			}
		}
		if (rwerror) // Catherine Aug. 2004
			lv->status = S_RW_ERROR;
		else
			lv->status = S_UNINITIALIZE;
		ret = Get_Partition_Name(lv->drive_no_list[0],DATA_PART,lv->device_name);
		if (ret<0) return ret;
		ret = Add_LVM_Volume_Record(lv);

		// Catherine Aug. 2004 : is this disk marked as "r/w error" once?
		if (rwerror) return ERROR_IO_ERROR;

		ret = init_lvm_single_disk_volume(lv);
	}
	else {
		ret = create_lvm_raid_volume(lv);
	}

	return ret;
}

/**********************************************************************
** Activate_LV
** Description:	Activate or deactivate a logical volume.
**		For deactivating, if any related snapshot volumes exist,
**		remove them.
** Input:
**		lv_path :	the LV name
**		active :	TRUE to activate, FALSE to deactivate
** Output:	Error code
**		SUCCESS
**		ERROR_FAIL
***********************************************************************/
int Activate_LV(char* lv_path, BOOL active)
{
	char cmd[BUF_SIZE];
	int ret;

	if (active) {
		sprintf(cmd, "%s -a y %s> /dev/null",
			LV_ACTIVATE_CMD, lv_path);
	}
	else {
		int sn_cnt, i, vol_no;
		SNAPSHOT_VOLUME_CONF *sn_vols=NULL;

		vol_no = Get_LV_Volume_No(lv_path);

		// remove all related snapshot volumes before deactivate
		sn_cnt = Get_Snapshot_Volumes_Conf_For_One_Volume(vol_no, &sn_vols);
		if (sn_cnt<0) return sn_cnt;

		for (i=0; i<sn_cnt; i++) {
			ret = Delete_Snapshot_Volume(sn_vols[i].i_Snapshot_No);
			if (ret<0) {
				if (sn_vols) free(sn_vols);
				return ret;
			}
		}

		if (sn_vols) free(sn_vols);

		sprintf(cmd, "%s -a n %s> /dev/null 1>/dev/null 2>/dev/null",
			LV_ACTIVATE_CMD, lv_path);
	}

	ret=system(cmd);
	if (!ret)
		return SUCCESS;

	return ERROR_FAIL;

}

/**********************************************************************
** Get_Snapshot_Volume_Usage_Rate
** Description:	Get the usage rate of a snapshot volume
**		(for resizing decision making)
** Output:	Usage rate or the error code ...
**		SUCCESS
**		ERROR_FAIL
***********************************************************************/
int Get_Snapshot_Volume_Use_Rate(char *snap_lv)
{
	int lv_handle = -1;
	int ret = 0;
	lv_snapshot_use_rate_req_t lv_snapshot_use_rate_req;

	if (!Is_LV_Exist(snap_lv))
		return ERROR_NOT_EXISTED;

	if ((lv_handle = open(snap_lv, O_RDONLY))==-1) {
		Mount_Snapshot_Volume(snap_lv);
		return ERROR_OPEN_FILE;
	}

	lv_snapshot_use_rate_req.rate = 0;
	lv_snapshot_use_rate_req.block = O_NONBLOCK; // get
	ret = ioctl ( lv_handle, LV_SNAPSHOT_USE_RATE, &lv_snapshot_use_rate_req);
	close(lv_handle);
	if (ret == 0)
		ret = lv_snapshot_use_rate_req.rate;
	else
		ret = ERROR_FAIL;
	return ret;
}

/**********************************************************************
** Resize_Snapshot_Volume_By_Rate
** Description:	Get the usage rate of a snapshot volume
**		(for resizing decision making)
**		snap_lv :	the lv path of a snapshot volume instance
**		rate :		the ratio == (new_size / old_size)
** Output:	Error code ...
**		SUCCESS
**		ERROR_FAIL
***********************************************************************/
/* Cat : for new snap repository size issue
int Resize_Snapshot_Volume_By_Rate(char* snap_lv , int rate)
{
	int ret, sn_vol_no;
	long long diffsize;
	char cmd[BUF_SIZE];
	SNAPSHOT_VOLUME_CONF snapconf;

	sn_vol_no = Get_Snapshot_Volume_No(snap_lv);
	if (sn_vol_no<0) return sn_vol_no;

	ret = Get_One_Snapshot_Volume_Conf(sn_vol_no, &snapconf);
	if (ret<0) return ret;

	diffsize = abs(rate-100)*snapconf.ll_Snapshot_Size/100;

	if (rate>100) {
		char pv_name[LV_PATH_MAX_LEN];

		ret = borrow_space_from_lv(snapconf.c_Src_LV, diffsize,
			pv_name, LV_PATH_MAX_LEN);
		if (ret<0) {
#ifdef DEBUG
			printf("Resize_Snapshot_Volume_By_Rate : "
				"borrow_space_from_lv fails, err=%d!\n",
				ret);
#endif
			return ret;
		}


		if (snapconf.b_Is_Scheduled_Snap) {
			ret = Share_Space_Among_All_Snapshot_Instances(sn_vol_no, diffsize);
		}
		else {
			sprintf(cmd, "%s -A y -L +%lldK %s %s>/dev/null 1>/dev/null 2>/dev/null",
				LV_EXTEND_CMD, diffsize, snap_lv, pv_name);

			Umount_Snapshot_Volume(snap_lv);
			if (system(cmd)) {
				Mount_Snapshot_Volume(snap_lv);
				return ERROR_FAIL;
			}

			ret = Mount_Snapshot_Volume(snap_lv);
		}
	}
	else {
		if (snapconf.b_Is_Scheduled_Snap) {
			ret = Share_Space_Among_All_Snapshot_Instances(sn_vol_no, (-1)*diffsize);
		}
		else {
			sprintf(cmd, "%s -A y -L -%lldK -f %s>/dev/null 1>/dev/null 2>/dev/null",
				LV_REDUCE_CMD, diffsize, snap_lv);
			Umount_Snapshot_Volume(snap_lv);
			if ((ret = system(cmd))!=0) {
#ifdef DEBUG
				printf("Resize_Snapshot_Volume_By_Rate : %d!\n"
				"\t(%s)\n", ret, cmd);
#endif

				Mount_Snapshot_Volume(snap_lv);
				return ERROR_FAIL;
			}

			ret = Mount_Snapshot_Volume(snap_lv);
		}

		ret = give_back_space_to_lv(snapconf.c_Src_LV, diffsize);
	}

	if (ret<0) return ret;

	ret = Set_Snapshot_Volume_Size(sn_vol_no, diffsize+snapconf.ll_Snapshot_Size);
	return SUCCESS;
}
*/

/**********************************************************************
** Delete_Snapshot_Volume
** Description:	Remove a snapshot volume
** Output:	Error code
**		SUCCESS
**		ERROR_UMOUNT_FAIL
**		ERROR_FAIL
**		ERROR_BUSY
***********************************************************************/
int Delete_Snapshot_Volume(int snapshot_vol_no)
{
	int ret, srcvolno, i=0;
	SNAPSHOT_VOLUME_INFO sninfo;
	char pvpath[LV_PATH_MAX_LEN];
#ifndef __EXT3_SNAP__
	long long left_size = 0;
#endif
	ret = Get_One_Snapshot_Volume_Info(snapshot_vol_no, &sninfo);
	if (ret<0) {
#ifdef DEBUG
		printf("Delete_Snapshot_Volume : "
			"Get_One_Snapshot_Volume_Conf fails, err=%d\n", ret);
#endif
		return ret;
	}

	ret = Set_Snapshot_Status(snapshot_vol_no, S_REMOVING_SNAPSHOT);
	srcvolno = Get_LV_Volume_No(sninfo.snap_conf.c_Src_LV);
	Get_PV_Path(srcvolno, pvpath, LV_PATH_MAX_LEN);

	if (sninfo.snap_conf.b_Is_Scheduled_Snap == FALSE) {
		if (sninfo.snap_conf.Restore_Status[0]==s_BUSY) {
#ifdef DEBUG
			printf("Delete_Snapshot_Volume : "
				"%s is busy restoring!\n",
				sninfo.snap_conf.c_Snapshot_Path);
#endif
			ret = ERROR_BUSY;
			Mount_Snapshot_Volume(sninfo.snap_conf.c_Snapshot_Path);
			Set_Snapshot_Status(snapshot_vol_no, S_READY);
			return ret;
		}

		ret = do_remove_snap_lv(sninfo.snap_conf.c_Snapshot_Path, pvpath);
		if (ret<0) {
#ifdef DEBUG
			printf("Delete_Snapshot_Volume : "
				"do_remove_snap_lv(%s, %s) fails, err=%d!\n",
				sninfo.snap_conf.c_Snapshot_Path,
				pvpath, ret);
#endif
			Mount_Snapshot_Volume(sninfo.snap_conf.c_Snapshot_Path);
			Set_Snapshot_Status(snapshot_vol_no, S_READY);
			return ret;
		}
#ifndef __EXT3_SNAP__
		left_size = sninfo.ll_Snapshot_Size;
#endif
	}
	else {
		SNAPSHOT_VOLUME_CONF snapconf;
#ifndef __EXT3_SNAP__
		left_size = 0;
#endif
		for (i=0; i < sninfo.i_Current_Instance_Count; i++) {
			snapconf.i_Snapshot_No = 0;
			strcpy(snapconf.c_Snapshot_Path, sninfo.snap_instances[i].c_Snap_Name);
			strcpy(snapconf.c_Src_LV, sninfo.snap_conf.c_Src_LV);
			snapconf.t_Build_Time = get_build_time_by_snap_instance_name(
						sninfo.snap_conf.c_Snapshot_Path,
						snapconf.c_Snapshot_Path);
			snapconf.i_Status = S_READY;
			snapconf.b_Is_Scheduled_Snap = FALSE;
			snapconf.Priority = sninfo.snap_conf.Priority;

			ret = do_add_snapshot_record(&snapconf);
			if (ret<=0) {
#ifdef DEBUG
				printf("Delete_Snapshot_Volume : "
					"do_add_snap_instance_conf(%s) fails, err=%d!\n",
					sninfo.snap_instances[i].c_Snap_Name, ret);
#endif
				Mount_Snapshot_Volume(sninfo.snap_instances[i].c_Snap_Name);
				Set_Snapshot_Status(snapconf.i_Snapshot_No, S_READY);
				if (sninfo.i_Current_Instance_Count>0)
					free(sninfo.snap_instances);
				return ret;
			}
		}
		ret = Remove_One_Snap_Daemon_Conf_Record(snapshot_vol_no);
	}
#ifndef __EXT3_SNAP__
	if (left_size>0) {
		ret = share_free_space_among_existent_snapshots(
			sninfo.snap_conf.c_Src_LV,
			left_size);
		if (ret<0) {
#ifdef DEBUG
			printf("Delete_Snapshot_Volume : "
				"share_free_space_among_existent_snapshots("
				"%s, %lld) fails, err=%d\n",
				sninfo.snap_conf.c_Src_LV,
				left_size, ret);
#endif
			ret = ERROR_FAIL;
			goto delete_snapshot_volume_out;
		}
	}
// <== Cat : for new snap repository size issue
delete_snapshot_volume_out:
#endif
	ret = Remove_Snapshot_Volume_Record(snapshot_vol_no);
	if (sninfo.snap_conf.b_Is_Scheduled_Snap && sninfo.i_Current_Instance_Count>0)
		free(sninfo.snap_instances);

	return ret;
}

/**********************************************************************
** Delete_Snapshot_Volume_Instance
** Description:	Retrieve all sheduled snapshot volumes' info
** Input:
**		snap_vol_no:	the snapshot volume's number
** Output:	error code
**		SUCCESS
**		ERROR_BUSY
**		ERROR_FAIL
***********************************************************************/
int Delete_Snapshot_Volume_Instance(char *snap_lv)
{
	int ret, snapshot_vol_no, srcvolno;
	SNAPSHOT_VOLUME_INFO sn_vol_info;
	char pvpath[LV_PATH_MAX_LEN];
#ifndef __EXT3_SNAP__
	long long space_left=0;
#endif
	// check if the snapshot instance is busy restoring
	if (Get_Snapshot_Restore_Status(snap_lv) == s_BUSY)
	{
#ifdef DEBUG
		printf("Delete_Snapshot_Volume_Instance : "
			"%s is busy restoring!\n",
			snap_lv);
#endif
		return ERROR_BUSY;
	}

	snapshot_vol_no = Get_Snapshot_Volume_No(snap_lv);
	if (snapshot_vol_no<=0) {
#ifdef DEBUG
		printf("Delete_Snapshot_Volume_Instance : "
			"Get_Snapshot_Volume_No(%s) fails, err=%d\n",
			snap_lv, snapshot_vol_no);
#endif
		return snapshot_vol_no;
	}

	ret = Get_One_Snapshot_Volume_Info(snapshot_vol_no, &sn_vol_info);
	if (ret<0) {
#ifdef DEBUG
		printf("Delete_Snapshot_Volume_Instance : "
			"Get_One_Snapshot_Volume_Info(%d) fails, err=%d\n",
			snapshot_vol_no, ret);
#endif
		return ret;
	}

	if (sn_vol_info.snap_conf.b_Is_Scheduled_Snap==FALSE) {
#ifdef DEBUG
		printf("Delete_Snapshot_Volume_Instance : "
			"is not a scheduled snapshot volume, "
			"call Delete_Snapshot_Volume(%d)\n",
			snapshot_vol_no);
#endif
		return Delete_Snapshot_Volume(snapshot_vol_no);
	}

	srcvolno = Get_LV_Volume_No(sn_vol_info.snap_conf.c_Src_LV);
	Get_PV_Path(srcvolno, pvpath, LV_PATH_MAX_LEN);

#ifndef __EXT3_SNAP__
	space_left = get_lv_total_size(snap_lv);
#endif
	ret = do_remove_snap_lv(snap_lv, pvpath);
	if (ret<0) {
#ifdef DEBUG
		printf("Delete_Snapshot_Volume_Instance : "
			"do_remove_snap_lv(%s) fails, ret=%d!\n", snap_lv, ret);
#endif
		//mount snapshot
		Mount_Snapshot_Volume(snap_lv);
		ret = ERROR_FAIL;
		goto delete_snapshot_volume_instance_out;
	}
	Remove_Instance_From_Snapshot_Record(snapshot_vol_no, snap_lv);
#ifndef __EXT3_SNAP__
// Cat : for new snap repository size issue ==>
	if (sn_vol_info.i_Current_Instance_Count > 1) {
//		ret = Share_Space_Among_All_Snapshot_Instances(snapshot_vol_no, space_left);

		ret = share_free_space_among_existent_snapshots(
			sn_vol_info.snap_conf.c_Src_LV,
			space_left);
		if (ret<0) {
#ifdef DEBUG
/*
			printf("Delete_Snapshot_Volume_Instance : "
				"Share_Space_Among_All_Snapshot_Instances(%d,%lld) fails, err=%d\n",
				snapshot_vol_no,space_left,ret);
*/
			printf("Delete_Snapshot_Volume_Instance : "
				"share_free_space_among_existent_snapshots("
				"%s, %lld) fails, err=%d\n",
				sn_vol_info.snap_conf.c_Src_LV, space_left, ret);
#endif
			ret = ERROR_FAIL;
		}
	}
// <== Cat : for new snap repository size issue
#endif
delete_snapshot_volume_instance_out:

	if (sn_vol_info.i_Current_Instance_Count>0)
		free(sn_vol_info.snap_instances);

	return ret;
}

/**********************************************************************
** Restore_Snapshot_To_Org_Volume
** Description:	Restore the content back to the source volume &
**		clear all blocks in this snapshot
** Output:	Error code
**		SUCCESS
**		ERROR_FAIL
**		ERROR_OPEN_FILE
***********************************************************************/
int Restore_Snapshot_To_Org_Volume(char* snap_lv)
{
	int sn_vol_no, ret, lv_dev=-1, org_lv_no, sn_lv_no;
	SNAPSHOT_VOLUME_CONF snapconf;

	sn_vol_no = Get_Snapshot_Volume_No(snap_lv);
	if (sn_vol_no<0) return sn_vol_no;

	ret = Get_One_Snapshot_Volume_Conf(
		sn_vol_no, &snapconf);
	if (ret<0) {
#ifdef DEBUG
		printf("Restore_Snapshot_To_Org_Volume : "
			"Get_One_Snapshot_Volume_Conf(%d) fails, err=%d\n",
			sn_vol_no, ret);
#endif
		Set_Snapshot_Restore_Status(snap_lv, s_FAILURE);
		return ret;
	}

	org_lv_no = Get_LV_Volume_No(snapconf.c_Src_LV);
	ret = Get_Volume_Status(org_lv_no);
	if (ret!=S_READY && ret!=S_DEGRADED && ret!= S_REBUILDING_RAID) {
#ifdef DEBUG
		printf("Restore_Snapshot_To_Org_Volume : "
			"Source volume being busy!\n");
#endif
		Set_Snapshot_Restore_Status(snap_lv, s_FAILURE);
		return ERROR_BUSY;
	}

	sn_lv_no = snapconf.i_Snapshot_No;
	Set_Volume_Status(org_lv_no, S_RESTORING_SNAPSHOT);
	Set_Snapshot_Status(sn_lv_no, S_RESTORING_SNAPSHOT);
	Set_Snapshot_Restore_Status(snap_lv, s_BUSY);

	Umount_Snapshot_Volume(snap_lv);
	if (Is_Mounted(snapconf.c_Src_LV))
	{
		Stop_Services(); // stop network services
		ret = Umount_LVM_Volume(org_lv_no);
		Start_Services(); // restart network services
		if (ret<0) {
#ifdef DEBUG
			printf("Restore_Snapshot_To_Org_Volume : "
				"Fails to unmount volume %d, err=%d. STOP!\n",
				org_lv_no, ret);
#endif
			goto restore_snapshot_to_org_volume_fail;
		}
	}

	if ((lv_dev = open(snap_lv, O_RDONLY))<0) {
#ifdef DEBUG
		printf("Restore_Snapshot_To_Org_Volume : "
			"Fails to open device %s\n",
			snap_lv);
#endif
		ret = ERROR_OPEN_FILE;
		goto restore_snapshot_to_org_volume_fail;
	}

	ret = ioctl(lv_dev, LV_SNAPSHOT_RESTORE);
	close(lv_dev);
	if (ret) {
#ifdef DEBUG
		printf("Restore_Snapshot_To_Org_Volume : "
			"ioctl fails, err=%d\n",
			ret);
#endif
		ret = ERROR_FAIL;
		goto restore_snapshot_to_org_volume_fail;
	}
	Set_Snapshot_Restore_Status(snap_lv, s_SUCCESS);
	ret = SUCCESS;

restore_snapshot_to_org_volume_out:
	Mount_LVM_Volume(org_lv_no);
	Mount_Snapshot_Volume(snap_lv);
	Set_Volume_Status(org_lv_no, S_READY);
	Set_Snapshot_Status(snapconf.i_Snapshot_No, S_READY);
	return ret;

restore_snapshot_to_org_volume_fail:
	Set_Snapshot_Restore_Status(snap_lv, s_FAILURE);
	goto restore_snapshot_to_org_volume_out;
}

/**********************************************************************
** Share_Space_Among_All_Snapshot_Instances
** Description:	Get more(or less) space and share among all instances
**		in a snapshot volume as evenly as possible
** Output:	Error code
**		SUCCESS
**		ERROR_FAIL
**		ERROR_OPEN_FILE
***********************************************************************/
/* Cat : for new snap repository size issue
int Share_Space_Among_All_Snapshot_Instances(int snap_no, long long more_size)
{
	SNAPSHOT_VOLUME_INFO snapinfo;
	int ret;
	char cmd[BUF_SIZE];

	ret = Get_One_Snapshot_Volume_Info(snap_no, &snapinfo);
	if (ret<0) return ret;

	if (snapinfo.snap_conf.b_Is_Scheduled_Snap==FALSE) {
		char pv_name[LV_PATH_MAX_LEN];
		char sign;

		Get_Snapshot_PV_Path(snap_no, pv_name, LV_PATH_MAX_LEN);

		sign = (more_size>0) ? '+' : '-';

		sprintf(cmd, "%s -A y -L %c%lldK %s %s>/dev/null 1>/dev/null 2>/dev/null",
			LV_EXTEND_CMD, sign, more_size, snapinfo.snap_conf.c_Snapshot_Path, pv_name);

		if (system(cmd))
			return ERROR_FAIL;

		return SUCCESS;
	}

	if (snapinfo.i_Current_Instance_Count==0)
		return SUCCESS;
	else {
		long long srclvsize = get_lv_total_size(snapinfo.snap_conf.c_Src_LV);
		ret = resize_all_instances_evenly(
			snapinfo.snap_instances,
			snapinfo.i_Current_Instance_Count,
			more_size,
			snapinfo.snap_conf.ll_Snapshot_Size,
			srclvsize * MAX_SNAP_SIZE_ALLOWED);
	}
	if (snapinfo.snap_conf.b_Is_Scheduled_Snap && snapinfo.i_Current_Instance_Count>0)
		free(snapinfo.snap_instances);

	return ret;
}
*/
/**********************************************************************
** Create_Snapshot_Instance
** Description:	Create a new instance in a snapshot volume
** Output:	Error code ....
**		SUCCESS
***********************************************************************/
int Create_Snapshot_Instance(int snap_no)
{
	int ret, src_vol_no;
	long long newsize;
	char newlv[LV_PATH_MAX_LEN], pv_path[LV_PATH_MAX_LEN];
// Cat : for new snap repository size issue ==>
//	SNAPSHOT_VOLUME_CONF sconf;
	SNAPSHOT_VOLUME_INFO sninfo;
	LVM_VOLUME_INFO lvinfo;
#ifdef DEBUG
printf("Create_Snapshot_Instance(%d) step 1\n", snap_no);
#endif
	ret = Get_One_Snapshot_Volume_Info(snap_no, &sninfo);
	if (ret<0) return ret;
#ifdef DEBUG
printf("Create_Snapshot_Instance(%d) step 2\n", snap_no);
#endif
	if (sninfo.snap_conf.b_Is_Scheduled_Snap==FALSE) {
#ifdef DEBUG
		printf("Create_Snapshot_Instance : "
			"Not a schedule instance!\n");
#endif
/*
		if (sconf.ll_Snapshot_Size>0)
			return (Create_Snapshot_Volume(&sconf));
		else
			return ERROR_INVALID_PARAM;
*/
		ret = (Create_Snapshot_Volume(&(sninfo.snap_conf)));
		goto create_snapshot_instance_out;
	}
/*
	if (sconf.b_Is_Snap_PV_Allocated==FALSE) {
#ifdef DEBUG
		printf("Create_Snapshot_Instance : "
			"Not well prepared!!\n");
#endif
		return ERROR_FAIL;
	}
*/	
#ifdef DEBUG
printf("Create_Snapshot_Instance(%d) step 3\n", snap_no);
#endif
	src_vol_no = Get_LV_Volume_No(sninfo.snap_conf.c_Src_LV);
	if (src_vol_no<0) {
		ret = ERROR_NOT_FOUND;
		goto create_snapshot_instance_out;
	}

	ret = Get_One_LVM_Volume_Info(src_vol_no, &lvinfo);
	if (ret<0) {
		ret = ret;
		goto create_snapshot_instance_out;
	}

	if (lvinfo.vol_data.status != S_READY
		&& lvinfo.vol_data.status != S_DEGRADED
		&& lvinfo.vol_data.status != S_REBUILDING_RAID)
	{
#ifdef DEBUG
		printf("Create_Snapshot_Instance : "
			"source volume is either not READY, DEGRADED, nor REBUILDING\n");
#endif
		ret = ERROR_INVALID_PARAM;
		goto create_snapshot_instance_out;
	}
#ifdef DEBUG
printf("Create_Snapshot_Instance(%d) step 4\n", snap_no);
#endif
	strcpy(pv_path, lvinfo.vol_data.device_name);
	
	Get_Next_Snapshot_Volume_Instance_Path(snap_no, newlv, LV_PATH_MAX_LEN);

// Cat : for new snap repository size issue ==>
#ifdef DEBUG
printf("Create_Snapshot_Instance(%d) step 5 kill the old instance if %d==%d\n",
	snap_no, sninfo.i_Current_Instance_Count, sninfo.snap_conf.i_Max_Instance_Size);
#endif
	if (sninfo.i_Current_Instance_Count==sninfo.snap_conf.i_Max_Instance_Size)
	{
		SNAP_RESTORE_STATUS status = Get_Snapshot_Restore_Status(
			sninfo.snap_instances[0].c_Snap_Name);

		if (status==s_BUSY) {
#ifdef DEBUG
			printf("Create_Snapshot_Instance : "
				"snap pool full but the oldest is busy restoring!\n");
#endif			
			ret = ERROR_BUSY;
			goto create_snapshot_instance_out;
		}
		ret = do_remove_snap_lv(sninfo.snap_instances[0].c_Snap_Name,
			lvinfo.vol_data.device_name);
		if (ret<0) {
#ifdef DEBUG
			printf("prepare_snap_for_new_instance : "
				"snap pool full but fails to remove the oldest instance(%s), err=%d!\n",
				sninfo.snap_instances[0].c_Snap_Name, ret);
#endif
			goto create_snapshot_instance_out;
		}
		Remove_Instance_From_Snapshot_Record(snap_no,
			sninfo.snap_instances[0].c_Snap_Name);
	}
#ifdef DEBUG
printf("Create_Snapshot_Instance(%d) step 5.5 src_lv=%s\n", snap_no, lvinfo.vol_data.lv_name);
#endif
	newsize = get_avail_snap_vol_size(lvinfo.vol_data.lv_name, TRUE);
	if (newsize<0) {
#ifdef DEBUG
		printf("Create_Snapshot_Instance : "
			"get_avail_snap_vol_size(%s, TRUE) fails, err=%d\n",
			lvinfo.vol_data.lv_name, (int)newsize);
#endif
		ret = newsize;
		goto create_snapshot_instance_out;
	}
	else if (newsize==0) {
#ifdef DEBUG
		printf("Create_Snapshot_Instance : "
			"not enough free space (%d%%) for the new snap instance!\n",
			SNAP_MIN_SIZE_RATIO);
#endif
		ret = ERROR_OUT_OF_SNAP_REPOSITORY_SPACE;
		goto create_snapshot_instance_out;
	}
#ifdef DEBUG
printf("Create_Snapshot_Instance(%d) step 6, newsize = %lld\n", snap_no, newsize);
#endif
/*
	ret = prepare_snap_for_new_instance(snap_no, &newsize);
	if (ret<0) {
#ifdef DEBUG
		printf("Create_Snapshot_Instance : "
			"prepare_snap_for_new_instance fails, err=%d\n", ret);
#endif	
		return ret;
	}
*/
// <== Cat : for new snap repository size issue
	ret = create_logical_volume_directly(newlv, pv_path, newsize, TRUE, sninfo.snap_conf.c_Src_LV);
	if (ret<0) {
#ifdef DEBUG
		printf("Create_Snapshot_Instance step 3 : "
			"create_logical_volume_directly(%s, %s, %lld, TRUE, %s) : %d\n",
			newlv, pv_path, newsize, sninfo.snap_conf.c_Src_LV, ret);
#endif
/* Cat : for new snap repository size issue
		// undo what has been done....
		Share_Space_Among_All_Snapshot_Instances(snap_no, newsize);
*/
		goto create_snapshot_instance_out;
	}
#ifdef DEBUG
printf("Create_Snapshot_Instance(%d) step 7\n", snap_no);
#endif
	ret = Mount_Snapshot_Volume(newlv);
#ifdef DEBUG
printf("Create_Snapshot_Instance(%d) step 8\n", snap_no);
#endif
	Add_Instance_To_Snapshot_Record(snap_no, newlv);
	Set_Snapshot_Status(snap_no, S_READY);
#ifdef DEBUG
printf("Create_Snapshot_Instance(%d) step 9\n", snap_no);
#endif
	ret = create_snap_share(newlv);

create_snapshot_instance_out:
	if (sninfo.i_Current_Instance_Count>0)
		free(sninfo.snap_instances);
	return ret;
}

/**********************************************************************
** LVM_Quick_Config_All_Disks
** Description:	re-config all volumes to the desired raid_level,
**		raid_level==SINGLE : all non-single volumes would be reinit;
**		raid_level==LINEAR : all disks would form a linear RAID;
**		raid_level==STRIPING : all disks would form a stripping volume;
**		raid_level==MIRROR : every 2 disks would form a RAID-1 volume;
**		raid_level==RAID5 : all disks would form a RAID-5 volume.
** Output:	error code --
**		SUCCESS
**		
***********************************************************************/
int LVM_Quick_Config_All_Disks(int raid_level)
{
	int ret=0, disk_no, vol_cnt, i;
	int real_drive_no, min_drive_no;
	LVM_VOLUME_INFO *vol_list=NULL;

	// check if the operation is allowed?
	vol_cnt = Get_All_LVM_Volumes_Info(&vol_list);
	if (vol_cnt<0) {
#ifdef DEBUG
		printf("LVM_Quick_Config_All_Disks(%d) : "
			"Get_All_LVM_Volumes_Info fails, err=%d\n",
			raid_level, vol_cnt);
#endif
		return vol_cnt;
	}
	for (i=0; i<vol_cnt; i++) {
		Set_Volume_Status(vol_list[i].vol_data.vol_no, S_INITIALIZING);
	}

	disk_no = Get_Profile_Integer(
			NASCONF_STORAGE_SECTION,
			NASCONF_DRIVE_NO_FIELD,
			0);
	ret = quick_remove_all_snapshot_volumes();
	if (ret<0) {
#ifdef DEBUG
		printf("LVM_Quick_Config_All_Disks(%d) : "
			"quick_remove_all_snapshot_volumes fails, err=%d\n",
			raid_level, ret);
#endif
		goto lvm_quick_not_much_changed_out;
	}

	// remove all shares
	ret = quick_remove_all_shares();
	if (ret<0) {
#ifdef DEBUG
		printf("LVM_Quick_Config_All_Disks(%d) : "
			"quick_remove_all_shares fails, err=%d\n",
			raid_level, ret);
#endif
		goto lvm_quick_not_much_changed_out;
	}

	// need to remove all volumes??
	ret = quick_check_finish(vol_list, vol_cnt, raid_level, disk_no,
		&min_drive_no, &real_drive_no);

	if (ret == ERROR_ALREADY_EXISTED) {
		int i;

		for(i=0; i<vol_cnt; i++) {
			ret = quick_reformat_volume(&vol_list[i]);
			if (ret<0) break;
		}
#ifdef DEBUG
		printf("LVM_Quick_Config_All_Disks(%d) : "
			"no need to rebuild, err=%d\n",
			raid_level, ERROR_ALREADY_EXISTED);
#endif
		goto lvm_quick_finish;
	}
	else if (ret<0) {
#ifdef DEBUG
		printf("LVM_Quick_Config_All_Disks(%d) : "
			"quick_check_finish fails, err=%d\n",
			raid_level, ret);
#endif
		goto lvm_quick_not_much_changed_out;
	}

	ret = quick_remove_old_stuff(vol_list, vol_cnt);
	if (ret<0) {
#ifdef DEBUG
		printf("LVM_Quick_Config_All_Disks(%d) : "
			"quick_remove_old_stuff fails, err=%d\n",
			raid_level, ret);
#endif
		goto lvm_quick_early_failure; 
	}

	Release_LVM_Info_List(vol_list, vol_cnt);
	vol_list = NULL;

	// create the new volume(s)
	vol_cnt = Get_All_LVM_Volumes_Info(&vol_list);
	if (vol_cnt<0) {
#ifdef DEBUG
		printf("LVM_Quick_Config_All_Disks(%d) : "
			"new Get_All_LVM_Volumes_Info fails, err=%d\n",
			raid_level, vol_cnt);
#endif
		return vol_cnt;
	}

	if (vol_cnt!=disk_no) {
		Release_LVM_Info_List(vol_list, vol_cnt);
		Restore_LVM_Storage_Conf();
		vol_cnt = Get_All_LVM_Volumes_Info(&vol_list);
	}

	ret = quick_init_disks(vol_list, vol_cnt);
	if (ret<0) {
#ifdef DEBUG
		printf("LVM_Quick_Config_All_Disks(%d) : "
			"quick_init_disks fails, err=%d\n",
			raid_level, ret);
#endif
		goto lvm_quick_early_failure; 
	}

	ret = quick_create_new_volumes(vol_list, vol_cnt, raid_level,
		disk_no, min_drive_no, real_drive_no);

lvm_quick_finish:
	Release_LVM_Info_List(vol_list, vol_cnt);
	return ret;

lvm_quick_early_failure:
	Release_LVM_Info_List(vol_list, vol_cnt);
	vol_cnt = Get_All_LVM_Volumes_Info(&vol_list);
	for (i=0; i<vol_cnt; i++) {
		if (vol_list[i].vol_data.raid_level == SINGLE
			&& vol_list[i].vol_data.status != S_READY
			&& vol_list[i].vol_data.status != S_RW_ERROR	// Catherine Aug. 2004
			&& vol_list[i].vol_data.status != S_NOT_EXIST)	// Catherine Aug. 2004
			Set_Volume_Status(vol_list[i].vol_data.vol_no, S_UNINITIALIZE);
	}
	goto lvm_quick_finish;

lvm_quick_not_much_changed_out:
	for (i=0; i<vol_cnt; i++) {
		if (vol_list[i].vol_data.status != S_RW_ERROR
			&& vol_list[i].vol_data.status != S_NOT_EXIST)	// Catherine Aug. 2004
			Set_Volume_Status(vol_list[i].vol_data.vol_no, S_READY);
	}
	goto lvm_quick_finish;
}

/**********************************************************************
** Prepare_Scheduled_Snapshot_Volume
** Description:	Prepare the space for new scheduled snapshot.
** Output:	error code --
**		SUCCESS
**		
***********************************************************************/
/* Cat : for new snap repository size issue
int Prepare_Scheduled_Snapshot_Volume(int snap_no)
{
	int ret;
	SNAPSHOT_VOLUME_INFO snapinfo;
	int srcvol;
	char pv_name[LV_PATH_MAX_LEN];
	char vol_section[50];
	long long snapsize;
	long long maxsize, minsize;

	ret = Get_One_Snapshot_Volume_Info(snap_no, &snapinfo);
	if (ret<0) return ret;

	// the first one
	if (snapinfo.i_Current_Instance_Count>0 ||
		snapinfo.snap_conf.b_Is_Snap_PV_Allocated)
	{
#ifdef DEBUG
		printf("Prepare_Scheduled_Snapshot_Volume : "
			"already initialized!\n");
#endif
		ret = SUCCESS;
		goto prepare_scheduled_snapshot_volume_out;
	}

	if (snapinfo.snap_conf.i_Max_Instance_Size<=0) {
#ifdef DEBUG
		printf("Prepare_Scheduled_Snapshot_Volume : "
			"Max instance size<=0!\n");
#endif
		ret = ERROR_INVALID_PARAM;
		goto prepare_scheduled_snapshot_volume_fail;
	}

	if (!Is_LV_Exist(snapinfo.snap_conf.c_Src_LV)) {
#ifdef DEBUG
		printf("Prepare_Scheduled_Snapshot_Volume : "
			"Source volume LV not exist!\n");
#endif
		ret = ERROR_INVALID_PARAM;
		goto prepare_scheduled_snapshot_volume_fail;
	}
	else {
		int volno = Get_LV_Volume_No(snapinfo.snap_conf.c_Src_LV);
		LVM_VOLUME_INFO lvinfo;

		ret = Get_One_LVM_Volume_Info(volno, &lvinfo);
		if (ret<0) {
#ifdef DEBUG
			printf("Prepare_Scheduled_Snapshot_Volume : "
				"fail to get source volume info (volno=%d), err=%d\n",
				volno, ret);
#endif
			goto prepare_scheduled_snapshot_volume_fail;
		}

		if (lvinfo.vol_data.status != S_READY && lvinfo.vol_data.status != S_DEGRADED)
		{
#ifdef DEBUG
			printf("Prepare_Scheduled_Snapshot_Volume : "
				"source volume is either not READY or not DEGRADED\n");
#endif
			ret = ERROR_INVALID_PARAM;
			goto prepare_scheduled_snapshot_volume_fail;
		}
	}

	srcvol = Get_LV_Volume_No(snapinfo.snap_conf.c_Src_LV);
	ret = Get_PV_Path(srcvol, pv_name, LV_PATH_MAX_LEN);
	if (ret<0) goto prepare_scheduled_snapshot_volume_fail;

	if (snapinfo.snap_conf.ll_Snapshot_Size>0) {
		snapsize = get_approx_size(snapinfo.snap_conf.ll_Snapshot_Size, 0);
		ret = get_allowed_snap_vol_size(snapinfo.snap_conf.c_Src_LV,
			snapinfo.snap_conf.i_Max_Instance_Size,
			&maxsize, &minsize);
		if (ret<0) {
#ifdef DEBUG
			printf("Prepare_Scheduled_Snapshot_Volume : "
				"get_allowed_snap_vol_size(%s, %d) failed!\n",
				snapinfo.snap_conf.c_Src_LV,
				snapinfo.snap_conf.i_Max_Instance_Size);
#endif
			goto prepare_scheduled_snapshot_volume_fail;
		}
		if (snapsize<minsize) snapsize = minsize;
		else if (snapsize > maxsize) snapsize = maxsize;

		ret = borrow_space_from_lv(
			snapinfo.snap_conf.c_Src_LV,
			snapsize,
			pv_name, LV_PATH_MAX_LEN);
		if (ret<0) {
#ifdef DEBUG
			printf("Prepare_Scheduled_Snapshot_Volume : "
				"borrow_space_from_lv(%s, %lld, %s) failed!\n",
				snapinfo.snap_conf.c_Src_LV,
				snapsize, pv_name);
#endif
			goto prepare_scheduled_snapshot_volume_fail;
		}
	}
	else {
		ret = ERROR_INVALID_PARAM;
		goto prepare_scheduled_snapshot_volume_fail;
	}

	sprintf(vol_section, "snapshot %d", snap_no);
	ret = Conf_Set_Field(SNAPSHOT_CONF,vol_section,"space allocated","TRUE");
	Set_Snapshot_Volume_Size(snap_no, snapsize);
	Set_Snapshot_Status(snap_no, S_READY);
	ret = SUCCESS;

prepare_scheduled_snapshot_volume_out:
	if (snapinfo.i_Current_Instance_Count>0) free(snapinfo.snap_instances);

	return ret;

prepare_scheduled_snapshot_volume_fail:
	Remove_Snapshot_Volume_Record(snap_no);
	Remove_One_Snap_Daemon_Conf_Record(snap_no);
	goto prepare_scheduled_snapshot_volume_out;
}
*/

/**********************************************************************
** Is_A_Valid_Snapshot_Size
** Description:	check if the snapshot size is in the acceptable range
** Output:	TRUE or FALSE		
***********************************************************************/
/* Cat : for new snap repository size issue
BOOL Is_A_Valid_Snapshot_Size(char* src_lv, long long snapsize, int max_inst_cnt)
{
	long long maxsize, minsize;
	int ret;

	ret = get_allowed_snap_vol_size(src_lv, max_inst_cnt, &maxsize, &minsize);
	if (ret<0) return FALSE;

	if (snapsize>maxsize || snapsize<minsize)
		return FALSE;

	return TRUE;
}
*/

/**********************************************************************
** Is_A_Valid_Snapshot_Size
** Description:	check if the snapshot size is in the acceptable range
** Output:	TRUE or FALSE		
***********************************************************************/
BOOL Is_Allocatable_Snapshot_Volume(char* src_lv, int max_inst_cnt)
{
// Cat : for new snap repository size issue ==>
//	long long maxsize, minsize;
	int ret, volno;
	BOOL result = TRUE;
	LVM_VOLUME_INFO lvinfo;
	long long availsize;
	int i, current_snap_count=0;

	// step 1 : is this volume snapshot-enabled ?
	volno = Get_LV_Volume_No(src_lv);
	if (volno<0) {
#ifdef DEBUG
		printf("Is_Allocatable_Snapshot_Volume(%s, %d) : "
			"Get_LV_Volume_No fails, err=%d\n",
			src_lv, max_inst_cnt, volno);
#endif
		return FALSE;
	}
	ret = Get_One_LVM_Volume_Info(volno, &lvinfo);
	if (ret<0) {
#ifdef DEBUG
		printf("Is_Allocatable_Snapshot_Volume(%s, %d) : "
			"Get_One_LVM_Volume_Info(%d) fails, err=%d\n",
			src_lv, max_inst_cnt, volno, ret);
#endif
		return FALSE;
	}
	if (!lvinfo.vol_data.snapshot_enable) {
#ifdef DEBUG
		printf("Is_Allocatable_Snapshot_Volume(%s, %d) : "
			"volume %d is not snapshot-enabled!\n",
			src_lv, max_inst_cnt, volno);
#endif
		result = FALSE;
		goto is_allocatable_snapshot_volume_out;
	}

	// step 2 : space enough for new snapshots ?
	for (i=0; i<lvinfo.snapshot_cnt; i++) {
		if (lvinfo.snap_list[i].b_Is_Scheduled_Snap)
			current_snap_count += lvinfo.snap_list[i].i_Max_Instance_Size;
		else
			current_snap_count++;
	}

#ifdef __EXT3_SNAP__
	if (lvinfo.vol_data.max_snap_count < current_snap_count + max_inst_cnt) {
#else // not __EXT3_SNAP__
	if (MAX_SNAP_NO_PER_VOLUME < current_snap_count + max_inst_cnt) {
#endif

#ifdef DEBUG
		printf("Is_Allocatable_Snapshot_Volume(%s, %d) : "
			"volume %d already has %d snapshots, max snapshot count per volume "
			"is %d\n",
			src_lv, max_inst_cnt, volno,
			current_snap_count, MAX_SNAP_NO_PER_VOLUME);
#endif
		result = FALSE;
		goto is_allocatable_snapshot_volume_out;
	}

	availsize = get_avail_snap_vol_size(src_lv, FALSE);
	if (availsize<=0) {
#ifdef DEBUG
		printf("Is_Allocatable_Snapshot_Volume(%s, %d) : "
			"get_avail_snap_vol_size(%s, FALSE) returns %lld\n",
			src_lv, max_inst_cnt, src_lv, availsize);
#endif
		result = FALSE;
		goto is_allocatable_snapshot_volume_out;
	}
/*
	ret = get_allowed_snap_vol_size(src_lv, max_inst_cnt, &maxsize, &minsize);
	if (ret<0) return FALSE;
*/
is_allocatable_snapshot_volume_out:
	if (lvinfo.snapshot_cnt>0) free(lvinfo.snap_list);

// <== Cat : for new snap repository size issue
	return result;
}

/**********************************************************************
** Reset_Disk
** Description:	reinitialize all disks to STRIPING and create a samba share "public"
** Output:	error code --
**		SUCCESS
**		
***********************************************************************/
int Reset_Disk()
{
	int ret;
	NAS_SHARE_INFO shareinfo;
	char plist[][USER_GROUP_NAME_LENGTH] = 
		{ "@everyone", "guest", };

	ret = LVM_Quick_Config_All_Disks(STRIPING);
	if (ret<0) return ret;

	ret = create_directory("/share/MD0_DATA/public");

	strcpy(shareinfo.share_name, "public");
	strcpy(shareinfo.path, "/share/MD0_DATA/public");
	strcpy(shareinfo.comment, "");
	shareinfo.read_list_cnt = 0;
	shareinfo.read_list_ptr = NULL;
	shareinfo.write_list_cnt = 2;
	shareinfo.write_list_ptr = plist;
	
	ret = Create_NAS_Share(&shareinfo);
	return ret;
}

/**********************************************************************
** Enable_Snapshot
** Description:	enable a volume to snapshot-support,
**		the volume is shrank to (100-ratio)% of PV size
**		to free some PEs to future snapshots
** Output:	error code --
**		SUCCESS
**		ERROR_NOT_LVM_VOLUME -- not a LVM volume, cannot enable snapshot
**		ERROR_SNAPSHOT_ALREADY_ENABLED -- try to enable a snapshot-enabled
**						volume with a different ratio value
**		
***********************************************************************/
int Enable_Snapshot(int volno, int ratio, int max_snap_cnt)
{
	int ret = SUCCESS;
	char volstr[BUF_SIZE], tmp[BUF_SIZE];
	LVM_VOLUME_INFO volinfo;
	PV_INFO pvinfo;
	long long snapsize;

	// step 0 : get volume info
	ret = Get_One_LVM_Volume_Info(volno, &volinfo);
	if (ret<0) {
#ifdef DEBUG
		printf("Enable_Snapshot(%d, %d, %d) : failed, err=%d\n",
			volno, ratio, max_snap_cnt, ret);
#endif
		return ret;
	}

	// step 1 : check if LV exist and already enabled
	if (!Is_LV_Exist(volinfo.vol_data.lv_name)) {
#ifdef DEBUG
		printf("Enable_Snapshot(%d, %d, %d) : not a LVM volume!\n",
			volno, ratio, max_snap_cnt);
#endif
		return ERROR_NOT_LVM_VOLUME;
	}

	if (volinfo.vol_data.snapshot_enable) {
		if (volinfo.vol_data.snap_ratio == ratio &&
			volinfo.vol_data.max_snap_count == max_snap_cnt)
			return SUCCESS;
		else {
#ifdef DEBUG
			printf("Enable_Snapshot(%d, %d, %d) : this volume already snapshot-enabled"
				"with ratio %d, try to disable it before enable it again!\n",
				volno, ratio, max_snap_cnt, volinfo.vol_data.snap_ratio);
#endif
			return ERROR_SNAPSHOT_ALREADY_ENABLED;
		}
	}

	ret = get_pv_info(volinfo.vol_data.device_name, &pvinfo);
	if (ret<0) {
#ifdef DEBUG
		printf("Enable_Snapshot(%d, %d, %d) : fail to get PV (%s) info, err=%d\n",
			volno, ratio, max_snap_cnt, volinfo.vol_data.device_name, ret);
#endif
		goto enable_snapshot_fail;
	}

	snapsize = get_approx_size(pvinfo.ll_PE_Size*pvinfo.i_PE_Count*ratio/100, -1);

	if (snapsize < max_snap_cnt * PE_SIZE) {
#ifdef DEBUG
		printf("Enable_Snapshot(%d, %d, %d) : the space not enough for so many snapshots!\n",
			volno, ratio, max_snap_cnt);
#endif
		ret = ERROR_OUT_OF_SNAP_REPOSITORY_SPACE;
		goto enable_snapshot_fail;
	}

	// step 2 : set the value to the config file
	sprintf(volstr, "volume %d", volno);
	ret = Conf_Set_Field(STORAGE_CONF, volstr, "enable snapshot", "TRUE");
	if (ret<0) {
#ifdef DEBUG
		printf("Enable_Snapshot(%d, %d, %d) : fail to set config file, err=%d\n",
			volno, ratio, max_snap_cnt, ret);
#endif
		return ret;
	}
	sprintf(tmp, "%d", ratio);
	ret = Conf_Set_Field(STORAGE_CONF, volstr, "snap repository ratio", tmp);
	if (ret<0) {
#ifdef DEBUG
		printf("Enable_Snapshot(%d, %d, %d) : fail to set config file, err=%d\n",
			volno, ratio, max_snap_cnt, ret);
#endif
		goto enable_snapshot_fail;
	}

	sprintf(tmp, "%d", max_snap_cnt);
	ret = Conf_Set_Field(STORAGE_CONF, volstr, "max snap count", tmp);
	if (ret<0) {
#ifdef DEBUG
		printf("Enable_Snapshot(%d, %d, %d) : fail to set config file, err=%d\n",
			volno, ratio, max_snap_cnt, ret);
#endif
		goto enable_snapshot_fail;
	}

	// step 3 : shrink LV size to (100-ratio)% of PV size
	ret = reduce_space_from_lv(volno, snapsize);
	if (ret<0) {
#ifdef DEBUG
		printf("Enable_Snapshot(%d, %d, %d) : reduce_space_from_lv(%lld), err=%d\n",
			volno, ratio, max_snap_cnt, snapsize, ret);
#endif
		goto enable_snapshot_fail;
	}

enable_snapshot_out:	
	return ret;

enable_snapshot_fail:
	sprintf(volstr, "volume %d", volno);
	Conf_Set_Field(STORAGE_CONF, volstr, "enable snapshot", "FALSE");
	goto enable_snapshot_out;
}

/**********************************************************************
** Reset_Disk
** Description:	disable a volume's snapshot support,
**		also kill all existent snapshots
** Output:	error code --
**		SUCCESS
**		ERROR_NOT_LVM_VOLUME
**		....
***********************************************************************/
int Disable_Snapshot(int volno)
{
	int ret = SUCCESS, i;
	char volstr[BUF_SIZE];
	long long snap_size;
	LVM_VOLUME_INFO volinfo;
	PV_INFO pvinfo;

	// step 0 : get volume info
	ret = Get_One_LVM_Volume_Info(volno, &volinfo);
	if (ret<0) {
#ifdef DEBUG
		printf("Disable_Snapshot(%d) : failed, err=%d\n", volno, ret);
#endif
		return ret;
	}

	// step 1 : check if LV exist and already disabled
	if (!Is_LV_Exist(volinfo.vol_data.lv_name)) {
#ifdef DEBUG
		printf("Disable_Snapshot(%d) : not a LVM volume!\n", volno);
#endif
		return ERROR_NOT_LVM_VOLUME;
	}

	if (!volinfo.vol_data.snapshot_enable)
		return SUCCESS;

	// step 2 : check if any existent snapshots, kill them
	for (i=0; i<volinfo.snapshot_cnt; i++) {
		ret = remove_snapshot_volume_completely(volinfo.snap_list[i].i_Snapshot_No);
		if (ret<0) {
#ifdef DEBUG
			printf("Disable_Snapshot(%d) : fail to kill snapshot %d, err=%d!\n",
				volno, volinfo.snap_list[i].i_Snapshot_No, ret);
#endif
			goto disable_snapshot_fail;
		}
	}

	// step 3 : set config file
	sprintf(volstr, "volume %d", volno);
	ret = Conf_Set_Field(STORAGE_CONF, volstr, "enable snapshot", "FALSE");
	if (ret<0) {
#ifdef DEBUG
		printf("Disable_Snapshot(%d) : fail to set config file, err=%d\n",
			volno, ret);
#endif
		return ret;
	}

	// step 4 : extend LV to full PV size
	ret = get_pv_info(volinfo.vol_data.device_name, &pvinfo);
	if (ret<0) {
#ifdef DEBUG
		printf("Disable_Snapshot(%d) : fail to get PV (%s) info, err=%d\n",
			volno, volinfo.vol_data.device_name, ret);
#endif
		goto disable_snapshot_fail;
	}

	snap_size = get_approx_size(pvinfo.ll_PE_Size*pvinfo.i_PE_Count*volinfo.vol_data.snap_ratio/100, -1);
#ifndef __EXT3_SNAP__
	if (Is_Mounted(volinfo.vol_data.lv_name)) {
		Stop_Services(); // stop network services
		ret = Umount_LVM_Volume(volno);
		Start_Services(); // restart network services
		if (ret<0) goto disable_snapshot_fail;
	}
#endif

	ret = extend_space_to_lv(volno, snap_size);

#ifndef __EXT3_SNAP__
	Mount_LVM_Volume(volno);
	if (ret<0) {
#ifdef DEBUG
		printf("Disable_Snapshot(%d) : extend_space_to_lv(%lld), err=%d\n",
			volno, snap_size, ret);
#endif
		goto disable_snapshot_fail;
	}
#endif
disable_snapshot_out:
	return ret;

disable_snapshot_fail:
	sprintf(volstr, "volume %d", volno);
	Conf_Set_Field(STORAGE_CONF, volstr, "enable snapshot", "TRUE");
	goto disable_snapshot_out;
}

/**********************************************************************
** Clear_All_PV_Headers
** Description:	clean up all PV's first LVM_VGDA_SIZE bytes;
**		the caller is responsible for unmounting all LVs...
**		since if some volumes are busy,
**		this function may fail
** Output:	error code --
**		SUCCESS
**		....
***********************************************************************/
int Clear_All_PV_Headers()
{
	int ret = SUCCESS, drive_no;
	char part_name[HD_DEVICE_NAME_LENGTH];
	int disk_cnt = Get_Profile_Integer(NASCONF_STORAGE_SECTION, NASCONF_DRIVE_NO_FIELD, 0);

#ifdef DEBUG
	printf("Clear_All_PV_Headers : starts with %d drives!\n", disk_cnt);
#endif
	Deactivate_LVM();

	for (drive_no=1; drive_no<=disk_cnt; drive_no++) {
		Get_Partition_Name(drive_no, DATA_PART, part_name);//DATA
		if (!Is_Partition_Exist(part_name)) {
#ifdef DEBUG
			printf("Clear_All_PV_Headers : partition %s not exist!\n", part_name);
#endif
			continue;
		}

		if (Is_Mounted(part_name))
			Unmount_Partition(part_name);

		clear_pv_header(part_name);
	}
	return ret;
}
