//****************************************************************************
//	Copyright (c) 2000	ICP Electronics Inc.	All right reserved.
//	FILE:
//			msg.c
//
//	Abstract:
//			Simulate Window Message Passing Mechanism.
//
//	HISTORY:
//			03/09/2001		Ethan Wang -- Created
//*****************************************************************************

#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>

#include <msgeng.h>

static int	Msg_Initialized = 0;
static int	FD_Msg;

int Open_Message_Engine()
{
	if ( Msg_Initialized )
		return FD_Msg;	// the process already open 

	FD_Msg = open(MSGENG_DEV, O_RDWR);
	if ( FD_Msg >= 0 )
		Msg_Initialized = 1;
	return FD_Msg;
}

void Close_Message_Engine()
{
	if ( Msg_Initialized )
	{
		close(FD_Msg);
		Msg_Initialized = 0;
	}
}

int GetMessage(MSG_PARM *pmsg)
{
	int errcode = 0;

	if ( !Msg_Initialized && (errcode = Open_Message_Engine()) < 0 )
		return errcode;

	ioctl(FD_Msg, IOCTL_MSG_GET_MESSAGE, pmsg);

	return errcode;
}

int SendMessageEx(const char *progname, int message, unsigned long parm1, unsigned long parm2, char *extra)
{
	int		errcode;
	MSG_PARM	msg;

	errcode = 0;

	if ( !Msg_Initialized && (errcode = Open_Message_Engine()) < 0 )
		return errcode;

	if ( progname == NULL )
	{
		msg.comm[0] = 0;
	}
	else
	{
		strncpy(msg.comm, progname, MAX_PROGNAME_SIZE-1);
		msg.comm[MAX_EXTRAMSG_SIZE-1] = 0;
	}

	if ( extra == NULL )
	{
		msg.extra[0] = 0;
	}
	else
	{
		strncpy(msg.extra, extra, MAX_EXTRAMSG_SIZE-1);
		msg.extra[MAX_EXTRAMSG_SIZE-1] = 0;
	}

	msg.message = message;
	msg.parm1 = parm1;
	msg.parm2 = parm2;

	ioctl(FD_Msg, IOCTL_MSG_SEND_MESSAGE, &msg);

	return errcode;
}

int SendMultipleMessage(const char *progname, int message, unsigned long parm1, unsigned long msg_count, char **extramsgs)
{
	int	i;
	int	errcode;

	errcode = 0;

	if ( extramsgs == NULL )
		return -1;

	for (i = 0 ; i < msg_count ; i++)
	{
		if ( extramsgs[i] != NULL )
		{
			errcode = SendMessageEx(progname, message, parm1, i, extramsgs[i]);
			if ( errcode < 0 )
				break;
		}
	}

	return errcode;
}

int SendMessage(const char *progname, int message, unsigned long parm1, unsigned long parm2)
{
	return SendMessageEx(progname, message, parm1, parm2, NULL);
}

int SetTimer(int id, int msec)
{
	return SendMessage(NULL, MSG_SET_TIMER, id, msec);
}

int KillTimer(int id)
{
	return SendMessage(NULL, MSG_KILL_TIMER, id, 0);
}
