//**************************************************************************
//
//	Copyright (c) 2002  ICP Electronics Inc.  All Rights Reserved.
//
//	FILE:
//		cfg_system_recover.c
//
//	Abstract: 
//		Backup/recover system configuration to/from the 'backup' dirctories of
//		all disks' ROOT partitions
//
//	FUNCTIONS:	TBD.
//
//	COMMENTS: 	N/A
//
//	HISTORY:
//		2002/11/15	Catherine Shen -- created
//
//**************************************************************************
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>
#include <assert.h>
#include <utime.h>
#include <fcntl.h>

/* NAS include file */
#include "config.h"
#include "storage.h"

extern int create_dir(char *dirname);

/* ------------------------------------------------------------------------- **
 * Type definitions...
 */
struct CONF_FILE_ATT {
	char* name;
	char* premise; // need "xxx" support? if not, NULL
	BOOL is_dir;
	BOOL optional;
};

/* ------------------------------------------------------------------------- **
 * Constants...
 */
#define TRUE			1
#define	FALSE			0
#define BACKUP_DIR		"backup"
#define STORAGE_CONF_FILE	"/etc/config/storage.conf"
#define SYSTEM_CONFIG_FILE	"/etc/config/uLinux.conf"

struct CONF_FILE_ATT NAS_CONFIG_TABLE[] = {
	/*-------------- general config files ---------------*/
	{ "/etc/passwd",				NULL,		FALSE,	FALSE },
	{ "/etc/group",					NULL,		FALSE,	FALSE },
	{ "/etc/raidtab",				NULL,		FALSE,	FALSE },
	{ "/etc/localtime",				NULL,		FALSE,	FALSE },
	{ "/etc/nfs.conf",				NULL,		FALSE,	FALSE },
	{ "/etc/config/smb.conf",			NULL,		FALSE,	FALSE },
	{ "/etc/config/uLinux.conf",			NULL,		FALSE,	FALSE },
	{ "/etc/config/smbpasswd",			NULL,		FALSE,	FALSE },
	{ "/etc/config/smbusers",			NULL,		FALSE,	FALSE },
	{ "/etc/config/clock",				NULL,		FALSE,	FALSE },
	{ "/etc/snmp/snmpd.conf",			NULL,		FALSE,	FALSE },
	{ "/etc/config/storage.conf",			NULL,		FALSE,	TRUE },
	{ "/etc/config/snapshot.conf",			NULL,		FALSE,	TRUE },
	{ "/etc/config/snapd.conf",			NULL,		FALSE,	TRUE },
	{ "/etc/nwserv.conf",				NULL,		FALSE,	FALSE },
	{ "/etc/config/nwpwd",				NULL,		FALSE,	FALSE },
	{ "/etc/config/pdcgroup",			NULL,		FALSE,	TRUE },
	{ "/etc/config/codepage",			NULL,		FALSE,	TRUE },
	{ "/etc/config/printcap",			NULL,		FALSE,	FALSE },
	{ "/etc/atalk/etc/papd.conf",			NULL,		FALSE,	FALSE },
	{ "/etc/config/usr_quota.conf",			NULL,		FALSE,	FALSE },
	{ "/etc/config/grp_quota.conf",			NULL,		FALSE,	FALSE },
	{ "/etc/config/backup_schedule.conf",		NULL,		FALSE,	TRUE },
	/*--------------- PPPOE config files ----------------*/
	{ "/etc/config/ppp/chap-secrets",		"PPPOE",	FALSE,	FALSE },
	{ "/etc/config/ppp/pap-secrets",		"PPPOE",	FALSE,	FALSE },
	{ "/etc/config/ppp/pppoe.conf",			"PPPOE",	FALSE,	FALSE },
	{ "/etc/config/ppp/resolv.conf",		"PPPOE",	FALSE,	FALSE },
	{ "/etc/config/ppp/options",			"PPPOE",	FALSE,	TRUE },
	/*-------------- firewall config files --------------*/
	{ "/etc/config/fw_vserver.conf",		"NAT",		FALSE,	TRUE },
	{ "/etc/config/fw_rule.conf",			"NAT",		FALSE,	TRUE },
	{ "/etc/config/fw_vdmz.conf",			"NAT",		FALSE,	TRUE },
	{ "/etc/config/fw_filter.conf",			"NAT",		FALSE,	TRUE },
	{ "/etc/config/fw_routing_table.conf",		"NAT",		FALSE,	TRUE },
	{ "/etc/config/fw_special_application.conf",	"NAT",		FALSE,	FALSE },
	{ "/etc/config/fw_mapping.conf",		"NAT",		FALSE,	TRUE },
	{ "/etc/config/fw_string_filter.conf",		"NAT",		FALSE,	TRUE },
	/*---------------- tape config files -----------------*/
	{ "/etc/config/tape",				"TAPE",		TRUE,	FALSE },
};


/* ------------------------------------------------------------------------- **
 * Variables...
 *
 */

/* ------------------------------------------------------------------------- */
/*         		Internal Function Implementation                     */
/* ------------------------------------------------------------------------- */

//==========================================================================
// name : get_usable_root_partitions
// description : get the mount points of all existent disks' root partitions
// output : existent root partition count or error number
//==========================================================================
int get_usable_root_partitions(char (*root_parts)[HD_DEVICE_NAME_LENGTH], int max_count)
{
	int count=0, i;
	char root[HD_DEVICE_NAME_LENGTH], mt_path[HD_DEVICE_NAME_LENGTH];
	int maxdiskcnt = Get_Profile_Integer("Storage", "Disk Drive Number", 0);

	for (i=1; i<=maxdiskcnt; i++) {
		if (count >= max_count)
			return ERROR_BUFFER_TOO_SMALL;

		Get_Partition_Name(i, ROOT_PART, root);
		if ((Get_Mount_Point(root, mt_path)==SUCCESS) &&	// root is mounted
			(Verify_Hidden_Conf_File(i)==SUCCESS))		// hidden file is valid
		{
			strcpy(root_parts[count++], mt_path);
		}	
	}
	return count;
}

//==========================================================================
// name : config_backup_dir_exists
// description : test if the config file in the partition valid
// output : 1 (TRUE) or 0 (FALSE)
//==========================================================================
int config_backup_dir_exists(char* part_mp)
{
	DIR *dp;
	char tmpname[BUF_SIZE];

	// test if the BACKUP_DIR dir exist
	if (part_mp[strlen(part_mp)-1]=='/')
		sprintf(tmpname, "%s%s", part_mp, BACKUP_DIR);
	else
		sprintf(tmpname, "%s/%s", part_mp, BACKUP_DIR);

	if ((dp = opendir(tmpname)) == NULL) return FALSE;

	closedir(dp);		

	return TRUE;
}

//==========================================================================
// name : get_file_name
// description : get the file name only
// output : file name string length
//==========================================================================
static int get_file_name(char* filepath, char* outname)
{
	char* p;

	if (filepath==NULL || outname==NULL)
		return -1;

	if ((p = strrchr(filepath, '/')) == NULL)
		strcpy(outname, filepath);
	else
		strcpy(outname, p+1);
	return (strlen(outname));
}

//==========================================================================
// name : does_file_exist
// description : test if the file exists
// output : 1 (TRUE) or 0 (FALSE)
//==========================================================================
BOOL does_file_exist(const char *filename)
{
	struct stat	status;
	
	assert(filename);
	
	return ( 0 == lstat(filename, &status) );
}

//==========================================================================
// name : does_dir_exist
// description : test if the directory exists
// output : 1 (TRUE) or 0 (FALSE)
//==========================================================================
BOOL does_dir_exist(const char *dirname)
{
	struct stat	status;
	
	assert(dirname);
	
	if ( 0 > lstat(dirname, &status) ) return FALSE;
	
	return S_ISDIR(status.st_mode);
}

//==========================================================================
// name : is_a_dir
// another name of does_dir_exist
// output : 1 (TRUE) or 0 (FALSE)
//==========================================================================
#define is_a_dir(a)	does_dir_exist(a)

//==========================================================================
// name : make_pathname
// description : concatenate the dirname & filename
// output : 0 (Success) or -1 (Failure)
//==========================================================================
int make_pathname(char *namebuf, int bufsize, const char *dirname, const char *filename)
{
	int totalsize;

	assert(namebuf);
	assert(dirname);
	assert(filename);

	totalsize = strlen(filename);
	if ( dirname[strlen(dirname)-1] == '/' )
		totalsize += strlen(dirname);
	else
		totalsize += strlen(dirname) + 1;

	if (totalsize >= bufsize) return -1;

	strcpy(namebuf, dirname);
	if ( '/' != namebuf[strlen(dirname) - 1] )
		strcat(namebuf, "/");
	strcat(namebuf, filename);

	return SUCCESS;
}

//==========================================================================
// name : remove_dir
// description : remove a directory along with all its contents
// output : 0 (Success) or -1 (Failure)
//==========================================================================
int remove_dir(const char* dirpath)
{
	char cmd[BUF_SIZE];

	assert( dirpath );
	assert( does_dir_exist(dirpath) );

	sprintf(cmd, "/bin/rm -rf %s", dirpath);
	if ( system(cmd) )
		return -1;
	return 0;
}

//==========================================================================
// name : my_copy_file
// description : copy a file, overwrite if force == TRUE && dest exists
// output : 0 (Success) or -1 (Failure)
//==========================================================================
int my_copy_file(const char *src, const char *dest, BOOL force)
{
	struct stat	src_status;
	struct utimbuf	old_utimbuf;
	int		src_fd, dest_fd;
	int 		count;
	char		buf[BUF_SIZE];

	if ( src == NULL || dest == NULL) {
#ifdef DEBUG
		printf("my_copy_file : invalid input\n");
#endif
		return -1;
	}

	// get src old status included: old atime, mtime
	if ( 0 > lstat(src, &src_status) ) {
#ifdef DEBUG
		printf("my_copy_file : source file not exist!\n");
#endif
		return -1;
	}
	if (S_ISLNK(src_status.st_mode)) {
#ifdef DEBUG
		printf("my_copy_file : source file %s is a symbolic link.\n",
			src);
#endif
		return 0;
	}
		
	if ( !force && does_file_exist(dest) ) {
#ifdef DEBUG
		printf("my_copy_file : not force copy && %s already exist!\n", dest);
#endif
		return -1;
	}
	
	// copy src to dest
	src_fd	= open(src, O_RDONLY);
	if (0 > src_fd) {
#ifdef DEBUG
		printf("my_copy_file : fail to open %s!\n", src);
#endif
		return -1;
	}
	
	dest_fd = open(dest, O_WRONLY | O_CREAT | O_TRUNC, src_status.st_mode);
	if (0 > dest_fd) {
#ifdef DEBUG
		printf("my_copy_file : fail to open %s for write!\n", dest);
#endif
		close(src_fd);
		return -1;
	}

	while ( 0 < ( count = read(src_fd, buf, BUF_SIZE) ) ) {
		if ( count != write(dest_fd, buf, count) ) {
			close(src_fd);
			close(dest_fd);
#ifdef DEBUG
			printf("my_copy_file : write fail!\n");
#endif		
			return -1;
		}
	}
	
	if (0 != count) {
#ifdef DEBUG
		printf("my_copy_file : fails to read!\n");
#endif
		close(src_fd);
		close(dest_fd);
		return -1;
	}

	if ( 0 > close(src_fd) ) {
#ifdef DEBUG
		printf("my_copy_file : fails to close %s!\n", src);
#endif
		close(dest_fd);
		return -1;
	}
	
	if ( 0 > close(dest_fd) ) {
#ifdef DEBUG
		printf("my_copy_file : fails to close %s!\n", dest);
#endif
		return -1;
	}

	// set src and dest time as old time
	old_utimbuf.actime	= src_status.st_atime;
	old_utimbuf.modtime	= src_status.st_mtime;

	if ( 0 > utime(src,  &old_utimbuf) ) {
#ifdef DEBUG
		printf("my_copy_file : fails to set create time!\n");
#endif
		return -1;
	}
	
	if ( 0 > utime(dest, &old_utimbuf) ) {
#ifdef DEBUG
		printf("my_copy_file : fails to set create time!\n");
#endif
		return -1;
	}
	
	return 0;
}

//==========================================================================
// name : my_copy_dir
// description : copy a directory
// output : 0 (Success) or -1 (Failure)
//==========================================================================
int my_copy_dir(char *src, char *dest, BOOL is_recursive)
{
	char		src_fn[BUF_SIZE], dest_fn[BUF_SIZE];
	int		ret_val	= SUCCESS;
	DIR		*src_dir=NULL;
	struct dirent	*dep=NULL;

	if ( src==NULL || dest == NULL) {
#ifdef DEBUG
		printf("my_copy_dir : invalid input\n");
#endif
		return -1;
	}

	if ( !does_dir_exist(src) ) {	
#ifdef DEBUG
		printf("my_copy_dir : %s not exist\n", src);
#endif
		return -1;
	}

	if ( !does_dir_exist(dest) && create_dir(dest) ) {
#ifdef DEBUG
		printf("my_copy_dir : fails to create a new directory %s\n", dest);
#endif
		return -1;
	}

	// open source
	src_dir = opendir(src);
	if (NULL == src_dir) {
#ifdef DEBUG
		printf("my_copy_dir : fails to opendir(%s)\n", src);
#endif
		return -1;
	}

	while ( NULL != ( dep = readdir(src_dir) ) ) {
		/** skip dot and dot-dot */
		if ( 0 == strcmp(".",  dep->d_name) )	continue;
		if ( 0 == strcmp("..", dep->d_name) )	continue;
#ifdef DEBUG
		printf("my_copy_dir : item [%s] found\n", dep->d_name);
#endif	
		// make the new src file/dir name
		ret_val = make_pathname(src_fn, BUF_SIZE, src, dep->d_name);
		if (SUCCESS != ret_val) break;
		
		// make the new dest file/dir name
		ret_val = make_pathname(dest_fn, BUF_SIZE, dest, dep->d_name);
		if (SUCCESS != ret_val) break;
		
		if ( is_a_dir(src_fn) ) {
			if ( is_recursive )
				ret_val = my_copy_dir(src_fn, dest_fn, TRUE);
			else
				ret_val = 0;
		}
		else
			ret_val = my_copy_file(src_fn, dest_fn, TRUE);

		if (ret_val) {
#ifdef DEBUG
			printf("my_copy_dir : fails to copy [%s] to [%s]\n", src_fn, dest_fn);
#endif
			break;
		}
	}

	if ( 0 > closedir(src_dir) ) {
#ifdef DEBUG
		printf("my_copy_dir : fails to close [%s]\n", src);
#endif
		ret_val = -1;
	}
	
	return -ret_val;
}

/* ------------------------------------------------------------------------- */
/*         		Public Function Implementation                       */
/* ------------------------------------------------------------------------- */

//==========================================================================
// name : Recover_System_Config
// description : copy the config files from the first existent & valid root partition
//		to their working directories
// output : the times error occurs
//==========================================================================
int Recover_System_Config(void)
{
	int ret = SUCCESS, part_cnt, i, j, errcnt=0;
	BOOL is_supported, op_mask;
	char roots[MAX_DRIVE_NO][HD_DEVICE_NAME_LENGTH];
	char file_path[BUF_SIZE], tmp[BUF_SIZE];

	// find all valid root partitions
	part_cnt = get_usable_root_partitions(roots, MAX_DRIVE_NO);
	if (part_cnt<0) {
#ifdef DEBUG
		printf("Recover_System_Config : get_usable_root_partitions fails, err=%d\n",
			part_cnt);
#endif
		return part_cnt;
	}

	for (j=0; j<part_cnt; j++) {
		if (!config_backup_dir_exists(roots[j])) // not a backup partition
			continue;

		for (i=0; i< sizeof(NAS_CONFIG_TABLE)/sizeof(NAS_CONFIG_TABLE[0]); i++) {
			if (NAS_CONFIG_TABLE[i].premise == NULL)
				is_supported = TRUE;
			else
				is_supported = Get_Profile_Boolean(NAS_CONFIG_TABLE[i].premise, "Support", FALSE);
			op_mask = is_supported & ~(NAS_CONFIG_TABLE[i].optional);

			get_file_name(NAS_CONFIG_TABLE[i].name, tmp);
			sprintf(file_path, "%s/%s/%s", roots[j], BACKUP_DIR, tmp);
			if (NAS_CONFIG_TABLE[i].is_dir)
				ret = my_copy_dir(file_path, NAS_CONFIG_TABLE[i].name, TRUE);
			else
				ret = my_copy_file(file_path, NAS_CONFIG_TABLE[i].name, TRUE);
			ret &= op_mask;
			if (ret) {
#ifdef DEBUG
				printf("Recover_System_Config : fails to recover %s\n",
					NAS_CONFIG_TABLE[i].name);
#endif
				errcnt++;
			}
		}
		break; // only once is enough
	}

	return -errcnt;
}

//==========================================================================
// name : Backup_System_Config
// description : copy the config files from the working directories to
//		the all existent & valid root partitions
//		if fails to copy a config file when the function is not supported
//		or the file's optional,	ignore the failure
// output : the times error occurs
//==========================================================================
int Backup_System_Config(void)
{
	int ret = SUCCESS, part_cnt, i, j, errcnt=0;
	BOOL op_mask, is_supported;
	char roots[MAX_DRIVE_NO][HD_DEVICE_NAME_LENGTH];
	char backupdir[BUF_SIZE], file_path[BUF_SIZE], tmp[BUF_SIZE];

	// find all valid root partitions
	part_cnt = get_usable_root_partitions(roots, MAX_DRIVE_NO);
	if (part_cnt<0) {
#ifdef DEBUG
		printf("Backup_Config_System : get_usable_root_partitions fails, err=%d\n",
			part_cnt);
#endif
		return part_cnt;
	}

	for (j=0; j<part_cnt; j++) {
		sprintf(backupdir, "%s/%s", roots[j], BACKUP_DIR);
		if (!config_backup_dir_exists(roots[j])) {
			ret = create_dir(backupdir);
			if (ret<0) {
#ifdef DEBUG
				printf("Backup_System_Config : fails to create_dir(%s)\n", backupdir);
#endif
				continue;
			}
		}

		// backup all config files or directories
		for (i=0; i< sizeof(NAS_CONFIG_TABLE)/sizeof(NAS_CONFIG_TABLE[0]); i++) {
			if (NAS_CONFIG_TABLE[i].premise == NULL)
				is_supported = TRUE;
			else
				is_supported = Get_Profile_Boolean(NAS_CONFIG_TABLE[i].premise, "Support", FALSE);
			op_mask = is_supported & ~(NAS_CONFIG_TABLE[i].optional);

			get_file_name(NAS_CONFIG_TABLE[i].name, tmp);
			sprintf(file_path, "%s/%s", backupdir, tmp);
			if (NAS_CONFIG_TABLE[i].is_dir)
				ret = my_copy_dir(NAS_CONFIG_TABLE[i].name, file_path, TRUE);
			else
				ret = my_copy_file(NAS_CONFIG_TABLE[i].name, file_path, TRUE);
			ret &= op_mask;
			if (ret) {
#ifdef DEBUG
				printf("Backup_Config_System : fails to backup %s\n",
					NAS_CONFIG_TABLE[i].name);
#endif
				errcnt++;
			}
		}
	}

	return errcnt;
}
