//**************************************************************************
//
//	Copyright (c) 2000  ICP Electronics Inc.  All Rights Reserved.
//
//	FILE:
//		cfg_network.c
//
//	Abstract: 
//		System configuration Access Functions.
//
//	FUNCTIONS:	TBD.
//
//	COMMENTS: 	N/A
//
//	HISTORY:
//		05/02/01	Louis Tsai modified
//		12/06/00	kw lee created
//
//**************************************************************************

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "config.h"
#include "Util.h"

extern int print_if(IFInfo *pif);
extern int set_samba_WINS_server(char *wins_ip);

/* ------------------------------------------------------------------------- **
 * Build options...
 */

//#define CFG_SYSTEM_TEST		// Build as a standalone program (with main), for testing only...



/* ------------------------------------------------------------------------- **
 * Constants...
 */



/* ------------------------------------------------------------------------- **
 * Variables...
 *
 *  bufr        - pointer to a global buffer.  This is probably a kludge,
 *                but it was the nicest kludge I could think of (for now).
 *  bSize       - The size of the global buffer <bufr>.
 */






/* ------------------------------------------------------------------------- */
/*         		Function Implementation                              */
/* ------------------------------------------------------------------------- */



#define	SZ_NETWORK_SECTION		"Network"
#define	SZ_INTERFACE_NUMBER_FIELD	"Interface Number"
#define	SZ_USE_DHCP_FIELD		"Use DHCP"
#define	SZ_MAC_ADDRESS_FIELD		"MAC Address"
#define	SZ_IP_ADDRESS_FIELD		"IP Address"
#define	SZ_SUBNET_MASK_FIELD		"Subnet Mask"
#define	SZ_GATEWAY_FIELD		"Gateway"
#define	SZ_BROADCAST_FIELD		"Broadcast"
#define	SZ_WINS_IP_FIELD		"WINS IP"
#define	SZ_CONFIGURED			"Configured"



BOOL Is_Network_Started()
{
	int ret = system("/sbin/ifconfig eth0 > /dev/null");
	return (ret == 0);
}


int Restart_Network()
{
	int ret = system("/etc/init.d/network.sh restart > /dev/null 2>&1");
	return ret;
}


/*
 * Get network Interface number
 * Return: number of if
 */
int Get_Network_Interface_Number(int *in)
{
	char bRet[BUF_SIZE];
	int  val;

	memset(bRet, 0, sizeof(bRet));
	val = GetProfileString(SZ_NETWORK_SECTION, SZ_INTERFACE_NUMBER_FIELD, "1", bRet, sizeof(bRet));
	if(!val)
		return 0;
	if(bRet[0] == 0)
		return 0;
	*in = atoi(bRet);
	return 1;
}

/*
 * BUGBUG: Currently we only support up to 8 eth interface
 *	No other interface like "fiber" is supported
 *	To support other type of NIC, some field in [Network] may need:
 *		[Network]
 *			Interface 1 = eth0
 *			Interface 2 = fic0 ... 
 *	However, it's not currently supported...
 * Return:
 *	0			pass
 *	ERROR_INVALID_PARAM		invalid parameter
 */
int Get_Network_Interface_Name(int index, char *buf, int buf_size)
{
	int ret = SUCCESS;
	char *p = NULL;

	switch (index) {
		case 0: 
			p = "eth0";	break;
		case 1: 
			p = "eth1";	break;
		case 2: 
			p = "eth2";	break;
		case 3: 
			p = "eth3";	break;
		case 4: 
			p = "eth4";	break;
		case 5: 
			p = "eth5";	break;
		case 6: 
			p = "eth6";	break;
		case 7: 
			p = "eth7";	break;
		default:
#ifdef	DEBUG		
			printf("GetNetworkInterfaceName: invalid index.\n");
#endif
			break;
	}
	if (p != NULL && NULL != buf) strncpy(buf, p, buf_size);
	else ret = ERROR_INVALID_PARAM;

	return ret;	
}

/*
 * Get working network Interface informartion 
 * Return:
 *	0			pass
 *	ERROR_FAIL	can not get network interface from Linux system
 */
int Get_Network_Interface_Volatile(int index, IFInfo *pif)
{
	int ret;
	char bRet[BUF_SIZE];

	Get_Network_Interface_Name(index, pif->device_name, sizeof(pif->device_name));
	GetProfileString(pif->device_name, SZ_USE_DHCP_FIELD, "FALSE", bRet, sizeof(bRet));
	pif->use_dhcp = IS_TRUE(bRet);
	if((ret=print_if(pif))<=0) {
		ret = ERROR_FAIL;
	}
	else 
		ret = SUCCESS;
	return ret;
}

/*
 * Get network Interface MAC address no matter cable is plugged or not 
 * Return:
 *	0			pass
 *	ERROR_FAIL	can not get MAC address from Linux system
 */
int Get_Network_Interface_MAC_Address(int index, char *MAC_addr, int addr_len)
{
	int ret = SUCCESS;
	int len;
	IFInfo ifi;

	if(addr_len > HWADDR_STR_LEN)
		len = HWADDR_STR_LEN;
	else
		len = addr_len;
	memset(&ifi, 0, sizeof(ifi));
	Get_Network_Interface_Name(index, ifi.device_name, sizeof(ifi.device_name));
	print_if(&ifi);
	memset(MAC_addr, 0, addr_len);
	strncpy(MAC_addr, ifi.hw_addr, len);
	return ret;
}


/*
 * Get Interface information from config file
 * Return:
 *	0			pass
 */
int Get_Network_Interface_Config(int index, IFInfo *pif)
{
	char bRet[BUF_SIZE];
	int len;

	Get_Network_Interface_Name(index, pif->device_name, sizeof(pif->device_name));
	GetProfileString(pif->device_name, SZ_USE_DHCP_FIELD, "FALSE", bRet, sizeof(bRet));
	pif->use_dhcp = IS_TRUE(bRet);
	len = sizeof(pif->hw_addr);
	GetProfileString(pif->device_name, SZ_MAC_ADDRESS_FIELD, "", pif->hw_addr, len);
	len = sizeof(pif->ip_addr);
	GetProfileString(pif->device_name, SZ_IP_ADDRESS_FIELD, "", pif->ip_addr, len);
	len = sizeof(pif->net_mask);
	GetProfileString(pif->device_name, SZ_SUBNET_MASK_FIELD, "", pif->net_mask, len);
	len = sizeof(pif->gateway);
	GetProfileString(pif->device_name, SZ_GATEWAY_FIELD, "", pif->gateway, len);
	len = sizeof(pif->broadcast);
	GetProfileString(pif->device_name, SZ_BROADCAST_FIELD, "", pif->broadcast, len);
	return SUCCESS;
}

/*
 * Set Interface information to config file
 * Return:
 *	0			pass
 *  ERROR_WRITE_FILE	can not write to config file
 */
int Set_Network_Interface_Config(IFInfo *pif,int restart_now)
{
	int val, ret = SUCCESS;

	val = WriteProfileString(pif->device_name, SZ_USE_DHCP_FIELD, BOOL2STR(pif->use_dhcp));
	if(!val)
		return ERROR_WRITE_FILE;
	val = WriteProfileString(pif->device_name, SZ_MAC_ADDRESS_FIELD, pif->hw_addr);
	if(!val)
		return ERROR_WRITE_FILE;
	val = WriteProfileString(pif->device_name, SZ_IP_ADDRESS_FIELD, pif->ip_addr);
	if(!val)
		return ERROR_WRITE_FILE;
	val = WriteProfileString(pif->device_name, SZ_SUBNET_MASK_FIELD, pif->net_mask);
	if(!val)
		return ERROR_WRITE_FILE;
	val = WriteProfileString(pif->device_name, SZ_GATEWAY_FIELD, pif->gateway);
	if(!val)
		return ERROR_WRITE_FILE;
	if (strcasecmp(pif->broadcast,"0.0.0.0") == 0 ) {
		ULONG bin_ip,bin_netmask,bin_broadcast;
		struct in_addr in;
		bin_ip = inet_addr ( pif->ip_addr);
		bin_netmask = inet_addr (pif->net_mask);
		bin_broadcast = (bin_ip & bin_netmask) | (~bin_netmask);
		in.s_addr = bin_broadcast;
			
		sprintf(pif->broadcast, "%s", (char *)inet_ntoa(in));
	}
	val = WriteProfileString(pif->device_name, SZ_BROADCAST_FIELD, pif->broadcast);
	if(!val)
		return ERROR_WRITE_FILE;
		
	// set configured flag for ethx to TRUE, thereafter it will follow the definition now.
	WriteProfileString(pif->device_name, SZ_CONFIGURED, "TRUE");
		
	if(restart_now) {
		Restart_Network();		
		Update_System_Display();
	}	
	return ret;
}

/*
 * Get WINDS Server IP Address from config file
 * Return:
 *	0			pass
 *	ERROR_READ_FILE	can not read from config file
 */
int Get_WINS_Server_IP(char *wins_ip, int buf_size)
{
	int val, ret = SUCCESS;
	
	val = GetProfileString(SZ_NETWORK_SECTION, SZ_WINS_IP_FIELD, "", wins_ip, buf_size);
	if(!val)
		return ERROR_READ_FILE;
	return ret;
}

/*
 * Set WINDS Server IP Address to config file
 * Return:
 *	0			pass
 *	ERROR_WRITE_FILE	can not write to config file
 */
int Set_WINS_Server_IP(char *wins_ip)
{
	int val, ret = SUCCESS;
	
	val = WriteProfileString(SZ_NETWORK_SECTION, SZ_WINS_IP_FIELD, wins_ip);
	if(!val)
		return ERROR_WRITE_FILE;
	
	// What should we do after set WINS IP?
	//John modified 2001-10-18:
	//ret = set_samba_WINS_server(wins_ip);
	if(!set_samba_WINS_server(wins_ip))	ret = ERROR_FAIL;

	return ret;
}


BOOL Is_Network_Interface_Configured(int index)
{
	char if_name[BUF_SIZE], buf[BUF_SIZE];
	if (Get_Network_Interface_Name(index, if_name, sizeof(if_name)) != SUCCESS)
		return FALSE;
	GetProfileString(if_name, SZ_CONFIGURED, "TRUE", buf, sizeof(buf));
	return NOT_FALSE(buf);
}
