//**************************************************************************
//
//	Copyright (c) 2001  ICP Electronics Inc.  All Rights Reserved.
//
//	FILE:
//		cfg_dhcp.c
//
//	Abstract: 
//		System configuration Access Functions.
//
//	FUNCTIONS:	Setting and getting the dhcp configuration from uLinux.
//		public function:
//			BOOL Is_DNS_Enable (void);
//			BOOL Enable_DNS (int Enable);
//			char * Get_Name_Server (int index);
//if not use new function{
//			int GetProfileInteger (char * section_name, char * key_name, 
//							int default_value);
//			BOOL GetProfileBOOL (char * section_name, char * key_name, 
//							BOOL default_value);
//}
//	COMMENTS: 	It must working with util.c
//
//	HISTORY:
//		V1.0 2001/08/09		Tiger Fu created
//
//**************************************************************************

#include	<stdio.h>
#include	<ctype.h>
#include	<stdlib.h>
#include	<string.h>

#include	"Util.h"

#include	"cfg_dns.h"

#define		DNS_BUF_SIZE		1024

//  system command
#define		DEVICE_NULL		"/dev/null"
#define		DNS_SHELL_SCRIPT	"/etc/init.d/named.sh"
#define		NAME_DAEMON_CONTROL	"ndc"

#define		SLAVE_ZONE_EXT		".slv"
#define		MASTER_ZONE_EXT		".for"

/*
 * Check if NAS support DNS
 * Return:
 *      true    Support
 *      false   Not support
 */
BOOL Is_Support_DNS()
{
        char buf[256];
        GetProfileString(DNS_SECTION, "Support", "FALSE", buf, sizeof(buf));
        return NOT_FALSE(buf);
}


//**************************************************************************
//	Function: 	Is_DNS_Enable
//	Description:	Check DNS whether enable or eisable, base on 
//			contents of uLinux.conf.
//	Parameter:	None
//	Return value:	TRUE = DNS enable | FALSE = DNS disable
//
BOOL Is_DNS_Enable (void)
{
	return is_private_dns_enable (DEFAULT_CFG_FILE);
}

BOOL is_private_dns_enable (char * cfg_file)
{
	BOOL result = FALSE;
	result = Get_Private_Profile_Boolean (DNS_SECTION, ENABLE, FALSE, cfg_file);
	if (!result)
		result = Get_Private_Profile_Boolean (DNS_SECTION, MODIFY_ACTION, FALSE, cfg_file);
	return result;
}

int set_private_dns_enable (BOOL Enable, char * cfg_file)
{
	BOOL result = FALSE;
	switch (Enable)
	{
	case FALSE:
		result = SET_PRIVATE_PROFILE_STRING (DNS_SECTION, ENABLE, "FALSE", cfg_file);
		break;
	case TRUE:
		result = SET_PRIVATE_PROFILE_STRING (DNS_SECTION, ENABLE, "TRUE", cfg_file);
		break;
	}
	return result;
}
//**************************************************************************
//	Function: 	Enable_DNS
//	Description:	Start or stop DNS serveice
//	Parameter:	int Enable(0=disable, 1=enable).
//	Return value:	
//
BOOL Enable_DNS (BOOL Enable)
{
	BOOL result = TRUE;
	char command[256];
	memset (command, '\0', sizeof (command));

	result = set_private_dns_enable (Enable, DEFAULT_CFG_FILE);
	switch (Enable)
	{
	case FALSE:
		sprintf (command, "%s stop > %s", DNS_SHELL_SCRIPT, DEVICE_NULL);
		system (command);
		break;
	case TRUE:
		memset (command, '\0', sizeof (command));
		sprintf (command,"%s restart > %s", DNS_SHELL_SCRIPT, DEVICE_NULL);
		system (command);
		break;
	}
	return result;
}

//**************************************************************************
//	Function: 	Reload_DNS_Config
//	Description:	
//	Parameter:	
//			
//	Return value:	
//
void Reload_DNS_Config (void)
{
	char command[256];
	sprintf (command, "%s reload > %s", DNS_SHELL_SCRIPT, DEVICE_NULL);
	system (command);
}

//**************************************************************************
//**************************************************************************
//	private function
//**************************************************************************
//**************************************************************************
//  0 == basic page, 1 == advance page
int get_which_page (void)
{
	return get_private_which_page (DEFAULT_CFG_FILE);
}

int get_private_which_page (char * cfg_file)
{
	char which_page[256];
	bzero (which_page, sizeof (which_page));

	GET_PRIVATE_PROFILE_STRING (DNS_SECTION, WHICH_PAGE, "", which_page, sizeof (which_page), cfg_file);

	if (!strcmp (which_page, "Basic"))
		return 0;
	
	if (!strcmp (which_page, "Advance"))
		return 1;
	return 0;
}
//  0 == basic page, 1 == advance page
void set_which_page (int page)
{
	set_private_which_page (page, DEFAULT_CFG_FILE);
}

void set_private_which_page (int page, char * cfg_file)
{
	if (page == 0)
		SET_PRIVATE_PROFILE_STRING (DNS_SECTION, WHICH_PAGE, "Basic", cfg_file);
	else
		SET_PRIVATE_PROFILE_STRING (DNS_SECTION, WHICH_PAGE, "Advance", cfg_file);
}
//
void set_modify_action (BOOL action)
{
	set_private_modify_action (action, DEFAULT_CFG_FILE);
}

void set_private_modify_action (BOOL action, char * cfg_file)
{
	if (action)
		SET_PRIVATE_PROFILE_STRING (DNS_SECTION, MODIFY_ACTION, "TRUE", cfg_file);
	else
		SET_PRIVATE_PROFILE_STRING (DNS_SECTION, MODIFY_ACTION, "FALSE", cfg_file);
}
////////////
char * get_domain_name (void)
{
	return get_private_domain_name (DEFAULT_CFG_FILE);
}

char * get_private_domain_name (char * cfg_file)
{
	int result = 0;
	static char domain_name[256];
	memset (domain_name, '\0', sizeof (domain_name));
	
	result = GET_PRIVATE_PROFILE_STRING (DNS_SECTION, DOMAIN_NAME, "", domain_name, sizeof (domain_name), cfg_file);
	if (!result)
		return "";
		
	return domain_name;
}
////////////
BOOL set_domain_name (char * domain_name)
{
	return set_private_domain_name (domain_name, DEFAULT_CFG_FILE);
}

BOOL set_private_domain_name (char * domain_name, char * cfg_file)
{
	return SET_PRIVATE_PROFILE_STRING (DNS_SECTION, DOMAIN_NAME, domain_name, cfg_file);
}
////////////
char * get_conf_file_name (void)
{
	int result = 0;
	static char conf_file_name[256];
	static char default_conf_file_name[] = "/etc/named.conf";
	memset (conf_file_name, '\0', sizeof (conf_file_name));
	
	result = GET_PROFILE_STRING (DNS_SECTION, NAMED_CONF, "/etc/named.conf", conf_file_name, sizeof (conf_file_name));
	if (!result)
		return default_conf_file_name;
		
	return conf_file_name;
}

/////
enum e_nstype get_ns_type (void)
{
	return get_private_ns_type (DEFAULT_CFG_FILE);
}
enum e_nstype get_private_ns_type (char * cfg_file)
{
	char type[256];
	memset (type, '\0', sizeof (type));
	
	GET_PRIVATE_PROFILE_STRING (DNS_SECTION, NAME_SERVER_TYPE, "Master", type, sizeof (type), cfg_file);
	if (!strcmp  (String_To_Upper_Case (type), "MASTER"))
		return E_Master;
	else if (!strcmp  (String_To_Upper_Case (type), "SLAVE"))
		return E_Slave;
	else if (!strcmp  (String_To_Upper_Case (type), "CACHE"))
		return E_Cache;
	else 
		return E_Un_Know_Type;
}
//////

//  success returns TRUE, else returns FALSE
BOOL set_ns_type (enum e_nstype type)
{
	return set_private_ns_type (type, DEFAULT_CFG_FILE);
}
BOOL set_private_ns_type (enum e_nstype type, char * cfg_file)
{
	BOOL result = FALSE;
	switch (type)
	{
	case E_Master:
		result = SET_PRIVATE_PROFILE_STRING (DNS_SECTION, NAME_SERVER_TYPE, "Master", cfg_file);
		break;
	case E_Slave:
		result = SET_PRIVATE_PROFILE_STRING (DNS_SECTION, NAME_SERVER_TYPE, "Slave", cfg_file);
		break;
	case E_Cache:
		result = SET_PRIVATE_PROFILE_STRING (DNS_SECTION, NAME_SERVER_TYPE, "Cache", cfg_file);
		break;	
	case E_Un_Know_Type:
		break;
	}
	return result;
}
/////


BOOL is_modify (void)
{
	return Get_Profile_Boolean (DNS_SECTION, CONFIG_MODIFIED, FALSE);
}

char * get_db_path (void)
{
	int result = 0;
	static char db_path[256];
	static char default_db_path[] = "/var/named";
	
	memset (db_path, '\0', sizeof (db_path));
	
	result = GET_PROFILE_STRING (DNS_SECTION, DB_PATH, "/var/named", db_path, sizeof (db_path));
	if (!result)
		return default_db_path;
		
	return db_path;
}
///////////
BOOL check_section_whether_exist (char * zone_name)
{
	return check_private_section_whether_exist (zone_name, DEFAULT_CFG_FILE);
}

BOOL check_private_section_whether_exist (char * zone_name, char * cfg_file)
{
	FILE * fp = NULL;
	char section_name[64];
	char s[256];
	
	memset (section_name, '\0', sizeof (section_name));
	sprintf (section_name, "[%s]", zone_name);
	fp = fopen (cfg_file, "r");

	if (fp)
	{
		while (fgets (s, sizeof (s), fp))
		{
			if (strstr (s, section_name))
			{
				fclose (fp);
				return TRUE;
			}
		}
		fclose (fp);
	}
	return FALSE;
}
///////////
int get_master_zone_number (void)
{
	return get_private_master_zone_number (DEFAULT_CFG_FILE);
}
int get_private_master_zone_number (char * cfg_file)
{
	char buf[80];
	char keyname[16];
	int index = 1;
	
	memset (buf, '\0', sizeof (buf));
	
	while (TRUE)
	{
		memset (keyname, '\0', sizeof (keyname));
		sprintf (keyname, "MasterZone%d", index);

		GET_PRIVATE_PROFILE_STRING (DNS_SECTION, keyname, "???", buf, sizeof (buf), cfg_file);
		if (!strcmp (buf, "???") || !strcmp (buf, ""))
			break;
		index++;
	}
	return index - 1;
}
/////////
char * get_master_zone_name (int index)
{
	return get_private_master_zone_name (index, DEFAULT_CFG_FILE);
}
char * get_private_master_zone_name (int index, char * cfg_file)
{
	static char zone_name[128];
	char keyname[16];
	
	memset (zone_name, '\0', sizeof (zone_name));
	memset (keyname, '\0', sizeof (keyname));
	sprintf (keyname, "MasterZone%d", index);

	GET_PRIVATE_PROFILE_STRING (DNS_SECTION, keyname, "???", zone_name, sizeof (zone_name), cfg_file);
	if (!strcmp (zone_name, "???") || !strcmp (zone_name, ""))
		return NULL;
		
	return zone_name;
}
/////////
int get_master_zone_info (char * zone_name, ZONE_INFO * zone_info)
{
	return get_private_master_zone_info (zone_name, zone_info, DEFAULT_CFG_FILE);
}
int get_private_master_zone_info (char * zone_name, ZONE_INFO * zone_info, char * cfg_file)
{
	char buf[128];
	int result = 0;  //  success

	if (check_private_section_whether_exist (zone_name, cfg_file) == TRUE)
	{
		strcpy (zone_info->zone_name, zone_name);
		strcpy (zone_info->zone_type, "master");

		GET_PRIVATE_PROFILE_STRING (zone_name, "file", "???", buf, sizeof (buf), cfg_file);

		if (strcmp (buf, "???") && strcmp (buf, ""))
			strcpy (zone_info->zone_file, buf);
		else
			result = -2;  //  the key for name of "file" do not exist
	}
	else
		result = -1;  //  the section do not exist

	return result;
}
/////////
BOOL set_master_zone_info (ZONE_INFO * zone_info)
{
	return set_private_master_zone_info (zone_info, DEFAULT_CFG_FILE);
}
BOOL set_private_master_zone_info (ZONE_INFO * zone_info, char * cfg_file)
{
	char keyname[32];
	BOOL result;
	int index = get_private_master_zone_number (cfg_file);
	
	while (index)
	{
		if (!strcmp (zone_info->zone_name, get_private_master_zone_name (index--, cfg_file)))
			return FALSE;
	}

	index = get_private_master_zone_number (cfg_file);

	memset (keyname, '\0', sizeof (keyname));
	sprintf (keyname, "MasterZone%d", index + 1);
	
	strcpy (zone_info->zone_file, zone_info->zone_name);
	strcat (zone_info->zone_file, MASTER_ZONE_EXT);

	result = SET_PRIVATE_PROFILE_STRING (DNS_SECTION, keyname, zone_info->zone_name, cfg_file);
	result = SET_PRIVATE_PROFILE_STRING (zone_info->zone_name, "file", zone_info->zone_file, cfg_file);

	return result;
}
////////
int modify_master_zone_info (char * zone_name, int index)
{
	return modify_private_master_zone_info (zone_name, index, DEFAULT_CFG_FILE);
}
int modify_private_master_zone_info (char * zone_name, int index, char * cfg_file)
{
	char keyname[32];
	char original_name[128];
	char buf[256], rr_buf[256];
	BOOL result;
	int rr_index = 0, rr_total, num = get_private_master_zone_number (cfg_file);

	while (num)
	{
		if (!strcmp (zone_name, get_private_master_zone_name (num--, cfg_file)))
			return FALSE;
	}

	memset (keyname, '\0', sizeof (keyname));
	sprintf (keyname, "MasterZone%d", index);

	strcpy (original_name, get_private_master_zone_name (index, cfg_file));

	GET_PRIVATE_PROFILE_STRING (original_name, "file", "", buf, sizeof (buf), cfg_file);
	SET_PRIVATE_PROFILE_STRING (zone_name, "file", buf, cfg_file);

	rr_total = get_private_resource_record_number (original_name, cfg_file);
	while (rr_index++ < rr_total)
	{
		strcpy (rr_buf, get_private_resource_record (original_name, rr_index, cfg_file));
		set_private_resource_record (zone_name, rr_buf, cfg_file);
	}

	Conf_Remove_Section (cfg_file, original_name);

	result = SET_PRIVATE_PROFILE_STRING (DNS_SECTION, keyname, zone_name, cfg_file);

	return result;
}
///////////
int remove_master_zone_info (int index)
{
	return remove_private_master_zone_info (index, DEFAULT_CFG_FILE);
}
int remove_private_master_zone_info (int index, char * cfg_file)
{
	char keyname[32];
	char buf[128];
	int result = -1;  //  the key name do not exist
	int total = 0;

	sprintf (keyname, "MasterZone%d", index);
	
	total = get_private_master_zone_number (cfg_file);

	GET_PRIVATE_PROFILE_STRING (DNS_SECTION, keyname, "???", buf, sizeof (buf), cfg_file);
	Conf_Remove_Field (cfg_file, DNS_SECTION, keyname);
	
	if (strcmp (buf, "???") && strcmp (buf, ""))
	{
		permutation_private_zone (S_MASTER_ZONE, index, total, cfg_file);
		Conf_Remove_Section (cfg_file, buf);
		result = 0;
	}

	return result;
}
///////////
int get_zone_resource_record_number (char * zone_name)
{
	return get_private_resource_record_number (zone_name, DEFAULT_CFG_FILE);
}

int get_private_resource_record_number (char * zone_name, char * cfg_file)
{
	char buf[80];
	char keyname[16];
	int index = 1;
	
	memset (buf, '\0', sizeof (buf));
	
	while (TRUE)
	{
		memset (keyname, '\0', sizeof (keyname));
		sprintf (keyname, "ResourceRecord%d", index);
		GET_PRIVATE_PROFILE_STRING (zone_name, keyname, "???", buf, sizeof (buf), cfg_file);
		if (!strcmp (buf, "???") || !strcmp (buf, ""))
			break;
		index++;
	}
	return index - 1;
}
////////

//  if success return great then 0, else less then and equel 0
int set_zone_resource_record (char * zone_name, char * resource_record)
{
	return set_private_resource_record (zone_name, resource_record, DEFAULT_CFG_FILE);
}

int set_private_resource_record (char * zone_name, char * resource_record, char * cfg_file)
{
	int result = -1;  //  the section not exist
	int index = 0;
	char keyname[32];
	
	memset (keyname, '\0', sizeof (keyname));
	if (check_private_section_whether_exist (zone_name, cfg_file) == TRUE)
	{
		index = get_private_resource_record_number (zone_name, cfg_file);
		while (index)
		{
			if (!strcmp (resource_record, get_private_resource_record (zone_name, index--, cfg_file)))
				return -2;  //  the resource record was exist
		}

		index = get_private_resource_record_number (zone_name, cfg_file);
		sprintf (keyname, "ResourceRecord%d", index + 1);
		result = SET_PRIVATE_PROFILE_STRING (zone_name, keyname, resource_record, cfg_file);
	}
	return result;
}
////////
char * get_zone_resource_record (char * zone_name, int index)
{
	return get_private_resource_record (zone_name, index, DEFAULT_CFG_FILE);
}

char * get_private_resource_record (char * zone_name, int index, char * cfg_file)
{
	static char resource_record[256];
	char keyname[32];
	
	memset (resource_record, '\0', sizeof (resource_record));
	memset (keyname, '\0', sizeof (keyname));

	sprintf (keyname, "ResourceRecord%d", index);

	GET_PRIVATE_PROFILE_STRING (zone_name, keyname, "???", resource_record, sizeof (resource_record), cfg_file);

	if (!strcmp (resource_record, "???") || !strcmp (resource_record, ""))
		return NULL;
	return resource_record;
}
////////
int remove_zone_record_by_index (char * zone_name, int index)
{
	return remove_private_record_by_index (zone_name, index, DEFAULT_CFG_FILE);
}
int remove_private_record_by_index (char * zone_name, int index, char * cfg_file)
{
	char keyname[16];
	int total = get_private_resource_record_number (zone_name, cfg_file);

	memset (keyname, '\0', sizeof (keyname));
	sprintf (keyname, "ResourceRecord%d", index);
	
	if (Conf_Remove_Field (cfg_file, zone_name, keyname) == ERROR_NOT_FOUND)
		return -1;
	permutation_private_resource_record (zone_name, index, total, cfg_file);
	
	return 0;
}
////////
int remove_zone_record_by_value (char * zone_name, char * value)
{
	return remove_private_record_by_value (zone_name, value, DEFAULT_CFG_FILE);  //  not found
}
int remove_private_record_by_value (char * zone_name, char * value, char * cfg_file)
{
	char keyname[32];
	int index = 0, total = get_private_resource_record_number (zone_name, cfg_file);
	
	while (index++ < total)
	{
		sprintf (keyname, "ResourceRecord%d", index);
		if (!strcmp (value, get_private_resource_record (zone_name, index, cfg_file)))
		{
			Conf_Remove_Field (cfg_file, zone_name, keyname);
			permutation_private_resource_record (zone_name, index, total, cfg_file);
			return 0;
		}
	}
	return -1;  //  not found
}
////////
void permutation_private_resource_record (char * zone_name, int index, int total, char * cfg_file)
{
	char buf[64];
	char keyname[32], keyname1[32];

	while (index++ < total)
	{
		sprintf (keyname, "ResourceRecord%d", index - 1);
		sprintf (keyname1, "ResourceRecord%d", index);
		GET_PRIVATE_PROFILE_STRING (zone_name, keyname1, "", buf, sizeof (buf), cfg_file);
		SET_PRIVATE_PROFILE_STRING (zone_name, keyname, buf, cfg_file);
	}
	Conf_Remove_Field (cfg_file, zone_name, keyname1);	
}

//==================================================================
//==================================================================
//==================================================================
//==================================================================

int get_slave_zone_number (void)
{
	return get_private_slave_zone_number (DEFAULT_CFG_FILE);
}
int get_private_slave_zone_number (char * cfg_file)
{
	char buf[80];
	char keyname[16];
	int index = 1;
	
	memset (buf, '\0', sizeof (buf));
	
	while (TRUE)
	{
		memset (keyname, '\0', sizeof (keyname));
		sprintf (keyname, "SlaveZone%d", index);
		GET_PRIVATE_PROFILE_STRING (DNS_SECTION, keyname, "???", buf, sizeof (buf), cfg_file);
		if (!strcmp (buf, "???") || !strcmp (buf, ""))
			break;
		index++;
	}
	return index - 1;
}
////////////

char * get_slave_zone_name (int index)
{
	return get_private_slave_zone_name (index, DEFAULT_CFG_FILE);
}

char * get_private_slave_zone_name (int index, char * cfg_file)
{
	static char zone_name[128];
	char keyname[16];
	
	memset (zone_name, '\0', sizeof (zone_name));
	memset (keyname, '\0', sizeof (keyname));
	sprintf (keyname, "SlaveZone%d", index);

	GET_PRIVATE_PROFILE_STRING (DNS_SECTION, keyname, "???", zone_name, sizeof (zone_name), cfg_file);
	if (!strcmp (zone_name, "???") || !strcmp (zone_name, ""))
		return NULL;

	return zone_name;
}

////////////
BOOL set_slave_zone_info (SLAVE_ZONE_INFO * zone_info)
{
	return set_private_slave_zone_info (zone_info, DEFAULT_CFG_FILE);
}

BOOL set_private_slave_zone_info (SLAVE_ZONE_INFO * zone_info, char * cfg_file)
{
	char keyname[32];
	BOOL result;
	int index = get_private_slave_zone_number (cfg_file);
	
	while (index)
	{
		if (!strcmp (zone_info->zone_info.zone_name, get_private_slave_zone_name (index--, cfg_file)))
			return FALSE;
	}
	
	//if (check_private_section_whether_exist (zone_info->zone_info.zone_name, cfg_file) == TRUE)
	//	return FALSE;
	
	index = get_private_slave_zone_number (cfg_file);
	memset (keyname, '\0', sizeof (keyname));
	sprintf (keyname, "SlaveZone%d", index + 1);
	
	result = SET_PRIVATE_PROFILE_STRING (DNS_SECTION, keyname, zone_info->zone_info.zone_name, cfg_file);
	result = SET_PRIVATE_PROFILE_STRING (zone_info->zone_info.zone_name, "masters", zone_info->masters_ip, cfg_file);
	strcpy (zone_info->zone_info.zone_file, zone_info->zone_info.zone_name);
	strcat (zone_info->zone_info.zone_file, SLAVE_ZONE_EXT);
	result = SET_PRIVATE_PROFILE_STRING (zone_info->zone_info.zone_name, "file", zone_info->zone_info.zone_file, cfg_file);

	return result;
}
////////////
int get_slave_zone_info (char * zone_name, SLAVE_ZONE_INFO * zone_info)
{
	return get_private_slave_zone_info (zone_name, zone_info, DEFAULT_CFG_FILE);
}
int get_private_slave_zone_info (char * zone_name, SLAVE_ZONE_INFO * zone_info, char * cfg_file)
{
	char buf[128];
	int result = -1;  //  the section do not exist

	if (check_private_section_whether_exist (zone_name, cfg_file) == TRUE)
	{
		strcpy (zone_info->zone_info.zone_name, zone_name);
		strcpy (zone_info->zone_info.zone_type, "slave");
		strcpy (zone_info->zone_info.zone_file, zone_name);
		strcat (zone_info->zone_info.zone_file, SLAVE_ZONE_EXT);
		
		GET_PRIVATE_PROFILE_STRING (zone_name, "masters", "???", buf, sizeof (buf), cfg_file);
		if (strcmp (buf, "???") && strcmp (buf, ""))
		{
			strcpy (zone_info->masters_ip, buf);
			result = 0;  //  success
		}
		else
			result = -2;  //  the key for name of "masters" do not exist
	}

	return result;
}
////////////
int remove_slave_zone_by_index (int index)
{
	return remove_private_slave_zone_by_index (index, DEFAULT_CFG_FILE);
}
int remove_private_slave_zone_by_index (int index, char * cfg_file)
{
	char keyname[32];
	char buf[128];
	int result = -1;  //  the key name do not exist
	int total = get_private_slave_zone_number (cfg_file);

	sprintf (keyname, "SlaveZone%d", index);
	
	GET_PRIVATE_PROFILE_STRING (DNS_SECTION, keyname, "???", buf, sizeof (buf), cfg_file);
	Conf_Remove_Field (cfg_file, DNS_SECTION, keyname);
	
	if (strcmp (buf, "???")  && strcmp (buf, ""))
	{
		permutation_private_zone (S_SLAVE_ZONE, index, total, cfg_file);
		Conf_Remove_Section (cfg_file, buf);
		result = 0;
	}

	return result;
}
////////////
int remove_slave_zone_by_value (char * value)
{
	return remove_private_slave_zone_by_value (value, DEFAULT_CFG_FILE);  //  the value not exist
}
int remove_private_slave_zone_by_value (char * value, char * cfg_file)
{
	char keyname[32];
	char buf[128];
	int index = 0, total = get_private_slave_zone_number (cfg_file);

	while (index++ < total)
	{
		sprintf (keyname, "SlaveZone%d", index);

		if (!strcmp (value, get_private_slave_zone_name (index, cfg_file)))
		{
			GET_PRIVATE_PROFILE_STRING (DNS_SECTION, keyname, "???", buf, sizeof (buf), cfg_file);
			Conf_Remove_Field (cfg_file, DNS_SECTION, keyname);

			permutation_private_zone (S_SLAVE_ZONE, index, total, cfg_file);
			Conf_Remove_Section (cfg_file, value);
			return 0;
		}
	}

	return -1;  //  the value not exist
}
//==================================================================
//==================================================================
//==================================================================
//==================================================================
void permutation_zone (int type, int index, int total)
{
	permutation_private_zone (type, index, total, DEFAULT_CFG_FILE);
}
void permutation_private_zone (int type, int index, int total, char * cfg_file) //  type = 1->master 2->slave 3->resrouce_record
{
	char buf[64];
	char keyname[16], keyname1[16];
	if (type <= 0 || type > 2)
		return;

	while (index++ < total)
	{
		switch (type)
		{
		case S_MASTER_ZONE:
			sprintf (keyname, "MasterZone%d", index - 1);
			sprintf (keyname1, "MasterZone%d", index);
			
			GET_PRIVATE_PROFILE_STRING (DNS_SECTION, keyname1, "", buf, sizeof (buf), cfg_file);
			SET_PRIVATE_PROFILE_STRING (DNS_SECTION, keyname, buf, cfg_file);
			break;
			
		case S_SLAVE_ZONE:
			sprintf (keyname, "SlaveZone%d", index - 1);
			sprintf (keyname1, "SlaveZone%d", index);
			GET_PRIVATE_PROFILE_STRING (DNS_SECTION, keyname1, "", buf, sizeof (buf), cfg_file);
			SET_PRIVATE_PROFILE_STRING (DNS_SECTION, keyname, buf, cfg_file);
			break;
		}
	}
	Conf_Remove_Field (cfg_file, DNS_SECTION, keyname1);
}
/////
int check_value_exist (char * keyname)
{
	return check_private_value_exist (keyname, DEFAULT_CFG_FILE);
	
}
int check_private_value_exist (char * keyname, char * cfg_file)
{
	char buf[256];

	GET_PRIVATE_PROFILE_STRING (DNS_SECTION, keyname, "???", buf, sizeof (buf), cfg_file);
	if (!strcmp (buf, "???"))
		return 0;
	return 1;
	
}
//==================================================================
//==================================================================
//==================================================================
//==================================================================
int get_forwarders_number (void)
{
	return get_private_forwarders_number (DEFAULT_CFG_FILE);
}
int get_private_forwarders_number (char * cfg_file)
{
	char buf[80];
	char keyname[16];
	int index = 1;
	
	memset (buf, '\0', sizeof (buf));
	
	while (TRUE)
	{
		memset (keyname, '\0', sizeof (keyname));
		sprintf (keyname, "Forwarders%d", index);
		GET_PRIVATE_PROFILE_STRING (DNS_SECTION, keyname, "???", buf, sizeof (buf), cfg_file);
		if (!strcmp (buf, "???") || !strcmp (buf, ""))
			break;
		index++;
	}
	return index - 1;
}
////////

BOOL set_forwarders (char * ip)
{
	return set_private_forwarders (ip, DEFAULT_CFG_FILE);
}
BOOL set_private_forwarders (char * ip, char * cfg_file)
{
	char keyname[16];
	int index = get_private_forwarders_number (cfg_file);

	while (index)
	{
		if (!strcmp (get_private_forwarders (index--, cfg_file), ip))
			return FALSE;
	}
	index = get_private_forwarders_number (cfg_file);

	bzero (keyname, sizeof (keyname));
	sprintf (keyname, "Forwarders%d", index + 1);

	return SET_PRIVATE_PROFILE_STRING (DNS_SECTION, keyname, ip, cfg_file);
}
///////////

int remove_forwarders (int index)
{
	return remove_private_forwarders (index, DEFAULT_CFG_FILE);
}
int remove_private_forwarders (int index, char * cfg_file)
{
	char keyname[32];
	char buf[128];
	int result = -1;  //  the key name do not exist
	int total = get_private_forwarders_number (cfg_file);

	sprintf (keyname, "Forwarders%d", index);
	
	GET_PRIVATE_PROFILE_STRING (DNS_SECTION, keyname, "???", buf, sizeof (buf), cfg_file);
	Conf_Remove_Field (cfg_file, DNS_SECTION, keyname);
	
	if (strcmp (buf, "???")  && strcmp (buf, ""))
	{
		permutation_private_forwarders (index, total, cfg_file);
		result = 0;
	}

	return result;
}
//////////

//  index start from 1
char * get_forwarders (int index)
{
	return get_private_forwarders (index, DEFAULT_CFG_FILE);
}
char * get_private_forwarders (int index, char * cfg_file)
{
	static char buf[32];
	char keyname[16];
	
	memset (buf, '\0', sizeof (buf));
	memset (keyname, '\0', sizeof (keyname));

	sprintf (keyname, "Forwarders%d", index);
	
	GET_PRIVATE_PROFILE_STRING (DNS_SECTION, keyname, "???", buf, sizeof (buf), cfg_file);
	if (!strcmp (buf, "???") || !strcmp (buf, ""))
		return NULL;
	return buf;
}
////////

int get_forwarders_index (char * forwarders)
{
	return get_private_forwarders_index (forwarders, DEFAULT_CFG_FILE);
}
int get_private_forwarders_index (char * forwarders, char * cfg_file)
{
	int index = 0, total = get_private_forwarders_number (cfg_file);

	while (index++ < total)
	{
		if ( !strcmp (get_private_forwarders (index, cfg_file), forwarders) )
			return index;
	}
	return 0;
}
//////////
void permutation_forwarders (int index, int total)
{
	permutation_private_forwarders (index, total, DEFAULT_CFG_FILE);
}

void permutation_private_forwarders (int index, int total, char * cfg_file)
{
	char buf[64];
	char keyname[32], keyname1[32];
	while (index++ < total)
	{
		sprintf (keyname, "Forwarders%d", index - 1);
		sprintf (keyname1, "Forwarders%d", index);
		GET_PRIVATE_PROFILE_STRING (DNS_SECTION, keyname1, "", buf, sizeof (buf), cfg_file);
		SET_PRIVATE_PROFILE_STRING (DNS_SECTION, keyname, buf, cfg_file);
	}
	Conf_Remove_Field (cfg_file, DNS_SECTION, keyname1);
}
