/**************************************************************************
**	Copyright (c) 2000  ICP Electronics Inc.  All Rights Reserved.
**
**	FILE:
**		backup.c
**
**	ABSTRACT: 
**
**	FUNCTIONS:
**
**	COMMENTS:
**
**	HISTORY:
**		2002/06/03	Created by Ellis
**
**************************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <fcntl.h>
#include <unistd.h>
#include <utime.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "backup.h"
#include "Util.h"

/*
** macros
*/
#ifndef NDEBUG
	#define TRACE	( !(_Debug) )?	(void)(0) : (void)printf
#else
	#define TRACE	(TRUE)?		(void)(0) : (void)printf
#endif

#ifndef __defined__bool_t
#define __defined__bool_t
typedef int	bool_t;
#endif

#ifndef BLK_SIZE
#define BLK_SIZE	(8 * 512)
#endif

#ifndef B2S
#define B2S(b)		( (b) ? "TRUE" : "FALSE" )
#endif

#ifndef N2S
#define N2S(str)	( (str)? (str) : "NULL" )
#endif

#ifndef TRUE
#define TRUE		(1)
#endif

#ifndef FALSE
#define FALSE		(0)
#endif

#ifndef NULL
#define NULL		(0)
#endif

/*
** local function declarations
*/
static int	backup_copy_purely	(const char *src, const char *dest, int option, char *name);
static int	backup_compress		(const char *src, const char *dest, int option, int pid);

static bool_t	create_dir		(const char *dirname);
static int	copy_file		(const char *src, const char *dest, bool_t checktime);
//static int	copy_slink		(const char *src, const char *dest);
//static int	copy_dev		(const char *src, const char *dest);
static int	sync_dir		(const char *src, const char *dest);

static int	create_empty_archive	(const char *archive);
static int	append_to_archive	(const char *filename, const char *archive);
static int	zip_archive		(const char *archive, const char *zip);
static int	tar_with_sync		(const char *src, const char *dest, const char *archive, 
					 bool_t checktime, bool_t sync);

static char*	make_archive		(const char *dirname, int pid);
static char*	make_zip		(const char *dirname, int pid);
static int	make_pathname		(char **pathname, const char *dirname, const char *filename);

static int	remove_recursively	(const char *pathname);

static char*	get_cwd			();
static char*	make_absolute		(const char *absolute_cwd, const char *pathname);

static char*	escape_filename		(const char *filename);

static bool_t	does_file_exist		(const char *filename);
static bool_t	does_dir_exist		(const char *dirname);
static bool_t	is_same_file		(const char *file1, const char *file2);

/*
** global variable definitions
*/
#ifndef NDEBUG
int	_Debug	= 0;
#endif

/*
** local function definitions
*/

int translate_str(char *str)
{
	char	*ptr1, *ptr2, tmp[2048];

	if (Is_Mb_String(str)==TRUE)
	{
		return 1;
	}
	ptr1=str;
	ptr2=tmp;
	while (*ptr1!=0x0)
	{
		if (*ptr1=='\\')
			*ptr2++='\\';
		*ptr2++=*ptr1++;
	}
	*ptr2=0x0;
	strcpy(str, tmp);
	return 1;
}

char *get_mapping_file(char *file)
{
	char	*ptr;
	int	cnt=0;

	ptr=file;
	while (*ptr!=0x0)
	{
		if (*ptr=='/')
			cnt++;
		if (cnt==3)
			return ptr;
		ptr++;
	}
	return file;
}


/*
** global function definitions
*/
/** About the function's spec, please reference the document: "Backup_Copy Spec.doc". */
int Backup_Copy(const char *src, const char *dest, int option, int pid, char *name)
{
	assert(src);
	assert(dest);
	
	TRACE("Backup_Copy(\"%s\", \"%s\", %d, %d);\n", src, dest, option, pid);
		
	if ( FALSE == does_dir_exist(src)  ) return E_SRC;
	if ( FALSE == does_dir_exist(dest) ) return E_DEST;

	if (option & BACKUP_COPY_COMPRESS) {
		char *absolute_dest	= NULL;
		char *old_wd  		= NULL;
		int  ret_val 		= SUCCESS;
		
		old_wd = get_cwd();
		if (NULL == old_wd) return E_MALLOC;
		
		absolute_dest = make_absolute(old_wd, dest);
		if (NULL == absolute_dest) { free(old_wd); return E_MALLOC; }

		TRACE("chdir(\"%s\")\n", src);
		if ( 0 > chdir(src) ) {
			ret_val = E_SRC;
		} else {
			ret_val = backup_compress("./", absolute_dest, option, pid);
			
			TRACE("chdir(\"%s\")\n", old_wd);
			if ( 0 > chdir(old_wd) )
				if (SUCCESS == ret_val) ret_val = E_SRC;
		}
				
		free(absolute_dest);
		free(old_wd);				
		
		return ret_val;
	} else {
		translate_str((char *)src);
		translate_str((char *)dest);
		return backup_copy_purely(src, dest, option, name);
	}
}

/** About the function's spec, please reference the document: "Backup_Sync Spec.doc". */
int Backup_Sync(const char *src, const char *dest, int bcompress, int pid, char *name)
{
	assert(src);
	assert(dest);
		
	TRACE( "Backup_Sync(\"%s\", \"%s\", %s, %d);\n", src, dest, B2S(bcompress), pid );

	if (bcompress) 
		return Backup_Copy(src, dest, 
			BACKUP_COPY_SYNC | BACKUP_COPY_TIMESTAMP | BACKUP_COPY_COMPRESS, pid, name);
	else
		return Backup_Copy(src, dest, 
			BACKUP_COPY_SYNC | BACKUP_COPY_TIMESTAMP, pid, name);
}

/*
**	Only perform the backup copy from src to dest
**	RETURN:	SUCCESS
**		E_SRC
**	    	E_DEST
**	    	E_COPY
**		E_REMOVE
**		E_MALLOC
**	NOTE: 
**		About the meaning of return values, please reference the document: "Backup_Copy Spec.doc".
*/
static int backup_copy_purely(const char *src, const char *dest, int option, char *name)
{
	char*		src_fn	= NULL;
	char*		dest_fn	= NULL;
	int		ret_val	= SUCCESS;
	
	DIR 		*src_dir;
	struct dirent	*dep;
	struct stat	src_status;

	char		cmd[1024], *ptr;

	assert(src);
	assert(dest);
//	assert( does_dir_exist(src)  );
//	assert( does_dir_exist(dest) );
	
	TRACE("backup_copy_purely(\"%s\", \"%s\", %d);\n", src, dest, option);		
	
	/** browse src tree */
	src_dir = opendir(src);
	if (NULL == src_dir)	return E_SRC;

	while ( NULL != ( dep = readdir(src_dir) ) ) {
		/** skip dot and dot-dot */
		if ( 0 == strcmp(".",  dep->d_name) )	continue;
		if ( 0 == strcmp("..", dep->d_name) )	continue;
				
		/** make the new src file/dir name */
		ret_val = make_pathname(&src_fn, src,  dep->d_name);
		if (SUCCESS != ret_val)
		{
			sprintf(cmd, "/sbin/write_log \"Remote backup schedule %s can't make local site directory %s/%s\" 2", name, src, dep->d_name);
			system(cmd);
//			break;
		}
		
		/** make the new dest file/dir name */
		ret_val = make_pathname(&dest_fn, dest, dep->d_name);
		if (SUCCESS != ret_val)
		{
			sprintf(cmd, "/sbin/write_log \"Remote backup schedule %s can't make remote site directory %s/%s\" 2", name, dest, dep->d_name);
			system(cmd);
//			break;
		}
		
		/** get src file's status */
		if ( 0 > lstat(src_fn, &src_status) )
		{
			ptr=get_mapping_file(src_fn);
			sprintf(cmd, "/sbin/write_log \"Remote backup schedule %s can't get the status of file %s\" 2", name, ptr);
			system(cmd);
			ret_val = E_SRC;
//			break;
		}

		if ( S_ISREG(src_status.st_mode) ) {
			ret_val = copy_file(src_fn, dest_fn, option & BACKUP_COPY_TIMESTAMP);
			if (SUCCESS != ret_val)
			{
				if (strstr(src_fn, ".AppleDouble")==NULL && strstr(src_fn, ".AppleDesktop")==NULL &&
					strstr(src_fn, ".AppleDB")==NULL)
				{
					ptr=get_mapping_file(src_fn);
					if (strlen(ptr)>=1024)
						sprintf(cmd, "/sbin/write_log \"Remote backup schedule %s can't copy file (path is too long) %s\" 2", name, ptr);
					else
						sprintf(cmd, "/sbin/write_log \"Remote backup schedule %s can't copy file (using special character) %s\" 2", name, ptr);
					system(cmd);
				}
//				break;
			}
			
		} else if ( S_ISDIR(src_status.st_mode) ) {			 
			if ( FALSE == create_dir(dest_fn) )
			{
				ptr=get_mapping_file(dest_fn);
				sprintf(cmd, "/sbin/write_log \"Remote backup schedule %s can't create remote site directory %s\" 2", name, ptr);
				system(cmd);
//				ret_val = E_DEST;
//				break;
			}
			
			translate_str(src_fn);
			translate_str(dest_fn);
			ret_val = backup_copy_purely(src_fn, dest_fn, option, name);
			if (SUCCESS != ret_val)
			{
				if (strstr(src_fn, ".AppleDouble")==NULL && strstr(src_fn, ".AppleDesktop")==NULL &&
					strstr(src_fn, ".AppleDB")==NULL)
				{
					char	*esc_ptr = NULL;
					ptr=get_mapping_file(src_fn);

					esc_ptr = escape_filename(ptr);
					if ( !esc_ptr ) 
						sprintf(cmd, "/sbin/write_log \"Remote backup schedule %s copy file fail (using special character)\" 2", name);
					else {
						if (strlen(ptr)>=1024)
							sprintf(cmd, "/sbin/write_log \"Remote backup schedule %s cannot copy share folder (path is too long) %s\" 2", name, src_fn);
						else
							sprintf(cmd, "/sbin/write_log \"Remote backup schedule %s cannot copy share folder (using special character) %s\" 2", name, esc_ptr);
						free(esc_ptr);
					}
					system(cmd);
				}
//				break;
			}
			
		} else if ( S_ISLNK(src_status.st_mode) ) {
/*			ret_val = copy_slink(src_fn, dest_fn);
			if (SUCCESS != ret_val)
			{
				ptr=get_mapping_file(src_fn);
				sprintf(cmd, "/sbin/write_log \"Remote backup schedule %s can't copy slink %s\" 2", name, ptr);
				system(cmd);
//				break;
			}
*/			
		} else if ( S_ISCHR(src_status.st_mode) || S_ISBLK(src_status.st_mode) ) {
/*			ret_val = copy_dev(src_fn, dest_fn);
			if (SUCCESS != ret_val)
			{
				ptr=get_mapping_file(src_fn);
				sprintf(cmd, "/sbin/write_log \"Remote backup schedule %s can't copy device %s\" 2", name, ptr);
				system(cmd);
//				break;
			}
*/			
		} else {
			if (strstr(src_fn, ".AppleDouble")==NULL && strstr(src_fn, ".AppleDesktop")==NULL &&
				strstr(src_fn, ".AppleDB")==NULL)
			{
				char	*esc_ptr = NULL;
				ptr=get_mapping_file(src_fn);

				esc_ptr = escape_filename(ptr);
				if ( !esc_ptr ) 
					sprintf(cmd, "/sbin/write_log \"Remote backup schedule %s copy file fail (using special character)\" 2", name);
				else {
					if (strlen(ptr)>=1024)
						sprintf(cmd, "/sbin/write_log \"Remote backup schedule %s copy file (path is too long) %s fail\" 2", name, src_fn);
					else
						sprintf(cmd, "/sbin/write_log \"Remote backup schedule %s copy file (using special character) %s fail\" 2", name, esc_ptr);
					free(esc_ptr);
				}
				system(cmd);
			}
//			ret_val = E_SRC;
//			break;
		}
	}
						
	if ( 0 > closedir(src_dir) )
		if (SUCCESS == ret_val)
		{
			sprintf(cmd, "/sbin/write_log \"Remote backup schedule %s can't close local directory %s\" 2", name, src);
			system(cmd);
			ret_val = E_SRC;
		}
		
	if (NULL != src_fn)	free(src_fn);
	if (NULL != dest_fn)	free(dest_fn);

	ret_val=SUCCESS;
	/** sync dest */
	if ( (SUCCESS == ret_val) && (option & BACKUP_COPY_SYNC) )
		ret_val = sync_dir(src, dest);

	return ret_val;
}

/*
**	Only perform the backup compress from src to dest
**	RETURN:	SUCCESS
**		E_SRC
**	    	E_DEST
**	    	E_COMPRESS
**		E_REMOVE
**		E_MALLOC
**	NOTE: 
**		About the meaning of return values, please reference the document: "Backup_Copy Spec.doc".
*/
static int backup_compress(const char *src, const char *dest, int option, int pid)
{
	char *archive	= NULL;
	char *zip	= NULL;
	int  ret_val	= SUCCESS;
	
	assert(src);
	assert(dest);
	assert( does_dir_exist(src)  );
	assert( does_dir_exist(dest) );

	TRACE("backup_compress(\"%s\", \"%s\", %d, %d);\n", src, dest, option, pid);
	
	/** make archive file name */
	archive = make_archive(src, pid);
	if (NULL == archive) return E_MALLOC;			

	TRACE("archive filename: \"%s\"\n", archive);

	/** create an empty archive */
	ret_val = create_empty_archive(archive);
	if (SUCCESS != ret_val) return ret_val;
	
	/** tar files */	
	ret_val = tar_with_sync(src, dest, archive, option & BACKUP_COPY_TIMESTAMP, 
				option & BACKUP_COPY_SYNC);
	if (SUCCESS != ret_val) { remove_recursively(archive); free(archive); return ret_val; }
	
	/** make archive file name */
	zip = make_zip(dest, pid);
	if (NULL == zip) { remove_recursively(archive); free(archive); return E_MALLOC; }

	TRACE("zip filename: \"%s\"\n", zip);
	
	/** zip archive file */
	ret_val = zip_archive(archive, zip);
	if (SUCCESS != ret_val)	{
		remove_recursively(zip);
		remove_recursively(archive); 
		free(archive); 
		free(zip); 
		return ret_val;
	}

	/** remove the tar file */	
	ret_val = remove_recursively(archive);
	
	free(archive);
	free(zip);
	
	return ret_val;
}

/*
**	If the specified dir doesn't exist, then create it.
**	RETURN:	TRUE	failed to create the dir.
**	    	FALSE	succeed to create the dir.
*/
static bool_t create_dir(const char *dirname)
{
	assert(dirname);
		
	if ( does_dir_exist(dirname) )	return TRUE;
	
	if ( does_file_exist(dirname) )	remove_recursively(dirname);

	TRACE("mkdir(\"%s\");\n", dirname);
	return ( 0 == mkdir(dirname, 0777) );
	return FALSE;
}

/*
**	Copy normal file from src to dest with the time checking.
**		It will keep the time attibute of the dest file.
**	RETURN:	SUCCESS
**		E_SRC
**	    	E_DEST
**		E_COPY
**		E_REMOVE
**	NOTE: 
**		About the meaning of return values, please reference the document: "Backup_Copy Spec.doc".
*/
static int copy_file(const char *src, const char *dest, bool_t checktime)
{
	struct stat	src_status, dest_status;
	struct utimbuf	old_utimbuf;
	int		src_fd, dest_fd;
	int 		count;
	char		buf[BLK_SIZE];

	assert(src);
	assert(dest);
	assert( does_file_exist(src) );
	
	TRACE( "copy_file(\"%s\", \"%s\", %s);\n", src, dest, B2S(checktime) );		

	/** get src old status included: old atime, mtime */
//	if ( 0 > lstat(src, &src_status) ) return E_SRC;
	if (0>lstat(src, &src_status))
		goto ignore_time;

	if ( does_file_exist(dest) ) {
//		int ret_val;
		
		/** check time */
		if (checktime) {
			int src_mode, dest_mode;
			
//			if ( 0 > lstat(dest, &dest_status) ) return E_DEST;

			if (lstat(dest, &dest_status)>=0)
			{			
				src_mode  = src_status.st_mode  & S_IFMT;
				dest_mode = dest_status.st_mode & S_IFMT;
			
				if ( (dest_mode		   == src_mode		 ) &&
				     (dest_status.st_mtime == src_status.st_mtime) &&
				     (dest_status.st_size  == src_status.st_size ) ) {
					TRACE("*WARNING* skip dest: \"%s\"\n", dest);
					return SUCCESS;
				}
			}
		}


		/** remove dest */	
//		ret_val = remove_recursively(dest);
//		if (SUCCESS != ret_val)	return ret_val;	
	}

ignore_time:
	/** copy src to dest */
	src_fd	= open(src, O_RDONLY);
	if (0 > src_fd)
		return E_SRC;


	dest_fd = open(dest, O_WRONLY | O_CREAT | O_TRUNC, src_status.st_mode);
	if (0 > dest_fd) {
		close(src_fd);
		return E_COPY;
	}

	while ( 0 < ( count = read(src_fd, buf, BLK_SIZE) ) ) {
		if ( count != write(dest_fd, buf, count) ) {
			close(src_fd);
			close(dest_fd);
			return E_COPY;
		}
	}
	
	if (0 != count) {
		close(src_fd);
		close(dest_fd);
		return E_SRC;
	}

	if ( 0 > close(src_fd) ) {
		close(dest_fd);
		return E_SRC;
	}
	
	if ( 0 > close(dest_fd) )
		return E_COPY;

	/** set src and dest time as old time*/
	old_utimbuf.actime	= src_status.st_atime;
	old_utimbuf.modtime	= src_status.st_mtime;

// Catherine 2003/05/07 -- in case the source filesystem is read-only (ex. snapshot)
//	if ( 0 > utime(src,  &old_utimbuf) )
//		return E_SRC;
	
// Kent 2003/07/28
// in some case, utime can't success
// for example: user not exist
// ignore this error
	if ( 0 > utime(dest, &old_utimbuf) )
	{
//		return E_COPY;
	}
	
	return SUCCESS;
}

/*
**	Copy symbolic link from src to dest
**	RETURN:	SUCCESS
**		E_SRC
**	    	E_DEST
**		E_COPY
**		E_REMOVE
**	NOTE: 
**		About the meaning of return values, please reference the document: "Backup_Copy Spec.doc".
*/
/* Commented by Charles, begin */
#if 0
static int copy_slink(const char *src, const char *dest)
{
	int 	count;
	char	buf[PATH_MAX + 1];
	int	ret_val = SUCCESS;

	assert(src);
	assert(dest);
	
	TRACE("copy_slink(\"%s\", \"%s\");\n", src, dest);

	/** remove the dest */
	if ( does_file_exist(dest) ) {
		ret_val = remove_recursively(dest);
		if (SUCCESS != ret_val)	return ret_val;
	}
	
	/** copy src to dest */
	count = readlink(src, buf, PATH_MAX);
	if (0 > count) return E_SRC;
	
	buf[count] = '\0';
	
	if ( 0 > symlink(buf, dest) ) return E_COPY;
	
	return SUCCESS;
}

/*
**	Copy device file from src to dest
**	RETURN:	SUCCESS
**		E_SRC
**	    	E_DEST
**		E_COPY
**		E_REMOVE
**	NOTE: 
**		About the meaning of return values, please reference the document: "Backup_Copy Spec.doc".
*/
static int copy_dev(const char *src, const char *dest)
{
	char *escaped_src, *escaped_dest;
	char *cmd;
	int  ret_val = SUCCESS;
	
	assert(src);
	assert(dest);
	
	TRACE("copy_dev(\"%s\", \"%s\");\n", src, dest);

	/** remove the dest */
	if ( does_file_exist(dest) ) {
		ret_val = remove_recursively(dest);
		if (SUCCESS != ret_val)	return ret_val;
	}
			
	/** make cmd string */
	escaped_src	= escape_filename(src);
	if (NULL == escaped_src)	return E_MALLOC;
	escaped_dest	= escape_filename(dest);
	if (NULL == escaped_dest)	{ free(escaped_src); return E_MALLOC; }
	cmd = (char *)malloc( strlen(escaped_src) + strlen(escaped_dest) + 50 );
	if (NULL == cmd)		{ free(escaped_dest); free(escaped_src); return E_MALLOC; }
	
	sprintf(cmd, "cp -a \"%s\" \"%s\"", escaped_src, escaped_dest);
		
	TRACE("cp cmd: %s\n", cmd);
	
	/** perform rm */
	if ( 0 != system(cmd) )	ret_val = E_COPY;

	free(cmd);
	free(escaped_dest);
	free(escaped_src);

	return ret_val;	
}
#endif
/* commented by Charles, end */

/*
**	Synchronize dest based on src non-recursively.		
**		It will remove files and dirs which only exist int src.
**	RETURN:	SUCCESS
**		E_SRC
**	    	E_DEST
**		E_REMOVE
**	NOTE: 
**		About the meaning of return values, please reference the document: "Backup_Copy Spec.doc".
*/
static int sync_dir(const char *src, const char *dest)
{
	char*		src_fn	= NULL;
	char*		dest_fn	= NULL;
	int		ret_val	= SUCCESS;
#ifdef SYNC_BY_NFS
	char		*dest_nfs = NULL;
#endif
	DIR		*dest_dir;
	struct dirent	*dep;
	
	assert(src);
	assert(dest);
	assert( does_dir_exist(src)  );
	assert( does_dir_exist(dest) );

	TRACE("sync_dir(\"%s\", \"%s\");\n", src, dest);
		
#ifdef SYNC_BY_NFS
	// Andy Wu
	{
		int	idx=5;	// dest=/mnt/1234/....
		int	size=strlen(dest)+5;

		while ( dest[idx] != '/' ) idx++;
		dest_nfs = (char *)malloc(sizeof(char)*size);
		if ( !dest_nfs )
			dest_nfs = (char *)dest;
		else {
			bzero(dest_nfs, size);
			strncpy(dest_nfs, dest, idx);
			strcat(dest_nfs, ".nfs");
			strcat(dest_nfs, &dest[idx]);
		}
	}
	dest_dir = opendir(dest_nfs);
	// Andy Wu
#else
	dest_dir = opendir(dest);
#endif
	if (NULL == dest_dir)	{
		
		return E_DEST;
	}
	
	while ( NULL != ( dep = readdir(dest_dir) ) ) {
		/** skip dot and dot-dot */
		if ( 0 == strcmp(".",  dep->d_name) )	continue;
		if ( 0 == strcmp("..", dep->d_name) )	continue;
				
		/** make the new src file/dir name */
		ret_val = make_pathname(&src_fn, src,  dep->d_name);
		if (SUCCESS != ret_val) break;
		
		/** make the new dest file/dir name */
#ifdef SYNC_BY_NFS
		ret_val = make_pathname(&dest_fn, dest_nfs, dep->d_name);
#else
		ret_val = make_pathname(&dest_fn, dest, dep->d_name);
#endif
		if (SUCCESS != ret_val) break;
		
		/** if the same name file doesn't exist in src, then rm it in dest */
		if ( FALSE == does_file_exist(src_fn) ) {		
			ret_val = remove_recursively(dest_fn);
			if (SUCCESS != ret_val)	break;
		}
	}
				
#ifdef SYNC_BY_NFS
	// Andy Wu
	{
		if ( dest_nfs )
			free(dest_nfs);
		dest_nfs = NULL;
	}
	// Andy Wu
#endif
	if ( 0 > closedir(dest_dir) ) {
		if (SUCCESS == ret_val) ret_val = E_DEST;
	}
		
	if (NULL != src_fn)	free(src_fn);
	if (NULL != dest_fn)	free(dest_fn);
	
	return ret_val;
}

/*
**	Create a empty archive file
**	RETURN:	SUCCESS
**	    	E_MALLOC
**		E_COMPRESS
**	NOTE: 
**		About the meaning of return values, please reference the document: "Backup_Copy Spec.doc".
*/
static int create_empty_archive(const char *archive)
{
	char *escaped_archive;
	char *cmd;
	int  ret_val = SUCCESS;
	
	assert(archive);
	
	TRACE("create_empty_archive(\"%s\");\n", archive);

	/** make cmd string */
	escaped_archive = escape_filename(archive);
	if (NULL == escaped_archive)	return E_MALLOC;
	cmd = (char *)malloc( strlen(escaped_archive) + 50 );
	if (NULL == cmd)		{ free(escaped_archive); return E_MALLOC; }
	
	sprintf(cmd, "tar rf \"%s\" --atime-preserve", escaped_archive);
		
	TRACE("tar cmd: %s\n", cmd);
	
	/** perform tar */
	if ( 0 != system(cmd) )	ret_val = E_COMPRESS;

	free(cmd);
	free(escaped_archive);

	return ret_val;
}

/*
**	Append file/dir the to archive file
**	RETURN:	SUCCESS
**	    	E_MALLOC
**		E_COMPRESS
**	NOTE: 
**		About the meaning of return values, please reference the document: "Backup_Copy Spec.doc".
*/
static int append_to_archive(const char *filename, const char *archive)
{
	char *escaped_filename, *escaped_archive;
	char *cmd;
	int  ret_val = SUCCESS;
	
	assert(filename);
	assert(archive);
	
	TRACE("append_to_archive(\"%s\", \"%s\");\n", filename, archive);

	/** skip archive himself */
	if ( is_same_file(filename, archive) ) return SUCCESS;
	
	/** make cmd string */
	escaped_filename	= escape_filename(filename);
	if (NULL == escaped_filename)	return E_MALLOC;
	escaped_archive		= escape_filename(archive);
	if (NULL == escaped_archive)	{ free(escaped_filename); return E_MALLOC; }
	cmd = (char *)malloc( strlen(escaped_filename) + strlen(escaped_archive) + 50 );
	if (NULL == cmd)		{ free(escaped_archive); free(escaped_filename); return E_MALLOC; }
	
	sprintf(cmd, "tar rf \"%s\" \"%s\" --atime-preserve", escaped_archive, escaped_filename);
		
	TRACE("tar cmd: %s\n", cmd);
	
	/** perform tar */
	if ( 0 != system(cmd) )	ret_val = E_COMPRESS;

	free(cmd);
	free(escaped_archive);
	free(escaped_filename);

	return ret_val;
}

/*
**	Zip the archive file
**	RETURN:	SUCCESS
**	    	E_MALLOC
**		E_COMPRESS if zip failed
**	NOTE: 
**		About the meaning of return values, please reference the document: "Backup_Copy Spec.doc".
*/
static int zip_archive(const char *archive, const char *zip)
{
	char *escaped_archive, *escaped_zip;
	char *cmd;
	int  ret_val = SUCCESS;
	
	assert(archive);
	assert(zip);
	
	TRACE("zip_archive(\"%s\", \"%s\");\n", archive, zip);

	/** make cmd string */
	escaped_archive	= escape_filename(archive);
	if (NULL == escaped_archive)	return E_MALLOC;
	escaped_zip	= escape_filename(zip);
	if (NULL == escaped_zip)	{ free(escaped_archive); return E_MALLOC; }
	cmd = (char *)malloc( strlen(escaped_archive) + strlen(escaped_zip) + 50 );
	if (NULL == cmd)		{ free(escaped_zip); free(escaped_archive); return E_MALLOC; }
	
	sprintf(cmd, "gzip -c \"%s\" > \"%s\"", escaped_archive, escaped_zip);
		
	TRACE("gzip cmd: %s\n", cmd);
	
	/** perform zip */
	if ( 0 != system(cmd) )	ret_val = E_COMPRESS;

	free(cmd);
	free(escaped_zip);
	free(escaped_archive);

	return ret_val;
}

/*
**	Tar src file to the archive file with sync recursively.
**		It will check time between src and dest.
**	RETURN:	SUCCESS
**	    	E_SRC
**		E_DEST
**		E_COMPRESS
**		E_REMOVE
**	    	E_MALLOC
**	NOTE: 
**		About the meaning of return values, please reference the document: "Backup_Copy Spec.doc".
*/
static int tar_with_sync(const char *src, const char *dest, const char *archive, 
			 bool_t checktime, bool_t sync)
{
	char*		src_fn	= NULL;
	char*		dest_fn	= NULL;
	int		ret_val	= SUCCESS;
	
	DIR 		*src_dir;
	struct dirent	*dep;
	struct stat	src_status, dest_status;
	
	assert(src);
	assert(dest);
	assert(archive);
	assert( does_dir_exist(src)  );
	assert( does_dir_exist(dest) );
	
	TRACE( "tar_with_sync(\"%s\", \"%s\", \"%s\", %s, %s);\n", 
		src, dest, archive, B2S(checktime), B2S(sync) );
		
	/** cmp src and dest */
	src_dir = opendir(src);
	if (NULL == src_dir)	return E_SRC;

	while ( NULL != ( dep = readdir(src_dir) ) ) {
		/** skip dot and dot-dot */
		if ( 0 == strcmp(".",  dep->d_name) )	continue;
		if ( 0 == strcmp("..", dep->d_name) )	continue;
				
		/** make the new src file/dir name */
		ret_val = make_pathname(&src_fn, src,  dep->d_name);
		if (SUCCESS != ret_val) break;
		
		/** make the new dest file/dir name */
		ret_val = make_pathname(&dest_fn, dest, dep->d_name);
		if (SUCCESS != ret_val) break;

		/** get src file's status */
		if ( 0 > lstat(src_fn, &src_status) ) { ret_val = E_SRC; break; }
		
		if ( S_ISREG(src_status.st_mode) ) {
			/** check time */
			if ( checktime && does_file_exist(dest_fn) ) {			
				int src_mode, dest_mode;
				
				if ( 0 > lstat(dest, &dest_status) ) return E_DEST;
				
				src_mode  = src_status.st_mode  & S_IFMT;
				dest_mode = dest_status.st_mode & S_IFMT;
				
				if ( (dest_mode		   == src_mode		 ) &&
				     (dest_status.st_mtime == src_status.st_mtime) &&
				     (dest_status.st_size  == src_status.st_size ) ) {
					continue;
				}
			}						
		} else if ( S_ISDIR(src_status.st_mode) ) {
			if ( does_dir_exist(dest_fn) ) {
				ret_val = tar_with_sync(src_fn, dest_fn, archive, checktime, sync);
				if (SUCCESS != ret_val) break;
				
				if (sync) {
					/** if dest_fn become a empty dest, then remove it */
					TRACE("*WARNING* if dir is empty, rm it: %s\n", dest_fn);
					rmdir(dest_fn);
				}
				
				continue;
			}
		} else if ( S_ISLNK(src_status.st_mode) ||
			    S_ISCHR(src_status.st_mode) || 
			    S_ISBLK(src_status.st_mode) ) {
			/** do nothing */
		} else {
			ret_val = E_SRC;
			break;
		}

		/** append src file to archive */
		ret_val = append_to_archive(src_fn, archive);
// mark by Kent 2002/12/24, to avoid illegal filename
//		if (SUCCESS != ret_val)	break;
		
		if ( sync && does_file_exist(dest_fn) ) {
			ret_val = remove_recursively(dest_fn);
			if (SUCCESS != ret_val)	break;
		}
	}
						
	if ( 0 > closedir(src_dir) )
		if (SUCCESS == ret_val) ret_val = E_SRC;
		
	if (NULL != src_fn)	free(src_fn);
	if (NULL != dest_fn)	free(dest_fn);

	/** sync dest */
	if ( (SUCCESS == ret_val) && (sync) )
		ret_val = sync_dir(src, dest);

	return ret_val;
}

/*
**	Alloc a string and make the archive file name
**	RETURN:	the archive file name
**	    	NULL means lack of memory
*/
static char* make_archive(const char *dirname, int pid)
{
	char prefix[50];
	
	assert(dirname);
	
	sprintf( prefix, "%d", pid );
	
	return tempnam(dirname, prefix);
}

/*
**	Alloc a string and make the zip file name
**	RETURN:	the zip file name
**	    	NULL means lack of memory
*/
static char* make_zip(const char *dirname, int pid)
{
	char* zip;
	
	assert(dirname);
	
	zip = (char *)malloc( strlen(dirname) + 50 );
	if (NULL == zip) return NULL;

	strcpy(zip, dirname);
	if ( '/' != zip[strlen(dirname) - 1] ) strcat(zip, "/");
	sprintf( zip + strlen(zip), "~~%d.tar.gz", pid );
	
	return zip;
}

/*
**	Realloc a string and make the pathname by concatenate the dirname and filename.
**	RETURN:	SUCCESS
**	    	E_MALLOC
*/
static int make_pathname(char **pathname, const char *dirname, const char *filename)
{
	char* tmp;
	
	assert(pathname);
	assert(dirname);
	assert(filename);

	tmp = (char*)realloc( *pathname, strlen(dirname) + strlen(filename) + 512 );
	if (NULL == tmp) return E_MALLOC;

	strcpy(tmp, dirname);

	if ( '/' != tmp[strlen(dirname) - 1] )	strcat(tmp, "/");

	strcat(tmp, filename);
	
	*pathname = tmp;

	return SUCCESS;
}

/*
**	Remove the file/dir recursively.
**	RETURN:	SUCCESS
**	    	E_REMOVE
*/
static int remove_recursively(const char *pathname)
{
	char *escaped_pathname;
	char *cmd;
	int  ret_val = SUCCESS;
	
	assert(pathname);
	
	TRACE("*WARNING* remove_recursively(\"%s\");\n", pathname);

	/** make cmd string */
	escaped_pathname = escape_filename(pathname);
	if (NULL == escaped_pathname)	return E_MALLOC;
	cmd = (char *)malloc( strlen(escaped_pathname) + 50 );
	if (NULL == cmd)		{ free(escaped_pathname); return E_MALLOC; }
		
	sprintf(cmd, "/bin/rm -rf \"%s\"", escaped_pathname);
		
	TRACE("rm cmd: %s\n", cmd);
	
	/** perform rm */
	if ( 0 != system(cmd) )	ret_val = E_REMOVE;

	free(cmd);
	free(escaped_pathname);

	return ret_val;
}

/*
**	Alloc a string and get the current working directory.
**	RETURN:	the current working directory
**	    	NULL means lack of memory
*/
static char* get_cwd()
{	
	char* cwd	= NULL;
	int   size	= 0;
	char* tmp;
	
	while (TRUE) {
		size += PATH_MAX + 1;
		tmp = (char*)realloc(cwd, size);
		if (NULL == tmp) {
			if (NULL != cwd) free(cwd); 
			return NULL;
		}
		cwd = tmp;
		
		if ( NULL != getcwd(cwd, size) ) return cwd;
	}
}

/*
**	Alloc a string and make the pathname to an absolute pathname.
**		If the pathname is already an absolute path, then only duplicate it.
**	RETURN:	the absolute pathname
**	    	NULL means lack of memory
*/
static char* make_absolute(const char *absolute_cwd, const char *pathname)
{
	char* tmp = NULL;
	
	assert(absolute_cwd);
	assert(pathname);
	
	if ('/' == pathname[0]) {
		tmp = (char *)malloc( strlen(pathname) + 1 );
		if (NULL == tmp) return NULL;
		
		strcpy(tmp, pathname);			
		return tmp;
	} else {
		assert(NULL == tmp);
		if ( SUCCESS == make_pathname(&tmp, absolute_cwd, pathname) )	return tmp;
		else								return NULL;
	}
}

/*
**	Escape the metacharater in filename.
**	RETURN:	the escaped filename
**	    	NULL means lack of memory
*/
static char* escape_filename(const char *filename)
{
	int	i, j;
	char	*escaped_filename;
	
	assert(filename);
	
	escaped_filename = (char *)malloc( strlen(filename) * 2 + 1 );
	if (NULL == escaped_filename) return NULL;
	
	for (i = 0, j = 0; '\0' != filename[i]; i++, j++) {
		switch (filename[i]) {
		case '$': 
		case '`':
		case '\\':
		/*
		 * A Linux rm bug: rm -rf "\'G" will return 0 but no operation performed!!
		 * To avoid this, not to translate '
		 */
		//case '\'':
		case '\"':
			escaped_filename[j] = '\\';
			j++;
		}
		
		escaped_filename[j] = filename[i];
	}
	
	escaped_filename[j] = '\0';

	return escaped_filename;
}


/*
**	Check if the file(normal file/dir/symbolic link/...) exists. 
**		It won't follow the symbolic link.
**	RETURN:	TRUE means existance
**	    	FALSE means nonexistence
*/
static bool_t does_file_exist(const char *filename)
{
	struct stat	status;
	
	assert(filename);
	
	return ( 0 == lstat(filename, &status) );
}

/*
**	Check if the dir exists. 
**		It won't follow the symbolic link.
**	RETURN:	TRUE means existance
**	    	FALSE means nonexistence
*/
static bool_t does_dir_exist(const char *dirname)
{
	struct stat	status;
	
	assert(dirname);
	
	if ( 0 > lstat(dirname, &status) ) return FALSE;
	
	return S_ISDIR(status.st_mode);
}

/*
**	Check if is the same file
**		It won't follow the symbolic link.
**	RETURN:	TRUE is the same file.
**	    	FALSE is different.
*/
static bool_t is_same_file(const char *file1, const char *file2)
{
	struct stat status1, status2;

	assert(file1);
	assert(file2);
	
	if ( 0 > lstat(file1, &status1) ) return FALSE;
	if ( 0 > lstat(file2, &status2) ) return FALSE;
	
	return (status1.st_ino == status2.st_ino);
}

