/*
   Unix SMB/Netbios implementation.
   Version 1.9.
   NBT netbios routines and daemon - version 2
   Copyright (C) Andrew Tridgell 1994-1998
   Copyright (C) Luke Kenneth Casson Leighton 1994-1998
   Copyright (C) Jeremy Allison 1994-1998

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "includes.h"

extern pstring global_myname;

extern uint16 samba_nb_type; /* Samba's NetBIOS name type. */

/*******************************************************************
 Utility function to add a name to the unicast subnet, or add in
 our IP address if it already exists.
******************************************************************/

void insert_permanent_name_into_unicast( struct subnet_record *subrec,
                                                struct nmb_name *nmbname, uint16 nb_type )
{
  struct name_record *namerec;

  if((namerec = find_name_on_subnet(unicast_subnet, nmbname, FIND_SELF_NAME)) == NULL)
  {
    /* The name needs to be created on the unicast subnet. */
    (void)add_name_to_subnet( unicast_subnet, nmbname->name,
                              nmbname->name_type, nb_type,
                              PERMANENT_TTL, PERMANENT_NAME, 1, &subrec->myip);
  }
  else
  {
    /* The name already exists on the unicast subnet. Add our local
       IP for the given broadcast subnet to the name. */
    add_ip_to_name_record( namerec, subrec->myip);
  }
}

/*******************************************************************
 Utility function to remove a name from the unicast subnet.
******************************************************************/

static void remove_permanent_name_from_unicast( struct subnet_record *subrec,
                                                struct nmb_name *nmbname )
{
  struct name_record *namerec;

  if((namerec = find_name_on_subnet(unicast_subnet, nmbname, FIND_SELF_NAME)) != NULL)
  {
    /* Remove this broadcast subnet IP address from the name. */
    remove_ip_from_name_record( namerec, subrec->myip);
    if(namerec->data.num_ips == 0)
      remove_name_from_namelist( unicast_subnet, namerec);
  }
}

/*******************************************************************
 Utility function always called to set our workgroup and server
 state back to potential browser, or none.
******************************************************************/

static void reset_workgroup_state( struct subnet_record *subrec, char *workgroup_name,
                                   BOOL force_new_election )
{
  struct work_record *work;
  struct server_record *servrec;
  struct nmb_name nmbname;

  if((work = find_workgroup_on_subnet( subrec, workgroup_name)) == NULL)
  {
((void)0);\
    return;
  }

  if((servrec = find_server_in_workgroup( work, global_myname)) == NULL)
  {
((void)0);\
    work->mst_state = lp_local_master() ? MST_POTENTIAL : MST_NONE;
    return;
  }

  /* Update our server status - remove any master flag and replace
   it with the potential browser flag. */
  servrec->serv.type &= ~SV_TYPE_MASTER_BROWSER;
  servrec->serv.type |= (lp_local_master() ? SV_TYPE_POTENTIAL_BROWSER : 0);

  /* Tell the namelist writer to write out a change. */
  subrec->work_changed = True;

  /* Reset our election flags. */
  work->ElectionCriterion &= ~0x4;

  work->mst_state = lp_local_master() ? MST_POTENTIAL : MST_NONE;

  /* Forget who the local master browser was for
     this workgroup. */

  set_workgroup_local_master_browser_name( work, "");

  /*
   * Ensure the IP address of this subnet is not registered as one
   * of the IP addresses of the WORKGROUP<1d> name on the unicast
   * subnet. This undoes what we did below when we became a local
   * master browser.
   */

  make_nmb_name(&nmbname, work->work_group, 0x1d);

  remove_permanent_name_from_unicast( subrec, &nmbname);

  if(force_new_election)
    work->needelection = True;
}

/*******************************************************************
  Unbecome the local master browser name release success function.
******************************************************************/

static void unbecome_local_master_success(struct subnet_record *subrec,
                             struct userdata_struct *userdata,
                             struct nmb_name *released_name,
                             struct in_addr released_ip)
{
  BOOL force_new_election = False;

  memcpy((char *)&force_new_election, userdata->data, sizeof(BOOL));

((void)0);

  /* Now reset the workgroup and server state. */
  reset_workgroup_state( subrec, released_name->name, force_new_election );

  if( DEBUGLVL( 0 ) )
  {
((void)0);
((void)0);
((void)0);
((void)0);
((void)0);
  }

}

/*******************************************************************
  Unbecome the local master browser name release fail function.
******************************************************************/

static void unbecome_local_master_fail(struct subnet_record *subrec, struct response_record *rrec,
                       struct nmb_name *fail_name)
{
  struct name_record *namerec;
  struct userdata_struct *userdata = rrec->userdata;
  BOOL force_new_election = False;

  memcpy((char *)&force_new_election, userdata->data, sizeof(BOOL));

((void)0);\

  /* Do it anyway. */
  namerec = find_name_on_subnet(subrec, fail_name, FIND_SELF_NAME);
  if(namerec)
    remove_name_from_namelist(subrec, namerec);

  /* Now reset the workgroup and server state. */
  reset_workgroup_state( subrec, fail_name->name, force_new_election );

  if( DEBUGLVL( 0 ) )
  {
((void)0);
((void)0);
((void)0);
((void)0);
((void)0);
  }
}

/*******************************************************************
 Utility function to remove the WORKGROUP<1d> name.
******************************************************************/

static void release_1d_name( struct subnet_record *subrec, char *workgroup_name,
                             BOOL force_new_election)
{
  struct nmb_name nmbname;
  struct name_record *namerec;

  make_nmb_name(&nmbname, workgroup_name, 0x1d);
  if((namerec = find_name_on_subnet( subrec, &nmbname, FIND_SELF_NAME))!=NULL)
  {
    struct userdata_struct *userdata;
    int size = sizeof(struct userdata_struct) + sizeof(BOOL);

    if((userdata = (struct userdata_struct *)malloc(size)) == NULL)
    {
((void)0);
      return;
    }

    userdata->copy_fn = NULL;
    userdata->free_fn = NULL;
    userdata->userdata_len = sizeof(BOOL);
    memcpy((char *)userdata->data, &force_new_election, sizeof(BOOL));

    release_name(subrec, namerec,
                 unbecome_local_master_success,
                 unbecome_local_master_fail,
                 userdata);

    zero_free(userdata, size);
  }
}

/*******************************************************************
 Unbecome the local master browser MSBROWSE name release success function.
******************************************************************/

static void release_msbrowse_name_success(struct subnet_record *subrec,
                      struct userdata_struct *userdata,
                      struct nmb_name *released_name,
                      struct in_addr released_ip)
{
((void)0);

  /* Remove the permanent MSBROWSE name added into the unicast subnet. */
  remove_permanent_name_from_unicast( subrec, released_name);
}

/*******************************************************************
 Unbecome the local master browser MSBROWSE name release fail function.
******************************************************************/

static void release_msbrowse_name_fail( struct subnet_record *subrec,
                       struct response_record *rrec,
                       struct nmb_name *fail_name)
{
  struct name_record *namerec;

((void)0);

  /* Release the name anyway. */
  namerec = find_name_on_subnet(subrec, fail_name, FIND_SELF_NAME);
  if(namerec)
    remove_name_from_namelist(subrec, namerec);

  /* Remove the permanent MSBROWSE name added into the unicast subnet. */
  remove_permanent_name_from_unicast( subrec, fail_name);
}

/*******************************************************************
  Unbecome the local master browser. If force_new_election is true, restart
  the election process after we've unbecome the local master.
******************************************************************/

void unbecome_local_master_browser(struct subnet_record *subrec, struct work_record *work,
                                   BOOL force_new_election)
{
  struct name_record *namerec;
  struct nmb_name nmbname;

  /* Sanity check. */

((void)0);\

  if(find_server_in_workgroup( work, global_myname) == NULL)
  {
((void)0);\
    work->mst_state = lp_local_master() ? MST_POTENTIAL : MST_NONE;
    return;
  }

  /* Set the state to unbecoming. */
  work->mst_state = MST_UNBECOMING_MASTER;

  /*
   * Release the WORKGROUP<1d> name asap to allow another machine to
   * claim it.
   */

  release_1d_name( subrec, work->work_group, force_new_election);

  /* Deregister any browser names we may have. */
  make_nmb_name(&nmbname, MSBROWSE, 0x1);
  if((namerec = find_name_on_subnet( subrec, &nmbname, FIND_SELF_NAME))!=NULL)
  {
    release_name(subrec, namerec,
                 release_msbrowse_name_success,
                 release_msbrowse_name_fail,
                 NULL);
  }

  /*
   * Ensure we have sent and processed these release packets
   * before returning - we don't want to process any election
   * packets before dealing with the 1d release.
   */

  retransmit_or_expire_response_records(time(NULL));
}

/****************************************************************************
  Success in registering the WORKGROUP<1d> name.
  We are now *really* a local master browser.
  ****************************************************************************/

static void become_local_master_stage2(struct subnet_record *subrec,
                                        struct userdata_struct *userdata,
                                        struct nmb_name *registered_name,
                                        uint16 nb_flags,
                                        int ttl, struct in_addr registered_ip)
{
  int i = 0;
  struct server_record *sl;
  struct work_record *work = find_workgroup_on_subnet( subrec, registered_name->name);
  struct server_record *servrec;

  if(!work)
  {
((void)0);\
    return;
  }

  if((servrec = find_server_in_workgroup( work, global_myname)) == NULL)
  {
((void)0);\
    work->mst_state = lp_local_master() ? MST_POTENTIAL : MST_NONE;
    return;
  }

((void)0);\

  work->mst_state = MST_BROWSER; /* registering WORKGROUP(1d) succeeded */

  /* update our server status */
  servrec->serv.type |= SV_TYPE_MASTER_BROWSER;
  servrec->serv.type &= ~SV_TYPE_POTENTIAL_BROWSER;

  /* Tell the namelist writer to write out a change. */
  subrec->work_changed = True;

  /* Add this name to the workgroup as local master browser. */
  set_workgroup_local_master_browser_name( work, global_myname);

  /* Count the number of servers we have on our list. If it's
     less than 10 (just a heuristic) request the servers
     to announce themselves.
   */
  for( sl = work->serverlist; sl != NULL; sl = sl->next)
    i++;

  if (i < 10)
  {
    /* Ask all servers on our local net to announce to us. */
    broadcast_announce_request(subrec, work);
  }

  /*
   * Now we are a local master on a broadcast subnet, we need to add
   * the WORKGROUP<1d> name to the unicast subnet so that we can answer
   * unicast requests sent to this name. We can create this name directly on
   * the unicast subnet as a WINS server always returns true when registering
   * this name, and discards the registration. We use the number of IP
   * addresses registered to this name as a reference count, as we
   * remove this broadcast subnet IP address from it when we stop becoming a local
   * master browser for this broadcast subnet.
   */

  insert_permanent_name_into_unicast( subrec, registered_name, nb_flags);

  /* Reset the announce master browser timer so that we try and tell a domain
     master browser as soon as possible that we are a local master browser. */
  reset_announce_timer();

  if( DEBUGLVL( 0 ) )
  {
((void)0);
((void)0);
((void)0);
((void)0);
((void)0);
  }

}

/****************************************************************************
  Failed to register the WORKGROUP<1d> name.
  ****************************************************************************/
static void become_local_master_fail2(struct subnet_record *subrec,
                                      struct response_record *rrec,
                                      struct nmb_name *fail_name)
{
  struct work_record *work = find_workgroup_on_subnet( subrec, fail_name->name);

((void)0);\

  if(!work)
  {
((void)0);\
    return;
  }

  /* Roll back all the way by calling unbecome_local_master_browser(). */
  unbecome_local_master_browser(subrec, work, False);
}

/****************************************************************************
  Success in registering the MSBROWSE name.
  ****************************************************************************/

static void become_local_master_stage1(struct subnet_record *subrec,
                                        struct userdata_struct *userdata,
                                        struct nmb_name *registered_name,
                                        uint16 nb_flags,
                                        int ttl, struct in_addr registered_ip)
{
  char *work_name = userdata->data;
  struct work_record *work = find_workgroup_on_subnet( subrec, work_name);

  if(!work)
  {
((void)0);\
    return;
  }

((void)0);

  work->mst_state = MST_MSB; /* Registering MSBROWSE was successful. */

  /*
   * We registered the MSBROWSE name on a broadcast subnet, now need to add
   * the MSBROWSE name to the unicast subnet so that we can answer
   * unicast requests sent to this name. We create this name directly on
   * the unicast subnet.
   */

  insert_permanent_name_into_unicast( subrec, registered_name, nb_flags);

  /* Attempt to register the WORKGROUP<1d> name. */
  register_name(subrec, work->work_group,0x1d,samba_nb_type,
                become_local_master_stage2,
                become_local_master_fail2,
                NULL);
}

/****************************************************************************
  Failed to register the MSBROWSE name.
  ****************************************************************************/

static void become_local_master_fail1(struct subnet_record *subrec,
                                      struct response_record *rrec,
                                      struct nmb_name *fail_name)
{
  char *work_name = rrec->userdata->data;
  struct work_record *work = find_workgroup_on_subnet(subrec, work_name);

  if(!work)
  {
((void)0);\
    return;
  }

  if(find_server_in_workgroup(work, global_myname) == NULL)
  {
((void)0);\
    return;
  }

  reset_workgroup_state( subrec, work->work_group, False );

((void)0);\
}

/******************************************************************
  Become the local master browser on a subnet.
  This gets called if we win an election on this subnet.

  Stage 1: mst_state was MST_POTENTIAL - go to MST_BACK register ^1^2__MSBROWSE__^2^1.
  Stage 2: mst_state was MST_BACKUP  - go to MST_MSB  and register WORKGROUP<1d>.
  Stage 3: mst_state was MST_MSB  - go to MST_BROWSER.
******************************************************************/

void become_local_master_browser(struct subnet_record *subrec, struct work_record *work)
{
  struct userdata_struct *userdata;
  int size = sizeof(struct userdata_struct) + sizeof(fstring) + 1;

  /* Sanity check. */
  if (!lp_local_master())
  {
((void)0);
    return;
  }

  if(!AM_POTENTIAL_MASTER_BROWSER(work))
  {
((void)0);
    return;
  }

  if(find_server_in_workgroup( work, global_myname) == NULL)
  {
((void)0);\
    return;
  }

((void)0);\

((void)0);
  work->mst_state = MST_BACKUP; /* an election win was successful */

  work->ElectionCriterion |= 0x5;

  /* Tell the namelist writer to write out a change. */
  subrec->work_changed = True;

  /* Setup the userdata_struct. */
  if((userdata = (struct userdata_struct *)malloc(size)) == NULL)
  {
((void)0);
    return;
  }

  userdata->copy_fn = NULL;
  userdata->free_fn = NULL;
  userdata->userdata_len = strlen(work->work_group)+1;
  pstrcpy(userdata->data, work->work_group);

  /* Register the special browser group name. */
  register_name(subrec, MSBROWSE, 0x01, samba_nb_type|NB_GROUP,
                become_local_master_stage1,
                become_local_master_fail1,
                userdata);

  zero_free(userdata, size);
}

/***************************************************************
 Utility function to set the local master browser name. Does
 some sanity checking as old versions of Samba seem to sometimes
 say that the master browser name for a workgroup is the same
 as the workgroup name.
****************************************************************/

void set_workgroup_local_master_browser_name( struct work_record *work, const char *newname)
{
((void)0);\

#if 0
  /*
   * Apparently some sites use the workgroup name as the local
   * master browser name. Arrrrggghhhhh ! (JRA).
   */
  if(strequal( work->work_group, newname))
  {
((void)0);\
    return;
  }
#endif

  StrnCpy(work->local_master_browser_name, newname,
            sizeof(work->local_master_browser_name)-1);
}
