/*
 * linux/drivers/ide/sc12000.c		Version 0.1	Mar. 1, 2001
 *
 * Copyright (C) 2000			Andre Hedrick <andre@linux-ide.org>
 * Ditto of GNU General Public License.
 *
 * Copyright (C) 2000			Mark Lord <mlord@pobox.com>
 * May be copied or modified under the terms of the GNU General Public License
 *
 * Development of this chipset driver was funded
 * by the nice folks at National Semiconductor.
 */

#include <linux/config.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/timer.h>
#include <linux/mm.h>
#include <linux/ioport.h>
#include <linux/blkdev.h>
#include <linux/hdreg.h>
#include <linux/interrupt.h>
#include <linux/pci.h>
#include <linux/init.h>
#include <linux/ide.h>
#include <asm/io.h>
#include <asm/irq.h>

#include "ide_modes.h"

#define DISPLAY_SC1200_TIMINGS

#if defined(DISPLAY_SC1200_TIMINGS) && defined(CONFIG_PROC_FS)
#include <linux/stat.h>
#include <linux/proc_fs.h>

static int sc1200_get_info(char *, char **, off_t, int);
extern int (*sc1200_display_info)(char *, char **, off_t, int); /* ide-proc.c */
extern char *ide_media_verbose(ide_drive_t *);
static struct pci_dev *bmide_dev;

static int sc1200_get_info (char *buffer, char **addr, off_t offset, int count)
{
	char *p = buffer;
	u32 bibma = pci_resource_start(bmide_dev, 4);
	u8  c0 = 0, c1 = 0;

	/*
	 * at that point bibma+0x2 et bibma+0xa are byte registers
	 * to investigate:
	 */

	c0 = inb_p((unsigned short)bibma + 0x02);
	c1 = inb_p((unsigned short)bibma + 0x0a);

	p += sprintf(p, "\n                               National SCx200 Chipset.\n");
	p += sprintf(p, "--------------- Primary Channel ---------------- Secondary Channel -------------\n");
	p += sprintf(p, "                %sabled                         %sabled\n",
			(c0&0x80) ? "dis" : " en",
			(c1&0x80) ? "dis" : " en");
	p += sprintf(p, "--------------- drive0 --------- drive1 -------- drive0 ---------- drive1 ------\n");
	p += sprintf(p, "DMA enabled:    %s              %s             %s               %s\n",
			(c0&0x20) ? "yes" : "no ", (c0&0x40) ? "yes" : "no ",
			(c1&0x20) ? "yes" : "no ", (c1&0x40) ? "yes" : "no " );

	p += sprintf(p, "UDMA\n");
	p += sprintf(p, "DMA\n");
	p += sprintf(p, "PIO\n");

	return p-buffer;
}
#endif /* DISPLAY_SC1200_TIMINGS && CONFIG_PROC_FS */

byte sc1200_proc = 0;

extern char *ide_xfer_verbose (byte xfer_rate);

/*
 * Set a new transfer mode at the drive
 */
int sc1200_set_xfer_mode (ide_drive_t *drive, byte mode)
{
	int error = 0;

	printk("%s: sc1200_set_xfer_mode(%s)\n", drive->name, ide_xfer_verbose(mode));
	error = ide_config_drive_speed(drive, mode);

	return error;
}

/*
 * Here are the standard PIO mode 0-4 timings for each "format".
 * Format-0 uses fast data reg timings, with slower command reg timings.
 * Format-1 uses fast timings for all registers, but won't work with all drives.
 */
static unsigned int sc1200_pio_timings[2][5] =
	{{0x00009172, 0x00012171, 0x00020080, 0x00032010, 0x00040010},
	 {0xd1329172, 0x71212171, 0x30200080, 0x20102010, 0x00100010}};


/*
 * After chip reset, the PIO timings are set to 0x00009172, which is not valid.
 */
#define SC1200_BAD_PIO(timings) (((timings)&~0x80000000)==0x00009172)

/*  to read/write the timing registers of the sc1200, one needs to use the
    pci read/write mechanism, not the simple i/o of the cx5530....  
*/ 

/* base is the first PIO timing reg, e.g. BAR 12 in the header
   adjusted by the ide channel offset...  */
#define BASE_REGISTER (0x80009240 + ((hwif)->channel ? 0x10 : 0x00))

unsigned int    port1 = (unsigned int)0xcf8;
unsigned int    port2 = (unsigned int)0xcfc;



/*
 * sc1200_tuneproc() handles selection/setting of PIO modes
 * for both the chipset and drive.
 *
 * The ide_init_sc1200() routine guarantees that all drives
 * will have valid default PIO timings set up before we get here.
 */
static void sc1200_tuneproc (ide_drive_t *drive, byte pio)	/* pio=255 means "autotune" */
{
	ide_hwif_t	*hwif = HWIF(drive);
	unsigned int	format;
	static byte	modes[5] = {XFER_PIO_0, XFER_PIO_1, XFER_PIO_2, XFER_PIO_3, XFER_PIO_4};

        printk("SC1200: setting PIO modes\n");
	pio = ide_get_best_pio_mode(drive, pio, 4, NULL);
	if (!sc1200_set_xfer_mode(drive, modes[pio])) {
  	        outl((BASE_REGISTER),port1);
		format = (inl(port2) >> 31) & 1;

		outl((BASE_REGISTER+((drive->select.b.unit<<3))),port1);
		outl(sc1200_pio_timings[format][pio],port2);
 	}
}

#ifdef CONFIG_BLK_DEV_IDEDMA
/*
 * sc1200_config_dma() handles selection/setting of DMA/UDMA modes
 * for both the chipset and drive.
 */
static int sc1200_config_dma (ide_drive_t *drive)
{
	int			udma_ok = 1, mode = 0;
	ide_hwif_t		*hwif = HWIF(drive);
	int			unit = drive->select.b.unit;
	ide_drive_t		*mate = &hwif->drives[unit^1];
	struct hd_driveid	*id = drive->id;
	unsigned int		basereg, reg, timings;

	/*
	 * Default to DMA-off in case we run into trouble here.
	 */
	(void)hwif->dmaproc(ide_dma_off_quietly, drive);	/* turn off DMA while we fiddle */
	outb(inb(hwif->dma_base+2)&~(unit?0x40:0x20), hwif->dma_base+2); /* clear DMA_capable bit */

	/*
	 * The SC1200 specifies that two drives sharing a cable cannot
	 * mix UDMA/MDMA.  It has to be one or the other, for the pair,
	 * though different timings can still be chosen for each drive.
	 * We could set the appropriate timing bits on the fly,
	 * but that might be a bit confusing.  So, for now we statically
	 * handle this requirement by looking at our mate drive to see
	 * what it is capable of, before choosing a mode for our own drive.
	 */
	if (mate->present) {
		struct hd_driveid *mateid = mate->id;
		if (mateid && (mateid->capability & 1) && !hwif->dmaproc(ide_dma_bad_drive, mate)) {
			if ((mateid->field_valid & 4) && (mateid->dma_ultra & 7))
				udma_ok = 1;
			else if ((mateid->field_valid & 2) && (mateid->dma_mword & 7))
				udma_ok = 0;
			else
				udma_ok = 1;
		}
	}

	/*
	 * Now see what the current drive is capable of,
	 * selecting UDMA only if the mate said it was ok.
	 */
	if (id && (id->capability & 1) && hwif->autodma && !hwif->dmaproc(ide_dma_bad_drive, drive)) {
		if (udma_ok && (id->field_valid & 4) && (id->dma_ultra & 7)) {
			if      (id->dma_ultra & 4)
				mode = XFER_UDMA_2;
			else if (id->dma_ultra & 2)
				mode = XFER_UDMA_1;
			else if (id->dma_ultra & 1)
				mode = XFER_UDMA_0;
		}
		if (!mode && (id->field_valid & 2) && (id->dma_mword & 7)) {
			if      (id->dma_mword & 4)
				mode = XFER_MW_DMA_2;
			else if (id->dma_mword & 2)
				mode = XFER_MW_DMA_1;
			else if (id->dma_mword & 1)
				mode = XFER_MW_DMA_0;
		}
	}

	/*
	 * Tell the drive to switch to the new mode; abort on failure.
	 */
	if (!mode || sc1200_set_xfer_mode(drive, mode)) {
		return 1;	/* failure */
	}
	/*
	 * Now tune the chipset to match the drive:
	 */
	switch (mode) {
		case XFER_UDMA_0:	timings = 0x00921250; break;
		case XFER_UDMA_1:	timings = 0x00911140; break;
		case XFER_UDMA_2:	timings = 0x00911030; break;
		case XFER_MW_DMA_0:	timings = 0x00077771; break;
		case XFER_MW_DMA_1:	timings = 0x00012121; break;
		case XFER_MW_DMA_2:	timings = 0x00002020; break;
		default:
			printk("%s: sc1200_config_dma: huh? mode=%02x\n", drive->name, mode);
			return 1;	/* failure */
	}

        outl((BASE_REGISTER+4),port1);
        reg = inl(port2);	 	     /* get unit0 config register */

	if (unit == 0) {		     /* are we configuring drive0? */
	        timings |= reg & 0x80000000;	/* preserve PIO format bit */

		outl((BASE_REGISTER+4),port1);
		outl(timings,port2);	     /* write drive0 timing register */

	} 
	else {

		outl((BASE_REGISTER+12),port1);
		outl(timings,port2);    	/* write drive1 config register */

	}
	
	outb(inb(hwif->dma_base+2)|(unit?0x40:0x20), hwif->dma_base+2);	/* set DMA_capable bit */

	/*
	 * Finally, turn DMA on in software, and exit.
	 */
	return hwif->dmaproc(ide_dma_on, drive);	/* success */
}



/*  Replacement for the standard ide_dma_end action in
    dma_proc.
   
    returns 1 on error, 0 otherwise
*/

int sc1200_ide_dma_end (ide_drive_t *drive)
{
	ide_hwif_t *hwif = HWIF(drive);
	unsigned long dma_base = hwif->dma_base;
	unsigned int count, reading = 0;
	byte dma_stat;
	int ret = 0;

	
	dma_stat = inb(dma_base+2);		/* get DMA status */

	if (!(dma_stat & 4))
	    printk(" ide_dma_end dma_stat=%0x err=%x newerr=%x\n", 
                    dma_stat, ((dma_stat&7)!=4), ((dma_stat&2)==2));
 
	outb(dma_stat|0x1b, dma_base+2);      /* clear the INTR & ERROR bits */
	outb(inb(dma_base)&~1, dma_base);     /* !! DO THIS HERE !! stop DMA */
        
       drive->waiting_for_dma = 0;
       ide_destroy_dmatable(drive);          /* purge DMA mappings */

	return (dma_stat & 7) != 4;	      /* verify good DMA status */

}





/*
 * This is a SC1200-specific wrapper for the standard ide_dmaproc().
 * We need it for our custom "ide_dma_check" function.
 * All other requests are forwarded to the standard ide_dmaproc().
 */
int sc1200_dmaproc (ide_dma_action_t func, ide_drive_t *drive)
{
	switch (func) {
		case ide_dma_check:
			return sc1200_config_dma(drive);
	        case ide_dma_end:
		        return sc1200_ide_dma_end(drive);
		default:
			break;
	}
	/* Other cases are done by generic IDE-DMA code. */
	return ide_dmaproc(func, drive);
}
#endif /* CONFIG_BLK_DEV_IDEDMA */

/*
 * Initialize the sc1200 bridge for reliable IDE DMA operation.
 */
unsigned int __init pci_init_sc1200 (struct pci_dev *dev, const char *name)
{
	struct pci_dev *master_0 = NULL;
	unsigned short pcicmd = 0;
	unsigned long flags;

#if defined(DISPLAY_SC1200_TIMINGS) && defined(CONFIG_PROC_FS)
	if (!sc1200_proc) {
		sc1200_proc = 1;
		bmide_dev = dev;
		sc1200_display_info = &sc1200_get_info;
	}
#endif /* DISPLAY_SC1200_TIMINGS && CONFIG_PROC_FS */

	pci_for_each_dev (dev) {
		if (dev->vendor == PCI_VENDOR_ID_NATIONAL) {
			switch (dev->device) {
				case PCI_DEVICE_ID_NATIONAL_SC1200_IDE:
					master_0 = dev;
					break;

			}
		}
	}
	if (!master_0) {
		printk("%s: unable to locate PCI MASTER function\n", name);
		return 0;
	}


	save_flags(flags);
	cli();	/* all CPUs (there should only be one CPU with this chipset) */

	restore_flags(flags);

	return 0;
}

/*
 * This gets invoked by the IDE driver once for each channel,
 * and performs channel-specific pre-initialization before drive probing.
 */
void __init ide_init_sc1200 (ide_hwif_t *hwif)
{
	if (hwif->mate)
		hwif->serialized = hwif->mate->serialized = 1;
	if (!hwif->dma_base) {
		hwif->autodma = 0;
	} else {

#ifdef CONFIG_BLK_DEV_IDEDMA
		hwif->dmaproc  = &sc1200_dmaproc;
#else
		hwif->autodma = 0;
#endif /* CONFIG_BLK_DEV_IDEDMA */

		hwif->tuneproc = &sc1200_tuneproc;

		/* ??? PIO TIMINGS ??? */
		/*   should not need to be set, as all
                 *   bios's set the PIO timings during power on
                 */
	}
}













