/*
   Unix SMB/Netbios implementation.
   Version 1.9.
   Samba utility functions
   Copyright (C) Andrew Tridgell 1992-1998

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* fork a child process to exec passwd and write to its
 * tty to change a users password. This is running as the
 * user who is attempting to change the password.
 */

/*
 * This code was copied/borrowed and stolen from various sources.
 * The primary source was the poppasswd.c from the authors of POPMail. This software
 * was included as a client to change passwords using the 'passwd' program
 * on the remote machine.
 *
 * This routine is called by set_user_password() in password.c only if ALLOW_PASSWORD_CHANGE
 * is defined in the compiler directives located in the Makefile.
 *
 * This code has been hacked by Bob Nance (nance@niehs.nih.gov) and Evan Patterson
 * (patters2@niehs.nih.gov) at the National Institute of Environmental Health Sciences
 * and rights to modify, distribute or incorporate this change to the CAP suite or
 * using it for any other reason are granted, so long as this disclaimer is left intact.
 */

/*
   This code was hacked considerably for inclusion in Samba, primarily
   by Andrew.Tridgell@anu.edu.au. The biggest change was the addition
   of the "password chat" option, which allows the easy runtime
   specification of the expected sequence of events to change a
   password.
   */

#include "includes.h"

extern struct passdb_ops pdb_ops;

#if ALLOW_CHANGE_PASSWORD

static int findpty(char **slave)
{
	int master;
	static fstring line;
	DIR *dirp;
	char *dpname;

#if defined(HAVE_GRANTPT)
	/* Try to open /dev/ptmx. If that fails, fall through to old method. */
	if ((master = sys_open("/dev/ptmx", O_RDWR, 0)) >= 0)
	{
		grantpt(master);
		unlockpt(master);
		*slave = (char *)ptsname(master);
		if (*slave == NULL)
		{
((void)0);
			/* Stop fd leak on error. */
			close(master);
			return -1;
		}
		else
		{
((void)0);
			return (master);
		}
	}
#endif /* HAVE_GRANTPT */

	fstrcpy(line, "/dev/ptyXX");

	dirp = opendir("/dev");
	if (!dirp)
		return (-1);
	while ((dpname = readdirname(dirp)) != NULL)
	{
		if (strncmp(dpname, "pty", 3) == 0 && strlen(dpname) == 5)
		{
((void)0);
			line[8] = dpname[3];
			line[9] = dpname[4];
			if ((master = sys_open(line, O_RDWR, 0)) >= 0)
			{
((void)0);
				line[5] = 't';
				*slave = line;
				closedir(dirp);
				return (master);
			}
		}
	}
	closedir(dirp);
	return (-1);
}

static int dochild(int master, char *slavedev, char *name,
		   char *passwordprogram, BOOL as_root)
{
	int slave;
	struct termios stermios;
	struct passwd *pass = Get_Pwnam(name, True);
	gid_t gid;
	uid_t uid;

	if (pass == NULL)
	{
((void)0);
		return False;
	}

	gid = pass->pw_gid;
	uid = pass->pw_uid;

	gain_root_privilege();

	/* Start new session - gets rid of controlling terminal. */
	if (setsid() < 0)
	{
((void)0);
		return (False);
	}

	/* Open slave pty and acquire as new controlling terminal. */
	if ((slave = sys_open(slavedev, O_RDWR, 0)) < 0)
	{
((void)0);
		return (False);
	}
#ifdef I_PUSH
	ioctl(slave, I_PUSH, "ptem");
	ioctl(slave, I_PUSH, "ldterm");
#elif defined(TIOCSCTTY)
	if (ioctl(slave, TIOCSCTTY, 0) < 0)
	{
((void)0);
		/* return(False); */
	}
#endif

	/* Close master. */
	close(master);

	/* Make slave stdin/out/err of child. */

	if (dup2(slave, STDIN_FILENO) != STDIN_FILENO)
	{
((void)0);
		return (False);
	}
	if (dup2(slave, STDOUT_FILENO) != STDOUT_FILENO)
	{
((void)0);
		return (False);
	}
	if (dup2(slave, STDERR_FILENO) != STDERR_FILENO)
	{
((void)0);
		return (False);
	}
	if (slave > 2)
		close(slave);

	/* Set proper terminal attributes - no echo, canonical input processing,
	   no map NL to CR/NL on output. */

	if (tcgetattr(0, &stermios) < 0)
	{
((void)0);
		return (False);
	}
	stermios.c_lflag &= ~(ECHO | ECHOE | ECHOK | ECHONL);
	stermios.c_lflag |= ICANON;
#ifdef ONLCR
 	stermios.c_oflag &= ~(ONLCR);
#endif
	if (tcsetattr(0, TCSANOW, &stermios) < 0) {
((void)0);
		return (False);
	}

	/* make us completely into the right uid */
	if (!as_root) {
		become_user_permanently(uid, gid);
	}

((void)0);

	/* execl() password-change application */
	if (execl("/bin/sh", "sh", "-c", passwordprogram, NULL) < 0)
	{
((void)0);
		return (False);
	}
	return (True);
}

static int expect(int master, char *issue, char *expected)
{
	pstring buffer;
	int attempts, timeout, nread, len;
	BOOL match = False;

	for (attempts = 0; attempts < 2; attempts++) {
		if (!strequal(issue, ".")) {
			if (lp_passwd_chat_debug())
((void)0);

			if ((len = write(master, issue, strlen(issue))) != strlen(issue)) {
((void)0);
				return False;
			}
		}

		if (strequal(expected, "."))
			return True;

		timeout = 2000;
		nread = 0;
		buffer[nread] = 0;

		while ((len = read_with_timeout(master, buffer + nread, 1,
						sizeof(buffer) - nread - 1,
						timeout)) > 0) {
			nread += len;
			buffer[nread] = 0;

			{
				/* Eat leading/trailing whitespace before match. */
				pstring str;
				pstrcpy( str, buffer);
				trim_string( str, " ", " ");

				if ((match = (unix_wild_match(expected, str) == 0)))
					timeout = 200;
			}
		}

		if (lp_passwd_chat_debug())
((void)0);

		if (match)
			break;

		if (len < 0) {
((void)0);
			return False;
		}
	}

((void)0);
	return match;
}

static void pwd_sub(char *buf)
{
	all_string_sub(buf, "\\n", "\n", 0);
	all_string_sub(buf, "\\r", "\r", 0);
	all_string_sub(buf, "\\s", " ", 0);
	all_string_sub(buf, "\\t", "\t", 0);
}

static int talktochild(int master, const char *seq)
{
	int count = 0;
	fstring issue, expected;

	fstrcpy(issue, ".");

	while (next_token(&seq, expected, NULL, sizeof(expected)))
	{
		pwd_sub(expected);
		count++;

		if (!expect(master, issue, expected))
		{
((void)0);
			return False;
		}

		if (!next_token(&seq, issue, NULL, sizeof(issue)))
			fstrcpy(issue, ".");

		pwd_sub(issue);
	}
	if (!strequal(issue, ".")) {
		/* we have one final issue to send */
		fstrcpy(expected, ".");
		if (!expect(master, issue, expected))
			return False;
	}

	return (count > 0);
}

static BOOL chat_with_program(char *passwordprogram, char *name,
			      char *chatsequence, BOOL as_root)
{
	char *slavedev;
	int master;
	pid_t pid, wpid;
	int wstat;
	BOOL chstat = False;

	/* allocate a pseudo-terminal device */
	if ((master = findpty(&slavedev)) < 0)
	{
((void)0);
		return (False);
	}

	/*
	 * We need to temporarily stop CatchChild from eating
	 * SIGCLD signals as it also eats the exit status code. JRA.
	 */

	CatchChildLeaveStatus();

	if ((pid = sys_fork()) < 0)
	{
((void)0);
		close(master);
		CatchChild();
		return (False);
	}

	/* we now have a pty */
	if (pid > 0)
	{			/* This is the parent process */
		if ((chstat = talktochild(master, chatsequence)) == False)
		{
((void)0);
			kill(pid, SIGKILL);	/* be sure to end this process */
		}

		while ((wpid = sys_waitpid(pid, &wstat, 0)) < 0)
		{
			if (errno == EINTR)
			{
				errno = 0;
				continue;
			}
			break;
		}

		if (wpid < 0)
		{
((void)0);
			close(master);
			CatchChild();
			return (False);
		}

		/*
		 * Go back to ignoring children.
		 */
		CatchChild();

		close(master);

		if (pid != wpid)
		{
((void)0);
			return (False);
		}
		if (WIFEXITED(wstat) == 0)
		{
((void)0);
			return (False);
		}
		if (WEXITSTATUS(wstat) != 0)
		{
((void)0);
			return (False);
		}

	}
	else
	{
		/* CHILD */

		/*
		 * Lose any oplock capabilities.
		 */
		oplock_set_capability(False, False);

		/* make sure it doesn't freeze */
		alarm(20);

		if (as_root)
			become_root();

((void)0);
		chstat =
			dochild(master, slavedev, name, passwordprogram,
				as_root);

		if (as_root)
			unbecome_root();

		/*
		 * The child should never return from dochild() ....
		 */

((void)0);
		exit(1);
	}

	if (chstat)
((void)0);
	return (chstat);
}


BOOL chgpasswd(char *name, const char *oldpass, const char *newpass, BOOL as_root)
{
	pstring passwordprogram;
	pstring chatsequence;
	size_t i;
	size_t len;

	strlower(name);
((void)0);

#if DEBUG_PASSWORD
((void)0);
#endif

	/* Take the passed information and test it for minimum criteria */
	/* Minimum password length */
	if (strlen(newpass) < lp_min_passwd_length()) {
		/* too short, must be at least MINPASSWDLENGTH */
((void)0);
		return (False);	/* inform the user */
	}

	/* Password is same as old password */
	if (strcmp(oldpass, newpass) == 0) {
		/* don't allow same password */
((void)0);
		return (False);	/* inform the user */
	}

	/*
	 * Check the old and new passwords don't contain any control
	 * characters.
	 */

	len = strlen(oldpass);
	for (i = 0; i < len; i++) {
		if (iscntrl((int)oldpass[i])) {
((void)0);
			return False;
		}
	}

	len = strlen(newpass);
	for (i = 0; i < len; i++) {
		if (iscntrl((int)newpass[i])) {
((void)0);
			return False;
		}
	}

#ifdef WITH_PAM
	if (lp_pam_password_change()) {
		BOOL ret;

		if (as_root)
			become_root();

		ret = smb_pam_passchange(name, oldpass, newpass);

		if (as_root)
			unbecome_root();

		return ret;
	}
#endif

	pstrcpy(passwordprogram, lp_passwd_program());
	pstrcpy(chatsequence, lp_passwd_chat());

	if (!*chatsequence) {
((void)0);
		return (False);
	}

	if (!*passwordprogram) {
((void)0);
		return (False);
	}

	if (as_root) {
		/* The password program *must* contain the user name to work. Fail if not. */
		if (strstr(passwordprogram, "%u") == NULL) {
((void)0);\
			return False;
		}
	}

	pstring_sub(passwordprogram, "%u", name);
	/* note that we do NOT substitute the %o and %n in the password program
	   as this would open up a security hole where the user could use
	   a new password containing shell escape characters */

	pstring_sub(chatsequence, "%u", name);
	all_string_sub(chatsequence, "%o", oldpass, sizeof(pstring));
	all_string_sub(chatsequence, "%n", newpass, sizeof(pstring));
	return (chat_with_program
		(passwordprogram, name, chatsequence, as_root));
}

#else /* ALLOW_CHANGE_PASSWORD */

BOOL chgpasswd(char *name, const char *oldpass, const char *newpass, BOOL as_root)
{
((void)0);
	return (False);
}
#endif /* ALLOW_CHANGE_PASSWORD */

/***********************************************************
 Code to check the lanman hashed password.
************************************************************/

BOOL check_lanman_password(char *user, uchar * pass1,
			   uchar * pass2, SAM_ACCOUNT **hnd)
{
	static uchar null_pw[16];
	uchar unenc_new_pw[16];
	uchar unenc_old_pw[16];
	SAM_ACCOUNT *sampass = NULL;
	uint16 acct_ctrl;
	uint8 *lanman_pw;
	BOOL ret;

	become_root();
	ret = pdb_getsampwnam(sampass, user);
	unbecome_root();

	if (ret == False) {
((void)0);
		pdb_free_sam(sampass);
		return False;
	}

	acct_ctrl = pdb_get_acct_ctrl     (sampass);
	lanman_pw = pdb_get_lanman_passwd (sampass);

	if (acct_ctrl & ACB_DISABLED) {
((void)0);
		pdb_free_sam(sampass);
		return False;
	}

	if ((lanman_pw == NULL) && (acct_ctrl & ACB_PWNOTREQ)) {
		uchar no_pw[14];
		memset(no_pw, '\0', 14);
		E_P16(no_pw, null_pw);
		if (!pdb_set_lanman_passwd(sampass, null_pw)) {
			pdb_free_sam(sampass);
			return False;
		}
	}
	else if (lanman_pw == NULL) {
((void)0);
		pdb_free_sam(sampass);
		return False;
	}

	/* Get the new lanman hash. */
	D_P16(lanman_pw, pass2, unenc_new_pw);

	/* Use this to get the old lanman hash. */
	D_P16(unenc_new_pw, pass1, unenc_old_pw);

	/* Check that the two old passwords match. */
	if (memcmp(lanman_pw, unenc_old_pw, 16)) {
((void)0);
		pdb_free_sam(sampass);
		return False;
	}

	/* this saves the pointer for the caller */
	*hnd = sampass;

	return True;
}

/***********************************************************
 Code to change the lanman hashed password.
 It nulls out the NT hashed password as it will
 no longer be valid.
************************************************************/

BOOL change_lanman_password(SAM_ACCOUNT *sampass, uchar * pass1,
			    uchar * pass2)
{
	static uchar null_pw[16];
	uchar unenc_new_pw[16];
	BOOL ret;
	uint16 acct_ctrl;
	uint8 *pwd;

	if (sampass == NULL) {
((void)0);
		return False;
	}

	acct_ctrl = pdb_get_acct_ctrl(sampass);
	pwd = pdb_get_lanman_passwd(sampass);

	if (acct_ctrl & ACB_DISABLED) {
((void)0);
		return False;
	}

	if ((pwd == NULL) && (acct_ctrl & ACB_PWNOTREQ)) {
		uchar no_pw[14];
		memset(no_pw, '\0', 14);
		E_P16(no_pw, null_pw);
		if (!pdb_set_lanman_passwd(sampass, null_pw))
			return False;
	}
	else if (pwd == NULL) {
((void)0);
		return False;
	}

	/* Get the new lanman hash. */
	D_P16(pwd, pass2, unenc_new_pw);

	if (!pdb_set_lanman_passwd(sampass, unenc_new_pw))
		return False;
	pdb_set_nt_passwd    (sampass, NULL);	/* We lose the NT hash. Sorry. */

	/* Now flush the sam_passwd struct to persistent storage */
	become_root();
	ret = pdb_update_sam_account (sampass, False);
	unbecome_root();

	return ret;
}

/***********************************************************
 Code to check and change the OEM hashed password.
************************************************************/
BOOL pass_oem_change(char *user,
		     uchar * lmdata, uchar * lmhash,
		     uchar * ntdata, uchar * nthash)
{
	fstring new_passwd;
	SAM_ACCOUNT *sampass = NULL;
	BOOL ret = check_oem_password(user, lmdata, lmhash, ntdata, nthash,
				      &sampass, new_passwd, sizeof(new_passwd));

	/*
	 * At this point we have the new case-sensitive plaintext
	 * password in the fstring new_passwd. If we wanted to synchronise
	 * with UNIX passwords we would call a UNIX password changing
	 * function here. However it would have to be done as root
	 * as the plaintext of the old users password is not
	 * available. JRA.
	 */

	if (ret && lp_unix_password_sync())
		ret = chgpasswd(user, "", new_passwd, True);

	if (ret)
		ret = change_oem_password(sampass, new_passwd, False);

	memset(new_passwd, 0, sizeof(new_passwd));

	pdb_free_sam(sampass);

	return ret;
}

/***********************************************************
 Code to check the OEM hashed password.

 this function ignores the 516 byte nt OEM hashed password
 but does use the lm OEM password to check the nt hashed-hash.

************************************************************/

BOOL check_oem_password(char *user,
			uchar * lmdata, uchar * lmhash,
			uchar * ntdata, uchar * nthash,
			SAM_ACCOUNT **hnd, char *new_passwd,
			int new_passwd_size)
{
	static uchar null_pw[16];
	static uchar null_ntpw[16];
	SAM_ACCOUNT *sampass = NULL;
	uint8 *lanman_pw, *nt_pw;
	uint16 acct_ctrl;
	int new_pw_len;
	uchar new_ntp16[16];
	uchar unenc_old_ntpw[16];
	uchar new_p16[16];
	uchar unenc_old_pw[16];
	char no_pw[2];
	BOOL ret;

	BOOL nt_pass_set = (ntdata != NULL && nthash != NULL);

	*hnd = NULL;

	pdb_init_sam(&sampass);

	become_root();
	ret = pdb_getsampwnam(sampass, user);
	unbecome_root();

	if (ret == False) {
((void)0);
		pdb_free_sam(sampass);
		return False;
	}

	acct_ctrl = pdb_get_acct_ctrl(sampass);

	if (acct_ctrl & ACB_DISABLED) {
((void)0);
		pdb_free_sam(sampass);
		return False;
	}

	/* construct a null password (in case one is needed */
	no_pw[0] = 0;
	no_pw[1] = 0;
	nt_lm_owf_gen(no_pw, null_ntpw, null_pw);

	/* save pointers to passwords so we don't have to keep looking them up */
	lanman_pw = pdb_get_lanman_passwd(sampass);
	nt_pw = pdb_get_nt_passwd(sampass);

	/* check for null passwords */
	if (lanman_pw == NULL) {
		if (acct_ctrl & ACB_PWNOTREQ) {
			if (!pdb_set_lanman_passwd(sampass, null_pw)) {
				pdb_free_sam(sampass);
				return False;
			}
			lanman_pw = pdb_get_lanman_passwd(sampass);
			if (!lanman_pw) {
				pdb_free_sam(sampass);
				return False;
			}
		} else {
((void)0);
			pdb_free_sam(sampass);
			return False;
		}
	}

	if (pdb_get_nt_passwd(sampass) == NULL && nt_pass_set) {
		if (acct_ctrl & ACB_PWNOTREQ) {
			pdb_set_nt_passwd(sampass, null_ntpw);
			nt_pw = pdb_get_nt_passwd(sampass);
			if (!nt_pw) {
				pdb_free_sam(sampass);
				return False;
			}
		} else {
((void)0);
			pdb_free_sam(sampass);
			return False;
		}
	}

	/*
	 * Call the hash function to get the new password.
	 */
	SamOEMhash((uchar *) lmdata, (uchar *)lanman_pw, 516);

	/*
	 * The length of the new password is in the last 4 bytes of
	 * the data buffer.
	 */

	new_pw_len = IVAL(lmdata, 512);
	if (new_pw_len < 0 || new_pw_len > new_passwd_size - 1) {
((void)0);
		pdb_free_sam(sampass);
		return False;
	}

	if (nt_pass_set) {
		/*
		 * nt passwords are in unicode
		 */
		int uni_pw_len = new_pw_len;
		char *pw;
		new_pw_len /= 2;
		pw = dos_unistrn2((uint16 *)(&lmdata[512 - uni_pw_len]), new_pw_len);
		memcpy(new_passwd, pw, new_pw_len + 1);
	} else {
		memcpy(new_passwd, &lmdata[512 - new_pw_len], new_pw_len);
		new_passwd[new_pw_len] = 0;
	}

	/*
	 * To ensure we got the correct new password, hash it and
	 * use it as a key to test the passed old password.
	 */

	nt_lm_owf_gen(new_passwd, new_ntp16, new_p16);

	if (!nt_pass_set) {
		/*
		 * Now use new_p16 as the key to see if the old
		 * password matches.
		 */
		D_P16(new_p16, lmhash, unenc_old_pw);

		if (memcmp(lanman_pw, unenc_old_pw, 16)) {
((void)0);
			pdb_free_sam(sampass);
			return False;
		}

#ifdef DEBUG_PASSWORD
((void)0);
#endif
		*hnd = sampass;
		return True;
	}

	/*
	 * Now use new_p16 as the key to see if the old
	 * password matches.
	 */
	D_P16(new_ntp16, lmhash, unenc_old_pw);
	D_P16(new_ntp16, nthash, unenc_old_ntpw);

	if (memcmp(lanman_pw, unenc_old_pw, 16)) {
((void)0);
		pdb_free_sam(sampass);
		return False;
	}

	if (memcmp(nt_pw, unenc_old_ntpw, 16)) {
((void)0);
		pdb_free_sam(sampass);
		return False;
	}
#ifdef DEBUG_PASSWORD
((void)0);
#endif
	*hnd = sampass;
	return True;
}

/***********************************************************
 Code to change the oem password. Changes both the lanman
 and NT hashes.
 override = False, normal
 override = True, override XXXXXXXXXX'd password
************************************************************/

BOOL change_oem_password(SAM_ACCOUNT *hnd, char *new_passwd,
			 BOOL override)
{
	BOOL ret;
	uchar new_nt_p16[16];
	uchar new_p16[16];

	nt_lm_owf_gen(new_passwd, new_nt_p16, new_p16);

	if (!pdb_set_lanman_passwd (hnd, new_p16))
		return False;
	if (!pdb_set_nt_passwd(hnd, new_nt_p16))
		return False;

	/* Now write it into the file. */
	become_root();
	ret = pdb_update_sam_account (hnd, override);
	unbecome_root();

	memset(new_passwd, '\0', strlen(new_passwd));

	return ret;
}

/***********************************************************
 Code to check a plaintext password against smbpasswd entries.
***********************************************************/

BOOL check_plaintext_password(char *user, char *old_passwd,
			      int old_passwd_size, SAM_ACCOUNT **hnd)
{
	SAM_ACCOUNT  *sampass = NULL;
	uchar old_pw[16], old_ntpw[16];
	BOOL ret;

	*hnd = NULL;

	pdb_init_sam(&sampass);

	become_root();
	ret = pdb_getsampwnam(sampass, user);
	unbecome_root();

	if (ret == False) {
((void)0);
		pdb_free_sam(sampass);
		return False;
	}

	if (pdb_get_acct_ctrl(sampass) & ACB_DISABLED) {
((void)0);
		pdb_free_sam(sampass);
		return (False);
	}

	nt_lm_owf_gen(old_passwd, old_ntpw, old_pw);

#ifdef DEBUG_PASSWORD
((void)0);
	dump_data(100, pdb_get_nt_passwd(sampass), 16);
((void)0);
	dump_data(100, old_ntpw, 16);
((void)0);
	dump_data(100, pdb_get_lanman_passwd(sampass), 16);
((void)0);
	dump_data(100, old_pw, 16);
#endif

	if (memcmp(pdb_get_nt_passwd(sampass), old_ntpw, 16)
	    && memcmp(pdb_get_lanman_passwd(sampass), old_pw, 16)) {
		pdb_free_sam(sampass);
		return (False);
	} else {
		*hnd = sampass;
		return (True);
	}
}
