
/******************************************************************************/
/*                                                                            */
/* Broadcom BCM5700 Linux Network Driver, Copyright (c) 2000 Broadcom         */
/* Corporation.                                                               */
/* All rights reserved.                                                       */
/*                                                                            */
/* This program is free software; you can redistribute it and/or modify       */
/* it under the terms of the GNU General Public License as published by       */
/* the Free Software Foundation, located in the file LICENSE.                 */
/*                                                                            */
/* /proc file system handling code.                                           */
/*                                                                            */
/******************************************************************************/

#include "mm.h"
#ifdef CONFIG_PROC_FS

#define NICINFO_PROC_DIR "nicinfo"

static struct proc_dir_entry *bcm5700_procfs_dir;

extern char bcm5700_driver[], bcm5700_version[];

static struct proc_dir_entry *
proc_getdir(char *name, struct proc_dir_entry *proc_dir)
{
	struct proc_dir_entry *pde = proc_dir;

	lock_kernel();
	for (pde=pde->subdir; pde; pde = pde->next) {
		if (pde->namelen && (strcmp(name, pde->name) == 0)) {
			/* directory exists */
			break;
		}
	}
	if (pde == (struct proc_dir_entry *) 0)
	{
		/* create the directory */
#if (LINUX_VERSION_CODE > 0x20300)
		pde = proc_mkdir(name, proc_dir);
#else
		pde = create_proc_entry(name, S_IFDIR, proc_dir);
#endif
		if (pde == (struct proc_dir_entry *) 0) {
			unlock_kernel();
			return (pde);
		}
	}
	unlock_kernel();
	return (pde);
}

int
bcm5700_proc_create(void)
{
	bcm5700_procfs_dir = proc_getdir(NICINFO_PROC_DIR, proc_net);

	if (bcm5700_procfs_dir == (struct proc_dir_entry *) 0) {
		printk(KERN_DEBUG "Could not create procfs nicinfo directory %s\n", NICINFO_PROC_DIR);
		return -1;
	}
	return 0;
}

int
bcm5700_read_pfs(char *page, char **start, off_t off, int count,
	int *eof, void *data)
{
	struct net_device *dev = (struct net_device *) data;
	PUM_DEVICE_BLOCK pUmDevice = (PUM_DEVICE_BLOCK) dev->priv;
	PLM_DEVICE_BLOCK pDevice = &pUmDevice->lm_dev;
	PT3_STATS_BLOCK pStats = (PT3_STATS_BLOCK) pDevice->pStatsBlkVirt;
	int len = 0;

	len += sprintf(page+len, "Description\t\t%s\n", pUmDevice->name);
	len += sprintf(page+len, "Driver_Name\t\t%s\n", bcm5700_driver);
	len += sprintf(page+len, "Driver_Version\t\t%s\n", bcm5700_version);
	len += sprintf(page+len, "PCI_Vendor\t\t0x%04x\n", pDevice->PciVendorId);
	len += sprintf(page+len, "PCI_Device_ID\t\t0x%04x\n",
		pDevice->PciDeviceId);
	len += sprintf(page+len, "PCI_Subsystem_Vendor\t0x%04x\n",
		pDevice->SubsystemVendorId);
	len += sprintf(page+len, "PCI_Subsystem_ID\t0x%04x\n",
		pDevice->SubsystemId);
	len += sprintf(page+len, "PCI_Revision_ID\t\t0x%02x\n",
		pDevice->PciRevId);
	len += sprintf(page+len, "PCI_Slot\t\t%d\n",
		PCI_SLOT(pUmDevice->pdev->devfn));
	len += sprintf(page+len, "PCI_Bus\t\t\t%d\n",
		pUmDevice->pdev->bus->number);
	len += sprintf(page+len, "Memory\t\t\t0x%lx\n", pUmDevice->dev->base_addr);
	len += sprintf(page+len, "IRQ\t\t\t%d\n", dev->irq);
	len += sprintf(page+len, "System_Device_Name\t%s\n", dev->name);
	len += sprintf(page+len, "Current_HWaddr\t\t%02x:%02x:%02x:%02x:%02x:%02x\n",
		dev->dev_addr[0], dev->dev_addr[1], dev->dev_addr[2],
		dev->dev_addr[3], dev->dev_addr[4], dev->dev_addr[5]);
	len += sprintf(page+len,
		"Permanent_HWaddr\t%02x:%02x:%02x:%02x:%02x:%02x\n",
		pDevice->NodeAddress[0], pDevice->NodeAddress[1],
		pDevice->NodeAddress[2], pDevice->NodeAddress[3],
		pDevice->NodeAddress[4], pDevice->NodeAddress[5]);
	len += sprintf(page+len, "Part_Number\t\t%s\n\n", pDevice->PartNo);

	len += sprintf(page+len, "Link\t\t\t%s\n", 
		(pUmDevice->opened == 0) ? "unknown" :
    		((pDevice->LinkStatus == LM_STATUS_LINK_ACTIVE) ? "up" :
		"down"));
	len += sprintf(page+len, "Speed\t\t\t%s\n", 
    		(pDevice->LinkStatus == LM_STATUS_LINK_DOWN) ? "N/A" :
    		((pDevice->LineSpeed == LM_LINE_SPEED_1000MBPS) ? "1000" :
    		((pDevice->LineSpeed == LM_LINE_SPEED_100MBPS) ? "100" :
    		(pDevice->LineSpeed == LM_LINE_SPEED_10MBPS) ? "10" : "N/A")));
	len += sprintf(page+len, "Duplex\t\t\t%s\n", 
    		(pDevice->LinkStatus == LM_STATUS_LINK_DOWN) ? "N/A" :
		((pDevice->DuplexMode == LM_DUPLEX_MODE_FULL) ? "full" :
			"half"));
	len += sprintf(page+len, "Flow_Control\t\t%s\n", 
    		(pDevice->LinkStatus == LM_STATUS_LINK_DOWN) ? "N/A" :
		((pDevice->FlowControl == LM_FLOW_CONTROL_NONE) ? "off" :
		(((pDevice->FlowControl & LM_FLOW_CONTROL_RX_TX_PAUSE) ==
			LM_FLOW_CONTROL_RX_TX_PAUSE) ? "receive/transmit" :
		(pDevice->FlowControl & LM_FLOW_CONTROL_RECEIVE_PAUSE) ?
			"receive" : "transmit")));
	len += sprintf(page+len, "State\t\t\t%s\n", 
    		(dev->flags & IFF_UP) ? "up" : "down");
	len += sprintf(page+len, "MTU_Size\t\t%d\n\n", dev->mtu);
	len += sprintf(page+len, "Rx_Packets\t\t%u\n", 
			((pStats == 0) ? 0 :
			pStats->ifHCInUcastPkts.Low +
			pStats->ifHCInMulticastPkts.Low +
			pStats->ifHCInBroadcastPkts.Low));
	len += sprintf(page+len, "Tx_Packets\t\t%u\n",
		((pStats == 0) ? 0 :
		pStats->COSIfHCOutPkts[0].Low));
	len += sprintf(page+len, "Rx_Bytes\t\t%u\n",
		((pStats == 0) ? 0 :
		pStats->ifHCInOctets.Low));
	len += sprintf(page+len, "Tx_Bytes\t\t%u\n",
		((pStats == 0) ? 0 :
		pStats->ifHCOutOctets.Low));
	len += sprintf(page+len, "Rx_Errors\t\t%u\n",
		((pStats == 0) ? 0 :
		pStats->ifInErrors.Low));
	len += sprintf(page+len, "Tx_Errors\t\t%u\n",
		((pStats == 0) ? 0 :
		pStats->ifOutErrors.Low));
	len += sprintf(page+len, "Rx_CRC_Errors\t\t%u\n",
		((pStats == 0) ? 0 :
		pStats->dot3StatsFCSErrors.Low));
	len += sprintf(page+len, "Tx_Carrier_Errors\t%u\n",
		((pStats == 0) ? 0 :
		pStats->dot3StatsCarrierSenseErrors.Low));
	len += sprintf(page+len, "Tx_Abort_Excess_Coll\t%u\n",
		((pStats == 0) ? 0 :
    		pStats->dot3StatsExcessiveCollisions.Low));
	len += sprintf(page+len, "Tx_Abort_Late_Coll\t%u\n",
		((pStats == 0) ? 0 :
    		pStats->dot3StatsLateCollisions.Low));
	len += sprintf(page+len, "Tx_Deferred_Ok\t\t%u\n",
		((pStats == 0) ? 0 :
    		pStats->dot3StatsDeferredTransmissions.Low));
	len += sprintf(page+len, "Tx_Single_Coll_Ok\t%u\n",
		((pStats == 0) ? 0 :
    		pStats->dot3StatsSingleCollisionFrames.Low));
	len += sprintf(page+len, "Tx_Multi_Coll_Ok\t%u\n",
		((pStats == 0) ? 0 :
    		pStats->dot3StatsMultipleCollisionFrames.Low));
	len += sprintf(page+len, "Rx_Short_Length_Errors\t%u\n",
		((pStats == 0) ? 0 :
		pStats->etherStatsUndersizePkts.Low));
	len += sprintf(page+len, "Rx_Long_Length_Errors\t%u\n",
		((pStats == 0) ? 0 :
		pStats->dot3StatsFramesTooLong.Low));
	len += sprintf(page+len, "Rx_Align_Errors\t\t%u\n",
		((pStats == 0) ? 0 :
		pStats->dot3StatsAlignmentErrors.Low));
	len += sprintf(page+len, "Tx_MAC_Errors\t\t%u\n",
		((pStats == 0) ? 0 :
		pStats->dot3StatsInternalMacTransmitErrors.Low));
	len += sprintf(page+len, "Rx_MAC_Errors\t\t%u\n\n",
		((pStats == 0) ? 0 :
		pStats->ifInErrors.Low));

	len += sprintf(page+len, "Tx_Checksum\t\t%s\n",
		((pDevice->TaskToOffload & LM_TASK_OFFLOAD_TX_TCP_CHECKSUM) ?
		"ON" : "OFF"));
	len += sprintf(page+len, "Rx_Checksum\t\t%s\n",
		((pDevice->TaskToOffload & LM_TASK_OFFLOAD_RX_TCP_CHECKSUM) ?
		"ON" : "OFF"));
	len += sprintf(page+len, "Scatter_Gather\t\t%s\n",
#if (LINUX_VERSION_CODE >= 0x20400)
		((dev->features & NETIF_F_SG) ? "ON" : "OFF"));
#else
		"OFF");
#endif
	len += sprintf(page+len, "Tx_Desc_Count\t\t%u\n",
		pDevice->TxPacketDescCnt);
	len += sprintf(page+len, "Rx_Desc_Count\t\t%u\n",
		pDevice->RxStdDescCnt);
	len += sprintf(page+len, "Rx_Jumbo_Desc_Count\t%u\n",
		pDevice->RxJumboDescCnt);
	len += sprintf(page+len, "Rx_Adaptive_Coalescing\t%s\n",
		(pUmDevice->rx_adaptive_coalesce ? "ON" : "OFF"));
	len += sprintf(page+len, "Rx_Coalescing_Ticks\t%u\n",
		pUmDevice->rx_curr_coalesce_ticks);
	len += sprintf(page+len, "Rx_Coalesced_Frames\t%u\n",
		pUmDevice->rx_curr_coalesce_frames);
	len += sprintf(page+len, "Tx_Coalescing_Ticks\t%u\n",
		pDevice->TxCoalescingTicks);
	len += sprintf(page+len, "Tx_Coalesced_Frames\t%u\n",
		pDevice->TxMaxCoalescedFrames);
	len += sprintf(page+len, "Stats_Coalescing_Ticks\t%u\n",
		pDevice->StatsCoalescingTicks);
	len += sprintf(page+len, "Wake_On_LAN\t\t%s\n",
        	((pDevice->WakeUpMode & LM_WAKE_UP_MODE_MAGIC_PACKET) ?
		"ON" : "OFF"));

	*eof = 1;
	return len;
}

int
bcm5700_proc_create_dev(struct net_device *dev)
{
	PUM_DEVICE_BLOCK pUmDevice = (PUM_DEVICE_BLOCK) dev->priv;

	if (!bcm5700_procfs_dir)
		return -1;

	sprintf(pUmDevice->pfs_name, "%s.info", dev->name);
	pUmDevice->pfs_entry = create_proc_entry(pUmDevice->pfs_name,
		S_IFREG, bcm5700_procfs_dir);
	if (pUmDevice->pfs_entry == 0)
		return -1;
	pUmDevice->pfs_entry->read_proc = bcm5700_read_pfs;
	pUmDevice->pfs_entry->data = dev;
	return 0;
}
int
bcm5700_proc_remove_dev(struct net_device *dev)
{
	PUM_DEVICE_BLOCK pUmDevice = (PUM_DEVICE_BLOCK) dev->priv;

	remove_proc_entry(pUmDevice->pfs_name, bcm5700_procfs_dir);
	return 0;
}

#endif
