/* vi: set sw=4 ts=4 autoindent: */
/*
 * Mini syslogd implementation for busybox
 *
 * Copyright (C) 1999,2000 by Lineo, inc. and Erik Andersen
 * Copyright (C) 1999,2000,2001 by Erik Andersen <andersee@debian.org>
 *
 * Copyright (C) 2000 by Karl M. Hegbloom <karlheg@debian.org>
 *
 * "circular buffer" Copyright (C) 2001 by Gennady Feldman <gfeldman@cachier.com>
 *
 * Maintainer: Gennady Feldman <gena01@cachier.com> as of Mar 12, 2001
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <stdio.h>
/* PROBLEM with vdprintf not being defined. */
/* Write formatted output to a file descriptor.  */
extern int vdprintf (int __fd, __const char *__restrict __fmt,
		    __gnuc_va_list __arg)
		    __THROW __attribute__ ((__format__ (__printf__, 2, 0)));

#include <stdlib.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <paths.h>
#include <signal.h>
#include <stdarg.h>
#include <time.h>
#include <string.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/un.h>
#include <sys/param.h>

#include "sysinfo.h"
#include "open_debug.h"

//---prototype---
extern unsigned long get_ip(int type);
extern int device_open(char *device, int mode);


#ifndef SOCK_DGRAM
#	define SOCK_DGRAM 1
#endif

#ifndef SOCK_STREAM
#	define SOCK_STREAM 2
#endif

/* SYSLOG_NAMES defined to pull some extra junk from syslog.h */
#define SYSLOG_NAMES
#include <sys/syslog.h>
#include <sys/uio.h>

/* Path for the file where all log messages are written */
#define __LOG_FILE "/var/log/messages"
#define __LOG_FTP_FILE "/var/log/vsftpd.log"
#define PID_FILE	"/var/run/syslogd.pid"
#define LOG_TEMP	"/var/log/msg2"
#define LOG_FTP_TEMP	"/var/log/ftpmsg2"
#define CHG_LOG_LINE	400	//Ƥj CHG_LOG_LINE ɭnAgJtemp file
#define	MAX_LOG_LINE	600	//Ƶ MAX_LOG_LINE ɭn line_count ksHΧɦWʧ@

/* Path to the unix socket */
static char lfile[BUFSIZ];

static char *logFilePath = __LOG_FILE;

/* interval between marks in seconds */
static int MarkInterval = 20 * 60;

/* localhost's name */
static char LocalHostName[32];

//-------added by Lily
#define FALSE	0
#define TRUE	1
#define RESERVE_BB_BUFFER(a, x)	a = malloc(x)
#define RELEASE_BB_BUFFER(a)	free(a)

char *tmpbuf;
const char *applet_name = "syslogd";

#undef BB_FEATURE_IPC_SYSLOG
#define BB_FEATURE_REMOTE_LOG
//-------added by Lily----END

#ifdef BB_FEATURE_REMOTE_LOG
#include <netinet/in.h>
/* udp socket for logging to remote host */
static int remotefd = -1;
/* where do we log? */
static char *RemoteHost=NULL;
/* what port to log to? */
static int RemotePort = 514;
/* To remote log or not to remote log, that is the question. */
static int doRemoteLog = FALSE;
static int local_logging = FALSE;
#endif

#ifdef BB_FEATURE_IPC_SYSLOG
#include <sys/ipc.h>
#include <sys/sem.h>
#include <sys/shm.h>

/* our shared key */
static const long KEY_ID = 0x414e4547; /*"GENA"*/

// Semaphore operation structures
static struct shbuf_ds {
	int size;		// size of data written
	int head;		// start of message list
	int tail;		// end of message list
	char data[1];		// data/messages
} *buf = NULL;			// shared memory pointer

static struct sembuf SMwup[1] = {{1, -1, IPC_NOWAIT}}; // set SMwup
static struct sembuf SMwdn[3] = {{0, 0}, {1, 0}, {1, +1}}; // set SMwdn

static int 	shmid = -1;	// ipc shared memory id
static int 	s_semid = -1;	// ipc semaphore id
int	data_size = 16000; // data size
int	shm_size = 16000 + sizeof(*buf); // our buffer size
static int circular_logging = FALSE;


/*
 * sem_up - up()'s a semaphore.
 */
static inline void sem_up(int semid)
{
	if ( semop(semid, SMwup, 1) == -1 )
		perror_msg_and_die("semop[SMwup]");
}

/*
 * sem_down - down()'s a semaphore
 */
static inline void sem_down(int semid)
{
	if ( semop(semid, SMwdn, 3) == -1 )
		perror_msg_and_die("semop[SMwdn]");
}


void ipcsyslog_cleanup(void){
	printf("Exiting Syslogd!\n");
	if (shmid != -1)
		shmdt(buf);

	if (shmid != -1)
		shmctl(shmid, IPC_RMID, NULL);
	if (s_semid != -1)
		semctl(s_semid, 0, IPC_RMID, 0);
}

void ipcsyslog_init(void)
{
	if (buf == NULL)
	{
		if ((shmid = shmget(KEY_ID, shm_size, IPC_CREAT | 1023)) == -1)
			perror_msg_and_die("shmget");

		if ((buf = shmat(shmid, NULL, 0)) == NULL)
    		perror_msg_and_die("shmat");


		buf->size=data_size;
		buf->head=buf->tail=0;

		// we'll trust the OS to set initial semval to 0 (let's hope)
		if ((s_semid = semget(KEY_ID, 2, IPC_CREAT | IPC_EXCL | 1023)) == -1)
		{
			if (errno == EEXIST)
			{
				if ((s_semid = semget(KEY_ID, 2, 0)) == -1)
					perror_msg_and_die("semget");
			}
			else
				perror_msg_and_die("semget");
		}
	}
	else
	{
		Dprintf("Buffer already allocated just grab the semaphore?");
	}
}

/* write message to buffer */
void circ_message(const char *msg)
{
	int l = strlen(msg)+1; /* count the whole message w/ '\0' included */

	sem_down(s_semid);

	/*
	 * Circular Buffer Algorithm:
	 * --------------------------
	 *
	 * Start-off w/ empty buffer of specific size SHM_SIZ
	 * Start filling it up w/ messages. I use '\0' as separator to break up messages.
	 * This is also very handy since we can do printf on message.
	 *
	 * Once the buffer is full we need to get rid of the first message in buffer and
	 * insert the new message. (Note: if the message being added is >1 message then
	 * we will need to "remove" >1 old message from the buffer). The way this is done
	 * is the following:
	 *	When we reach the end of the buffer we set a mark and start from the beginning.
	 *	Now what about the beginning and end of the buffer? Well we have the "head"
	 *	index/pointer which is the starting point for the messages and we have "tail"
	 *	index/pointer which is the ending point for the messages. When we "display" the
	 *	messages we start from the beginning and continue until we reach "tail". If we
	 *	reach end of buffer, then we just start from the beginning (offset 0). "head" and
	 *	"tail" are actually offsets from the beginning of the buffer.
	 *
	 * Note: This algorithm uses Linux IPC mechanism w/ shared memory and semaphores to provide
	 * 	 a threasafe way of handling shared memory operations.
	 */
	if ( (buf->tail + l) < buf->size )
	{
		/* before we append the message we need to check the HEAD so that we won't
		   overwrite any of the message that we still need and adjust HEAD to point
		   to the next message! */
		if ( buf->tail < buf->head)
		{
			if ( (buf->tail + l) >= buf->head )
			{
			  /* we need to move the HEAD to point to the next message
			   * Theoretically we have enough room to add the whole message to the
			   * buffer, because of the first outer IF statement, so we don't have
			   * to worry about overflows here!
			   */
			   int k= buf->tail + l - buf->head; /* we need to know how many bytes
			   					we are overwriting to make
								enough room */
				char *c=memchr(buf->data+buf->head + k,'\0',buf->size - (buf->head + k));
				if (c != NULL) /* do a sanity check just in case! */
				{
					buf->head = c - buf->data + 1; /* we need to convert pointer to
						offset + skip the '\0' since
						we need to point to the beginning
						of the next message */
					/* Note: HEAD is only used to "retrieve" messages, it's not used
						when writing messages into our buffer */
				}else /* show an error message to know we messed up? */
				{
			   		Dprintf("Weird! Can't find the terminator token??? \n");
			   		buf->head=0;
				}
			}
		} /* in other cases no overflows have been done yet, so we don't care! */

		/* we should be ok to append the message now */
		strncpy(buf->data + buf->tail,msg,l); /* append our message */
		buf->tail+=l; /* count full message w/ '\0' terminating char */
	}
	else
	{
		/* we need to break up the message and "circle" it around */
		char *c;
		int k = buf->tail + l - buf->size; /* count # of bytes we don't fit */
		
		/* We need to move HEAD! This is always the case since we are going
		 * to "circle" the message.
		 */
		c = memchr(buf->data + k ,'\0', buf->size - k);
		
		if (c != NULL) /* if we don't have '\0'??? weird!!! */
		{
			/* move head pointer*/
			buf->head=c-buf->data+1; 
			
			/* now write the first part of the message */			
			strncpy(buf->data + buf->tail, msg, l - k - 1);
			
			/* ALWAYS terminate end of buffer w/ '\0' */
			buf->data[buf->size-1]='\0'; 
			
			/* now write out the rest of the string to the beginning of the buffer */
			strcpy(buf->data, &msg[l-k-1]);

			/* we need to place the TAIL at the end of the message */
			buf->tail = k + 1;
		}
		else
		{
			Dprintf("Weird! Can't find the terminator token from the beginning??? \n");
			buf->head = buf->tail = 0; /* reset buffer, since it's probably corrupted */
		}
		
	}
	sem_up(s_semid);
}
#endif

//added by Lily
void
perror_msg_and_die(char * str)
{
	printf("%s\n", str);
	exit(1);
}

/* Note: There is also a function called "message()" in init.c */
/* Print a message to the log file. */
//static void message (int pri, char *fmt, ...) __attribute__ ((format (printf, 1, 2)));
static void message (int pri, char *fmt, ...)
{
	int fd;
	struct flock fl;
	va_list arguments;
	static int line_syslog_count=0;
	static int line_ftp_count=0;
	int *line_count;
	char *logpath, *logtmppath;

	fl.l_whence = SEEK_SET;
	fl.l_start  = 0;
	fl.l_len    = 1;

	if(pri == LOG_FTP)	//added by Lily for vsftpd log
	{
		logpath = __LOG_FTP_FILE;
		logtmppath = LOG_FTP_TEMP;
		line_count = &line_ftp_count;
	}
	else
	{
		logpath = logFilePath;
		logtmppath = LOG_TEMP;
		line_count = &line_syslog_count;
	}

#ifdef BB_FEATURE_IPC_SYSLOG
	if ((circular_logging == TRUE) && (buf != NULL))
	{
		char b[1024];
		va_start (arguments, fmt);
		vsnprintf (b, sizeof(b)-1, fmt, arguments);
		va_end (arguments);
		circ_message(b);
	}
	else
#endif
	if ((fd = device_open (logpath,
						   O_WRONLY | O_CREAT | O_NOCTTY | O_APPEND |
						   O_NONBLOCK)) >= 0) 
	{
		fl.l_type = F_WRLCK;
		fcntl (fd, F_SETLKW, &fl);
		va_start (arguments, fmt);
		vdprintf (fd, fmt, arguments);
		va_end (arguments);
		fl.l_type = F_UNLCK;
		fcntl (fd, F_SETLKW, &fl);
		close (fd);
		(*line_count)++;

		if(*line_count > CHG_LOG_LINE) //}lgJĤGɮ
		{	
			if ((fd = device_open (logtmppath,
						   O_WRONLY | O_CREAT | O_NOCTTY | O_APPEND |
						   O_NONBLOCK)) >= 0) 
			{
				fl.l_type = F_WRLCK;
				fcntl (fd, F_SETLKW, &fl);
				va_start (arguments, fmt);
				vdprintf (fd, fmt, arguments);
				va_end (arguments);
				
				fl.l_type = F_UNLCK;
				fcntl (fd, F_SETLKW, &fl);
				
				close (fd);
			}
		}

		if(*line_count >= MAX_LOG_LINE) 	//ɮ
		{
			unlink(logpath);
			rename(logtmppath, logpath);
			*line_count = (MAX_LOG_LINE - CHG_LOG_LINE);
		}
	} 
	else 
	{
		/* Always send console messages to /dev/console so people will see them. */
		if ((fd = device_open (_PATH_CONSOLE,
							   O_WRONLY | O_NOCTTY | O_NONBLOCK)) >= 0) 
		{
			va_start (arguments, fmt);
			vdprintf (fd, fmt, arguments);
			va_end (arguments);
			close (fd);
		} 
		else 
		{
			fprintf (stderr, "Bummer, can't print: ");
			va_start (arguments, fmt);
			vfprintf (stderr, fmt, arguments);
			fflush (stderr);
			va_end (arguments);
		}
	}
}

typedef struct _code 
{
	char *c_name;
	int c_val;
} CODE;

static void logMessage (int pri, char *msg)
{
	time_t now;
	char *timestamp;
	static char res[20] = "";
#if 0
	CODE *c_pri, *c_fac;

	if (pri != 0) 
	{
		for (c_fac = facilitynames;
			c_fac->c_name && !(c_fac->c_val == LOG_FAC(pri) << 3); c_fac++);
		for (c_pri = prioritynames;
			c_pri->c_name && !(c_pri->c_val == LOG_PRI(pri)); c_pri++);
		if (c_fac->c_name == NULL || c_pri->c_name == NULL)
			snprintf(res, sizeof(res), "<%d>", pri);
		else
			snprintf(res, sizeof(res), "%s.%s", c_fac->c_name, c_pri->c_name);
	}
#endif

	if (strlen(msg) < 16 || msg[3] != ' ' || msg[6] != ' ' ||
			msg[9] != ':' || msg[12] != ':' || msg[15] != ' ') 
	{
		time(&now);
		timestamp = ctime(&now) + 4;
		timestamp[15] = '\0';
	} 
	else 
	{
		timestamp = msg;
		timestamp[15] = '\0';
		msg += 16;
	}

#if	!defined(_NTP) && !defined(_HAVE_RTC)
	//SҰNTPUISNTP]w, tήɶiणT
	//ҥHsyslogbUIܤWNnjavascript覡
	//ܦOڭ̪sysloguO{b(system)uptime
	//UIܮ, ssisysloguptime, AഫP{bɶ(PC)t()
	sprintf(timestamp, "%lu", get_current_time());
#endif

	/* todo: supress duplicates */

#ifdef BB_FEATURE_REMOTE_LOG
	/* send message to remote logger */
	if ( -1 != remotefd)
	{
		static const int IOV_COUNT = 2;
		struct iovec iov[IOV_COUNT];
		struct iovec *v = iov;

		memset(&res, 0, sizeof(res));
		snprintf(res, sizeof(res), "<%d>", pri);
		v->iov_base = res ;
		v->iov_len = strlen(res);          
		v++;

		v->iov_base = msg;
		v->iov_len = strlen(msg);          
writev_retry:
		if ( -1 == writev(remotefd,iov, IOV_COUNT))
		{
			if (errno == EINTR) 
				goto writev_retry;
			/*error_msg_and_die("syslogd: cannot write to remote file handle on"
				"%s:%d",RemoteHost,RemotePort);*/
			Dprintf("syslogd: cannot write to remote file handle on" \
					"%s:%d errno=%d",RemoteHost,RemotePort,errno);
		}
	}
	if (local_logging == TRUE)
#endif
		/* now spew out the message to wherever it is supposed to go */
		//message("%s %s %s %s\n", timestamp, LocalHostName, res, msg);	//modified by Lily
		message(pri,"%s %s\n", timestamp, msg);
}

static void quit_signal(int sig)
{
	logMessage(LOG_SYSLOG | LOG_INFO, "syslogd: System log daemon exiting.");
	unlink(lfile);
#ifdef BB_FEATURE_IPC_SYSLOG
	ipcsyslog_cleanup();
#endif

	exit(TRUE);
}

static void domark(int sig)
{
	if (MarkInterval > 0) 
	{
		//logMessage(LOG_SYSLOG | LOG_INFO, "-- MARK --");	//marked by hsujp
		alarm(MarkInterval);
	}
}

/* This must be a #define, since when DODEBUG and BUFFERS_GO_IN_BSS are
 * enabled, we otherwise get a "storage size isn't constant error. */
#define BUFSIZE 1023
static int serveConnection (int conn)
{
	int    n_read, pri_check_ok;
	char *p;

	RESERVE_BB_BUFFER(tmpbuf, BUFSIZE + 1);
	p = tmpbuf;
	n_read = read (conn, tmpbuf, BUFSIZE );

	while (p < tmpbuf + n_read) 
	{
		int           pri = (LOG_USER | LOG_NOTICE);
		char          line[ BUFSIZE + 1 ];
		unsigned char c;

		char *q = line;

		tmpbuf[ n_read - 1 ] = '\0';

		pri_check_ok = 0;
		while (p && (c = *p) && q < &line[ sizeof (line) - 1 ]) 
		{
			if (c == '<' && !pri_check_ok) 
			{
				/* Parse the magic priority number. */
				pri = 0;
				while (isdigit (*(++p))) 
				{
					pri = 10 * pri + (*p - '0');
				}
				pri_check_ok = 1;

				if (pri & ~(LOG_FACMASK | LOG_PRIMASK))
				{
					pri = (LOG_USER | LOG_NOTICE);
				}
			} 
			else if (c == '\n') 
			{
				*q++ = ' ';
			} 
			else if (iscntrl (c) && (c < 0177)) 
			{
				*q++ = '^';
				*q++ = c ^ 0100;
			} 
			else 
			{
				*q++ = c;
			}
			p++;
		}
		*q = '\0';
		p++;
		/* Now log it */
		logMessage (pri, line);
	}
	RELEASE_BB_BUFFER (tmpbuf);
	return n_read;
}


#ifdef BB_FEATURE_REMOTE_LOG
static void init_RemoteLog (void)
{
	struct sockaddr_in remoteaddr;
	struct hostent *hostinfo;
	int len = sizeof(remoteaddr);

	memset(&remoteaddr, 0, len);

	remotefd = socket(AF_INET, SOCK_DGRAM, 0);

	if (remotefd < 0) 
	{
		perror_msg_and_die("syslogd: cannot create socket");
	}

	hostinfo = gethostbyname(RemoteHost);
	if(hostinfo == NULL)
	{
		close(remotefd);
		doRemoteLog = FALSE;
		logMessage (LOG_SYSLOG | LOG_INFO, "syslogd: syslogd cannot connect to remote host" );
		return;
	}

	remoteaddr.sin_family = AF_INET;
	remoteaddr.sin_addr = *(struct in_addr *) *hostinfo->h_addr_list;
	remoteaddr.sin_port = htons(RemotePort);

	/* 
		Since we are using UDP sockets, connect just sets the default host and port 
		for future operations
	*/
	if ( 0 != (connect(remotefd, (struct sockaddr *) &remoteaddr, len)))
	{
		//instead of killing self while connecting failed, we disable the remote log flag -- Lily
		//perror_msg_and_die("syslogd: cannot connect to remote host");
		doRemoteLog = FALSE;
		close(remotefd);
		logMessage (LOG_SYSLOG | LOG_INFO, "syslogd: syslogd cannot connect to remote host" );
	}
}
#endif

static void doSyslogd (void) __attribute__ ((noreturn));
static void doSyslogd (void)
{
	struct sockaddr_un sunx;
	socklen_t addrLength;
	FILE *fp;
	pid_t pid;
	char buf[20];

	int sock_fd;
	fd_set fds;

	//added by Lily: kill previous process, if existed
	if( (fp = fopen(PID_FILE,"r"))!=NULL )
	{
		fscanf(fp, "%d", &pid);
		sprintf(buf, "kill %d", pid);
		System(buf);
		fclose(fp);
	}

	//added by Lily: record the pid into PID_FILE
	if( (fp = fopen(PID_FILE,"w"))!=NULL )
	{
		pid = getpid();
		fprintf(fp,"%d",pid);
		fclose(fp);
	}
	else
	{
		printf("Can't write into %s\n", PID_FILE);
		exit(1);	//can't record pid
	}

	/* Set up signal handlers. */
	signal (SIGINT,  quit_signal);
	signal (SIGTERM, quit_signal);
	signal (SIGQUIT, quit_signal);
	signal (SIGHUP,  SIG_IGN);
	signal (SIGCHLD,  SIG_IGN);
#ifdef SIGCLD
	signal (SIGCLD,  SIG_IGN);
#endif
	signal (SIGALRM, domark);
	alarm (MarkInterval);

	/* Create the syslog file so realpath() can work. */
	if (realpath (_PATH_LOG, lfile) != NULL)
		unlink (lfile);

	memset (&sunx, 0, sizeof (sunx));
	sunx.sun_family = AF_UNIX;
	strncpy (sunx.sun_path, lfile, sizeof (sunx.sun_path));
	if ((sock_fd = socket (AF_UNIX, SOCK_STREAM, 0)) < 0)
		perror_msg_and_die ("Couldn't get file descriptor for socket " _PATH_LOG);

	addrLength = sizeof (sunx.sun_family) + strlen (sunx.sun_path);
	if ((bind (sock_fd, (struct sockaddr *) &sunx, addrLength)) || (listen (sock_fd, 5)))
		perror_msg_and_die ("Could not connect to socket " _PATH_LOG);

	if (chmod (lfile, 0666) < 0)
		perror_msg_and_die ("Could not set permission on " _PATH_LOG);

	FD_ZERO (&fds);
	FD_SET (sock_fd, &fds);

#ifdef BB_FEATURE_IPC_SYSLOG
	if (circular_logging == TRUE )
	{
	   ipcsyslog_init();
	}
#endif

#ifdef BB_FEATURE_REMOTE_LOG
	if (doRemoteLog == TRUE)
	{
		init_RemoteLog();
	}
#endif

	logMessage (LOG_SYSLOG | LOG_INFO, "syslogd: syslogd started" );

	for (;;) 
	{
		fd_set readfds;
		int    n_ready;
		int    fd;

		memcpy (&readfds, &fds, sizeof (fds));

		if ((n_ready = select (FD_SETSIZE, &readfds, NULL, NULL, NULL)) < 0) 
		{
			if (errno == EINTR) continue; /* alarm may have happened. */
			perror_msg_and_die ("select error");
		}

		for (fd = 0; (n_ready > 0) && (fd < FD_SETSIZE); fd++) 
		{
			if (FD_ISSET (fd, &readfds)) 
			{
				--n_ready;

				if (fd == sock_fd) 
				{
					int   conn;
					//printf("New Connection request.\n");
					if ((conn = accept (sock_fd, (struct sockaddr *) &sunx, &addrLength)) < 0) 
					{
						perror_msg_and_die ("accept error");
					}

					FD_SET(conn, &fds);
					//printf("conn: %i, set_size: %i\n",conn,FD_SETSIZE);
			  	} 
				else 
				{
					//printf("Serving connection: %i\n",fd);
					if ( serveConnection(fd) <= 0 ) 
					{
						close (fd);
						FD_CLR(fd, &fds);
					}
				} /* fd == sock_fd */
			}/* FD_ISSET() */
		}/* for */
	} /* for main loop */
}


extern int main(int argc, char **argv)
{
	int opt;
	int doFork = TRUE;
	char *p;
	int str_index=0;
	
	Dprintf("-------------------------------------------------------------------------------\r\n");
	Dprintf("amit_syslogd: Enter syslogd!\r\n");

	/* do normal option parsing */
	while ((opt = getopt(argc, argv, "m:nO:R:LC")) > 0) 
	{
		switch (opt) 
		{
			case 'm':
				MarkInterval = atoi(optarg) * 60;
				break;
			case 'n':
				doFork = FALSE;
				break;
			case 'O':
				logFilePath = strdup(optarg);
				break;
#ifdef BB_FEATURE_REMOTE_LOG
			case 'R':
				RemoteHost = strdup(optarg);
				RemoteHost++; //skip ":"
				while(*(RemoteHost+str_index)!=NULL)
				{
					if(*(RemoteHost+str_index)==":")
					{
						p=RemoteHost+str_index+1; 		//skip ":", and let p point to port number.
						RemotePort = atoi(p+1);
						*(RemoteHost+str_index) = '\0';	//cut the port segment.
						break;
					}
					str_index++;
				}
				Dprintf("RemoteHost=%s,RemotePort=%d\n", RemoteHost, RemotePort);
				doRemoteLog = TRUE;
				break;
			case 'L':
				local_logging = TRUE;
				break;
#endif
#ifdef BB_FEATURE_IPC_SYSLOG
			case 'C':
				circular_logging = TRUE;
				break;
#endif
			default:
				//show_usage();
				break;
		}
	}


#ifdef BB_FEATURE_REMOTE_LOG
	/* If they have not specified remote logging, then log locally */
	if (doRemoteLog == FALSE)
		local_logging = TRUE;
#endif

	/* Store away localhost's name before the fork */
	gethostname(LocalHostName, sizeof(LocalHostName));
	if ((p = strchr(LocalHostName, '.'))) {
		*p++ = '\0';
	}

	umask(0);

	if (doFork == TRUE) {
/* #if !defined(__UCLIBC__) || defined(__UCLIBC_HAS_MMU__) */
		if (daemon(0, 1) < 0)
			perror_msg_and_die("daemon");
/* #else */
/* 			error_msg_and_die("daemon not supported"); */
/* #endif */
	}
	doSyslogd();

	return EXIT_SUCCESS;
}

/*
Local Variables
c-file-style: "linux"
c-basic-offset: 4
tab-width: 4
End:
*/
