/* fdisk.c -- Partition table manipulator for Linux.
 *
 * Copyright (C) 1992  A. V. Le Blanc (LeBlanc@mcc.ac.uk)
 *
 * This program is free software.  You can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation: either version 1 or
 * (at your option) any later version.
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <ctype.h>
#include <setjmp.h>
#include <errno.h>
#include <getopt.h>
#include <sys/stat.h>

#include "nls.h"
#include "common.h"
#include "fdisk.h"


//#include "../defines.h"
#ifdef HAVE_blkpg_h
#include <linux/blkpg.h>
#endif


#define hex_val(c)	({ \
				char _c = (c); \
				isdigit(_c) ? _c - '0' : \
				tolower(_c) + 10 - 'a'; \
			})


#define LINE_LENGTH	800
#define pt_offset(b, n)	((struct partition *)((b) + 0x1be + \
				(n) * sizeof(struct partition)))
#define sector(s)	((s) & 0x3f)
#define cylinder(s, c)	((c) | (((s) & 0xc0) << 2))

#define hsc2sector(h,s,c) (sector(s) - 1 + sectors * \
				((h) + heads * cylinder(s,c)))
#define set_hsc(h,s,c,sector) { \
				s = sector % sectors + 1;	\
				sector /= sectors;	\
				h = sector % heads;	\
				sector /= heads;	\
				c = sector & 0xff;	\
				s |= (sector >> 2) & 0xc0;	\
			}

/* A valid partition table sector ends in 0x55 0xaa */
int
valid_part_table_flag(unsigned char *b) {
	return (b[510] == 0x55 && b[511] == 0xaa);
}

static void
write_part_table_flag(char *b) {
	b[510] = 0x55;
	b[511] = 0xaa;
}

/* start_sect and nr_sects are stored little endian on all machines */
/* moreover, they are not aligned correctly */
static void
store4_little_endian(unsigned char *cp, unsigned int val) {
	cp[0] = (val & 0xff);
	cp[1] = ((val >> 8) & 0xff);
	cp[2] = ((val >> 16) & 0xff);
	cp[3] = ((val >> 24) & 0xff);
}

static unsigned int
read4_little_endian(unsigned char *cp) {
	return (unsigned int)(cp[0]) + ((unsigned int)(cp[1]) << 8)
		+ ((unsigned int)(cp[2]) << 16)
		+ ((unsigned int)(cp[3]) << 24);
}

static void
set_start_sect(struct partition *p, unsigned int start_sect) {
	store4_little_endian(p->start4, start_sect);
}

unsigned int
get_start_sect(struct partition *p) {
	return read4_little_endian(p->start4);
}

static void
set_nr_sects(struct partition *p, unsigned int nr_sects) {
	store4_little_endian(p->size4, nr_sects);
}

unsigned int
get_nr_sects(struct partition *p) {
	return read4_little_endian(p->size4);
}

/* normally O_RDWR, -l option gives O_RDONLY */
static int type_open = O_RDWR;

/*
 * Raw disk label. For DOS-type partition tables the MBR,
 * with descriptions of the primary partitions.
 */
char MBRbuffer[MAX_SECTOR_SIZE];

/*
 * per partition table entry data
 *
 * The four primary partitions have the same sectorbuffer (MBRbuffer)
 * and have NULL ext_pointer.
 * Each logical partition table entry has two pointers, one for the
 * partition and one link to the next one.
 */
struct pte {
	struct partition *part_table;	/* points into sectorbuffer */
	struct partition *ext_pointer;	/* points into sectorbuffer */
	char changed;			/* boolean */
	unsigned int offset;		/* disk sector number */
	char *sectorbuffer;		/* disk sector contents */
} ptes[MAXIMUM_PARTS];

char	*disk_device,			/* must be specified */
	*line_ptr,			/* interactive input */
	line_buffer[LINE_LENGTH];

int	fd,				/* the disk */
	ext_index,			/* the prime extended partition */
	listing = 0,			/* no aborts for fdisk -l */
	nowarn = 0,			/* no warnings for fdisk -l/-s */
	dos_compatible_flag = ~0,
	dos_changed = 0,
	partitions = 4;			/* maximum partition + 1 */

unsigned int	user_cylinders, user_heads, user_sectors;
unsigned int	pt_heads, pt_sectors;
unsigned int	kern_heads, kern_sectors;

unsigned int	heads,
	sectors,
	cylinders,
	sector_size = DEFAULT_SECTOR_SIZE,
	user_set_sector_size = 0,
	sector_offset = 1,
	units_per_sector = 1,
	display_in_cyl_units = 1,
	extended_offset = 0;		/* offset of link pointers */

unsigned long long total_number_of_sectors;

#define dos_label (!sun_label && !sgi_label && !aix_label && !osf_label)
int     sun_label = 0;                  /* looking at sun disklabel */
int	sgi_label = 0;			/* looking at sgi disklabel */
int	aix_label = 0;			/* looking at aix disklabel */
int	osf_label = 0;			/* looking at OSF/1 disklabel */
int	possibly_osf_label = 0;

jmp_buf listingbuf;

void fatal(enum failure why) {
	char	error[LINE_LENGTH],
		*message = error;

	if (listing) {
		close(fd);
		longjmp(listingbuf, 1);
	}

	switch (why) {
		case unable_to_open:
			snprintf(error, sizeof(error),
				 _("Unable to open %s\n"), disk_device);
			break;
		case unable_to_read:
			snprintf(error, sizeof(error),
				 _("Unable to read %s\n"), disk_device);
			break;
		case unable_to_seek:
			snprintf(error, sizeof(error),
				_("Unable to seek on %s\n"),disk_device);
			break;
		case unable_to_write:
			snprintf(error, sizeof(error),
				_("Unable to write %s\n"), disk_device);
			break;
		default:
			message = _("Fatal error\n");
	}

	fputc('\n', stderr);
	fputs(message, stderr);
	exit(1);
}

static void
seek_sector(int fd, unsigned int secno) {
	long long offset = (long long) secno * sector_size;
	if (ext2_llseek(fd, offset, SEEK_SET) == (long long) -1)
		fatal(unable_to_seek);
}

static void
write_sector(int fd, unsigned int secno, char *buf) {
	seek_sector(fd, secno);
	if (write(fd, buf, sector_size) != sector_size)
		fatal(unable_to_write);
}

static unsigned int
get_partition_start(struct pte *pe) {
	return pe->offset + get_start_sect(pe->part_table);
}
void
set_all_unchanged(void) {
	int i;

	for (i = 0; i < MAXIMUM_PARTS; i++)
		ptes[i].changed = 0;
}

void
set_changed(int i) {
	ptes[i].changed = 1;
}

static int
is_garbage_table(void) {
	int i;

	for (i = 0; i < 4; i++) {
		struct pte *pe = &ptes[i];
		struct partition *p = pe->part_table;

		if (p->boot_ind != 0 && p->boot_ind != 0x80)
			return 1;
	}
	return 0;
}

static struct systypes *
get_sys_types(void) {
	return (i386_sys_types);
}

char *partition_type(unsigned char type)
{
	int i;
	struct systypes *types = get_sys_types();

	for (i=0; types[i].name; i++)
		if (types[i].type == type)
			return _(types[i].name);

	return NULL;
}
static int
is_cleared_partition(struct partition *p) {
	return !(!p || p->boot_ind || p->head || p->sector || p->cyl ||
		 p->sys_ind || p->end_head || p->end_sector || p->end_cyl ||
		 get_start_sect(p) || get_nr_sects(p));
}

static void
set_partition(int i, int doext, unsigned int start, unsigned int stop,
	      int sysid) {
	struct partition *p;
	unsigned int offset;

	p = ptes[i].part_table;
	offset = ptes[i].offset;
	p->boot_ind = 0;
	p->sys_ind = sysid;
	set_start_sect(p, start - offset);
	set_nr_sects(p, stop - start + 1);
	if (dos_compatible_flag && (start/(sectors*heads) > 1023))
		start = heads*sectors*1024 - 1;
	set_hsc(p->head, p->sector, p->cyl, start);
	if (dos_compatible_flag && (stop/(sectors*heads) > 1023))
		stop = heads*sectors*1024 - 1;
	set_hsc(p->end_head, p->end_sector, p->end_cyl, stop);
	ptes[i].changed = 1;
}

static int
test_c(char **m, char *mesg) {
	int val = 0;
	if (!*m)
		fprintf(stderr, _("You must set"));
	else {
		fprintf(stderr, " %s", *m);
		val = 1;
	}
	*m = mesg;
	return val;
}

static int
warn_geometry(void) {
	char *m = NULL;
	int prev = 0;

	if (sgi_label)	/* cannot set cylinders etc anyway */
		return 0;
	if (!heads)
		prev = test_c(&m, _("heads"));
	if (!sectors)
		prev = test_c(&m, _("sectors"));
	if (!cylinders)
		prev = test_c(&m, _("cylinders"));
	if (!m)
		return 0;
	fprintf(stderr,
		_("%s%s.\nYou can do this from the extra functions menu.\n"),
		prev ? _(" and ") : " ", m);
	return 1;
}

void update_units(void)
{
	int cyl_units = heads * sectors;

	if (display_in_cyl_units && cyl_units)
		units_per_sector = cyl_units;
	else
		units_per_sector = 1; 	/* in sectors */
}

static void
warn_cylinders(void) {
	if (dos_label && cylinders > 1024 && !nowarn){}
//		fprintf(stderr, _("\n"
//"The number of cylinders for this disk is set to %d.\n"
//"There is nothing wrong with that, but this is larger than 1024,\n"
//"and could in certain setups cause problems with:\n"
//"1) software that runs at boot time (e.g., old versions of LILO)\n"
//"2) booting and partitioning software from other OSs\n"
//"   (e.g., DOS FDISK, OS/2 FDISK)\n"),
//			cylinders);
}

static void
create_doslabel(void) {
	int i;

//	fprintf(stderr,
//	_("Building a new DOS disklabel. Changes will remain in memory only,\n"
//	  "until you decide to write them. After that, of course, the previous\n"
//	  "content won't be recoverable.\n\n"));
	aix_label = osf_label = possibly_osf_label = 0;
	partitions = 4;

	for (i = 510-64; i < 510; i++)
		MBRbuffer[i] = 0;
	write_part_table_flag(MBRbuffer);
	extended_offset = 0;
	set_all_unchanged();
	set_changed(0);
	get_boot(create_empty_dos);
}

#include <sys/utsname.h>
#define MAKE_VERSION(p,q,r)     (65536*(p) + 256*(q) + (r))

static int
linux_version_code(void) {
	static int kernel_version = 0;
        struct utsname my_utsname;
        int p, q, r;

        if (!kernel_version && uname(&my_utsname) == 0) {
                p = atoi(strtok(my_utsname.release, "."));
                q = atoi(strtok(NULL, "."));
                r = atoi(strtok(NULL, "."));
                kernel_version = MAKE_VERSION(p,q,r);
        }
        return kernel_version;
}

static void
get_sectorsize(int fd) {
#if defined(BLKSSZGET)
	if (!user_set_sector_size &&
	    linux_version_code() >= MAKE_VERSION(2,3,3)) {
		int arg;
		if (ioctl(fd, BLKSSZGET, &arg) == 0)
			sector_size = arg;
		if (sector_size != DEFAULT_SECTOR_SIZE)
			printf(_("Note: sector size is %d (not %d)\n"),
			       sector_size, DEFAULT_SECTOR_SIZE);
	}
#else
	/* maybe the user specified it; and otherwise we still
	   have the DEFAULT_SECTOR_SIZE default */
#endif
}

static void
get_kernel_geometry(int fd) {
#ifdef HDIO_GETGEO
	struct hd_geometry geometry;

	if (!ioctl(fd, HDIO_GETGEO, &geometry)) {
		kern_heads = geometry.heads;
		kern_sectors = geometry.sectors;
		/* never use geometry.cylinders - it is truncated */
	}
#endif
}

static int
is_probably_full_disk(char *name) {
#ifdef HDIO_GETGEO
	struct hd_geometry geometry;
	int fd, i = 0;

	fd = open(name, O_RDONLY);
	if (fd >= 0) {
		i = ioctl(fd, HDIO_GETGEO, &geometry);
		close(fd);
	}
	return (fd >= 0 && i == 0 && geometry.start == 0);
#else
	/* silly heuristic */
	while (*name)
		name++;
	return !isdigit(name[-1]);
#endif
}

static void
get_partition_table_geometry(void) {
	unsigned char *bufp = MBRbuffer;
	struct partition *p;
	int i, h, s, hh, ss;
	int first = 1;
	int bad = 0;

	if (!(valid_part_table_flag(bufp)))
		return;

	hh = ss = 0;
	for (i=0; i<4; i++) {
		p = pt_offset(bufp, i);
		if (p->sys_ind != 0) {
			h = p->end_head + 1;
			s = (p->end_sector & 077);
			if (first) {
				hh = h;
				ss = s;
				first = 0;
			} else if (hh != h || ss != s)
				bad = 1;
		}
	}

	if (!first && !bad) {
		pt_heads = hh;
		pt_sectors = ss;
	}
}

void
get_geometry(int fd, struct geom *g) {
	int sec_fac;
	unsigned long long llsectors, llcyls;

	get_sectorsize(fd);
	sec_fac = sector_size / 512;
	heads = cylinders = sectors = 0;
	kern_heads = kern_sectors = 0;
	pt_heads = pt_sectors = 0;

	get_kernel_geometry(fd);
	get_partition_table_geometry();

	heads = user_heads ? user_heads :
		pt_heads ? pt_heads :
		kern_heads ? kern_heads : 255;
	sectors = user_sectors ? user_sectors :
		pt_sectors ? pt_sectors :
		kern_sectors ? kern_sectors : 63;

	if (disksize(fd, &llsectors))
		llsectors = 0;

	total_number_of_sectors = llsectors;

	sector_offset = 1;
	if (dos_compatible_flag)
		sector_offset = sectors;

	llcyls = total_number_of_sectors / (heads * sectors * sec_fac);
	cylinders = llcyls;
	if (cylinders != llcyls)	/* truncated? */
		cylinders = ~0;
	if (!cylinders)
		cylinders = user_cylinders;

	if (g) {
		g->heads = heads;
		g->sectors = sectors;
		g->cylinders = cylinders;
	}
}

/*
 * Read MBR.  Returns:
 *   -1: no 0xaa55 flag present (possibly entire disk BSD)
 *    0: found or created label
 *    1: I/O error
 */
int
get_boot(enum action what) {
	int i;

	partitions = 4;
	ext_index = 0;
	extended_offset = 0;

	for (i = 0; i < 4; i++) {
		struct pte *pe = &ptes[i];

		pe->part_table = pt_offset(MBRbuffer, i);
		pe->ext_pointer = NULL;
		pe->offset = 0;
		pe->sectorbuffer = MBRbuffer;
		pe->changed = (what == create_empty_dos);
	}

	memset(MBRbuffer, 0, 512);

	if (what == create_empty_dos)
		goto got_dos_table;		/* skip reading disk */

	if ((fd = open(disk_device, type_open)) < 0) {
	    if ((fd = open(disk_device, O_RDONLY)) < 0) {
		if (what == try_only)
		    return 1;
		fatal(unable_to_open);
	    } else
		printf(_("You will not be able to write "
			 "the partition table.\n"));
	}

	if (512 != read(fd, MBRbuffer, 512)) {
		if (what == try_only)
			return 1;
		fatal(unable_to_read);
	}

	get_geometry(fd, NULL);

	update_units();


got_dos_table:

	if (!valid_part_table_flag(MBRbuffer)) {
		switch(what) {
		case fdisk:
//			fprintf(stderr,
//				_("Device contains neither a valid DOS "
//				  "partition table, nor Sun, SGI or OSF "
//				  "disklabel\n"));
			create_doslabel();
			return 0;
		case require:
			return -1;
		case try_only:
		        return -1;
		case create_empty_dos:
			break;
		default:
//			fprintf(stderr, _("Internal error\n"));
			exit(1);
		}
	}

	warn_cylinders();
	warn_geometry();


	for (i = 3; i < partitions; i++) {
		struct pte *pe = &ptes[i];

		if (!valid_part_table_flag(pe->sectorbuffer)) {
//			fprintf(stderr,
//				_("Warning: invalid flag 0x%04x of partition "
//				"table %d will be corrected by w(rite)\n"),
//				part_table_flag(pe->sectorbuffer), i + 1);
			pe->changed = 1;
		}
	}

	return 0;
}

char * const
str_units(int n) {	/* n==1: use singular */
	if (n == 1)
		return display_in_cyl_units ? _("cylinder") : _("sector");
	else
		return display_in_cyl_units ? _("cylinders") : _("sectors");
}

void change_units(void)
{
	display_in_cyl_units = !display_in_cyl_units;
	update_units();
	printf(_("Changing display/entry units to %s\n"),
		str_units(PLURAL));
}

static void
toggle_active(int i) {
	struct pte *pe = &ptes[i];
	struct partition *p = pe->part_table;

	if (IS_EXTENDED (p->sys_ind) && !p->boot_ind)
		fprintf(stderr,
			_("WARNING: Partition %d is an extended partition\n"),
			i + 1);
	p->boot_ind = (p->boot_ind ? 0 : ACTIVE_FLAG);
	pe->changed = 1;
}

/* check_consistency() and long2chs() added Sat Mar 6 12:28:16 1993,
 * faith@cs.unc.edu, based on code fragments from pfdisk by Gordon W. Ross,
 * Jan.  1990 (version 1.2.1 by Gordon W. Ross Aug. 1990; Modified by S.
 * Lubkin Oct.  1991). */

static void
long2chs(ulong ls, unsigned int *c, unsigned int *h, unsigned int *s) {
	int spc = heads * sectors;

	*c = ls / spc;
	ls = ls % spc;
	*h = ls / sectors;
	*s = ls % sectors + 1;	/* sectors count from 1 */
}

static void check_consistency(struct partition *p, int partition) {
	unsigned int pbc, pbh, pbs;	/* physical beginning c, h, s */
	unsigned int pec, peh, pes;	/* physical ending c, h, s */
	unsigned int lbc, lbh, lbs;	/* logical beginning c, h, s */
	unsigned int lec, leh, les;	/* logical ending c, h, s */

	if (!heads || !sectors || (partition >= 4))
		return;		/* do not check extended partitions */

/* physical beginning c, h, s */
	pbc = (p->cyl & 0xff) | ((p->sector << 2) & 0x300);
	pbh = p->head;
	pbs = p->sector & 0x3f;

/* physical ending c, h, s */
	pec = (p->end_cyl & 0xff) | ((p->end_sector << 2) & 0x300);
	peh = p->end_head;
	pes = p->end_sector & 0x3f;

/* compute logical beginning (c, h, s) */
	long2chs(get_start_sect(p), &lbc, &lbh, &lbs);

/* compute logical ending (c, h, s) */
	long2chs(get_start_sect(p) + get_nr_sects(p) - 1, &lec, &leh, &les);

/* Same physical / logical beginning? */
	if (cylinders <= 1024 && (pbc != lbc || pbh != lbh || pbs != lbs)) {
		printf(_("Partition %d has different physical/logical "
			"beginnings (non-Linux?):\n"), partition + 1);
		printf(_("     phys=(%d, %d, %d) "), pbc, pbh, pbs);
		printf(_("logical=(%d, %d, %d)\n"),lbc, lbh, lbs);
	}

/* Same physical / logical ending? */
	if (cylinders <= 1024 && (pec != lec || peh != leh || pes != les)) {
		printf(_("Partition %d has different physical/logical "
			"endings:\n"), partition + 1);
		printf(_("     phys=(%d, %d, %d) "), pec, peh, pes);
		printf(_("logical=(%d, %d, %d)\n"),lec, leh, les);
	}

#if 0
/* Beginning on cylinder boundary? */
	if (pbh != !pbc || pbs != 1) {
		printf(_("Partition %i does not start on cylinder "
			"boundary:\n"), partition + 1);
		printf(_("     phys=(%d, %d, %d) "), pbc, pbh, pbs);
		printf(_("should be (%d, %d, 1)\n"), pbc, !pbc);
	}
#endif

/* Ending on cylinder boundary? */
	if (peh != (heads - 1) || pes != sectors) {
		printf(_("Partition %i does not end on cylinder boundary.\n"),
			partition + 1);
#if 0
		printf(_("     phys=(%d, %d, %d) "), pec, peh, pes);
		printf(_("should be (%d, %d, %d)\n"),
		pec, heads - 1, sectors);
#endif
	}
}

static void
list_disk_geometry(void) {
	long long bytes = (total_number_of_sectors << 9);
	long megabytes = bytes/1000000;

	if (megabytes < 10000)
		printf(_("\nDisk %s: %ld MB, %lld bytes\n"),
		       disk_device, megabytes, bytes);
	else
		printf(_("\nDisk %s: %ld.%ld GB, %lld bytes\n"),
		       disk_device, megabytes/1000, (megabytes/100)%10, bytes);
	printf(_("%d heads, %d sectors/track, %d cylinders"),
	       heads, sectors, cylinders);
	if (units_per_sector == 1)
		printf(_(", total %llu sectors"),
		       total_number_of_sectors / (sector_size/512));
	printf("\n");
	printf(_("Units = %s of %d * %d = %d bytes\n\n"),
	       str_units(PLURAL),
	       units_per_sector, sector_size, units_per_sector * sector_size);
}

/*
 * Check whether partition entries are ordered by their starting positions.
 * Return 0 if OK. Return i if partition i should have been earlier.
 * Two separate checks: primary and logical partitions.
 */
static int
wrong_p_order(int *prev) {
	struct pte *pe;
	struct partition *p;
	unsigned int last_p_start_pos = 0, p_start_pos;
	int i, last_i = 0;

	for (i = 0 ; i < partitions; i++) {
		if (i == 4) {
			last_i = 4;
			last_p_start_pos = 0;
		}
		pe = &ptes[i];
		if ((p = pe->part_table)->sys_ind) {
			p_start_pos = get_partition_start(pe);

			if (last_p_start_pos > p_start_pos) {
				if (prev)
					*prev = last_i;
				return i;
			}

			last_p_start_pos = p_start_pos;
			last_i = i;
		}
	}
	return 0;
}

static void
list_table(int xtra) {
	struct partition *p;
	char *type;
	int i, w;

	list_disk_geometry();


	if (is_garbage_table()) {
		printf(_("This doesn't look like a partition table\n"
			 "Probably you selected the wrong device.\n\n"));
	}
		

	/* Heuristic: we list partition 3 of /dev/foo as /dev/foo3,
	   but if the device name ends in a digit, say /dev/foo1,
	   then the partition is called /dev/foo1p3. */
	w = strlen(disk_device);
	if (w && isdigit(disk_device[w-1]))
		w++;
	if (w < 5)
		w = 5;

	printf(_("%*s Boot      Start         End      Blocks   Id  System\n"),
	       w+1, _("Device"));

	for (i = 0; i < partitions; i++) {
		struct pte *pe = &ptes[i];

		p = pe->part_table;
		if (p && !is_cleared_partition(p)) {
			unsigned int psects = get_nr_sects(p);
			unsigned int pblocks = psects;
			unsigned int podd = 0;

			if (sector_size < 1024) {
				pblocks /= (1024 / sector_size);
				podd = psects % (1024 / sector_size);
			}
			if (sector_size > 1024)
				pblocks *= (sector_size / 1024);
                        printf(
			    "%s  %c %11lu %11lu %11lu%c  %2x  %s\n",
			partname(disk_device, i+1, w+2),
/* boot flag */		!p->boot_ind ? ' ' : p->boot_ind == ACTIVE_FLAG
			? '*' : '?',
/* start */		(unsigned long) cround(get_partition_start(pe)),
/* end */		(unsigned long) cround(get_partition_start(pe) + psects
				- (psects ? 1 : 0)),
/* odd flag on end */	(unsigned long) pblocks, podd ? '+' : ' ',
/* type id */		p->sys_ind,
/* type name */		(type = partition_type(p->sys_ind)) ?
			type : _("Unknown"));
			check_consistency(p, i);
		}
	}

	/* Is partition table in disk order? It need not be, but... */
	/* partition table entries are not checked for correct order if this
	   is a sgi, sun or aix labeled disk... */
	if (dos_label && wrong_p_order(NULL)) {
		printf(_("\nPartition table entries are not in disk order\n"));
	}
}


static void
fill_bounds(unsigned int *first, unsigned int *last) {
	int i;
	struct pte *pe = &ptes[0];
	struct partition *p;

	for (i = 0; i < partitions; pe++,i++) {
		p = pe->part_table;
		if (!p->sys_ind || IS_EXTENDED (p->sys_ind)) {
			first[i] = 0xffffffff;
			last[i] = 0;
		} else {
			first[i] = get_partition_start(pe);
			last[i] = first[i] + get_nr_sects(p) - 1;
		}
	}
}


//	auto make partition 
// 		partition size is in MB type
//						Laroche
unsigned int auto_add_partition(int n, int sys, unsigned int start,long partition_size) 
{
	char mesg[256];		/* 48 does not suffice in Japanese */
	int i;//, read = 0;
	int absolute = 1048576;	//1024*1024
	struct partition *p = ptes[n].part_table;
//	struct partition *q = ptes[ext_index].part_table;
	long long llimit, stop = 0;
	//unsigned int start=0, 
	unsigned int limit=0, //temp,
		first[partitions], last[partitions];

	if (absolute) { // trans MB to byte
		unsigned long long bytes;
		unsigned long unit;

		bytes = (unsigned long long) partition_size * absolute;
		unit = sector_size * units_per_sector;
		bytes += unit/2;	/* round */
		bytes /= unit;
		stop = (long long) bytes ;
	}
	if (p && p->sys_ind) {
		printf(_("Partition %d is already defined.  Delete "
			 "it before re-adding it.\n"), n + 1);
		return 0;
	}
	fill_bounds(first, last);
	if (n < 4) {
		if (display_in_cyl_units || !total_number_of_sectors)
			llimit = heads * sectors * cylinders - 1;
		else
			llimit = total_number_of_sectors - 1;
		limit = llimit;
		if (limit != llimit)
			limit = 0x7fffffff;
	}
	if (display_in_cyl_units)
		for (i = 0; i < partitions; i++)
			first[i] = (cround(first[i]) - 1) * units_per_sector;

	snprintf(mesg, sizeof(mesg), _("First %s"), str_units(SINGULAR));

	for (i = 0; i < partitions; i++) {
		struct pte *pe = &ptes[i];

		if (start < pe->offset && limit >= pe->offset)
			limit = pe->offset - 1;
		if (start < first[i] && limit >= first[i])
			limit = first[i] - 1;
	}
	if (start > limit) {
		printf(_("No free sectors available\n"));
		return 0;
	}
	if (cround(start) == cround(limit)) {
		stop = limit;
	} else {
		stop += cround(start);
		if (display_in_cyl_units) {
			stop = stop * units_per_sector - 1;
			if (stop >limit)
				stop = limit;
		}
	}
//	printf("partition_size = %d limit = %d\n",(int)partition_size,limit);
	if ((int)partition_size == 0)	//fdisk a disk into only one partition 
	{
		stop = limit;
	}
//	printf("start = %d stop = %d\n",start,(unsigned int)stop);
	set_partition(n, 0, start,(unsigned int)stop, sys);
	return stop;
}

static void
write_table(void) {
	int i;

	if (dos_label) {
		for (i=0; i<3; i++)
			if (ptes[i].changed)
				ptes[3].changed = 1;
		for (i = 3; i < partitions; i++) {
			struct pte *pe = &ptes[i];

			if (pe->changed) {
				write_part_table_flag(pe->sectorbuffer);
				write_sector(fd, pe->offset, pe->sectorbuffer);
			}
		}
	}

	printf(_("The partition table has been altered!\n\n"));
	reread_partition_table(1);
}

void
reread_partition_table(int leave) {
	int error = 0;
	int i;

	printf(_("Calling ioctl() to re-read partition table.\n"));
	sync();
	sleep(2);
	if ((i = ioctl(fd, BLKRRPART)) != 0) {
                error = errno;
        } else {
                /* some kernel versions (1.2.x) seem to have trouble
                   rereading the partition table, but if asked to do it
		   twice, the second time works. - biro@yggdrasil.com */
                sync();
                sleep(2);
                if ((i = ioctl(fd, BLKRRPART)) != 0)
                        error = errno;
        }

	if (i) {
		printf(_("\nWARNING: Re-reading the partition table "
			 "failed with error %d: %s.\n"
			 "The kernel still uses the old table.\n"
			 "The new table will be used "
			 "at the next reboot.\n"),
			error, strerror(error));
	}

	if (dos_changed)
	    printf(
		_("\nWARNING: If you have created or modified any DOS 6.x\n"
		"partitions, please see the fdisk manual page for additional\n"
		"information.\n"));

	if (leave) {
		if (fsync(fd) || close(fd)) {
			fprintf(stderr, _("\nError closing file\n"));
			exit(1);
		}

		printf(_("Syncing disks.\n"));
		sync();
		sleep(4);		/* for sync() */
		exit(!!i);
	}
}

#define MAX_PER_LINE	16
static void
try(char *device, int user_specified) {
	int gb;

	disk_device = device;
	if (setjmp(listingbuf))
		return;
//	if (!user_specified)
//		if (is_ide_cdrom_or_tape(device))
//			return;
	if ((fd = open(disk_device, type_open)) >= 0) {
		gb = get_boot(try_only);
		if (gb > 0) { /* I/O error */
		} else if (gb < 0) { /* no DOS signature */
			list_disk_geometry();
	//		if (!aix_label && btrydev(device) < 0)
	//			fprintf(stderr,
	//				_("Disk %s doesn't contain a valid "
	//				  "partition table\n"), device);
		} else {
			list_table(0);
		}
		close(fd);
	} else {
		/* Ignore other errors, since we try IDE
		   and SCSI hard disks which may not be
		   installed on the system. */
		if (errno == EACCES) {
			fprintf(stderr, _("Cannot open %s\n"), device);
			return;
		}
	}
}

/*
 * for fdisk -l:
 * try all things in /proc/partitions that look like a full disk
 */
static void
tryprocpt(void) {
	FILE *procpt;
	char line[100], ptname[100], devname[120];
	int ma, mi, sz;

	procpt = fopen(PROC_PARTITIONS, "r");
	if (procpt == NULL) {
		fprintf(stderr, _("cannot open %s\n"), PROC_PARTITIONS);
		return;
	}

	while (fgets(line, sizeof(line), procpt)) {
		if (sscanf (line, " %d %d %d %[^\n ]",
			    &ma, &mi, &sz, ptname) != 4)
			continue;
		snprintf(devname, sizeof(devname), "/dev/%s", ptname);
		if (is_probably_full_disk(devname))
			try(devname, 0);
	}
	fclose(procpt);
}

int
main(int argc, char **argv) {
	int c;
	int optl = 0;	//opts = 0;
	int optp = 0; 	//add by Laroche in order to add partition auto
	int opta = 0;	//add by Laroche to make one partitions of total capability disk
	int optw = 0;	//add by Laroche to choose make a swap partition or not
	long auto_partition_size[4]={0};
	int sys_type = LINUX_NATIVE;
	setlocale(LC_ALL, "");
	bindtextdomain(PACKAGE, LOCALEDIR);
	textdomain(PACKAGE);

	/*
	 * Calls:
	 *  fdisk -v
	 *  fdisk -l [-b sectorsize] [-u] device ...
	 *  fdisk -s [partition] ...
	 *  fdisk [-b sectorsize] [-u] device
	 *
	 * Options -C, -H, -S set the geometry.
	 * 
	 */
	while ((c = getopt(argc, argv, "vV:aA:pP:lL:tT:wW")) != -1) {
		switch (c) 
		{
			case 'w':	//enable make a linux swap in specified size or 128MB 
			case 'W':	// in default
				optw = 1;
				if (isdigit(argv[optind][0]))
				{
					auto_partition_size[0] = atol(argv[optind]);
					optind++;
				}
				else
				{
					auto_partition_size[0] = 128;
				}
				printf("size[0] = %ld\n",auto_partition_size[0]);
				optp = 1;
				break;
			case 't':	//decide use FAT32 instead of  Linux_NATIVE
			case 'T':
				sys_type = WIN95_FAT32;
				break;
			case 'P':
			case 'p':
				while((optp < 4 )&&(optind < argc) && (isdigit(argv[optind][0])))
				{
					auto_partition_size[optp] = atol(argv[optind]);
					printf("size[%d] = %ld\n",optp,auto_partition_size[optp]);
					optp++;
					optind++;
				}
				break;
			case 'A':
			case 'a':
				opta = 1;	
				break;
			case 'L':
			case 'l':
				optl = 1;
				break;;
			case 'V':
			case 'v':
				printf("fdisk v" UTIL_LINUX_VERSION "\n");
				exit(0);
			default:
				exit(0);
		}
	}

	if (user_set_sector_size && argc-optind != 1)
		printf(_("Warning: the -b (set sector size) option should"
			 " be used with one specified device\n"));

	if (optl) {
		nowarn = 1;
		type_open = O_RDONLY;
		tryprocpt();
		exit(0);
	}
	if (argc-optind == 1)
		disk_device = argv[optind];
	else
		return -1;
	get_boot(fdisk);

//	auto add partition for MGB100 in AMIT 
//									add by Laroche
	if ((optp > 0 ) || (opta == 1))
	{
		int i = 0;
		unsigned int start=sectors;	//start index
		printf("Start fdisk\n");
		create_doslabel();
		if (optw == 1)
		{
			//start = auto_add_partition(0,LINUX_SWAP,start,auto_partition_size[0]);
			start = auto_add_partition(3,LINUX_SWAP,start,auto_partition_size[0]);
			start++;
		}
		
		if (opta == 1)
		{
			//auto_add_partition(optw,sys_type,start,0);
			auto_add_partition(0,sys_type,start,0);
		}
		else
		{
			//for (i = optw; i < optp+optw; i++)
			//{
			//	start = auto_add_partition(i,sys_type,start,auto_partition_size[i]);
			//	start++;
			//}
			for (i = 0; i < optp-optw; i++)
			{
				start = auto_add_partition(i,sys_type,start,auto_partition_size[i+optw]);
				start++;
			}
		}
		//toggle_active(optw);
		toggle_active(0);
		write_table();
		return 0;	
	}
	return 0;
}
