/*
 * linux/drivers/scsi/iteraid.h
 *
 * (C) Copyright 2002-2004 ITE Tech, inc.
 *
 * Nov 11, 2002	Mark Lu	file created.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * ITE IT8211 IDE Controller device driver header for Linux
 */

#ifndef _ITERAID_H_
#define _ITERAID_H_

#include <linux/version.h>
#include <linux/types.h>
#include <linux/kdev_t.h>

#define ITE_VENDOR_ID	0x1283			/* Vendor ID (ITE)	*/
#define ITE_DEVICE_ID	0x8211			/* Device IF (IT8211)	*/
#define MAX_ADAPTERS	2			/* Max Board supported	*/
#define MAX_DEVICES	(MAX_ADAPTERS * 4)	/* Max Dev supported	*/

#define TRUE		1
#define FALSE 		0

/*
 * Undef macros which may conflict
 */
#undef	START_STOP

/************************************************************************
 * Debugging macro
 ************************************************************************/
#ifdef ITE_DEBUG
#define dprintk(msg...) printk(msg)
#else
#define dprintk(msg...) do { } while(0)
#endif

/************************************************************************
 * Raid level definitions
 ************************************************************************/
#define RAID_LEVEL_0			0
#define RAID_LEVEL_1			1
#define RAID_LEVEL_10			2
#define RAID_LEVEL_JBOD			3
#define RAID_LEVEL_NORMAL		4
#define RAID_LEVEL_NODISK		5

/************************************************************************
 * Physical disk status definitions
 ************************************************************************/
#define DISK_KEY_OFF			0
#define DISK_OFF_LINE			1
#define DISK_ON_LINE			2
#define DISK_REBUILDING			3
#define DISK_PLUGGING			4
#define DISK_PLUGGING_OK		5

#define MaximumLBAOf28Bit		0x10000000

#define DisableChannel			1
#define EnableChannel			2

#define CABLE_40_PIN			0
#define CABLE_80_PIN			1

#define RaidActive			0
#define RaidInactive			1

#define IDE_CLOCK_66			0
#define IDE_CLOCK_50			1

#define USE_ULTRA_DMA			0
#define USE_MULTIWORD_DMA		1

/************************************************************************
 *
 ************************************************************************/
typedef struct _PHYSICAL_DISK_STATUS
{
 u8		ModelNumber[40];		/* Byte 00-39		*/
 u32		UserAddressableSectors_LOW;	/* Byte 40-43		*/
 u32		UserAddressableSectors_HIGH;	/* Byte 44-47		*/
 u8		MultiWordDMASupport;		/* Byte 48		*/
 u8		MultiWordDMAActive;		/* Byte 49		*/
 u8		UltraDMASupport;		/* Byte 50		*/
 u8		UltraDMAActive;			/* Byte 51		*/
 u8		RaidType;			/* Byte 52		*/
 u8		RaidNumber;			/* Byte 53		*/
 u8		SerialNumber[20];		/* Byte 54-73		*/
 u8		DiskStatus;			/* Byte 74		*/
 u8		DiskOriginalLocation;		/* Byte 75		*/
 u8		Cable80Pin;			/* Byte 76		*/
 u8		BootableDisk;			/* Byte 77		*/
 u8		StorageSize[8];			/* Byte 78-85		*/
 u8		Reserved[35];			/* Byte 86-120		*/
 u8		UpdateYear;			/* Byte 121		*/
 u8		UpdateMonth;			/* Byte 122		*/
 u8		UpdateDay;			/* Byte 123		*/
 u8		FirmwareVer;			/* Byte 124		*/
 u8		RebuildStatus;			/* Byte 125		*/
 u8		StripeSize;			/* Byte 126		*/
 u8		AutoRebuildEnable;		/* Byte 127		*/
} PHYSICAL_DISK_STATUS, *PPHYSICAL_DISK_STATUS;

/************************************************************************
 *
 ************************************************************************/
typedef struct _IT8211_SET_CHIP_STATUS_INFO
{
 u16		RaidType;			/* Word 129		*/
 u16		ContainingDisks;		/* Word 130		*/
 u16		UltraDmaTiming01;		/* Word 131		*/
 u16		UltraDmaTiming23;		/* Word 132		*/
 u16		UltraDmaTiming45;		/* Word 133		*/
 u16		UltraDmaTiming6;		/* Word 134		*/
 u16		MultiWordDmaTiming01;		/* Word 135		*/
 u16		UltraDmaTiming2;		/* Word 136		*/
 u16		PioTiming4;			/* Word 137		*/
 u16		AutoRebuildEnable;		/* Word 138		*/
 u16		IdeClkUDma01;			/* Word 139		*/
 u16		IdeClkUDma23;			/* Word 140		*/
 u16		IdeClkUDma45;			/* Word 141		*/
 u16		IdeClkUDma6;			/* Word 142		*/
 u16		IdeClkMDma01;			/* Word 143		*/
 u16		IdeClkMDma2;			/* Word 144		*/
 u16		IdeClkPio4;			/* Word 145		*/
 u16		StripeSize;			/* Word 146		*/
 u16		BootableDisk;			/* Word 147		*/
 u16		CheckHotSwapInterval;		/* Word 148		*/
 u16		TargetSourceDisk;		/* Word 149		*/
 u16		RebuildBlockSize;		/* Word 150		*/
 u16		ResetInterval1;			/* Word 151		*/
 u16		ResetInterval2;			/* Word 152		*/
 u16		RebuildRetryTimes;		/* Word 153		*/
 u16		NewlyCreated;			/* Word 154		*/
} IT8211_SET_CHIP_STATUS_INFO, *PIT8211_SET_CHIP_STATUS_INFO;

/************************************************************************
 * Serial number written to HDD (20 bytes)
 ************************************************************************/
typedef struct _RAID_SERIAL_NUMBER
{
 u16		Year;
 u8		Month;
 u8		Date;
 u8		Day;
 u8		Hour;
 u8		Minute;
 u8		Second;
 u8		MiniSec;
 u8		RaidType;
 u8		ContainingDisks;
 u8		DontCare[9];
} RAID_SERIAL_NUMBER, *PRAID_SERIAL_NUMBER;

/************************************************************************
 * Disk array create information
 *
 * Following items index definition
 * 0: Primary Master
 * 1: Secondary Master
 * 2: Primary Slave
 * 3: Secondary Slave
 ************************************************************************/
typedef struct _RAID_CREATE_INFO
{
 u8			DiskArrayId;
 RAID_SERIAL_NUMBER	SerialNumber;
 u8			ModelNumber[40];
 u16			RaidType;
 u16			ContainingDisks;
 u16			AutoRebuildEnable;
 u16			StripeSize;
 u16			BootableDisk;
 u16			TargetSourceDisk;
 u8			ErasePartition;
 u8			DMASupported[4];
 u8			UDMASupported[4];
 u8			AddressableSectors[4];
 u8			NewlyCreated;
 u8			Reserved;
} RAID_CREATE_INFO, *PRAID_CREATE_INFO;

/************************************************************************
 * Rebuild data structure
 ************************************************************************/
typedef struct _RAID_REBUILD_INFO
{
 u8		DiskArrayId;	/* Virtual device number (0-3)		*/
 u8		SrcDisk;	/* Source disk (0-3)			*/
 u8		DestDisk;	/* Destination disk (0-3)		*/
 u8		Resume;		/* 1: Resume the last time rebuild	*/
				/* 0: Rebuild from LBA 0		*/
 u8		Status;		/* Indicate the status of the current	*/
				/* rebuild command filled by drivers	*/
 u8		Reserved[3];	/* For aligement			*/
} RAID_REBUILD_INFO, *PRAID_REBUILD_INFO;

/************************************************************************
 * ATA transfer modes
 ************************************************************************/
#define PIO_DEFAULT			0x00
#define PIO_DEFAULT_IORDY_DISABLE	0x01
#define PIO_FLOW_CONTROL		0x08
#define SINGLEWORD_DMA			0x10
#define MULTIWORD_DMA			0x20
#define ULTRA_DMA			0x40

#define ITE_DRV_SIGNATURE		"ITE RAID CONTROLLER"
#define ITE_DRV_BYPASS			"ITE BYPASS MODE"

/************************************************************************
 * Extra IDE commands supported by Accusys
 ************************************************************************/
#define IDE_COMMAND_GET_CHIP_STATUS	0xFA
#define IDE_COMMAND_SET_CHIP_STATUS	0xFB
#define IDE_COMMAND_REBUILD		0xFC
#define IDE_COMMAND_REBUILD_STATUS	0xFD

#define REBUILD_ERR_WRONG_ARRAY_TYPE		0x01
#define REBUILD_ERR_DISK_TOO_SMALL		0x02
#define REBUILD_ERR_SRC_DISK_LOCATION_INCORRECT	0x03
#define REBUILD_ERR_SRC_DISK_OFFLINE		0x04
#define REBUILD_ERR_DEST_DISK_OFFLINE		0x05
#define REBUILD_ERR_DISK_BUSY			0x10	// defined by Andy, not accusys

/************************************************************************
 * ATA transfer modes
 ************************************************************************/
#define PIO_DEFAULT			0x00
#define PIO_DEFAULT_IORDY_DISABLE	0x01
#define PIO_FLOW_CONTROL		0x08
#define SINGLEWORD_DMA			0x10
#define MULTIWORD_DMA			0x20
#define ULTRA_DMA			0x40

/************************************************************************
 * IDE registers offset
 ************************************************************************/
#define IDE_NR_PORTS			10

#define IDE_DATA_OFFSET			0
#define IDE_ERROR_OFFSET		1
#define IDE_NSECTOR_OFFSET		2
#define IDE_LOCYL_OFFSET		3
#define IDE_MIDCYL_OFFSET		4
#define IDE_HCYL_OFFSET			5
#define IDE_SELECT_OFFSET		6
#define IDE_STATUS_OFFSET		7
#define IDE_CONTROL_OFFSET		8
#define IDE_IRQ_OFFSET			9

#define	IDE_FEATURE_OFFSET		IDE_ERROR_OFFSET
#define IDE_COMMAND_OFFSET		IDE_STATUS_OFFSET
#define IDE_ALTERNATE_OFFSET		IDE_CONTROL_OFFSET

/************************************************************************
 * ATAPI registers offset
 ************************************************************************/
#define ATAPI_DATA_OFFSET		0
#define ATAPI_ERROR_OFFSET		1
#define ATAPI_INTREASON_OFFSET		2
#define ATAPI_UNUSED1_OFFSET		3
#define ATAPI_LCYL_OFFSET		4
#define ATAPI_HCYL_OFFSET		5
#define ATAPI_SELECT_OFFSET		6
#define ATAPI_STATUS_OFFSET		7
#define ATAPI_CONTROL_OFFSET		8

#define ATAPI_COMMAND_OFFSET		ATAPI_STATUS_OFFSET
#define	ATAPI_FEATURE_OFFSET		ATAPI_ERROR_OFFSET

/************************************************************************
 * Following structures are according to SPC-3 (by Chanel)
 ************************************************************************/
typedef struct _SCSI_MODE_SENSE6
{
 u8		OperationCode;
 u8		Reserved1 : 3;
 u8		Dbd : 1;
 u8		Reserved2 : 4;
 u8		PageCode : 6;
 u8		Pc : 2;
 u8		SubpageCode;
 u8		AllocationLength;
 u8		Control;
} SCSI_MODE_SENSE6, *PSCSI_MODE_SENSE6;

typedef struct _SCSI_MODE_SENSE10
{
 u8		OperationCode;
 u8		Reserved1 : 3;
 u8		Dbd : 1;
 u8		LLBAA : 1;
 u8		Reserved2 : 3;
 u8		PageCode : 6;
 u8		Pc : 2;
 u8		SubpageCode;
 u8		Reserved3[3];
 u8		AllocationLengthMsb;
 u8		AllocationLengthLsb;
 u8		Control;
} SCSI_MODE_SENSE10, *PSCSI_MODE_SENSE10;

typedef struct _SCSI_MODE_SELECT6
{
 u8		OperationCode;
 u8		SPBit : 1;
 u8		Reserved1 : 3;
 u8		PFBit : 1;
 u8		Reserved2 : 3;
 u8		Reserved3[2];
 u8		ParameterListLength;
 u8		Control;
} SCSI_MODE_SELECT6, *PSCSI_MODE_SELECT6;

typedef struct _SCSI_MODE_SELECT10
{
 u8		OperationCode;
 u8		SPBit : 1;
 u8		Reserved1 : 3;
 u8		PFBit : 1;
 u8		Reserved2 : 3;
 u8		Reserved3[5];
 u8		ParameterListLengthMsb;
 u8		ParameterListLengthLsb;
 u8		Control;
} SCSI_MODE_SELECT10, *PSCSI_MODE_SELECT10;

typedef struct _SCSI_MODE_PARAMETER_HEADER6
{
 u8		ModeDataLength;
 u8		MediumType;
 u8		DeviceSpecificParameter;
 u8		BlockDescriptorLength;
} SCSI_MODE_PARAMETER_HEADER6, *PSCSI_MODE_PARAMETER_HEADER6;

typedef struct _SCSI_MODE_PARAMETER_HEADER10
{
 u8		ModeDataLengthMsb;
 u8		ModeDataLengthLsb;
 u8		MediumType;
 u8		DeviceSpecificParameter;
 u8		Reserved[2];
 u8		BlockDescriptorLengthMsb;
 u8		BlockDescriptorLengthLsb;
} SCSI_MODE_PARAMETER_HEADER10, *PSCSI_MODE_PARAMETER_HEADER10;

typedef struct _SCSI_MODE_PARAMTER_BLOCK_DESCRIPTER
{
 u8		DesityCode;
 u8		NumberOfBlocks2;
 u8		NumberOfBlocks1;
 u8		NumberOfBlocks0;
 u8		Reserved;
 u8		BlockLength2;
 u8		BlockLength1;
 u8		BlockLength0;
} SCSI_MODE_PARAMTER_BLOCK_DESCRIPTER, *PSCSI_MODE_PARAMTER_BLOCK_DESCRIPTER;

/************************************************************************
 * IDE command definitions
 ************************************************************************/
#define IDE_COMMAND_ATAPI_RESET			0x08
#define IDE_COMMAND_RECALIBRATE			0x10
#define IDE_COMMAND_READ_SECTOR			0x20
#define IDE_COMMAND_READ_SECTOR_EXT		0x24
#define IDE_COMMAND_READ_DMA_EXT		0x25
#define IDE_COMMAND_READ_MULTIPLE_EXT		0x29
#define IDE_COMMAND_WRITE_SECTOR		0x30
#define IDE_COMMAND_WRITE_SECTOR_EXT		0x34
#define IDE_COMMAND_WRITE_DMA_EXT		0x35
#define IDE_COMMAND_WRITE_MULTIPLE_EXT		0x39
#define IDE_COMMAND_READ_VERIFY			0x40
#define IDE_COMMAND_READ_VERIFY_EXT		0x42
#define IDE_COMMAND_SEEK			0x70
#define IDE_COMMAND_SET_DRIVE_PARAMETERS	0x91
#define IDE_COMMAND_ATAPI_PACKET		0xA0
#define IDE_COMMAND_ATAPI_IDENTIFY		0xA1
#define IDE_COMMAND_READ_MULTIPLE		0xC4
#define IDE_COMMAND_WRITE_MULTIPLE		0xC5
#define IDE_COMMAND_SET_MULTIPLE		0xC6
#define IDE_COMMAND_READ_DMA			0xC8
#define IDE_COMMAND_WRITE_DMA			0xCA
#define IDE_COMMAND_GET_MEDIA_STATUS		0xDA
#define IDE_COMMAND_ENABLE_MEDIA_STATUS		0xEF
#define IDE_COMMAND_SET_FEATURE			0xEF
#define IDE_COMMAND_IDENTIFY			0xEC
#define IDE_COMMAND_MEDIA_EJECT			0xED

/************************************************************************
 * IDE status definitions
 ************************************************************************/
#define IDE_STATUS_ERROR			0x01
#define IDE_STATUS_INDEX			0x02
#define IDE_STATUS_CORRECTED_ERROR		0x04
#define IDE_STATUS_DRQ				0x08
#define IDE_STATUS_DSC				0x10
#define IDE_STATUS_DRDY				0x40
#define IDE_STATUS_IDLE				0x50
#define IDE_STATUS_BUSY				0x80

/************************************************************************
 * IDE drive control definitions
 ************************************************************************/
#define IDE_DC_DISABLE_INTERRUPTS		0x02
#define IDE_DC_RESET_CONTROLLER			0x04
#define IDE_DC_REENABLE_CONTROLLER		0x00

/************************************************************************
 * IDE error definitions
 ************************************************************************/
#define IDE_ERROR_BAD_BLOCK			0x80
#define IDE_ERROR_DATA_ERROR			0x40
#define IDE_ERROR_MEDIA_CHANGE			0x20
#define IDE_ERROR_ID_NOT_FOUND			0x10
#define IDE_ERROR_MEDIA_CHANGE_REQ		0x08
#define IDE_ERROR_COMMAND_ABORTED		0x04
#define IDE_ERROR_END_OF_MEDIA			0x02
#define IDE_ERROR_ILLEGAL_LENGTH		0x01

/************************************************************************
 * IDENTIFY data
 ************************************************************************/
typedef struct _IDENTIFY_DATA
{
 u16		GeneralConfiguration;		/* 00 00		*/
 u16		NumberOfCylinders;		/* 02  1		*/
 u16		Reserved1;			/* 04  2		*/
 u16		NumberOfHeads;			/* 06  3		*/
 u16		UnformattedBytesPerTrack;	/* 08  4		*/
 u16		UnformattedBytesPerSector;	/* 0A  5		*/
 u16		SectorsPerTrack;		/* 0C  6		*/
 u16		VendorUnique1[3];		/* 0E  7-9		*/
 u16		SerialNumber[10];		/* 14  10-19		*/
 u16		BufferType;			/* 28  20		*/
 u16		BufferSectorSize;		/* 2A  21		*/
 u16		NumberOfEccBytes;		/* 2C  22		*/
 u16		FirmwareRevision[4];		/* 2E  23-26		*/
 u16		ModelNumber[20];		/* 36  27-46		*/
 u8		MaximumBlockTransfer;		/* 5E  47		*/
 u8		VendorUnique2;			/* 5F			*/
 u16		DoubleWordIo;			/* 60  48		*/
 u16		Capabilities;			/* 62  49		*/
 u16		Reserved2;			/* 64  50		*/
 u8		VendorUnique3;			/* 66  51		*/
 u8		PioCycleTimingMode;		/* 67			*/
 u8		VendorUnique4;			/* 68  52		*/
 u8		DmaCycleTimingMode;		/* 69			*/
 u16		TranslationFieldsValid : 1;	/* 6A  53		*/
 u16		Reserved3 : 15;			/*			*/
 u16		NumberOfCurrentCylinders;	/* 6C  54		*/
 u16		NumberOfCurrentHeads;		/* 6E  55		*/
 u16		CurrentSectorsPerTrack;		/* 70  56		*/
 u32		CurrentSectorCapacity;		/* 72  57-58		*/
 u16		CurrentMultiSectorSetting;	/*     59		*/
 u32		UserAddressableSectors;		/*     60-61		*/
 u16		SingleWordDMASupport : 8;	/*     62		*/
 u16		SingleWordDMAActive : 8;	/*			*/
 u16		MultiWordDMASupport : 8;	/*     63		*/
 u16		MultiWordDMAActive : 8;		/*			*/
 u16		AdvancedPIOModes : 8;		/*     64		*/
 u16		Reserved4 : 8;			/*			*/
 u16		MinimumMWXferCycleTime;		/*     65		*/
 u16		RecommendedMWXferCycleTime;	/*     66		*/
 u16		MinimumPIOCycleTime;		/*     67		*/
 u16		MinimumPIOCycleTimeIORDY;	/*     68		*/
 u16		Reserved5[2];			/*     69-70		*/
 u16		ReleaseTimeOverlapped;		/*     71		*/
 u16		ReleaseTimeServiceCommand;	/*     72		*/
 u16		MajorRevision;			/*     73		*/
 u16		MinorRevision;			/*     74		*/
 u16		Reserved6[50];			/*     75-126		*/
 u16		SpecialFunctionsEnabled;	/*     127		*/
 u16		Reserved7[128];			/*     128-255		*/
} IDENTIFY_DATA, *PIDENTIFY_DATA;

/************************************************************************
 * Identify data without the Reserved4
 ************************************************************************/
typedef struct _IDENTIFY_DATA2
{
 u16		GeneralConfiguration;		/* 00			*/
 u16		NumberOfCylinders;		/* 01			*/
 u16		Reserved1;			/* 02			*/
 u16		NumberOfHeads;			/* 03			*/
 u16		Reserved2[2];			/* 04-05		*/
 u16		SectorsPerTrack;		/* 06			*/
 u16		Reserved3[3];			/* 07-09		*/
 u16		SerialNumber[10];		/* 10-19		*/
 u16		Reserved4[3];			/* 20-22		*/
 u16		FirmwareRevision[4];		/* 23-26		*/
 u16		ModelNumber[20];		/* 27-46		*/
 u16		MaximumBlockTransfer;		/* 47			*/
 u16		Reserved5;			/* 48			*/
 u16		Capabilities[2];		/* 49-50		*/
 u16		Reserved6[2];			/* 51-52		*/
 u16		ValidFieldIndicator;		/* 53			*/
 u16		NumberOfCurrentCylinders;	/* 54			*/
 u16		NumberOfCurrentHeads;		/* 55			*/
 u16		CurrentSectorsPerTrack;		/* 56			*/
 u16		CurrentSectorCapacityLow;	/* 57			*/
 u16		CurrentSectorCapacityHigh;	/* 58			*/
 u16		CurrentMultiSectorSetting;	/* 59			*/
 u32		UserAddressableSectors;		/* 60-61		*/
 u16		Reserved7;			/* 62			*/
 u8		MultiWordDMASupport;		/* 63			*/
 u8		MultiWordDMAActive;		/*			*/
 u16		AdvancedPIOModes;		/* 64			*/
 u16		MinimumMWXferCycleTime;		/* 65			*/
 u16		RecommendedMWXferCycleTime;	/* 66			*/
 u16		MinimumPIOCycleTime;		/* 67			*/
 u16		MinimumPIOCycleTimeIORDY;	/* 68			*/
 u16		Reserved8[6];			/* 69-74		*/
 u16		QueueDepth;			/* 75			*/
 u16		Reserved9[4];			/* 76-79		*/
 u16		MajorVersionNumber;		/* 80			*/
 u16		MinorVersionNumber;		/* 81			*/
 u32		CmdSetSupported;		/* 82-83		*/
 u16		CmdSetFeatureSupportedExt;	/* 84			*/
 u16		CmdSetFeatureEnabledLow;	/* 85			*/
 u16		CmdSetFeatureEnabledHigh;	/* 86			*/
 u16		CmdSetFeatureDefault;		/* 87			*/
 u8		UltraDMASupport;		/* 88			*/
 u8		UltraDMAActive;			/*			*/
 u16		SecurityEraseTime;		/* 89			*/
 u16		EnhancedSecurityEraseTime;	/* 90			*/
 u16		PowerManagementValue;		/* 91			*/
 u16		MasterPasswordRevision;		/* 92			*/
 u16		HwResetResult;			/* 93			*/
 u16		Reserved11[6];			/* 94-99		*/
 u32		Capacity_48bit_LOW;		/* 100-101		*/
 u32		Capacity_48bit_HIGH;		/* 102-103		*/
 u16		Reserved12[24];			/* 104-127		*/
 u16		SecurityStatus;			/* 128			*/
 u16		Reserved13[31];			/* 129-159 vendor spec  */
 u16		Reserved14[96];			/* 160-255		*/
} IDENTIFY_DATA2, *PIDENTIFY_DATA2;

#define IDENTIFY_DATA_SIZE sizeof(IDENTIFY_DATA)

/************************************************************************
 * IDENTIFY capability bit definitions.
 ************************************************************************/
#define IDENTIFY_CAPABILITIES_DMA_SUPPORTED	0x0100
#define IDENTIFY_CAPABILITIES_LBA_SUPPORTED	0x0200

/************************************************************************
 * IDENTIFY DMA timing cycle modes.
 ************************************************************************/
#define IDENTIFY_DMA_CYCLES_MODE_0		0x00
#define IDENTIFY_DMA_CYCLES_MODE_1		0x01
#define IDENTIFY_DMA_CYCLES_MODE_2		0x02

typedef struct _SENSE_DATA
{
 u8		ErrorCode:7;
 u8		Valid:1;
 u8		SegmentNumber;
 u8		SenseKey:4;
 u8		Reserved:1;
 u8		IncorrectLength:1;
 u8		EndOfMedia:1;
 u8		FileMark:1;
 u8		Information[4];
 u8		AdditionalSenseLength;
 u8		CommandSpecificInformation[4];
 u8		AdditionalSenseCode;
 u8		AdditionalSenseCodeQualifier;
 u8		FieldReplaceableUnitCode;
 u8		SenseKeySpecific[3];
} SENSE_DATA, *PSENSE_DATA;

/************************************************************************
 * Sense codes
 ************************************************************************/
#define SCSI_SENSE_NO_SENSE		0x00
#define SCSI_SENSE_RECOVERED_ERROR	0x01
#define SCSI_SENSE_NOT_READY		0x02
#define SCSI_SENSE_MEDIUM_ERROR		0x03
#define SCSI_SENSE_HARDWARE_ERROR	0x04
#define SCSI_SENSE_ILLEGAL_REQUEST	0x05
#define SCSI_SENSE_UNIT_ATTENTION	0x06
#define SCSI_SENSE_DATA_PROTECT		0x07
#define SCSI_SENSE_BLANK_CHECK		0x08
#define SCSI_SENSE_UNIQUE		0x09
#define SCSI_SENSE_COPY_ABORTED		0x0A
#define SCSI_SENSE_ABORTED_COMMAND	0x0B
#define SCSI_SENSE_EQUAL		0x0C
#define SCSI_SENSE_VOL_OVERFLOW		0x0D
#define SCSI_SENSE_MISCOMPARE		0x0E
#define SCSI_SENSE_RESERVED		0x0F

/************************************************************************
 * Additional Sense codes
 ************************************************************************/
#define SCSI_ADSENSE_NO_SENSE		0x00
#define SCSI_ADSENSE_MAN_INTERV		0x03
#define SCSI_ADSENSE_LUN_NOT_READY	0x04
#define SCSI_ADSENSE_ILLEGAL_COMMAND	0x20
#define SCSI_ADSENSE_ILLEGAL_BLOCK	0x21
#define SCSI_ADSENSE_INVALID_LUN	0x25
#define SCSI_ADSENSE_SELECT_TIMEOUT	0x45
#define SCSI_ADSENSE_MUSIC_AREA		0xA0
#define SCSI_ADSENSE_DATA_AREA		0xA1
#define SCSI_ADSENSE_VOLUME_OVERFLOW	0xA7

#define SCSI_ADSENSE_NO_MEDIA_IN_DEVICE	0x3A
#define SCSI_ADWRITE_PROTECT		0x27
#define SCSI_ADSENSE_MEDIUM_CHANGED	0x28
#define SCSI_ADSENSE_BUS_RESET		0x29
#define SCSI_ADSENSE_TRACK_ERROR	0x14
#define SCSI_ADSENSE_SEEK_ERROR		0x15
#define SCSI_ADSENSE_REC_DATA_NOECC	0x17
#define SCSI_ADSENSE_REC_DATA_ECC	0x18
#define SCSI_ADSENSE_ILLEGAL_MODE	0x64
#define SCSI_ADSENSE_BAD_CDB		0x24
#define SCSI_ADSENSE_BAD_PARM_LIST	0x26
#define SCSI_ADSENSE_CANNOT_READ_MEDIUM	0x30

#define SCSISTAT_CHECK_CONDITION	0x02

/************************************************************************
 * Inquiry buffer structure. This is the data returned from the target
 * after it receives an inquiry.
 *
 * This structure may be extended by the number of bytes specified
 * in the field AdditionalLength. The defined size constant only
 * includes fields through ProductRevisionLevel.
 *
 * The NT SCSI drivers are only interested in the first 36 bytes of data.
 ************************************************************************/

#define INQUIRYDATABUFFERSIZE 36

typedef struct _INQUIRYDATA
{
 u8		DeviceType : 5;
 u8		DeviceTypeQualifier : 3;
 u8		DeviceTypeModifier : 7;
 u8		RemovableMedia : 1;
 u8		Versions;
 u8		ResponseDataFormat;
 u8		AdditionalLength;
 u8		Reserved[2];
 u8		SoftReset : 1;
 u8		CommandQueue : 1;
 u8		Reserved2 : 1;
 u8		LinkedCommands : 1;
 u8		Synchronous : 1;
 u8		Wide16Bit : 1;
 u8		Wide32Bit : 1;
 u8		RelativeAddressing : 1;
 u8		VendorId[8];
 u8		ProductId[16];
 u8		ProductRevisionLevel[4];
 u8		VendorSpecific[20];
 u8		Reserved3[40];
} INQUIRYDATA, *PINQUIRYDATA;

#define DIRECT_ACCESS_DEVICE		0x00	/* Disks		*/

/************************************************************************
 * Read Capacity Data - returned in Big Endian format
 ************************************************************************/
typedef struct _READ_CAPACITY_DATA
{
 u32		LogicalBlockAddress;
 u32		BytesPerBlock;
} READ_CAPACITY_DATA, *PREAD_CAPACITY_DATA;

#define MAXIMUM_CDB_SIZE	12

/************************************************************************
 * CDB (Command Descriptor Block)
 ************************************************************************/
typedef union _CDB
{
 /*
  * Standard 6-byte CDB
  */
 struct _CDB6READWRITE
    {
     u8		OperationCode;		/* Opcode			*/
     u8		LogicalBlockMsb1 : 5;	/* Logical block MSB 5-bit	*/
     u8		LogicalUnitNumber : 3;	/* LUN				*/
     u8		LogicalBlockMsb0;	/* Logical block MSB 8-bit	*/
     u8		LogicalBlockLsb;	/* Logical block LSB 8-bit	*/
     u8		TransferBlocks;		/* Data length			*/
     u8		Control;		/* Control byte			*/
    } CDB6READWRITE, *PCDB6READWRITE;

 /*
  * Standard 10-byte CDB
  */
 struct _CDB10
    {
     u8		OperationCode;
     u8		Reserved1 : 5;
     u8		LogicalUnitNumber : 3;
     u8		LogicalBlockByte0;
     u8		LogicalBlockByte1;
     u8		LogicalBlockByte2;
     u8		LogicalBlockByte3;
     u8		Reserved2;
     u8		TransferBlocksMsb;
     u8		TransferBlocksLsb;
     u8		Control;
    } CDB10, *PCDB10;

 struct _START_STOP
    {
     u8		OperationCode;
     u8		Immediate: 1;
     u8		Reserved1 : 4;
     u8		LogicalUnitNumber : 3;
     u8		Reserved2[2];
     u8		Start : 1;
     u8		LoadEject : 1;
     u8		Reserved3 : 6;
     u8		Control;
    } START_STOP, *PSTART_STOP;

} CDB, *PCDB;

/************************************************************************
 * SCSI CDB operation codes
 ************************************************************************/
#define SCSIOP_TEST_UNIT_READY		0x00
#define SCSIOP_REZERO_UNIT		0x01
#define SCSIOP_REWIND			0x01
#define SCSIOP_REQUEST_BLOCK_ADDR	0x02
#define SCSIOP_REQUEST_SENSE		0x03
#define SCSIOP_FORMAT_UNIT		0x04
#define SCSIOP_READ_BLOCK_LIMITS	0x05
#define SCSIOP_REASSIGN_BLOCKS		0x07
#define SCSIOP_READ6			0x08
#define SCSIOP_RECEIVE			0x08
#define SCSIOP_WRITE6			0x0A
#define SCSIOP_PRINT			0x0A
#define SCSIOP_SEND			0x0A
#define SCSIOP_SEEK6			0x0B
#define SCSIOP_TRACK_SELECT		0x0B
#define SCSIOP_SLEW_PRINT		0x0B
#define SCSIOP_SEEK_BLOCK		0x0C
#define SCSIOP_PARTITION		0x0D
#define SCSIOP_READ_REVERSE		0x0F
#define SCSIOP_WRITE_FILEMARKS		0x10
#define SCSIOP_FLUSH_BUFFER		0x10
#define SCSIOP_SPACE			0x11
#define SCSIOP_INQUIRY			0x12
#define SCSIOP_VERIFY6			0x13
#define SCSIOP_RECOVER_BUF_DATA		0x14
#define SCSIOP_MODE_SELECT		0x15
#define SCSIOP_RESERVE_UNIT		0x16
#define SCSIOP_RELEASE_UNIT		0x17
#define SCSIOP_COPY			0x18
#define SCSIOP_ERASE			0x19
#define SCSIOP_MODE_SENSE		0x1A
#define SCSIOP_START_STOP_UNIT		0x1B
#define SCSIOP_STOP_PRINT		0x1B
#define SCSIOP_LOAD_UNLOAD		0x1B
#define SCSIOP_RECEIVE_DIAGNOSTIC	0x1C
#define SCSIOP_SEND_DIAGNOSTIC		0x1D
#define SCSIOP_MEDIUM_REMOVAL		0x1E
#define SCSIOP_READ_CAPACITY		0x25
#define SCSIOP_READ			0x28
#define SCSIOP_WRITE			0x2A
#define SCSIOP_SEEK			0x2B
#define SCSIOP_LOCATE			0x2B
#define SCSIOP_WRITE_VERIFY		0x2E
#define SCSIOP_VERIFY			0x2F
#define SCSIOP_SEARCH_DATA_HIGH		0x30
#define SCSIOP_SEARCH_DATA_EQUAL	0x31
#define SCSIOP_SEARCH_DATA_LOW		0x32
#define SCSIOP_SET_LIMITS		0x33
#define SCSIOP_READ_POSITION		0x34
#define SCSIOP_SYNCHRONIZE_CACHE	0x35
#define SCSIOP_COMPARE			0x39
#define SCSIOP_COPY_COMPARE		0x3A
#define SCSIOP_WRITE_DATA_BUFF		0x3B
#define SCSIOP_READ_DATA_BUFF		0x3C
#define SCSIOP_CHANGE_DEFINITION	0x40
#define SCSIOP_READ_SUB_CHANNEL		0x42
#define SCSIOP_READ_TOC			0x43
#define SCSIOP_READ_HEADER		0x44
#define SCSIOP_PLAY_AUDIO		0x45
#define SCSIOP_PLAY_AUDIO_MSF		0x47
#define SCSIOP_PLAY_TRACK_INDEX		0x48
#define SCSIOP_PLAY_TRACK_RELATIVE	0x49
#define SCSIOP_PAUSE_RESUME		0x4B
#define SCSIOP_LOG_SELECT		0x4C
#define SCSIOP_LOG_SENSE		0x4D
#define SCSIOP_MODE_SELECT10		0x55
#define SCSIOP_MODE_SENSE10		0x5A
#define SCSIOP_LOAD_UNLOAD_SLOT		0xA6
#define SCSIOP_MECHANISM_STATUS		0xBD
#define SCSIOP_READ_CD			0xBE

#define DRIVER_NAME		"Device Driver for IT8211 RAID Controller"
#define COMPANY_NAME		"Integrated Technology Express, Inc."
#define CONTROLLER_NAME_IT8211	"IT8211 UDMA/ATA133 RAID Controller"
#define PROC_DIR_NAME		"it8211"
#define ITE_MAX_CMDS		124

#define PCI_IOSEN	0x01	/* Enable IO space			*/
#define PCI_BMEN	0x04	/* Enable IDE bus master		*/

/************************************************************************
 * PRD (Physical Region Descriptor) = Scatter-gather table
 *
 * |  byte3   |	 byte2	 |  byte1   |   byte0   |
 * +--------------------------------------------+
 * | Memory Region Physical Base Address[31:1]  |
 * +----+----------------+----------------------+
 * |EOT |  reserved      |   Byte count[15:1]   |
 * +----+----------------+----------------------+
 ************************************************************************/
typedef struct _PRD_TABLE_ENTRY
{
 u32		PhysicalBaseAddress;		/* Byte0 - Byte3	*/
 u16		ByteCount;			/* Byte4 - Byte5	*/
 u16		EndOfTable;			/* Byte6 - Byte7	*/
} PRD_TABLE_ENTRY, *PPRD_TABLE_ENTRY;

#define SG_FLAG_EOT			0x8000	/* End of PRD		*/
#define MAX_SG_DESCRIPTORS		17	/* 17 -- maximum 64K	*/

#define NUM_OF_PRD_TABLE_ENTRY		0x10

/************************************************************************
 * Bus master register bits definition
 ************************************************************************/
#define BM_CMD_FLG_START		0x01
#define BM_CMD_FLG_WRTTOMEM		0x08
#define BM_CMD_FLG_WRTTODSK		0x00

#define BM_STAT_FLG_ACTIVE		0x01
#define BM_STAT_FLG_ERR			0x02
#define BM_STAT_FLG_INT			0x04
#define BM_DRV0_DMA_CAPABLE		0x20
#define BM_DRV1_DMA_CAPABLE		0x40

#define BM_PRD_FLG_EOT  		0x8000

/************************************************************************
 * SRB Functions
 ************************************************************************/
#define SRB_FUNCTION_EXECUTE_SCSI	0x00
#define SRB_FUNCTION_IO_CONTROL		0x02
#define SRB_FUNCTION_SHUTDOWN		0x07
#define SRB_FUNCTION_FLUSH		0x08

/************************************************************************
 * SRB Status
 ************************************************************************/
#define SRB_STATUS_PENDING		0x00
#define SRB_STATUS_SUCCESS		0x01
#define SRB_STATUS_ABORTED		0x02
#define SRB_STATUS_ABORT_FAILED		0x03
#define SRB_STATUS_ERROR		0x04
#define SRB_STATUS_BUSY			0x05
#define SRB_STATUS_INVALID_REQUEST	0x06
#define SRB_STATUS_INVALID_PATH_ID	0x07
#define SRB_STATUS_NO_DEVICE		0x08
#define SRB_STATUS_TIMEOUT		0x09
#define SRB_STATUS_SELECTION_TIMEOUT	0x0A
#define SRB_STATUS_COMMAND_TIMEOUT	0x0B
#define SRB_STATUS_MESSAGE_REJECTED	0x0D
#define SRB_STATUS_BUS_RESET		0x0E
#define SRB_STATUS_PARITY_ERROR		0x0F
#define SRB_STATUS_REQUEST_SENSE_FAILED	0x10
#define SRB_STATUS_NO_HBA		0x11
#define SRB_STATUS_DATA_OVERRUN		0x12
#define SRB_STATUS_UNEXPECTED_BUS_FREE	0x13
#define SRB_STATUS_BAD_SRB_BLOCK_LENGTH	0x15
#define SRB_STATUS_REQUEST_FLUSHED	0x16
#define SRB_STATUS_INVALID_LUN		0x20
#define SRB_STATUS_INVALID_TARGET_ID	0x21
#define SRB_STATUS_BAD_FUNCTION		0x22
#define SRB_STATUS_ERROR_RECOVERY	0x23
#define SRB_STATUS_NEED_REQUEUE		0x24

/************************************************************************
 * SRB Status Masks
 ************************************************************************/
#define SRB_STATUS_QUEUE_FROZEN		0x40
#define SRB_STATUS_AUTOSENSE_VALID	0x80

#define SRB_STATUS(Status)	\
	(Status & ~(SRB_STATUS_AUTOSENSE_VALID | SRB_STATUS_QUEUE_FROZEN))

/************************************************************************
 * SRB Flag Bits
 ************************************************************************/
#define SRB_FLAGS_DATA_IN		0x00000040
#define SRB_FLAGS_DATA_OUT		0x00000080

/************************************************************************
 * SRB Working flags define area
 ************************************************************************/
#define	SRB_WFLAGS_USE_INTERNAL_BUFFER	0x00000001
#define	SRB_WFLAGS_IGNORE_ARRAY		0x00000002
#define	SRB_WFLAGS_HAS_CALL_BACK	0x00000004
#define	SRB_WFLAGS_MUST_DONE		0x00000008
#define	SRB_WFLAGS_ON_MIRROR_DISK	0x00000010
#define	SRB_WFLAGS_ON_SOURCE_DISK	0x00000020
#define SRB_WFLAGS_ARRAY_IO_STARTED	0x10000000
#define SRB_WFLAGS_WATCHTIMER_CALLED	0x20000000
#define SRB_WFLAGS_USE_SG		0x40000000

/************************************************************************
 * SCSI I/O Request Block
 ************************************************************************/
typedef struct _SCSI_REQUEST_BLOCK
{
 u16			Length;
 u8			Function;
 u8			SrbStatus;
 u8			ScsiStatus;
 u8			TargetId;
 u8			Lun;
 u8			CdbLength;
 u8			SenseInfoBufferLength;
 u8			UseSg;
 u8			reseved[2];
 u32			WorkingFlags;
 u32			SrbFlags;
 u32			DataTransferLength;
 u32			TimeOutValue;
 void *			DataBuffer;
 void *			SenseInfoBuffer;
 u8			Cdb[16];
 Scsi_Cmnd *		pREQ;
} SCSI_REQUEST_BLOCK, *PSCSI_REQUEST_BLOCK;

#define SCSI_REQUEST_BLOCK_SIZE sizeof(SCSI_REQUEST_BLOCK)

/************************************************************************
 * Second device flags
 ***********************************************************************/
#define DFLAGS_REDUCE_MODE	        0x00010000
#define DFLAGS_DEVICE_DISABLED		0x00020000
#define DFLAGS_BOOTABLE_DEVICE		0x00080000
#define DFLAGS_BOOT_MARK		0x00100000
#define DFLAGS_NEW_ADDED		0x40000000
#define DFLAGS_REMAINED_MEMBER		0x80000000

/************************************************************************
 * Device Extension Device Flags
 ************************************************************************/
/*
 * Indicates that some device is present.
 */
#define DFLAGS_DEVICE_PRESENT		0x0001

/*
 * Indicates whether ATAPI commands can be used.
 */
#define DFLAGS_ATAPI_DEVICE		0x0002

/*
 * Indicates whether this is a tape device.
 */
#define DFLAGS_TAPE_DEVICE		0x0004

/*
 * Indicates whether device interrupts as DRQ is set after
 * receiving Atapi Packet Command.
 */
#define DFLAGS_INT_DRQ			0x0008

/*
 * Indicates that the drive has the 'removable' bit set in
 * identify data (offset 128)
 */
#define DFLAGS_REMOVABLE_DRIVE		0x0010

/*
 * Media status notification enabled.
 */
#define DFLAGS_MEDIA_STATUS_ENABLED	0x0020

/*
 * Indicates atapi 2.5 changer present.
 */
#define DFLAGS_ATAPI_CHANGER		0x0040

/*
 * Indicates multi-platter device, not conforming to the 2.5 spec.
 */
#define DFLAGS_SANYO_ATAPI_CHANGER	0x0080

/*
 * Indicates that the init path for changers has already been done.
 */
#define DFLAGS_CHANGER_INITED		0x0100
#define DFLAGS_CONFIG_CHANGED		0x0200

#define UDMA_MODE_5_6			0x80

/************************************************************************
 * Used to disable 'advanced' features.
 ************************************************************************/
#define MAX_ERRORS			4

/************************************************************************
 * ATAPI command definitions
 ************************************************************************/
#define ATAPI_MODE_SENSE		0x5A
#define ATAPI_MODE_SELECT		0x55
#define ATAPI_FORMAT_UNIT		0x24

/************************************************************************
 * User IOCTL structure
 * Notes:
 * (1) Data transfers are limited to PAGE_SIZE (4k on i386, 8k for alpha)
 ************************************************************************/
typedef struct	_uioctl_t
{
 u16		inlen;		/* Length of data written to device	*/
 u16		outlen;		/* Length of data read from device	*/
 void *		data;		/* Data read from devic starts here	*/
 u8		status;		/* Status return from driver		*/
 u8		reserved[3];	/* For 4-byte alignment			*/
}		uioctl_t;

/************************************************************************
 * IOCTL commands for RAID
 ************************************************************************/
#define ITE_IOCMAGIC			't'

#define ITE_IOC_GET_PHY_DISK_STATUS	_IO(ITE_IOCMAGIC, 1)
#define ITE_IOC_CREATE_DISK_ARRAY	_IO(ITE_IOCMAGIC, 2)
#define ITE_IOC_REBUILD_START		_IO(ITE_IOCMAGIC, 3)
#define ITE_IOC_GET_REBUILD_STATUS	_IO(ITE_IOCMAGIC, 4)
#define ITE_IOC_RESET_ADAPTER		_IO(ITE_IOCMAGIC, 5)
#define ITE_IOC_GET_DRIVER_VERSION	_IO(ITE_IOCMAGIC, 6)

/************************************************************************
 * _Channel
 ************************************************************************/
typedef struct _Channel
{
 /*
  * IDE (ATAPI) io port address.
  */
 unsigned long		io_ports[IDE_NR_PORTS];

 /*
  * DMA base address.
  */
 unsigned long		dma_base;

 /*
  * Flags word for each possible device.
  */
 u16			DeviceFlags[2];

 /*
  * Indicates number of platters on changer-ish devices.
  */
 u32			DiscsPresent[2];

 /*
  * Indicates expecting an interrupt.
  */
 u8			ExpectingInterrupt;

 /*
  * Indicate last tape command was DSC Restrictive.
  */
 u8			RDP;

 /*
  * Interrupt level.
  */
 u8			InterruptLevel;

 /*
  * Placeholder for status register after a GET_MEDIA_STATUS command.
  */
 u8			ReturningMediaStatus;

 /*
  * Remember the channel number (0, 1)
  */
 u8			channel;

 /*
  * Indicates cable status.
  */
 u8			Cable80[2];

 /*
  * Reserved for alignment.
  */
 u8			reserved1[0];

 /*
  * Data buffer pointer.
  */
 unsigned short *	DataBuffer;

 /*
  * Data words left.
  */
 u32			WordsLeft;

 /*
  * Retry count.
  */
 u32			RetryCount;

 /*
  * Keep DMA type (MULTIWORD_DMA or ULTRA_DMA) for each device.
  */
 u8			DmaType[2];

 /*
  * Keep UDMA timing for each device.
  */
 u8			UdmaTiming[2];

 /*
  * Keep PIO/DMA timing for each channel. PioDmaTiming[clock][channel]
  */
 u8			PioDmaTiming[2];

 /*
  * Keep IDE clock (50 MHz or 66 MHz) for each device.
  */
 u8			IdeClock[2];

 /*
  * Keep the active device for each channel.
  */
 u8			ActiveDevice;

 /*
  * Indicate whether we should perform DMA mode switch on this channel?
  */
 u8			DoSwitch;

 /*
  * ???
  */
 u8			ConvertCdb;

 /*
  * Use or do not use DMA.
  */
 u8			UseDma[2];

 /*
  * Reserved for alignment.
  */
 u8			reserved2[3];

 /*
  * Identify data for device.
  */
 IDENTIFY_DATA		FullIdentifyData;
 IDENTIFY_DATA2		IdentifyData[2];

 /*
  * DMA PRD table physical address.
  */
 dma_addr_t		dmatable_dma;

 /*
  * DMA PRD table virtual address.
  */
 unsigned long *	dmatable_cpu;

 /*
  * Point to SCATTER/GATHER data buffer.
  */
 struct scatterlist *	sg_table;

 /*
  * DMA read or write.
  */
 int			sg_dma_direction;

 /*
  * Current request on controller.
  */
 PSCSI_REQUEST_BLOCK	CurrentSrb;

 /*
  * Original request on controller.
  */
 PSCSI_REQUEST_BLOCK	OriginalSrb;

 /*
  * Internal SRB.
  */
 SCSI_REQUEST_BLOCK	_Srb;

 /*
  * Remember the PCI device.
  */
 struct pci_dev *	pPciDev;

 /*
  * Placeholder for CDB.
  */
 u8			TempCdb[MAXIMUM_CDB_SIZE];
} Channel, *PChannel;

/************************************************************************
 * _Adapter
 ************************************************************************/
typedef struct _Adapter
{
 char *			name;		/* Adapter's name		*/
 u8			num_channels;	/* How many channels support	*/
 u8			irq;		/* irq number			*/
 u8			irqOwned;	/* If any irq is use		*/
 u8			pci_bus;	/* PCI bus number		*/
 u8			devfn;		/* Device and function number	*/
 u8			offline;	/* On line or off line		*/
 u8			bypass_mode;	/* bypass or firware mode	*/
 u8			reserved2[1];	/* Reserved for alignment	*/
 Channel *		IDEChannel;	/* IT8211 supports two channels	*/
 struct pci_dev *	pci_dev;	/* For PCI device		*/
}
ITE_ADAPTER, *PITE_ADAPTER;

/************************************************************************
 * Scsi_Host_Template Initializer
 ************************************************************************/
#define ITERAID								\
{									\
 next:				NULL,					\
 module:			NULL,					\
 proc_name:			"it8211",				\
 proc_info:			iteraid_proc_info,			\
 name:				"RAIDExpress133",			\
 detect:			iteraid_detect,				\
 release:			iteraid_release,			\
 info:	     			NULL,					\
 ioctl:				NULL,					\
 command:			iteraid_command,			\
 queuecommand:  		iteraid_queuecommand,			\
 abort:	    			iteraid_old_abort,			\
 reset:	    			iteraid_old_reset,			\
 eh_abort_handler:		iteraid_abort_eh,			\
 eh_device_reset_handler:	NULL,					\
 eh_bus_reset_handler:		NULL,					\
 eh_host_reset_handler:		iteraid_reset_eh,			\
 slave_attach:			NULL,					\
 bios_param:			iteraid_biosparam,			\
 can_queue:			1,					\
 this_id:			-1,					\
 sg_tablesize:   		32,					\
 cmd_per_lun:			1,					\
 present:			0,					\
 unchecked_isa_dma:		0,					\
 use_clustering:   		DISABLE_CLUSTERING,			\
 use_new_eh_code:		1					\
}

/************************************************************************
 * Beautification macros
 ************************************************************************/
#define ScheduleRetryProcess(pChan) do {		\
	pChan->retry_timer->expires = jiffies + 10;	\
	add_timer(pChan->retry_timer);			\
	} while (0)

#define CancelRetryProcess(pChan) del_timer(pChan->retry_timer)

#define GetStatus(pChan, Status)	\
    Status = inb(pChan->io_ports[IDE_CONTROL_OFFSET]);

#define GetBaseStatus(pChan, Status)	\
    Status = inb(pChan->io_ports[IDE_COMMAND_OFFSET]);

#define GetError(pChan, Error)		\
    Error = inb(pChan->io_ports[IDE_ERROR_OFFSET]);

#define ReadBuffer(pChan, Buffer, Count) \
    insw(pChan->io_ports[IDE_DATA_OFFSET], Buffer, Count);

#define WriteCommand(BaseIoAddress, Command) \
    outb(pChan->io_ports[IDE_COMMAND_OFFSET], Command);

#define WriteBuffer(pChan, Buffer, Count) \
    outsw(pChan->io_ports[IDE_DATA_OFFSET], Buffer, Count);

#define WaitOnBusy(pChan, Status)		\
{						\
 int		i;				\
 for (i = 0; i < 20000; i++)			\
    {						\
     GetStatus(pChan, Status);			\
     if (Status & IDE_STATUS_BUSY)		\
	{					\
	 udelay(150);				\
	 continue;				\
	}					\
     else 					\
	{					\
	 break;					\
	}					\
    }						\
}

#define WaitOnBaseBusy(pChan, Status)		\
{						\
 int		i;				\
 for (i = 0; i < 20000; i++)			\
    {						\
     GetBaseStatus(pChan, Status);		\
     if (Status & IDE_STATUS_BUSY)		\
	{					\
	 udelay(150);				\
	 continue;				\
	} 					\
     else					\
	{					\
	 break;					\
	}					\
    }						\
}

#define WaitForDrq(pChan, Status)		\
{						\
 int		i;				\
 for (i = 0; i < 1000; i++)			\
    {						\
     GetStatus(pChan, Status);			\
     if (Status & IDE_STATUS_BUSY)		\
	{					\
	 udelay(100);				\
	}					\
     else if (Status & IDE_STATUS_DRQ)		\
	{					\
	 break;					\
	}					\
     else					\
	{					\
	 udelay(200);				\
	}					\
    }						\
}

#define WaitForBaseDrq(pChan, Status)		\
{						\
 int		i;				\
 for (i = 0; i < 50000; i++)			\
    {						\
     GetBaseStatus(pChan, Status);		\
     if (Status & IDE_STATUS_BUSY)		\
	{					\
	 udelay(100);				\
	}					\
     else if (Status & IDE_STATUS_DRQ)		\
	{					\
	 break;					\
	}					\
     else					\
	{					\
	 udelay(200);				\
	}					\
    }						\
}

#define CheckBusyDrq(pChan, Status)		\
{						\
 int		i;				\
 for (i = 0; i < 50000; i++)			\
    {						\
     GetBaseStatus(pChan, Status);		\
     if ((Status & IDE_STATUS_BUSY) ||		\
	!(Status & IDE_STATUS_DRQ))		\
	{					\
	 udelay(200);				\
	}					\
     else					\
	{					\
	 break;					\
	}					\
    }						\
}

#define WaitShortForDrq(pChan, Status)		\
{						\
 int		i;				\
 for (i = 0; i < 2; i++)			\
    {						\
     GetStatus(pChan, Status);			\
     if (Status & IDE_STATUS_BUSY)		\
	{					\
	 udelay(100);				\
	}					\
     else if (Status & IDE_STATUS_DRQ)		\
	{					\
	 break;					\
	}					\
     else					\
	{					\
	 udelay(100);				\
	}					\
    }						\
}

#define WaitForDeviceReady(pChan, Status)	\
{						\
 int		i;				\
 for (i = 0; i < 50000; i++)			\
    {						\
     GetStatus(pChan, Status);			\
     if (Status == 0)				\
	{					\
	 break;					\
	}					\
     if ((Status & IDE_STATUS_BUSY) || (Status & IDE_STATUS_DRQ)) \
	{					\
	 udelay(200);				\
	 continue;				\
	}					\
     else					\
	{					\
	 break;					\
	}					\
    }						\
}

#define WaitForCommandComplete(pChan, Status)		\
{							\
 int		i;					\
 for (i = 0; i < 50000; i++)				\
    { 							\
     GetStatus(pChan, Status);				\
     if ((Status == 0) || (Status & IDE_STATUS_ERROR)	\
      || (Status == IDE_STATUS_IDLE))			\
	{						\
	 break;						\
	}						\
     udelay(200);					\
     continue;						\
    }							\
}

#define WaitForBaseCommandComplete(pChan, Status)	\
{							\
 int		i;					\
 for (i = 0; i < 50000; i++)				\
    { 							\
     GetBaseStatus(pChan, Status);			\
     if ((Status == 0) || (Status & IDE_STATUS_ERROR)	\
      || (Status == IDE_STATUS_IDLE))			\
	{						\
	 break;						\
	}						\
     udelay(200);					\
     continue;						\
    }							\
}

#define AtapiSoftReset(pChan, DevNum)		\
{						\
 unsigned char		statusByte;		\
 outb((unsigned char)(((DevNum & 0x1) << 4) | 0xA0), pChan->io_ports[IDE_SELECT_OFFSET]); \
 udelay(500);									  \
 outb(IDE_COMMAND_ATAPI_RESET, pChan->io_ports[IDE_COMMAND_OFFSET]);		  \
 mdelay(1000);									  \
 outb((unsigned char)(((DevNum & 0x1) << 4) | 0xA0), pChan->io_ports[IDE_SELECT_OFFSET]); \
 WaitOnBusy(pChan, statusByte);			\
 udelay(500);					\
}

#define IdeHardReset(pChan, result)		\
{						\
 unsigned char		statusByte;		\
 int			i;			\
 outb(IDE_DC_RESET_CONTROLLER, pChan->io_ports[IDE_CONTROL_OFFSET]);	\
 mdelay(50);								\
 outb(IDE_DC_REENABLE_CONTROLLER, pChan->io_ports[IDE_CONTROL_OFFSET]);	\
 for (i = 0; i < 1000 * 1000; i++)					\
    {									\
     statusByte = inb(pChan->io_ports[IDE_CONTROL_OFFSET]);		\
     if (statusByte != IDE_STATUS_IDLE && statusByte != 0x0)		\
	{					\
	 udelay(30);				\
	}					\
     else					\
	{					\
	 break;					\
	}					\
    }						\
 if (i == 1000 * 1000)				\
    {						\
     printk("IdeHardReset Fail!\n");		\
     result = FALSE;				\
    }						\
 else						\
    {						\
     dprintk("IdeHardReset Success!\n");	\
     result = TRUE;				\
    }						\
}

/************************************************************************
 * Function prototypes
 ************************************************************************/
const char *	iteraid_info (struct Scsi_Host *);
int		iteraid_detect (Scsi_Host_Template *);
int		iteraid_release (struct Scsi_Host *);
int		iteraid_command (Scsi_Cmnd *);
int		iteraid_queuecommand (Scsi_Cmnd *, void (*done)(Scsi_Cmnd *));
int		iteraid_biosparam (Disk *, kdev_t, int *);
int		iteraid_proc_info (char *buffer, char **start, off_t offset,
			int length, int hostno, int inout);

static void	TaskStart(PChannel, Scsi_Cmnd *);
static void	TaskQueue(void);
static void	TaskDone(PChannel, PSCSI_REQUEST_BLOCK);
static u32	IdeSendCommand(PChannel, PSCSI_REQUEST_BLOCK);
static void	IdeMediaStatus(u8, PChannel, u8);
static void	IdeSetupDma(PChannel, unsigned long, unsigned short);
static void	MapRequest(Scsi_Cmnd *, PSCSI_REQUEST_BLOCK);
static u8	IssueIdentify(PChannel, u8, u8);
static u8	IT8211ResetAdapter(PITE_ADAPTER);
static void	AtapiStartIo(PChannel, PSCSI_REQUEST_BLOCK);
static u8	AtapiInterrupt(PChannel);
static void	AtapiResetController(PITE_ADAPTER pAdap, PChannel pChan);

static int	itedev_open(struct inode *, struct file *);
static int	itedev_ioctl_entry(struct inode *, struct file *, unsigned int, unsigned long);
static int	itedev_ioctl(struct inode *, struct file *, unsigned int, unsigned long);
static int	itedev_close(struct inode *, struct file *);
#endif /* #ifndef _ITERAID_H_ */
