#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <syslog.h>
#include <stdarg.h>
#include <errno.h>
#include <time.h>
#include <signal.h>
#include <utmp.h>
#include <sys/ioctl.h>
#include <sys/resource.h>
#include "LOG_log.h"
#include "PRO_file.h"
#include "fntp.h"
#include "fmacro.h"
#include "ntp.h"
#include "cgi_err.h"

int sys_numservers = 0; 	     /* number of servers to poll */
int complete_servers = 0;
int initializing = 1;
u_long current_time = 0;
volatile int alarm_flag = 0;
int sys_samples = DEFSAMPLES;	/* number of samples/server */
u_long sys_timeout = DEFTIMEOUT; /* timeout time, in TIMER_HZ units */
struct server *sys_servers;	/* the server list */
int sys_authenticate = 0;	/* true when authenticating */
u_int32 sys_authkey = 0;	/* set to authentication key in use */
int sys_version = NTP_VERSION;	/* version to poll with */
double sys_residual = 0;	/* residual from previous adjustment */

struct timeval timeout = {60,0};


u_long volatile full_recvbufs;	     /* number of recvbufs on fulllist */
struct recvbuf *volatile freelist;   /* free buffers */
struct recvbuf *volatile fulllist;   /* lifo buffers with data */
struct recvbuf *volatile beginlist;  /* fifo buffers with data */
u_long volatile total_recvbufs;	     /* total recvbufs currently in use */
u_long volatile free_recvbufs;	     /* number of recvbufs on freelist */
u_long volatile lowater_adds;	     /* number of times we have added memory */

int fd;
fd_set fdmask;

/*
 * Storage declarations
 */
char lib_stringbuf[LIB_NUMBUFS][LIB_BUFLENGTH];
int lib_nextbuf;
int lib_inited = 0;
static const char *months[] = {
	"Jan", "Feb", "Mar", "Apr", "May", "Jun",
	"Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
};


/*
 * initialization routine.  Might be needed if the code is ROMized.
 */
void
init_lib(void)
{
	lib_nextbuf = 0;
	lib_inited = 1;
}

void PPC_Debug(int level,char *task, int index)
{
	char foo1[MAX_TASK_LEN+1];
	int length;
	char msg_buff[1][128];
	
	length = strlen(task);	       /* check task name length */
	if (length >= MAX_TASK_LEN) 
		length = MAX_TASK_LEN;
	memset(&foo1[0], '\0', MAX_TASK_LEN+1);
	strncpy(&foo1[0], task, length);
	
	PRO_GetMsgByIndex(NTP_SECTION,index,1,msg_buff,MSG_F);
	LOG_LogMsg(foo1,level,msg_buff[0]);
} 


char *
dolfptoa(
	u_long fpi,
	u_long fpv,
	int neg,
	int ndec,
	int msec
	)
{
	register u_char *cp, *cpend;
	register u_long lwork;
	register int dec;
	u_char cbuf[24];
	u_char *cpdec;
	char *buf;
	char *bp;

	/*
	 * Get a string buffer before starting
	 */
	LIB_GETBUF(buf);

	/*
	 * Zero the character buffer
	 */
	memset((char *) cbuf, 0, sizeof(cbuf));

	/*
	 * Work on the integral part.  This is biased by what I know
	 * compiles fairly well for a 68000.
	 */
	cp = cpend = &cbuf[10];
	lwork = fpi;
	if (lwork & 0xffff0000) {
		register u_long lten = 10;
		register u_long ltmp;

		do {
			ltmp = lwork;
			lwork /= lten;
			ltmp -= (lwork << 3) + (lwork << 1);
			*--cp = (u_char)ltmp;
		} while (lwork & 0xffff0000);
	}
	if (lwork != 0) {
		register u_short sten = 10;
		register u_short stmp;
		register u_short swork = (u_short)lwork;

		do {
			stmp = swork;
			swork /= sten;
			stmp -= (swork<<3) + (swork<<1);
			*--cp = (u_char)stmp;
		} while (swork != 0);
	}

	/*
	 * Done that, now deal with the problem of the fraction.  First
	 * determine the number of decimal places.
	 */
	if (msec) {
		dec = ndec + 3;
		if (dec < 3)
		    dec = 3;
		cpdec = &cbuf[13];
	} else {
		dec = ndec;
		if (dec < 0)
		    dec = 0;
		cpdec = &cbuf[10];
	}
	if (dec > 12)
	    dec = 12;
	
	/*
	 * If there's a fraction to deal with, do so.
	 */
	if (fpv != 0) {
		l_fp work;

		work.l_ui = 0;
		work.l_uf = fpv;
		while (dec > 0) {
			l_fp ftmp;

			dec--;
			/*
			 * The scheme here is to multiply the
			 * fraction (0.1234...) by ten.  This moves
			 * a junk of BCD into the units part.
			 * record that and iterate.
			 */
			work.l_ui = 0;
			L_LSHIFT(&work);
			ftmp = work;
			L_LSHIFT(&work);
			L_LSHIFT(&work);
			L_ADD(&work, &ftmp);
			*cpend++ = (u_char)work.l_ui;
			if (work.l_uf == 0)
			    break;
		}

		/*
		 * Rounding is rotten
		 */
		if (work.l_uf & 0x80000000) {
			register u_char *tp = cpend;

			*(--tp) += 1;
			while (*tp >= 10) {
				*tp = 0;
				*(--tp) += 1;
			};
			if (tp < cp)
			    cp = tp;
		}
	}
	cpend += dec;


	/*
	 * We've now got the fraction in cbuf[], with cp pointing at
	 * the first character, cpend pointing past the last, and
	 * cpdec pointing at the first character past the decimal.
	 * Remove leading zeros, then format the number into the
	 * buffer.
	 */
	while (cp < cpdec) {
		if (*cp != 0)
		    break;
		cp++;
	}
	if (cp == cpdec)
	    --cp;

	bp = buf;
	if (neg)
	    *bp++ = '-';
	while (cp < cpend) {
		if (cp == cpdec)
		    *bp++ = '.';
		*bp++ = (char)(*cp++ + '0');	/* ascii dependent? */
	}
	*bp = '\0';

	/*
	 * Done!
	 */
	return buf;
}
		
char *
mfptoa(u_long fpi,	u_long fpf,	int ndec)
{
	int isneg;

	if (M_ISNEG(fpi, fpf)) {
		isneg = 1;
		M_NEG(fpi, fpf);
	} else
	    isneg = 0;

	return dolfptoa(fpi, fpf, isneg, ndec, 0);
}

/** Level 2 **/
void * emalloc(u_int size)
{
	char *mem;
	
	if ((mem = (char *)malloc(size)) == 0) {
		PPC_Debug(LOG_LEVEL_4,LOG_TASK,MEMORY_ERR);
		exit(1);
	}
	return mem;
}

static void initialise_buffer(struct recvbuf *buff)
{
	memset((char *) buff, 0, sizeof(struct recvbuf));
}

/*
 * alarming - record the occurance of an alarm interrupt
 */

static void alarming(int sig)
{
	alarm_flag++;
}

void signal_no_reset(int sig,void (*func)(int))
{
	int n;
	struct sigaction vec;

	vec.sa_handler = func;
	sigemptyset(&vec.sa_mask);

	vec.sa_flags = 0;

	while (1){
		struct sigaction ovec;

		n = sigaction(sig, &vec, &ovec);
		if (n == -1 && errno == EINTR)
			continue;
		break;
	}
	if (n == -1) {
		exit(1);
	}
}

void init_recvbuff(int nbufs)
{
	register struct recvbuf *buf;
	int i;

	/*
	 * Init buffer free list and stat counters
	 */
	freelist = 0;

	buf = (struct recvbuf *)
	    emalloc(nbufs*sizeof(struct recvbuf));
	for (i = 0; i < nbufs; i++)
	{
		initialise_buffer(buf);
		buf->next = (struct recvbuf *) freelist;
		freelist = buf;
		buf++;
	}

	fulllist = 0;
	free_recvbufs = total_recvbufs = nbufs;
	full_recvbufs = lowater_adds = 0;
}

static void create_buffers(void)
{
	register struct recvbuf *buf;
	int i;
	buf = (struct recvbuf *)
	    emalloc(RECV_INC*sizeof(struct recvbuf));
	for (i = 0; i < RECV_INC; i++)
	{
		initialise_buffer(buf);
		buf->next = (struct recvbuf *) freelist;
		freelist = buf;
		buf++;
	}

	free_recvbufs += RECV_INC;
	total_recvbufs += RECV_INC;
	lowater_adds++;
}

int decodenetnum(const char *num, u_int32 *netnum)
{
	register const char *cp;
	register char *bp;
	register int i;
	register int temp;
	register int eos;
	char buf[80];		/* will core dump on really stupid stuff */

	cp = num;
	*netnum = 0;

	if (*cp == '[') {
		eos = ']';
		cp++;
	} else {
		eos = '\0';
	}

	for (i = 0; i < 4; i++) {
		bp = buf;
		while (isdigit((int)*cp))
		    *bp++ = *cp++;
		if (bp == buf)
		    break;

		if (i < 3) {
			if (*cp++ != '.')
			    break;
		} else if (*cp != eos)
		    break;

		*bp = '\0';
		temp = atoi(buf);
		if (temp > 255)
		    break;
		*netnum <<= 8;
		*netnum += temp;
	}
	
	if (i < 4)
	    return 0;
	*netnum = htonl(*netnum);
	return 1;
}

/*
 * getnetnum - given a host name, return its net number
 */
static int getnetnum(const char *host,u_int32 *num)
{
	struct hostent *hp;

	if (decodenetnum(host, num))
		return 1;
	else if ((hp = gethostbyname(host)) != 0) {
		memmove((char *)num, hp->h_addr, sizeof(u_int32));
		return (1);
	}
	PPC_Debug(LOG_LEVEL_4,LOG_TASK,RESOLVE_HOST_ERR);
	return (0);
}

/*
 * addserver - determine a server's address and allocate a new structure
 *		   for it.
 */

static void addserver(char *serv)
{
	register struct server *server;
	u_int32 netnum;

	if (!getnetnum(serv, &netnum))
		return;
	
	server = (struct server *)emalloc(sizeof(struct server));
	memset((char *)server, 0, sizeof(struct server));

	server->srcadr.sin_family = AF_INET;
	server->srcadr.sin_addr.s_addr = netnum;
	server->srcadr.sin_port = htons(NTP_PORT);

	server->event_time = ++sys_numservers;
	if (sys_servers == NULL){
		sys_servers = server;
	}
	else {
		struct server *sp;

		for (sp = sys_servers; sp->next_server != NULL;
		     sp = sp->next_server) ;
		sp->next_server = server;
	}
}



/** Level 1 **/
/*
void msyslog(int level, const char *fmt, ...)
{
	va_list ap;
	char buf[1025], nfmt[256];
	register int c;
	register char *n, *prog;
	register const char *f;
	int olderrno;
	char *err;

	va_start(ap, fmt);

	olderrno = errno;
	n = nfmt;
	f = fmt;
	while ((c = *f++) != '\0' && c != '\n' && n < &nfmt[252]) {
		if (c != '%') {
			*n++ = c;
			continue;
		}
		if ((c = *f++) != 'm') {
			*n++ = '%';
			*n++ = c;
			continue;
		}
		err = 0;

		err = strerror(olderrno);

		if ((n + strlen(err)) < &nfmt[254]) {
			strcpy(n, err);
			n += strlen(err);
		}
	}
	if (!syslogit)
	    *n++ = '\n';
	*n = '\0';

	vsprintf(buf, nfmt, ap);
	if (syslogit)
	    syslog(level, "%s", buf);
	else
	{
		FILE *out_file = syslog_file ? syslog_file
			: level <= LOG_ERR ? stderr : stdout;
		prog = strrchr(progname, '/');
		if (prog == NULL)
		    prog = progname;
		else
		    prog++;
		(void) fprintf(out_file, "%s ", humanlogtime ());
		(void) fprintf(out_file, "%s[%d]: %s", prog, (int)getpid(), buf);
		fflush (out_file);
	}
	va_end(ap);
}
*/

/*
 * init_io - initialize I/O data and open socket
 */
static void init_io(void)
{

	struct sockaddr_in addr;

	/*
	 * Init buffer free list and stat counters
	 */
	init_recvbuff(sys_numservers + 2);
	/*
	 * Open the socket
	 */
	/* create a datagram (UDP) socket */
	if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		PPC_Debug(LOG_LEVEL_4,LOG_TASK,OPEN_SOCKET_ERR);
		exit(1);
		/*NOTREACHED*/
	}
	/*
	 * bind the socket to the NTP port
	 */

	memset((char *)&addr, 0, sizeof addr);
	addr.sin_family = AF_INET;
	addr.sin_port = htons(NTP_PORT);
	addr.sin_addr.s_addr = htonl(INADDR_ANY);
	if (bind(fd, (struct sockaddr *)&addr, sizeof(addr)) < 0) {
		PPC_Debug(LOG_LEVEL_4,LOG_TASK,BIND_SOCKET_ERR);
		exit(1);
	}
	FD_ZERO(&fdmask);
	FD_SET(fd, &fdmask);
	/*
	 * set non-blocking,
	 */
	if (fcntl(fd, F_SETFL, O_NONBLOCK) < 0) {
		PPC_Debug(LOG_LEVEL_4,LOG_TASK,SET_NONBLOCKING_ERR);
		exit(1);
	}
}


static void init_alarm(void)
{
	struct itimerval itimer;

	alarm_flag = 0;

	/*
	 * Set up the alarm interrupt.	The first comes 1/(2*TIMER_HZ)
	 * seconds from now and they continue on every 1/TIMER_HZ seconds.
	 */
	(void) signal_no_reset(SIGALRM,alarming);
	itimer.it_interval.tv_sec = itimer.it_value.tv_sec = 0;
	itimer.it_interval.tv_usec = 1000000/TIMER_HZ;
	itimer.it_value.tv_usec = 1000000/(TIMER_HZ<<1);
	setitimer(ITIMER_REAL, &itimer, (struct itimerval *)0);
}

/*
 * getrecvbufs - get receive buffers which have data in them
 */

struct recvbuf * getrecvbufs(void){
	struct recvbuf *rb = NULL; /* nothing has arrived */;

	if (full_recvbufs == 0)
	{
		;
	}
	else {

		/*
		 * Get the fulllist chain and mark it empty
		 */
		rb = beginlist;
		fulllist = 0;
		full_recvbufs = 0;

		/*
		 * Check to see if we're below the low water mark.
		 */
		if (free_recvbufs <= RECV_LOWAT)
		{
			if (total_recvbufs >= RECV_TOOMANY)
//			    msyslog(LOG_ERR, "too many recvbufs allocated (%ld)",
//				    total_recvbufs);
;
			else
			{
				create_buffers();
			}
		}
	}

	/*
	 * Return the chain
	 */
	return rb;
}

/*
 * server_data - add a sample to the server's filter registers
 */
static void server_data(register struct server *server,s_fp d,l_fp *c,u_fp e)
{
	register int i;

	i = server->filter_nextpt;
	if (i < NTP_SHIFT) {
		server->filter_delay[i] = d;
		server->filter_offset[i] = *c;
		server->filter_soffset[i] = LFPTOFP(c);
		server->filter_error[i] = e;
		server->filter_nextpt = i + 1;
	}
}

/*
 * get_systime - return the system time in timestamp format biased by
 * the current time offset.
 */
void get_systime(l_fp *now)
{
	struct timeval tv;
	double dtemp;

	/*
	 * We use nanosecond time if we can get it. Watch out for
	 * rounding wiggles, which may overflow the fraction.
	 */
	(void) GETTIMEOFDAY(&tv, (struct timezone *)0);
	now->l_i = tv.tv_sec + JAN_1970;


	dtemp = tv.tv_usec * FRAC / 1e6;

	if (dtemp >= FRAC)
		now->l_i++;
	now->l_uf = (u_int32)dtemp;

}

/*
 * sendpkt - send a packet to the specified destination
 */
static void sendpkt(struct sockaddr_in *dest, struct pkt *pkt,int len)
{
	int cc;

	cc = sendto(fd, (char *)pkt, (size_t)len, 0, (struct sockaddr *)dest,
			sizeof(struct sockaddr_in));
	if (cc == -1) {
		if (errno != EWOULDBLOCK && errno != ENOBUFS)
//			msyslog(LOG_ERR, "sendto(%s): %m", ntoa(dest));
;
	}
}


/*
 * transmit - transmit a packet to the given server, or mark it completed.
 *  This is called by the timeout routine and by the receive
 *		  procedure.
 */
static void transmit(register struct server *server)
{
	struct pkt xpkt;

//	if (debug)
//		printf("transmit(%s)\n", ntoa(&server->srcadr));

	if (server->filter_nextpt < server->xmtcnt) {
		l_fp ts;
		/*
		 * Last message to this server timed out.  Shift
		 * zeros into the filter.
		 */
		L_CLR(&ts);
		server_data(server, 0, &ts, 0);
	}

	if ((int)server->filter_nextpt >= sys_samples) {
		/*
		 * Got all the data we need.  Mark this guy
		 * completed and return.
		 */
		server->event_time = 0;
		complete_servers++;
		return;
	}

	/*
	 * If we're here, send another message to the server.    Fill in
	 * the packet and let 'er rip.
	 */
	xpkt.li_vn_mode = PKT_LI_VN_MODE(LEAP_NOTINSYNC,
					 sys_version, MODE_CLIENT);
	xpkt.stratum = STRATUM_TO_PKT(STRATUM_UNSPEC);
	xpkt.ppoll = NTP_MINPOLL;
	xpkt.precision = NTPDATE_PRECISION;
	xpkt.rootdelay = htonl(NTPDATE_DISTANCE);
	xpkt.rootdispersion = htonl(NTPDATE_DISP);
	xpkt.refid = htonl(NTPDATE_REFID);
	L_CLR(&xpkt.reftime);
	L_CLR(&xpkt.org);
	L_CLR(&xpkt.rec);

	/*
	 * Determine whether to authenticate or not.	If so,
	 * fill in the extended part of the packet and do it.
	 * If not, just timestamp it and send it away.
	 */
		get_systime(&(server->xmt));
		HTONL_FP(&server->xmt, &xpkt.xmt);
		sendpkt(&(server->srcadr), &xpkt, LEN_PKT_NOMAC);

//		if (debug > 1)
//			printf("transmit to %s\n", ntoa(&(server->srcadr)));

	/*
	 * Update the server timeout and transmit count
	 */
	server->event_time = current_time + sys_timeout;
	server->xmtcnt++;
}


/*
 * findserver - find a server in the list given its address
 */
static struct server *findserver(struct sockaddr_in *addr)
{
	register u_int32 netnum;
	struct server *server;
	struct server *mc_server;

	mc_server = NULL;
	if (htons(addr->sin_port) != NTP_PORT)
		return 0;
	netnum = addr->sin_addr.s_addr;

	for (server = sys_servers; server != NULL; 
	     server = server->next_server) {
		register u_int32 servnum;
		
		servnum = server->srcadr.sin_addr.s_addr;
		if (netnum == servnum)
			return server;
		if (IN_MULTICAST(ntohl(servnum)))
			mc_server = server;
	}

	if (mc_server != NULL) {	
		struct server *sp;

		if (mc_server->event_time != 0) {
			mc_server->event_time = 0;
			complete_servers++;
		}
		server = (struct server *)emalloc(sizeof(struct server));
		memset((char *)server, 0, sizeof(struct server));

		server->srcadr.sin_family = AF_INET;
		server->srcadr.sin_addr.s_addr = netnum;
		server->srcadr.sin_port = htons(NTP_PORT);

		server->event_time = ++sys_numservers;
		for (sp = sys_servers; sp->next_server != NULL;
		     sp = sp->next_server) ;
		sp->next_server = server;
		transmit(server);
	}
	return NULL;
}

u_long authdecryptions;		/* calls to decrypt */
/*
 * authdecrypt - verify message authenticator
 *
 * Returns one if authenticator valid, zero if invalid or key not found.
 */
int authdecrypt(keyid_t keyno,u_int32 *pkt,int length,int size)
{
	/*
	 * A zero key identifier means the sender has not verified
	 * the last message was correctly authenticated. Nevertheless,
	 * the authenticator itself is considered valid.
	 */
	authdecryptions++;
	if (keyno == 0)
		return (0);

	return (0);
}

/*
 * receive - receive and process an incoming frame
 */
static void receive(struct recvbuf *rbufp)
{
	register struct pkt *rpkt;
	register struct server *server;
	register s_fp di;
	l_fp t10, t23;
	l_fp org;
	l_fp rec;
	l_fp ci;
	int has_mac;
	int is_authentic;

//	if (debug)
//		printf("receive(%s)\n", ntoa(&rbufp->recv_srcadr));
	/*
	 * Check to see if the packet basically looks like something
	 * intended for us.
	 */
	if (rbufp->recv_length == LEN_PKT_NOMAC)
		has_mac = 0;
	else if (rbufp->recv_length >= LEN_PKT_NOMAC)
		has_mac = 1;
	else {
//		if (debug)
//			printf("receive: packet length %d\n",
//			   rbufp->recv_length);
		return; 		/* funny length packet */
	}

	rpkt = &(rbufp->recv_pkt);
	if (PKT_VERSION(rpkt->li_vn_mode) < NTP_OLDVERSION ||
		PKT_VERSION(rpkt->li_vn_mode) > NTP_VERSION) {
		return;
	}

	if ((PKT_MODE(rpkt->li_vn_mode) != MODE_SERVER
		 && PKT_MODE(rpkt->li_vn_mode) != MODE_PASSIVE)
		|| rpkt->stratum >= STRATUM_UNSPEC) {
//		if (debug)
//			printf("receive: mode %d stratum %d\n",
//			   PKT_MODE(rpkt->li_vn_mode), rpkt->stratum);
		return;
	}

	/*
	 * So far, so good.  See if this is from a server we know.
	 */
	server = findserver(&(rbufp->recv_srcadr));
	if (server == NULL) {
//		if (debug)
//			printf("receive: server not found\n");
		return;
	}

	/*
	 * Decode the org timestamp and make sure we're getting a response
	 * to our last request.
	 */
	NTOHL_FP(&rpkt->org, &org);
	if (!L_ISEQU(&org, &server->xmt)) {
//		if (debug)
//			printf("receive: pkt.org and peer.xmt differ\n");
		return;
	}

	/*
	 * Check out the authenticity if we're doing that.
	 */
	if (!sys_authenticate)
		is_authentic = 1;
	else {
		is_authentic = 0;

//		if (debug > 3)
//			printf("receive: rpkt keyid=%ld sys_authkey=%ld decrypt=%ld\n",
//			   (long int)ntohl(rpkt->exten[0]), (long int)sys_authkey,
//			   (long int)authdecrypt(sys_authkey, (u_int32 *)rpkt,
//				LEN_PKT_NOMAC, (int)(rbufp->recv_length - LEN_PKT_NOMAC)));

		if (has_mac && ntohl(rpkt->exten[0]) == sys_authkey &&
			authdecrypt(sys_authkey, (u_int32 *)rpkt, LEN_PKT_NOMAC,
			(int)(rbufp->recv_length - LEN_PKT_NOMAC)))
			is_authentic = 1;
//		if (debug)
//			printf("receive: authentication %s\n",
//			   is_authentic ? "passed" : "failed");
	}
	server->trust <<= 1;
	if (!is_authentic)
		server->trust |= 1;

	/*
	 * Looks good.	Record info from the packet.
	 */
	server->leap = PKT_LEAP(rpkt->li_vn_mode);
	server->stratum = PKT_TO_STRATUM(rpkt->stratum);
	server->precision = rpkt->precision;
	server->rootdelay = ntohl(rpkt->rootdelay);
	server->rootdispersion = ntohl(rpkt->rootdispersion);
	server->refid = rpkt->refid;
	NTOHL_FP(&rpkt->reftime, &server->reftime);
	NTOHL_FP(&rpkt->rec, &rec);
	NTOHL_FP(&rpkt->xmt, &server->org);

	/*
	 * Make sure the server is at least somewhat sane.	If not, try
	 * again.
	 */
	if (L_ISZERO(&rec) || !L_ISHIS(&server->org, &rec)) {
		transmit(server);
		return;
	}

	/*
	 * Calculate the round trip delay (di) and the clock offset (ci).
	 * We use the equations (reordered from those in the spec):
	 *
	 * d = (t2 - t3) - (t1 - t0)
	 * c = ((t2 - t3) + (t1 - t0)) / 2
	 */
	t10 = server->org;		/* pkt.xmt == t1 */
	L_SUB(&t10, &rbufp->recv_time); /* recv_time == t0*/

	t23 = rec;			/* pkt.rec == t2 */
	L_SUB(&t23, &org);		/* pkt->org == t3 */

	/* now have (t2 - t3) and (t0 - t1).	Calculate (ci) and (di) */
	ci = t10;
	L_ADD(&ci, &t23);
	L_RSHIFT(&ci);

	/*
	 * Calculate di in t23 in full precision, then truncate
	 * to an s_fp.
	 */
	L_SUB(&t23, &t10);
	di = LFPTOFP(&t23);

//	if (debug > 3)
//		printf("offset: %s, delay %s\n", lfptoa(&ci, 6), fptoa(di, 5));

	di += (FP_SECOND >> (-(int)NTPDATE_PRECISION))
		+ (FP_SECOND >> (-(int)server->precision)) + NTP_MAXSKW;

	if (di <= 0) {		/* value still too raunchy to use? */
		L_CLR(&ci);
		di = 0;
	} else {
		di = max(di, NTP_MINDIST);
	}

	/*
	 * Shift this data in, then transmit again.
	 */
	server_data(server, (s_fp) di, &ci, 0);
	transmit(server);
}




/*
 * freerecvbuf - make a single recvbuf available for reuse
 */
void freerecvbuf(struct recvbuf *rb)
{
	rb->next = (struct recvbuf *) freelist;
	freelist = rb;
	free_recvbufs++;
}

/*
 * timer - process a timer interrupt
 */
void timer(void){
	struct server *server;

	/*
	 * Bump the current idea of the time
	 */
	current_time++;

	/*
	 * Search through the server list looking for guys
	 * who's event timers have expired.  Give these to
	 * the transmit routine.
	 */
	for (server = sys_servers; server != NULL; 
	     server = server->next_server) {
		if (server->event_time != 0
		    && server->event_time <= current_time)
			transmit(server);
	}
}



const char *set_tod_using = "UNKNOWN";
int ntp_set_tod(struct timeval *tvp,void *tzp)
{
	int rc;
	{

		rc = SETTIMEOFDAY(tvp, tzp);
		if (!rc)
		{
			set_tod_using = "settimeofday";
			return rc;
		}
	}
	set_tod_using = "Failed!";
	return -1;
}

/*
 * step_systime - step the system clock.
 */
int step_systime(double now)
{
	struct timeval timetv, adjtv, oldtimetv;
	int isneg = 0;
	double dtemp;
	
	dtemp = sys_residual + now;
	if (dtemp < 0) {
		isneg = 1;
		dtemp = 0 - dtemp;
		adjtv.tv_sec = (int32)dtemp;
		adjtv.tv_usec = (u_int32)((dtemp - (double)adjtv.tv_sec) *
					  1e6 + .5);
	} else {
		adjtv.tv_sec = (int32)dtemp;
		adjtv.tv_usec = (u_int32)((dtemp - (double)adjtv.tv_sec) *
					  1e6 + .5);
	}
	(void) GETTIMEOFDAY(&timetv, (struct timezone *)0);

	oldtimetv = timetv;
#ifdef DEBUG
	if (debug)
		printf("step_systime: step %.6f residual %.6f\n", now, sys_residual);
#endif
	
	if (isneg) {
		timetv.tv_sec -= adjtv.tv_sec;
		timetv.tv_usec -= adjtv.tv_usec;
		if (timetv.tv_usec < 0) {
			timetv.tv_sec--;
			timetv.tv_usec += 1000000;
		}
	} else {
		timetv.tv_sec += adjtv.tv_sec;
		timetv.tv_usec += adjtv.tv_usec;
		if (timetv.tv_usec >= 1000000) {
			timetv.tv_sec++;
			timetv.tv_usec -= 1000000;
		}
	}
	if (ntp_set_tod(&timetv, (struct timezone *)0) != 0) {
//		msyslog(LOG_ERR, "Can't set time of day: %m");
		return (0);
	}
	sys_residual = 0;

	/*
	 * FreeBSD, for example, has:
	 * struct utmp {
	 *	   char    ut_line[UT_LINESIZE];
	 *	   char    ut_name[UT_NAMESIZE];
	 *	   char    ut_host[UT_HOSTSIZE];
	 *	   long    ut_time;
	 * };
	 * and appends line="|", name="date", host="", time for the OLD
	 * and appends line="{", name="date", host="", time for the NEW
	 * to _PATH_WTMP .
	 *
	 * Some OSes have utmp, some have utmpx.
	 */

	/*
	 * Write old and new time entries in utmp and wtmp if step adjustment
	 * is greater than one second.
	 *
	 * This might become even Uglier...
	 */
	if (oldtimetv.tv_sec != timetv.tv_sec)
	{
		struct utmp ut;

		memset((char *)&ut, 0, sizeof(ut));

	}
	return (1);
}

/*
 * This fuction is not the same as lib/systime step_systime!!!
 */
static int l_step_systime(l_fp *ts)
{
	double dtemp;

//	if (debug)
//		return 1;
	LFPTOD(ts, dtemp);
	return step_systime(dtemp);
}

/*
 * clock_select - select the pick-of-the-litter clock from the samples
 *		  we've got.
 */
static struct server *clock_select(void)
{
	register struct server *server;
	register int i;
	register int nlist;
	register s_fp d;
	register int j;
	register int n;
	s_fp local_threshold;
	struct server *server_list[NTP_MAXCLOCK];
	u_fp server_badness[NTP_MAXCLOCK];
	struct server *sys_server;

	/*
	 * This first chunk of code is supposed to go through all
	 * servers we know about to find the NTP_MAXLIST servers which
	 * are most likely to succeed.	We run through the list
	 * doing the sanity checks and trying to insert anyone who
	 * looks okay.	We are at all times aware that we should
	 * only keep samples from the top two strata and we only need
	 * NTP_MAXLIST of them.
	 */
	nlist = 0;	/* none yet */
	for (server = sys_servers; server != NULL; server = server->next_server) {
		if (server->delay == 0)
			continue;	/* no data */
#if 0			
		if (server->stratum > NTP_INFIN)
			continue;	/* stratum no good */
#endif
		if (server->delay > NTP_MAXWGT) {
			continue;	/* too far away */
		}
#if 0		
		if (server->leap == LEAP_NOTINSYNC)
			continue;	/* he's in trouble */
#endif
		if (!L_ISHIS(&server->org, &server->reftime)) {
			continue;	/* very broken host */
		}
#if 0		
		if ((server->org.l_ui - server->reftime.l_ui)
			>= NTP_MAXAGE) {
			continue;	/* too long without sync */
		}
#endif
		if (server->trust != 0) {
			continue;
		}
		/*
		 * This one seems sane.  Find where he belongs
		 * on the list.
		 */
		d = server->dispersion + server->dispersion;
		for (i = 0; i < nlist; i++)
			if (server->stratum <= server_list[i]->stratum)
			break;
		for ( ; i < nlist; i++) {
			if (server->stratum < server_list[i]->stratum)
				break;
			if (d < (s_fp) server_badness[i])
				break;
		}

		/*
		 * If i points past the end of the list, this
		 * guy is a loser, else stick him in.
		 */
		if (i >= NTP_MAXLIST)
			continue;
		for (j = nlist; j > i; j--)
			if (j < NTP_MAXLIST) {
				server_list[j] = server_list[j-1];
				server_badness[j]
					= server_badness[j-1];
			}

		server_list[i] = server;
		server_badness[i] = d;
		if (nlist < NTP_MAXLIST)
			nlist++;
	}

	/*
	 * Got the five-or-less best.	 Cut the list where the number of
	 * strata exceeds two.
	 */
	j = 0;
	for (i = 1; i < nlist; i++)
		if (server_list[i]->stratum > server_list[i-1]->stratum)
		if (++j == 2) {
			nlist = i;
			break;
		}

	/*
	 * Whew!  What we should have by now is 0 to 5 candidates for
	 * the job of syncing us.  If we have none, we're out of luck.
	 * If we have one, he's a winner.  If we have more, do falseticker
	 * detection.
	 */

	if (nlist == 0)
		sys_server = 0;
	else if (nlist == 1) {
		sys_server = server_list[0];
	} else {
		/*
		 * Re-sort by stratum, bdelay estimate quality and
		 * server.delay.
		 */
		for (i = 0; i < nlist-1; i++)
			for (j = i+1; j < nlist; j++) {
				if (server_list[i]->stratum
				< server_list[j]->stratum)
				break;	/* already sorted by stratum */
				if (server_list[i]->delay
				< server_list[j]->delay)
				continue;
				server = server_list[i];
				server_list[i] = server_list[j];
				server_list[j] = server;
			}

		/*
		 * Calculate the fixed part of the dispersion limit
		 */
		local_threshold = (FP_SECOND >> (-(int)NTPDATE_PRECISION))
			+ NTP_MAXSKW;

		/*
		 * Now drop samples until we're down to one.
		 */
		while (nlist > 1) {
			for (n = 0; n < nlist; n++) {
				server_badness[n] = 0;
				for (j = 0; j < nlist; j++) {
					if (j == n) /* with self? */
						continue;
					d = server_list[j]->soffset
						- server_list[n]->soffset;
					if (d < 0)	/* absolute value */
						d = -d;
					/*
					 * XXX This code *knows* that
					 * NTP_SELECT is 3/4
					 */
					for (i = 0; i < j; i++)
						d = (d>>1) + (d>>2);
					server_badness[n] += d;
				}
			}

			/*
			 * We now have an array of nlist badness
			 * coefficients.	Find the badest.  Find
			 * the minimum precision while we're at
			 * it.
			 */
			i = 0;
			n = server_list[0]->precision;;
			for (j = 1; j < nlist; j++) {
				if (server_badness[j] >= server_badness[i])
					i = j;
				if (n > server_list[j]->precision)
					n = server_list[j]->precision;
			}

			/*
			 * i is the index of the server with the worst
			 * dispersion.	If his dispersion is less than
			 * the threshold, stop now, else delete him and
			 * continue around again.
			 */
			if ( (s_fp) server_badness[i] < (local_threshold
							 + (FP_SECOND >> (-n))))
				break;
			for (j = i + 1; j < nlist; j++)
				server_list[j-1] = server_list[j];
			nlist--;
		}

		/*
		 * What remains is a list of less than 5 servers.  Take
		 * the best.
		 */
		sys_server = server_list[0];
	}

	/*
	 * That's it.    Return our server.
	 */
	return sys_server;
}

/*
 * clock_filter - determine a server's delay, dispersion and offset
 */
static void clock_filter(register struct server *server)
{
	register int i, j;
	int ord[NTP_SHIFT];

	/*
	 * Sort indices into increasing delay order
	 */
	for (i = 0; i < sys_samples; i++)
		ord[i] = i;

	for (i = 0; i < (sys_samples-1); i++) {
		for (j = i+1; j < sys_samples; j++) {
			if (server->filter_delay[ord[j]] == 0)
				continue;
			if (server->filter_delay[ord[i]] == 0
				|| (server->filter_delay[ord[i]]
				> server->filter_delay[ord[j]])) {
				register int tmp;

				tmp = ord[i];
				ord[i] = ord[j];
				ord[j] = tmp;
			}
		}
	}

	/*
	 * Now compute the dispersion, and assign values to delay and
	 * offset.	If there are no samples in the register, delay and
	 * offset go to zero and dispersion is set to the maximum.
	 */
	if (server->filter_delay[ord[0]] == 0) {
		server->delay = 0;
		L_CLR(&server->offset);
		server->soffset = 0;
		server->dispersion = PEER_MAXDISP;
	} else {
		register s_fp d;

		server->delay = server->filter_delay[ord[0]];
		server->offset = server->filter_offset[ord[0]];
		server->soffset = LFPTOFP(&server->offset);
		server->dispersion = 0;
		for (i = 1; i < sys_samples; i++) {
			if (server->filter_delay[ord[i]] == 0)
				d = PEER_MAXDISP;
			else {
				d = server->filter_soffset[ord[i]]
					- server->filter_soffset[ord[0]];
				if (d < 0)
					d = -d;
				if (d > PEER_MAXDISP)
					d = PEER_MAXDISP;
			}
			/*
			 * XXX This *knows* PEER_FILTER is 1/2
			 */
			server->dispersion += (u_fp)(d) >> i;
		}
	}
	/*
	 * We're done
	 */
}

char *
numtoa(
	u_int32 num
	)
{
	register u_int32 netnum;
	register char *buf;

	netnum = ntohl(num);
	LIB_GETBUF(buf);
	(void) sprintf(buf, "%lu.%lu.%lu.%lu", ((u_long)netnum >> 24) & 0xff,
		       ((u_long)netnum >> 16) & 0xff, ((u_long)netnum >> 8) & 0xff,
		       (u_long)netnum & 0xff);
	return buf;
}

#define	ntoa(_sin)		numtoa((_sin)->sin_addr.s_addr)

/*
 * clock_adjust - process what we've received, and adjust the time
 *		 if we got anything decent.
 */
static int clock_adjust(void)
{
	register struct server *sp=NULL, *server=NULL;
	FILE *fp;
	char time_str[32],tmp[128]={0};
	time_t cursec ;
	struct tm *tm;
	
	for (sp = sys_servers; sp != NULL; sp = sp->next_server)
		clock_filter(sp);
	server = clock_select();

	if (server == 0) {
		PPC_Debug(LOG_LEVEL_4,LOG_TASK,ADJUST_TIME_ERR);
		return(1);
	}
	remove(NTP_OFFSET_FILE);
	if (l_step_systime(&server->offset)) {
		cursec = time((time_t *) 0);
		tm= localtime(&cursec);
		PRO_GetMsgByIndex(NTP_SECTION,SYNC_OK,1,(Char128 *)tmp,ERRMSG_FILE);
		if(!tmp[0])
			strcpy(tmp,"Synchronization with time server %s succeeded%cCurrent Time: %s%cOffset: %s sec");
		fp=fopen(NTP_OFFSET_FILE,"wt");
		if(fp){	
			(void) sprintf(time_str, "%s %d %02d:%02d:%02d %d", months[tm->tm_mon],tm->tm_mday,tm->tm_hour, tm->tm_min, tm->tm_sec, tm->tm_year+1900);
			fprintf(fp,tmp, ntoa(&server->srcadr),'\n',time_str,'\n',lfptoa(&server->offset, 6));
			fflush(fp);
			fclose(fp);
		}
		if(!fork()){	
			int i;
			
			setsid();
			for(i=0;i<64;i++)
				close(i);
			if(fork()>0)
				exit(0);
			chdir("/");
			umask(0);
			signal(SIGCLD,SIG_IGN);		
			system("/usr/sbin/hwclock -w 2>/dev/null");
			exit(0);
		}
	}
	return(0);
}

void add_full_recv_buffer(struct recvbuf *rb)
{
	if (full_recvbufs == 0)
	{
		beginlist = rb;
		rb->next = 0;
	}
	else
	{
		rb->next = fulllist->next;
		fulllist->next = rb;
	}
	fulllist = rb;
	full_recvbufs++;
}

struct recvbuf * get_free_recv_buffer(void)
{
	struct recvbuf * buffer = NULL;
	if (free_recvbufs <= RECV_LOWAT)
		{
			if (total_recvbufs >= RECV_TOOMANY) {
//			    msyslog(LOG_ERR, "too many recvbufs allocated (%ld)",
//				    total_recvbufs);
			}
			else
			{
				create_buffers();
			}
		}

	if (free_recvbufs > 0)
	{
		buffer = freelist;
		freelist = buffer->next;
		buffer->next = NULL;
		--free_recvbufs;
	}
	return buffer;
}

u_long free_recvbuffs (void)
{
	return free_recvbufs;
}


/*
 * input_handler - receive packets asynchronously
 */
void input_handler(void)
{
	register int n;
	register struct recvbuf *rb;
	struct timeval tvzero;
	int fromlen;
	l_fp ts;
	fd_set fds;

	/*
	 * Do a poll to see if we have data
	 */
	for (;;) {
		fds = fdmask;
		tvzero.tv_sec = tvzero.tv_usec = 0;
		n = select(fd+1, &fds, (fd_set *)0, (fd_set *)0, &tvzero);

		/*
		 * If nothing to do, just return.  If an error occurred,
		 * complain and return.  If we've got some, freeze a
		 * timestamp.
		 */
		if (n == 0)
			return;
		else if (n == -1) {
			if (errno != EINTR)
//				msyslog(LOG_ERR,
//					"select() error: %m"
//					);
			;
			return;
		}
		get_systime(&ts);

		/*
		 * Get a buffer and read the frame.  If we
		 * haven't got a buffer, or this is received
		 * on the wild card socket, just dump the packet.
		 */
		if (initializing || free_recvbuffs() == 0) {
			char buf[100];

			(void) read(fd, buf, sizeof buf);
			continue;
		}

		rb = get_free_recv_buffer();

		fromlen = sizeof(struct sockaddr_in);
		rb->recv_length = recvfrom(fd, (char *)&rb->recv_pkt,
		   sizeof(rb->recv_pkt), 0,
		   (struct sockaddr *)&rb->recv_srcadr, &fromlen);
		if (rb->recv_length == -1) {
			freerecvbuf(rb);
			continue;
		}

		/*
		 * Got one.  Mark how and when it got here,
		 * put it on the full list.
		 */
		rb->recv_time = ts;
		add_full_recv_buffer(rb);
	}
}


int Get_NTP_Time(char *serv){
	int was_alarmed;
	struct recvbuf *rbuflist;
	struct recvbuf *rbuf;

	addserver(serv);

	if (sys_numservers == 0) {
		return -1;
	}

	init_io();
	
	init_alarm();
	/*
	 * Set the priority.
	 */

	initializing = 0;
	was_alarmed = 0;
	rbuflist = (struct recvbuf *)0;
	

	while (complete_servers < sys_numservers) {
	
		fd_set rdfdes;
		int nfound;

		if (alarm_flag) {		/* alarmed? */
			was_alarmed = 1;
			alarm_flag = 0;
		}
		rbuflist = getrecvbufs();	/* get received buffers */

		if (!was_alarmed && rbuflist == (struct recvbuf *)0) {
			/*
			 * Nothing to do.	 Wait for something.
			 */
			rdfdes = fdmask;
			nfound = select(fd+1, &rdfdes, (fd_set *)0,
					(fd_set *)0, &timeout);

			if (nfound > 0){
				input_handler();
			}
			else if (nfound == -1) {
				if (errno != EINTR)
//					msyslog(LOG_ERR,"select() error: %m");
					printf("enter select() error\n");
			} else {
//					msyslog(LOG_DEBUG,"select(): nfound = %d, error: %m",nfound);
					printf("enter select():nfound error\n");
			}
			if (alarm_flag) {		/* alarmed? */
				was_alarmed = 1;
				alarm_flag = 0;
			}
			rbuflist = getrecvbufs();	/* get received buffers */
		}

		/*
		 * Out here, signals are unblocked.  Call receive
		 * procedure for each incoming packet.
		 */
		while (rbuflist != (struct recvbuf *)0) {
			rbuf = rbuflist;
			rbuflist = rbuf->next;
			receive(rbuf);
			freerecvbuf(rbuf);
		}

		/*
		 * Call timer to process any timeouts
		 */
		if (was_alarmed) {
			timer();
			was_alarmed = 0;
		}

		/*
		 * Go around again
		 */
	}
	return clock_adjust();
}

int Set_Time(const struct timeval *tv){
	
	return 0;

}



