/**
 * ============================================================================
 * = COPYRIGHT
 * File Version: $Revision: 1.1.1.1 $
 * 
 * -- Intel Copyright Notice --
 * 
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * -- End Intel Copyright Notice --
 * = PRODUCT
 *      Intel(r) IXP425 Software Release
 *
 * = LIBRARY
 *      OSSL ( Operating System Services)  Library
 *
 * = MODULE
 *      OSSL Semaphores
 *
 * = FILENAME
 *      osslSema.c
 *
 * = DESCRIPTION
 *      
 *
 * = AUTHOR
 *      Intel Corporation
 *
 * = AKNOWLEDGEMENTS
 *      
 *
 * = CREATION TIME
 *      1/9/2002 1:56:24 PM
 *
 * = CHANGE HISTORY
 *
 * ============================================================================
 */


/* INCLUDES */
#include "ix_ossl.h" 

#ifdef __linux
#include <linux/slab.h>
#else
#include <stdio.h>
#include "internal-osslThreadUtils.h"
#include "os_api.h"
#include "string.h"
#endif
 
/* FUNCTIONS */

/**
 * NAME:  ix_ossl_sem_init
 *
 * DESCRIPTION: This function initializes a new semaphore. 'sid' is a 
 *              pointer to an ix_ossl_sem_t. Upon success, '*sid' will be
 *              the semaphore id used in all other ix_ossl_sem 
 *              functions. The newly created semaphore will be initialized
 *              the value of 'start_value'.
 *
 * @Param:  - IN start_value:  initial value of the semaphore
 * 
 * @Param:  - OUT sid: Address where the newly created semaphore id will
 *                     returned.
 *
 * @Return: IX_OSSL_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */

ix_error ix_ossl_sem_init(int start_value, ix_ossl_sem_t* sid) 
{

#ifdef __linux
    *sid = kmalloc(sizeof(struct semaphore), GFP_KERNEL);
    if ( !*sid)
	return IX_OSSL_ERROR_OS_CALL_FAILURE;

    sema_init(*sid, start_value);

    return IX_OSSL_ERROR_SUCCESS;
#else
	int retVal;
    os_error osError;


#ifdef DEBUG
    printf(" in ossl_sem_init \n");
#endif /* DEBUG */
    
    retVal = os_thread_sema_create( start_value,sid, &osError);
    
    if(retVal != OS_SUCCESS){
#ifdef DEBUG
	   printf("ix_ossl_sema_create: sem_init() failed\n"
                            "\t[%d] %s\n", osError.errnum, osError.errname);
#endif /* DEBUG */
	   return IX_OSSL_ERROR_NEW(retVal,IX_ERROR_LEVEL_LOCAL);    
	}

    return IX_OSSL_ERROR_SUCCESS;
#endif
}

/**
 * NAME:  ix_ossl_sem_take
 *
 * DESCRIPTION: If the semaphore is 'empty', the calling thread is blocked. 
 *              If the semaphore is 'full', it is taken and control is returned
 *              to the caller. If the time indicated in 'timeout' is reached, 
 *              the thread will unblock and return an error indication. If the
 *              timeout is set to 'IX_OSSL_WAIT_NONE', the thread will never block;
 *              if it is set to 'IX_OSSL_WAIT_FOREVER', the thread will block until
 *              the semaphore is available. 
 *
 *
 * @Param:  - IN sid:  semaphore id.
 * @Param:  - IN timeout:  timeout value of type ix_uint32 expressed in miliseconds
 *
 * @Return: IX_OSSL_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */
ix_error ix_ossl_sem_take(ix_ossl_sem_t sid, ix_uint32 timeout)
{

#ifdef __linux

    ix_error ret = IX_OSSL_ERROR_SUCCESS;

    switch (timeout)
    {
        case IX_OSSL_WAIT_FOREVER:
            down(sid);
	    break;
	    
        case IX_OSSL_WAIT_NONE:
	    if (down_trylock(sid))
                ret = IX_OSSL_ERROR_OS_CALL_FAILURE;
	    break;
	    
        default:
	{
	    unsigned int time = jiffies + timeout;
            while (1)
            {
                if (!down_trylock(sid))
		{
		    break;
		}
                else
		{
                    if (time < jiffies)
		    {
                        ret = IX_OSSL_ERROR_OS_CALL_FAILURE;
                        break;
		    }
                }
                schedule();
	    }
	}
    }
    
    return ret;

#else
    int retVal;
    os_error osError;

    retVal = os_thread_sema_P(&sid,timeout, &osError);

    if(retVal != OS_SUCCESS) {
#ifdef DEBUG
	   printf("ix_ossl_sem_take: os_thread_sema_P failed\n"
               "\t[%d] %s\n", osError.errnum, osError.errname);
#endif /* DEBUG */	 
	   return IX_OSSL_ERROR_NEW(retVal,IX_ERROR_LEVEL_LOCAL);  	 
	}

    return IX_OSSL_ERROR_SUCCESS;    
#endif
}

/**
 * NAME:   ix_ossl_sem_give
 *
 * DESCRIPTION: This function causes the next available thread in the pend queue
 *              to be unblocked. If no thread is pending on this semaphore, the 
 *              semaphore becomes 'full'. 
 *
 * @Param:  - IN sid:  semaphore id.
 *
 * @Return: IX_OSSL_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */
ix_error ix_ossl_sem_give(ix_ossl_sem_t sid)
{
#ifdef __linux
    up(sid);
    return IX_OSSL_ERROR_SUCCESS;
#else

    int retVal;
    os_error osError;

	retVal = os_thread_sema_V(&sid, &osError);
    if(retVal != OS_SUCCESS) {
#ifdef DEBUG	      
		printf("ix_ossl_sem_give: os_thread_semaV() failed\n"
                            "\t[%d] %s\n", osError.errnum, osError.errname);
#endif /* DEBUG */	       
		return IX_OSSL_ERROR_NEW(retVal,IX_ERROR_LEVEL_LOCAL);
		   
    }

    return IX_OSSL_ERROR_SUCCESS;
#endif
}


/**
 * NAME:   ix_ossl_sem_fini
 *
 * DESCRIPTION: This function frees a semaphore.'sid' is semaphore id.  
 *              The semaphore is terminated, all resources are freed. 
 *              The threads pending on this semaphore will be released 
 *              and return an error.
 *
 * @Param:  - IN sid:  semaphore id
 *
 * @Return: IX_OSSL_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */
ix_error ix_ossl_sem_fini(ix_ossl_sem_t  sid)
{ 
#ifdef __linux
    return IX_OSSL_ERROR_OS_CALL_FAILURE;
#else

	int retVal;
    os_error osError;

    retVal = os_thread_sema_destroy(&sid, &osError);

    if(retVal != OS_SUCCESS) {
			    
		printf("ix_ossl_sem_fini: os_thread_destroy() failed\n"
                            "\t[%d] %s\n", osError.errnum, osError.errname);				
		return IX_OSSL_ERROR_NEW(retVal,IX_ERROR_LEVEL_LOCAL);
    }     
    return IX_OSSL_ERROR_SUCCESS;
#endif
}


/**
*   OS Semaphore APIs
*
*   Contains the VxWorks wrapper functions that allows for initialization,
*   posting, waiting and deletion of thread semaphores.
*
*   Function: os_thread_sema_create
*             os_thread_sema_P
*             os_thread_sema_flush
*             os_thread_sema_V
*             os_thread_sema_destroy
*
 * Author: Intel Corporation
*
*/


/**
 * NAME: os_thread_sema_create
 *
 * DESCRIPTION: Create a thread semaphore object.
 *
 * @Param:  - IN    value that the semaphore should be initialize to.
 *
 * @Param:  - OUT   id of the thread semaphore created.
 * @Param:  - OUT   osError   pointer to the datastructure where the 
 *                  error conditions are returned.
 */
int  os_thread_sema_create(
                           int value, 
                           ix_ossl_sem_t* sid, 
                           os_error* osError
                          )
{
#ifdef __linux
    return IX_OSSL_ERROR_OS_CALL_FAILURE;
#else
  
	int retVal; 
	SEM_ID semId; 
  
	semId = semCCreate(SEM_Q_FIFO,value); 

	if (semId == NULL)	{
		retVal = OS_NOMEM;
		osError->errnum = retVal;
     	strcpy(osError->errname,"ENOMEM");
      	return retVal;	
	}
  
	else {		
	
		*sid = (ix_ossl_sem_t) semId;	
		return OS_SUCCESS; 
	}
	return IX_OSSL_ERROR_OS_CALL_FAILURE;
#endif  
}

/**
 * NAME: os_thread_sema_P
 *
 * DESCRIPTION: Waits on the specified semaphore until the units are available
 *              or timeout occurs. 
 *
 * @Param:  - IN    sid      pointer  to the semaphore object.
 * @Param:  - IN    timeout  timeout value.
 * @Param:  - OUT   osError  pointer to the datastructure where the 
 *                           error conditions are returned.
 */
int  os_thread_sema_P(
                      ix_ossl_sem_t* sid,
                      ix_uint32 timeout,
                      os_error* osError
                     )
{
#ifdef __linux
    return IX_OSSL_ERROR_OS_CALL_FAILURE;
#else
	
	int retVal;
  
	retVal = semTake(*(SEM_ID *)sid,timeout);
	switch(retVal)	{
	case SEMA_SUCCESS :
		{ 
			return OS_SUCCESS;	
		}
 
	case S_objLib_OBJ_TIMEOUT :
    	{
			osError->errnum = retVal;
			strcpy(osError->errname,"S_objLib_OBJ_TUMEOUT");
			return OS_TIMEDOUT;
		}
	case S_objLib_OBJ_ID_ERROR : 
		{
			/* The value specified by attr is invalid */
			osError->errnum = retVal;
			strcpy(osError->errname,"S_objLib_OBJ_ID_ERROR");
			return OS_INVALID_ATTR;
		}
	case S_objLib_OBJ_UNAVAILABLE : 
		{    
			osError->errnum = retVal;
			strcpy(osError->errname,"S_objLib_OBJ_UNAVAILABLE");
			return OS_NOSYS;
		}
  
	case S_intLib_NOT_ISR_CALLABLE :
		{
			osError->errnum = retVal;
			strcpy(osError->errname,"S_intLib,NOT_ISR_CALLABLE");
			return OS_ERR_INTR;
		}
	default: 
		{  
			osError->errnum = retVal;
			strcpy(osError->errname,"UNKNOWN_ERROR");
      		return IX_OSSL_ERROR_OS_CALL_FAILURE;    
		}
	}   
#endif	
}


/**
 * NAME: os_thread_sema_V
 *
 * DESCRIPTION: This function releases the specified  semaphore and the
 *              semaphore state becomes available after this function call.
 *
 * @Param:  - IN      sid       pointer to the thread semaphore object.
 * @Param:  - OUT     osError   pointer to the datastructure where the 
 *                              error conditions are returned.
 *
 */
int  os_thread_sema_V(ix_ossl_sem_t* sid, os_error* osError)
{
#ifdef __linux
    return IX_OSSL_ERROR_OS_CALL_FAILURE;
#else

	int retVal;

	retVal = semGive(*(SEM_ID *)sid);

	switch(retVal){
	case SEMA_SUCCESS  :   
		{	
			return OS_SUCCESS;
		}
	case  S_objLib_OBJ_ID_ERROR :
		{			    
			/* The value specified by attr is invalid */     
			osError->errnum = retVal;    
			strcpy(osError->errname,"S_objLib_OBJ_ID_ERROR");     
			return OS_INVALID_ATTR;   
		}
	case  S_semLib_INVALID_OPERATION :
		{           
			osError->errnum = retVal; 
			strcpy(osError->errname,"S_semLib_INVALID_OPERATION");  
			return OS_INVALID_OPERATION;  
		}
 		  
	case  S_intLib_NOT_ISR_CALLABLE :
		{
			osError->errnum = retVal;
			strcpy(osError->errname,"S_intLib_NOT_ISR_CALLABLE");
			return OS_ERR_INTR;
		}
	default:  
		{   
			osError->errnum = retVal;
			strcpy(osError->errname,"UNKNOWN_ERROR");     
			return IX_OSSL_ERROR_OS_CALL_FAILURE;  
		}  
	}    
#endif
}


/**
 * NAME:  os_thread_sema_destroy
 *
 * DESCRIPTION: Destroy the thread semaphore object.
 *
 * @Param:  - IN   sid       pointer to the thread semaphore object.
 * @Param:  - OUT  osError   pointer to the datastructure where the 
 *                           error conditions are returned.
 */
int  os_thread_sema_destroy(ix_ossl_sem_t* sid, os_error* osError)
{
#ifdef __linux
    return IX_OSSL_ERROR_OS_CALL_FAILURE;
#else

	int retVal;
   
	retVal = semDelete(*(SEM_ID *)sid); 
	switch(retVal){
 
	case SEMA_SUCCESS  :   
		{
			return OS_SUCCESS;      
		}
   
	case  S_objLib_OBJ_ID_ERROR : 
		{      
			/* The value specified by attr is invalid */
			osError->errnum = retVal;
			strcpy(osError->errname,"S_objLib_OBJ_ID_ERROR"); 
			return OS_INVALID_ATTR;  
		}
	case  S_smObjLib_NO_OBJECT_DESTROY :  
		{     
			osError->errnum = retVal;  
			strcpy(osError->errname,"S_semLib_INVALID_OPERATION");  
			return OS_INVALID_OPERATION; 
		}  
	case  S_intLib_NOT_ISR_CALLABLE : 
		{
			osError->errnum = retVal;
			strcpy(osError->errname,"S_intLib_NOT_ISR_CALLABLE"); 
			return OS_ERR_INTR;
		}
	default: 
		{
			osError->errnum = retVal;
			strcpy(osError->errname,"UNKNOWN_ERROR");
			return IX_OSSL_ERROR_OS_CALL_FAILURE;   
		} 
	}   
#endif
}

