/**
 * @file IxNpeDlImageMgr.c
 *
 * @author Intel Corporation
 * @date 09 January 2002
 *
 * @brief This file contains the implementation of the private API for the 
 *        IXP425 NPE Downloader ImageMgr module
 *
 * @version $Revision: 1.1.1.1 $
 * 
 * @par
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * @par
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * @par
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * @par
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * @par
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * @par
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * @par
 * -- End Intel Copyright Notice --
*/


/*
 * Put the system defined include files required.
 */
#include <stdio.h>


/*
 * Put the user defined include files required.
 */
#include "IxNpeDlImageMgr_p.h"
#include "IxNpeDlMacros_p.h"

/*
 * define the flag which toggles the firmare inclusion
 */
#define IX_NPE_MICROCODE_FIRMWARE_INCLUDED 1
#include "IxNpeMicrocode.h"


/*
 * Typedefs whose scope is limited to this file.
 */

typedef struct
{
    UINT32 size;
    UINT32 offset;
    UINT32 id;
} IxNpeDlImageMgrImageEntry;

typedef union
{
    IxNpeDlImageMgrImageEntry image;
    UINT32 eohMarker;
} IxNpeDlImageMgrHeaderEntry;

typedef struct
{
    UINT32 signature;
    /* 1st entry in the header (there may be more than one) */
    IxNpeDlImageMgrHeaderEntry entry[1];
} IxNpeDlImageMgrImageLibraryHeader;

/* module statistics counters */
typedef struct
{
    UINT32 invalidSignature;
    UINT32 imageIdListOverflow;
    UINT32 imageIdNotFound;
} IxNpeDlImageMgrStats;


/*
 * Variable declarations global to this file only.  Externs are followed by
 * static variables.
 */
static IxNpeDlImageMgrStats ixNpeDlImageMgrStats;

/* default image */
static UINT32 *IxNpeMicroCodeImageLibrary = IX_NPEDL_MicrocodeImageLibrary; 


/*
 * static function prototypes.
 */
PRIVATE BOOL
ixNpeDlImageMgrSignatureCheck (UINT32 *microCodeImageLibrary);

PRIVATE void  
ixNpeDlImageMgrImageIdFormat (UINT32 rawImageId, IxNpeDlImageId *imageId);

PRIVATE BOOL
ixNpeDlImageMgrImageIdCompare (IxNpeDlImageId *imageIdA, 
				 IxNpeDlImageId *imageIdB);
				 
PRIVATE BOOL
ixNpeDlImageMgrNpeFunctionIdCompare (IxNpeDlImageId *imageIdA,
    				       IxNpeDlImageId *imageIdB);


/*
 * Function definition: ixNpeDlImageMgrMicrocodeImageLibraryOverride
 */
IX_STATUS
ixNpeDlImageMgrMicrocodeImageLibraryOverride (
    UINT32 *clientImageLibrary)
{
    IX_STATUS status = IX_SUCCESS;

    IX_NPEDL_TRACE0 (IX_NPEDL_FN_ENTRY_EXIT, 
		     "Entering ixNpeDlImageMgrMicrocodeImageLibraryOverride\n");

    if (ixNpeDlImageMgrSignatureCheck (clientImageLibrary))
    {
	IxNpeMicroCodeImageLibrary = clientImageLibrary;
    }
    else
    {
	IX_NPEDL_ERROR_REPORT ("ixNpeDlImageMgrMicrocodeImageLibraryOverride: "
			       "Client-supplied image has invalid signature\n");
	status = IX_FAIL;
    }

    IX_NPEDL_TRACE1 (IX_NPEDL_FN_ENTRY_EXIT, 
		     "Exiting ixNpeDlImageMgrMicrocodeImageLibraryOverride: status = %d\n",
		     status);
    return status;
}


/*
 * Function definition: ixNpeDlImageMgrImageListExtract
 */
IX_STATUS
ixNpeDlImageMgrImageListExtract (
    IxNpeDlImageId *imageListPtr,
    UINT32 *numImages)
{
    UINT32 rawImageId;
    IxNpeDlImageId formattedImageId;
    IX_STATUS status = IX_SUCCESS;
    UINT32 imageCount = 0;
    IxNpeDlImageMgrImageLibraryHeader *header;

    IX_NPEDL_TRACE0 (IX_NPEDL_FN_ENTRY_EXIT, 
		     "Entering ixNpeDlImageMgrImageListExtract\n");

    header = (IxNpeDlImageMgrImageLibraryHeader *) IxNpeMicroCodeImageLibrary;

    if (ixNpeDlImageMgrSignatureCheck (IxNpeMicroCodeImageLibrary))
    {
	/* for each image entry in the image header ... */
	while (header->entry[imageCount].eohMarker !=
	       IX_NPEDL_IMAGEMGR_END_OF_HEADER)
	{
	    /*
	     * if the image list container from calling function has capacity,
	     * add the image id to the list 
	     */
	    if ((imageListPtr != NULL) && (imageCount < *numImages))
	    {
		rawImageId = header->entry[imageCount].image.id;
	        ixNpeDlImageMgrImageIdFormat (rawImageId, &formattedImageId);
		imageListPtr[imageCount] = formattedImageId;
	    }
	    /* imageCount reflects no. of image entries in image library header */
	    imageCount++;  
	}
	
	/*
	 * if image list container from calling function was too small to
	 * contain all image ids in the header, set return status to FAIL
	 */
	if ((imageListPtr != NULL) && (imageCount > *numImages))
	{
	    status = IX_FAIL;
	    IX_NPEDL_ERROR_REPORT ("ixNpeDlImageMgrImageListExtract: "
				   "number of Ids found exceeds list capacity\n");
	    ixNpeDlImageMgrStats.imageIdListOverflow++;
	}
	/* return number of image ids found in image library header */
	*numImages = imageCount;  
    }
    else
    {
	status = IX_FAIL;
	IX_NPEDL_ERROR_REPORT ("ixNpeDlImageMgrImageListExtract: "
			       "invalid signature in image\n");
    }
    
    IX_NPEDL_TRACE1 (IX_NPEDL_FN_ENTRY_EXIT, 
		     "Exiting ixNpeDlImageMgrImageListExtract: status = %d\n",
		     status);
    return status;
}


/*
 * Function definition: ixNpeDlImageMgrImageLocate
 */
IX_STATUS
ixNpeDlImageMgrImageLocate (
    IxNpeDlImageId *imageId,
    UINT32 **imagePtr,
    UINT32 *imageSize)
{
    UINT32 imageOffset;
    UINT32 rawImageId;
    IxNpeDlImageId formattedImageId;
    /* used to index image entries in image library header */
    UINT32 imageCount = 0;   
    IX_STATUS status = IX_FAIL;
    IxNpeDlImageMgrImageLibraryHeader *header;

    IX_NPEDL_TRACE0 (IX_NPEDL_FN_ENTRY_EXIT,
		     "Entering ixNpeDlImageMgrImageLocate\n");

    header = (IxNpeDlImageMgrImageLibraryHeader *) IxNpeMicroCodeImageLibrary;

    if (ixNpeDlImageMgrSignatureCheck (IxNpeMicroCodeImageLibrary))
    {
	/* for each image entry in the image library header ... */
	while (header->entry[imageCount].eohMarker !=
	       IX_NPEDL_IMAGEMGR_END_OF_HEADER)
	{
	    rawImageId = header->entry[imageCount].image.id;
	    ixNpeDlImageMgrImageIdFormat (rawImageId, &formattedImageId);
	    /* if a match for imageId is found in the image library header... */
	    if (ixNpeDlImageMgrImageIdCompare (imageId, &formattedImageId))
	    {
		/*
		 * get pointer to the image in the image library using offset from
		 * 1st word in image library
		 */
		imageOffset = header->entry[imageCount].image.offset;
		*imagePtr = &IxNpeMicroCodeImageLibrary[imageOffset];
		/* get the image size */
		*imageSize = header->entry[imageCount].image.size;
		status = IX_SUCCESS;
		break;
	    }
	    imageCount++;
	}
	if (status != IX_SUCCESS)
	{
	    IX_NPEDL_ERROR_REPORT ("ixNpeDlImageMgrImageLocate: "
				   "imageId not found in image library header\n");
	    ixNpeDlImageMgrStats.imageIdNotFound++;
	}
    }
    else
    {
	IX_NPEDL_ERROR_REPORT ("ixNpeDlImageMgrImageLocate: "
			       "invalid signature in image library\n");
    }

    IX_NPEDL_TRACE1 (IX_NPEDL_FN_ENTRY_EXIT,
		     "Exiting ixNpeDlImageMgrImageLocate: status = %d\n", status);
    return status;
}

/*
 * Function definition: ixNpeDlImageMgrLatestImageExtract
 */
IX_STATUS
ixNpeDlImageMgrLatestImageExtract (IxNpeDlImageId *imageId)
{
    UINT32 imageCount = 0; 
    UINT32 rawImageId;
    IxNpeDlImageId formattedImageId;
    IX_STATUS status = IX_FAIL;
    IxNpeDlImageMgrImageLibraryHeader *header;
    
    
    IX_NPEDL_TRACE0 (IX_NPEDL_FN_ENTRY_EXIT,
		     "Entering ixNpeDlImageMgrLatestImageExtract\n");
		     
    header = (IxNpeDlImageMgrImageLibraryHeader *) IxNpeMicroCodeImageLibrary;
    
    if (ixNpeDlImageMgrSignatureCheck (IxNpeMicroCodeImageLibrary))
    {
	/* for each image entry in the image library header ... */
	while (header->entry[imageCount].eohMarker !=
	       IX_NPEDL_IMAGEMGR_END_OF_HEADER)
	{
	    rawImageId = header->entry[imageCount].image.id;
	    ixNpeDlImageMgrImageIdFormat (rawImageId, &formattedImageId);
	    /* 
	     * if a match for the npe Id and functionality Id of the imageId is
	     *  found in the image library header... 
	     */
            if(ixNpeDlImageMgrNpeFunctionIdCompare(imageId, &formattedImageId))
            {
                if(imageId->major <= formattedImageId.major)
                {
                    if(imageId->minor < formattedImageId.minor)
                    {
                        imageId->minor = formattedImageId.minor;
                    }
                    imageId->major = formattedImageId.major;
                }
                status = IX_SUCCESS;
            }
            imageCount++;
	}
	if (status != IX_SUCCESS)
	{
	    IX_NPEDL_ERROR_REPORT ("ixNpeDlImageMgrLatestImageExtract: "
				   "imageId not found in image library header\n");
	    ixNpeDlImageMgrStats.imageIdNotFound++;
	}
    }
    else
    {
	IX_NPEDL_ERROR_REPORT ("ixNpeDlImageMgrLatestImageGet: "
			       "invalid signature in image library\n");
    }

    IX_NPEDL_TRACE1 (IX_NPEDL_FN_ENTRY_EXIT,
		     "Exiting ixNpeDlImageMgrLatestImageGet: status = %d\n", status);
    return status;
}

/*
 * Function definition: ixNpeDlImageMgrSignatureCheck
 */
PRIVATE BOOL
ixNpeDlImageMgrSignatureCheck (UINT32 *microCodeImageLibrary)
{
    IxNpeDlImageMgrImageLibraryHeader *header =
	(IxNpeDlImageMgrImageLibraryHeader *) microCodeImageLibrary;
    BOOL result = TRUE;

    if (header->signature != IX_NPEDL_IMAGEMGR_SIGNATURE)
    {
	result = FALSE;
	ixNpeDlImageMgrStats.invalidSignature++;
    }

    return result;
}


/*
 * Function definition: ixNpeDlImageMgrImageIdFormat
 */
PRIVATE void
ixNpeDlImageMgrImageIdFormat (
    UINT32 rawImageId, 
    IxNpeDlImageId *imageId)
{  
    imageId->npeId = (rawImageId >>
				IX_NPEDL_IMAGEID_NPEID_OFFSET) &
	IX_NPEDL_NPEIMAGE_FIELD_MASK;
    imageId->functionalityId = (rawImageId >> 
				  IX_NPEDL_IMAGEID_FUNCTIONID_OFFSET) &
	IX_NPEDL_NPEIMAGE_FIELD_MASK;
    imageId->major = (rawImageId >>
				IX_NPEDL_IMAGEID_MAJOR_OFFSET) &
	IX_NPEDL_NPEIMAGE_FIELD_MASK;
    imageId->minor = (rawImageId >>
				IX_NPEDL_IMAGEID_MINOR_OFFSET) &
	IX_NPEDL_NPEIMAGE_FIELD_MASK;

}


/*
 * Function definition: ixNpeDlImageMgrImageIdCompare
 */
PRIVATE BOOL
ixNpeDlImageMgrImageIdCompare (
    IxNpeDlImageId *imageIdA,
    IxNpeDlImageId *imageIdB)
{
    if ((imageIdA->npeId   == imageIdB->npeId)   &&
	(imageIdA->functionalityId == imageIdB->functionalityId) &&
	(imageIdA->major   == imageIdB->major)   &&
	(imageIdA->minor   == imageIdB->minor))
    {
	return TRUE;
    }
    else
    {
	return FALSE;
    }
}

/*
 * Function definition: ixNpeDlImageMgrNpeFunctionIdCompare
 */
PRIVATE BOOL
ixNpeDlImageMgrNpeFunctionIdCompare (
    IxNpeDlImageId *imageIdA,
    IxNpeDlImageId *imageIdB)
{
    if ((imageIdA->npeId   == imageIdB->npeId)   &&
	(imageIdA->functionalityId == imageIdB->functionalityId))
    {
	return TRUE;
    }
    else
    {
	return FALSE;
    }
}


/*
 * Function definition: ixNpeDlImageMgrStatsShow
 */
void
ixNpeDlImageMgrStatsShow (void)
{
    printf ("\nixNpeDlImageMgrStatsShow:\n");

    printf ("\tInvalid Image Signatures: %u\n",
	    ixNpeDlImageMgrStats.invalidSignature);
    printf ("\tImage Id List capacity too small: %u\n",
	    ixNpeDlImageMgrStats.imageIdListOverflow);
    printf ("\tImage Id not found: %u\n",
	    ixNpeDlImageMgrStats.imageIdNotFound);
    
    printf ("\n");
}


/*
 * Function definition: ixNpeDlImageMgrStatsReset
 */
void
ixNpeDlImageMgrStatsReset (void)
{
    ixNpeDlImageMgrStats.invalidSignature = 0;
    ixNpeDlImageMgrStats.imageIdListOverflow = 0;
    ixNpeDlImageMgrStats.imageIdNotFound = 0;
}


/*
 * Function definition: ixNpeDlImageMgrImageFind
 */
IX_STATUS
ixNpeDlImageMgrImageFind (
    UINT32 *imageLibrary,
    UINT32 imageId,
    UINT32 **imagePtr,
    UINT32 *imageSize)
{
    UINT32 imageOffset;
    /* used to index image entries in image library header */
    UINT32 imageCount = 0;   
    IX_STATUS status = IX_FAIL;
    IxNpeDlImageMgrImageLibraryHeader *header;
    BOOL imageFound = FALSE;

    IX_NPEDL_TRACE0 (IX_NPEDL_FN_ENTRY_EXIT,
		     "Entering ixNpeDlImageMgrImageFind\n");


    /* If user didn't specify a library to use, use the default
     * one from IxNpeMicrocode.h
     */
    if (imageLibrary == NULL)
    {
	imageLibrary = IxNpeMicroCodeImageLibrary;
    }
    
    if (ixNpeDlImageMgrSignatureCheck (imageLibrary))
    {
	header = (IxNpeDlImageMgrImageLibraryHeader *) imageLibrary;
    
	/* for each image entry in the image library header ... */
	while ((header->entry[imageCount].eohMarker !=
               IX_NPEDL_IMAGEMGR_END_OF_HEADER) && !(imageFound))
	{
	    /* if a match for imageId is found in the image library header... */
	    if (imageId == header->entry[imageCount].image.id)
	    {
		/*
		 * get pointer to the image in the image library using offset from
		 * 1st word in image library
		 */
		imageOffset = header->entry[imageCount].image.offset;
		*imagePtr = &imageLibrary[imageOffset];
		/* get the image size */
		*imageSize = header->entry[imageCount].image.size;
		status = IX_SUCCESS;
		imageFound = TRUE;
	    }
	    imageCount++;
	}
	if (status != IX_SUCCESS)
	{
	    IX_NPEDL_ERROR_REPORT ("ixNpeDlImageMgrImageFind: "
				   "imageId not found in image library header\n");
	    ixNpeDlImageMgrStats.imageIdNotFound++;
	}
    }
    else
    {
	IX_NPEDL_ERROR_REPORT ("ixNpeDlImageMgrImageFind: "
			       "invalid signature in image library\n");
    }

    IX_NPEDL_TRACE1 (IX_NPEDL_FN_ENTRY_EXIT,
		     "Exiting ixNpeDlImageMgrImageFind: status = %d\n", status);
    return status;
}
