/**
 * ============================================================================
 * = COPYRIGHT
 * File Version: $Revision: 1.1.1.1 $
 * 
 * -- Intel Copyright Notice --
 * 
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * -- End Intel Copyright Notice --
 *
 * = PRODUCT
 *      Intel(r) IXP425 Software Release
 *
 * = LIBRARY
 *      OSSL - Operating System Services  Library
 *
 * = MODULE
 *      OSSL  Abstraction layer header file
 *
 * = FILENAME
 *       ix_ossl.h
 *
 * = DESCRIPTION
 *   This file contains the prototypes of OS-independent wrapper
 *   functions which allow the programmer not to be tied to a specific
 *   operating system. The OSSL functions can be divided into three classes:
 *
 *   1) synchronization-related wrapper functions around thread system calls
 *   2) thread-related wrapper functions around thread calls
 *   3) transactor/workbench osapi calls -- defined in osApi.h
 *
 *   Both 1 and 2 classes of functions provide Thread Management, Thread 
 *   Synchronization, Mutual Exclusion and  Timer primitives. Namely, 
 *   creation and deletion functions as well as the standard "wait" and
 *   "exit". Additionally, a couple of utility functions which enable to 
 *   pause the execution of a thread are also provided.
 *
 *   The 3rd class provides a  slew of other OSAPI functions to handle
 *   Transactor/WorkBench OS calls.
 *
 *
 *   OSSL Thread APIs:
 *             The OSSL thread functions that allow for thread creation,
 *             get thread id, thread deletion and set thread priroity.
 *
 *             ix_ossl_thread_create
 *             ix_ossl_thread_get_id
 *             ix_ossl_thread_exit
 *             ix_ossl_thread_kill
 *             ix_ossl_thread_set_priority
 *             ix_ossl_thread__delay
 * 
 *   OSSL Semaphore APIs:
 *             The OSSL semaphore functions that allow for initialization,
 *             posting, waiting and deletion of semaphores.
 *
 *             ix_ossl_sem_init
 *             ix_ossl_sem_fini
 *             ix_ossl_sem_take
 *             ix_ossl_sem_give
 *             ix_ossl_sem_flush
 *
 *   OSSL Mutex APIs:
 *             The OSSL wrapper functions that allow for initialization,
 *             posting, waiting and deletion of mutexes.
 *
 *             ix_ossl_mutex_init
 *             ix_ossl_mutex_fini
 *             ix_ossl_mutex_lock
 *             ix_ossl_mutex_unlock
 *
 *   OSSL Timer APIs:
 *     	       The timer APIs provide sleep and get time functions.
 *
 *             ix_ossl_sleep
 *             ix_ossl_sleep_tick
 *             ix_ossl_time_get
 * 
 *   OSAPIs for Transactor/WorkBench:
 *             These OSAPI functions are used for transator OS calls. 
 *             They are defined in osApi.h. 
 *
 *             Sem_Init 			
 *             Sem_Destroy 		
 *             Sem_Wait 			
 *             Sem_Wait			
 *             Thread_Create		
 *             Thread_Cancel 		
 *             Thread_SetPriority 		
 *             delayMs 			
 *             delayTick			
 *             
 *
 *
 **********************************************************************
 *      
 *
 * = AUTHOR
 *      Intel Corporation
 *
 * = ACKNOWLEDGEMENTS
 *      
 *
 * = CREATION TIME
 *      1/8/2002 1:53:42 PM
 *
 * = CHANGE HISTORY
 *   02/22/2002 : Renamed osapi.h os_api.h
 *                Moved OS header file includes from OSSL.h to os_api.h
 *                Moved OS specific datatypes to os_api.h 
 *                Modified data types, macros and functions as per
 *                'C' coding guidelines.
 *
 *
 * ============================================================================
 */

#ifndef _IX_OSSL_H
#ifndef __doxygen_hide
#define _IX_OSSL_H
#endif /* __doxygen_hide */

#ifdef __cplusplus
extern "C"{
#endif

/* Header file includes */
#ifdef __vxworks
#include <private/timerLibP.h> /* needed for IX_OSSL_TIME_* macros */
#endif

#include "ix_os_type.h"
#include "ix_types.h"
#include "ix_error.h"
#include "ix_symbols.h"
#include "os_datatypes.h"

/* Following header file is required for transactor/workbench OSAPI functions */
/* #include "os_api.h"*/ 

/**
 * @defgroup IxOSSL IXP425 Operating System Services Library (IxOSSL) API
 *
 * @brief This service provides a layer of OS dependency services. 
 *
 *   This file contains the prototypes of OS-independent wrapper
 *   functions which allow the programmer not to be tied to a specific
 *   operating system. The OSSL functions can be divided into three classes:
 *
 *   1) synchronization-related wrapper functions around thread system calls
 *   2) thread-related wrapper functions around thread calls
 *   3) transactor/workbench osapi calls -- defined in osApi.h
 *
 *   Both 1 and 2 classes of functions provide Thread Management, Thread 
 *   Synchronization, Mutual Exclusion and  Timer primitives. Namely, 
 *   creation and deletion functions as well as the standard "wait" and
 *   "exit". Additionally, a couple of utility functions which enable to 
 *   pause the execution of a thread are also provided.
 *
 *   The 3rd class provides a slew of other OSAPI functions to handle
 *   Transactor/WorkBench OS calls.
 *
 *   @note WHEREVER POSSIBLE, PLEASE USE THE EQUIVALENT API FUNCTION
 *         FROM THE @ref IxOsServices COMPONENT INSTEAD IF ONE EXISTS.
 * @{
 */

/* OSSL Data Structures */

/**
 * @typedef ix_ossl_thread_t
 *
 * @brief This type defines OSSL thread type.
 *
 */
typedef os_thread_t     	ix_ossl_thread_t; 

/**
 * @typedef ix_ossl_sem_t
 * 
 * @brief This type defines OSSL semaphore type.
 *
 */
typedef os_sem_t         	ix_ossl_sem_t;

/**
 * @typedef ix_ossl_mutex_t
 * 
 * @brief This type defines OSSL mutex type.
 *
 */
typedef os_mutex_t	ix_ossl_mutex_t;

/**
 * @brief This type defines OSSL time.
 *
 * The ix_ossl_time_t struct has two fields. 'sec' 
 * and 'nsec'. Time value is computed as: sec * 10^9 + nsec 
 *
 */
typedef struct
{
    unsigned long tv_sec; /**< seconds */
    unsigned long tv_nsec; /**< nanoseconds [1, 999,999,999] */
} ix_ossl_time_t;




/* OSSL function types */

/**
 * @typedef ix_ossl_thread_entry_point_t
 * 
 * @brief This function type defines OSSL thread
 *              entry point function.
 *. 
 * @param void * arg (in) - pointer to a custom thread argument
 * @param void ** ptrRetObj (out) - address where a pointer to a custom data
 *                                  structure or object will be returned by the
 *                                  thread on exit.
 * 
 */
typedef ix_error (*ix_ossl_thread_entry_point_t)(
                                                  void* arg, 
                                                  void** ptrRetObj
                                                );

/**
 * @brief This type defines thread main info.
 *
 * The struct has two fields: threadMain
 * and threadArg. The threadMain is pointer to
 * thread entry point function. threadArg is 
 * a void pointer to a user defined thread 
 * entry point argument.  
 *
 */
typedef struct
{
    ix_ossl_thread_entry_point_t threadMain; /**< pointer to thread entry point function */
    void* threadArg;     /**< void pointer to a user defined thread entry point argument */
} ix_ossl_thread_main_info_t;


/* Enumerations are defined here */

/**
 * @enum ix_ossl_error_code
 * 
 * @brief This type defines error codes returned by OSSL calls.
 *
 */
typedef enum
{
    IX_OSSL_ERROR_SUCCESS =  0, /**< success */
    IX_OSSL_ERROR_INVALID_ARGUMENTS = OS_INVALID_ATTR, /**< invalid arguments */
    IX_OSSL_ERROR_INVALID_OPERATION,            /**< invalid operation */
    IX_OSSL_ERROR_THREAD_CALL_FAILURE,          /**< thread operation failed */
    IX_OSSL_ERROR_INVALID_PID,                  /**< invalid process id */
    IX_OSSL_ERROR_INVALID_TID,                  /**< invalid thread id */
    IX_OSSL_ERROR_OS_CALL_FAILURE = OS_FAILURE, /**< invalid arguments */
    IX_OSSL_ERROR_TIMEOUT = OS_TIMEDOUT,        /**< OS operation failed */
    IX_OSSL_ERROR_NOMEM = OS_NOMEM,             /**< memory unavailable */
    IX_OSSL_ERROR_NOSYS = OS_NOSYS              /**< system resource unavailable */
} ix_ossl_error_code;


/**
 * @enum ix_ossl_sem_state
 * 
 * @brief This type defines OSSL binary semaphore states.
 *
 */
typedef enum
{
    IX_OSSL_SEM_UNAVAILABLE = OS_SEM_UNVAILABLE,  /**< Semaphore unavailable */
    IX_OSSL_SEM_AVAILABLE = OS_SEM_AVAILABLE      /**< Semaphore available   */
} ix_ossl_sem_state;



/**
 * @enum ix_ossl_mutex_state
 * 
 * @brief This type defines OSSL mutex states.
 *
 */
typedef enum
{
    IX_OSSL_MUTEX_UNLOCK = OS_MUTEX_UNLOCK,    /**< Mutex unlocked */
    IX_OSSL_MUTEX_LOCK = OS_MUTEX_LOCK         /**< Mutex locked */
} ix_ossl_mutex_state;


/**
 * @enum ix_ossl_thread_priority
 * 
 * @brief This type define OSSL thread priority levels.
 *
 */
typedef enum
{
    IX_OSSL_THREAD_PRI_HIGH = OS_THREAD_PRI_HIGH,     /**< High priority thread */
    IX_OSSL_THREAD_PRI_MEDIUM = OS_THREAD_PRI_MEDIUM, /**< Medium priority thread */
    IX_OSSL_THREAD_PRI_LOW = OS_THREAD_PRI_LOW        /**< Low priority thread */
} ix_ossl_thread_priority;


/* Pre-processor Symbols */

/**
 * @def IX_OSSL_ERROR_SUCCESS
 *
 * @brief This symbol defines an error token that indicates the successful
 *        completion of the OSSL calls.
 *
 */
#define IX_OSSL_ERROR_SUCCESS     (ix_error)0UL 

/**
 * @def IX_OSSL_ERROR_FAILURE
 *
 * @brief This symbol defines an error token that indicates the failed
 *        completion of the OSSL calls.
 *
 */
#if defined __linux
#define IX_OSSL_ERROR_FAILURE 	  (ix_error)1
#endif


/**
 * @def IX_OSSL_WAIT_FOREVER
 *
 * @brief This symbol is useful for specifying an 'indefinite wait'
 * timeout value in ix_ossl_sem_take and ix_ossl_mutex_lock function calls
 *
 */
#define IX_OSSL_WAIT_FOREVER    OS_WAIT_FOREVER /* (-1)UL */

/**
 * @def IX_OSSL_WAIT_NONE
 *
 * @brief This symbol is useful for specifying a 'no wait'
 * timeout value in ix_ossl_sem_take and ix_ossl_mutex_lock function calls
 *
 */
#define IX_OSSL_WAIT_NONE       OS_WAIT_NONE     /* 0 */

/**
 *  OSSL TIME Macros :
 *  Following macros provide functions for ix_time handling. 
 *  These macros will be used for different operations on 
 *  'ix_ossl_time_t' data type:
 *  '==' , '>' , '<' , '+' , '-' , 'normalize' , 'isvalid'
 *  'iszero' , and 'conversion to os ticks'.
 */

/**
 *  Note: For the following macros, 'true' is a non-zero 
 *        integer value and 'false' is a zero' integer
 *        value. The result has type 'int'.
 *
 */

/**
 * @def BILLION
 *
 * @brief Define a constant for the value 1 billion, used by OSSL TIME macros.
 *        Equivalent to (1000 million nanoseconds / second)
 */
#define BILLION         1000000000


/**
 * @def TICKS_PER_NSEC
 *
 * @brief The number of OS Ticks per nano-second on Linux
 */
#ifdef __linux
#define TICKS_PER_NSEC  (BILLION/HZ)
#endif


/**
 * @def IX_OSSL_TIME_EQ
 *
 * @brief Compares a operand with b operand. Returns true if 
 *              they are equal and false otherwise.
 *
 * @param a ix_ossl_time_t (in) - operand a
 * @param b ix_ossl_time_t (in) - operand b
 *
 * @return true if a == b and false otherwise.
 */
#define IX_OSSL_TIME_EQ(a,b) \
        ((a).tv_sec == (b).tv_sec && (a).tv_nsec == (b).tv_nsec)

/**
 * @def IX_OSSL_TIME_GT
 *
 * @brief Compares a operand with b operand. Returns true if
 *        a > b, and false otherwise.
 *
 * @param a ix_ossl_time_t (in) - operand a
 * @param b ix_ossl_time_t (in) - operand b
 *
 * @return true if a > b and false otherwise.
 */

#define IX_OSSL_TIME_GT(a,b)  \
        ((a).tv_sec  > (b).tv_sec ||	\
        ((a).tv_sec == (b).tv_sec && (a).tv_nsec > (b).tv_nsec))

/**
 * @def IX_OSSL_TIME_LT
 *
 * @brief Compares a operand with b operand. Returns true
 *        if a < b,  and false otherwise. 
 *            
 * @param a ix_ossl_time_t (in) - operand a
 * @param b ix_ossl_time_t (in) - operand b
 *
 * @return true if a < b and false otherwise.
 */	
#define IX_OSSL_TIME_LT(a,b) \
        ((a).tv_sec  < (b).tv_sec ||	\
        ((a).tv_sec == (b).tv_sec && (a).tv_nsec < (b).tv_nsec))

/**
 * @def IX_OSSL_TIME_ISZERO
 *
 * @brief This macro checks if the operand a is zero. Returns
 *        true if a is zero (both sec and nsec fields must be zero)
 *        and false otherwise. 
 *
 * @param a ix_ossl_time_t (in) - operand a
 *
 * @return  true if a is zero and false otherwise.
 */	
#define IX_OSSL_TIME_ISZERO(a) \
        ((a).tv_sec == 0 && (a).tv_nsec == 0)

/**
 * @def IX_OSSL_TIME_SET
 *
 * @brief This macro sets operand a to the value of operand b.
 *
 * @param a ix_ossl_time_t (out) - operand a
 * @param b ix_ossl_time_t (in)  - operand b
 *
 */
#define IX_OSSL_TIME_SET(a,b) \
        (a).tv_sec = (b).tv_sec; (a).tv_nsec = (b).tv_nsec

/**
 * @def IX_OSSL_TIME_ADD
 *
 * @brief This macro performs  a += b operation. 
 *        
 * @param a ix_ossl_time_t (in|out) - operand a
 * @param b ix_ossl_time_t (in)     - operand b
 *
 */	    
#define IX_OSSL_TIME_ADD(a,b)       \
        (a).tv_sec += (b).tv_sec;   \
        (a).tv_nsec += (b).tv_nsec; \
        if ((a).tv_nsec >= BILLION) \
	{ \
        (a).tv_sec++; \
        (a).tv_nsec -= BILLION; }

/**
 * @def IX_OSSL_TIME_SUB
 *
 * @brief This macro performs  a -= b operation.
 *
 * @param a ix_ossl_time_t (in|out) - operand a
 * @param b ix_ossl_time_t (in)     - operand b
 *
 */		
#define IX_OSSL_TIME_SUB(a,b)        	\
        if ((a).tv_nsec >= (b).tv_nsec) \
        { \
          (a).tv_sec -= (b).tv_sec; \
          (a).tv_nsec -= (b).tv_nsec; \
        } \
        else \
        { \
          (a).tv_sec -= ((b).tv_sec + 1); \
          (a).tv_nsec += BILLION - (b).tv_nsec; \
        }

/**
 * @def IX_OSSL_TIME_NORMALIZE
 *
 * @brief This macro normalizes the value of a. If 'a.nsec' > 10^9,
 *        it is decremented by 10^9 and 'a.sec' is incremented by 1.
 *
 * @param a ix_ossl_time_t (in|out) - operand a
 *
 */	
#define	IX_OSSL_TIME_NORMALIZE(a) 	\
        if ((a).tv_nsec >= BILLION)	\
	    { (a).tv_sec++; (a).tv_nsec -= BILLION; }	\
	else if ((a).tv_nsec < 0)	\
	    { (a).tv_sec--; (a).tv_nsec += BILLION; }

/**
 * @def IX_OSSL_TIME_VALID
 *
 * @brief This macro checks whether a is a valid ix_ossl_time_t i.e.
 *        0 =< a.nsec < 10^9. Returns true if a is valid and
 *        false otherwise.
 *
 * @param a ix_ossl_time_t (in) - operand a
 *
 * @return  true if a is valid ix_ossl_time_t and false otherwise
 *
 */
#define IX_OSSL_TIME_VALID(a) \
        ((a).tv_nsec >= 0 && (a).tv_nsec < BILLION)
	

/**
 * @def IX_OSSL_TIME_ZERO
 *
 * @brief This macro sets the value of a to zero.
 *
 * @param a ix_ossl_time_t (in|out) - operand a
 *
 */
#define IX_OSSL_TIME_ZERO(a)   	\
        (a).tv_sec = 0; (a).tv_nsec = 0

/**
 * @def IX_OSSL_TIME_CONVERT_TO_TICK
 *
 * @brief This macro converts b value in ix_ossl_time_t to
 *        a value in os ticks.
 *
 * @param a unsigned int (out)  - operand a
 * @param b ix_ossl_time_t (in) - operand b
 *
 */
#ifdef __linux
#define IX_OSSL_TIME_CONVERT_TO_TICK(a,b)   \
              (a) = (b).tv_sec * HZ + \
	      (b).tv_nsec / TICKS_PER_NSEC + \
	      (((b).tv_nsec % TICKS_PER_NSEC > (TICKS_PER_NSEC/2)) ? 1 : 0)
#else
#define IX_OSSL_TIME_CONVERT_TO_TICK(a,b)   \
              TV_CONVERT_TO_TICK(a,b)
#endif



/*  OSSL  functions */

/**
 * @brief create a thread
 *
 * This function creates a cancellable thread that will 
 * execute the user-provided entry point function. Custom 
 * arguments can be passed to this function using the "arg"
 * argument. 
 *
 * @param entryPoint ix_ossl_thread_entry_point_t (in) -  thread's entry point
 *                                                        function.
 * @param arg void* (in) - pointer to custom arguments that
 *                         will be passed to entry point function
 *                         as the first argument.
 *
 * @param ptrTid ix_ossl_thread_t* (out) - address at which the thread id of the
 *                                         newly created thread will be returned
 *
 * @return IX_OSSL_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */
IX_EXPORT_FUNCTION 
ix_error ix_ossl_thread_create(
                                ix_ossl_thread_entry_point_t entryPoint,
                                void* arg,
                                ix_ossl_thread_t* ptrTid
                              );

/**
 * @brief get id of calling thread
 *
 * This function returns id of the calling thread.
 *
 * @param ptrTid ix_ossl_thread_t* (out) - address at which the id of the
 *                                         calling thread will be returned
 *
 * @return IX_OSSL_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */
IX_EXPORT_FUNCTION
ix_error ix_ossl_thread_get_id(ix_ossl_thread_t* ptrTid);


/**
*   @brief wrapper for user-provided thread function
*
*   This function provides the needed pthread-compliant
*   entry point function. It basically acts as a wrapper
*   around the user-provided function. It still does one
*   important thing which is to set the cancellation type
*   of the thread to ASYNCHRONOUS (which translates into
*   instantaneous")
*
*   @param ptrThreadInfo void* (in) - pointer to our temporary structure
*                                     containing pointers to the thread's
*                                     entry point function and its
*                                     argument structure
*
*   @return void *
*/
IX_EXPORT_FUNCTION
void* ix_ossl_thread_main_wrapper(void* ptrThreadInfo);


/**
 * @brief Causes the calling thread to exit
 *
 * This function causes the calling thread to exit. It
 * gives the opportunity to the caller to pass back to
 * a waiting parent a pointer to a custom data structure
 * and an ix_error token.
 *
 * @param retError ix_error (in) - ix_error token to be returned to the waiting
 *                                 parent (0 if no error is returned)
 *
 * @param retObj void* (in) - pointer to custom data structure returned to
 *                            the waiting parent on thread exit.It is used for
 *                            post-mortem debugging.(null if no data structure
 *                            is returned)
 *
 * @return none
 */
IX_EXPORT_FUNCTION
void ix_ossl_thread_exit(ix_error retError, void* retObj);

/**
 * @brief kills the specified thread
 *
 * Kills the running thread specified by its thread id.
 * Because the thread will be killed instantly, the caller 
 * must be extremely careful when using this function as 
 * the thread will not have time to release any of the 
 * resources it is currently owning. ix_ossl_thread_exit
 * should be used to delete a thread and its resources instead!. 
 *
 * @param tid ix_ossl_thread_t (in) - id of the thread to be killed
 *
 * @return  IX_OSSL_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 *
 * @warning This function does not kill linux kernel threads. It only sends
 * the SIG_TERM signal and it is the resposibility of the thread to check for
 * this signal (see signal_pending) and terminate itself.
 */
IX_EXPORT_FUNCTION 
ix_error ix_ossl_thread_kill(ix_ossl_thread_t tid);



/**
 * @brief sets the priority of the specified thread
 *
 * This function sets the priority of the indicated thread.
 * Possible values for 'priority' are IX_OSSL_THREAD_PRI_HIGH,
 * IX_OSSL_THREAD_PRI_MED, and IX_OSSL_THREAD_PRI_LOW.
 *
 * @param tid ix_ossl_thread_t (in) - id of the thread
 * @param priority ix_ossl_thread_priority (in) - valid priority values are:
 *                                              IX_OSSL_THREAD_PRI_HIGH,
 *                                              IX_OSSL_THREAD_PRI_MED, and
 *                                              IX_OSSL_THREAD_PRI_LOW.
 *
 * @return  IX_OSSL_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */
IX_EXPORT_FUNCTION
ix_error ix_ossl_thread_set_priority(
                                     ix_ossl_thread_t tid,
                                     ix_ossl_thread_priority priority
                                    );


/**
 * @brief delay the current task for specified number of OS ticks
 *
 * This function causes the current task to delay for the specified
 * number of OS ticks. Control of the CPU is relinquished during this time
 * allowing other system tasks a chance to execute.
 *
 *
 * @param ticks int (in) - number of OS ticks to delay task.
 *
 * @return IX_OSSL_ERROR_SUCCESS if successful or IX_OSSL_ERROR_FAILURE for failure.
 */
IX_EXPORT_FUNCTION
ix_error ix_ossl_thread_delay(
                              int ticks
			     );


/**
 * @brief creates a thread (use @ref ix_ossl_thread_create instead)
 *
 * Create a cancellable thread that will execute the
 * user-provided entry point function. Custom arguments
 * can be passed to this function using the "arg"
 * argument.
 *
 * @param start_routine void * (*)(void *) (in ) - pointer to thread's entry
 *                                                 point function
 *
 * @param ptrThreadInfo ix_ossl_thread_main_info_t* (in) - pointer to custom
 *                                                   argument structure that
 *                                      will be passed to entry point function
 *
 * @param ptrTid ix_ossl_thread_t* (out) - address at which the thread id of the
 *                                         newly created thread will be returned
 * @param osError os_error* (out) - pointer to the datastructure where OS
 *                                  error codes are returned.
 */
IX_EXPORT_FUNCTION
int  os_thread_create(
                      void * (*start_routine)(void *),
                      ix_ossl_thread_main_info_t *ptrThreadInfo,
                      ix_ossl_thread_t* ptrTid,
                      os_error* osError
                     );


/**
 * @brief get thread id (use @ref ix_ossl_thread_get_id instead)
 *
 * Returns the thread id of the calling thread
 *
 * @param ptrTid ix_ossl_thread_t* (out) - address at which the thread id of
 *                                         the inquiring thread will be returned
 *
 */
IX_EXPORT_FUNCTION
int os_thread_get_id(ix_ossl_thread_t* ptrTid);


/**
 * @brief exits the calling thread (use @ref ix_ossl_thread_exit instead)
 *
 * This function causes the calling thread to exit. It
 * gives the opportunity (this is not a requirement!) to
 * the caller to pass back to a waiting parent a pointer
 * to a custom data structure and an ix_error token.
 *
 * @param ptrRetObj void* (out) - pointer to custom data structure
 *                                (null if no data structure is returned)
 *
 */
IX_EXPORT_FUNCTION
int os_thread_exit(void* ptrRetObj);


/**
 * @brief kill the specified thread (use @ref ix_ossl_thread_kill instead)
 *
 * Kills the running thread specified by its thread id.
 *
 * @param tid ix_ossl_thread_t (in) - id of the thread to be killed
 *
 * @param osError os_error* (out) - pointer to the datastructure where OS
 *                                  error conditions are returned.
 */
IX_EXPORT_FUNCTION
int  os_thread_kill(ix_ossl_thread_t tid, os_error* osError);


/**
 * @brief sets priority of a thread (use @ref ix_ossl_thread_set_priority instead)
 *
 * This function sets the priority of the indicated thread.
 * possible values for 'priority' are IX_OSSL_PRI_HIGH,
 * IX_OSSL_PRI_MED, and IX_OSSL_PRI_LOW. The effect of priority
 * is OS dependant.
 *
 * @param tid ix_ossl_thread_t* (in) - pointer to the thread object
 * @param priority ix_ossl_thread_priority (in) - priority  priority level.
 * @param osError os_error* (out) - pointer to the datastructure where OS
 *                                  error conditions are returned.
 *
 */
IX_EXPORT_FUNCTION
ix_error os_thread_set_priority(
                                ix_ossl_thread_t* tid,
                                ix_ossl_thread_priority priority,
			                    os_error* osError
                               );


/**
 * @brief gets number of OS ticks per second
 *
 * This function returns the number of os ticks per second.
 *
 * @param pticks int* (out) - pointer to location  where data will be returned.
 *
 * @return IX_OSSL_ERROR_SUCCESS.
 */
IX_EXPORT_FUNCTION
ix_error ix_ossl_tick_get(
			  int *pticks
			  );


/**
 * @brief causes the calling thread to sleep for specified time (milliseconds)
 *
 * This function causes the calling thread to sleep for the specified time.
 *
 * @param sleeptime_ms ix_uint32 (in) - sleep time specified in milliseconds.
 * @param osError os_error* (out) - pointer to the datastructure where the
 *                                  error conditions are returned.
 *
 */
IX_EXPORT_FUNCTION
int os_sleep(ix_uint32 sleeptime_ms, os_error* osError);


/**
 * @brief causes the calling thread to sleep for specified time (os ticks)
 *
 * This function causes the calling thread to sleep for the
 * time specified in OS ticks.
 *
 * @param sleeptime_ticks ix_uint32 (in) - sleep time specified in OS ticks.
 * @param osError os_error* (out) - pointer to the datastructure where the
 *                                  error conditions are returned.
 *
 */
IX_EXPORT_FUNCTION
int os_sleep_tick(ix_uint32 sleeptime_ticks, os_error *osError);


/**
 * @brief returns the system time with nano-second resolution
 *
 * This function places the current value of time, in
 * (seconds, nanoseconds), into the '*ptime' structure. This
 * function does not provide a time-of-day. The purpose is
 * to provide a nano-second resolution time.
 *
 * @param ptime ix_ossl_time_t* (out) - address to the time structure.
 * @param osError os_error* (out) - pointer to the datastructure where the
 *                                  error conditions are returned.
 *
 */
IX_EXPORT_FUNCTION
int os_time_get(ix_ossl_time_t *ptime, os_error *osError);


/**
 * @brief initialises a new semaphore
 *
 * This function initializes a new semaphore. 'sid' is a
 * pointer to an ix_ossl_sem_t. Upon success, '*sid' will be
 * the semaphore id used in all other ix_ossl_sem 
 * functions. The newly created semaphore will be initialized
 * the value of 'start_value'.
 *
 * @param start_value int (in) - initial value of the semaphore
 * 
 * @param sid ix_ossl_sem_t* (out) - Address where the newly created semaphore id
 *                                   will returned.
 *
 * @return IX_OSSL_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */
IX_EXPORT_FUNCTION 
ix_error ix_ossl_sem_init(int start_value, ix_ossl_sem_t* sid);

/**
 * @brief take a semaphore, and block if semaphore not available
 *
 * If the semaphore is 'empty', the calling thread is blocked. 
 * If the semaphore is 'full', it is taken and control is returned
 * to the caller. If the time indicated in 'timeout' is reached, 
 * the thread will unblock and return an error indication. If the
 * timeout is set to 'IX_OSSL_WAIT_NONE', the thread will never block;
 * if it is set to 'IX_OSSL_WAIT_FOREVER', the thread will block until
 * the semaphore is available. 
 *
 *
 * @param sid ix_ossl_sem_t (in) - semaphore id.
 * @param timeout ix_uint32 (in) - timeout value of type ix_uint32 expressed in
 *                                 miliseconds
 *
 * @return IX_OSSL_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */
IX_EXPORT_FUNCTION 
ix_error ix_ossl_sem_take(
                          ix_ossl_sem_t sid,
                          ix_uint32 timeout
                         );

/**
 * @brief give back a semaphore
 *
 * This function causes the next available thread in the pend queue
 * to be unblocked. If no thread is pending on this semaphore, the 
 * semaphore becomes 'full'. 
 *
 * @param sid ix_ossl_sem_t (in) - semaphore id.
 *
 * @return IX_OSSL_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */
IX_EXPORT_FUNCTION 
ix_error ix_ossl_sem_give(ix_ossl_sem_t sid);


/**
 * @brief unblocks all threads pending on the semaphore
 *
 * This function unblocks all pending threads
 * without altering the semaphore count. 'sid' 
 * is the id for the semaphore. '*result' will 
 * be non-zero if a thread was unblocked during
 * this call.
 *
 * @param sid ix_ossl_sem_t (in) - semaphore id
 * @param result int* (out) - the value referred will be non-zero if a
 *                            thread was unblocked during this call 
 *
 * @return IX_OSSL_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */
IX_EXPORT_FUNCTION 
ix_error ix_ossl_sem_flush(ix_ossl_sem_t sid, int* result);


/**
 * @brief terminate the semaphore (free semaphore resources)
 *
 * This function frees a semaphore.'sid' is semaphore id.  
 * The semaphore is terminated, all resources are freed. 
 * The threads pending on this semaphore will be released 
 * and return an error.
 *
 * @param sid ix_ossl_sem_t (in) - semaphore id
 *
 * @return IX_OSSL_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */
IX_EXPORT_FUNCTION
ix_error ix_ossl_sem_fini(ix_ossl_sem_t sid);


/**
 * @brief create a thread semaphore 
 *
 * Create a thread semaphore object.
 *
 * @param value int (in) - value that the semaphore should be initialize to.
 *
 * @param sid ix_ossl_sem_t* (out) - id of the thread semaphore created.
 * @param osError os_error* (out) - pointer to the datastructure where the
 *                                  error conditions are returned.
 */
IX_EXPORT_FUNCTION
int  os_thread_sema_create(
                           int value,
                           ix_ossl_sem_t* sid,
                           os_error* osError
                          );


/**
 * @brief pend on a semaphore (with timeout)
 *
 * Waits on the specified semaphore until the units are available
 * or timeout occurs.
 *
 * @param sid ix_ossl_sem_t* (in) - pointer  to the semaphore object.
 * @param timeout ix_uint32 (in)  - timeout value.
 * @param osError os_error* (out) - pointer to the datastructure where the
 *                                  error conditions are returned.
 */
IX_EXPORT_FUNCTION
int  os_thread_sema_P(
                      ix_ossl_sem_t* sid,
                      ix_uint32 timeout,
                      os_error* osError
                     );


/**
 * @brief release a semaphore
 *
 * This function releases the specified  semaphore and the
 * semaphore state becomes available after this function call.
 *
 * @param sid ix_ossl_sem_t* (in) - pointer to the thread semaphore object.
 * @param osError os_error* (out) - pointer to the datastructure where the
 *                                  error conditions are returned.
 */
IX_EXPORT_FUNCTION
int  os_thread_sema_V(ix_ossl_sem_t* sid, os_error* osError);


/**
 * @brief destroy semaphore object
 *
 * Destroy the thread semaphore object.
 *
 * @param sid ix_ossl_sem_t* (in) - pointer to the thread semaphore object.
 * @param osError os_error* (out) - pointer to the datastructure where the
 *                                  error conditions are returned.
 */
IX_EXPORT_FUNCTION
int  os_thread_sema_destroy(ix_ossl_sem_t* sid, os_error* osError);


/**
 * @brief initialises a new mutex object
 *
 * This function initializes a new mutex. 'mid' is a pointer
 * to an ix_ossl_mutex_t. Upon success, '*mid' will contain
 * the mutex id. 'start_state' is the initial locking
 * state.
 *
 * @note 
 *
 * @param start_state ix_ossl_mutex_state (in) - 'start_state' is  initial locking state.
 *                            Valid values are :-
 *                            'OSSL_MUTEX_LOCK': Lock the mutex
 *                            'OSSL_MUTEX_UNLOCK': Leave the mutex unlocked
 *
 * @param mid ix_ossl_mutex_t* (out) - pointer to id of the mutex created
 *
 * @return IX_OSSL_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */
IX_EXPORT_FUNCTION 
ix_error ix_ossl_mutex_init(ix_ossl_mutex_state start_state, ix_ossl_mutex_t* mid);


/**
 * @brief lock a mutex
 *
 * This function locks the mutex. If the mutex is already
 * locked, the thread will block. If the time indicated in
 * 'timeout' is reached, the thread will unblock and return
 * error indication. If the timeout is set to 'IX_OSSL_WAIT_NONE', 
 * the thread will never block (this is trylock). If the timeout
 * is set to 'IX_OSSL_WAIT_FOREVER',  the thread will block until
 * the mutex is  unlocked. 
 *
 * @param mid ix_ossl_mutex_t (in) - mutex id.
 * @param timeout ix_uint32 (in) - timeout value of type ix_uint32 expressed
 *                                 in miliseconds.
 *
 * @return IX_OSSL_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */
IX_EXPORT_FUNCTION 
ix_error ix_ossl_mutex_lock(
                            ix_ossl_mutex_t mid, 
                            ix_uint32 timeout
                           );


/**
 * @brief unlocks a mutex
 *
 * This function unlocks the mutex.  'mid' is the mutex id.
 * If there are threads pending on the mutex, the next one is 
 * given the lock. If there are no pending threads, then the 
 * mutex is unlocked.
 *
 * @param mid ix_ossl_mutex_t (in) - mutex id
 *
 * @return IX_OSSL_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */
IX_EXPORT_FUNCTION 
ix_error ix_ossl_mutex_unlock(ix_ossl_mutex_t mid);


/**
 * @brief free a mutex
 *
 * This function frees a mutex.'mid' is the id mutex id.
 * The mutex is deleted, all resources are freed. Any threads
 * pending on this mutex will be unblocked and return an error.
 *
 * @param mid ix_ossl_mutex_t (in) - mutex id
 *
 * @return IX_OSSL_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */
IX_EXPORT_FUNCTION
ix_error ix_ossl_mutex_fini(ix_ossl_mutex_t mid);


/**
 * @brief creates a thread mutex object
 *
 * Create a thread mutex object.
 *
 * See @ref ixOsServMutexInit for an alternative.
 *
 * @param start_state ix_ossl_mutex_state (in) - the value that the mutex state
 *                                               should be initialized to.
 *
 * @param mid ix_ossl_mutex_t* (out) - id of the thread mutex created.
 * @param osError os_error* (out) - pointer to the datastructure where the
 *                                  error conditions are returned.
 *
 */
IX_EXPORT_FUNCTION
int  os_thread_mutex_create(
                            ix_ossl_mutex_state start_state,
                            ix_ossl_mutex_t* mid,
                            os_error* osError
                           );


/**
 * @brief lock a mutex
 *
 * This function locks the mutex. If the mutex is already
 * locked, the task will block. If the time indicated in
 * 'timeout' is reached, the task will unblock and retun
 * error indication. If timeout is set to '0', the task will
 * never block.
 *
 * See @ref ixOsServMutexLock for an alternative.
 *
 * @param mutex ix_ossl_mutex_t* (in) - pointer to the thread mutex object.
 * @param timeout ix_uint32 (in) - 'timeout' value
 * @param osError os_error* (out) - pointer to the datastructure where the
 *                                  error conditions are returned.
 */
IX_EXPORT_FUNCTION
int  os_thread_mutex_lock(
                          ix_ossl_mutex_t* mutex,
                          ix_uint32 timeout,
                          os_error* osError
                         );


/**
 * @brief unlock a mutex
 *
 * This function unlocks the mutex. If there are tasks
 * pending on the mutex, the next one is given the lock.
 * If there are no pending tasks, then the mutex is
 * unlocked.
 *
 * See @ref ixOsServMutexUnlock for an alternative.
 *
 * @param mutex ix_ossl_mutex_t* (in) - pointer to the thread mutex object.
 * @param osError os_error* (out) - pointer to the datastructure where the
 *                                  error conditions are returned.
 */
IX_EXPORT_FUNCTION
int  os_thread_mutex_unlock(ix_ossl_mutex_t* mutex, os_error* osError);


/**
 * @brief destroy a mutex object
 *
 * Destroy the thread mutex object.
 *
 * see @ref ixOsServMutexDestroy for an alternative
 *
 * @param mutex ix_ossl_mutex_t* (in) - pointer to the thread mutex object.
 * @param osError os_error* (out) - pointer to the datastructure where the
 *                                  error conditions are returned.
 */
IX_EXPORT_FUNCTION
int  os_thread_mutex_destroy(ix_ossl_mutex_t* mutex, os_error* osError);


/**
 * @brief causes calling thread to sleep for specified time (milliseconds)
 *
 * This function causes the calling thread to sleep for the
 * specified time.
 *
 * @param sleeptime_ms ix_uint32 (in) - sleep time specified in milliseconds.
 *
 * @return IX_OSSL_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 *
 * @Warning: In VxWorks this function has a resolution dictated by sysClkRateGet().
 *           Very small delays could be truncated to 0 i.e. no delay at all.
 *           See ixOsServTaskSleep() for an alternative.
 */

ix_error ix_ossl_sleep(ix_uint32 sleeptime_ms);


/**
 * @brief causes calling thread to sleep for specified time (os ticks)
 *
 * This function causes the calling thread to sleep for the 
 * time specified in OS ticks.
 *
 * @param sleeptime_ticks ix_uint32 (in) - sleep time specified in os ticks.
 *
 * @return IX_OSSL_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */
IX_EXPORT_FUNCTION 
ix_error ix_ossl_sleep_tick(ix_uint32 sleeptime_ticks);


/**
 * @brief gets current system time with nano-second resolution
 *
 * This function places the current value of a timer,
 * in seconds and  nano-seconds, into an ix_ossl_time_t structure 
 * pointed by 'ptime'. This function does not provide a 
 * time-of-day. The intention is to provide a nano-second 
 * resolution time.
 *
 * @param ptime ix_ossl_time_t* (out) - pointer to 'ix_ossl_time_t' structure
 *                                      where data will be returned.
 *
 * @return IX_OSSL_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */
IX_EXPORT_FUNCTION 
ix_error ix_ossl_time_get(ix_ossl_time_t*  ptime);



/**
 * This section describe memory management functions
 */


/**
 * @typedef ix_ossl_size_t
 * 
 * @brief This type describes a generic size type.
 *
 */
typedef unsigned int ix_ossl_size_t;



/**
 * @brief allocate a block of memory
 *
 * This function will allocate a memory block.
 * the function returns a void pointer to the allocated space, or NULL if there is 
 * insufficient memory available. To return a pointer to a type other than void, use
 * a type cast on the return value. The storage space pointed to by the return value 
 * is guaranteed to be suitably aligned for storage of any type of object. If size is 0,
 * ix_ossl_malloc allocates a zero-length item in the heap and returns a valid pointer 
 * to that item. Always check the return from ix_ossl_malloc, even if the amount of memory
 * requested is small.
 * 
 * See @ref ixOsServCacheDmaAlloc for an alternative.
 *
 * @param arg_Size ix_ossl_size_t (in) - the size of the memory block requested.
 *
 * @return Returns a valid address if successful or a NULL for failure.
 */
IX_EXPORT_FUNCTION 
void* ix_ossl_malloc(
                      ix_ossl_size_t arg_Size
                    );




/**
 * @brief free a block of memory
 *
 * This function will free a memory block specified by the passed address.
 * The ix_ossl_free function deallocates a memory block (arg_pMemory) that was previously 
 * allocated by a call to ix_ossl_malloc.If arg_pMemory is NULL, the pointer is ignored 
 * and ix_ossl_free immediately returns. Attempting to free an invalid pointer (a 
 * pointer to a memory block that was not allocated by ix_ossl_malloc) may affect 
 * subsequent allocation requests and cause errors.
 *
 * See @ref ixOsServCacheDmaFree for an alternative.
 * 
 * @param arg_pMemory void* (in) - address of the memory block to be deallocated.
 */
IX_EXPORT_FUNCTION 
void ix_ossl_free(
                   void* arg_pMemory
                 );




/**
 * @brief copy number of bytes from one memory location to another
 *
 * This function will copy memory bytes between buffers.
 * The ix_ossl_memcpy function copies count bytes of arg_pSrc to arg_pDest. If the source and 
 * destination overlap, this function does not ensure that the original source bytes in the 
 * overlapping region are copied before being overwritten. 
 * 
 * @param arg_pDest void* (in|out)  - destination buffer address 
 * @param arg_pSrc const void* (in) - source buffer address
 * @param arg_Count ix_ossl_size_t  - number of bytes to copy
 *
 * @return Returns the address of the destination buffer.
 */
IX_EXPORT_FUNCTION 
void* ix_ossl_memcpy(
                      void* arg_pDest, 
                      const void* arg_pSrc,
                      ix_ossl_size_t arg_Count 
                    );



/**
 * @brief fill a region of memory with a specified byte value
 *
 * This function sets buffers to a specified character.
 * The ix_ossl_memset function sets the first arg_Count bytes of arg_pDest to the 
 * character arg_Char.
 * 
 * @param arg_pDest void* (in|out) - destination buffer address 
 * @param arg_pChar int (in) - character to set
 * @param arg_Count ix_ossl_size_t (in) - number of characters to set
 *
 * @return Returns the address of the destination buffer.
 */
IX_EXPORT_FUNCTION 
void* ix_ossl_memset(
                      void* arg_pDest, 
                      int arg_pChar,
                      ix_ossl_size_t arg_Count 
                    );



/**
 * Message logging API
 */

/**
 * @brief initialise the error message logging facility
 *
 * This function is used to initialize the error message logging. For each OS the 
 * messages will be logged into an implementation dependent stream. Further details 
 * will be provided on where the messages will be logged! This function should be called 
 * before any call to ix_ossl_message_log().
 *
 * See @ref ixOsServLogLevelSet for an alternative.
 * 
 * @return IX_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */
IX_EXPORT_FUNCTION 
ix_error ix_ossl_message_log_init(void);


/**
 * @brief log a printf-style message
 *
 * This routine is used to log a specified message.  This routine's  syntax is similar to
 * printf() - a format string is followed by  a variable number of arguments which will 
 * be interpreted and formated according to the fmt_string passed as the first argument.
 * Further details will be provided on where the messages will be logged!
 *
 * See @ref ixOsServLog for an alternative.
 *
 * @param arg_pFmtString char* (in) -  format string for the log message
 *
 * @return IX_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */
IX_EXPORT_FUNCTION 
ix_error ix_ossl_message_log(
                              char* arg_pFmtString, 
                              ...
                            );


#ifdef __cplusplus
}
#endif
 
#endif /* _IX_OSSL_H */

/**
 * @} defgroup IxOSSL
 */
