/**
 * @file IxOsServicesMemMap.h
 * 
 * @brief Header file for memory access maps
 * 
 * @par
 * @version $Revision: 1.1.1.1 $
 * 
 * @par
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright 1999-2003 Intel Corporation All Rights Reserved.
 * 
 * @par
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * @par
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * @par
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * @par
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * @par
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * @par
 * -- End Intel Copyright Notice --
 */

#ifndef IxOsServicesMemMap_H
#define IxOsServicesMemMap_H

#include <ixp425.h>
#include <IxTypes.h>
#include <IxOsServicesMemAccess.h>
 
/** 
 * @addtogroup IxOsServices 
 * @{ 
 */

/* map sizes to be used when requesting memory with IX_OS_SERV_MEM_MAP */

#define IX_OSSERV_QMGR_MAP_SIZE           (0x4000)     /**< Queue Manager map size */
#define IX_OSSERV_EXP_REG_MAP_SIZE        (0x1000)     /**< Exp Bus Registers map size */
#define IX_OSSERV_UART1_MAP_SIZE          (0x1000)     /**< UART1 map size */
#define IX_OSSERV_UART2_MAP_SIZE          (0x1000)     /**< UART2 map size */
#define IX_OSSERV_PMU_MAP_SIZE            (0x1000)     /**< PMU map size */
#define IX_OSSERV_OSTS_MAP_SIZE            (0x1000)    /**< OS Timers map size */
#define IX_OSSERV_NPEA_MAP_SIZE           (0x1000)     /**< NPE A map size */
#define IX_OSSERV_NPEB_MAP_SIZE           (0x1000)     /**< NPE B map size */
#define IX_OSSERV_NPEC_MAP_SIZE           (0x1000)     /**< NPE C map size */
#define IX_OSSERV_ETHA_MAP_SIZE           (0x1000)     /**< Eth A map size */
#define IX_OSSERV_ETHB_MAP_SIZE           (0x1000)     /**< Eth B map size */
#define IX_OSSERV_USB_MAP_SIZE            (0x1000)     /**< USB map size */
#define IX_OSSERV_GPIO_MAP_SIZE           (0x1000)     /**< GPIO map size */
#define IX_OSSERV_EXP_BUS_MAP_SIZE        (0x08000000) /**< Expansion bus map size */
#define IX_OSSERV_EXP_BUS_CS1_MAP_SIZE    (0x01000000) /**< CS1 map size */
#define IX_OSSERV_EXP_BUS_CS4_MAP_SIZE    (0x01000000) /**< CS4 map size */

/* physical addresses to be used when requesting
   memory with IX_OS_SERV_MEM_MAP */
#ifdef IX_OSSERV_LINUX_BE

#define IX_OSSERV_GPIO_PHYS_BASE           IXP425_GPIO_BASE_PHYS
#define IX_OSSERV_UART1_PHYS_BASE          IXP425_UART1_BASE_PHYS
#define IX_OSSERV_UART2_PHYS_BASE          IXP425_UART2_BASE_PHYS
#define IX_OSSERV_ETHA_PHYS_BASE           IXP425_EthA_BASE_PHYS
#define IX_OSSERV_ETHB_PHYS_BASE           IXP425_EthB_BASE_PHYS
#define IX_OSSERV_NPEA_PHYS_BASE           IXP425_NPEA_BASE_PHYS
#define IX_OSSERV_NPEB_PHYS_BASE           IXP425_NPEB_BASE_PHYS
#define IX_OSSERV_NPEC_PHYS_BASE           IXP425_NPEC_BASE_PHYS
#define IX_OSSERV_PERIPHERAL_PHYS_BASE     IXP425_PERIPHERAL_BASE_PHYS
#define IX_OSSERV_QMGR_PHYS_BASE           IXP425_QMGR_BASE_PHYS
#define IX_OSSERV_OSTS_PHYS_BASE           IXP425_TIMER_BASE_PHYS
#define IX_OSSERV_USB_PHYS_BASE            IXP425_USB_BASE_PHYS
#define IX_OSSERV_EXP_BUS_PHYS_BASE        IXP425_EXP_BUS_BASE2_PHYS
#define IX_OSSERV_EXP_BUS_BOOT_PHYS_BASE   IXP425_EXP_BUS_BASE1_PHYS
#define IX_OSSERV_EXP_BUS_CS1_PHYS_BASE    IXP425_EXP_BUS_CS1_BASE_PHYS
#define IX_OSSERV_EXP_BUS_CS4_PHYS_BASE    IXP425_EXP_BUS_CS4_BASE_PHYS
#define IX_OSSERV_EXP_BUS_REGS_PHYS_BASE   IXP425_EXP_CFG_BASE_PHYS

#elif defined (IX_OSSERV_VXWORKS_LE) || defined (IX_OSSERV_VXWORKS_BE)

#define IX_OSSERV_GPIO_PHYS_BASE           IXP425_GPIO_BASE
#define IX_OSSERV_UART1_PHYS_BASE          IXP425_UART1_BASE
#define IX_OSSERV_UART2_PHYS_BASE          IXP425_UART2_BASE
#define IX_OSSERV_ETHA_PHYS_BASE           IXP425_EthA_BASE
#define IX_OSSERV_ETHB_PHYS_BASE           IXP425_EthB_BASE
#define IX_OSSERV_NPEA_PHYS_BASE           IXP425_NPEA_BASE
#define IX_OSSERV_NPEB_PHYS_BASE           IXP425_NPEB_BASE
#define IX_OSSERV_NPEC_PHYS_BASE           IXP425_NPEC_BASE
#define IX_OSSERV_PERIPHERAL_PHYS_BASE     IXP425_PERIPHERAL_BASE
#define IX_OSSERV_QMGR_PHYS_BASE           IXP425_QMGR_BASE
#define IX_OSSERV_OSTS_PHYS_BASE           IXP425_OSTS
#define IX_OSSERV_USB_PHYS_BASE            IXP425_USB_BASE
#define IX_OSSERV_EXP_BUS_PHYS_BASE        IXP425_EXPANSION_BUS_BASE2
#define IX_OSSERV_EXP_BUS_BOOT_PHYS_BASE   IXP425_EXPANSION_BUS_BASE1
#define IX_OSSERV_EXP_BUS_CS1_PHYS_BASE    IXP425_EXPANSION_BUS_CS1_BASE
#define IX_OSSERV_EXP_BUS_CS4_PHYS_BASE    IXP425_EXPANSION_BUS_CS4_BASE
#define IX_OSSERV_EXP_BUS_REGS_PHYS_BASE   IXP425_EXP_CONFIG_BASE

#else

#error This operating system is not supported - only VxWorks and Linux are supported.

#endif

/* @} */

typedef enum
{
  IX_STATIC_MAP = 0, /**<Set map entry type to static map */
  IX_DYNAMIC_MAP /**<Set map entry type to dynamic map */
} IxOsServMapEntryType;

typedef enum
{
  IX_BE_MAP    = 0x1, /**<Set map coherency type to BE map */
  IX_LE_AC_MAP = 0x2, /**<Set map coherency type to AC map */
  IX_LE_DC_MAP = 0x4  /**<Set map coherency type to DC map */
} IxOsServMapCoherencyType;

/* determine coherency type required for the component 
   including this file */
#if defined (CSR_BE_MAPPING)
  
#define IX_COMPONENT_COHERENCY (IX_BE_MAP)
  
#elif defined (CSR_LE_ADDRESS_COHERENT_MAPPING)

#define IX_COMPONENT_COHERENCY (IX_LE_AC_MAP)

#elif defined (CSR_LE_DATA_COHERENT_MAPPING)

#define IX_COMPONENT_COHERENCY (IX_LE_DC_MAP)

#else

#error This component does not define a usable coherency mode in \
IxOsServicesComponents.h. Either define the coherency or do not \
include this file.

#endif

/* prototypes */
PUBLIC void *
ixOsServMemMap(UINT32 physicalAddress, UINT32 size, UINT32 coherency);

PUBLIC void
ixOsServMemUnmap(UINT32 virtualAddress, UINT32 coherency);

PUBLIC UINT32
ixOsServMemVirtToPhys(UINT32 virtualAddress, UINT32 coherency);

/**
 * @ingroup IxOsServices
 *
 * @def IX_OSSERV_MEM_MAP
 *
 * @brief Maps an I/O memory zone
 *
 * @param requestedAddress UINT32 (in) - physical address to map
 * @param size UINT32 (in) - size of map (should be large enough to hold the 
 * largest offset access made in this zone)
 *
 * This macro maps an I/O mapped physical memory zone of the given size
 * into a virtual memory zone accessible by the caller and returns a cookie - 
 * the start address of the virtual memory zone. 
 * IX_MMU_PHYS_TO_VIRT_TRANSLATION should NOT therefore be used on the returned
 * virtual address.
 * The memory zone should be unmapped using IX_OSSERV_MEM_UNMAP once the caller 
 * has finished using this zone (e.g. on driver unload) using the cookie as 
 * parameter.
 * The IX_OSSERV_READ/WRITE_LONG/SHORT macros should be used to read and write 
 * the mapped memory, adding the necessary offsets to the address cookie.
 *
 * @warning Not to be called from interrupt level
 *
 * @note The size parameter is only used for identifying a suitable (i.e. large
 * enough) map in the global memory map (ixOsServGlobalMemoryMap). 
 * It is NOT used to actually map the memory zone.
 * Instead, the zone indicated in the global memory map is used. 
 * Mapping will work only if the zone is predefined in the global memory map.
 *
 * @return the mapped virtual address or NULL if the operation could not be 
 * completed. This virtual address (cookie) has to be saved and used to 
 * unmap the memory zone during any clean-up or unload operation, using
 * the IX_OSSERV_MEM_UNMAP macro.
 */
#define IX_OSSERV_MEM_MAP(requestedPhysicalAddress, size) \
  ixOsServMemMap(requestedPhysicalAddress, size, IX_COMPONENT_COHERENCY)

/**
 * @ingroup IxOsServices
 *
 * @def IX_OSSERV_MEM_UNMAP
 *
 * @brief unmaps a previously mapped I/O memory zone
 *
 * @param requestedAddress UINT32 (in) - cookie (virtual address) to unmap
 *
 * This macro unmaps a previously mapped I/O memory zone using
 * the cookie obtained in the mapping operation. The memory zone in question 
 * becomes unavailable to the caller once unmapped and the cookie should be 
 * discarded.
 *
 * This macro cannot fail if the given parameter is correct and does not
 * return a value.
 *
 * @warning Not to be called from interrupt level
 */
#define IX_OSSERV_MEM_UNMAP(requestedVirtualAddress) \
  ixOsServMemUnmap(requestedVirtualAddress, IX_COMPONENT_COHERENCY)
  
/**
 * @ingroup IxOsServices
 *
 * @brief provides a mapped memory-aware virtual to physical translation
 *
 * @param virtualAddress UINT32 (in) - cookie (virtual address) to translate
 *
 * This macro searches through the global memory mapped for a virtualAddress
 * match; if found, it will return the physical address defined in the map.
 * Otherwise it will default to IX_MMU_VIRTUAL_TO_PHYSICAL_TRANSLATION.
 *
 * @return the physical address translation
 */
#define IX_OSSERV_MMAP_VIRT_TO_PHYS_TRANSLATION(virtualAddress) \
  ixOsServMemVirtToPhys(virtualAddress, IX_COMPONENT_COHERENCY)

typedef struct _IxOsServMemoryMap
{
  /* map type - IX_STATIC_MAP or IX_DYNAMIC_MAP;
     dynamic maps are allocated by the OS, static maps are only simulated
     by this component for portability */
  IxOsServMapEntryType type;
     
  /* physical address of the memory mapped I/O zone 
     - initialized in the global memory map */
  UINT32 physicalAddress;
  
  /* size of map - initialized in the global memory map */
  UINT32 size;
  
  /* virtual address of the zone; must be predefined in 
     the global memory map for static maps and has to be NULL 
     for dynamic maps (populated on allocation) */
  UINT32 virtualAddress;
  
  /* pointer to a map function called to map a dynamic map; 
     will populate the virtualAddress field */
  void (*mapFunction)(struct _IxOsServMemoryMap *map);
    
  /* pointer to a map function called to unmap a dynamic map; 
     will reset the virtualAddress field to NULL */
  void (*unmapFunction)(struct _IxOsServMemoryMap *map);
  
  /* reference count describing how many components share this map; 
     actual allocation/deallocation for dynamic maps is done only 
     between 0 <=> 1 transitions of the counter */
  UINT32 refCount;
     
  /* memory coherency for the map; can be a combination of IX_BE_MAP (Big 
     Endian) and IX_LE_AC_MAP or IX_LE_DC_MAP (Little Endian, Address
     Coherent or Data Coherent). Any combination is allowed provided it
     contains at most one LE flag - e.g. (IX_BE_MAP), (IX_LE_AC_MAP),
     (IX_BE_MAP | IX_LE_DC_MAP) are valid combinations while
     (IX_BE_MAP | IX_LE_DC_MAP | IX_LE_AC_MAP) is not. */
  IxOsServMapCoherencyType coherency;
     
  /* user-friendly name */
  char *name;
} IxOsServMemoryMap;
  
#if defined (__linux)

/* Linux-specific map/unmap functions to be used with dynamic maps */
#include <asm/io.h>
#include <linux/ioport.h>
void ixOsServLinuxMemMap(IxOsServMemoryMap *map);
void ixOsServLinuxMemUnmap(IxOsServMemoryMap *map);

/* prototypes for linux map/unmap functions */
PUBLIC void 
ixOsServLinuxMemMap(IxOsServMemoryMap *map);

PUBLIC void 
ixOsServLinuxMemUnmap(IxOsServMemoryMap *map);

#endif

#ifdef IxOsServicesMem_C

#if defined (__linux)

/* Note: - dynamic maps will be mapped using ioremap() with the base addresses and sizes declared in this array (matched to include the requested zones, 
           but not using the actual ixOsServMemMap requests) 
         - static maps have to be also declared in arch/arm/mach-ixp425/mm.c using the exact same addresses and size
         - the user-friendly name will be made available in /proc for dynamically allocated maps 
         - the declared order of the maps is important if the maps overlap - when a common zone is requested only the
           first usable map will be always chosen */
IxOsServMemoryMap ixOsServGlobalMemoryMap[] = 
{
  /* Queue Manager */
  { 
    IX_STATIC_MAP,                   /* type            */
    IXP425_QMGR_BASE_PHYS,           /* physicalAddress */
    IXP425_QMGR_REGION_SIZE,         /* size            */
    IXP425_QMGR_BASE_VIRT,           /* virtualAddress  */
    NULL,                            /* mapFunction     */
    NULL,                            /* unmapFunction   */
    0,                               /* refCount        */
    IX_BE_MAP,                       /* coherency       */
    "qMgr"                           /* name            */
  },

  /* APB Peripherals */
  { 
    IX_STATIC_MAP,                   /* type            */
    IXP425_PERIPHERAL_BASE_PHYS,     /* physicalAddress */
    IXP425_PERIPHERAL_REGION_SIZE,   /* size            */
    IXP425_PERIPHERAL_BASE_VIRT,     /* virtualAddress  */
    NULL,                            /* mapFunction     */
    NULL,                            /* unmapFunction   */
    0,                               /* refCount        */
    IX_BE_MAP,                       /* coherency       */
    "peripherals"                    /* name            */
  },

  /* Expansion Bus Config Registers */
  { 
    IX_STATIC_MAP,                   /* type            */
    IXP425_EXP_CFG_BASE_PHYS,        /* physicalAddress */
    IXP425_EXP_CFG_REGION_SIZE,      /* size            */
    IXP425_EXP_CFG_BASE_VIRT,        /* virtualAddress  */
    NULL,                            /* mapFunction     */
    NULL,                            /* unmapFunction   */
    0,                               /* refCount        */
    IX_BE_MAP,                       /* coherency       */
    "Exp Cfg"                        /* name            */
  },
  
  /* PCI config Registers */      
  {
    IX_STATIC_MAP,                   /* type            */
    IXP425_PCI_CFG_BASE_PHYS,        /* physicalAddress */
    IXP425_PCI_CFG_REGION_SIZE,      /* size            */
    IXP425_PCI_CFG_BASE_VIRT,        /* virtualAddress  */
    NULL,                            /* mapFunction     */
    NULL,                            /* unmapFunction   */
    0,                               /* refCount        */
    IX_BE_MAP,                       /* coherency       */
    "pciConfig"                      /* name            */
  },
        
  /* Expansion Bus */
  {
    IX_DYNAMIC_MAP,                  /* type            */
    IXP425_EXP_BUS_BASE_PHYS,        /* physicalAddress */
    IXP425_EXP_BUS_REGION_SIZE,      /* size            */
    0,                               /* virtualAddress  */
    ixOsServLinuxMemMap,             /* mapFunction     */
    ixOsServLinuxMemUnmap,           /* unmapFunction   */
    0,                               /* refCount        */
    IX_BE_MAP,                       /* coherency       */
    "Exp Bus"                        /* name            */
  }
};

#elif defined (__vxworks)

/* In VxWorks there's a 1:1 mapping between the virtual address space and the 
   physical address space, therefore all requests for a zone will be satisfied
   with the very same pointer 
   (cookie = requestedAddress - physicalAddress + virtualAddress, 
   and physicalAddress == virtualAddress == 0x00000000)
   and the "global" map is obviously large enough (0xFFFFFFFF) to host any zone,
   therefore it will be matched against every request not overrunning 4Gb.
   Note: this is valid only for the default LE coherency mode (AC). Maps
   aliasing I/O memory at different addresses in DC mode require additional
   definition - for an example see the qMgr LE_DC alias */
IxOsServMemoryMap ixOsServGlobalMemoryMap[] = 
{
  /* Global BE and LE_AC map */
  { 
    IX_STATIC_MAP,                   /* type            */
    0x00000000,                      /* physicalAddress */
    0xFFFFFFFF,                      /* size            */
    0x00000000,                      /* virtualAddress  */
    NULL,                            /* mapFunction     */
    NULL,                            /* unmapFunction   */
    0,                               /* refCount        */
    IX_BE_MAP | IX_LE_AC_MAP,        /* coherency       */
    "global"                         /* name            */
  },
        
  /* QMgr LE_DC map */
  { 
    IX_STATIC_MAP,                   /* type            */
    IXP425_QMGR_BASE,                /* physicalAddress */
    IXP425_QMGR_SIZE,                /* size            */
    IXP425_QMGR_LE_DC_VIRT,          /* virtualAddress  */
    NULL,                            /* mapFunction     */
    NULL,                            /* unmapFunction   */
    0,                               /* refCount        */
    IX_LE_DC_MAP,                    /* coherency       */
    "qMgr LE_DC"                     /* name            */
  },

  /* Expansion Bus Config LE_DC map */
  {
      IX_STATIC_MAP,                 /* type            */
      IXP425_EXP_CONFIG_BASE,        /* physicalAddress */
      IXP425_EXP_CONFIG_SIZE,        /* size            */
      IXP425_EXP_CONFIG_LE_DC_VIRT,  /* virtualAddress  */
      NULL,                          /* mapFunction     */
      NULL,                          /* unmapFunction   */
      0,                             /* refCount        */
      IX_LE_DC_MAP,                  /* coherency       */
      "Exp Cfg LE_DC"                /* name            */
  },
        
  /* Peripherals LE_DC map */
  { 
    IX_STATIC_MAP,                   /* type            */
    IXP425_PERIPHERAL_BASE,          /* physicalAddress */
    IXP425_PERIPHERAL_SIZE,          /* size            */
    IXP425_PERIPHERAL_LE_DC_VIRT,    /* virtualAddress  */
    NULL,                            /* mapFunction     */
    NULL,                            /* unmapFunction   */
    0,                               /* refCount        */
    IX_LE_DC_MAP,                    /* coherency       */
    "peripherals LE_DC"              /* name            */
  }
};

#else

#error This operating system is not supported - only VxWorks and Linux are supported.

#endif

#define IX_OSSERV_MEM_MAP_ELEMENTS (sizeof (ixOsServGlobalMemoryMap) / sizeof (IxOsServMemoryMap))

#endif /* IxOsServicesMem.c visible definitions */

#endif /* IxOsServicesMemMap_H */
