/**
 * @file IxFeatureCtrl.c
 *
 * @author Intel Corporation
 * @date 29-Jan-2003
 *
 * @brief Feature Control Public API Implementation
 *
 * @version $Revision: 1.1.1.1 $
 * 
 * @par
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * @par
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * @par
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * @par
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * @par
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * @par
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * @par
 * -- End Intel Copyright Notice --
*/


#include "IxAssert.h"
#include "IxOsServicesMemMap.h"
#include "IxOsServices.h"
#include "IxVersionId.h"
#include "IxFeatureCtrl.h"

/* Macro to read from the Feature Control Register */
#define IX_FEATURE_CTRL_READ(result) \
do { \
ixFeatureCtrlExpMap(); \
(result) = IX_OSSERV_READ_LONG(ixFeatureCtrlRegister); \
} while (0)

/* Macro to write to the Feature Control Register */
#define IX_FEATURE_CTRL_WRITE(value) \
do { \
ixFeatureCtrlExpMap(); \
IX_OSSERV_WRITE_LONG(ixFeatureCtrlRegister, (value)); \
} while (0)

/*
 * This is the offset of the feature register relative to the base of the
 * Expansion Bus Controller MMR.
 */
#define IX_FEATURE_CTRL_REG_OFFSET (0x00000028)


/* Boolean to mark the fact that the EXP_CONFIG address space was mapped */
PRIVATE BOOL ixFeatureCtrlExpCfgRegionMapped = FALSE;

/* Pointer holding the virtual address of the Feature Control Register */
PRIVATE VUINT32 *ixFeatureCtrlRegister = NULL;

/* Place holder to store the software configuration */
PRIVATE BOOL swConfiguration[IX_FEATURECTRL_SWCONFIG_MAX];

/* Flag to control swConfiguration[] is initialized once */
PRIVATE BOOL swConfigurationFlag = FALSE ;

/**
 * Forward declaration
 */
PRIVATE
void ixFeatureCtrlExpMap(void);

PRIVATE 
void ixFeatureCtrlSwConfigurationInit(void);

/**
 * Function to map EXP_CONFIG space
 */
PRIVATE
void ixFeatureCtrlExpMap(void)
{
    UINT32 expCfgBaseAddress = 0;

    /* If the EXP Configuration space has already been mapped then
     * return */
    if (ixFeatureCtrlExpCfgRegionMapped == TRUE)
    {
	return;
    }

    /* Map (get virtual address) for the EXP_CONFIG space */
    expCfgBaseAddress = (UINT32)
	(IX_OSSERV_MEM_MAP(IX_OSSERV_EXP_BUS_REGS_PHYS_BASE,
			   IX_OSSERV_EXP_REG_MAP_SIZE));

    /* Assert that the mapping operation succeeded */
    IX_ASSERT(expCfgBaseAddress);

    /* Set the address of the Feature register */
    ixFeatureCtrlRegister =
	(VUINT32 *) (expCfgBaseAddress + IX_FEATURE_CTRL_REG_OFFSET);

    /* Mark the fact that the EXP_CONFIG space has already been mapped */
    ixFeatureCtrlExpCfgRegionMapped = TRUE;
}

/**
 * Function definition: ixFeatureCtrlSwConfigurationInit
 * This function will only initialize software configuration once.
 */
PRIVATE void ixFeatureCtrlSwConfigurationInit(void)
{
  UINT32 i;
  if (FALSE == swConfigurationFlag)
  {
    for (i=0; i<IX_FEATURECTRL_SWCONFIG_MAX ; i++)
    {
        /* By default, all software configuration are enabled */
        swConfiguration[i]= TRUE ;
    }
    /*Make sure this function only initializes swConfiguration[] once*/
    swConfigurationFlag = TRUE ;
  }  
}

/**
 * Function definition: ixFeatureCtrlRead
 */
IxFeatureCtrlReg 
ixFeatureCtrlRead (void)
{
    IxFeatureCtrlReg result;

#if CPU!=SIMSPARCSOLARIS
    /* Read the feature control register */
    IX_FEATURE_CTRL_READ(result);
    return result;
#else
    /* Return an invalid value for VxWorks simulation */
    result = 0xFFFFFFFF;
    return result;
#endif
}

/**
 * Function definition: ixFeatureCtrlWrite
 */
void
ixFeatureCtrlWrite (IxFeatureCtrlReg expUnitReg)
{
#if CPU!=SIMSPARCSOLARIS
    /* Write value to feature control register */
    IX_FEATURE_CTRL_WRITE(expUnitReg);
#endif
}


/**
 * Function definition: ixFeatureCtrlHwCapabilityRead
 */
IxFeatureCtrlReg
ixFeatureCtrlHwCapabilityRead (void)
{ 
  IxFeatureCtrlReg currentReg, hwCapability;
  
  /* Capture a copy of feature control register */
  currentReg = ixFeatureCtrlRead(); 

  /* Try to enable all hardware components. 
   * Only software disable hardware can be enabled again */
  ixFeatureCtrlWrite(0);
  
  /* Read feature control register to know the hardware capability. */ 
  hwCapability = ixFeatureCtrlRead();
     
  /* Restore initial feature control value */
  ixFeatureCtrlWrite(currentReg);

  /* return Hardware Capability */
  return hwCapability;  
}


/**
 * Function definition: ixFeatureCtrlComponentCheck
 */
IX_STATUS 
ixFeatureCtrlComponentCheck (IxFeatureCtrlComponentType componentType)
{
  IxFeatureCtrlReg expUnitReg; 

  /* Read feature control register to know current hardware capability. */ 
  expUnitReg = ixFeatureCtrlRead();

  /* For example: To check for Hashing Coprocessor (bit-2) 
   *                   expUniteg    = 0x0010
   *                  ~expUnitReg   = 0x1101 
   *                  componentType = 0x0100
   *    ~expUnitReg & componentType = 0x0100 (Not zero)                      
   */
 
  /* 
   * Inverse the bit value because available component is 0 in value 
   */
  expUnitReg = ~expUnitReg ;

  if (expUnitReg & componentType)
  {
     return (IX_FEATURE_CTRL_COMPONENT_ENABLED);
  }   
  else
  {  
     return (IX_FEATURE_CTRL_COMPONENT_DISABLED);
  } 
}


/**
 * Function definition: ixFeatureCtrlProductIdRead
 */
IxFeatureCtrlProductId
ixFeatureCtrlProductIdRead ()
{
#if CPU!=SIMSPARCSOLARIS
  IxFeatureCtrlProductId  pdId = 0 ;
   
  /* Use ARM instruction to move register0 from coprocessor to ARM register */ 
    
  __asm("mrc p15, 0, %0, cr0, cr0, 0;" : "=r"(pdId) :); 

  return (pdId);
#else
  /* Return an invalid value for VxWorks simulation */
  return 0xffffffff;
#endif
}


/**
 * Function definition: ixFeatureCtrlSwConfigurationCheck
 */
IX_STATUS
ixFeatureCtrlSwConfigurationCheck (IxFeatureCtrlSwConfig swConfigType)
{
  if (swConfigType >= IX_FEATURECTRL_SWCONFIG_MAX)  
  {
     ixOsServLog(LOG_WARNING, "FeatureCtrl: Invalid software configuraiton input.\n",0,0,0,0,0,0);  
     return IX_FEATURE_CTRL_SWCONFIG_DISABLED;
  }

  /* The function will only initialize once. */
  ixFeatureCtrlSwConfigurationInit();
  
  /* Check and return software configuration */
  return  ((swConfiguration[(UINT32)swConfigType] == TRUE) ? IX_FEATURE_CTRL_SWCONFIG_ENABLED: IX_FEATURE_CTRL_SWCONFIG_DISABLED);
}

/**
 * Function definition: ixFeatureCtrlSwConfigurationWrite
 */
void
ixFeatureCtrlSwConfigurationWrite (IxFeatureCtrlSwConfig swConfigType, BOOL enabled)
{
  if (swConfigType >= IX_FEATURECTRL_SWCONFIG_MAX)  
  {
     ixOsServLog(LOG_WARNING, "FeatureCtrl: Invalid software configuraiton input.\n",0,0,0,0,0,0);  
     return;
  }

  /* The function will only initialize once. */
  ixFeatureCtrlSwConfigurationInit();
  
  /* Write software configuration */
  swConfiguration[(UINT32)swConfigType]=enabled ;
}

/**
 * Function definition: ixFeatureCtrlIxp400SwVersionShow
 */
void
ixFeatureCtrlIxp400SwVersionShow (void)
{
    printf ("\nIXP400 Software Release %s %s\n\n", IX_VERSION_ID, IX_VERSION_INTERNAL_ID);

}
