/**
 * @file IxEthDBAPISupport.c
 *
 * @brief Public API support functions
 * @version $Revision: 1.1.1.1 $
 * 
 * @par
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * @par
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * @par
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * @par
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * @par
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * @par
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * @par
 * -- End Intel Copyright Notice --
 */

#include <IxTypes.h>
#include <IxEthDB.h>
#include <IxNpeMh.h>
#include <IxFeatureCtrl.h>

#include "IxEthDB_p.h"
#include "IxEthDBMessages_p.h"
#include "IxEthDB_p.h"
#include "IxEthDBLog_p.h"

#ifdef IX_UNIT_TEST

int dbAccessCounter = 0;
int overflowEvent   = 0;

#endif

/*
 * External declaration 
 */
extern HashTable dbHashtable;

/*
 * Internal declaration 
 */
PortInfo ixEthDBPortInfo[IX_ETH_DB_NUMBER_OF_PORTS];

/**
 * @brief adds an entry to the database and triggers port updates
 *
 * @param portID port ID of the new entry
 * @param macAddr MAC address of the new entry
 * @param age age of the new entry (typically 0)
 * @param staticEntry use STATIC_ENTRY or DYNAMIC_ENTRY (static entries
 * are not subject to aging)
 *
 * This function is a wrapper over @ref ixEthDBAdd(), the only extra
 * functionality being the port update operation.
 *
 * @internal
 */
IX_ETH_DB_PUBLIC
IxEthDBStatus ixEthDBFilteringEntryAdd(IxEthDBPortId portID, IxEthDBMacAddr *macAddr, UINT8 age, BOOL staticEntry)
{
    MacDescriptor *newMacDescriptor = NULL;
    IxEthDBStatus result;

    result = ixEthDBAdd(portID, macAddr, age, staticEntry, &newMacDescriptor);
	
    if (newMacDescriptor != NULL)
    {
        IX_ETH_DB_SUPPORT_TRACE("DB: (Support) Added new MAC descriptor, triggering update\n");

	ixEthDBTriggerAddPortUpdate(portID, newMacDescriptor);
    }

    return result;
}

IX_ETH_DB_PUBLIC
void ixEthDBPortInit(IxEthDBPortId portID)
{
    PortInfo *portInfo;

    if (portID >= IX_ETH_DB_NUMBER_OF_PORTS)
    {
        return;
    }

    portInfo = &ixEthDBPortInfo[portID];

    if (ixEthDBSingleEthNpeCheck(portID) != IX_ETH_DB_SUCCESS)
    {
        WARNING_LOG("EthDB: Unavailable Eth %d: Cannot initialize EthDB Port.\n", (UINT32) portID);
        
        return;
    }

    if (portInfo->dependencyPortMap != 0)
    {
        /* redundant */
        return;
    }

    /* initialize core fields */
    portInfo->portID             = portID;
    portInfo->dependencyPortMap  = DEPENDENCY_MAP(portID);

    /* default values */
    portInfo->agingEnabled       = FALSE;
    portInfo->enabled            = FALSE;
    portInfo->macAddressUploaded = FALSE;
    portInfo->maximumFrameSize   = IX_ETHDB_DEFAULT_FRAME_SIZE;

    /* default update control values */
    portInfo->updateMethod.accessRequestInProgress = FALSE;
    portInfo->updateMethod.searchTree              = NULL;
    portInfo->updateMethod.searchTreePendingWrite  = FALSE;
    portInfo->updateMethod.treeInitialized         = FALSE;
    portInfo->updateMethod.treeWriteAccess         = FALSE;
    portInfo->updateMethod.updateEnabled           = FALSE;
    portInfo->updateMethod.userControlled          = FALSE;

    /* Ethernet NPE-specific initializations */
    if (ixEthDBPortDefinitions[portID].type == ETH_NPE)
    {
        /* update handler */
        portInfo->updateMethod.updateHandler = ixEthDBNPEUpdateHandler;
        portInfo->updateMethod.syncAllowed   = TRUE; /* allow read-back of NPE tree entries after AccessGrant or BalanceRequest */

        /* initialize and empty PortAddress mutex */
        ixOsServMutexInit(&portInfo->ackPortAddressLock);
        ixOsServMutexLock(&portInfo->ackPortAddressLock);
        ixOsServMutexInit(&portInfo->ackFrameSizeLock);
        ixOsServMutexLock(&portInfo->ackFrameSizeLock);
    }
}

IX_ETH_DB_PUBLIC
IxEthDBStatus ixEthDBPortEnable(IxEthDBPortId portID)
{
    IxEthDBPortMap triggerPorts;
    IxEthDBPortMap excludePorts;

    IX_ETH_DB_CHECK_PORT_EXISTS(portID);

    IX_ETH_DB_CHECK_SINGLE_NPE(portID);
    
    if (ixEthDBPortInfo[portID].enabled)
    {
        /* redundant */
        return IX_ETH_DB_SUCCESS;
    }

    triggerPorts = DEPENDENCY_MAP(portID);
    excludePorts = EMPTY_DEPENDENCY_MAP;

    /* mark as enabled */
    ixEthDBPortInfo[portID].enabled = TRUE;

    /* Operation stops here when Ethernet Learning is not enabled */
    if(IX_FEATURE_CTRL_SWCONFIG_DISABLED == 
       ixFeatureCtrlSwConfigurationCheck(IX_FEATURECTRL_ETH_LEARNING)) 
    {
        return IX_ETH_DB_SUCCESS;
    } 

    if (ixEthDBPortDefinitions[portID].type == ETH_NPE && !ixEthDBPortInfo[portID].macAddressUploaded)
    {
        IX_ETH_DB_SUPPORT_TRACE("DB: (Support) MAC address not set on port %d, enable failed\n", portID);

        /* must use UnicastAddressSet() before enabling an NPE port */
        return IX_ETH_DB_MAC_UNINITIALIZED;
    }

    if (ixEthDBPortDefinitions[portID].type == ETH_NPE)
    {
        IX_ETH_DB_SUPPORT_TRACE("DB: (Support) Attempting to enable the NPE callback for port %d...\n", portID);

        /* connect event processor callback for this port */
        if (ixEthDBDefaultEventCallbackEnable(portID, TRUE) != IX_ETH_DB_SUCCESS)
        {
            return IX_ETH_DB_FAIL;
        }

        if (!ixEthDBPortInfo[portID].updateMethod.userControlled)
        {  
          ixEthDBPortInfo[portID].updateMethod.updateEnabled = TRUE;
        }
        
        /* if this is first time initialization then we already have
           write access to the tree and can AccessRelease directly */
        if (!ixEthDBPortInfo[portID].updateMethod.treeInitialized)
        {
            ixEthDBPortInfo[portID].updateMethod.treeWriteAccess = TRUE;

            IX_ETH_DB_SUPPORT_TRACE("DB: (Support) Initializing tree for port %d using trigger set %x\n", portID, triggerPorts);

            /* Allow read-back of NPE tree entries after AccessGrant or BalanceRequest */
            ixEthDBPortInfo[portID].updateMethod.syncAllowed = TRUE;

            /* create an initial tree and release access into it */
            ixEthDBUpdatePortLearningTrees(triggerPorts, excludePorts);

            /* mark tree as being initialized */
            ixEthDBPortInfo[portID].updateMethod.treeInitialized = TRUE;
        }
    }

    IX_ETH_DB_SUPPORT_TRACE("DB: (Support) Enabling succeeded for port %d\n", portID);

    return IX_ETH_DB_SUCCESS;
}

IX_ETH_DB_PUBLIC
IxEthDBStatus ixEthDBPortDisable(IxEthDBPortId portID)
{
    HashIterator iterator;
    /* ports who will have deleted records and therefore will need updating */
    IxEthDBPortMap triggerPorts = EMPTY_DEPENDENCY_MAP;

    IX_ETH_DB_CHECK_PORT_EXISTS(portID);

    IX_ETH_DB_CHECK_SINGLE_NPE(portID);
    
    if (!ixEthDBPortInfo[portID].enabled)
    {
        /* redundant */
        return IX_ETH_DB_SUCCESS;
    }

    if(IX_FEATURE_CTRL_SWCONFIG_ENABLED == 
       ixFeatureCtrlSwConfigurationCheck(IX_FEATURECTRL_ETH_LEARNING)) 
    { 
        /* disconnect event processor callback for this port */
        if (ixEthDBPortDefinitions[portID].type == ETH_NPE && ixEthDBDefaultEventCallbackEnable(portID, FALSE) != IX_ETH_DB_SUCCESS)
        {
            return IX_ETH_DB_FAIL;
        }
    }

    /* prevents read-back of NPE tree entries after AccessGrant or BalanceRequest */
    ixEthDBPortInfo[portID].updateMethod.syncAllowed = FALSE;

    /* wipe out current entries for this port */

    /* browse database and entries */
    BUSY_RETRY(ixEthDBInitHashIterator(&dbHashtable, &iterator));

    while (IS_ITERATOR_VALID(&iterator))
    {
	MacDescriptor *descriptor =  (MacDescriptor *) iterator.node->data;

	/* check if the port match. If so, remove the entry  */
	if (descriptor->portID == portID && !descriptor->staticEntry)
	{
	    /* delete entry */
	    BUSY_RETRY(ixEthDBRemoveEntryAtHashIterator(&dbHashtable, &iterator));
	    /* add port to the set of update trigger ports */
	    triggerPorts = JOIN_PORT_TO_MAP(triggerPorts, portID);
	}
	else
        {
	    /* move to the next record */
	    BUSY_RETRY(ixEthDBIncrementHashIterator(&dbHashtable, &iterator));
	}
    }
    
    /* if we've removed any records or lost any events make sure to force an update */
    if (triggerPorts != EMPTY_DEPENDENCY_MAP)
    {
        /* prevents read-back of NPE tree entries after AccessGrant or BalanceRequest */
        ixEthDBPortInfo[portID].updateMethod.syncAllowed = FALSE;
        ixEthDBTriggerRemovePortUpdate(triggerPorts);
    }

    /* mark as disabled */
    ixEthDBPortInfo[portID].enabled = FALSE;

    /* disable updates unless the user has specifically altered the default behavior */
    if (ixEthDBPortDefinitions[portID].type == ETH_NPE)
    {
        if (!ixEthDBPortInfo[portID].updateMethod.userControlled)
        {  
          ixEthDBPortInfo[portID].updateMethod.updateEnabled = FALSE;
        }
            
        /* make sure we re-initialize the NPE learning tree when the port is re-enabled */
        ixEthDBPortInfo[portID].updateMethod.treeInitialized = FALSE;
    }

    return IX_ETH_DB_SUCCESS;
}

IX_ETH_DB_PUBLIC
void ixEthDBMaximumFrameSizeAckCallback(IxNpeMhNpeId npeID, IxNpeMhMessage msg)
{
    IxEthDBPortId portID = IX_ETH_DB_NPE_TO_PORT_ID(npeID);
    PortInfo *portInfo;

    if (portID >= IX_ETH_DB_NUMBER_OF_PORTS || ixEthDBPortDefinitions[portID].type != ETH_NPE)
    {
        ERROR_IRQ_LOG("DB: (Support) FrameSiseSetAck callback received an invalid port value [0x%X], malformed response - possible deadlock condition\n", portID, 0, 0, 0, 0, 0);

        return;
    }

    portInfo = &ixEthDBPortInfo[portID];

    IX_ETH_DB_SUPPORT_IRQ_TRACE("DB: (Support) Received FrameSiseSetAck from port %d\n", portID, 0, 0, 0, 0, 0);

    /* unblock pending ixEthDBPortAddressSet */
    ixOsServMutexUnlock(&portInfo->ackFrameSizeLock);

}

IX_ETH_DB_PUBLIC
IxEthDBStatus ixEthDBFilteringPortMaximumFrameSizeSet(IxEthDBPortId portID,
						  UINT32 maximumFrameSize)
{
    IxEthDBPortId port;
    IxNpeMhMessage message;
    UINT32 frameSizeArray[6];

    IX_ETH_DB_CHECK_PORT_EXISTS(portID);

    IX_ETH_DB_CHECK_SINGLE_NPE(portID);

    if ((maximumFrameSize < IX_ETHDB_MIN_FRAME_SIZE) || 
	(maximumFrameSize > IX_ETHDB_MAX_FRAME_SIZE))
    {
	return IX_ETH_DB_INVALID_ARG;
    }

    if (ixEthDBPortDefinitions[portID].type == ETH_NPE)
    {
	if ((maximumFrameSize < IX_ETHDB_MIN_NPE_FRAME_SIZE) || 
	    (maximumFrameSize > IX_ETHDB_MAX_NPE_FRAME_SIZE))
	{
	    return IX_ETH_DB_INVALID_ARG;
	}
    }

    /* update internal structure */
    ixEthDBPortInfo[portID].maximumFrameSize = maximumFrameSize;

    /* The number of ports id fore size filtering is limited to 6.
     * The NPE has a hardcoded limit of 6 ports (numbered from 0 to 5)
     * Use a temporary array to store the frame size of all ports
     * (the ports which are not defined in the port definition
     * are set with a maximum frame size of 1522)
     */
    for (port = 0; 
	 port < sizeof(frameSizeArray)/sizeof(frameSizeArray[0]); 
	 port++)
    {
	frameSizeArray[port] = IX_ETHDB_DEFAULT_FRAME_SIZE;
    }

    for (port = 0; port < IX_ETH_DB_NUMBER_OF_PORTS; port++)
    {
        if (ixEthDBPortInfo[port].maximumFrameSize != 0)
        {
	    frameSizeArray[port] = ixEthDBPortInfo[port].maximumFrameSize;
        }
    }
   
    /* update all NPEs with new settings : send the first message */
    for (port = 0; port < IX_ETH_DB_NUMBER_OF_PORTS; port++)
    {
	if ((ixEthDBPortDefinitions[port].type == ETH_NPE) &&
	    (ixEthDBSingleEthNpeCheck(port) == IX_ETH_DB_SUCCESS))
	{
	    IX_ETH_DB_SUPPORT_TRACE("DB: (Support) Sending SetMaxFrameSize on port %d...\n", port);
    
	    message.data[0] = (IX_ETHNPE_X2P_NPE_SETMAXSIZEFILTERING1 << 24) |
		(((frameSizeArray[port] + 63) / 64) << 16) |
		(frameSizeArray[0] + 1);
	    message.data[1] = ((frameSizeArray[1] + 1) << 16) | 
		(frameSizeArray[2] + 1);

	    /* send a message for port 0 1 2 */
	    if (ixNpeMhMessageWithResponseSend(IX_ETH_DB_PORT_ID_TO_NPE(port),
		       message,
		       IX_ETHNPE_P2X_NPE_ACKMAXSIZEFILTERING1,
		       ixEthDBMaximumFrameSizeAckCallback,
		       IX_NPEMH_SEND_RETRIES_DEFAULT)
		!= IX_SUCCESS)
	    {
		IX_ETH_DB_SUPPORT_TRACE("DB: (Support) Unable to send first message on port %d...\n", port);
		return IX_ETH_DB_FAIL;
	    }

	    /* block until P2X_ELT_Ack arrives */
	    ixOsServMutexLock(&ixEthDBPortInfo[port].ackFrameSizeLock);

	    IX_ETH_DB_SUPPORT_TRACE("DB: (Support) Sending SetMaxFrameSize on port %d...\n", port);
    
	    message.data[0] = (IX_ETHNPE_X2P_NPE_SETMAXSIZEFILTERING2 << 24) |
		(frameSizeArray[3] + 1);
	    message.data[1] = ((frameSizeArray[4] + 1) << 16) | 
		(frameSizeArray[5] + 1);

	    /* send a message for port 3 4 5 */
	    if (ixNpeMhMessageWithResponseSend(IX_ETH_DB_PORT_ID_TO_NPE(port),
		       message,
		       IX_ETHNPE_P2X_NPE_ACKMAXSIZEFILTERING2,
		       ixEthDBMaximumFrameSizeAckCallback,
		       IX_NPEMH_SEND_RETRIES_DEFAULT)
		!= IX_SUCCESS)
	    {
		IX_ETH_DB_SUPPORT_TRACE("DB: (Support) Unable to send second message on port %d...\n", port);
		return IX_ETH_DB_FAIL;
	    }

	    /* block until P2X_ELT_Ack arrives */
	    ixOsServMutexLock(&ixEthDBPortInfo[port].ackFrameSizeLock);
	}
    }
    return IX_ETH_DB_SUCCESS;
}

IX_ETH_DB_PUBLIC
IxEthDBStatus ixEthDBPortAddressSet(IxEthDBPortId portID, IxEthDBMacAddr *macAddr)
{
    IxNpeMhMessage message;
    IxEthDBPortMap triggerPorts;
    IxEthDBPortMap excludePorts;
    IxMutex *ackPortAddressLock;

    IX_ETH_DB_CHECK_PORT_EXISTS(portID);

    triggerPorts = DEPENDENCY_MAP(portID);
    excludePorts = EMPTY_DEPENDENCY_MAP;

    ackPortAddressLock = &ixEthDBPortInfo[portID].ackPortAddressLock;

    /* Operation stops here when Ethernet Learning is not enabled */
    if(IX_FEATURE_CTRL_SWCONFIG_DISABLED == 
       ixFeatureCtrlSwConfigurationCheck(IX_FEATURECTRL_ETH_LEARNING)) 
    {
        return IX_ETH_DB_SUCCESS;
    }

    IX_ETH_DB_CHECK_SINGLE_NPE(portID);

    /* exit if the port is not an Ethernet NPE */
    if (ixEthDBPortDefinitions[portID].type != ETH_NPE)
    {
        return IX_ETH_DB_INVALID_PORT;
    }

    /* populate message */
    FILL_ELT_MAC_MESSAGE(message, IX_ETHNPE_X2P_ELT_SETPORTADDRESS, portID, macAddr);

    IX_ETH_DB_SUPPORT_TRACE("DB: (Support) Sending SetPortAddress on port %d...\n", portID);
    
    /* send a SetPortAddress message */
    if (ixNpeMhMessageWithResponseSend(IX_ETH_DB_PORT_ID_TO_NPE(portID),
        message,
        IX_ETHNPE_P2X_ELT_ACKPORTADDRESS,
        ixEthDBPortSetAckCallback,
        IX_NPEMH_SEND_RETRIES_DEFAULT) != IX_SUCCESS)
    {
        return IX_ETH_DB_FAIL;
    }

    /* block until P2X_ELT_AckPortAddress arrives */
    ixOsServMutexLock(ackPortAddressLock);

    /* if this is first time initialization then we already have
       write access to the tree and can AccessRelease directly */
    if (!ixEthDBPortInfo[portID].updateMethod.treeInitialized)
    {
        ixEthDBPortInfo[portID].updateMethod.treeWriteAccess = TRUE;

        IX_ETH_DB_SUPPORT_TRACE("DB: (Support) Initializing tree for port %d using trigger set %x\n", portID, triggerPorts);

        /* temporarily enable updating port to create an initial tree */
        ixEthDBPortInfo[portID].updateMethod.updateEnabled = TRUE;

        /* create an initial tree and release access into it */
        ixEthDBUpdatePortLearningTrees(triggerPorts, excludePorts);

        /* disable update until PortEnable() is used on this port */
        ixEthDBPortInfo[portID].updateMethod.updateEnabled = FALSE;

        /* mark tree as being initialized */
        ixEthDBPortInfo[portID].updateMethod.treeInitialized = TRUE;
    }

    return IX_ETH_DB_SUCCESS;
}

/**
 * @brief callback to acknowledge MAC address uploading
 *
 * @param npeID ID of NPE that generated the message
 * @param msg NPE message
 *
 * This routine is called when a P2X_ELT_AckPortAddress message is received
 * from an NPE. It updates the internal port-specific data structures to
 * indicate that the MAC address was uploaded successfully.
 *
 * @warning not to be called directly
 *
 * @internal
 */
IX_ETH_DB_PUBLIC
void ixEthDBPortSetAckCallback(IxNpeMhNpeId npeID, IxNpeMhMessage msg)
{
    IxEthDBPortId portID = IX_ETH_DB_NPE_TO_PORT_ID(npeID);
    PortInfo *portInfo;

    if (portID >= IX_ETH_DB_NUMBER_OF_PORTS || ixEthDBPortDefinitions[portID].type != ETH_NPE)
    {
        ERROR_IRQ_LOG("DB: (Support) PortSetAck callback received an invalid port value [0x%X], malformed response - possible deadlock condition\n", portID, 0, 0, 0, 0, 0);

        return;
    }

    portInfo = &ixEthDBPortInfo[portID];

    IX_ETH_DB_SUPPORT_IRQ_TRACE("DB: (Support) Received AckPortAddress from port %d\n", portID, 0, 0, 0, 0, 0);

    /* unblock pending ixEthDBPortAddressSet */
    ixOsServMutexUnlock(&portInfo->ackPortAddressLock);

    /* allow calls to PortEnable */
    portInfo->macAddressUploaded = TRUE;
}

