/**
 * @file IxEthAccMac.c
 *
 * @author Intel Corporation
 * @date 
 *
 * @brief  MAC control functions
 *
 * Design Notes:
 *
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * @par
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * @par
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * @par
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * @par
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * @par
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * @par
 * -- End Intel Copyright Notice --
 */

#include <string.h>
#include <stdio.h>
#include <IxTypes.h>
#include <IxAssert.h>
#include <IxOsServicesMemMap.h>
#include <IxNpeMh.h>
#include <IxEthDB.h>
#include <IxEthNpe.h>
#include <IxEthAcc.h>
#include <IxOsBuffPoolMgt.h>
#include <IxEthAccDataPlane_p.h>
#include "IxEthAcc_p.h"
#include "IxEthAccMac_p.h"


#define IX_ETH_ACC_VALIDATE_PORT_ID(portId) \
    do                                                           \
    {                                                            \
        if(!IX_ETH_ACC_IS_PORT_VALID(portId))   \
        {                                                        \
	    return IX_ETH_ACC_INVALID_PORT;                      \
        }                                                        \
    } while(0)

/* port Disable state machine : list of states */
typedef enum
{
    IX_STATE_DISABLE_ERR,  /* error state : something unexpected occured */
    IX_STATE_DISABLE_INIT, /* initial state */
    IX_STATE_DISABLE_ACK,
    IX_STATE_DISABLE,
    IX_STATE_DISABLE_COMPLETE,
    IX_STATE_DISABLE_COMPLETE_RX,
    IX_STATE_DISABLE_LAST,
    IX_STATE_DISABLE_LAST_ACK,
    IX_STATE_DISABLE_LAST_RX,
    IX_STATE_DISABLE_END   /* final state */
} IxEthAccPortDisableState;

typedef struct
{
    BOOL fullDuplex;
    BOOL rxFCSAppend;
    BOOL txFCSAppend;
    BOOL txPADAppend;
    BOOL enabled;
    BOOL promiscuous;
    BOOL joinAll;
    IxMutex ackMIBStatsLock;
    IxMutex ackMIBStatsResetLock;
    IxMutex MIBStatsGetAccessLock;
    IxMutex MIBStatsGetResetAccessLock;
    IxEthAccMacAddr mcastAddrsTable[IX_ETH_ACC_MAX_MULTICAST_ADDRESSES];
    UINT32 mcastAddrIndex;
    IX_MBUF *portDisableMbufPtr;
    volatile IxEthAccPortDisableState portDisableState;
    BOOL initDone;
    BOOL macInitialised;
}IxEthAccMacState;

PRIVATE IxEthAccMacState ixEthAccMacState[IX_ETH_ACC_NUMBER_OF_PORTS];

PRIVATE UINT32 ixEthAccMacBase[IX_ETH_ACC_NUMBER_OF_PORTS];

/*Forward function declarations*/
PRIVATE void
ixEthAccPortDisableRxCallback (UINT32 cbTag, 
			       IX_MBUF * mBufPtr,
			       IxEthAccPortId learnedPortId);

PRIVATE void
ixEthAccPortDisableMessageCallback (IxNpeMhNpeId npeId,
				    IxNpeMhMessage msg);

PRIVATE IxEthAccStatus
ixEthAccPortMulticastMacAddressGet (IxEthAccPortId portId,
				    IxEthAccMacAddr *macAddr);

PRIVATE IxEthAccStatus
ixEthAccPortMulticastMacFilterGet (IxEthAccPortId portId,
				   IxEthAccMacAddr *macAddr);

PRIVATE void
ixEthAccMacNpeStatsMessageCallback (IxNpeMhNpeId npeId,
				    IxNpeMhMessage msg);

PRIVATE void
ixEthAccMacNpeStatsResetMessageCallback (IxNpeMhNpeId npeId,
					 IxNpeMhMessage msg);

PRIVATE void
ixEthAccMulticastAddressSet(IxEthAccPortId portId);

PRIVATE BOOL
ixEthAccMacEqual(IxEthAccMacAddr *macAddr1,
		 IxEthAccMacAddr *macAddr2);

PRIVATE void
ixEthAccMacPrint(IxEthAccMacAddr *m);

PRIVATE void
ixEthAccMacStateUpdate(IxEthAccPortId portId);

PRIVATE void
ixEthAccPortDisableStateSet (IxEthAccPortId portId,
			     IxEthAccPortDisableState newState);

PRIVATE BOOL
ixEthAccPortDisableTerminalStateCheck (IxEthAccPortId portId);

IxEthAccStatus
ixEthAccMacMemInit(void)
{
  ixEthAccMacBase[IX_ETH_PORT_1] =
      (UINT32) IX_OSSERV_MEM_MAP(IX_ETH_ACC_MAC_0_BASE, IX_OSSERV_ETHA_MAP_SIZE);
  ixEthAccMacBase[IX_ETH_PORT_2] =
      (UINT32) IX_OSSERV_MEM_MAP(IX_ETH_ACC_MAC_1_BASE, IX_OSSERV_ETHB_MAP_SIZE);

  if (ixEthAccMacBase[IX_ETH_PORT_1] == 0
      || ixEthAccMacBase[IX_ETH_PORT_2] == 0)
  {
    ixOsServLog(LOG_FATAL, "EthAcc: Could not map MAC I/O memory\n", 0, 0, 0, 0, 0 ,0);

    return IX_ETH_ACC_FAIL;
  }

    return IX_ETH_ACC_SUCCESS;
}

void
ixEthAccMacUnload(void)
{
    IX_OSSERV_MEM_UNMAP(ixEthAccMacBase[IX_ETH_PORT_1]);
    IX_OSSERV_MEM_UNMAP(ixEthAccMacBase[IX_ETH_PORT_2]);
    
    ixEthAccMacBase[IX_ETH_PORT_1] = 0;
    ixEthAccMacBase[IX_ETH_PORT_2] = 0;
}

IxEthAccStatus
ixEthAccPortEnable(IxEthAccPortId portId)
{
    IxEthAccStatus result;

    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }

    IX_ETH_ACC_VALIDATE_PORT_ID(portId);

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Unavailable Eth %d: Cannot enable port.\n",(INT32)portId,0,0,0,0,0);
        return IX_ETH_ACC_SUCCESS ;
    }

    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }

    if (ixEthAccPortData[portId].ixEthAccTxData.txBufferDoneCallbackFn == NULL)
    {
        /* TxDone callback not registered */
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }

    if (ixEthAccPortData[portId].ixEthAccRxData.rxCallbackFn == NULL)
    {
        /* Rx callback not registered */
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }

    if(!ixEthAccMacState[portId].initDone)
    {
	return (IX_ETH_ACC_MAC_UNINITIALIZED);
    }

    /* if the state is being set to what it is already at, do nothing*/
    if (ixEthAccMacState[portId].enabled)
    {
        return IX_ETH_ACC_SUCCESS;
    }


    /* enable ethernet database for this port */
    if ((result = ixEthDBPortEnable(portId)) != IX_ETH_ACC_SUCCESS)
    {
        return result;
    }

    REG_WRITE(ixEthAccMacBase[portId],
	      IX_ETH_ACC_MAC_TX_CNTRL2,
	      IX_ETH_ACC_TX_CNTRL2_RETRIES_MASK);

    REG_WRITE(ixEthAccMacBase[portId],
	      IX_ETH_ACC_MAC_RANDOM_SEED,
	      IX_ETH_ACC_RANDOM_SEED_DEFAULT);

    REG_WRITE(ixEthAccMacBase[portId],
	      IX_ETH_ACC_MAC_THRESH_P_EMPTY,
	      IX_ETH_ACC_MAC_THRESH_P_EMPTY_DEFAULT);

    REG_WRITE(ixEthAccMacBase[portId],
	      IX_ETH_ACC_MAC_THRESH_P_FULL,
	      IX_ETH_ACC_MAC_THRESH_P_FULL_DEFAULT);

    REG_WRITE(ixEthAccMacBase[portId],
	      IX_ETH_ACC_MAC_TX_DEFER,
	      IX_ETH_ACC_MAC_TX_DEFER_DEFAULT);

    REG_WRITE(ixEthAccMacBase[portId],
	      IX_ETH_ACC_MAC_TX_TWO_DEFER_1,
	      IX_ETH_ACC_MAC_TX_TWO_DEFER_1_DEFAULT);

    REG_WRITE(ixEthAccMacBase[portId],
	      IX_ETH_ACC_MAC_TX_TWO_DEFER_2,
	      IX_ETH_ACC_MAC_TX_TWO_DEFER_2_DEFAULT);

    REG_WRITE(ixEthAccMacBase[portId],
	      IX_ETH_ACC_MAC_SLOT_TIME,
	      IX_ETH_ACC_MAC_SLOT_TIME_DEFAULT);

    REG_WRITE(ixEthAccMacBase[portId],
	      IX_ETH_ACC_MAC_INT_CLK_THRESH,
	      IX_ETH_ACC_MAC_INT_CLK_THRESH_DEFAULT);

    REG_WRITE(ixEthAccMacBase[portId],
	      IX_ETH_ACC_MAC_BUF_SIZE_TX,
	      IX_ETH_ACC_MAC_BUF_SIZE_TX_DEFAULT);

    REG_WRITE(ixEthAccMacBase[portId],
	      IX_ETH_ACC_MAC_TX_CNTRL1,
	      IX_ETH_ACC_TX_CNTRL1_DEFAULT);

    REG_WRITE(ixEthAccMacBase[portId],
	      IX_ETH_ACC_MAC_RX_CNTRL1,
	      IX_ETH_ACC_RX_CNTRL1_DEFAULT);

    ixEthAccMacState[portId].enabled = TRUE;

    /* rewrite the setup (including mac filtering) depending
     * on current options
     */
    ixEthAccMacStateUpdate(portId);

    return IX_ETH_ACC_SUCCESS;
}

static IxEthAccPortRxCallback ixEthAccPortDisableFn[IX_ETH_ACC_NUMBER_OF_PORTS];
static UINT32 ixEthAccPortDisableCbTag[IX_ETH_ACC_NUMBER_OF_PORTS];

PRIVATE void
ixEthAccPortDisableStateSet (IxEthAccPortId portId,
			     IxEthAccPortDisableState newState)
{
    /* update the statemachine state */
    ixEthAccMacState[portId].portDisableState = newState;
}


PRIVATE BOOL
ixEthAccPortDisableTerminalStateCheck (IxEthAccPortId portId)
{
    /* check for a statemachine terminal state */
    return ((ixEthAccMacState[portId].portDisableState == 
	    IX_STATE_DISABLE_ERR) ||
	(ixEthAccMacState[portId].portDisableState == 
	 IX_STATE_DISABLE_END));
}

PRIVATE void
ixEthAccPortDisableRxCallback (UINT32 cbTag, 
			       IX_MBUF * mBufPtr,
			       IxEthAccPortId learnedPortId)
{
    IxEthAccPortId portId = (IxEthAccPortId)cbTag;
    int key;
    IX_MBUF *mNextPtr;

    while (mBufPtr)
    {
	mNextPtr = IX_MBUF_NEXT_BUFFER_IN_PKT_PTR(mBufPtr);
	IX_MBUF_NEXT_BUFFER_IN_PKT_PTR(mBufPtr) = NULL;

	if (mBufPtr == ixEthAccMacState[portId].portDisableMbufPtr)
	{
	    key = ixOsServIntLock();
	    switch(ixEthAccMacState[portId].portDisableState)
	    {
	    case IX_STATE_DISABLE_INIT:
	    case IX_STATE_DISABLE:
	    case IX_STATE_DISABLE_ACK:
		IX_MBUF_MLEN(mBufPtr) = IX_ETHACC_RX_MBUF_MIN_SIZE;
		if (ixEthAccPortRxFreeReplenish(portId,
			ixEthAccMacState[portId].portDisableMbufPtr)
		    != IX_ETH_ACC_SUCCESS)
		{
		    ixEthAccPortDisableStateSet(portId,
						IX_STATE_DISABLE_ERR);
		}
		break;
	    case IX_STATE_DISABLE_COMPLETE:
		ixEthAccPortDisableStateSet(portId, 
					    IX_STATE_DISABLE_COMPLETE_RX);
		break;
	    case IX_STATE_DISABLE_LAST_ACK:
		ixEthAccPortDisableStateSet(portId, 
					    IX_STATE_DISABLE_END);
		break;
	    case IX_STATE_DISABLE_LAST:
		ixEthAccPortDisableStateSet(portId, 
					    IX_STATE_DISABLE_LAST_RX);
		break;
	    case IX_STATE_DISABLE_LAST_RX:
	    case IX_STATE_DISABLE_COMPLETE_RX:
	    case IX_STATE_DISABLE_END:
	    case IX_STATE_DISABLE_ERR:
		ixEthAccPortDisableStateSet(portId, IX_STATE_DISABLE_ERR);
		break;
	    }
	    ixOsServIntUnlock(key);
	}
	else
	{
	    key = ixOsServIntLock();
	    switch(ixEthAccMacState[portId].portDisableState)
	    {
	    case IX_STATE_DISABLE_INIT:
	    case IX_STATE_DISABLE:
	    case IX_STATE_DISABLE_ACK:
		/* nothing to do */
		break;
	    case IX_STATE_DISABLE_LAST_ACK:
	    case IX_STATE_DISABLE_COMPLETE:
		/* rx free mbuf from user was not expected 
		 * then, restart the disable sequence
		 */
		ixEthAccPortDisableStateSet(portId, 
					    IX_STATE_DISABLE_ACK);
		break;
	    case IX_STATE_DISABLE_LAST:
		/* rx free mbuf from user was not expected 
		 * then, restart the disable sequence
		 */
		ixEthAccPortDisableStateSet(portId, 
					    IX_STATE_DISABLE);
		break;
	    case IX_STATE_DISABLE_LAST_RX:
	    case IX_STATE_DISABLE_COMPLETE_RX:
		/* rx free mbuf from user was not expected 
		 * then, restart the disable sequence
		 * by issuing a new rxfree mbuf
		 */
		ixEthAccPortDisableStateSet(portId, 
					    IX_STATE_DISABLE_INIT);
		break;
	    case IX_STATE_DISABLE_END:
	    case IX_STATE_DISABLE_ERR:
		ixEthAccPortDisableStateSet(portId, 
					    IX_STATE_DISABLE_ERR);
		break;
	    }
	    ixOsServIntUnlock(key);
	    ixEthAccPortDisableFn[portId](ixEthAccPortDisableCbTag[portId],
					  mBufPtr,
					  learnedPortId);
	}
	mBufPtr = mNextPtr;
    }
}

PRIVATE void
ixEthAccPortDisableMessageCallback (IxNpeMhNpeId npeId,
				    IxNpeMhMessage msg)
{
    IxEthAccPortId portId = IX_ETH_ACC_NPE_TO_PORT_ID(npeId);
    int key;

    /* update the statemachine state */
    key = ixOsServIntLock();	
    switch(ixEthAccMacState[portId].portDisableState)
    {
    case IX_STATE_DISABLE:
	if (msg.data[1] == 0)
	{
	    ixEthAccPortDisableStateSet(portId, 
					IX_STATE_DISABLE_COMPLETE);
	}
	else
	{
	    ixEthAccPortDisableStateSet(portId,
					IX_STATE_DISABLE_ACK);
	}
	break;
    case IX_STATE_DISABLE_LAST:
	if (msg.data[1] == 0)
	{
	    ixEthAccPortDisableStateSet(portId, 
					    IX_STATE_DISABLE_LAST_ACK);
	}
	else
	{
	    ixEthAccPortDisableStateSet(portId, 
					IX_STATE_DISABLE_ACK);
	}
	break;
    case IX_STATE_DISABLE_LAST_RX:
	if (msg.data[1] == 0)
	{
	    ixEthAccPortDisableStateSet(portId, 
					IX_STATE_DISABLE_END);
	}
	else
	{
	    ixEthAccPortDisableStateSet(portId, 
					IX_STATE_DISABLE_INIT);
	}
	break;
    case IX_STATE_DISABLE_INIT:
    case IX_STATE_DISABLE_ACK:
    case IX_STATE_DISABLE_COMPLETE:
    case IX_STATE_DISABLE_COMPLETE_RX:
    case IX_STATE_DISABLE_LAST_ACK:
    case IX_STATE_DISABLE_END:
    case IX_STATE_DISABLE_ERR:
	ixEthAccPortDisableStateSet(portId, IX_STATE_DISABLE_ERR);
	break;
    }
    ixOsServIntUnlock(key);
}

IxEthAccStatus
ixEthAccPortDisable(IxEthAccPortId portId)
{
    IxEthAccStatus result;
    IxNpeMhMessage message;
    int key;
    int timeOutCount;
    int retry;
    IX_STATUS npeMhStatus;
    IX_MBUF *mbufPtr;

    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }

    IX_ETH_ACC_VALIDATE_PORT_ID(portId);

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Unavailable Eth %d: Cannot disable port.\n",(INT32)portId,0,0,0,0,0);
        return IX_ETH_ACC_SUCCESS ;
    } 

    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }

    /* if the state is being set to what it is already at, do nothing*/
    if (!ixEthAccMacState[portId].enabled)
    {
        return IX_ETH_ACC_SUCCESS;
    }
   
    /* return mbufs from s/w queue */
    ixEthAccRecoverSwQBuffers(portId);

    /* return mbufs from h/w queue */
    ixEthAccRecoverTxSubmittedQBuffers(portId);
    ixEthAccRecoverRxFreeQBuffers(portId);

    /* disable MAC receive */
    ixEthAccPortRxDisable(portId);

    /* enter the critical section */
    key = ixOsServIntLock();

    /* swap the rx callback */
    ixEthAccPortDisableFn[portId] = 
	ixEthAccPortData[portId].ixEthAccRxData.rxCallbackFn;
    ixEthAccPortDisableCbTag[portId] = 
	ixEthAccPortData[portId].ixEthAccRxData.rxCallbackTag;
    ixEthAccPortData[portId].ixEthAccRxData.rxCallbackFn = 
	ixEthAccPortDisableRxCallback;
    ixEthAccPortData[portId].ixEthAccRxData.rxCallbackTag = 
	portId;

    /* initialise the state machine */
    ixEthAccPortDisableStateSet(portId, IX_STATE_DISABLE_INIT);

    mbufPtr = ixEthAccMacState[portId].portDisableMbufPtr;

    /* exit the critical section */
    ixOsServIntUnlock(key);

    retry = 0;
    do
    {
        /* depending on the state left when port disable
         * failed, change the state to re-enter the state machine
         */
	key = ixOsServIntLock();	
	switch(ixEthAccMacState[portId].portDisableState)
        {
	case IX_STATE_DISABLE_INIT:
            /* first retry */
	    break;
	case IX_STATE_DISABLE:
	case IX_STATE_DISABLE_ACK:
	case IX_STATE_DISABLE_LAST:
	case IX_STATE_DISABLE_COMPLETE:
	case IX_STATE_DISABLE_LAST_ACK:
	    /* no response from NPE, mbuf in NPE internal queue */
	    ixEthAccPortDisableStateSet(portId, IX_STATE_DISABLE_ACK);
	    break;
	case IX_STATE_DISABLE_COMPLETE_RX:
	case IX_STATE_DISABLE_LAST_RX:
	    /* no response from NPE, mbuf received */
	    ixEthAccPortDisableStateSet(portId, IX_STATE_DISABLE_INIT);
	    break;
	    /* no mbuf received */
	case IX_STATE_DISABLE_ERR:
	case IX_STATE_DISABLE_END:
	    /* error recovery attempt */
	    ixEthAccPortDisableStateSet(portId, IX_STATE_DISABLE_INIT);
	    break;
	break;
	}
	ixOsServIntUnlock(key);

	/* iterate until the state is complete */
	timeOutCount = 0;

	do
	{
	/* enter the critical section */
	key = ixOsServIntLock();

	switch(ixEthAccMacState[portId].portDisableState)
	{
	case IX_STATE_DISABLE_INIT:
	    ixEthAccPortDisableStateSet(portId, IX_STATE_DISABLE_ACK);

	    /* send a specific rx free buffer message to NPE */
	    IX_MBUF_MLEN(mbufPtr) = IX_ETHACC_RX_MBUF_MIN_SIZE;
	    IX_MBUF_NEXT_BUFFER_IN_PKT_PTR(mbufPtr) = NULL;
	    if (ixEthAccPortRxFreeReplenish(portId, mbufPtr)
		!= IX_ETH_ACC_SUCCESS)
	    {
		/* unable to replenish : abort */
		ixEthAccPortDisableStateSet(portId, IX_STATE_DISABLE_ERR);
		break;
	    }
	    /* pass thru the next case */	    
	case IX_STATE_DISABLE_ACK:
	    ixEthAccPortDisableStateSet(portId, IX_STATE_DISABLE);

	    /* exit the critical section */
	    ixOsServIntUnlock(key);

	    message.data[0] = IX_ETHNPE_X2P_NPE_PORT_DISABLE << 
		IX_ETH_ACC_MAC_MSGID_SHL;
	    message.data[1] = 0;
	    
	    npeMhStatus = 
		ixNpeMhMessageWithResponseSend(
		       IX_ETH_ACC_PORT_ID_TO_NPE(portId), 
		       message,
		       IX_ETHNPE_P2X_NPE_PORT_DISABLE,
		       ixEthAccPortDisableMessageCallback, 
		       IX_NPEMH_SEND_RETRIES_DEFAULT);

	    /* enter the critical section again */
	    key = ixOsServIntLock();

	    if (npeMhStatus!= IX_SUCCESS)
	    {
		/* unable to send a message : abort */
		ixEthAccPortDisableStateSet(portId, IX_STATE_DISABLE_ERR);
	    }
 	    break;
	case IX_STATE_DISABLE_COMPLETE_RX:
	    ixEthAccPortDisableStateSet(portId, IX_STATE_DISABLE_LAST);

	    /* send a specific rx free buffer message to NPE */
	    IX_MBUF_MLEN(mbufPtr) = IX_ETHACC_RX_MBUF_MIN_SIZE;
	    IX_MBUF_NEXT_BUFFER_IN_PKT_PTR(mbufPtr) = NULL;
	    if (ixEthAccPortRxFreeReplenish(portId, mbufPtr)
		!= IX_ETH_ACC_SUCCESS)
	    {
		/* unable to replenish : abort */
		ixEthAccPortDisableStateSet(portId, IX_STATE_DISABLE_ERR);
		break;
	    }
	    /* exit the critical section */
	    ixOsServIntUnlock(key);

	    message.data[0] = IX_ETHNPE_X2P_NPE_PORT_DISABLE << 
		IX_ETH_ACC_MAC_MSGID_SHL;
	    message.data[1] = 0;
	    
	    npeMhStatus = 
		ixNpeMhMessageWithResponseSend(
		       IX_ETH_ACC_PORT_ID_TO_NPE(portId), 
		       message,
		       IX_ETHNPE_P2X_NPE_PORT_DISABLE,
		       ixEthAccPortDisableMessageCallback, 
		       IX_NPEMH_SEND_RETRIES_DEFAULT);

	    /* enter the critical section again */
	    key = ixOsServIntLock();

	    if (npeMhStatus!= IX_SUCCESS)
	    {
		/* unable to send a message : abort */
		ixEthAccPortDisableStateSet(portId, IX_STATE_DISABLE_ERR);
	    }
 	    break;
	case IX_STATE_DISABLE_COMPLETE:
	case IX_STATE_DISABLE:
	case IX_STATE_DISABLE_LAST:
	case IX_STATE_DISABLE_LAST_ACK:
	case IX_STATE_DISABLE_LAST_RX:
	case IX_STATE_DISABLE_END:
	case IX_STATE_DISABLE_ERR:
	    /* nothing to do */
	    break;
	}
	/* exit the critical section */
	ixOsServIntUnlock(key);
	
	/* wait for rx traffic to drain */
	ixOsServTaskSleep(IX_ETH_ACC_PORT_DISABLE_DELAY_MSECS);
    }
    while ((++timeOutCount < IX_ETH_ACC_PORT_DISABLE_DELAY_COUNT) && 
	   (ixEthAccPortDisableTerminalStateCheck(portId) == FALSE));
    }
    while ((++retry < IX_ETH_ACC_PORT_DISABLE_RETRY_COUNT) && 
	   (ixEthAccPortDisableTerminalStateCheck(portId) == FALSE));

    /* disable MAC Tx and Rx services */
    ixEthAccMacState[portId].enabled = FALSE;  
    ixEthAccMacStateUpdate(portId);

    /* restore the rx callback */
    key = ixOsServIntLock();
    ixEthAccPortData[portId].ixEthAccRxData.rxCallbackFn = 
	ixEthAccPortDisableFn[portId];
    ixEthAccPortData[portId].ixEthAccRxData.rxCallbackTag = 
	ixEthAccPortDisableCbTag[portId];
    ixOsServIntUnlock(key);

    /* disable ethernet database for this port */
    if ((result = ixEthDBPortDisable(portId)) != IX_ETH_ACC_SUCCESS)
    {
	ixEthAccPortDisableStateSet(portId, IX_STATE_DISABLE_ERR);
    }

    /* drain again possible traffic left : return mbufs from s/w queue */
    ixEthAccRecoverSwQBuffers(portId);

    /* drain again possible traffic left : return mbufs from h/w queue */
    ixEthAccRecoverTxSubmittedQBuffers(portId);
    ixEthAccRecoverRxFreeQBuffers(portId);

    /* the MAC core rx/tx disable may left the MAC hardware in an
     * unpredictable state. A hw reset is executed before resetting 
     * all the MAC parameters to a known value.
     */
    REG_WRITE(ixEthAccMacBase[portId], 
	      IX_ETH_ACC_MAC_CORE_CNTRL,
	      IX_ETH_ACC_CORE_RESET);

    ixOsServTaskSleep(IX_ETH_ACC_MAC_RESET_DELAY);   

    ixEthAccMacStateUpdate(portId);

    REG_WRITE(ixEthAccMacBase[portId],
	      IX_ETH_ACC_MAC_INT_CLK_THRESH,
	      IX_ETH_ACC_MAC_INT_CLK_THRESH_DEFAULT);

    REG_WRITE(ixEthAccMacBase[portId], 
	      IX_ETH_ACC_MAC_CORE_CNTRL,
	      IX_ETH_ACC_CORE_MDC_EN);
    
    if (ixEthAccMacState[portId].portDisableState != IX_STATE_DISABLE_END)
    {
	IX_ETH_ACC_FATAL_LOG(
	     "IXETHACC:ixEthAccPortDisable: ixEthAccPortDisable failed port %u (state = %u)\n", 
	     (unsigned)portId, 
	     ixEthAccMacState[portId].portDisableState, 
             0, 0, 0, 0);
	return IX_ETH_ACC_FAIL;
    }

    return IX_ETH_ACC_SUCCESS;
}

IxEthAccStatus
ixEthAccPortEnabledQuery(IxEthAccPortId portId, BOOL *enabled)
{
    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }

    IX_ETH_ACC_VALIDATE_PORT_ID(portId);  

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Unavailable Eth %d: Cannot enable port.\n",(INT32)portId,0,0,0,0,0);

        /* Since Eth NPE is not available, port must be disabled */  
        *enabled = FALSE ;
        return IX_ETH_ACC_SUCCESS ;
    }
 
    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
        /* Since Eth NPE is not available, port must be disabled */  
        *enabled = FALSE ;
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }
    
    *enabled = ixEthAccMacState[portId].enabled;
    
    return IX_ETH_ACC_SUCCESS;
}

IxEthAccStatus 
ixEthAccPortLoopbackEnable(IxEthAccPortId portId)
{
    UINT32 regval;

    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }

    /* Turn off promiscuous mode */    
    IX_ETH_ACC_VALIDATE_PORT_ID(portId);

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Eth %d: Cannot enable loopback.\n",(INT32)portId,0,0,0,0,0);
        return IX_ETH_ACC_SUCCESS ;
    } 

    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }
   
    /* read register */
    REG_READ(ixEthAccMacBase[portId], 
	     IX_ETH_ACC_MAC_RX_CNTRL1,
	     regval);
    
    /* update register */
    REG_WRITE(ixEthAccMacBase[portId],
	      IX_ETH_ACC_MAC_RX_CNTRL1,
	      regval | IX_ETH_ACC_RX_CNTRL1_LOOP_EN);

    return IX_ETH_ACC_SUCCESS;
}

IxEthAccStatus 
ixEthAccPortTxEnable(IxEthAccPortId portId)
{
    UINT32 regval;

    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }

    /* Turn off promiscuous mode */    
    IX_ETH_ACC_VALIDATE_PORT_ID(portId);

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Eth %d: Cannot enable TX.\n",(INT32)portId,0,0,0,0,0);
        return IX_ETH_ACC_SUCCESS ;
    } 

    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }
   
    /* read register */
    REG_READ(ixEthAccMacBase[portId], 
	     IX_ETH_ACC_MAC_TX_CNTRL1,
	     regval);
    
    /* update register */
    REG_WRITE(ixEthAccMacBase[portId],
	      IX_ETH_ACC_MAC_TX_CNTRL1,
	      regval | IX_ETH_ACC_TX_CNTRL1_TX_EN);

    return IX_ETH_ACC_SUCCESS;
}

IxEthAccStatus 
ixEthAccPortRxEnable(IxEthAccPortId portId)
{
    UINT32 regval;

    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }

    /* Turn off promiscuous mode */    
    IX_ETH_ACC_VALIDATE_PORT_ID(portId);

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Eth %d: Cannot enable RX.\n",(INT32)portId,0,0,0,0,0);
        return IX_ETH_ACC_SUCCESS ;
    } 

    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }
   
    /* read register */
    REG_READ(ixEthAccMacBase[portId], 
	     IX_ETH_ACC_MAC_RX_CNTRL1,
	     regval);
    
    /* update register */
    REG_WRITE(ixEthAccMacBase[portId],
	      IX_ETH_ACC_MAC_RX_CNTRL1,
	      regval | IX_ETH_ACC_RX_CNTRL1_RX_EN);

    return IX_ETH_ACC_SUCCESS;
}

IxEthAccStatus 
ixEthAccPortLoopbackDisable(IxEthAccPortId portId)
{
    UINT32 regval;

    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }

    IX_ETH_ACC_VALIDATE_PORT_ID(portId);

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Eth %d: Cannot disable loopback.\n",(INT32)portId,0,0,0,0,0);
        return IX_ETH_ACC_SUCCESS ;
    } 

    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }
   
    /*disable MAC loopabck */
    REG_READ(ixEthAccMacBase[portId], 
	     IX_ETH_ACC_MAC_RX_CNTRL1,
	     regval);
    
    REG_WRITE(ixEthAccMacBase[portId],
	      IX_ETH_ACC_MAC_RX_CNTRL1,
	      (regval & ~IX_ETH_ACC_RX_CNTRL1_LOOP_EN));

    return IX_ETH_ACC_SUCCESS;
}

IxEthAccStatus 
ixEthAccPortTxDisable(IxEthAccPortId portId)
{
    UINT32 regval;

    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }

    /* Turn off promiscuous mode */    
    IX_ETH_ACC_VALIDATE_PORT_ID(portId);

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Eth %d: Cannot disable TX.\n", (INT32)portId,0,0,0,0,0);
        return IX_ETH_ACC_SUCCESS ;
    } 

    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }
   
    /* read register */
    REG_READ(ixEthAccMacBase[portId], 
	     IX_ETH_ACC_MAC_TX_CNTRL1,
	     regval);
    
    /* update register */
    REG_WRITE(ixEthAccMacBase[portId],
	      IX_ETH_ACC_MAC_TX_CNTRL1,
	      (regval & ~IX_ETH_ACC_TX_CNTRL1_TX_EN));

    return IX_ETH_ACC_SUCCESS;
}

IxEthAccStatus 
ixEthAccPortRxDisable(IxEthAccPortId portId)
{
    UINT32 regval;

    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }

    /* Turn off promiscuous mode */    
    IX_ETH_ACC_VALIDATE_PORT_ID(portId);

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Eth %d: Cannot disable RX.\n", (INT32)portId,0,0,0,0,0);
        return IX_ETH_ACC_SUCCESS ;
    } 

    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }
   
    /* read register */
    REG_READ(ixEthAccMacBase[portId], 
	     IX_ETH_ACC_MAC_RX_CNTRL1,
	     regval);
    
    /* update register */
    REG_WRITE(ixEthAccMacBase[portId],
	      IX_ETH_ACC_MAC_RX_CNTRL1,
	      (regval & ~IX_ETH_ACC_RX_CNTRL1_RX_EN));

    return IX_ETH_ACC_SUCCESS;
}

IxEthAccStatus 
ixEthAccPortPromiscuousModeClear(IxEthAccPortId portId)
{
    UINT32 regval;

    /* Turn off promiscuous mode */    
    IX_ETH_ACC_VALIDATE_PORT_ID(portId);

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Unavailable Eth %d: Cannot clear promiscuous mode.\n",(INT32)portId,0,0,0,0,0);
        return IX_ETH_ACC_SUCCESS ;
    }

    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }
   
    /*set bit 5 of Rx control 1 - enable address filtering*/
    REG_READ(ixEthAccMacBase[portId],
	     IX_ETH_ACC_MAC_RX_CNTRL1,
	     regval);
    
    REG_WRITE(ixEthAccMacBase[portId],
	      IX_ETH_ACC_MAC_RX_CNTRL1,
	      regval | IX_ETH_ACC_RX_CNTRL1_ADDR_FLTR_EN);

    ixEthAccMacState[portId].promiscuous = FALSE;

    ixEthAccMulticastAddressSet(portId);

    return IX_ETH_ACC_SUCCESS;
}

IxEthAccStatus  
ixEthAccPortPromiscuousModeSet(IxEthAccPortId portId)
{
    UINT32 regval;

    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }

    IX_ETH_ACC_VALIDATE_PORT_ID(portId);

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Unavailable Eth %d: Cannot set promiscuous mode.\n",(INT32)portId,0,0,0,0,0);
        return IX_ETH_ACC_SUCCESS ;
    }

    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }
    
    /* 
     * Set bit 5 of Rx control 1 - We enable address filtering even in
     * promiscuous mode because we want the MAC to set the appropriate
     * bits in m_flags which doesn't happen if we turn off filtering.
     */
    REG_READ(ixEthAccMacBase[portId], 
	     IX_ETH_ACC_MAC_RX_CNTRL1,
	     regval);
    
    REG_WRITE(ixEthAccMacBase[portId],
	      IX_ETH_ACC_MAC_RX_CNTRL1,
	      regval | IX_ETH_ACC_RX_CNTRL1_ADDR_FLTR_EN);

    ixEthAccMacState[portId].promiscuous = TRUE;

    ixEthAccMulticastAddressSet(portId);

    return IX_ETH_ACC_SUCCESS;
}

IxEthAccStatus 
ixEthAccPortUnicastMacAddressSet (IxEthAccPortId portId,
				  IxEthAccMacAddr *macAddr)
{
    UINT32 i;
    IxEthAccStatus result;
    
    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }

    IX_ETH_ACC_VALIDATE_PORT_ID(portId);

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Unavailable Eth %d: Cannot set Unicast Mac Address.\n",(INT32)portId,0,0,0,0,0);
        return IX_ETH_ACC_SUCCESS ;
    }

    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }


    if (macAddr == NULL)
    {
	return IX_ETH_ACC_FAIL;
    }    

    if ( macAddr->macAddress[0] & IX_ETH_ACC_ETH_MAC_BCAST_MCAST_BIT )
    {
	/* This is a multicast/broadcast address cant set it ! */
	return IX_ETH_ACC_FAIL;
    }

    if ( macAddr->macAddress[0] == 0 &&
	 macAddr->macAddress[1] == 0 &&
	 macAddr->macAddress[2] == 0 &&
	 macAddr->macAddress[3] == 0 &&
	 macAddr->macAddress[4] == 0 &&
	 macAddr->macAddress[5] == 0  )
    {
	/* This is an invalid mac address cant set it ! */
	return IX_ETH_ACC_FAIL;
    }
	
    /* update the MAC address in the ethernet database */
    if ((result = ixEthDBPortAddressSet(portId, (IxEthDBMacAddr *) macAddr)) != IX_ETH_ACC_SUCCESS)
    {
        return result;
    }
    
    /*Set the Unicast MAC to the specified value*/
    for(i=0;i<IX_IEEE803_MAC_ADDRESS_SIZE;i++)
    {	
	REG_WRITE(ixEthAccMacBase[portId],
		  IX_ETH_ACC_MAC_UNI_ADDR_1 + i*sizeof(UINT32),
		  macAddr->macAddress[i]);	
    }
    ixEthAccMacState[portId].initDone = TRUE;

    return IX_ETH_ACC_SUCCESS;
}

IxEthAccStatus 
ixEthAccPortUnicastMacAddressGet (IxEthAccPortId portId, 
				  IxEthAccMacAddr *macAddr)
{
    /*Return the current value of the Unicast MAC from h/w
      for the specified port*/
    UINT32 i;

    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }

    IX_ETH_ACC_VALIDATE_PORT_ID(portId);

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Unavailable Eth %d: Cannot get Unicast Mac Address.\n",(INT32)portId,0,0,0,0,0);
        /* Since Eth Npe is unavailable, return invalid MAC Address = 00:00:00:00:00:00 */
        for(i=0;i<IX_IEEE803_MAC_ADDRESS_SIZE;i++)
        {
	    macAddr->macAddress[i] = 0;	
        }        
        return IX_ETH_ACC_SUCCESS ;
    }

    if(!ixEthAccMacState[portId].initDone)
    {
	return (IX_ETH_ACC_MAC_UNINITIALIZED);
    }
    
    if (macAddr == NULL)
    {
	return IX_ETH_ACC_FAIL;
    }    
    
    
    for(i=0;i<IX_IEEE803_MAC_ADDRESS_SIZE;i++)
    {
	REG_READ(ixEthAccMacBase[portId],
		 IX_ETH_ACC_MAC_UNI_ADDR_1 + i*sizeof(UINT32),
		 macAddr->macAddress[i]);	
    }
    return IX_ETH_ACC_SUCCESS;
}

PRIVATE IxEthAccStatus
ixEthAccPortMulticastMacAddressGet (IxEthAccPortId portId,
				    IxEthAccMacAddr *macAddr)
{
    /*Return the current value of the Multicast MAC from h/w
      for the specified port*/
    UINT32 i;

    for(i=0;i<IX_IEEE803_MAC_ADDRESS_SIZE;i++)
    {

	REG_READ(ixEthAccMacBase[portId],
		 IX_ETH_ACC_MAC_ADDR_1 + i*sizeof(UINT32),
		 macAddr->macAddress[i]);
    }

    return IX_ETH_ACC_SUCCESS;
}

PRIVATE IxEthAccStatus
ixEthAccPortMulticastMacFilterGet (IxEthAccPortId portId,
				   IxEthAccMacAddr *macAddr)
{
    /*Return the current value of the Multicast MAC from h/w
      for the specified port*/
    UINT32 i;

    for(i=0;i<IX_IEEE803_MAC_ADDRESS_SIZE;i++)
    {

	REG_READ(ixEthAccMacBase[portId],
		 IX_ETH_ACC_MAC_ADDR_MASK_1 + i*sizeof(UINT32),
		 macAddr->macAddress[i]);
    }
    return IX_ETH_ACC_SUCCESS;
}

IxEthAccStatus
ixEthAccPortMulticastAddressJoin (IxEthAccPortId portId,
				  IxEthAccMacAddr *macAddr)
{
    UINT32 i;
    IxEthAccMacAddr broadcastAddr = {{0xff,0xff,0xff,0xff,0xff,0xff}};

    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }

    /*Check that the port parameter is valid*/
    IX_ETH_ACC_VALIDATE_PORT_ID(portId);

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Unavailable Eth %d: Cannot join Multicast Mac Address.\n",(INT32)portId,0,0,0,0,0);
        return IX_ETH_ACC_SUCCESS ;
    }

    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }

    /*Check that the mac address is valid*/
    if(macAddr == NULL)
    {
	return IX_ETH_ACC_FAIL;
    }

    /* Check that this is a multicast address */
    if (!(macAddr->macAddress[0] & IX_ETH_ACC_ETH_MAC_BCAST_MCAST_BIT))
    {
	return IX_ETH_ACC_FAIL;
    }

    /* We don't add the Broadcast address */
    if(ixEthAccMacEqual(&broadcastAddr, macAddr))
    {
	return IX_ETH_ACC_FAIL;
    }

    for (i = 0;
	 i<ixEthAccMacState[portId].mcastAddrIndex;
	 i++)
    {
	/*Check if the current entry already match an existing matches*/
	if(ixEthAccMacEqual(&ixEthAccMacState[portId].mcastAddrsTable[i], macAddr))
	{
	    /* Address found in the list and already configured,
	     * return a success status
	     */
	    return IX_ETH_ACC_SUCCESS;
	}
    }

    /* check for availability at the end of the current table */
    if(ixEthAccMacState[portId].mcastAddrIndex >= IX_ETH_ACC_MAX_MULTICAST_ADDRESSES)
    {
	return IX_ETH_ACC_FAIL;
    }

    /*First add the address to the multicast table for the
      specified port*/
    i=ixEthAccMacState[portId].mcastAddrIndex;
    
    memcpy(&ixEthAccMacState[portId].mcastAddrsTable[i],
	   &macAddr->macAddress,
	   IX_IEEE803_MAC_ADDRESS_SIZE);
    
    /*Increment the index into the table, this must be done here
     as MulticastAddressSet below needs to know about the latest
     entry.
    */
    ixEthAccMacState[portId].mcastAddrIndex++;

    /*Then calculate the new value to be written to the address and
      address mask registers*/
    ixEthAccMulticastAddressSet(portId);

    return IX_ETH_ACC_SUCCESS;
}


IxEthAccStatus
ixEthAccPortMulticastAddressJoinAll (IxEthAccPortId portId)
{
    IxEthAccMacAddr mcastMacAddr = {{0x1,0x0,0x0,0x0,0x0,0x0}};

    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }

    /*Check that the port parameter is valid*/
    IX_ETH_ACC_VALIDATE_PORT_ID(portId);

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Unavailable Eth %d: Cannot join all Multicast Address.\n",(INT32)portId,0,0,0,0,0);
        return IX_ETH_ACC_SUCCESS ;
    }

    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }

    /* remove all entries from the database and
    *  insert a multicast entry
    */
    memcpy(&ixEthAccMacState[portId].mcastAddrsTable[0],
	   &mcastMacAddr.macAddress,
	   IX_IEEE803_MAC_ADDRESS_SIZE);

    ixEthAccMacState[portId].mcastAddrIndex = 1;
    ixEthAccMacState[portId].joinAll = TRUE;

    ixEthAccMulticastAddressSet(portId);

    return IX_ETH_ACC_SUCCESS;
}

IxEthAccStatus
ixEthAccPortMulticastAddressLeave (IxEthAccPortId portId,
				   IxEthAccMacAddr *macAddr)
{
    UINT32 i;
    IxEthAccMacAddr mcastMacAddr = {{0x1,0x0,0x0,0x0,0x0,0x0}};

    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }

    /*Check that the port parameter is valid*/
    IX_ETH_ACC_VALIDATE_PORT_ID(portId);

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Unavailable Eth %d: Cannot leave Multicast Address.\n",(INT32)portId,0,0,0,0,0);
        return IX_ETH_ACC_SUCCESS ;
    }

    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }

    /*Check that the mac address is valid*/
    if(macAddr == NULL)
    {
	return IX_ETH_ACC_FAIL;
    }
    /* Remove this mac address from the mask for the specified port
     * we copy down all entries above the blanked entry, and
     * decrement the index
     */    
    i=0;

    while(i<ixEthAccMacState[portId].mcastAddrIndex)
    {
	/*Check if the current entry matches*/
	if(ixEthAccMacEqual(&ixEthAccMacState[portId].mcastAddrsTable[i],
			    macAddr))
	{
	    if(ixEthAccMacEqual(macAddr, &mcastMacAddr))
	    {
		ixEthAccMacState[portId].joinAll = FALSE;
	    }
	    /*Decrement the index into the multicast address table
	      for the current port*/
	    ixEthAccMacState[portId].mcastAddrIndex--;

	    /*Copy down all entries above the current entry*/
	    while(i<ixEthAccMacState[portId].mcastAddrIndex)
	    {
		memcpy(&ixEthAccMacState[portId].mcastAddrsTable[i],
		       &ixEthAccMacState[portId].mcastAddrsTable[i+1], 
		       IX_IEEE803_MAC_ADDRESS_SIZE);
                i++;
	    }
	    /*recalculate the mask and write it to the MAC*/
	    ixEthAccMulticastAddressSet(portId);

	    return IX_ETH_ACC_SUCCESS;
	}
	/* search the next entry */
	i++;
    }
    /* no matching entry found */
    return IX_ETH_ACC_NO_SUCH_ADDR;
}

IxEthAccStatus
ixEthAccPortMulticastAddressLeaveAll (IxEthAccPortId portId)
{
    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }

    /*Check that the port parameter is valid*/
    IX_ETH_ACC_VALIDATE_PORT_ID(portId);

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Unavailable Eth %d: Cannot leave all Multicast Address.\n",(INT32)portId,0,0,0,0,0);
        return IX_ETH_ACC_SUCCESS ;
    }

    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }

    ixEthAccMacState[portId].mcastAddrIndex = 0;
    ixEthAccMacState[portId].joinAll = FALSE;

    ixEthAccMulticastAddressSet(portId);

    return IX_ETH_ACC_SUCCESS;
}


IxEthAccStatus
ixEthAccPortUnicastAddressShow (IxEthAccPortId portId)
{
    IxEthAccMacAddr macAddr;

    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }

    IX_ETH_ACC_VALIDATE_PORT_ID(portId);

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Unavailable Eth %d: Cannot show Unicast Address.\n",(INT32)portId,0,0,0,0,0);
        return IX_ETH_ACC_SUCCESS ;
    }

    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }
    
    /*Get the MAC (UINICAST) address from hardware*/
    if(ixEthAccPortUnicastMacAddressGet(portId, &macAddr) != IX_ETH_ACC_SUCCESS)
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: MAC address uninitialised port %u\n",
			       (INT32)portId,0,0,0,0,0);
	return IX_ETH_ACC_MAC_UNINITIALIZED;
    }

    /*print it out*/
    ixEthAccMacPrint(&macAddr);
    printf("\n");
    return IX_ETH_ACC_SUCCESS;
}



void
ixEthAccPortMulticastAddressShow(IxEthAccPortId portId)
{    
    IxEthAccMacAddr macAddr;
    UINT32 i;

    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return;
    }

    if(!IX_ETH_ACC_IS_PORT_VALID(portId))
    {
        return;
    } 

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Unavailable Eth %d: Cannot show Multicast Address.\n",(INT32)portId,0,0,0,0,0);
        return ;
    }

    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
	return;
    }

    printf("Multicast MAC: ");
    /*Get the MAC (MULTICAST) address from hardware*/
    ixEthAccPortMulticastMacAddressGet(portId, &macAddr);
    /*print it out*/
    ixEthAccMacPrint(&macAddr);
    /*Get the MAC (MULTICAST) filter from hardware*/
    ixEthAccPortMulticastMacFilterGet(portId, &macAddr);
    /*print it out*/
    printf(" ( ");
    ixEthAccMacPrint(&macAddr);
    printf(" )\n");
    printf("Constituent Addresses:\n");
    for(i=0;i<ixEthAccMacState[portId].mcastAddrIndex;i++)
    {
	ixEthAccMacPrint(&ixEthAccMacState[portId].mcastAddrsTable[i]);
	printf("\n");
    }
    return;
}

/*Set the duplex mode*/
IxEthAccStatus 
ixEthAccPortDuplexModeSet (IxEthAccPortId portId, 
			   IxEthAccDuplexMode mode)
{
    UINT32 txregval;
    UINT32 rxregval;

    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }

    /*This is bit 1 of the transmit control reg, set to 1 for half
      duplex, 0 for full duplex*/
    IX_ETH_ACC_VALIDATE_PORT_ID(portId);

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Unavailable Eth %d: Cannot set Duplex Mode.\n",(INT32)portId,0,0,0,0,0);
        return IX_ETH_ACC_SUCCESS ;
    }

    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }

    REG_READ(ixEthAccMacBase[portId], 
	     IX_ETH_ACC_MAC_TX_CNTRL1,
	     txregval);
    
    REG_READ(ixEthAccMacBase[portId],
	     IX_ETH_ACC_MAC_RX_CNTRL1,
	     rxregval);
    
    if (mode ==  IX_ETH_ACC_FULL_DUPLEX)
    {
	/*Clear half duplex bit in TX*/
	REG_WRITE(ixEthAccMacBase[portId],
		  IX_ETH_ACC_MAC_TX_CNTRL1,
		  txregval & ~IX_ETH_ACC_TX_CNTRL1_DUPLEX);

	/*We must set the pause enable in the receive logic when in
	  full duplex mode*/
	REG_WRITE(ixEthAccMacBase[portId],
		  IX_ETH_ACC_MAC_RX_CNTRL1,
		  rxregval | IX_ETH_ACC_RX_CNTRL1_PAUSE_EN);
	ixEthAccMacState[portId].fullDuplex = TRUE;
	
    }
    else if (mode ==  IX_ETH_ACC_HALF_DUPLEX)
    {
	/*Set half duplex bit in TX*/
	REG_WRITE(ixEthAccMacBase[portId],
		  IX_ETH_ACC_MAC_TX_CNTRL1,
		  txregval | IX_ETH_ACC_TX_CNTRL1_DUPLEX);

	/*We must clear pause enable in the receive logic when in
	  half duplex mode*/	
	REG_WRITE(ixEthAccMacBase[portId],
		  IX_ETH_ACC_MAC_RX_CNTRL1,
		  rxregval & ~IX_ETH_ACC_RX_CNTRL1_PAUSE_EN);

	ixEthAccMacState[portId].fullDuplex = FALSE;
    }
    else
    {
	return IX_ETH_ACC_FAIL;
    }
    
    
    return IX_ETH_ACC_SUCCESS;    
    
}



IxEthAccStatus 
ixEthAccPortDuplexModeGet (IxEthAccPortId portId, 
			   IxEthAccDuplexMode *mode)
{
    /*Return the duplex mode for the specified port*/
    UINT32 regval;

    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }

    /*This is bit 1 of the transmit control reg, set to 1 for half
      duplex, 0 for full duplex*/
    IX_ETH_ACC_VALIDATE_PORT_ID(portId);

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Unavailable Eth %d: Cannot get Duplex Mode.\n",(INT32)portId,0,0,0,0,0);
        /* return hald duplex */ 
        *mode = IX_ETH_ACC_HALF_DUPLEX ;
        return IX_ETH_ACC_SUCCESS ;
    }

    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }
    
    if (mode == NULL)
    {
	return (IX_ETH_ACC_FAIL);
    }

    REG_READ(ixEthAccMacBase[portId], 
	     IX_ETH_ACC_MAC_TX_CNTRL1,
	     regval);
    
    if( regval & IX_ETH_ACC_TX_CNTRL1_DUPLEX)
    {
	*mode = IX_ETH_ACC_HALF_DUPLEX;
    }
    else
    {
	*mode = IX_ETH_ACC_FULL_DUPLEX;
    }

    return IX_ETH_ACC_SUCCESS;
}



IxEthAccStatus 
ixEthAccPortTxFrameAppendPaddingEnable (IxEthAccPortId portId)
{
    UINT32 regval;
    /*Enable FCS computation by the MAC and appending to the
      frame*/
    
    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }

    IX_ETH_ACC_VALIDATE_PORT_ID(portId);

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Unavailable Eth %d: Cannot enable Tx Frame Append Padding.\n",(INT32)portId,0,0,0,0,0);
        return IX_ETH_ACC_SUCCESS ;
    }

    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }
    
    REG_READ(ixEthAccMacBase[portId], 
	     IX_ETH_ACC_MAC_TX_CNTRL1,
	     regval);
    
    REG_WRITE(ixEthAccMacBase[portId],
	      IX_ETH_ACC_MAC_TX_CNTRL1,
	      regval | 
	      IX_ETH_ACC_TX_CNTRL1_PAD_EN);

    ixEthAccMacState[portId].txPADAppend = TRUE;
    return IX_ETH_ACC_SUCCESS;  
}

IxEthAccStatus 
ixEthAccPortTxFrameAppendPaddingDisable (IxEthAccPortId portId)
{
    UINT32 regval;

    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }

    /*disable FCS computation and appending*/
    /*Set bit 4 of tx control register one to zero*/
    IX_ETH_ACC_VALIDATE_PORT_ID(portId);

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Unavailable Eth %d: Cannot disble Tx Frame Append Padding.\n",(INT32)portId,0,0,0,0,0);
        return IX_ETH_ACC_SUCCESS ;
    }

    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }
    
    REG_READ(ixEthAccMacBase[portId], 
	     IX_ETH_ACC_MAC_TX_CNTRL1,
	     regval);
    
    REG_WRITE(ixEthAccMacBase[portId],
	      IX_ETH_ACC_MAC_TX_CNTRL1,
	      regval & ~IX_ETH_ACC_TX_CNTRL1_PAD_EN);

    ixEthAccMacState[portId].txPADAppend = FALSE;
    return IX_ETH_ACC_SUCCESS; 
}

IxEthAccStatus 
ixEthAccPortTxFrameAppendFCSEnable (IxEthAccPortId portId)
{
    UINT32 regval;

    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }

    /*Enable FCS computation by the MAC and appending to the
      frame*/
    
    IX_ETH_ACC_VALIDATE_PORT_ID(portId);

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Unavailable Eth %d: Cannot enable Tx Frame Append FCS.\n",(INT32)portId,0,0,0,0,0);
        return IX_ETH_ACC_SUCCESS ;
    }

    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }
    
    REG_READ(ixEthAccMacBase[portId], 
	     IX_ETH_ACC_MAC_TX_CNTRL1,
	     regval);
    
    REG_WRITE(ixEthAccMacBase[portId],
	      IX_ETH_ACC_MAC_TX_CNTRL1,
	      regval | IX_ETH_ACC_TX_CNTRL1_FCS_EN);

    ixEthAccMacState[portId].txFCSAppend = TRUE;
    return IX_ETH_ACC_SUCCESS;  
}

IxEthAccStatus 
ixEthAccPortTxFrameAppendFCSDisable (IxEthAccPortId portId)
{
    UINT32 regval;

    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }

    /*disable FCS computation and appending*/
    /*Set bit 4 of tx control register one to zero*/
    IX_ETH_ACC_VALIDATE_PORT_ID(portId);

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Unavailable Eth %d: Cannot disable Tx Frame Append FCS.\n",(INT32)portId,0,0,0,0,0);
        return IX_ETH_ACC_SUCCESS ;
    }

    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }
    
    REG_READ(ixEthAccMacBase[portId], 
	     IX_ETH_ACC_MAC_TX_CNTRL1,
	     regval);
    
    REG_WRITE(ixEthAccMacBase[portId],
	      IX_ETH_ACC_MAC_TX_CNTRL1,
	      regval & ~IX_ETH_ACC_TX_CNTRL1_FCS_EN);

    ixEthAccMacState[portId].txFCSAppend = FALSE;
    return IX_ETH_ACC_SUCCESS; 
}

IxEthAccStatus 
ixEthAccPortRxFrameAppendFCSEnable (IxEthAccPortId portId)
{
    /*Set bit 2 of rx control 1*/
    UINT32 regval;

    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }

    IX_ETH_ACC_VALIDATE_PORT_ID(portId);

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Unavailable Eth %d: Cannot enable Rx Frame Append FCS.\n",(INT32)portId,0,0,0,0,0);
        return IX_ETH_ACC_SUCCESS ;
    }

    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }
    
    REG_READ(ixEthAccMacBase[portId], 
	     IX_ETH_ACC_MAC_RX_CNTRL1,
	     regval);
    
    REG_WRITE(ixEthAccMacBase[portId],
	      IX_ETH_ACC_MAC_RX_CNTRL1,
	      regval | IX_ETH_ACC_RX_CNTRL1_CRC_EN);

    ixEthAccMacState[portId].rxFCSAppend = TRUE;
    return IX_ETH_ACC_SUCCESS;
}

IxEthAccStatus 
ixEthAccPortRxFrameAppendFCSDisable (IxEthAccPortId portId)
{
    UINT32 regval;

    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }

    /*Clear bit 2 of rx control 1*/
    IX_ETH_ACC_VALIDATE_PORT_ID(portId);    

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Unavailable Eth %d: Cannot disable Rx Frame Append FCS.\n",(INT32)portId,0,0,0,0,0);
        return IX_ETH_ACC_SUCCESS ;
    }

    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }
    
    REG_READ(ixEthAccMacBase[portId], 
	     IX_ETH_ACC_MAC_RX_CNTRL1,
	     regval);
    
    REG_WRITE(ixEthAccMacBase[portId],
	      IX_ETH_ACC_MAC_RX_CNTRL1,
	      regval & ~IX_ETH_ACC_RX_CNTRL1_CRC_EN);

    ixEthAccMacState[portId].rxFCSAppend = FALSE;
    return IX_ETH_ACC_SUCCESS;
}



PRIVATE void
ixEthAccMacNpeStatsMessageCallback (IxNpeMhNpeId npeId,
				    IxNpeMhMessage msg)
{
    IxEthAccPortId portId = IX_ETH_ACC_NPE_TO_PORT_ID(npeId);

    /*Unblock Stats Get call*/
    ixOsServMutexUnlock(&ixEthAccMacState[portId].ackMIBStatsLock);

}

PRIVATE void
ixEthAccMibIIStatsEndianConvert (IxEthEthObjStats *retStats)
{
    /* endianness conversion */
    retStats->dot3StatsAlignmentErrors = 
	IX_OSSERV_SWAP_NPE_SHARED_LONG(retStats->dot3StatsAlignmentErrors);
    retStats->dot3StatsFCSErrors = 
	IX_OSSERV_SWAP_NPE_SHARED_LONG(retStats->dot3StatsFCSErrors);
    retStats->dot3StatsFrameTooLongs = 
	IX_OSSERV_SWAP_NPE_SHARED_LONG(retStats->dot3StatsFrameTooLongs);
    retStats->dot3StatsInternalMacReceiveErrors = 
	IX_OSSERV_SWAP_NPE_SHARED_LONG(retStats->dot3StatsInternalMacReceiveErrors);
    retStats->LearnedEntryDiscards = 
	IX_OSSERV_SWAP_NPE_SHARED_LONG(retStats->LearnedEntryDiscards);
    retStats->UnderflowEntryDiscards = 
	IX_OSSERV_SWAP_NPE_SHARED_LONG(retStats->UnderflowEntryDiscards);
    retStats->dot3StatsSingleCollisionFrames = 
	IX_OSSERV_SWAP_NPE_SHARED_LONG(retStats->dot3StatsSingleCollisionFrames);
    retStats->dot3StatsMultipleCollisionFrames = 
	IX_OSSERV_SWAP_NPE_SHARED_LONG(retStats->dot3StatsMultipleCollisionFrames);
    retStats->dot3StatsDeferredTransmissions = 
	IX_OSSERV_SWAP_NPE_SHARED_LONG(retStats->dot3StatsDeferredTransmissions);
    retStats->dot3StatsLateCollisions = 
	IX_OSSERV_SWAP_NPE_SHARED_LONG(retStats->dot3StatsLateCollisions);
    retStats->dot3StatsExcessiveCollsions = 
	IX_OSSERV_SWAP_NPE_SHARED_LONG(retStats->dot3StatsExcessiveCollsions);
    retStats->dot3StatsInternalMacTransmitErrors = 
	IX_OSSERV_SWAP_NPE_SHARED_LONG(retStats->dot3StatsInternalMacTransmitErrors);
    retStats->dot3StatsCarrierSenseErrors = 
	IX_OSSERV_SWAP_NPE_SHARED_LONG(retStats->dot3StatsCarrierSenseErrors);
}

IxEthAccStatus
ixEthAccMibIIStatsGet (IxEthAccPortId portId,
		       IxEthEthObjStats *retStats )
{
    IxNpeMhMessage message;

    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }

    IX_ETH_ACC_VALIDATE_PORT_ID(portId);

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Unavailable Eth %d: Cannot get MIB II Stats.\n",(INT32)portId,0,0,0,0,0);
        /* Return all zero stats */
        IX_ETH_ACC_MEMSET(retStats, 0, sizeof(IxEthEthObjStats));
        return IX_ETH_ACC_SUCCESS ;
    }

    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }

    if (retStats == NULL)
    {
	return (IX_ETH_ACC_FAIL);
    }

    if (!ixEthAccMacState[portId].enabled)
    {
        return (IX_ETH_ACC_FAIL);
    }

    IX_ACC_DATA_CACHE_INVALIDATE(retStats, sizeof(IxEthEthObjStats));
    retStats = (IxEthEthObjStats *) IX_MMU_VIRTUAL_TO_PHYSICAL_TRANSLATION(retStats);

    message.data[0] = IX_ETHNPE_X2P_STATS_SHOW << IX_ETH_ACC_MAC_MSGID_SHL;
    message.data[1] = (UINT32)retStats;

    /*Permit only one task to request MIB statistics Get operation
      at a time*/
    ixOsServMutexLock(&ixEthAccMacState[portId].MIBStatsGetAccessLock);

    if(ixNpeMhMessageWithResponseSend(IX_ETH_ACC_PORT_ID_TO_NPE(portId),
				      message,
				      IX_ETHNPE_P2X_STATS_REPORT,
				      ixEthAccMacNpeStatsMessageCallback,
				      IX_NPEMH_SEND_RETRIES_DEFAULT)
       != IX_SUCCESS)
    {
	ixOsServMutexUnlock(&ixEthAccMacState[portId].MIBStatsGetAccessLock);
	return IX_ETH_ACC_FAIL;
    }

    /*Wait for callback invocation indicating response to
      this request - we need this mutex in order to ensure
      that the return from this function is synchronous
    */
    ixOsServMutexLock(&ixEthAccMacState[portId].ackMIBStatsLock);
    /*Permit other tasks to perform MIB statistics Get operation*/
    ixOsServMutexUnlock(&ixEthAccMacState[portId].MIBStatsGetAccessLock);

    ixEthAccMibIIStatsEndianConvert (retStats);

    return IX_ETH_ACC_SUCCESS;
}


PRIVATE void
ixEthAccMacNpeStatsResetMessageCallback (IxNpeMhNpeId npeId,
					 IxNpeMhMessage msg)
{
    IxEthAccPortId portId = IX_ETH_ACC_NPE_TO_PORT_ID(npeId);

    /*Unblock Stats Get & reset call*/
    ixOsServMutexUnlock(&ixEthAccMacState[portId].ackMIBStatsResetLock);
    
}



IxEthAccStatus 
ixEthAccMibIIStatsGetClear (IxEthAccPortId portId, 
			    IxEthEthObjStats *retStats)
{
    IxNpeMhMessage message;

    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }

    IX_ETH_ACC_VALIDATE_PORT_ID(portId);

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Unavailable Eth %d: Cannot get and clear MIB II Stats.\n",(INT32)portId,0,0,0,0,0);
        return IX_ETH_ACC_SUCCESS ;
    }

    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }

    if (retStats == NULL)
    {
	return (IX_ETH_ACC_FAIL);
    }

    if (!ixEthAccMacState[portId].enabled)
    {
        return (IX_ETH_ACC_FAIL);
    }

    IX_ACC_DATA_CACHE_INVALIDATE(retStats, sizeof(IxEthEthObjStats));
    retStats = (IxEthEthObjStats *)IX_MMU_VIRTUAL_TO_PHYSICAL_TRANSLATION(retStats);
    
    message.data[0] = IX_ETHNPE_X2P_STATS_RESET << 
	IX_ETH_ACC_MAC_MSGID_SHL;
    message.data[1] = (UINT32)retStats;
    
    /*Permit only one task to request MIB statistics Get-Reset operation
      at a time*/
    ixOsServMutexLock(&ixEthAccMacState[portId].MIBStatsGetResetAccessLock);
    
    if(ixNpeMhMessageWithResponseSend(IX_ETH_ACC_PORT_ID_TO_NPE(portId), 
				      message,
				      IX_ETHNPE_P2X_STATS_REPORT,
				      ixEthAccMacNpeStatsResetMessageCallback,
				      IX_NPEMH_SEND_RETRIES_DEFAULT) 
       != IX_SUCCESS)
    {
	ixOsServMutexUnlock(&ixEthAccMacState[portId].MIBStatsGetResetAccessLock);
	return IX_ETH_ACC_FAIL;
    }

    /*Wait for callback invocation indicating response to 
      this request*/
    ixOsServMutexLock(&ixEthAccMacState[portId].ackMIBStatsResetLock);
    /*permit other tasks to get and reset MIB stats*/
    ixOsServMutexUnlock(&ixEthAccMacState[portId].MIBStatsGetResetAccessLock);

    ixEthAccMibIIStatsEndianConvert (retStats);

    return IX_ETH_ACC_SUCCESS;
    
}

IxEthAccStatus
ixEthAccMibIIStatsClear (IxEthAccPortId portId)
{
    IxEthEthObjStats retStats;

    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }

    IX_ETH_ACC_VALIDATE_PORT_ID(portId);

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Unavailable Eth %d: Cannot clear MIB II Stats.\n",(INT32)portId,0,0,0,0,0);
        return IX_ETH_ACC_SUCCESS ;
    }

    /*There is no reset operation without a corresponding Get
    */

    return ixEthAccMibIIStatsGetClear(portId, &retStats);

}

/* Initialize the ethernet MAC settings */
IxEthAccStatus
ixEthAccMacInit(IxEthAccPortId portId)
{
    IX_MBUF_POOL* portDisablePool;

    IX_ETH_ACC_VALIDATE_PORT_ID(portId);

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Unavailable Eth %d: Cannot initialize Mac.\n",(INT32)portId,0,0,0,0,0);
        return IX_ETH_ACC_SUCCESS ;
    }
    
    if(ixEthAccMacState[portId].macInitialised == FALSE)
    {
	ixEthAccMacState[portId].fullDuplex  = TRUE;
	ixEthAccMacState[portId].rxFCSAppend = TRUE;
	ixEthAccMacState[portId].txFCSAppend = TRUE;
	ixEthAccMacState[portId].txPADAppend = TRUE;
	ixEthAccMacState[portId].enabled     = FALSE;
	ixEthAccMacState[portId].promiscuous = TRUE;
	ixEthAccMacState[portId].joinAll     = FALSE;
	ixEthAccMacState[portId].initDone    = FALSE;
	ixEthAccMacState[portId].macInitialised = TRUE;
    
        /* initialize MIB stats mutexes */
        ixOsServMutexInit(&ixEthAccMacState[portId].ackMIBStatsLock);
        ixOsServMutexLock(&ixEthAccMacState[portId].ackMIBStatsLock);

        ixOsServMutexInit(&ixEthAccMacState[portId].ackMIBStatsResetLock);
        ixOsServMutexLock(&ixEthAccMacState[portId].ackMIBStatsResetLock);

        ixOsServMutexInit(&ixEthAccMacState[portId].MIBStatsGetAccessLock);

        ixOsServMutexInit(&ixEthAccMacState[portId].MIBStatsGetResetAccessLock);

	ixEthAccMacState[portId].portDisableMbufPtr = NULL;
	IX_MBUF_POOL_INIT(&portDisablePool, 
			  1, 
			  IX_ETHACC_RX_MBUF_MIN_SIZE,
			  "portDisable Pool");
	IX_MBUF_POOL_GET(portDisablePool, 
			 &ixEthAccMacState[portId].portDisableMbufPtr);
	IX_ENSURE(ixEthAccMacState[portId].portDisableMbufPtr != NULL, "Pool allocation failed");
    }
    
    IX_ASSERT (ixEthAccMacBase[portId] != 0);
  
    REG_WRITE(ixEthAccMacBase[portId], 
	      IX_ETH_ACC_MAC_CORE_CNTRL,
	      IX_ETH_ACC_CORE_RESET);
    
    ixOsServTaskSleep(IX_ETH_ACC_MAC_RESET_DELAY);   
    
    REG_WRITE(ixEthAccMacBase[portId], 
	      IX_ETH_ACC_MAC_CORE_CNTRL,
	      IX_ETH_ACC_CORE_MDC_EN);
    
    REG_WRITE(ixEthAccMacBase[portId],
	      IX_ETH_ACC_MAC_INT_CLK_THRESH,
	      IX_ETH_ACC_MAC_INT_CLK_THRESH_DEFAULT);
    
    ixEthAccMacStateUpdate(portId);
    
    return IX_ETH_ACC_SUCCESS; 
}

/* PRIVATE Functions*/

PRIVATE void
ixEthAccMacStateUpdate(IxEthAccPortId portId)
{
    UINT32 regval;

    if ( ixEthAccMacState[portId].enabled == FALSE )
    {
	/*  Just disable both the transmitter and reciver in the MAC.  */
        REG_READ(ixEthAccMacBase[portId], 
		 IX_ETH_ACC_MAC_RX_CNTRL1,
		 regval);
 	REG_WRITE(ixEthAccMacBase[portId],
		  IX_ETH_ACC_MAC_RX_CNTRL1,
		  regval & ~IX_ETH_ACC_RX_CNTRL1_RX_EN);

        REG_READ(ixEthAccMacBase[portId], 
		 IX_ETH_ACC_MAC_TX_CNTRL1,
		 regval);
 	REG_WRITE(ixEthAccMacBase[portId],
		  IX_ETH_ACC_MAC_TX_CNTRL1,
		  regval & ~IX_ETH_ACC_TX_CNTRL1_TX_EN);
    }
    
    if(ixEthAccMacState[portId].fullDuplex)
    {
	ixEthAccPortDuplexModeSet (portId, IX_ETH_ACC_FULL_DUPLEX);
    }
    else
    {
	ixEthAccPortDuplexModeSet (portId, IX_ETH_ACC_HALF_DUPLEX);
    }

    if(ixEthAccMacState[portId].rxFCSAppend)
    {
	ixEthAccPortRxFrameAppendFCSEnable (portId);
    }
    else
    {
	ixEthAccPortRxFrameAppendFCSDisable (portId);
    }

    if(ixEthAccMacState[portId].txFCSAppend)
    {
	ixEthAccPortTxFrameAppendFCSEnable (portId);
    }
    else
    {
	ixEthAccPortTxFrameAppendFCSDisable (portId);
    }
    
    if(ixEthAccMacState[portId].txPADAppend)
    {
	ixEthAccPortTxFrameAppendPaddingEnable (portId);
    }
    else
    {
	ixEthAccPortTxFrameAppendPaddingDisable (portId);
    }
    
    if(ixEthAccMacState[portId].promiscuous)
    {
	ixEthAccPortPromiscuousModeSet(portId);
    }
    else
    {
	ixEthAccPortPromiscuousModeClear(portId);
    }

    if ( ixEthAccMacState[portId].enabled == TRUE )
    {
        /*   Enable both the transmitter and reciver in the MAC.  */
        REG_READ(ixEthAccMacBase[portId],
		 IX_ETH_ACC_MAC_RX_CNTRL1,
		 regval);
        REG_WRITE(ixEthAccMacBase[portId],
		  IX_ETH_ACC_MAC_RX_CNTRL1,
		  regval | IX_ETH_ACC_RX_CNTRL1_RX_EN);

        REG_READ(ixEthAccMacBase[portId], 
		 IX_ETH_ACC_MAC_TX_CNTRL1,
		 regval);
 	REG_WRITE(ixEthAccMacBase[portId],
		  IX_ETH_ACC_MAC_TX_CNTRL1,
		  regval | IX_ETH_ACC_TX_CNTRL1_TX_EN);
    }
}


PRIVATE BOOL
ixEthAccMacEqual(IxEthAccMacAddr *macAddr1,
		 IxEthAccMacAddr *macAddr2)
{
    UINT32 i;
    for(i=0;i<IX_IEEE803_MAC_ADDRESS_SIZE; i++)
    {
	if(macAddr1->macAddress[i] != macAddr2->macAddress[i])
	{
	    return FALSE;
	}
    }
    return TRUE;
}

PRIVATE void
ixEthAccMacPrint(IxEthAccMacAddr *m)
{
    printf("%2.2x:%2.2x:%2.2x:%2.2x:%2.2x:%2.2x",
	   m->macAddress[0], m->macAddress[1],
	   m->macAddress[2], m->macAddress[3],
	   m->macAddress[4], m->macAddress[5]);    
}

/* Set the multicast address and address mask registers
 * 
 * A bit in the address mask register must be set if
 * all multicast addresses always have that bit set, or if
 * all multicast addresses always have that bit cleared.
 *
 * A bit in the address register must be set if all multicast
 * addresses have that bit set, otherwise, it should be cleared
 */

PRIVATE void
ixEthAccMulticastAddressSet(IxEthAccPortId portId)
{
    UINT32 i;
    UINT32 j;
    IxEthAccMacAddr addressMask;
    IxEthAccMacAddr address;
    IxEthAccMacAddr alwaysClearBits;
    IxEthAccMacAddr alwaysSetBits;

    /* calculate alwaysClearBits and alwaysSetBits:
     * alwaysClearBits is calculated by ORing all 
     * multicast addresses, those bits that are always
     * clear are clear in the result
     *
     * alwaysSetBits is calculated by ANDing all 
     * multicast addresses, those bits that are always set
     * are set in the result
     */
    
    if (ixEthAccMacState[portId].promiscuous == TRUE)
    {
	/* Promiscuous Mode is set, and filtering 
	 * allow all packets, and enable the mcast and
	 * bcast detection.
	 */
	memset(&addressMask.macAddress,
	       0, 
	       IX_IEEE803_MAC_ADDRESS_SIZE);
	memset(&address.macAddress, 
	       0, 
	       IX_IEEE803_MAC_ADDRESS_SIZE);
    }
    else
    {
	if(ixEthAccMacState[portId].joinAll == TRUE)
	{
	    /* Join all is set. The mask and address are
	     * the multicast settings.
	     */
	    IxEthAccMacAddr macAddr = {{0x1,0x0,0x0,0x0,0x0,0x0}};

	    memcpy(addressMask.macAddress, 
		   macAddr.macAddress, 
		   IX_IEEE803_MAC_ADDRESS_SIZE);
	    memcpy(address.macAddress, 
		   macAddr.macAddress,
		   IX_IEEE803_MAC_ADDRESS_SIZE);
	}
	else if(ixEthAccMacState[portId].mcastAddrIndex == 0)
	{
	    /* No entry in the filtering database,
	     * Promiscuous Mode is cleared, Broadcast filtering
	     * is configured.
	     */
	    memset(addressMask.macAddress, 
		   IX_ETH_ACC_MAC_ALL_BITS_SET, 
		   IX_IEEE803_MAC_ADDRESS_SIZE);
	    memset(address.macAddress, 
		   IX_ETH_ACC_MAC_ALL_BITS_SET,
		   IX_IEEE803_MAC_ADDRESS_SIZE);
	}
	else
	{
	    /* build a mask and an address which mix all entreis
	     * from the list of multicast addresses
	     */
	    memset(alwaysClearBits.macAddress, 
		   0, 
		   IX_IEEE803_MAC_ADDRESS_SIZE);
	    memset(alwaysSetBits.macAddress, 
		   IX_ETH_ACC_MAC_ALL_BITS_SET, 
		   IX_IEEE803_MAC_ADDRESS_SIZE);
	    
	    for(i=0;i<ixEthAccMacState[portId].mcastAddrIndex;i++)
	    {
		for(j=0;j<IX_IEEE803_MAC_ADDRESS_SIZE;j++)
		{
		    alwaysClearBits.macAddress[j] |= 
			ixEthAccMacState[portId].mcastAddrsTable[i].macAddress[j];
		    alwaysSetBits.macAddress[j] &= 
			ixEthAccMacState[portId].mcastAddrsTable[i].macAddress[j];
		}
	    }
       
	    for(i=0;i<IX_IEEE803_MAC_ADDRESS_SIZE;i++)
	    {
		addressMask.macAddress[i] = alwaysSetBits.macAddress[i]
		    | ~alwaysClearBits.macAddress[i];
		address.macAddress[i] = alwaysSetBits.macAddress[i];
	    }
	}
    }
    
    /*write the new addr filtering to h/w*/    
    for(i=0;i<IX_IEEE803_MAC_ADDRESS_SIZE;i++)
    {	
	REG_WRITE(ixEthAccMacBase[portId],
		  IX_ETH_ACC_MAC_ADDR_MASK_1+i*sizeof(UINT32),
		  addressMask.macAddress[i]);
	REG_WRITE(ixEthAccMacBase[portId],
		  IX_ETH_ACC_MAC_ADDR_1+i*sizeof(UINT32),
		  address.macAddress[i]);	    
    }
}





