/**
 * @file IxUSBRNDIS.h
 *
 * @brief include file for the IXP425 USB RNDIS implementation
 *
 * @version $Revision: 1.1.1.1 $
 * 
 * @par
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * @par
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * @par
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * @par
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * @par
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * @par
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * @par
 * -- End Intel Copyright Notice --
*/
#ifndef IXUSBRNDIS_H
#define IXUSBRNDIS_H

#include <IxTypes.h>
#include <usb.h>
#include <stdio.h>
#include <stdarg.h>

/**
 * @defgroup Codelets IXP425 Codelets
 *
 * @brief IXP425 Codelets
 *
 * @{
 */

/**
 * @ingroup Codelets
 *
 * @defgroup IxUSBRNDIS IXP425 USB RNDIS Codelet (IxUSBRNDIS) API
 *
 * @brief IXP425 codelet USB RNDIS API
 *
 * How to use the USB RNDIS codelet:
 *
 * - build a loadable object and load it into vxWorks
 * - start the codelet by typing ixUSBRNDISStart
 *
 * You should see the "usb" network interface in the output 
 * generated by the ifShow command.

 * Plug the board into the USB port of a Windows 98/ME/2000
 * machine and selected the driver provided with the codelet
 * when queried for it.
 *
 * Note:
 * - the IP and MAC addresses of the END driver (therefore
 * the board side of the link) are defined in ixUSBRNDISEnd.h
 * - the MAC address of the RNDIS driver (which will be
 * used by Windows as its own MAC address) is defined in ixUSBRNDIS.h
 *
 * Currently the END MAC address is 00:00:00:00:00:02 and the
 * RNDIS MAC address is 00:00:00:00:00:01.
 * The IP address of the END is 192.168.1.1, therefore you should
 * use a compatible address for the RNDIS controller on the Windows
 * side (such as 192.168.1.2) and set the END IP address as
 * gateway address for the RNDIS network device, or change
 * them to suitable values.
 *
 * The codelet was tested with Windows 2000 only, and telnet and
 * ftp traffic was passed by routing the PC through the IXP425 into
 * a network.
 *
 * Warning: this codelet is for demonstration purposes only, it should
 * not be considered a fully working application.
 *
 *
 *
 * @{
 */

#ifndef __doxygen_HIDE

#define USB_COMMUCATION_DEVICE          (0x02)
#define USB_COMMUNICATION_INTERFACE     (0x02)
#define USB_ABSTRACT_CONTROL_MODEL      (0x02)
#define USB_VENDOR_SPECIFIC_PROTOCOL    (0xff)
#define USB_DATA_INTERFACE              (0x0a)

#define RNDIS_SEND_ENCAPSULATED_COMMAND (0x00)
#define RNDIS_GET_ENCAPSULATED_RESPONSE (0x01)

#define RNDIS_RESPONSE_AVAILABLE                   (0x01)
#define RNDIS_RESPONSE_AVAILABLE_NOTIFICATION_SIZE (2) /* words */

/* endpoint definitions for RNDIS usage */
#define RNDIS_NOTIFICATION_ENDPOINT (ENDPOINT_5) /* Interrupt IN */
#define RNDIS_DATA_IN_ENDPOINT      (ENDPOINT_1) /* Bulk IN */
#define RNDIS_DATA_OUT_ENDPOINT     (ENDPOINT_2) /* Bulk OUT */
#define RNDIS_CONTROL_ENDPOINT      (ENDPOINT_0) /* Control */

/* RNDIS messages */
#define RNDIS_INITIALIZE        (0x2)
#define RNDIS_HALT              (0x3)
#define RNDIS_QUERY             (0x4)
#define RNDIS_SET               (0x5)
#define RNDIS_RESET             (0x6)
#define RNDIS_INDICATE_STATUS   (0x7)
#define RNDIS_KEEPALIVE         (0x8)
#define RNDIS_COMPLETED         (0x80000000) /* this flag is ORed with the message ID to indicate completion */

/* RNDIS message component offsets (in word units!) */
#define RNDIS_MESSAGE_TYPE_OFFSET  (0)
#define RNDIS_REQUEST_ID_OFFSET    (2)
#define RNDIS_OID_OFFSET           (3)
#define RNDIS_BUFFER_LENGTH_OFFSET (4)
#define RNDIS_BUFFER_OFFSET_OFFSET (5)

/* RNDIS data packets */
#define RNDIS_DATA_OFFSET_OFFSET                        (2) /* entry offset */
#define RNDIS_DATA_LENGTH_OFFSET                        (3) /* entry offset */
#define RNDIS_DATA_PAYLOAD_BYTE_OFFSET(UINT32_pPacket)  (44)
#define RNDIS_DATA_PAYLOAD_BYTE_LENGTH(UINT32_pPacket)  ((UINT32_pPacket)[RNDIS_DATA_LENGTH_OFFSET])
#define RNDIS_DATA_PACKET_HEADER_SIZE                   (44) /* size of the RNDIS packet header, in bytes */
#define RNDIS_PACKET_MSG                                (0x1)
#define RNDIS_PAYLOAD_OFFSET                            (36)

/* RNDIS message response packet sizes */
#define RNDIS_INITIALIZE_COMPLETE_SIZE   (52)
#define RNDIS_QUERY_COMPLETE_HEADER_SIZE (24)
#define RNDIS_GENERIC_MSG_COMPLETE_SIZE  (16)

/* RNDIS responses */
#define RNDIS_NUMBER_OF_SUPPORTED_OIDS (26)
#define RNDIS_MAXIMUM_FRAME_SIZE       (1520)
#define RNDIS_MAXIMUM_TOTAL_SIZE       (1558)
#define RNDIS_USB_SPEED                (12000000)
#define RNDIS_LINK_SPEED               (RNDIS_USB_SPEED / 100) /* link speed in units of 100bps - USB speed is 12Mbps */
#define RNDIS_RECEIVE_BLOCK_SIZE       (1514)
#define RNDIS_TRANSMIT_BLOCK_SIZE      (64)
#define RNDIS_MAX_PACKETS_PER_TRANSFER (1)
#define RNDIS_MAX_HOST_PACKET_SIZE     (2048)
#define RNDIS_PACKET_ALIGNMENT         (1) /* aligment, in powers of 2 - unused for 1 packet transfers */

/* RNDIS status values */
#define RNDIS_STATUS_SUCCESS            (0)
#define RNDIS_STATUS_FAILURE            (0xc0000001)
#define RNDIS_STATUS_MEDIA_CONNECT      (0x4001000b)
#define RNDIS_STATUS_MEDIA_DISCONNECT   (0x4001000c)

/* RNDIS versions */
#define RNDIS_MAJOR_VERSION (1)
#define RNDIS_MINOR_VERSION (0)

/* RNDIS device flags */
#define RNDIS_DF_CONNECTIONLESS      (1)
#define RNDIS_DF_CONNECTION_ORIENTED (2)

/* RNDIS medium definitions */
#define RNDIS_MEDIUM_802_3 (0)


/* -------------------- Imported MS RNDIS Constants --------------------- */

/*	Required OIDs */
#define OID_GEN_SUPPORTED_LIST					0x00010101
#define OID_GEN_HARDWARE_STATUS					0x00010102
#define OID_GEN_MEDIA_SUPPORTED					0x00010103
#define OID_GEN_MEDIA_IN_USE					0x00010104
#define OID_GEN_MAXIMUM_LOOKAHEAD				0x00010105
#define OID_GEN_MAXIMUM_FRAME_SIZE				0x00010106
#define OID_GEN_LINK_SPEED						0x00010107
#define OID_GEN_TRANSMIT_BUFFER_SPACE			0x00010108
#define OID_GEN_RECEIVE_BUFFER_SPACE			0x00010109
#define OID_GEN_TRANSMIT_BLOCK_SIZE				0x0001010A
#define OID_GEN_RECEIVE_BLOCK_SIZE				0x0001010B
#define OID_GEN_VENDOR_ID						0x0001010C
#define OID_GEN_VENDOR_DESCRIPTION				0x0001010D
#define OID_GEN_CURRENT_PACKET_FILTER			0x0001010E
#define OID_GEN_CURRENT_LOOKAHEAD				0x0001010F
#define OID_GEN_DRIVER_VERSION					0x00010110
#define OID_GEN_MAXIMUM_TOTAL_SIZE				0x00010111
#define OID_GEN_PROTOCOL_OPTIONS				0x00010112
#define OID_GEN_MAC_OPTIONS						0x00010113
#define OID_GEN_MEDIA_CONNECT_STATUS			0x00010114
#define OID_GEN_MAXIMUM_SEND_PACKETS			0x00010115
#define OID_GEN_VENDOR_DRIVER_VERSION			0x00010116
#define	OID_GEN_SUPPORTED_GUIDS					0x00010117
#define	OID_GEN_NETWORK_LAYER_ADDRESSES			0x00010118	// Set only
#define OID_GEN_TRANSPORT_HEADER_OFFSET			0x00010119  // Set only

/*	Optional OIDs */
#define	OID_GEN_MEDIA_CAPABILITIES				0x00010201
#define OID_GEN_PHYSICAL_MEDIUM					0x00010202

/*	Required statistics */
#define OID_GEN_XMIT_OK							0x00020101
#define OID_GEN_RCV_OK							0x00020102
#define OID_GEN_XMIT_ERROR						0x00020103
#define OID_GEN_RCV_ERROR						0x00020104
#define OID_GEN_RCV_NO_BUFFER					0x00020105

/* 802.3 Objects (Ethernet) */
#define OID_802_3_PERMANENT_ADDRESS				0x01010101
#define OID_802_3_CURRENT_ADDRESS				0x01010102
#define OID_802_3_MULTICAST_LIST				0x01010103
#define OID_802_3_MAXIMUM_LIST_SIZE				0x01010104
#define OID_802_3_MAC_OPTIONS					0x01010105

#define	NDIS_802_3_MAC_OPTION_PRIORITY			0x00000001

#define OID_802_3_RCV_ERROR_ALIGNMENT			0x01020101
#define OID_802_3_XMIT_ONE_COLLISION			0x01020102
#define OID_802_3_XMIT_MORE_COLLISIONS			0x01020103

#define OID_802_3_XMIT_DEFERRED					0x01020201
#define OID_802_3_XMIT_MAX_COLLISIONS			0x01020202
#define OID_802_3_RCV_OVERRUN					0x01020203
#define OID_802_3_XMIT_UNDERRUN					0x01020204
#define OID_802_3_XMIT_HEARTBEAT_FAILURE		0x01020205
#define OID_802_3_XMIT_TIMES_CRS_LOST			0x01020206
#define OID_802_3_XMIT_LATE_COLLISIONS			0x01020207

typedef enum _NDIS_HARDWARE_STATUS
{
	NdisHardwareStatusReady,
	NdisHardwareStatusInitializing,
	NdisHardwareStatusReset,
	NdisHardwareStatusClosing,
	NdisHardwareStatusNotReady
} NDIS_HARDWARE_STATUS, *PNDIS_HARDWARE_STATUS;

typedef enum _NDIS_MEDIA_STATE
{
	NdisMediaStateConnected,
	NdisMediaStateDisconnected
} NDIS_MEDIA_STATE, *PNDIS_MEDIA_STATE;

/* Medium the Ndis Driver is running on (OID_GEN_MEDIA_SUPPORTED/ OID_GEN_MEDIA_IN_USE). */
typedef enum _NDIS_MEDIUM
{
	NdisMedium802_3,
	NdisMedium802_5,
	NdisMediumFddi,
	NdisMediumWan,
	NdisMediumLocalTalk,
	NdisMediumDix,				/* defined for convenience, not a real medium */
	NdisMediumArcnetRaw,
	NdisMediumArcnet878_2,
	NdisMediumAtm,
	NdisMediumWirelessWan,
	NdisMediumIrda,
	NdisMediumBpc,
	NdisMediumCoWan,
	NdisMedium1394,
	NdisMediumMax				/* Not a real medium, defined as an upper-bound */
} NDIS_MEDIUM, *PNDIS_MEDIUM;


/* Physical Medium Type definitions. Used with OID_GEN_PHYSICAL_MEDIUM. */
typedef enum _NDIS_PHYSICAL_MEDIUM
{
	NdisPhysicalMediumReserved0,
	NdisPhysicalMediumWirelessLan,
	NdisPhysicalMediumCableModem,
	NdisPhysicalMediumPhoneLine,
	NdisPhysicalMediumPowerLine,
	NdisPhysicalMediumDSL,		/* includes ADSL and UADSL (G.Lite) */
	NdisPhysicalMediumMax		/* Not a real physical type, defined as an upper-bound */
} NDIS_PHYSICAL_MEDIUM, *PNDIS_PHYSICAL_MEDIUM;


/*
 * RNDIS layer state
 */
typedef enum
{
    RNDIS_HALTED = 0,   	/* RNDIS layer state - Halted */
    RNDIS_INITIALIZED,		/* RNDIS layer state - Initialized */
    RNDIS_DATA_INITIALIZED 	/* RNDIS layer state - Data Initialized */
} RndisLayerState;

/*
 * Structure to maintain RNDIS layer statistics
 */
typedef struct
{
    UINT32 Rx;			/* Number of packets recieved */
    UINT32 Tx;			/* Number of packets transmitted */
    UINT32 RxNoBuf;		/* Number of packets recieved but no buffer */
} RndisLayerStats;

/*
 * Structure to maintain RNDIS layer data information
 */
typedef struct
{
    UINT32 currentPacketFilter;		/* Current packet filter */
    RndisLayerStats stats;		/* RNDIS layer state statistics */
} RndisLayerData;
/* Protocol types supported by ndis. These values need to be consistent 
 * with ADDRESS_TYPE_XXX defined in TDI.H 
 */


#define	NDIS_PROTOCOL_ID_DEFAULT		0x00
#define	NDIS_PROTOCOL_ID_TCP_IP			0x02
#define	NDIS_PROTOCOL_ID_IPX			0x06
#define	NDIS_PROTOCOL_ID_NBF			0x07
#define	NDIS_PROTOCOL_ID_MAX			0x0F
#define	NDIS_PROTOCOL_ID_MASK			0x0F

/* Ndis Packet Filter Bits (OID_GEN_CURRENT_PACKET_FILTER). */

#define NDIS_PACKET_TYPE_DIRECTED 0x0001

#define NDIS_PACKET_TYPE_MULTICAST 0x0002

#define NDIS_PACKET_TYPE_ALL_MULTICAST 0x0004

#define NDIS_PACKET_TYPE_BROADCAST 0x0008

#define NDIS_PACKET_TYPE_SOURCE_ROUTING 0x0010

#define NDIS_PACKET_TYPE_PROMISCUOUS 0x0020

#define NDIS_PACKET_TYPE_SMT 0x0040

#define NDIS_PACKET_TYPE_ALL_LOCAL 0x0080

#define NDIS_PACKET_TYPE_MAC_FRAME 0x8000

#define NDIS_PACKET_TYPE_FUNCTIONAL 0x4000

#define NDIS_PACKET_TYPE_ALL_FUNCTIONAL 0x2000

#define NDIS_PACKET_TYPE_GROUP 0x1000

/* ------------------------------------------------------------- */


/* macros */
#define WORD_COUNT(packet) (sizeof(packet) / sizeof((packet)[0]))

#define MIN(a, b) ((a) < (b) ? (a) : (b))

#define NUM_ELEM(vector) (sizeof (vector) / sizeof ((vector)[0]))

/*
 * Ansi Incompatible; otherwise works fine
 */
#if 0 
#ifdef __vxworks
#define ENUM_VERBOSE_TRACE(_fmt_,vargs...) \
    { \
        char _buf_[1024]; \
        if (sprintf(_buf_,(const char *)_fmt_,##vargs) > 0) { \
            logMsg(_buf_,0,0,0,0,0,0); \
        } \
    }
#else
#define ENUM_VERBOSE_TRACE(_fmt_,vargs...) \
    { \
        char _buf_[1024]; \
        if (printf(_buf_,(const char *)_fmt_,##vargs) > 0) { \
        } \
    }
#endif

#define RNDIS_VERBOSE_TRACE(_fmt_,vargs...) \
    ENUM_VERBOSE_TRACE(_fmt_,##vargs)
#define ENUM_TRACE(_fmt_,vargs...) \
    ENUM_VERBOSE_TRACE(_fmt_,##vargs)
#define RNDIS_TRACE(_fmt_,vargs...) \
    ENUM_VERBOSE_TRACE(_fmt_,##vargs)

#endif /* #if 0 */

/*
 * This block of code fails to inline due to variable # of arguments
 */
#if 0
static __inline__ int
ENUM_VERBOSE_TRACE(char *_fmt_,...);

static __inline__ int
ENUM_VERBOSE_TRACE(char *_fmt_,...) 
{
    char _buf_[1024];
    int retStatus = 0;
    va_list vap;
 
    va_start(vap, _fmt_);
    retStatus = vsprintf(_buf_,_fmt_,vap);
    if (retStatus > 0) {
#ifdef __vxworks
        logMsg(_buf_,0,0,0,0,0,0);
#else
        printf(_buf_);
#endif
    }
    va_end(vap);
    return retStatus;
}
#endif 

/* 
 * This block of code results in declared but not used warning
 */
#if 0
static int
ENUM_VERBOSE_TRACE(char *_fmt_,...);

static int
ENUM_VERBOSE_TRACE(char *_fmt_,...)
{ 
    char _buf_[1024]; 
    int retStatus = 0;
    va_list vap; 

    va_start(vap, _fmt_); 
    retStatus = vsprintf(_buf_,_fmt_,vap);
    if (retStatus > 0) { 
#ifdef __vxworks
        logMsg(_buf_,0,0,0,0,0,0); 
#else
        printf(_buf_);
#endif
    } 
    va_end(vap); 
    return retStatus;
}
#endif

/*
 * This function is implemented in IxUSBRNDISEnumerator.c
 */
extern int
enum_verbose_log_trace(char *_fmt_,...);

#define ENUM_VERBOSE_TRACE\
    if(rndisEnumVerboseTrace) enum_verbose_log_trace
#define RNDIS_VERBOSE_TRACE \
    if(rndisEnumVerboseTrace) ENUM_VERBOSE_TRACE
#define ENUM_TRACE \
    if (rndisEnumTrace) ENUM_VERBOSE_TRACE
#define RNDIS_TRACE \
    if (rndisEnumTrace) ENUM_VERBOSE_TRACE

/* debug logging */
#ifndef __vxworks
#undef  logMsg
#define logMsg if(0) printf
#endif

#define ERROR_TRACE logMsg

#if 0
#define ENUM_TRACE if (rndisEnumTrace) logMsg
#define ENUM_VERBOSE_TRACE if (rndisEnumVerboseTrace) logMsg
#endif

#if 0
#define ENUM_TRACE logMsg
#define ENUM_VERBOSE_TRACE if (0) logMsg
#define RNDIS_TRACE logMsg
#define RNDIS_VERBOSE_TRACE if (0) logMsg 
#endif


#if 0
#define DUMP_ETHERNET_FRAME(mblk) { BOOL ipPacket; ixUSBRNDISEthernetHdrDump(IX_USB_MBLK_DATA(mblk), &ipPacket); }
#define DUMP_IP_FRAME(mblk)  { ixUSBRNDISIpHdrDump(IX_USB_MBLK_DATA(mblk); }
#else
#define DUMP_ETHERNET_FRAME(mblk) {  }
#define DUMP_IP_FRAME(mblk)  {  }
#endif

/* buffer abstraction */
#ifdef __vxworks
#include <IxOsBuffMgt.h>
#include <IxOsBuffPoolMgt.h>

/* RNDIS buffer abstraction layer */
#define RNDIS_BUF 	                        IX_USB_MBLK
#define RNDIS_BUF_LEN(buf) 		            IX_USB_MBLK_LEN(buf)
#define RNDIS_BUF_DATA(buf)                 IX_USB_MBLK_DATA(buf)
#define RNDIS_SYS_ALIGN_BUF(buf, alignment) { IX_USB_MBLK_DATA(buf)  += alignment; }
#define RNDIS_SYS_ALLOC_BUF(len)            ixUSBBufferAlloc((size_t) len)
#define RNDIS_SYS_SET_BUF_LEN(buf, len)     { IX_USB_MBLK_LEN(buf) = IX_USB_MBLK_PKT_LEN(buf) = len; buf->mBlkHdr.mFlags |= M_PKTHDR; }
#define RNDIS_SYS_FREE(buf)	                { IX_USB_MBLK_FREE(buf); }
#else /* linux */
#define RNDIS_BUF 	                        struct sk_buff
#define RNDIS_BUF_LEN(buf) 	                buf->len
#define RNDIS_BUF_DATA(buf)                 buf->data
#define RNDIS_SYS_ALIGN_BUF(buf, alignment) { skb_reserve(buf, alignment); }
#define RNDIS_SYS_ALLOC_BUF(len)            dev_alloc_skb(len)
#define RNDIS_SYS_SET_BUF_LEN(buf, len)     { skb_put(buf, len); }
#define RNDIS_SYS_FREE(buf)	                /* Linux END de-allocates */
#endif

#define RNDIS_FREE(buf)	                    { IX_USB_MBLK_FREE(buf); }

#endif /* ndef __doxygen_HIDE */

/* prototypes */

/**
 * @ingroup IxUSBRNDIS
 *
 * @fn PUBLIC IX_STATUS ixUSBRNDISSignalEncapsulatedCommand(void)
 *
 * @brief Function prototype for signalling encapsulation command
 *
 * @param None
 *
 * @return @li IX_SUCCESS - Succesfully signalling encapsulation command
 * @return @li IX_FAIL - Failed to signal encapsulation command for some 
 * internal reason
 */
PUBLIC IX_STATUS ixUSBRNDISSignalEncapsulatedCommand(void);

/**
 * @ingroup IxUSBRNDIS
 *
 * @fn PUBLIC IX_STATUS ixUSBRNDISProcessEncapsulatedCommand(IX_USB_MBLK *)
 *
 * @brief Function prototype for processing encapsulation command
 *
 * @param IX_USB_MBLK - memory buffer to be encapsulated
 *
 * @return @li IX_SUCCESS - Succesfully processing encapsulation command
 * @return @li IX_FAIL - Failed to process encapsulation command for some 
 * internal reason
 */
PUBLIC IX_STATUS ixUSBRNDISProcessEncapsulatedCommand(IX_USB_MBLK *);

/**
 * @ingroup IxUSBRNDIS
 *
 * @fn PUBLIC IX_STATUS ixUSBRNDISProcessDataPacket(IX_USB_MBLK *)
 *
 * @brief Function prototype for processing data packet
 *
 * @param IX_USB_MBLK - data packet to be processed
 *
 * @return @li IX_SUCCESS - Data packet successfully processed
 * @return @li IX_FAIL - Failed to process data packet
 */
PUBLIC IX_STATUS ixUSBRNDISProcessDataPacket(IX_USB_MBLK *);

/**
 * @ingroup IxUSBRNDIS
 *
 * @fn PUBLIC IX_STATUS ixUSBRNDISLayerInit(void *pDrvCtrl)
 *
 * @brief Function prototype for layer initialization
 *
 * @param pDrvCtrl - Pointer to the device to be initialized
 *
 * @return @li IX_SUCCESS - Successfully initalized
 * @return @li IX_FAIL - Failed to initialize
 */
PUBLIC IX_STATUS ixUSBRNDISLayerInit(void *pDrvCtrl);

/**
 * @ingroup IxUSBRNDIS
 *
 * @fn PUBLIC IX_STATUS ixUSBRNDISInit(void)
 *
 * @brief Function prototype for initializing RNDIS 
 *
 * @param None 
 *
 * @return @li IX_SUCCESS - Successfully initalized RNDIS
 * @return @li IX_FAIL - Failed to initialize RNDIS
 */
PUBLIC IX_STATUS ixUSBRNDISInit(void);

/**
 * @ingroup IxUSBRNDIS
 *
 * @fn PUBLIC void ixUSBRNDISUnload(void)
 *
 * @brief Function prototype for releasing the I/O memory and disconnecting the interrupt 
 *
 * @param None 
 *
 * @return None
 */
PUBLIC void ixUSBRNDISUnload(void);

/**
 * @ingroup IxUSBRNDIS
 *
 * @fn PUBLIC IX_USB_MBLK* ixUSBRNDISCreateMBuf(UINT8 *buffer, UINT32 len)
 *
 * @brief Function prototype for creating MBufs
 *
 * @param UINT8 *buffer - Pointer to a buffer  
 * @param UINT32 len - Length of buffer  
 *
 * @return @li IX_USB_MBLK - Successfully create MBUF
 * @return @li NULL - Failed to create MBUF
 */
PUBLIC IX_USB_MBLK* ixUSBRNDISCreateMBuf(UINT8 *buffer, UINT32 len);

/**
 * @ingroup IxUSBRNDIS
 *
 * @fn PUBLIC IX_STATUS ixUSBRNDISSendDataPacket(RNDIS_BUF *packet)
 *
 * @brief Function prototype for sending data packet. It is the hook for the
 * RNDIS END
 *
 * @param RNDIS *packet - Pointer to a data packet to be transmitted
 *
 * @return @li IX_SUCCESS - Successfully send a data packet
 * @return @li IX_FAIL - Failed to send a data packet
 */
PUBLIC IX_STATUS ixUSBRNDISSendDataPacket(RNDIS_BUF *packet);

/* debug */

/*
 * Function prototype for debuging purpose for dumping IP Header
 */
void ixUSBRNDISIpHdrDump(const char * const mData);

const char *ixUSBRNDISIpProtoStrGet (const UINT8 ipProto);

const char *ixUSBRNDISEthernetTypeStrGet (const UINT16 etherType);

/*
 * Function prototype for debuging purpose for dumping Ethernet Header
 */
void ixUSBRNDISEthernetHdrDump(const char * const mData, BOOL *nonIpHdrDetected);

/** @} defgroup IxUSBRNDIS*/

/** @} defgroup Codelets*/

#endif /* IxUSBRNDIS_H */
