/** @file
 * @version $Revision: 1.1.1.1 $
 * 
 * @par
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * @par
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * @par
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * @par
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * @par
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * @par
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * @par
 * -- End Intel Copyright Notice --
*/
/**
 * @brief implementation of the RNDIS layer over USB
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <net/mbuf.h>
#include <netBufLib.h>
#include <logLib.h>
#include <inetLib.h>
#include <ipProto.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/if_ether.h>

#include <IxOsServicesEndianess.h>

#include "usb.h"
#include "usbstd.h"

#include "IxUSBRNDIS.h"
#include "IxUSBRNDISVendor.h"
#include "IxUSBRNDISEnd.h"

/* prototypes */
PRIVATE IX_STATUS ixUSBRNDISSignalEncapsulatedResponse(void);
PRIVATE void ixUSBRNDISSwapWords(UINT32 *packet, UINT32 wordCount);
PRIVATE void ixUSBRNDISSendEncapsulatedResponse(void);
PRIVATE void ixUSBRNDISInitializeComplete(UINT32 requestID, UINT32 *rndisPacket);
PRIVATE void ixUSBRNDISHalt(void);
PRIVATE void ixUSBRNDISQueryComplete(UINT32 requestID, UINT32 *rndisPacket);
PRIVATE void ixUSBRNDISSetComplete(UINT32 requestID, UINT32 *rndisPacket);
PRIVATE void ixUSBRNDISReset(void);
PRIVATE void ixUSBRNDISKeepAliveComplete(UINT32 requestID);
PRIVATE void ixUSBRNDISGenericResponseCompleteSend(UINT32 messageID, UINT32 requestID);
PRIVATE void ixUSBRNDISHostToBusByteOrder(UINT32 *packet, UINT32 wordCount);
PRIVATE void ixUSBRNDISBusToHostByteOrder(UINT32 *packet, UINT32 wordCount);

extern  IX_USB_MBLK* ixUSBBufferAlloc(size_t size);

/* debug functions */
PUBLIC void ixUSBRNDISDumpPacket(void);
PRIVATE const char *ixUSBRNDISStringOIDGet(UINT32 oid);

/* imported data */
extern USBDevice usbDevice;

/* log control */
extern int rndisEnumTrace;
extern int rndisEnumVerboseTrace;

/* data */
RndisLayerData layerData;
RndisLayerState layerState;
IX_USB_MBLK *rndisResponse;
void *endDeviceDriverControl;

PUBLIC IX_STATUS
ixUSBRNDISLayerInit(void *pDrvCtrl)
{
    layerState    = RNDIS_HALTED;
    rndisResponse = NULL;

    endDeviceDriverControl = pDrvCtrl;

    layerData.currentPacketFilter = 0;
    layerData.stats.Rx            = 0;
    layerData.stats.Tx            = 0;
    layerData.stats.RxNoBuf       = 0;

    return IX_SUCCESS;
}

PRIVATE IX_STATUS
ixUSBRNDISSignalEncapsulatedResponse()
{
    IX_STATUS result;
    IX_USB_MBLK *buf;

    /* send a RESPONSE_AVAILABLE message on the interrupt IN endpoint */
    UINT32 packet[RNDIS_RESPONSE_AVAILABLE_NOTIFICATION_SIZE];

    packet[0] = RNDIS_RESPONSE_AVAILABLE;
    packet[1] = 0; /* reserved */

    /* convert packet to LE format */
    ixUSBRNDISHostToBusByteOrder(packet, WORD_COUNT(packet));

    /* submit */
    buf    = (IX_USB_MBLK *) ixUSBRNDISCreateMBuf((UINT8 *) packet, sizeof (packet));
    result = ixUSBBufferSubmit(&usbDevice, RNDIS_NOTIFICATION_ENDPOINT, buf);

    if (result != IX_SUCCESS && buf != NULL)
    {
        RNDIS_FREE(buf);
    }

    return result;
}

PRIVATE void
ixUSBRNDISHostToBusByteOrder(UINT32 *packet, UINT32 wordCount)
{
    ixUSBRNDISSwapWords(packet, wordCount);
}

PRIVATE void
ixUSBRNDISBusToHostByteOrder(UINT32 *packet, UINT32 wordCount)
{
    ixUSBRNDISSwapWords(packet, wordCount);
}

PRIVATE void
ixUSBRNDISSwapWords(UINT32 *packet, UINT32 wordCount)
{
#ifdef __BIG_ENDIAN
    UINT32 wordIndex;

    /* ONLY USE THIS FOR BE PLATFORMS */
    for (wordIndex = 0 ; wordIndex < wordCount ; wordIndex++)
    {
        UINT32 word = packet[wordIndex];

        packet[wordIndex] = (word >> 24) 
                            | ((word & 0xff0000) >> 8)
                            | ((word & 0x00ff00) << 8)
                            | ((word & 0xff) << 24);
    }
#endif /* __BIG_ENDIAN */
}

PUBLIC IX_STATUS 
ixUSBRNDISSignalEncapsulatedCommand(void)
{
    /* the actual processing takes place in ixUSBRNDISProcessEncapsulatedCommand once the data packet arrives, just reset the response here */
    if (rndisResponse != NULL)
    {
        RNDIS_FREE(rndisResponse);

        rndisResponse = NULL;
    }

    return IX_SUCCESS;
}

PUBLIC IX_STATUS 
ixUSBRNDISProcessEncapsulatedCommand(IX_USB_MBLK *packet)
{
    UINT32 messageType;
    UINT32 requestID;
    UINT32 *rndisPacket = (UINT32 *) IX_USB_MBLK_DATA(packet);

    /* convert packet to host format */
    ixUSBRNDISBusToHostByteOrder(rndisPacket, IX_USB_MBLK_LEN(packet) / sizeof(UINT32));

    /* extract the message type */
    messageType = rndisPacket[RNDIS_MESSAGE_TYPE_OFFSET];
    requestID   = rndisPacket[RNDIS_REQUEST_ID_OFFSET];

    switch (messageType)
    {
        case RNDIS_INITIALIZE:
        {
            RNDIS_VERBOSE_TRACE("USB: processing RNDIS_INITIALIZE message..\n");

            ixUSBRNDISInitializeComplete(requestID, rndisPacket);

            break;
        }
        case RNDIS_HALT:
        {
            RNDIS_VERBOSE_TRACE("USB: processing RNDIS_HALT message..\n");

            ixUSBRNDISHalt();

            break;
        }
        case RNDIS_QUERY:
        {
            RNDIS_VERBOSE_TRACE("USB: processing RNDIS_QUERY message..\n");

            ixUSBRNDISQueryComplete(requestID, rndisPacket);

            break;
        }
        case RNDIS_SET:
        {
            RNDIS_VERBOSE_TRACE("USB: processing RNDIS_SET message..\n");

            ixUSBRNDISSetComplete(requestID, rndisPacket);

            break;
        }
        case RNDIS_RESET:
        {
            RNDIS_VERBOSE_TRACE("USB: processing RNDIS_RESET message..\n");

            ixUSBRNDISReset();

            break;
        }
        case RNDIS_KEEPALIVE:
        {
            RNDIS_VERBOSE_TRACE("USB: processing RNDIS_KEEPALIVE message..\n");

            ixUSBRNDISKeepAliveComplete(requestID);

            break;
        }
        default:
        {
            RNDIS_TRACE("USB: unknown RNDIS message received..\n");

            return IX_FAIL;
        }
    }

    RNDIS_FREE(packet);

    return IX_SUCCESS;
}

PUBLIC IX_STATUS 
ixUSBRNDISProcessDataPacket(IX_USB_MBLK *encapsulatedPacket)
{
    UINT32 *rndisHeader;
    UINT8 *payloadOffset;
    UINT32 payloadLength;
    RNDIS_BUF *packet;

    if (encapsulatedPacket == NULL)
    {
        RNDIS_TRACE("Warning, we've received a null packet\n");

        return IX_FAIL;
    }

    ixUSBRNDISBusToHostByteOrder((UINT32 *)IX_USB_MBLK_DATA(encapsulatedPacket),
	    RNDIS_DATA_PACKET_HEADER_SIZE / sizeof (UINT32));

    rndisHeader   = (UINT32 *) IX_USB_MBLK_DATA(encapsulatedPacket);
    payloadOffset = ((UINT8 *) IX_USB_MBLK_DATA(encapsulatedPacket)) + RNDIS_DATA_PAYLOAD_BYTE_OFFSET(encapsulatedPacket);
    payloadLength = RNDIS_DATA_PAYLOAD_BYTE_LENGTH(rndisHeader);

    packet = RNDIS_SYS_ALLOC_BUF(payloadLength + 4); /* alignment space required */

    RNDIS_VERBOSE_TRACE("USB: received RNDIS data packet, %d bytes, encapsulated data %d bytes\n", 
        payloadLength, 
        IX_USB_MBLK_LEN(encapsulatedPacket));

    if (packet == NULL)
    {
        RNDIS_TRACE("Warning, could not get a free buffer for decapsulation\n");

        RNDIS_FREE(encapsulatedPacket);

        return IX_FAIL;
    }

    /* word align ip packet */
    RNDIS_SYS_ALIGN_BUF(packet, 2);
    
    /* remove RNDIS encapsulation */
    memcpy(RNDIS_BUF_DATA(packet), payloadOffset, payloadLength);

    /* prepare packet for stack */
    RNDIS_SYS_SET_BUF_LEN(packet, payloadLength);
    
    /* free original packet */
    RNDIS_FREE(encapsulatedPacket);

    /* increment our counters */
    layerData.stats.Tx++;

    DUMP_ETHERNET_FRAME(packet);

    /* send packet to END for processing */
    rndisInt(endDeviceDriverControl, (int) packet);

    return IX_SUCCESS;
}

PUBLIC IX_STATUS 
ixUSBRNDISSendDataPacket(RNDIS_BUF *packet)
{
    UINT32 payloadLength;
    UINT32 encapsulatedPacketLength;
    IX_USB_MBLK *encapsulatedPacket;
    UINT8* encapsulatedPacketOffset;
    UINT32 *rndisHeader;
    IX_STATUS result;

    if (packet == NULL)
    {
        RNDIS_TRACE("Warning, we were requested to send a null 'packet'\n");

        return IX_FAIL;
    }

    payloadLength            = RNDIS_BUF_LEN(packet);
    encapsulatedPacketLength = payloadLength + RNDIS_DATA_PACKET_HEADER_SIZE;

    /* create a new RNDIS packet */
    encapsulatedPacket = ixUSBBufferAlloc(encapsulatedPacketLength);

    if (encapsulatedPacket == NULL)
    {
        RNDIS_TRACE("Warning, could not get a free 'packet' for encapsulation\n");

        layerData.stats.RxNoBuf++;
	
	    RNDIS_SYS_FREE(packet);

        return IX_FAIL;
    }

    encapsulatedPacketOffset = ((UINT8 *) IX_USB_MBLK_DATA(encapsulatedPacket)) + RNDIS_DATA_PACKET_HEADER_SIZE;

    /* compose the RNDIS header */
    rndisHeader    = (UINT32 *) IX_USB_MBLK_DATA(encapsulatedPacket);
    rndisHeader[0] = RNDIS_PACKET_MSG;              /* message type */
    rndisHeader[1] = encapsulatedPacketLength;      /* total length */
    rndisHeader[2] = RNDIS_PAYLOAD_OFFSET;          /* payload offset */
    rndisHeader[3] = payloadLength;                 /* payload length */

    /* unused fields */
    rndisHeader[4]  = 0; /* OOB data offset */
    rndisHeader[5]  = 0; /* OOB data length */
    rndisHeader[6]  = 0; /* OOB data elements */
    rndisHeader[7]  = 0; /* per packet info offset */
    rndisHeader[8]  = 0; /* per packet info length */
    rndisHeader[9]  = 0; /* VcHandle */
    rndisHeader[10] = 0; /* reserved */

    /* add encapsulated data */
    memcpy(encapsulatedPacketOffset, RNDIS_BUF_DATA(packet), payloadLength);

    /* endianess conversion */
    ixUSBRNDISHostToBusByteOrder((UINT32 *)IX_USB_MBLK_DATA(encapsulatedPacket),
	    RNDIS_DATA_PACKET_HEADER_SIZE / sizeof (UINT32));

    RNDIS_VERBOSE_TRACE("USB: sending RNDIS data 'packet', %d bytes\n", encapsulatedPacketLength);

    /* increment stats */
    layerData.stats.Rx++;

    /* send packet over USB */
    result = ixUSBBufferSubmit(&usbDevice, RNDIS_DATA_IN_ENDPOINT, encapsulatedPacket);

    if (result != IX_SUCCESS)
    {
        RNDIS_FREE(encapsulatedPacket);
    }

    /* free original buffer */
    RNDIS_SYS_FREE(packet);

    return result;
}

PRIVATE void 
ixUSBRNDISSendEncapsulatedResponse()
{
    ixUSBRNDISSignalEncapsulatedResponse();
}

PRIVATE void 
ixUSBRNDISInitializeComplete(UINT32 requestID, UINT32 *rndisPacket)
{
    UINT32 *responsePacket;
    
    rndisResponse = ixUSBBufferAlloc(RNDIS_INITIALIZE_COMPLETE_SIZE);

    responsePacket = (UINT32 *) IX_USB_MBLK_DATA(rndisResponse);

    /* set layer to initialized state */
    layerState = RNDIS_INITIALIZED;

    /* compose response packet */
    responsePacket[0]  = RNDIS_INITIALIZE | RNDIS_COMPLETED;
    responsePacket[1]  = RNDIS_INITIALIZE_COMPLETE_SIZE;
    responsePacket[2]  = requestID;
    responsePacket[3]  = RNDIS_STATUS_SUCCESS;
    responsePacket[4]  = RNDIS_MAJOR_VERSION;
    responsePacket[5]  = RNDIS_MINOR_VERSION;
    responsePacket[6]  = RNDIS_DF_CONNECTIONLESS;
    responsePacket[7]  = RNDIS_MEDIUM_802_3;
    responsePacket[8]  = RNDIS_MAX_PACKETS_PER_TRANSFER; /* one packet per transfer */
    responsePacket[9]  = RNDIS_MAX_HOST_PACKET_SIZE;     /* max packet size we expect from host */
    responsePacket[10] = RNDIS_PACKET_ALIGNMENT;
    responsePacket[11] = 0;     /* reserved */
    responsePacket[12] = 0;     /* reserved */

    RNDIS_TRACE("USB: Initialized RNDIS layer on request ID 0x%X\n", requestID);

    /* endianess conversion */
    ixUSBRNDISHostToBusByteOrder((UINT32 *)IX_USB_MBLK_DATA(rndisResponse),
	IX_USB_MBLK_LEN(rndisResponse) / sizeof (UINT32));

    ixUSBRNDISSendEncapsulatedResponse();
}

PRIVATE void 
ixUSBRNDISHalt()
{
    layerState = RNDIS_HALTED;

    RNDIS_TRACE("USB: RNDIS NIC down\n");

    /* clear outgoing transfers */
    ixUSBEndpointClear(&usbDevice, RNDIS_DATA_IN_ENDPOINT);
}

PRIVATE void 
ixUSBRNDISQueryComplete(UINT32 requestID, UINT32 *rndisPacket)
{
    UINT32 oid  = rndisPacket[RNDIS_OID_OFFSET];

    UINT32 *responsePacket;
    UINT32 responseBufferLength = 0;

    RNDIS_VERBOSE_TRACE("USB: RNDIS QUERY [%s]\n",
            (char *)ixUSBRNDISStringOIDGet(oid));

    rndisResponse = ixUSBBufferAlloc(1000); /* @todo change this */

    responsePacket = (UINT32 *)IX_USB_MBLK_DATA(rndisResponse);

    responsePacket[0] = RNDIS_QUERY | RNDIS_COMPLETED;
    responsePacket[2] = requestID;
    responsePacket[3] = RNDIS_STATUS_SUCCESS;

    switch (oid)
    {
        case OID_GEN_SUPPORTED_LIST:
        {
            responseBufferLength = RNDIS_NUMBER_OF_SUPPORTED_OIDS * sizeof (UINT32);

            responsePacket[6]  = OID_GEN_SUPPORTED_LIST;
            responsePacket[7]  = OID_GEN_HARDWARE_STATUS;
            responsePacket[8]  = OID_GEN_MEDIA_SUPPORTED;
            responsePacket[9]  = OID_GEN_MEDIA_IN_USE;
            responsePacket[10] = OID_GEN_MAXIMUM_FRAME_SIZE;
            responsePacket[11] = OID_GEN_LINK_SPEED;
            responsePacket[12] = OID_GEN_TRANSMIT_BLOCK_SIZE;
            responsePacket[13] = OID_GEN_RECEIVE_BLOCK_SIZE;
            responsePacket[14] = OID_GEN_VENDOR_ID;
            responsePacket[15] = OID_GEN_VENDOR_DESCRIPTION;
            responsePacket[16] = OID_GEN_CURRENT_PACKET_FILTER;
            responsePacket[17] = OID_GEN_MAXIMUM_TOTAL_SIZE;
            responsePacket[18] = OID_GEN_MEDIA_CONNECT_STATUS;
            responsePacket[19] = OID_GEN_PHYSICAL_MEDIUM;
            responsePacket[20] = OID_GEN_XMIT_OK;
            responsePacket[21] = OID_GEN_RCV_OK;
            responsePacket[22] = OID_GEN_XMIT_ERROR;
            responsePacket[23] = OID_GEN_RCV_ERROR;
            responsePacket[24] = OID_GEN_RCV_NO_BUFFER;
            responsePacket[25] = OID_802_3_PERMANENT_ADDRESS;
            responsePacket[26] = OID_802_3_CURRENT_ADDRESS;
            responsePacket[27] = OID_802_3_MULTICAST_LIST;
            responsePacket[28] = OID_802_3_MAXIMUM_LIST_SIZE;
            responsePacket[29] = OID_802_3_RCV_ERROR_ALIGNMENT;
            responsePacket[30] = OID_802_3_XMIT_ONE_COLLISION;
            responsePacket[31] = OID_802_3_XMIT_MORE_COLLISIONS;

            break;
        }
        case OID_GEN_HARDWARE_STATUS:
        {
            responseBufferLength = sizeof (UINT32);
            responsePacket[6]    = NdisHardwareStatusReady;

            break;
        }
        case OID_GEN_MEDIA_SUPPORTED: /* we support and use only Ethernet */
        case OID_GEN_MEDIA_IN_USE:
        {
            responseBufferLength = sizeof (UINT32);
            responsePacket[6]    = NdisMedium802_3;

            break;
        }
        case OID_GEN_MAXIMUM_FRAME_SIZE:
            {
                responseBufferLength = sizeof (UINT32);
                responsePacket[6]    = RNDIS_MAXIMUM_FRAME_SIZE;

                break;
            }
        case OID_GEN_TRANSMIT_BLOCK_SIZE:
        {
            responseBufferLength = sizeof (UINT32);
            responsePacket[6]    = RNDIS_TRANSMIT_BLOCK_SIZE;

            break;
        }
        case OID_GEN_RECEIVE_BLOCK_SIZE:
        {
            responseBufferLength = sizeof (UINT32);
            responsePacket[6]    = RNDIS_RECEIVE_BLOCK_SIZE;

            break;
        }
        case OID_GEN_LINK_SPEED:
        {
            responseBufferLength = sizeof (UINT32);
            responsePacket[6]    = RNDIS_LINK_SPEED;

            break;
        }
        case OID_GEN_VENDOR_ID:
        {
            responseBufferLength = sizeof (UINT32);
            responsePacket[6]    = RNDIS_VENDOR_ID;

            break;
        }
        case OID_GEN_VENDOR_DESCRIPTION:
        {
            responseBufferLength = strlen(RNDIS_VENDOR_DESCRIPTION);

            memcpy(&responsePacket[6], RNDIS_VENDOR_DESCRIPTION, responseBufferLength + 1);

            break;
        }
        case OID_GEN_CURRENT_PACKET_FILTER:
        {
            responseBufferLength = sizeof (UINT32);
            responsePacket[6]    = layerData.currentPacketFilter;

            break;
        }
        case OID_GEN_MAXIMUM_TOTAL_SIZE:
        {
            responseBufferLength = sizeof (UINT32);
            responsePacket[6]    = RNDIS_MAXIMUM_TOTAL_SIZE;

            break;
        }
        case OID_GEN_MEDIA_CONNECT_STATUS:
        {
            responseBufferLength = sizeof (UINT32);
            responsePacket[6]    = NdisMediaStateConnected; /* temporary */

            break;
        }
        case OID_GEN_PHYSICAL_MEDIUM:
        {
            responseBufferLength = sizeof (UINT32);
            responsePacket[6]    = NdisMedium802_3;
        }
        case OID_GEN_XMIT_OK:
        {
            responseBufferLength = sizeof (UINT32);
            responsePacket[6]    = layerData.stats.Tx;

            break;
        }
        case OID_GEN_RCV_OK:
        {
            responseBufferLength = sizeof (UINT32);
            responsePacket[6]    = layerData.stats.Rx;

            break;
        }
        case OID_GEN_XMIT_ERROR:
        case OID_GEN_RCV_ERROR:
        case OID_802_3_RCV_ERROR_ALIGNMENT:
        case OID_802_3_XMIT_ONE_COLLISION:
        case OID_802_3_XMIT_MORE_COLLISIONS:
        {
            responseBufferLength = sizeof (UINT32);
            responsePacket[6]    = 0; /* no such errors in USB Bulk transactions */

            break;
        }
        case OID_GEN_RCV_NO_BUFFER:
        {
            responseBufferLength = sizeof (UINT32);
            responsePacket[6]    = layerData.stats.RxNoBuf;

            break;
        }
        case OID_802_3_PERMANENT_ADDRESS:
        case OID_802_3_CURRENT_ADDRESS:
        {
            UINT8 macAddress[] = RNDIS_MAC_ADDRESS;

            responseBufferLength = 6;

            memcpy(&responsePacket[6], macAddress, 6);

            break;
        }
        case OID_802_3_MULTICAST_LIST:
        {
            break;
        }
        case OID_802_3_MAXIMUM_LIST_SIZE:
        {
            responseBufferLength = sizeof (UINT32);
            responsePacket[6]    = 64;
        }
    }

    /* set information buffer length and offset */
    responsePacket[4] = responseBufferLength;
    responsePacket[5] = 16;

    /* set total packet length */
    responsePacket[1] = RNDIS_QUERY_COMPLETE_HEADER_SIZE + responseBufferLength;

    IX_USB_MBLK_LEN(rndisResponse) = responsePacket[1];

    ixUSBRNDISHostToBusByteOrder((UINT32 *)IX_USB_MBLK_DATA(rndisResponse),
	IX_USB_MBLK_LEN(rndisResponse) / sizeof (UINT32));
    ixUSBRNDISSendEncapsulatedResponse();
}

PRIVATE void 
ixUSBRNDISSetComplete(UINT32 requestID, UINT32 *rndisPacket)
{
    UINT32 oid          = rndisPacket[RNDIS_OID_OFFSET];

    UINT32 bufferOffset = rndisPacket[RNDIS_BUFFER_OFFSET_OFFSET] / sizeof (UINT32) + RNDIS_REQUEST_ID_OFFSET;
    UINT32 bufferLength = rndisPacket[RNDIS_BUFFER_LENGTH_OFFSET];

    RNDIS_VERBOSE_TRACE("USB: RNDIS SET [%s] %d bytes\n",
          (char *)ixUSBRNDISStringOIDGet(oid), bufferLength);

    if (oid == OID_GEN_CURRENT_PACKET_FILTER)
    {
        if(rndisPacket[bufferOffset] == 0)
        {
            layerState = RNDIS_INITIALIZED;

            RNDIS_TRACE("USB: RNDIS Layer is configured to NOT pass data\n");
        }
        else
        {
            layerState = RNDIS_DATA_INITIALIZED;

            RNDIS_TRACE("USB: RNDIS Layer is configured to pass data\n");
        }
    }

    ixUSBRNDISGenericResponseCompleteSend(RNDIS_SET, requestID);
}

PRIVATE void 
ixUSBRNDISReset()
{
    /* clear outgoing transfers */
    ixUSBEndpointClear(&usbDevice, RNDIS_DATA_IN_ENDPOINT);

    ixUSBRNDISGenericResponseCompleteSend(RNDIS_RESET, 0);
}

PRIVATE void 
ixUSBRNDISKeepAliveComplete(UINT32 requestID)
{
    ixUSBRNDISGenericResponseCompleteSend(RNDIS_KEEPALIVE, requestID);
}

PRIVATE void 
ixUSBRNDISGenericResponseCompleteSend(UINT32 messageID, UINT32 requestID)
{
    UINT32 *responsePacket;

    rndisResponse = ixUSBBufferAlloc(RNDIS_GENERIC_MSG_COMPLETE_SIZE);

    responsePacket = (UINT32 *)IX_USB_MBLK_DATA(rndisResponse);

    responsePacket[0] = messageID | RNDIS_COMPLETED;
    responsePacket[1] = RNDIS_GENERIC_MSG_COMPLETE_SIZE;
    responsePacket[2] = requestID;
    responsePacket[3] = RNDIS_STATUS_SUCCESS;

    /* endianess conversion */
    ixUSBRNDISHostToBusByteOrder((UINT32 *)IX_USB_MBLK_DATA(rndisResponse),
	IX_USB_MBLK_LEN(rndisResponse) / sizeof (UINT32));

    ixUSBRNDISSendEncapsulatedResponse();
}

/***************************************************************
 *                     Debug functions                         *
 ***************************************************************/
PRIVATE const char *
ixUSBRNDISStringOIDGet(UINT32 oid)
{
    switch (oid)
    {
        case OID_GEN_SUPPORTED_LIST:            return "OID_GEN_SUPPORTED_LIST";
        case OID_GEN_HARDWARE_STATUS:           return "OID_GEN_HARDWARE_STATUS";
        case OID_GEN_MEDIA_SUPPORTED:           return "OID_GEN_MEDIA_SUPPORTED";
        case OID_GEN_MEDIA_IN_USE:              return "OID_GEN_MEDIA_IN_USE";
        case OID_GEN_MAXIMUM_FRAME_SIZE:        return "OID_GEN_MAXIMUM_FRAME_SIZE";
        case OID_GEN_TRANSMIT_BLOCK_SIZE:       return "OID_GEN_TRANSMIT_BLOCK_SIZE";
        case OID_GEN_RECEIVE_BLOCK_SIZE:        return "OID_GEN_RECEIVE_BLOCK_SIZE";
        case OID_GEN_LINK_SPEED:                return "OID_GEN_LINK_SPEED";
        case OID_GEN_VENDOR_ID:                 return "OID_GEN_VENDOR_ID";
        case OID_GEN_VENDOR_DESCRIPTION:        return "OID_GEN_VENDOR_DESCRIPTION";
        case OID_GEN_CURRENT_PACKET_FILTER:     return "OID_GEN_CURRENT_PACKET_FILTER";
        case OID_GEN_MAXIMUM_TOTAL_SIZE:        return "OID_GEN_MAXIMUM_TOTAL_SIZE";
        case OID_GEN_MEDIA_CONNECT_STATUS:      return "OID_GEN_MEDIA_CONNECT_STATUS";
        case OID_GEN_XMIT_OK:                   return "OID_GEN_XMIT_OK";
        case OID_GEN_RCV_OK:                    return "OID_GEN_RCV_OK";
        case OID_GEN_XMIT_ERROR:                return "OID_GEN_XMIT_ERROR";
        case OID_GEN_RCV_ERROR:                 return "OID_GEN_RCV_ERROR";
        case OID_802_3_RCV_ERROR_ALIGNMENT:     return "OID_802_3_RCV_ERROR_ALIGNMENT";
        case OID_802_3_XMIT_ONE_COLLISION:      return "OID_802_3_XMIT_ONE_COLLISION";
        case OID_802_3_XMIT_MORE_COLLISIONS:    return "OID_802_3_XMIT_MORE_COLLISIONS";
        case OID_GEN_RCV_NO_BUFFER:             return "OID_GEN_RCV_NO_BUFFER";
        case OID_802_3_PERMANENT_ADDRESS:       return "OID_802_3_PERMANENT_ADDRESS";
        case OID_802_3_CURRENT_ADDRESS:         return "OID_802_3_CURRENT_ADDRESS";
        case OID_802_3_MULTICAST_LIST:          return "OID_802_3_MULTICAST_LIST";
        case OID_802_3_MAXIMUM_LIST_SIZE:       return "OID_802_3_MAXIMUM_LIST_SIZE";
        default:                                return "Unknown OID";
    }
}

PUBLIC void
ixUSBRNDISDumpPacket()
{
    UINT8 *packet = (UINT8 *) IX_USB_MBLK_DATA(rndisResponse);

    RNDIS_TRACE("    RNDIS packet header dump:\n");
    RNDIS_TRACE("        type      %02X %02X %02X %02X\n",
        packet[0],
        packet[1],
        packet[2],
        packet[3]);

    RNDIS_TRACE("        length    %02X %02X %02X %02X\n",
        packet[4],
        packet[5],
        packet[6],
        packet[7]);

    RNDIS_TRACE("        requestId %02X %02X %02X %02X\n",
        packet[8],
        packet[9],
        packet[10],
        packet[11]);

    RNDIS_TRACE("        status    %02X %02X %02X %02X\n",
        packet[12],
        packet[13],
        packet[14],
        packet[15]);
}
