/** @file
 * @version $Revision: 1.1.1.1 $
 * 
 * @par
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * @par
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * @par
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * @par
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * @par
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * @par
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * @par
 * -- End Intel Copyright Notice --
*/
/* rndisEnd.c - template END  network interface driver */

/* Copyright 1984-1999 Wind River Systems, Inc. */
#include "copyright_wrs.h"

/*
modification history
--------------------
01h,12oct99,dat  SPR 28492, fixed rndisSend.
01i,29mar99,dat  documentation, SPR 26119. fixed .bS/.bE usage
01h,28feb99,pul  changed the prototype for END NET functions, to have END_OBJ
		 reference as the first argument rather than device object,
		 fix for SPR#24285
01g,17feb99,dat  documentation, removed beta warning
01f,29sep98,dat  Fixed problem in PollRx relating to SPR 22325.
01e,28sep98,dat  SPR 22325, system mode transition, plus fixed warnings
		 and template is now compilable (SPR 22204).
01d,17jul98,db   changed "holder" in rndisParse from char** to char *.
		 fixed references to "holder" in rndisParse(spr #21464).
01c,15oct97,gnn  revised to reflect the latest API changes.
01b,05may97,dat  added TODO's for documentation and macros.
01a,04feb97,gnn	 written.
*/

/*
DESCRIPTION

In this example driver the macros RNDIS_OUT_SHORT and RNDIS_IN_SHORT
are sample macros to read/write data to a mock device.  If a device
communicates through formatted control blocks in shared memory, the
accesses to those control blocks should also be through redefinable
macros.

The macros SYS_INT_CONNECT, SYS_INT_DISCONNECT, and SYS_INT_ENABLE allow
the driver to be customized for BSPs that use special versions of these
routines.

The macro SYS_INT_CONNECT is used to connect the interrupt handler to
the appropriate vector.  By default it is the routine intConnect().

The macro SYS_INT_DISCONNECT is used to disconnect the interrupt handler prior
to unloading the module.  By default this is a dummy routine that
returns OK.

The macro SYS_INT_ENABLE is used to enable the interrupt level for the
end device.  It is called once during initialization.  By default this is
the routine sysLanIntEnable(), defined in the module sysLib.o.

The macro SYS_ENET_ADDR_GET is used to get the ethernet address (MAC)
for the device.  The single argument to this routine is the END_DEVICE
pointer.  By default this routine copies the ethernet address stored in
the global variable sysTemplateEnetAddr into the END_DEVICE structure.

INCLUDES:
end.h endLib.h etherMultiLib.h

SEE ALSO: muxLib, endLib
.I "Writing and Enhanced Network Driver"
*/

/* includes */

#include "vxWorks.h"
#include "stdlib.h"
#include "cacheLib.h"
#include "intLib.h"
#include "end.h"			/* Common END structures. */
#include "endLib.h"
#include "lstLib.h"			/* Needed to maintain protocol list. */
#include "wdLib.h"
#include "iv.h"
#include "semLib.h"
#include "etherLib.h"
#include "logLib.h"
#include "netLib.h"
#include "stdio.h"
#include "sysLib.h"
#include "errno.h"
#include "errnoLib.h"
#include "memLib.h"
#include "iosLib.h"
#include "ipProto.h"
#undef	ETHER_MAP_IP_MULTICAST
#include "etherMultiLib.h"		/* multicast stuff. */
#include "etherLib.h"
#include "inetLib.h"
#include "muxLib.h"

#include "net/mbuf.h"
#include "net/unixLib.h"
#include "net/protosw.h"
#include "net/systm.h"
#include "net/if_subr.h"
#include "net/route.h"

#include "sys/socket.h"
#include "sys/ioctl.h"
#include "sys/times.h"

#include "IxUSBRNDIS.h"
#include "IxUSBRNDISEnd.h" 


IMPORT	int endMultiLstCnt (END_OBJ* pEnd);

/* defines */

/* Configuration items */

#define END_BUFSIZ      (ETHERMTU + ENET_HDR_REAL_SIZ + 6)
#define EH_SIZE		(14)
#define END_SPEED_10M	10000000	/* 10Mbs */
#define END_SPEED_100M	100000000	/* 100Mbs */
#define END_SPEED       END_SPEED_10M

/* Cache macros */

#define END_CACHE_INVALIDATE(address, len) \
        CACHE_DRV_INVALIDATE (&pDrvCtrl->cacheFuncs, (address), (len))

#define END_CACHE_PHYS_TO_VIRT(address) \
        CACHE_DRV_PHYS_TO_VIRT (&pDrvCtrl->cacheFuncs, (address))

#define END_CACHE_VIRT_TO_PHYS(address) \
        CACHE_DRV_VIRT_TO_PHYS (&pDrvCtrl->cacheFuncs, (address))

/*
 * Default macro definitions for BSP interface.
 * These macros can be redefined in a wrapper file, to generate
 * a new module with an optimized interface.
 */

/* 
 * This RNDIS End driver has no actual device therefore we don't need
 * to connect any interrupts
 */

/* Macro to connect interrupt handler to vector */
#define SYS_INT_CONNECT(pDrvCtrl,rtn,arg,pResult)

/* Macro to disconnect interrupt handler from vector */
#define SYS_INT_DISCONNECT(pDrvCtrl,rtn,arg,pResult)

/* Macro to enable the appropriate interrupt level */
#define SYS_INT_ENABLE(pDrvCtrl)

/* Macro to get the ethernet address */
#define SYS_ENET_ADDR_GET(pDevice) \
	{ \
	    unsigned char enetAddr[] = RNDIS_END_MAC_ADDRESS; \
	    bcopy ((char *) enetAddr, (char *)(&pDevice->enetAddr), 6); \
	}

/*
 * Macros to do a short (UINT16) access to the chip. Default
 * assumes a normal memory mapped device.
 */

#ifndef RNDIS_OUT_SHORT
#   define RNDIS_OUT_SHORT(pDrvCtrl,addr,value) \
	(*(USHORT *)addr = value)
#endif

#ifndef RNDIS_IN_SHORT
#   define RNDIS_IN_SHORT(pDrvCtrl,addr,pData) \
	(*pData = *addr)
#endif

/* A shortcut for getting the hardware address from the MIB II stuff. */

#define END_HADDR(pEnd)	\
		((pEnd)->mib2Tbl.ifPhysAddress.phyAddress)

#define END_HADDR_LEN(pEnd) \
		((pEnd)->mib2Tbl.ifPhysAddress.addrLength)


/* typedefs */

typedef struct
    {
    int len;
    char * pData;
    } PKT;	/* A dummy DMA data packet */

#define RNDIS_PKT_LEN_GET(pPkt) (((PKT *)pPkt)->len)

typedef struct rfd
    {
    PKT *  pPkt;
    struct rfd * next;
    } RFD;	/* dummy rx frame descriptor */

typedef struct free_args
    {
    void* arg1;
    void* arg2;
    } FREE_ARGS;

/* The definition of the driver control structure */

typedef struct end_device
    {
    END_OBJ     end;			/* The class we inherit from. */
    int		unit;			/* unit number */
    int         ivec;                   /* interrupt vector */
    int         ilevel;                 /* interrupt level */
    char*       pShMem;                	/* real ptr to shared memory */
    long	flags;			/* Our local flags. */
    UCHAR	enetAddr[6];		/* ethernet address */
    CACHE_FUNCS cacheFuncs;             /* cache function pointers */
    FUNCPTR     freeRtn[128];           /* Array of free routines. */
    struct free_args    freeData[128];  /* Array of free arguments */
                                        /* the free routines. */
    CL_POOL_ID	pClPoolId;		/* cluster pool */
    BOOL	rxHandling;		/* rcv task is scheduled */
    } END_DEVICE;

/*
 * This will only work if there is only a single unit, for multiple
 * unit device drivers these should be integrated into the END_DEVICE
 * structure.
 */

M_CL_CONFIG rndisMclBlkConfig = 	/* network mbuf configuration table */
    {
    /*
    no. mBlks		no. clBlks	memArea		memSize
    -----------		----------	-------		-------
    */
    0, 			0, 		NULL, 		0
    };

CL_DESC rndisClDescTbl [] = 	/* network cluster pool configuration table */
    {
    /*
    clusterSize			num	memArea		memSize
    -----------			----	-------		-------
    */
    {ETHERMTU + EH_SIZE + 2,	0,	NULL,		0}
    };

int rndisClDescTblNumEnt = (NELEMENTS(rndisClDescTbl));

NET_POOL rndisCmpNetPool;

/* Definitions for the flags field */

#define RNDIS_PROMISCUOUS	0x1
#define RNDIS_POLLING	0x2

/* Status register bits, returned by rndisStatusRead() */

#define RNDIS_RINT		0x1	/* Rx interrupt pending */
#define RNDIS_TINT		0x2	/* Tx interrupt pending */
#define RNDIS_RXON		0x4	/* Rx on (enabled) */
#define RNDIS_VALID_INT	0x3	/* Any valid interrupt pending */

#define RNDIS_MIN_FBUF	(1536)	/* min first buffer size */

/* DEBUG MACROS */

#ifdef DEBUG
#   define LOGMSG(x,a,b,c,d,e,f) \
	if (endDebug) \
	    { \
	    logMsg (x,a,b,c,d,e,f); \
	    }
#else
#   define LOGMSG(x,a,b,c,d,e,f)
#endif /* ENDDEBUG */

#undef DRV_DEBUG

#ifdef	DRV_DEBUG
#define DRV_DEBUG_OFF		0x0000
#define DRV_DEBUG_RX		0x0001
#define	DRV_DEBUG_TX		0x0002
#define DRV_DEBUG_INT		0x0004
#define	DRV_DEBUG_POLL		(DRV_DEBUG_POLL_RX | DRV_DEBUG_POLL_TX)
#define	DRV_DEBUG_POLL_RX	0x0008
#define	DRV_DEBUG_POLL_TX	0x0010
#define	DRV_DEBUG_LOAD		0x0020
#define	DRV_DEBUG_IOCTL		0x0040
#define DRV_DEBUG_POLL_REDIR	0x10000
#define	DRV_DEBUG_LOG_NVRAM	0x20000

int	rndisDebug = 0x00;
int     rndisTxInts=0;

#define DRV_LOG(FLG, X0, X1, X2, X3, X4, X5, X6)                        \
	if (rndisDebug & FLG)                                        \
            logMsg(X0, X1, X2, X3, X4, X5, X6);

#define DRV_PRINT(FLG,X)                                                \
	if (rndisDebug & FLG) printf X;

#else /*DRV_DEBUG*/

#define DRV_LOG(DBG_SW, X0, X1, X2, X3, X4, X5, X6)
#define DRV_PRINT(DBG_SW,X)

#endif /*DRV_DEBUG*/

/* LOCALS */

/* forward static functions */

LOCAL void	rndisReset	(END_DEVICE *pDrvCtrl);
LOCAL void	rndisHandleRcvInt (END_DEVICE *pDrvCtrl, M_BLK_ID packet);
LOCAL STATUS rndisRecv	(END_DEVICE *pDrvCtrl, M_BLK_ID packet);
LOCAL void	rndisConfig	(END_DEVICE *pDrvCtrl);

/* END Specific interfaces. */

/* This is the only externally visible interface. */

END_OBJ* 	rndisLoad (char* initString, void *unused);

LOCAL STATUS	rndisStart	(END_OBJ* pDrvCtrl);
LOCAL STATUS	rndisStop	(END_DEVICE* pDrvCtrl);
LOCAL int	rndisIoctl   (END_DEVICE* pDrvCtrl, int cmd, caddr_t data);
LOCAL STATUS	rndisUnload	(END_DEVICE* pDrvCtrl);
LOCAL STATUS	rndisSend	(END_DEVICE* pDrvCtrl, M_BLK_ID pBuf);

LOCAL STATUS	rndisMCastAdd (END_DEVICE* pDrvCtrl, char* pAddress);
LOCAL STATUS	rndisMCastDel (END_DEVICE* pDrvCtrl, char* pAddress);
LOCAL STATUS	rndisMCastGet (END_DEVICE* pDrvCtrl,
				    MULTI_TABLE* pTable);
LOCAL STATUS	rndisPollStart (END_DEVICE* pDrvCtrl);
LOCAL STATUS	rndisPollStop (END_DEVICE* pDrvCtrl);
LOCAL STATUS	rndisPollSend (END_DEVICE* pDrvCtrl, M_BLK_ID pBuf);
LOCAL STATUS	rndisPollRcv (END_DEVICE* pDrvCtrl, M_BLK_ID pBuf);
LOCAL void	rndisAddrFilterSet(END_DEVICE *pDrvCtrl);

LOCAL STATUS	rndisParse	(END_DEVICE * pDrvCtrl, char * initString);
LOCAL STATUS	rndisMemInit	(END_DEVICE * pDrvCtrl);

/*
 * Declare our function table.  This is static across all driver
 * instances.
 */

LOCAL NET_FUNCS rndisFuncTable =
    {
    (FUNCPTR) rndisStart,		/* Function to start the device. */
    (FUNCPTR) rndisStop,		/* Function to stop the device. */
    (FUNCPTR) rndisUnload,		/* Unloading function for the driver. */
    (FUNCPTR) rndisIoctl,		/* Ioctl function for the driver. */
    (FUNCPTR) rndisSend,		/* Send function for the driver. */
    (FUNCPTR) rndisMCastAdd,		/* Multicast add function for the */
					/* driver. */
    (FUNCPTR) rndisMCastDel,		/* Multicast delete function for */
					/* the driver. */
    (FUNCPTR) rndisMCastGet,		/* Multicast retrieve function for */
					/* the driver. */
    (FUNCPTR) rndisPollSend,		/* Polling send function */
    (FUNCPTR) rndisPollRcv,		/* Polling receive function */
    endEtherAddressForm,		/* put address info into a NET_BUFFER */
    endEtherPacketDataGet,	 	/* get pointer to data in NET_BUFFER */
    endEtherPacketAddrGet  		/* Get packet addresses. */
    };


/*
 * Starts the RNDIS component
 */
STATUS ixUSBRNDISStart()
{
    static END_TBL_ENTRY pDevTbl = { 0, rndisLoad, "", 1, NULL, FALSE };
    END_OBJ *         pCookie = NULL;
    INT8 inetAddr[INET_ADDR_LEN];
    INT8 nameAndUnit[32];
    INT8 pNetDev[32];

    /* Add in mux END */
    pCookie = muxDevLoad (pDevTbl.unit,
                          pDevTbl.endLoadFunc,
                          pDevTbl.endLoadString,
                          pDevTbl.endLoan, pDevTbl.pBSP);

    if (pCookie == NULL)
    {
        printf("muxDevLoad failed for device entry %d!\n", 0);

        return ERROR;
    }
    else
    {
        /* Start END */
        if (muxDevStart(pCookie) == ERROR)
        {
            printf("muxDevStart failed for device entry %d!\n", 0);

            return ERROR;
        }
    }

    /* If the RNDIS interface is not already attached, we should bring it up here
      and assign an IP address*/
    sprintf(pNetDev,"%s","usb");

    sprintf(nameAndUnit,"%s%d","usb", 0);

    /* need to figure out if this has been configured already */
    if (ifAddrGet(nameAndUnit,inetAddr) == ERROR)
    {
        if (ipAttach(0, pNetDev) != OK)
        {
            printf ("Failed to attach to device %s, unit: 0", pNetDev);
        }
        else
        {
            ifAddrSet(nameAndUnit, RNDIS_END_INET_ADDR);
        }
    }

    return OK;
}

/*******************************************************************************
*
* rndisLoad - initialize the driver and device
*
* This routine initializes the driver and the device to the operational state.
* All of the device specific parameters are passed in the initString.
*
* The string contains the target specific parameters like this:
*
* "register addr:int vector:int level:shmem addr:shmem size:shmem width"
*
* RETURNS: An END object pointer or NULL on error.
*/

END_OBJ* rndisLoad
    (
    char* initString,		/* String to be parsed by the driver. */
    void *unused
    )
    {
    END_DEVICE 	*pDrvCtrl;

    if (initString == NULL)
    {
        printf("Failed to load RNDIS END, null init string was given\n");

        return NULL;
    }

    /* VxWorks needs the device name on the first call */
    if (initString[0] == '\0')
    {
        bcopy("usb", initString, 4);

        printf("Returning the usb device name string..\n");

        return (END_OBJ *) OK;
    }

    printf("Loading RNDIS...\n");

    /* allocate the device structure */

    pDrvCtrl = (END_DEVICE *) calloc (sizeof (END_DEVICE), 1);

    if (pDrvCtrl == NULL)
    {
        printf("Failed to allocate END device structure\n");

	    goto errorExit;
    }

    /* parse the init string, filling in the device structure */

    if (rndisParse (pDrvCtrl, initString) == ERROR)
    {
        printf("Failed to parse initialization string\n");

	    goto errorExit;
    }

    /* Get the ethernet address. */

    SYS_ENET_ADDR_GET(pDrvCtrl);

    /* initialize the END and MIB2 parts of the structure */

    /*
     * The M2 element must come from m2Lib.h
     * This template is set up for a DIX type ethernet device.
     */

    if (END_OBJ_INIT (&pDrvCtrl->end, (DEV_OBJ *)pDrvCtrl, "usb",
                      pDrvCtrl->unit, &rndisFuncTable,
                      "Intel IXP425 USB RNDIS Driver.") == ERROR
     || END_MIB_INIT (&pDrvCtrl->end, M2_ifType_ethernet_csmacd,
                      &pDrvCtrl->enetAddr[0], 6, END_BUFSIZ,
                      END_SPEED)
		    == ERROR)
    {
        printf("Failed to initialize the END or MIB object\n");

	    goto errorExit;
    }

    /* Perform memory allocation/distribution */

    if (rndisMemInit (pDrvCtrl) == ERROR)
    {
        printf("Failed to initialize the driver memory\n");

	    goto errorExit;
    }

    /* reset and reconfigure the device */

    rndisReset (pDrvCtrl);
    rndisConfig (pDrvCtrl);

    /* load the RNDIS layer */
    ixUSBRNDISInit();
    ixUSBRNDISLayerInit(pDrvCtrl);

    /* set the flags to indicate readiness */

    END_OBJ_READY (&pDrvCtrl->end,
		    IFF_UP | IFF_RUNNING | IFF_NOTRAILERS | IFF_BROADCAST
		    /* |  IFF_MULTICAST*/);

    printf("Done loading RNDIS...");

    return (&pDrvCtrl->end);

errorExit:

    printf("Failed to load RNDIS\n");

    if (pDrvCtrl != NULL)
	free ((char *)pDrvCtrl);

    return NULL;
    }

/*******************************************************************************
*
* rndisParse - parse the init string
*
* Parse the input string.  Fill in values in the driver control structure.
*
* The muxLib.o module automatically prepends the unit number to the user's
* initialization string from the BSP (configNet.h).
*
* .IP <unit>
* Device unit number, a small integer.
* .IP <vecNum>
* Interrupt vector number (used with sysIntConnect)
* .IP <intLvl>
* Interrupt level
* .LP
*
* RETURNS: OK or ERROR for invalid arguments.
*/

STATUS rndisParse
    (
    END_DEVICE * pDrvCtrl,	/* device pointer */
    char * initString		/* information string */
    )
    {
        /* RNDIS drivers take no init strings */
        return OK;
    }

/*******************************************************************************
*
* rndisMemInit - initialize memory for the chip
*
* This routine is highly specific to the device.
*
* RETURNS: OK or ERROR.
*/

STATUS rndisMemInit
    (
    END_DEVICE * pDrvCtrl	/* device to be initialized */
    )
    {
#if 0
    int count = 0;
#endif


    /*
     * This is how we would set up and END netPool using netBufLib(1).
     * This code is pretty generic.
     */

    if ((pDrvCtrl->end.pNetPool = malloc (sizeof(NET_POOL))) == NULL)
        return (ERROR);

    rndisMclBlkConfig.mBlkNum = 16;
    rndisClDescTbl[0].clNum = 16;
    rndisMclBlkConfig.clBlkNum = rndisClDescTbl[0].clNum;

    /* Calculate the total memory for all the M-Blks and CL-Blks. */

    rndisMclBlkConfig.memSize = (rndisMclBlkConfig.mBlkNum *
				    (MSIZE + sizeof (long))) +
			      (rndisMclBlkConfig.clBlkNum *
				    (CL_BLK_SZ + sizeof(long)));

    if ((rndisMclBlkConfig.memArea = (char *) memalign (sizeof(long),
                                                  rndisMclBlkConfig.memSize))
        == NULL)
        return (ERROR);

    /* Calculate the memory size of all the clusters. */

    rndisClDescTbl[0].memSize = (rndisClDescTbl[0].clNum *
				    (END_BUFSIZ + 8)) + sizeof(int);

    /* Allocate the memory for the clusters from cache safe memory. */

    rndisClDescTbl[0].memArea =
        (char *) cacheDmaMalloc (rndisClDescTbl[0].memSize);

    if (rndisClDescTbl[0].memArea == NULL)
        {
        DRV_LOG (DRV_DEBUG_LOAD, "system memory unavailable\n",
		1, 2, 3, 4, 5, 6);
        return (ERROR);
        }

    /* Initialize the memory pool. */

    if (netPoolInit(pDrvCtrl->end.pNetPool, &rndisMclBlkConfig,
                    &rndisClDescTbl[0], rndisClDescTblNumEnt,
		    NULL) == ERROR)
        {
        DRV_LOG (DRV_DEBUG_LOAD, "Could not init buffering\n",
		1, 2, 3, 4, 5, 6);
        return (ERROR);
        }

    /*
     * If you need clusters to store received packets into then get them
     * here ahead of time.
     */

    if ((pDrvCtrl->pClPoolId = netClPoolIdGet (pDrvCtrl->end.pNetPool,
	sizeof (RFD), FALSE))
        == NULL)
        return (ERROR);

/* DH - Fix for SCR# 1171 */
#if 0
    while (count < rndisClDescTbl[0].clNum)
        {
	char * pTempBuf;

        if ((pTempBuf = (char *)netClusterGet(pDrvCtrl->end.pNetPool,
                                              pDrvCtrl->pClPoolId))
            == NULL)
            {
            DRV_LOG (DRV_DEBUG_LOAD, "Could not get a buffer\n",
                     1, 2, 3, 4, 5, 6);
            return (ERROR);
            }

        }
#endif

    DRV_LOG (DRV_DEBUG_LOAD, "Memory setup complete\n", 1, 2, 3, 4, 5, 6);

    return OK;
    }

/*******************************************************************************
*
* rndisStart - start the device
*
* This function calls BSP functions to connect interrupts and start the
* device running in interrupt mode.
*
* RETURNS: OK or ERROR
*
*/

LOCAL STATUS rndisStart
    (
    END_OBJ * pDrvCtrl	/* device ID */
    )
    {
    DRV_LOG (DRV_DEBUG_LOAD, "Interrupt connected.\n", 1, 2, 3, 4, 5, 6);

    SYS_INT_ENABLE (pDrvCtrl);

    DRV_LOG (DRV_DEBUG_LOAD, "interrupt enabled.\n", 1, 2, 3, 4, 5, 6);

    return (OK);
    }


/*******************************************************************************
*
* rndisInt - handle controller interrupt
*
* This routine is called at interrupt level in response to an interrupt from
* the controller.
*
* RETURNS: N/A.
*/
void rndisInt(void *_pDrvCtrl, int _packet)
{
    END_DEVICE  *pDrvCtrl = (END_DEVICE *) _pDrvCtrl;	/* interrupting device */
    M_BLK_ID packet = (M_BLK_ID) _packet;

    /* Note: packets come ready-made from the USB RNDIS layer */

    DRV_LOG (DRV_DEBUG_INT, "Got an interrupt!\n", 1, 2, 3, 4, 5, 6);

    /* Have netTask handle any input packets */
/*    if (!(pDrvCtrl->rxHandling))
    {*/
        pDrvCtrl->rxHandling = TRUE;
        netJobAdd ((FUNCPTR) rndisHandleRcvInt, (int) pDrvCtrl, (int) packet, 0,0,0);
/*  }*/
}


/*******************************************************************************
*
* rndisRecv - process the next incoming packet
*
* Handle one incoming packet.  The packet is checked for errors.
*
* RETURNS: N/A.
*/

LOCAL STATUS rndisRecv
    (
    END_DEVICE *pDrvCtrl,	/* device structure */
    M_BLK_ID packet        /* packet to process */
    )
    {
    /* Add one to our unicast data. */
    END_ERR_ADD (&pDrvCtrl->end, MIB2_IN_UCAST, +1);

    DRV_LOG (DRV_DEBUG_RX, "Calling upper layer!\n", 1, 2, 3, 4, 5, 6);

    /* Call the upper layer's receive routine. */

    END_RCV_RTN_CALL(&pDrvCtrl->end, packet);

    return (OK);
    }

/*******************************************************************************
*
* rndisHandleRcvInt - task level interrupt service for input packets
*
* This routine is called at task level indirectly by the interrupt
* service routine to do any message received processing.
*
* The double loop is to protect against a race condition where the interrupt
* code see rxHandling as TRUE, but it is then turned off by task code.
* This race is not fatal, but does cause occassional delays until a second
* packet is received and then triggers the netTask to call this routine again.
*
* RETURNS: N/A.
*/

LOCAL void rndisHandleRcvInt
    (
    END_DEVICE *pDrvCtrl,	/* interrupting device */
    M_BLK_ID packet
    )
    {

#if 0

       /* dump packet */
       {
         int len = packet->m_len;
         int byteIndex;

         printf("Received packet, %d bytes\n : MBUF Data Ptr : 0x%08X\n", len,packet->m_data);
         printf("--------------------------------------------------------------------------\n");

         for (byteIndex = 0 ; byteIndex < len ; byteIndex++)
         {
             printf("%02X ", packet->m_data[byteIndex]);

             if (byteIndex > 0 && byteIndex % 24 == 0) printf("\n");
         }

         if (byteIndex % 24 != 0) printf("\n");

         printf("--------------------------------------------------------------------------\n");
       }
#endif

       rndisRecv (pDrvCtrl, packet);
    }

/*******************************************************************************
*
* rndisSend - the driver send routine
*
* This routine takes a M_BLK_ID sends off the data in the M_BLK_ID.
* The buffer must already have the addressing information properly installed
* in it.  This is done by a higher layer.  The last arguments are a free
* routine to be called when the device is done with the buffer and a pointer
* to the argument to pass to the free routine.
*
* RETURNS: OK, ERROR, or END_ERR_BLOCK.
*/

LOCAL STATUS rndisSend
    (
    END_DEVICE * pDrvCtrl,	/* device ptr */
    M_BLK_ID     pMblk		/* data to send */
    )
    {
    int         oldLevel = 0;

    /*
     * Obtain exclusive access to transmitter.  This is necessary because
     * we might have more than one stack transmitting at once.
     */

    if (!(pDrvCtrl->flags & RNDIS_POLLING))
	END_TX_SEM_TAKE (&pDrvCtrl->end, WAIT_FOREVER);

    /* This is the normal case where all the data is in one M_BLK_ID */

    /* Set pointers in local structures to point to data. */

    /*
     * If no buffers are available,
     * release the semaphore and return END_ERR_BLOCK.
     * Do not free packet
     */

    /* place a transmit request */

    if (!(pDrvCtrl->flags & RNDIS_POLLING))
        oldLevel = intLock ();	/* protect rndisInt */

    /* initiate device transmit */
    ixUSBRNDISSendDataPacket(pMblk);

    /* Advance our management index(es) */

    if (!(pDrvCtrl->flags & RNDIS_POLLING))
	END_TX_SEM_GIVE (&pDrvCtrl->end);

    if (!(pDrvCtrl->flags & RNDIS_POLLING))
        intUnlock (oldLevel);

    /* Bump the statistics counters. */

    END_ERR_ADD (&pDrvCtrl->end, MIB2_OUT_UCAST, +1);

    /*
     * Cleanup.  The driver must either free the packet now or
     * set up a structure so it can be freed later after a transmit
     * interrupt occurs.
     */
#if 0
    if (freeNow)
        netMblkClChainFree (pMblk);
#endif

    return (OK);
    }

/*******************************************************************************
*
* rndisIoctl - the driver I/O control routine
*
* Process an ioctl request.
*
* RETURNS: A command specific response, usually OK or ERROR.
*/

LOCAL int rndisIoctl
    (
    END_DEVICE * pDrvCtrl,	/* device receiving command */
    int cmd,			/* ioctl command code */
    caddr_t data		/* command argument */
    )
    {
    int error = 0;
    long value;

    printf("\nrndisIoctl : pDrvCtrl:0x%08X , cmd: 0x%08X \n", (unsigned int) pDrvCtrl,cmd);


    switch (cmd)
        {
        case EIOCSADDR:
	    if (data == NULL)
		return (EINVAL);
            bcopy ((char *)data, (char *)END_HADDR(&pDrvCtrl->end),
		   END_HADDR_LEN(&pDrvCtrl->end));
            break;

        case EIOCGADDR:
	    if (data == NULL)
		return (EINVAL);
            bcopy ((char *)END_HADDR(&pDrvCtrl->end), (char *)data,
		    END_HADDR_LEN(&pDrvCtrl->end));
            break;

        case EIOCSFLAGS:
	    value = (long)data;
	    if (value < 0)
		{
		value = -(--value);
		END_FLAGS_CLR (&pDrvCtrl->end, value);
		}
	    else
		{
		END_FLAGS_SET (&pDrvCtrl->end, value);
		}
	    rndisConfig (pDrvCtrl);
            break;

        case EIOCGFLAGS:
	    *(int *)data = END_FLAGS_GET(&pDrvCtrl->end);
            break;

	case EIOCPOLLSTART:	/* Begin polled operation */
	    rndisPollStart (pDrvCtrl);
	    break;

	case EIOCPOLLSTOP:	/* End polled operation */
	    rndisPollStop (pDrvCtrl);
	    break;

        case EIOCGMIB2:		/* return MIB information */
            if (data == NULL)
                return (EINVAL);
            bcopy((char *)&pDrvCtrl->end.mib2Tbl, (char *)data,
                  sizeof(pDrvCtrl->end.mib2Tbl));
            break;
        case EIOCGFBUF:		/* return minimum First Buffer for chaining */
            if (data == NULL)
                return (EINVAL);
            *(int *)data = RNDIS_MIN_FBUF;
            break;
        default:
            error = EINVAL;
        }

    return (error);
    }

/******************************************************************************
*
* rndisConfig - reconfigure the interface under us.
*
* Reconfigure the interface setting promiscuous mode, and changing the
* multicast interface list.
*
* RETURNS: N/A.
*/

LOCAL void rndisConfig
    (
    END_DEVICE *pDrvCtrl	/* device to be re-configured */
    )
    {

    /* Set promiscuous mode if it's asked for. */

    if (END_FLAGS_GET(&pDrvCtrl->end) & IFF_PROMISC)
	{
	DRV_LOG (DRV_DEBUG_IOCTL, "Setting promiscuous mode on!\n",
		1, 2, 3, 4, 5, 6);
	}
    else
	{
	DRV_LOG (DRV_DEBUG_IOCTL, "Setting promiscuous mode off!\n",
		1, 2, 3, 4, 5, 6);
	}

    /* Set up address filter for multicasting. */

    if (END_MULTI_LST_CNT(&pDrvCtrl->end) > 0)
	{
	rndisAddrFilterSet (pDrvCtrl);
	}

    return;
    }

/******************************************************************************
*
* rndisAddrFilterSet - set the address filter for multicast addresses
*
* This routine goes through all of the multicast addresses on the list
* of addresses (added with the endAddrAdd() routine) and sets the
* device's filter correctly.
*
* RETURNS: N/A.
*/

void rndisAddrFilterSet
    (
    END_DEVICE *pDrvCtrl	/* device to be updated */
    )
    {
    ETHER_MULTI* pCurr;

    pCurr = END_MULTI_LST_FIRST (&pDrvCtrl->end);

    while (pCurr != NULL)
	{
	pCurr = END_MULTI_LST_NEXT(pCurr);
	}

    }

/*******************************************************************************
*
* rndisPollRcv - routine to receive a packet in polled mode.
*
* This routine is called by a user to try and get a packet from the
* device.
*
* RETURNS: OK upon success.  EAGAIN is returned when no packet is available.
*/

LOCAL STATUS rndisPollRcv
    (
    END_DEVICE * pDrvCtrl,	/* device to be polled */
    M_BLK_ID      pMblk		/* ptr to buffer */
    )
    {
        return ERROR;
    }

/*******************************************************************************
*
* rndisPollSend - routine to send a packet in polled mode.
*
* This routine is called by a user to try and send a packet on the
* device.
*
* RETURNS: OK upon success.  EAGAIN if device is busy.
*/

LOCAL STATUS rndisPollSend
    (
    END_DEVICE* 	pDrvCtrl,	/* device to be polled */
    M_BLK_ID    pMblk	/* packet to send */
    )
    {
        return ERROR;
    }

/*****************************************************************************
*
* rndisMCastAdd - add a multicast address for the device
*
* This routine adds a multicast address to whatever the driver
* is already listening for.  It then resets the address filter.
*
* RETURNS: OK or ERROR.
*/

LOCAL STATUS rndisMCastAdd
    (
    END_DEVICE *pDrvCtrl,		/* device pointer */
    char* pAddress	/* new address to add */
    )
    {
    int error;

    if ((error = etherMultiAdd (&pDrvCtrl->end.multiList,
		pAddress)) == ENETRESET)
	rndisConfig (pDrvCtrl);

    return (OK);
    }

/*****************************************************************************
*
* rndisMCastDel - delete a multicast address for the device
*
* This routine removes a multicast address from whatever the driver
* is listening for.  It then resets the address filter.
*
* RETURNS: OK or ERROR.
*/

LOCAL STATUS rndisMCastDel
    (
    END_DEVICE *pDrvCtrl,		/* device pointer */
    char* pAddress		/* address to be deleted */
    )
    {
    int error;

    if ((error = etherMultiDel (&pDrvCtrl->end.multiList,
	     (char *)pAddress)) == ENETRESET)
	rndisConfig (pDrvCtrl);

    return (OK);
    }

/*****************************************************************************
*
* rndisMCastGet - get the multicast address list for the device
*
* This routine gets the multicast list of whatever the driver
* is already listening for.
*
* RETURNS: OK or ERROR.
*/

LOCAL STATUS rndisMCastGet
    (
    END_DEVICE *pDrvCtrl,			/* device pointer */
    MULTI_TABLE* pTable		/* address table to be filled in */
    )
    {
    return (etherMultiGet (&pDrvCtrl->end.multiList, pTable));
    }

/*******************************************************************************
*
* rndisStop - stop the device
*
* This function calls BSP functions to disconnect interrupts and stop
* the device from operating in interrupt mode.
*
* RETURNS: OK or ERROR.
*/

LOCAL STATUS rndisStop
    (
    END_DEVICE *pDrvCtrl	/* device to be stopped */
    )
    {
    STATUS result = OK;

    SYS_INT_DISCONNECT (pDrvCtrl, rndisInt, (int)pDrvCtrl, &result);

    if (result == ERROR)
	{
	DRV_LOG (DRV_DEBUG_LOAD, "Could not disconnect interrupt!\n",
		1, 2, 3, 4, 5, 6);
	}

    return (result);
    }

/******************************************************************************
*
* rndisUnload - unload a driver from the system
*
* This function first brings down the device, and then frees any
* stuff that was allocated by the driver in the load function.
*
* RETURNS: OK or ERROR.
*/

LOCAL STATUS rndisUnload
    (
    END_DEVICE* pDrvCtrl	/* device to be unloaded */
    )
    {
    END_OBJECT_UNLOAD (&pDrvCtrl->end);

    return (OK);
    }


/*******************************************************************************
*
* rndisPollStart - start polled mode operations
*
* RETURNS: OK or ERROR.
*/

LOCAL STATUS rndisPollStart
    (
    END_DEVICE * pDrvCtrl	/* device to be polled */
    )
    {
        return ERROR;
    }

/*******************************************************************************
*
* rndisPollStop - stop polled mode operations
*
* This function terminates polled mode operation.  The device returns to
* interrupt mode.
*
* The device interrupts are enabled, the current mode flag is switched
* to indicate interrupt mode and the device is then reconfigured for
* interrupt operation.
*
* RETURNS: OK or ERROR.
*/

LOCAL STATUS rndisPollStop
    (
    END_DEVICE * pDrvCtrl	/* device to be polled */
    )
    {
    return (OK);
    }

/*******************************************************************************
*
* rndisReset - reset device
*
* RETURNS: N/A.
*/

LOCAL void rndisReset
    (
    END_DEVICE *pDrvCtrl
    )
    {
    }
