/*
 * FileName:    IxTimersCodeletMain.c
 * Author: Intel Corporation
 * Created:     15 May 2002
 * Description: IXP425 Timers  
 *
 *
 * Design Notes:
 *
 * This codelet shows how to use the IXP425 hardware timers.
 *
 * Note that multiplications are done before divisions to
 * limit rounding errors. Also, Unary operators are used to
 * manipulate UINT64, to bypass a gcc bug.
 *
 * File Version: $Revision: 1.1.1.1 $
 * 
 * -- Intel Copyright Notice --
 * 
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * -- End Intel Copyright Notice --
 */

#include <stdio.h>

#include "IxOsServices.h"
#include "IxOsServicesMemMap.h"
#include "IxOsCacheMMU.h"
#include "ixp425.h"
#include "IxTimersCodelet_p.h"
#include "IxTimersCodeletMain.h"

/* Delay used by the timers
 *
 * if 2 timers are running with a peroid of X and Y, and if T 
 * is the graetest integer > 1 that divide both X and Y, then
 * X = T * x and Y = T * y, they both timers will expire at the same
 * cycle every T * x * y cycles
 * 
 * This test is to demonstrate timers that frequently clashes,
 */
#define IX_TIMER_TEST_RELOAD 1412   /* T */

#define IX_TIMER_1_CLOCK_RELOAD     (IX_TIMER_TEST_RELOAD * 3)
#define IX_TIMER_2_CLOCK_RELOAD     (IX_TIMER_TEST_RELOAD * 5)
#define IX_TIMER_TS_CLOCK_RELOAD    (IX_TIMER_TEST_RELOAD * 7)
#define IX_TIMER_WDOG_CLOCK_RELOAD  (IX_TIMER_TEST_RELOAD * 11)
PRIVATE UINT32 ixTimersPmuClockReload = 0;

#define IX_TIMER_TEST_DELAY_MINUTES 4   /* test delay */

#define IX_TIMER_TEST_CBTAG         0x20 /* arbitrary callback value */

#define IX_TIMER_EXP_CONFIG0_REG_OFFSET (0x20)

/* ISR setup : togle these flags to enable/disable the timers */
static BOOL timer1TestEnabled = FALSE;
static BOOL timer2TestEnabled = FALSE;
static BOOL wdogTestEnabled = FALSE;
static BOOL tsTestEnabled = FALSE;
static BOOL pmuTestEnabled = FALSE;

/* ISR counts */
static UINT64 ixTimersCodeletTimer1Count = 0;
static UINT64 ixTimersCodeletTimer2Count = 0;
static UINT64 ixTimersCodeletWdogCount = 0;
static UINT64 ixTimersCodeletTsCount = 0;
static UINT64 ixTimersCodeletPmuCount = 0;
static UINT64 ixTimersCodeletTestDuration = 0; 
static UINT64 ixTimersCodeletTimer1StartCount = 0;
static UINT64 ixTimersCodeletTimer2StartCount = 0;
static UINT64 ixTimersCodeletWdogStartCount = 0;
static UINT64 ixTimersCodeletTsStartCount = 0;
static UINT64 ixTimersCodeletPmuStartCount = 0;
static UINT64 ixTimersCodeletTimer1EndCount = 0;
static UINT64 ixTimersCodeletTimer2EndCount = 0;
static UINT64 ixTimersCodeletWdogEndCount = 0;
static UINT64 ixTimersCodeletTsEndCount = 0;
static UINT64 ixTimersCodeletPmuEndCount = 0;

static UINT32 ixXscaleCpuRate = 0;
static UINT32 ixCrystalRate = 21333;
static UINT32 ixXClockDivider = 0;
static UINT32 ixPClockDivider = 0;

static UINT32 ixp425ExpCfgBaseVirtAddress = 0;

/* Get the Xscale configuration (CPU frequency and initialise
 * the PMU clock. (OS clock may be affected by operations on
 * timers and timestamps)
 */
static IX_STATUS ixTimersPmuTickInit(void)
{
    volatile UINT32 *ixXscaleCpuClockAddr = NULL;
    UINT32 speed;
    UINT32 val;

    if (ixp425ExpCfgBaseVirtAddress == 0)
    {
	ixOsServLog(LOG_FATAL,
		    "ERROR - Must map the exp cfg base before calling the ixTimersPmuTickInit function\n",
		    0,0,0,0,0,0);
	return IX_FAIL;
    }

    /* get the processor clock speed from A21-A23 */
    ixXscaleCpuClockAddr = 
	(UINT32 *)(ixp425ExpCfgBaseVirtAddress + IX_TIMER_EXP_CONFIG0_REG_OFFSET);

    speed = (*ixXscaleCpuClockAddr >> 21) & 7;
    ixXscaleCpuRate = speed == 7 ? 533 : speed == 1 ? 400 : speed == 3 ? 266 : 0;

    if (ixXscaleCpuRate == 0 || IXP425_PERIPHERAL_BUS_CLOCK != 66)
    {
	return IX_FAIL;
    }
    /* CCLK register : configure clock speed 
     * (Refer to the 80200 coprocessor documents for more information)
    */
    val = ixXscaleCpuRate == 533 ? 6 : ixXscaleCpuRate == 400 ? 4 : 2;
    __asm__ (" mcr p14, 0, %0, c6, c1, 0; " : : "r" (val));

    /* get the ratio for the xscale core frequency (xclock)
     * and the ratio for the APB frequency (pClock)
     */
    ixXClockDivider = ixXscaleCpuRate == 533 ? 4 : ixXscaleCpuRate == 400 ? 5 : 8;
    ixPClockDivider = 32;

    return IX_SUCCESS;
}

/* Get the current time in xclock cycles 
 * (Refer to the 80200 coprocessor documents for more information)
 */
UINT32 ixTimersPmuTickGet(void)
{
    UINT32 retval;
    __asm__ volatile (" mrc p14,0,%0,c1,c1,0; " : "=r" (retval) );
    return retval;
}

/*
 * Convert the Xscale PMU ticks to nanoseconds 
 */
static UINT64 ixTimersXClockToNanosecsConvert(UINT64 local_tick)
{
    local_tick *= 10000;
    local_tick *= ixXClockDivider;
    local_tick /= ixCrystalRate;
    return local_tick;
}

/*
 * Convert the Xscale APB ticks to nanoseconds
 */
static UINT64 ixTimersPClockToNanosecsConvert(UINT64 local_tick)
{
    local_tick *= 10000;
    local_tick *= ixPClockDivider;
    local_tick /= ixCrystalRate;
    return local_tick;
}

/* 
 * Convert the Xscale p clock (66 Mz) to events/minute
 */
static UINT64 ixTimersNsecPClockRateToRateConvert(UINT64 delayNsec, 
						  UINT64 tickPeriod)
{
    if (tickPeriod == 0) 
    {
	tickPeriod = 1;
	tickPeriod <<= 32;
    }

    delayNsec /= ixTimersPClockToNanosecsConvert(tickPeriod);
    return delayNsec;
}

/* 
 * Convert the Xscale p clock (66 Mz) to events/minute
 */
static UINT64 ixTimersNsecXClockRateToRateConvert(UINT64 delayNsec, 
						  UINT64 tickPeriod)
{
    if (tickPeriod == 0) 
    {
	tickPeriod = 1;
	tickPeriod <<= 32;
    }

    delayNsec /= ixTimersXClockToNanosecsConvert(tickPeriod);
    return delayNsec;
}

/* 
 * Convert the Xscale p clock (66 Mz) to events/minute
 */
static UINT64 ixTimersNsecRateToMinutesRateConvert(UINT64 delayNsec, 
						   UINT64 tickCount)
{
    UINT64 rate = tickCount;
    rate *= 60;
    rate *= 1000000000;
    rate /= delayNsec;
    return rate;
}

/* 
 * Convert the Xscale p clock (66 Mz) to events/minute
 */
static UINT64 ixTimersPClockRateToMinutesRateConvert(UINT64 tickPeriod)
{
    UINT64 delay = 60;
    delay *= 1000000000;
    return ixTimersNsecPClockRateToRateConvert(delay, tickPeriod);
}

/* 
 * Convert the Xscale x clock (533 Mz) to events/minute
 */
static UINT64 ixTimersXClockRateToMinutesRateConvert(UINT64 tickPeriod)
{
    UINT64 delay = 60;
    delay *= 1000000000;
    return ixTimersNsecXClockRateToRateConvert(delay, tickPeriod);
}

/* Show ISR counts */
void ixTimersIsrShow(void)
{
    printf("Timer 1 ISR count .......... : %10u\n", 
	   (UINT32)ixTimersCodeletTimer1Count);
    printf("Timer 2 ISR count .......... : %10u\n", 
	   (UINT32)ixTimersCodeletTimer2Count);
    printf("Timestamp ISR count ........ : %10u\n", 
	   (UINT32)ixTimersCodeletTsCount);
    printf("Wdog ISR count ............. : %10u\n", 
	   (UINT32)ixTimersCodeletWdogCount);
    printf("Pmu ISR count .............. : %10u\n", 
	   (UINT32)ixTimersCodeletPmuCount);
}

/* Display timing estimates for this test 
 *
 * These estimates are based on the configured timers
 * and will show the expected number of interrupts per
 * timer and per minute.
 *
 * The number of interrupts per timer per minute is equal to
 * the number of bus clock per minute divided by the number 
 * of bus clock between each timer interrupt.
 * (numIntPerMinute = 60 * BusClokPerSec / timerPeriod)
 *
 * The total number of interrupts per minute is the sum of the
 * the number of interrupts per timer.
 *
 * Note that multiplications are done before divisions to
 * limit rounding errors. Also, Unary operators are used to
 * manipulate UINT64, to bypass a gcc bug.
*/
PUBLIC void ixTimersCodeletEstimatesShow(void)
{
    UINT64 interruptRatePerMinute;
    UINT64 totalInterruptRatePerMinute;

    printf("\n------------------------------------------\n");
    printf("ESTIMATES :\n\n");

    printf("Test time .................. : %10u seconds\n", 
	   (UINT32)(IX_TIMER_TEST_DELAY_MINUTES * 60));

    totalInterruptRatePerMinute = 0;

    /* timer 1 interrupts */
    if (timer1TestEnabled)
    {
	interruptRatePerMinute = 
	    ixTimersPClockRateToMinutesRateConvert(1 + IX_TIMER_1_CLOCK_RELOAD);
	totalInterruptRatePerMinute += interruptRatePerMinute;
    }
    else
    {
	interruptRatePerMinute = 0;
    }
    printf("   Timer 1 ISR rate ........ : %10u / minute\n", 
	   (UINT32)interruptRatePerMinute);

    /* timer 2 interrupts */
    if (timer2TestEnabled)
    {
	interruptRatePerMinute = 
	    ixTimersPClockRateToMinutesRateConvert(1 + IX_TIMER_2_CLOCK_RELOAD);
	totalInterruptRatePerMinute += interruptRatePerMinute;
    }
    else
    {
	interruptRatePerMinute = 0;
    }
    printf("   Timer 2 ISR rate ........ : %10u / minute\n", 
	   (UINT32)interruptRatePerMinute);

    /*
     * Timestamp interrupts
     * Timestamp interrupt rate cannot be predicted accurately
     * because it needs to be re-loaded by the ISR which cannot
     * be accurately predicted.
     */
    printf("   Timestamp ISR rate ...... :        n/a\n");

    /* Wdog interrupts */
    if (wdogTestEnabled)
    {
	interruptRatePerMinute = 
	    ixTimersPClockRateToMinutesRateConvert(1 + IX_TIMER_WDOG_CLOCK_RELOAD);
	totalInterruptRatePerMinute += interruptRatePerMinute;
    }
    else
    {
	interruptRatePerMinute = 0;
    }
    printf("   Wdog ISR rate ........... : %10u / minute\n", 
	   (UINT32)interruptRatePerMinute);

    /* pmu interrupts */
    if (pmuTestEnabled)
    {
	interruptRatePerMinute = 
	    ixTimersXClockRateToMinutesRateConvert(ixTimersPmuClockReload);
	totalInterruptRatePerMinute += interruptRatePerMinute;
    }
    else
    {
	interruptRatePerMinute = 0;
    }
    printf("   Pmu ISR rate ............ : %10u / minute\n", 
	   (UINT32)interruptRatePerMinute);

    printf("                               ----------\n");
    printf("Total ISR rate ............. : %10u / minute\n", 
	   (UINT32)totalInterruptRatePerMinute);

    /* allow some time for display */
    ixOsServTaskSleep(1000);
}


/* Display timing estimates for this test 
 *
 * These results are based on counters incremented inside each
 * ISR will show the average number of interrupts per
 * timer and per minute.
 *
 * (numIntPerMinute = IsrCount * nSPerMinute / testTimeNs)
 *
 * The total number of interrupts per minute is the sum of the
 * the number of interrupts per timer.
 * 
 * Note that multiplications are done before divisions to
 * limit rounding errors. Also, Unary operators are used to
 * manipulate UINT64, to bypass a gcc bug.
*/
PUBLIC void ixTimersCodeletResultsShow(void)
{
    UINT64 totalInterruptRatePerMinute;
    UINT64 interruptRatePerMinute;
    UINT64 testTimeInSeconds;
    UINT64 testTimeInNanoSecs;

    printf("\n------------------------------------------\n");
    printf("RESULTS :\n\n");

    /* convert the real test time to nanoseconds */
    testTimeInNanoSecs = 
	ixTimersXClockToNanosecsConvert(ixTimersCodeletTestDuration);

    testTimeInSeconds = testTimeInNanoSecs;
    testTimeInSeconds /= 1000000000;

    printf("Test time .................. : %10u seconds\n", 
	   (UINT32)testTimeInSeconds);

    totalInterruptRatePerMinute = 0;

    interruptRatePerMinute = 
	ixTimersNsecRateToMinutesRateConvert(testTimeInNanoSecs,
					     ixTimersCodeletTimer1Count);
    totalInterruptRatePerMinute += ixTimersCodeletTimer1Count;
    printf("   Timer1 ISR rate ......... : %10u / minute\n", 
	   (UINT32)interruptRatePerMinute);

    interruptRatePerMinute = 
	ixTimersNsecRateToMinutesRateConvert(testTimeInNanoSecs,
					     ixTimersCodeletTimer2Count);
    totalInterruptRatePerMinute += ixTimersCodeletTimer2Count;
    printf("   Timer2 ISR rate ......... : %10u / minute\n", 
	   (UINT32)interruptRatePerMinute);

    interruptRatePerMinute = 
	ixTimersNsecRateToMinutesRateConvert(testTimeInNanoSecs,
					     ixTimersCodeletTsCount);
    totalInterruptRatePerMinute += ixTimersCodeletTsCount;
    printf("   Timestamp ISR rate ...... : %10u / minute\n", 
	   (UINT32)interruptRatePerMinute);

    interruptRatePerMinute = 
	ixTimersNsecRateToMinutesRateConvert(testTimeInNanoSecs,
					     ixTimersCodeletWdogCount);
    totalInterruptRatePerMinute += ixTimersCodeletWdogCount;
    printf("   WatchDog ISR rate ....... : %10u / minute\n", 
	   (UINT32)interruptRatePerMinute);

    interruptRatePerMinute = 
	ixTimersNsecRateToMinutesRateConvert(testTimeInNanoSecs,
					     ixTimersCodeletPmuCount);
    totalInterruptRatePerMinute += ixTimersCodeletPmuCount;
    printf("   Pmu ISR rate ............ : %10u / minute\n", 
	   (UINT32)interruptRatePerMinute);

    totalInterruptRatePerMinute = 
	ixTimersNsecRateToMinutesRateConvert(testTimeInNanoSecs,
					     totalInterruptRatePerMinute);
    printf("                               ----------\n");
    printf("Total ISR .................. : %10u / minute\n", 
	   (UINT32)totalInterruptRatePerMinute);
    
    /* allow some time for display */
    ixOsServTaskSleep(1000);
}

/* ---------------------------------------------------------- 
 * Isr functions for each timer
 */

/* TIMER 1 ISR : this ISR increments its own counter */
void ixTimersCodeletTimer1Isr(void *arg)
{
    ixTimersCodeletTimer1Count++;
}

/* TIMER 2 ISR : this ISR increments its own counter */
void ixTimersCodeletTimer2Isr(void *arg)
{
    ixTimersCodeletTimer2Count++;
}

/* TIMESTAMP ISR : this ISR increments its own counter */
void ixTimersCodeletTsIsr(void *arg)
{
    /* get the time since the timer reach 0. This time is close
     * to 0xffff... (some unsigned negative value) since the timer
     * already expired.
     */
    UINT32 downCounter;
    ixTimerDownCounterGet(IX_TIMER_TS, &downCounter);
    /* remove 14 cycles because it is the estimated delay between
     * the counter is retrieved and the timer is reset.
     * also, this depend on compile-time options
     * if (downCounter < (UINT32)-14) 
     * {
     *   downCounter += 14;
     * }
     * else
     * {
     *   downCounter = 0;
     * }
     */
    /* reload the one-shot timer value */
    ixTimerEnable(IX_TIMER_TS, IX_TIMER_TS_CLOCK_RELOAD + downCounter, TRUE);
    /* increment the ISR counter */
    ixTimersCodeletTsCount++;
}

/* WATCHDOG ISR : this ISR increments its own counter */
void ixTimersCodeletWdogIsr(void *arg)
{
    /* reload the one-shot timer value */
    ixTimerEnable(IX_TIMER_WDOG, IX_TIMER_WDOG_CLOCK_RELOAD, TRUE);
    /* increment the ISR counter */
    ixTimersCodeletWdogCount++;
}

/* PMU ISR : this ISR increments its own counter */
void ixTimersCodeletPmuIsr(void *arg)
{
    /* reload the one-shot timer value */
    ixTimerEnable(IX_TIMER_PMU, ixTimersPmuClockReload, TRUE);
    /* increment the ISR counter */
    ixTimersCodeletPmuCount++;
}

 /* Map (get virtual address) for the EXP_CONFIG space */
static IX_STATUS ixTimersCodeletMemMap(void)
{    
    ixp425ExpCfgBaseVirtAddress = 
	(UINT32) IX_OSSERV_MEM_MAP(IX_OSSERV_EXP_BUS_REGS_PHYS_BASE,
				   IX_OSSERV_EXP_REG_MAP_SIZE);
    
    if (ixp425ExpCfgBaseVirtAddress == 0)
    {
	ixOsServLog(LOG_FATAL, "TimersCodelet: Could not map ExpCfg I/O mapped memory\n", 0, 0, 0, 0, 0, 0);
	return IX_FAIL;
    }
    return IX_SUCCESS;
}

PUBLIC void ixTimersCodeletUnload(void)
{
    IX_OSSERV_MEM_UNMAP(ixp425ExpCfgBaseVirtAddress);
    ixp425ExpCfgBaseVirtAddress = 0;
}

/* ---------------------------------------------------------- 
 * Test entry point
 */

PUBLIC IX_STATUS ixTimersCodeletMain(void)
{
    IX_STATUS status;
    int key;
    UINT64 waitDelay;
    UINT64 startClock, endClock;
    
    /* Select the timers to test 
     * (The selected timers should not already be used by the system)
     */
    timer1TestEnabled = FALSE;  /* used by the OS */
    timer2TestEnabled = TRUE;
    wdogTestEnabled = FALSE;
    tsTestEnabled = TRUE;
    pmuTestEnabled = TRUE;      /* use PMU as timing source */
    ixTimersPmuClockReload = 0; /* interrupt every 2 ^ 32 xclock cycles */

    /* Map (get virtual address) for the EXP_CONFIG space */
    if (IX_SUCCESS != ixTimersCodeletMemMap())
    {
	return IX_FAIL;
    }
    
    /* initialise the PMU clock */
    status = ixTimersPmuTickInit();
    if (status != IX_SUCCESS)
    {
	ixOsServLog(LOG_ERROR, 
		    "Unable to initialise the Xscale Pmu clock\n",
		    1,2,3,4,5,6);
	return IX_FAIL;
    }

    /* display the expected rates */
    ixTimersCodeletEstimatesShow();

    /* initialiset the ixTimer utilities */
    ixTimerInit(TRUE);

    /* reset the ISR counts */
    ixTimersCodeletTimer1Count = 0;
    ixTimersCodeletTimer2Count = 0;
    ixTimersCodeletWdogCount = 0;
    ixTimersCodeletTsCount = 0;
    ixTimersCodeletPmuCount = 0;

    /* setup the timers */
    if (timer1TestEnabled)
    {
	status = ixTimerBind(IX_TIMER_1, 
			     ixTimersCodeletTimer1Isr, 
			     (void *)IX_TIMER_TEST_CBTAG);
	if (status != IX_SUCCESS)
	{
	    ixOsServLog(LOG_ERROR, 
			"Unable to initialise Timer 1\n",1,2,3,4,5,6);
	    return IX_FAIL;
	}
    }

    if (timer2TestEnabled)
    {
	status = ixTimerBind(IX_TIMER_2, 
			     ixTimersCodeletTimer2Isr, 
			     (void *)IX_TIMER_TEST_CBTAG);
	if (status != IX_SUCCESS)
	{
	    ixOsServLog(LOG_ERROR, 
			"Unable to initialise Timer 2\n",1,2,3,4,5,6);
	    return IX_FAIL;
	}
    }

    if (wdogTestEnabled)
    {
	status = ixTimerBind(IX_TIMER_WDOG, 
			     ixTimersCodeletWdogIsr, 
			     (void *)IX_TIMER_TEST_CBTAG);
	if (status != IX_SUCCESS)
	{
	    ixOsServLog(LOG_ERROR, 
			"Unable to initialise Watchdog Timer\n",1,2,3,4,5,6);
	    return IX_FAIL;
	}
    }

    if (tsTestEnabled)
    {
	status = ixTimerBind(IX_TIMER_TS, 
			     ixTimersCodeletTsIsr, 
			     (void *)IX_TIMER_TEST_CBTAG);
	if (status != IX_SUCCESS)
	{
	    ixOsServLog(LOG_ERROR, 
			"Unable to initialise Timestamp Timer\n",1,2,3,4,5,6);
	    return IX_FAIL;
	}
    }

    /* enable the timers */
    if (timer1TestEnabled)
    {
	status = ixTimerEnable(IX_TIMER_1, IX_TIMER_1_CLOCK_RELOAD, FALSE);
	if (status != IX_SUCCESS)
	{
	    ixOsServLog(LOG_ERROR, 
			"Unable to enable Timer 1\n",1,2,3,4,5,6);
	    return IX_FAIL;
	}
    }

    if (timer2TestEnabled)
    {
	status = ixTimerEnable(IX_TIMER_2, IX_TIMER_2_CLOCK_RELOAD, FALSE);
	if (status != IX_SUCCESS)
	{
	    ixOsServLog(LOG_ERROR, 
			"Unable to enable Timer 2\n",1,2,3,4,5,6);
	    return IX_FAIL;
	}
    }

    if (wdogTestEnabled)
    {
	status = ixTimerEnable(IX_TIMER_WDOG, IX_TIMER_WDOG_CLOCK_RELOAD, TRUE);
	if (status != IX_SUCCESS)
	{
	    ixOsServLog(LOG_ERROR, 
			"Unable to enable Watchdog Timer\n",1,2,3,4,5,6);
	    return IX_FAIL;
	}
    }

    if (tsTestEnabled)
    {
	status = ixTimerEnable(IX_TIMER_TS, IX_TIMER_TS_CLOCK_RELOAD, TRUE);
	if (status != IX_SUCCESS)
	{
	    ixOsServLog(LOG_ERROR, 
			"Unable to enable Timestamp Timer\n",1,2,3,4,5,6);
	    return IX_FAIL;
	}
    }

    status = ixTimerBind(IX_TIMER_PMU, 
			 ixTimersCodeletPmuIsr, 
			 (void *)IX_TIMER_TEST_CBTAG);
    if (status != IX_SUCCESS)
    {
	ixOsServLog(LOG_ERROR, 
		    "Unable to initialise Pmu Timer\n",1,2,3,4,5,6);
	return IX_FAIL;
    }
    status = ixTimerEnable(IX_TIMER_PMU, 0, TRUE);
    if (status != IX_SUCCESS)
    {
	ixOsServLog(LOG_ERROR, 
		    "Unable to enable Pmu Timer\n",1,2,3,4,5,6);
	return IX_FAIL;
    }

    /*
     * Get a snapshot 
     */
    key = ixOsServIntLock();
    ixTimersCodeletTimer1StartCount = ixTimersCodeletTimer1Count;
    ixTimersCodeletTimer2StartCount = ixTimersCodeletTimer2Count;
    ixTimersCodeletWdogStartCount = ixTimersCodeletWdogCount;
    ixTimersCodeletTsStartCount = ixTimersCodeletTsCount;
    ixTimersCodeletPmuStartCount = ixTimersCodeletPmuCount;
    startClock = ixTimersPmuTickGet();
    ixOsServIntUnlock(key);

    /* get the number of PMU interrupts */
    waitDelay = 1 + IX_TIMER_TEST_DELAY_MINUTES *
	    ixTimersXClockRateToMinutesRateConvert(0);
        
    /* wait until the amount of time is expired, but collect the
     * current time on a regular basis. Standard OS functions can
     * not be used because they may rely on the timers being 
     * tested.
     */
    while (1)
    {
	/* do nothing for a while */
	UINT32 count = 10000000;
	while (count--)
	{
	    /* on Montavista, it is not -safe- to stay in a loop
	     * within the same 8 bytes of instructions. This is the 
	     * reason of the multiple nop operations.
	     */
	    __asm__(" nop; nop; nop; nop; nop; nop; nop; nop; nop; nop; ");
	}
	/* get the current time */
	key = ixOsServIntLock();
	if ((ixTimersCodeletPmuCount - ixTimersCodeletPmuStartCount) > waitDelay)
	{
	    ixOsServIntUnlock(key);
	    break;
	}
	ixOsServIntUnlock(key);
    }

    /* get the current snapshot */
    key = ixOsServIntLock();
    ixTimersCodeletTimer1EndCount = ixTimersCodeletTimer1Count;
    ixTimersCodeletTimer2EndCount = ixTimersCodeletTimer2Count;
    ixTimersCodeletWdogEndCount = ixTimersCodeletWdogCount;
    ixTimersCodeletTsEndCount = ixTimersCodeletTsCount;
    ixTimersCodeletPmuEndCount = ixTimersCodeletPmuCount;
    endClock = ixTimersPmuTickGet();
    ixOsServIntUnlock(key);

    /* display current timer setup */
    printf("\n------------------------------------------\n");
    ixTimerShow();

    /* disable all timers (best effort) */
    if (timer1TestEnabled)
	ixTimerDisable(IX_TIMER_1);

    if (timer2TestEnabled)
	ixTimerDisable(IX_TIMER_2);

    if (wdogTestEnabled)
	ixTimerDisable(IX_TIMER_WDOG);

    if (tsTestEnabled)
	ixTimerDisable(IX_TIMER_TS);

    ixTimerDisable(IX_TIMER_PMU);

    /* unconfigure all timers (best effort) */
    if (timer1TestEnabled)
	ixTimerUnbind(IX_TIMER_1);

    if (timer2TestEnabled)
	ixTimerUnbind(IX_TIMER_2);

    if (wdogTestEnabled)
	ixTimerUnbind(IX_TIMER_WDOG);

    if (tsTestEnabled)
	ixTimerUnbind(IX_TIMER_TS);

    ixTimerUnbind(IX_TIMER_PMU);

    /* get the very exact number of interrupts during the 
     * sampling time
     */
    ixTimersCodeletTimer1Count = 
	ixTimersCodeletTimer1EndCount - ixTimersCodeletTimer1StartCount;
    ixTimersCodeletTimer2Count =
	ixTimersCodeletTimer2EndCount - ixTimersCodeletTimer2StartCount;
    ixTimersCodeletWdogCount = 
	ixTimersCodeletWdogEndCount -ixTimersCodeletWdogStartCount;
    ixTimersCodeletTsCount = 
	ixTimersCodeletTsEndCount - ixTimersCodeletTsStartCount;
    ixTimersCodeletPmuCount = 
	ixTimersCodeletPmuEndCount - ixTimersCodeletPmuStartCount;

    /* get the exact test time (xclock) from the number of interrupts
     * and the current counters at start and end of test
     */
    ixTimersCodeletTestDuration = ixTimersCodeletPmuCount;
    ixTimersCodeletTestDuration <<= 32;
    ixTimersCodeletTestDuration += (endClock - startClock);

    /* display codelet results */
    ixTimersCodeletResultsShow();
    
    /* display internal counters */
    printf("\n------------------------------------------\n");
    printf("\nISR count details :\n\n");

    ixTimersIsrShow();

    printf("\n------------------------------------------\n");
    printf("\n");

    return IX_SUCCESS;
}

#if 0
/* ---------------------------------------------------------- 
 * Test entry point
 */

PUBLIC IX_STATUS ixTimersCodeletSmallTest(void)
{
    IX_STATUS status;
    IxTimerId timer;
    IxTimerIsr isr = NULL;
    int key;
    BOOL oneShot = FALSE;
    ixTimersPmuClockReload = 100000;

    /* initialiset the ixTimer utilities */
    ixTimerInit(FALSE);

    /* note : timer_1 is used by the system */
    for (timer = IX_TIMER_1; timer < IX_TIMER_MAX; timer++)
    {
#ifdef __vxworks
	if (timer == IX_TIMER_1) continue;  /* used by os */
#endif
#ifdef __linux
	if (timer == IX_TIMER_1) continue;  /* used by os */
#endif
	/* reset the ISR counts */
	ixTimersCodeletTimer1Count = 0;
	ixTimersCodeletTimer2Count = 0;
	ixTimersCodeletWdogCount = 0;
	ixTimersCodeletTsCount = 0;
	ixTimersCodeletPmuCount = 0;

	switch(timer)
	{
	case IX_TIMER_1:
	    printf("Timer 1 test\n");
	    isr = ixTimersCodeletTimer1Isr;
	    oneShot = FALSE;
	    break;
	case IX_TIMER_2:
	    printf("Timer 2 test\n");
	    isr = ixTimersCodeletTimer2Isr;
	    oneShot = FALSE;
	    break;
	case IX_TIMER_WDOG:
	    printf("Timer Wdog test\n");
	    isr = ixTimersCodeletWdogIsr;
	    oneShot = TRUE;
	    break;
	case IX_TIMER_TS:
	    printf("Timer Timestamp test\n");
	    isr = ixTimersCodeletTsIsr;
	    oneShot = TRUE;
	    break;
	case IX_TIMER_PMU:
	    printf("Timer Pmu test\n");
	    isr = ixTimersCodeletPmuIsr;
	    oneShot = TRUE;
	    break;
	default:
	    return IX_FAIL;
	}

	status = ixTimerBind(timer, 
			     isr, 
			     (void *)IX_TIMER_TEST_CBTAG);
	if (status != IX_SUCCESS)
	{
	    ixOsServLog(LOG_ERROR, 
			"Unable to initialise Timer %d\n",
			(int)timer,2,3,4,5,6);
	    return IX_FAIL;
	}
    

	/* enable the timer to run every 100000 cycles */
	status = ixTimerEnable(timer, 100000, oneShot);
	if (status != IX_SUCCESS)
	{
	    ixOsServLog(LOG_ERROR, 
			"Unable to enable Timer %d\n",
			(int)timer,2,3,4,5,6);
	    return IX_FAIL;
	}

	/* display current timer setup */
	printf("\n------------------------------------------\n");
	ixTimerShow();

	/*
	 * Get a snapshot 
	 */
	key = ixOsServIntLock();
	ixTimersCodeletTimer1StartCount = ixTimersCodeletTimer1Count;
	ixTimersCodeletTimer2StartCount = ixTimersCodeletTimer2Count;
	ixTimersCodeletWdogStartCount = ixTimersCodeletWdogCount;
	ixTimersCodeletTsStartCount = ixTimersCodeletTsCount;
	ixTimersCodeletPmuStartCount = ixTimersCodeletPmuCount;
	ixOsServIntUnlock(key);

	ixOsServTaskSleep(1000); /* 1 second */
	
	/* get the current snapshot */
	key = ixOsServIntLock();
	ixTimersCodeletTimer1EndCount = ixTimersCodeletTimer1Count;
	ixTimersCodeletTimer2EndCount = ixTimersCodeletTimer2Count;
	ixTimersCodeletWdogEndCount = ixTimersCodeletWdogCount;
	ixTimersCodeletTsEndCount = ixTimersCodeletTsCount;
	ixTimersCodeletPmuEndCount = ixTimersCodeletPmuCount;
	ixOsServIntUnlock(key);

	/* disable all timers (best effort) */
	ixTimerDisable(timer);
	ixTimerUnbind(timer);

	/* get the very exact number of interrupts during the 
	 * sampling time
	 */
	ixTimersCodeletTimer1Count = 
	    ixTimersCodeletTimer1EndCount - ixTimersCodeletTimer1StartCount;
	ixTimersCodeletTimer2Count =
	    ixTimersCodeletTimer2EndCount - ixTimersCodeletTimer2StartCount;
	ixTimersCodeletWdogCount = 
	    ixTimersCodeletWdogEndCount -ixTimersCodeletWdogStartCount;
	ixTimersCodeletTsCount = 
	    ixTimersCodeletTsEndCount - ixTimersCodeletTsStartCount;
	ixTimersCodeletPmuCount = 
	    ixTimersCodeletPmuEndCount - ixTimersCodeletPmuStartCount;

	printf("\n------------------------------------------\n");
	
	/* Show ISR counts */
	ixTimersIsrShow();
    }

    printf("\n------------------------------------------\n");
    printf("\n");

    return IX_SUCCESS;
}

#endif
