 /**
 * @file IxEthAccCodelet.h
 *
 * @date 22 April 2002
 *
 * @brief This file contains the interface for the Ethernet Access Codelet.
 *
 * @version $Revision: 1.1.1.1 $
 * 
 * @par
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * @par
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * @par
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * @par
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * @par
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * @par
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * @par
 * -- End Intel Copyright Notice --
 *
 */
#ifndef IXETHACCCODELET_H
#define IXETHACCCODELET_H

/**
 * @defgroup Codelets IXP425 Codelets
 *
 * @brief IXP425 EthAcc Codelet
 *
 * @{
 */

/**
 * @ingroup Codelets
 *
 * @defgroup IxEthAccCodelet IXP425 Ethernet Access Codelet (IxEthAccCodelet) API
 *
 * @brief IXP425 Ethernet Access Codelet API
 *
 * This codelet demonstrates both Ethernet Data and Control plane services and 
 * Ethernet Management services. 
 * <UL>
 *   <LI> A) Ethernet Data and Control plane services: 
 *   <UL>
 *        <LI> Configuring both ports as a receiver sink from an external source (such 
 *            as Smartbits).
 *        <LI> Configuring Port-1 to automatically transmit frames and receive frames on Port-2. 
 *             Frames generated and transmitted in Port-1 are loopbacked into Port-2 by using cross cable.
 *        <LI> Configuring and performing a software loopback on each of the two ethernet ports.
 *	  <LI> Configuring both ports to act as a bridge so that frames received on one port are retransmitted on the other.
 *   </UL>
 * </UL>
 *
 *   <UL>
 *   <LI> B) Ethernet Management services: 
 *      <UL>
 *	<LI> Adding and removing static/dynamic entries.
 *	<LI> Calling the maintenance interface (shall be run as a separate background task)
 *      <LI> Calling the show routine to display the MAC address filtering tables.    
 *      </UL>
 *   </UL>
 *
 * <b> Definition </b><br>
 * In the context of this codelet, the following definitions are applicable.<br>
 * Port 1 = ixe0 = Ethernet port associated with NPE-B Ethernet Coprocessor.<br> 
 * Port 2 = ixe1 = Ethernet port associated with NPE-C Ethernet Coprocessor.<br>    
 *
 * <b> Design constraints </b><br>
 * This codelet assumes that the underlying IXP425 Product Line Silicons have two Ethernet NPEs.
 * For silicon with single Ethernet NPE, operation will be only functional in the particular 
 * Ethernet port that corresponds to the available Ethernet NPE. Particularly, bridge operation
 * will not work as two Ethernet ports are needed in this operation. 
 *
 * <b> Assumptions </b><br>
 * This codelet illustrates the use EthAcc APIs. The operations provided may not be
 * working on the best performance as the target of this codelet is just to show the 
 * functionality of APIs. In order to get better performance, #undef
 * IX_ETHACC_CODELET_TXGENRXSINK_VERIFY to disable traffic verification. <br>
 *
 * Please note that this codelet is not optimized for production quality.
 *   
 * For performance testing, please use the operations below:
 * <UL>
 * <LI> Rx Sink Operation.
 * <LI> TxGenRxSink Operation.
 * <LI> Bridge Operation with Ethernet frames sent into either one of the Ethernet Ports.
 * </UL>
 * 
 * The operations below need special tuning to optimize them. Tuning can be done by either 
 * using a lower traffic(frames/second), reducing the value of  IX_ETHACC_CODELET_TXGEN_PCKS
 * or  #undef IX_ETHACC_CODELET_TXGENRXSINK_VERIFY.
 * <UL>  
 *  <LI> Software Loopback Operation.
 *  <LI> PHY Loopback Operation.
 *  <LI> Bridge Operation with Ethernet frames sent into both Ethernet Ports.
 * </UL>
 *
 * <b> VxWorks User Guide </b><br>
 * ixEthAccCodeletMain() function is used as a single point of execution for 
 * EthAcc Codelet. It allows user to enter selection for different type 
 * of supported operations described below: 
 *
 * <pre>
 *  <i> Usage :
 *      >ixEthAccCodeletMain (operationType)
 *      Where operationType:
 *           1 = To sink received frames as fast as possible for available ports.
 *           2 = To software loopback received frames to the same port for available ports.
 *           3 = To generate and transmit frames from port 1, remote loopback by using 
 *               an external cross cable to port 2, and received on port 2 (TxGenRxSink).
 *           4 = To generate frames and perform PHY loopback on the same port for available ports. 
 *           5 = To transmit any frame received on one port through the other one (Bridge).
 *           6 = To activate Ethernet MAC learning facility.            
 * </i>
 * </pre>
 *
 * <b> Linux User Guide </b><br>
 * The idea of using the ixEthAccCodeletMain() as a single point of execution for
 * EthAcc codelet. The operation selected will be executed when user issue 'insmod' 
 * in command prompt.
 *
 * <pre>
 * <i>  Usage :
 *      >insmod ixp400_codelets_ethAcc.o operationType=<x>
 *      Where x:
 *           1 = To sink received frames as fast as possible for available ports.
 *           2 = To software loopback received frames to the same port for available ports.
 *           3 = To generate and transmit frames from port 1, remote loopback by using 
 *               an external cross cable to port 2, and received on port 2 (TxGenRxSink).  
 *           4 = To generate frames and perform PHY loopback on the same port for available ports. 
 *           5 = To transmit any frame received on one port through the other one (Bridge).
 *           6 = To activate Ethernet MAC learning facility.  
 * </i>
 * </pre>
 *
 * <P>
 * <B>MAC Setup</B>
 * <P>
 * The default MAC setup will be:
 * <UL>
 *	<LI>Promiscuous mode enabled (for learning example)
 *	<LI>Frame Check Sequence appended for all frames generated on the XScale
 * </UL>
 *
 * <P>
 * <B>PHY Setup</B>
 * <P>
 * This codelet uses two PHYs as defined by IX_ETHACC_CODELET_MAX_PHY 
 * The default PHY setup will be: 
 * <UL>
 *	<LI>100Mbits,
 *	<LI>full duplex,
 *	<LI>auto-negotiation on.
 * </UL>
 *
 * <P>
 * <B>Jumbo frames</B>
 * <P>
 * This codelet setup enable Jumbo frame reception
 * The default setup will be: 
 * <UL>
 *	<LI>frames up to a msdu size of 9018 are supported.
 * </UL>
 *
 * <P>
 * <B>Test Equipment</B>
 * <P>
 * The test harness will consist of external test equipment capable of
 * generating Ethernet packets (e.g. SmartBits).
 * <P>
 * The test equipment must be capable of performing at least the following
 * actions to support the scenarios outlined for the Codelet:
 * <UL>
 *   <LI> Send/receive an Ethernet data stream.
 *   <LI> Send/receive frames of different length.
 *   <LI> Detect CRC errors.
 *   <LI> Append FCS.
 *   <LI> Support 100Mbit full duplex mode.
 * </UL>
 *
 * @{
 */

#include "IxTypes.h"
#include "IxEthAcc.h"
#include "IxEthAccCodelet_p.h"

/*
 * #defines for function return types, etc.
 */

/**
 * @ingroup IxEthAccCodelet 
 *
 * @def IX_ETHACC_CODELET_NPEB_MAC
 *
 * @brief Hard-encoded MAC address for NPEB.  
 */
#define IX_ETHACC_CODELET_NPEB_MAC {{0x2, 0x0, 0xa, 0xb, 0xc, 0xd}}

/**
 * @ingroup IxEthAccCodelet 
 *
 * @def IX_ETHACC_CODELET_NPEC_MAC
 *
 * @brief Hard-encoded MAC address for NPEC.
 */
#define IX_ETHACC_CODELET_NPEC_MAC {{0x2, 0x0, 0xe, 0xf, 0xa, 0xb}}

/**
 * @ingroup IxEthAccCodelet 
 *
 * @def IX_ETHACC_CODELET_RX_MBUF_POOL_SIZE
 *
 * @brief Size of receive MBuf pool.  
 */
#define IX_ETHACC_CODELET_RX_MBUF_POOL_SIZE   128

/**
 * @ingroup IxEthAccCodelet 
 *
 * @def IX_ETHACC_CODELET_TX_MBUF_POOL_SIZE
 *
 * @brief Size of transmit MBuf pool.  
 */
#define IX_ETHACC_CODELET_TX_MBUF_POOL_SIZE   128

/**
 * @ingroup IxEthAccCodelet 
 *
 * @def IX_ETHACC_CODELET_MAX_PORT
 *
 * @brief Number of Ethernet Ports supported for this codelet.
 */
#define IX_ETHACC_CODELET_MAX_PORT IX_ETH_ACC_NUMBER_OF_PORTS

/**
 * @ingroup IxEthAccCodelet 
 *
 * @def IX_ETHACC_CODELET_MBUF_POOL_SIZE
 *
 * @brief Size of MBuf pool.
 */
#define IX_ETHACC_CODELET_MBUF_POOL_SIZE \
  ((IX_ETHACC_CODELET_RX_MBUF_POOL_SIZE + IX_ETHACC_CODELET_TX_MBUF_POOL_SIZE) \
  * IX_ETHACC_CODELET_MAX_PORT)

/**
 * @ingroup IxEthAccCodelet 
 *
 * @def IX_ETHACC_CODELET_PCK_SIZE
 *
 * @brief Size of MBuf packet (recommaended size for ethAcc component)
 */
#define IX_ETHACC_CODELET_PCK_SIZE IX_ETHACC_RX_MBUF_MIN_SIZE

/**
 * @ingroup IxEthAccCodelet 
 *
 * @def IX_ETHACC_CODELET_PCK_LEN
 *
 * @brief Length of MBuf payload (in bytes).
 */
#define IX_ETHACC_CODELET_PCK_LEN 1536

/**
 * @ingroup IxEthAccCodelet 
 *
 * @def IX_ETHACC_CODELET_MBUF_DATA_POOL_SIZE
 *
 * @brief Size of MBuf data pool.
 */
#define IX_ETHACC_CODELET_MBUF_DATA_POOL_SIZE \
  (IX_ETHACC_CODELET_MBUF_POOL_SIZE * IX_ETHACC_CODELET_PCK_SIZE)

/**
 * @ingroup IxEthAccCodelet 
 *
 * @def IX_ETHACC_CODELET_TXGEN_PCK_LEN
 *
 * @brief Size of packets for TxGenRxSink Operation.
 */
#define IX_ETHACC_CODELET_TXGEN_PCK_LEN 60

/**
 * @ingroup IxEthAccCodelet 
 *
 * @def IX_ETHACC_CODELET_TXGEN_PCKS 
 *
 * @brief Number of packets to generate for the TxGenRxSink Operation.
 */
#define IX_ETHACC_CODELET_TXGEN_PCKS 128

/* 
 *  Compilation trap to ensure 
 *  IX_ETHACC_CODELET_TXGEN_PCKS <= IX_ETHACC_CODELET_TX_MBUF_POOL_SIZE
 */
#if IX_ETHACC_CODELET_TXGEN_PCKS > IX_ETHACC_CODELET_TX_MBUF_POOL_SIZE
    #error Number of TX Gen packets is greater than Tx MBuf pool
#endif

/**
 * @ingroup IxEthAccCodelet 
 *
 * @def IX_ETHACC_CODELET_TXGENRXSINK_VERIFY
 *
 * @brief Verify payload for TxGenRxSink operation. 
 *        To undefine, change to #undef.
 * 
 * @Warning Defining this will affect performance
 */
#undef IX_ETHACC_CODELET_TXGENRXSINK_VERIFY 

/**
 * @ingroup IxEthAccCodelet 
 *
 * @def IX_ETHACC_CODELET_RX_FCS_STRIP
 *
 * @brief Strip FCS from incoming frames.
 *        To undefine, change to #undef. 
 */
#define IX_ETHACC_CODELET_RX_FCS_STRIP 

/**
 * @ingroup IxEthAccCodelet 
 *
 * @def IX_ETHACC_CODELET_RX_FCS_STRIP
 *
 * @brief Append FCS for outgoing frames.
 *        To undefine, change to #undef. 
 */
#undef IX_ETHACC_CODELET_TX_FCS_APPEND

/**
 * @ingroup IxEthAccCodelet 
 *
 * @def IX_ETHACC_CODELET_FRAME_SIZE
 *
 * @brief Maximum size of a frame. 

 * This maximum frame size includes different network settings :
 * @li Ethernet frames (up to 1518 bytes), 
 * @li BabyJumbo frames (up to nearly 1600 bytes) 
 * @li Jumbo frames (9K bytes). 
 * Note that different encapsulation types may
 * extend the MTU size of 9000. The NPE firmware
 * compares onlu the overall ethernet frame size (MSDU), with may be
 * stripped from the FCS at the time of comparrison.
 *
 */
#ifdef IX_ETHACC_CODELET_RX_FCS_STRIP
#define IX_ETHACC_CODELET_FRAME_SIZE (14+9000) /* eth header + 9K MTU */
#else
#define IX_ETHACC_CODELET_FRAME_SIZE (14+9000+4)/* eth header + 9K MTU + FCS */
#endif

/*
 * Variable declarations global to this file. Externs are followed by
 * statics.
 */

/*
 * Function definitions
 */

/** 
 * @ingroup IxEthAccCodelet
 *
 * @fn ixEthAccCodeletMain(IxEthAccCodeletOperation operationType)
 *
 * @brief This function is used as a single point of execution for EthAcc codelet.
 *
 * @param "IxEthAccCodeletOperation [in] operationType" - The type of operation 
 *        to be executed. Refer to the descriptions above.
 *
 * @return
 * - IX_SUCCESS : If operation selected is successfully setup
 * - IX_FAIL    : If operation selected fails to be setup.
 */
PUBLIC IX_STATUS
ixEthAccCodeletMain(IxEthAccCodeletOperation operationType);

#endif /* IXETHACCCODELET_H */

/** @} defgroup IxEthAccCodelet*/

/** @} defgroup Codelets*/

