/**
 * @file IxAdslCtrleLib.c
 *
 * @date 2-Jun-2003
 *
 * @brief  ADSL Driver CTRL-E Library adopted from STMicro(formerly Alcatel) reference design
 *         dated 29-April-2000
 *
 * Design Notes:
 *
 * @version $Revision: 1.1.1.1 $
 * 
 * @par
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright 2000-2003 Intel Corporation All Rights Reserved.
 * 
 * @par
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * @par
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * @par
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * @par
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * @par
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * @par
 * -- End Intel Copyright Notice --
 */

#include <stdio.h>
#include "IxAdslCtrleLib.h"
#include "IxAdsl_p.h"
#include "IxAdslUtil.h"
#include "IxTypes.h"
#include "IxAdsl.h"

#if CPU == XSCALE
unsigned int ADSL_getVersion(unsigned int lineNum, T_swVersion* p_swVersion)
{	/* check whether modem in showtime  */
    unsigned char status[RECTRLE_BUFFSIZE];
    bool readstatusok,inshowtime=0;
    unsigned char versions[RECTRLE_BUFFSIZE];
    bool readversionsok;
    unsigned int returnvalue;

    if ((readstatusok = ixAdslUtilCtrleRead(lineNum,status,CTEAD_MODEM_STATUS,1)))
    {	inshowtime = (status[0]==MODEM_STATUS_UP);
    }
    /* read	far end	and near end versions  */
    if ((readversionsok = ixAdslUtilCtrleRead(lineNum,versions,CTEAD_RVERSION_NEAR_END,2)))
    {	p_swVersion->near_end =	(unsigned char)versions[0];
        p_swVersion->far_end = inshowtime?(unsigned char)versions[1]:-1;
    }

    /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_REJ if	comms ok but modem   */
    /* not in showtime, C_CONF_DATA_COMMPROBLEM if comms not ok  */

    if (!readstatusok || !readversionsok)
    {
    returnvalue =	C_CONF_DATA_COMMPROBLEM;
    }
    else
    {
        if (!inshowtime) returnvalue = C_CONF_DATA_REJ;
        else returnvalue = C_CONF_DATA_ACK;
    }
    return returnvalue;
}
#endif

#if CPU == XSCALE
unsigned int ADSL_getVendorId(unsigned int lineNum,T_VendorId * p_VendorId)
{	/* check whether modem in showtime  */
    unsigned char status[RECTRLE_BUFFSIZE];
    bool readstatusok,inshowtime=0;
    unsigned char vendorids[RECTRLE_BUFFSIZE];
    bool readvendoridsok;
    unsigned int returnvalue;

    if ((readstatusok = ixAdslUtilCtrleRead(lineNum,status,CTEAD_MODEM_STATUS,1)))
    {	inshowtime = (status[0]==MODEM_STATUS_UP);
    }

    /* read	far end	and near end vendor ids	 */

    if ((readvendoridsok = ixAdslUtilCtrleRead(lineNum,vendorids,CTEAD_RVENDOR_NEAR_END_LSB,4)))
    {	unsigned short lsb,msb;
        lsb = (unsigned	short)((unsigned char)vendorids[0]);/* first cast necessary  */
        msb = (unsigned	short)((unsigned char)vendorids[1]);
        p_VendorId->near_end = msb<<8 |	lsb;
        lsb = (unsigned	short)((unsigned char)vendorids[2]);/* first cast necessary;  */
        msb = (unsigned	short)((unsigned char)vendorids[3]);
        p_VendorId->far_end = inshowtime? msb<<8 | lsb :-1;
    }

    /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_REJ if	comms ok but modem   */
    /* not in showtime, C_CONF_DATA_COMMPROBLEM if comms not ok  */

    if (!readstatusok || !readvendoridsok) returnvalue = C_CONF_DATA_COMMPROBLEM;
    else
    {	if (!inshowtime) returnvalue = C_CONF_DATA_REJ;
        else returnvalue = C_CONF_DATA_ACK;
    }
    return returnvalue;
}
#endif

#if CPU == XSCALE
unsigned int ADSL_get_controller_SWVersion(unsigned int lineNum, T_Controller_SW_version* p_CtrlrSWVersion)
{
    /* read	controller SW version bytes  */
    unsigned char swversionbytes[RECTRLE_BUFFSIZE];
    bool readok;
    unsigned int returnvalue;

    if ((readok = ixAdslUtilCtrleRead(lineNum, swversionbytes,CTEAD_CONTROLLER_SW_VERSION,2)))
    {	unsigned char maskuppernibble=0x0F;
        p_CtrlrSWVersion->number3=(unsigned char)swversionbytes[0];
        p_CtrlrSWVersion->number2=(unsigned char)swversionbytes[1] & maskuppernibble;
        p_CtrlrSWVersion->number1=(unsigned char)swversionbytes[1]>>4 &	maskuppernibble;
    }
    /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_REJ if	comms ok but modem   */
    /* not in showtime, C_CONF_DATA_COMMPROBLEM if comms not ok  */


    if (!readok) returnvalue = C_CONF_DATA_COMMPROBLEM;
    else returnvalue = C_CONF_DATA_ACK;

    return returnvalue;
}
#endif

unsigned int ADSL_sendDMTEngineInfoCommand(unsigned int lineNum)
{

    bool commerror=FALSE;
    unsigned int returnvalue;

    /*  write address and length of	memory to be read	  */
    {	unsigned int tempint=SACHEM_VERSION_ADDRESS;
        unsigned char putdata[6];
        putdata[0]=(char)(tempint%256);	tempint=tempint/256;
        putdata[1]=(char)(tempint%256);	tempint=tempint/256;
        putdata[2]=(char)(tempint%256);	tempint=tempint/256;
        putdata[3]=(char)(tempint%256);
        putdata[4]=(char)SACHEM_VERSION_LENGTH;
        putdata[5]=(char)0;
        if (!ixAdslUtilCtrleWrite(lineNum,putdata,BLCKBADDRREG,6))
            commerror =TRUE;
    }

    /* write opcode	for Modem_Load_Block  */
    if (!commerror)
    {	unsigned char putdata[1];
        putdata[0]=(char)MODEM_LOAD_BLOCK;
        if (!ixAdslUtilCtrleWrite(lineNum,putdata,TXCOMMANDREG,1))
            commerror =TRUE;
    }

    /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_COMMPROBLEM if	comms not ok  */

    if (commerror)
        returnvalue = C_CONF_DATA_COMMPROBLEM;
    else
        returnvalue = C_CONF_DATA_ACK;

    return returnvalue;
}

unsigned int ADSL_getDMTEngineInfo(unsigned int lineNum, T_DMTEngineInfo* pinfo)
{

    bool commerror=FALSE;
    unsigned int returnvalue;


    unsigned char getdata[RECTRLE_BUFFSIZE];
    if (ixAdslUtilCtrleRead(lineNum,getdata,BLCKDATAREG,2))
    {	unsigned short lsb,msb;
        lsb = (unsigned	short)((unsigned char)getdata[0]);
        msb = (unsigned	short)((unsigned char)getdata[1]);
        pinfo->version = msb<<8	| lsb;
    }
    else commerror=TRUE;

    /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_COMMPROBLEM if	comms not ok  */

    if (commerror)
        returnvalue = C_CONF_DATA_COMMPROBLEM;
    else
        returnvalue = C_CONF_DATA_ACK;

    return returnvalue;
}

unsigned int ADSL_sendITUVendorIDInfoCommand(unsigned int lineNum, bool nearnotfar)
{	bool commerror=FALSE;
    unsigned int returnvalue;

    /* write opcode	for Modem_Load_Block  */
    if (!commerror)
    {	unsigned char putdata[1];
    putdata[0]=(char)(nearnotfar?GET_NEAR_ITU_VENDOR_ID:GET_FAR_ITU_VENDOR_ID);
        if (!ixAdslUtilCtrleWrite(lineNum,putdata,TXCOMMANDREG,1))
            commerror =TRUE;
    }

    /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_COMMPROBLEM if	comms not ok  */

    if (commerror)
        returnvalue = C_CONF_DATA_COMMPROBLEM;
    else
        returnvalue = C_CONF_DATA_ACK;

    return returnvalue;
}

unsigned int ADSL_getITUVendorIDInfo(unsigned int lineNum,T_ADSL_Requested_Info* pinfo)
{

    bool commerror=FALSE;
    unsigned int returnvalue;

    unsigned char getdata[RECTRLE_BUFFSIZE];
    if (ixAdslUtilCtrleRead(lineNum,getdata,CTEAD_ADSL_REQUESTED_INFO_0,9))
    {	pinfo->adsl_requested_info_0 = (unsigned char)getdata[0];
        pinfo->adsl_requested_info_1 = (unsigned char)getdata[1];
        pinfo->adsl_requested_info_2 = (unsigned char)getdata[2];
        pinfo->adsl_requested_info_3 = (unsigned char)getdata[3];
        pinfo->adsl_requested_info_4 = (unsigned char)getdata[4];
        pinfo->adsl_requested_info_5 = (unsigned char)getdata[5];
        pinfo->adsl_requested_info_6 = (unsigned char)getdata[6];
        pinfo->adsl_requested_info_7 = (unsigned char)getdata[7];
        pinfo->adsl_requested_info_8 = (unsigned char)getdata[8];
    }
    else commerror=TRUE;

    /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_COMMPROBLEM if	comms not ok  */

    if (commerror)
        returnvalue = C_CONF_DATA_COMMPROBLEM;
    else
        returnvalue = C_CONF_DATA_ACK;

    return returnvalue;
}

unsigned int ADSL_send_getoperationalmode_InfoCommand(unsigned int lineNum)
{	bool commerror=FALSE;
    unsigned int returnvalue;

    /* write opcode	for Modem_Load_Block  */
    if (!commerror)
    {
    unsigned char putdata[1];
    putdata[0]=(char)(GET_OPERATIONAL_MODE);
        if (!ixAdslUtilCtrleWrite(lineNum,putdata,TXCOMMANDREG,1))
            commerror =TRUE;
    }

    /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_COMMPROBLEM if	comms not ok  */

    if (commerror)
        returnvalue = C_CONF_DATA_COMMPROBLEM;
    else
        returnvalue = C_CONF_DATA_ACK;

    return returnvalue;
}

unsigned int ADSL_getoperationalmode_Info(unsigned int lineNum,T_ADSL_Requested_Info*pinfo)
{
    bool commerror=FALSE;
    unsigned int returnvalue;
    unsigned char getdata[RECTRLE_BUFFSIZE];;
    if (ixAdslUtilCtrleRead(lineNum,getdata,CTEAD_ADSL_REQUESTED_INFO_0,1))
    {	pinfo->adsl_requested_info_0 = (unsigned char)getdata[0];
    }
    else commerror=TRUE;

    /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_COMMPROBLEM if	comms not ok  */

    if (commerror)
        returnvalue = C_CONF_DATA_COMMPROBLEM;
    else
        returnvalue = C_CONF_DATA_ACK;

    return returnvalue;
}


unsigned int ADSL_sendGabaritInfoCommand(unsigned int lineNum, unsigned int offset) /* offset	in words  */
{
    bool commerror=FALSE;
    unsigned int returnvalue;

    /*  write address and length of	memory to be read	  */
    {	unsigned int tempint=GABARIT_BASE_ADDRESS+(offset*4);
        unsigned char putdata[6];
        putdata[0]=(char)(tempint%256);	tempint=tempint/256;
        putdata[1]=(char)(tempint%256);	tempint=tempint/256;
        putdata[2]=(char)(tempint%256);	tempint=tempint/256;
        putdata[3]=(char)(tempint%256);
        putdata[4]=(char)(GABARIT_BLOCK_LENGTH*4); /* GABARIT_BLOCK_LENGTH in words  */
        putdata[5]=(char)0;
        if (!ixAdslUtilCtrleWrite(lineNum,putdata,BLCKBADDRREG,6))
            commerror =TRUE;
    }

    /* write opcode	for Modem_Load_Block  */
    if (!commerror)
    {	unsigned char putdata[1];
        putdata[0]=(char)MODEM_LOAD_BLOCK;
        if (!ixAdslUtilCtrleWrite(lineNum,putdata,TXCOMMANDREG,1))
            commerror =TRUE;
    }

    /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_COMMPROBLEM if	comms not ok  */

    if (commerror)
        returnvalue = C_CONF_DATA_COMMPROBLEM;
    else
        returnvalue = C_CONF_DATA_ACK;

    return returnvalue;

}

unsigned int ADSL_getGabaritInfo(unsigned int lineNum, T_GabaritInfo*	pinfo,unsigned int offset) /* offset in	words  */
{
    bool commerror=FALSE;
    unsigned int returnvalue;
    unsigned char getdata[RECTRLE_BUFFSIZE];

    if (ixAdslUtilCtrleRead(lineNum,getdata,BLCKDATAREG,(GABARIT_BLOCK_LENGTH*4))) /* GABARIT_BLOCK_LENGTH in words  */
    {	unsigned int lsb,l2b,m3b,msb;
        int i=0;
        for ( i=0;i<GABARIT_BLOCK_LENGTH;i++)
        {	lsb = (unsigned	int)((unsigned char)getdata[i*4+0]);
            l2b = (unsigned	int)((unsigned char)getdata[i*4+1]);
            m3b = (unsigned	int)((unsigned char)getdata[i*4+2]);
            msb = (unsigned	int)((unsigned char)getdata[i*4+3]);
            pinfo->data[offset+i] =	(signed	int)(msb<<24 | m3b<<16 | l2b<<8	| lsb);
        }
    }
    else commerror=TRUE;

    /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_COMMPROBLEM if	comms not ok  */

    if (commerror)
        returnvalue = C_CONF_DATA_COMMPROBLEM;
    else
        returnvalue = C_CONF_DATA_ACK;

    return returnvalue;
}


#if CPU == XSCALE
unsigned int LT_ADSL_get_NearEndLineOperData(unsigned int lineNum, T_LTadslNearEndLineOperData* p_OperData)
{	bool commerror=FALSE; unsigned int returnvalue;

    /* check whether modem in showtime  */
    unsigned char status[RECTRLE_BUFFSIZE];
    bool inshowtime=FALSE;
    int i;

    if (ixAdslUtilCtrleRead(lineNum,status,CTEAD_MODEM_STATUS,1))
    {	inshowtime = (status[0]==MODEM_STATUS_UP);
    }
    else commerror=TRUE;

    /* read	LT near	end operational	data  */
    if (!commerror)
    {	/* relCapacityOccupation  */
        unsigned char getdata[RECTRLE_BUFFSIZE];
        if (ixAdslUtilCtrleRead(lineNum, getdata,CTEAD_LT_NEAR_RELCAPACITYOCC_UP_LSB,2))
        {	unsigned short lsb,msb;
            lsb = (unsigned	short)((unsigned char)getdata[0]);
            msb = (unsigned	short)((unsigned char)getdata[1]);
            p_OperData->relCapacityOccupationUpstr = msb<<8	| lsb;
        }
        else commerror =TRUE;
    }

    if (!commerror)
    {	/* attainable bitrate  */
        unsigned char getdata[RECTRLE_BUFFSIZE];
        if (ixAdslUtilCtrleRead(lineNum,getdata,CTEAD_LT_NEAR_ATTAINABLE_BITRATE_LSB,2))
        {	unsigned short lsb,msb;
            lsb = (unsigned	short)((unsigned char)getdata[0]);
            msb = (unsigned	short)((unsigned char)getdata[1]);
            p_OperData->attainableBitrate =	msb<<8 | lsb;
        }
        else commerror =TRUE;
    }

    if (!commerror)
    {	/* noiseMarginUpstr  */
        unsigned char getdata[RECTRLE_BUFFSIZE];
        if (ixAdslUtilCtrleRead(lineNum,getdata,CTEAD_LT_NEAR_NOISEMARGIN_UP,1))
        {	p_OperData->noiseMarginUpstr = (signed char)getdata[0];
        }
        else commerror =TRUE;
    }

    if (!commerror)
    {	/* outputpowerdownstream  */
        unsigned char getdata[RECTRLE_BUFFSIZE];
        if (ixAdslUtilCtrleRead(lineNum,getdata,CTEAD_LT_NEAR_OUTPUTPOWER_DN,1))
        {	p_OperData->outputPowerDnstr = (signed char)getdata[0];
        }
        else commerror =TRUE;
    }

    if (!commerror)
    {	/* attenuationupstr  */
        unsigned char getdata[RECTRLE_BUFFSIZE];
        if (ixAdslUtilCtrleRead(lineNum,getdata,CTEAD_LT_NEAR_ATTENUATION_UP,1))
        {	p_OperData->attenuationUpstr = (unsigned char)getdata[0];
        }
        else commerror =TRUE;
    }

    if (!commerror)
    {	/* carrierLoad	*/
        unsigned char getdata[RECTRLE_BUFFSIZE];
        if (ixAdslUtilCtrleRead(lineNum,getdata,CTEAD_LT_NEAR_CARRIER_LOAD,64))
        {	for ( i=0;i<=63;i++)
                p_OperData->carrierLoad[i] = (unsigned char)getdata[i];
        }
        else commerror =TRUE;
    }
    if (!commerror)
    {	/* carrierLoad	*/
        unsigned char getdata[RECTRLE_BUFFSIZE];
        if (ixAdslUtilCtrleRead(lineNum,getdata,CTEAD_LT_NEAR_CARRIER_LOAD+64,64))
        {	for ( i=0;i<=63;i++)
                p_OperData->carrierLoad[i+64] =	(unsigned char)getdata[i];
        }
        else commerror =TRUE;
    }
    if (!commerror)
    {	/* CTEAD_FAR_INT_CHANNELDATA_LSB  */
        unsigned char getdata[RECTRLE_BUFFSIZE];
        if (ixAdslUtilCtrleRead(lineNum,getdata,CTEAD_NEAR_INT_CHANNELDATA_LSB,2))
        {
            unsigned short lsb,msb;
                        lsb = (unsigned short)((unsigned char)getdata[0]);
                        msb = (unsigned short)((unsigned char)getdata[1]);
                        p_OperData->interleavedBitRate = inshowtime? msb<<8 | lsb:-1;

        }
        else commerror =TRUE;
    }

    if (!commerror)
    {	/* CTEAD_FAR_FAST_CHANNELDATA_LSB  */
        unsigned char getdata[RECTRLE_BUFFSIZE];
        if (ixAdslUtilCtrleRead(lineNum,getdata,CTEAD_NEAR_FAST_CHANNELDATA_LSB,2))
        {
            unsigned short lsb,msb;
                        lsb = (unsigned short)((unsigned char)getdata[0]);
                        msb = (unsigned short)((unsigned char)getdata[1]);
                        p_OperData->fastBitRate = inshowtime? msb<<8 | lsb:-1;


        }
        else commerror =TRUE;
    }

    /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_REJ if	comms ok but modem   */
    /* not in showtime, C_CONF_DATA_COMMPROBLEM if comms not ok  */
    /*unsigned int returnvalue*/
    if (commerror) returnvalue = C_CONF_DATA_COMMPROBLEM;
    else
    {	if (!inshowtime) returnvalue = C_CONF_DATA_REJ;
        else returnvalue = C_CONF_DATA_ACK;
    }
    return returnvalue;
}
#endif

#if CPU == XSCALE
unsigned int LT_ADSL_get_FarEndLineOperData(unsigned int lineNum, T_LTadslFarEndLineOperData*	p_OperData)
{	bool commerror=FALSE; unsigned int returnvalue;

    /* check whether modem in showtime  */
    unsigned char status[RECTRLE_BUFFSIZE];
    bool inshowtime=FALSE;
    if (ixAdslUtilCtrleRead(lineNum,status,CTEAD_MODEM_STATUS,1))
    {	inshowtime = (status[0]==MODEM_STATUS_UP);
    }
    else commerror=TRUE;

    /* read	LT far end operational data  */
    if (!commerror)
    {	/* relCapacityOccupation  */
        unsigned char getdata[RECTRLE_BUFFSIZE];
        if (ixAdslUtilCtrleRead(lineNum,getdata,CTEAD_LT_FAR_RELCAPACITYOCC_DN_LSB,2))
        {	unsigned short lsb,msb;
            lsb = (unsigned	short)((unsigned char)getdata[0]);
            msb = (unsigned	short)((unsigned char)getdata[1]);
            p_OperData->relCapacityOccupationDnstr = inshowtime? msb<<8 | lsb:-1;
        }
        else commerror =TRUE;
    }

    if (!commerror)
    {	/* attainable bitrate  */
        unsigned char getdata[RECTRLE_BUFFSIZE];
        if (ixAdslUtilCtrleRead(lineNum,getdata,CTEAD_LT_FAR_ATTAINABLE_BITRATE_LSB,2))
        {	unsigned short lsb,msb;
            lsb = (unsigned	short)((unsigned char)getdata[0]);
            msb = (unsigned	short)((unsigned char)getdata[1]);
            p_OperData->attainableBitrate =	inshowtime? msb<<8 | lsb:-1;
        }
        else commerror =TRUE;
    }

    if (!commerror)
    {	/* noiseMarginDnstr  */
        unsigned char getdata[RECTRLE_BUFFSIZE];
        if (ixAdslUtilCtrleRead(lineNum,getdata,CTEAD_LT_FAR_NOISEMARGIN_DN,1))
        {	p_OperData->noiseMarginDnstr = inshowtime? (signed char)getdata[0]:-1;
        }
        else commerror =TRUE;
    }

    if (!commerror)
    {	/* outputpowerupstream	*/
        unsigned char getdata[RECTRLE_BUFFSIZE];
        if (ixAdslUtilCtrleRead(lineNum,getdata,CTEAD_LT_FAR_OUTPUTPOWER_UP,1))
        {	p_OperData->outputPowerUpstr = inshowtime? (signed char)getdata[0]:-1;
        }
        else commerror =TRUE;
    }

    if (!commerror)
    {	/* attenuationdnstr  */
        unsigned char getdata[RECTRLE_BUFFSIZE];
        if (ixAdslUtilCtrleRead(lineNum,getdata,CTEAD_LT_FAR_ATTENUATION_DN,1))
        {	p_OperData->attenuationDnstr = inshowtime?(unsigned char)getdata[0]:-1;
        }
        else commerror =TRUE;
    }

    if (!commerror)
    {	/* CTEAD_FAR_INT_CHANNELDATA_LSB  */
        unsigned char getdata[RECTRLE_BUFFSIZE];
        if (ixAdslUtilCtrleRead(lineNum,getdata,CTEAD_FAR_INT_CHANNELDATA_LSB,2))
        {
            unsigned short lsb,msb;
                        lsb = (unsigned short)((unsigned char)getdata[0]);
                        msb = (unsigned short)((unsigned char)getdata[1]);
                        p_OperData->interleavedBitRate = inshowtime? msb<<8 | lsb:-1;

        }
        else commerror =TRUE;
    }

    if (!commerror)
    {	/* CTEAD_FAR_FAST_CHANNELDATA_LSB  */
        unsigned char getdata[RECTRLE_BUFFSIZE];
        if (ixAdslUtilCtrleRead(lineNum,getdata,CTEAD_FAR_FAST_CHANNELDATA_LSB,2))
        {
            unsigned short lsb,msb;
                        lsb = (unsigned short)((unsigned char)getdata[0]);
                        msb = (unsigned short)((unsigned char)getdata[1]);
                        p_OperData->fastBitRate = inshowtime? msb<<8 | lsb:-1;


        }
        else commerror =TRUE;
    }

    /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_REJ if	comms ok but modem   */
    /* not in showtime, C_CONF_DATA_COMMPROBLEM if comms not ok  */
    /*unsigned int returnvalue*/
    if (commerror) returnvalue = C_CONF_DATA_COMMPROBLEM;
    else
    {	if (!inshowtime) returnvalue = C_CONF_DATA_REJ;
        else returnvalue = C_CONF_DATA_ACK;
    }
    return returnvalue;
}
#endif

unsigned int NT_ADSL_get_NearEndLineOperData(unsigned int lineNum, T_NTadslNearEndLineOperData* p_OperData)
{	bool commerror=FALSE; unsigned int returnvalue;

    /* check whether modem in showtime  */
    unsigned char status[RECTRLE_BUFFSIZE];
    bool inshowtime=FALSE;
    if (ixAdslUtilCtrleRead(lineNum,status,CTEAD_MODEM_STATUS,1))
    {	inshowtime = (status[0]==MODEM_STATUS_UP);
    }
    else commerror=TRUE;

    /* read	NT near	end operational	data  */
    if (!commerror)
    {	/* relCapacityOccupation  */
        unsigned char getdata[RECTRLE_BUFFSIZE];
        if (ixAdslUtilCtrleRead(lineNum,getdata,CTEAD_NT_NEAR_RELCAPACITYOCC_DN_LSB,2))
        {	unsigned short lsb,msb;
            lsb = (unsigned	short)((unsigned char)getdata[0]);
            msb = (unsigned	short)((unsigned char)getdata[1]);
            p_OperData->relCapacityOccupationDnstr = msb<<8	| lsb;
        }
        else commerror =TRUE;
    }

    if (!commerror)
    {	/* noiseMargindnstr  */
        unsigned char getdata[RECTRLE_BUFFSIZE];
        if (ixAdslUtilCtrleRead(lineNum,getdata,CTEAD_NT_NEAR_NOISEMARGIN_DN,1))
        {	p_OperData->noiseMarginDnstr = (signed char)getdata[0];
        }
        else commerror =TRUE;
    }

    if (!commerror)
    {	/* outputpowerupstream	*/
        unsigned char getdata[RECTRLE_BUFFSIZE];
        if (ixAdslUtilCtrleRead(lineNum,getdata,CTEAD_NT_NEAR_OUTPUTPOWER_UP,1))
        {	p_OperData->outputPowerUpstr = (signed char)getdata[0];
        }
        else commerror =TRUE;
    }

    if (!commerror)
    {	/* attenuationdownstr  */
        unsigned char getdata[RECTRLE_BUFFSIZE];
        if (ixAdslUtilCtrleRead(lineNum,getdata,CTEAD_NT_NEAR_ATTENUATION_DN,1))
        {	p_OperData->attenuationDnstr = (unsigned char)getdata[0];
        }
        else commerror =TRUE;
    }

    /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_REJ if	comms ok but modem   */
    /* not in showtime, C_CONF_DATA_COMMPROBLEM if comms not ok  */
    /*unsigned int returnvalue*/
    if (commerror) returnvalue = C_CONF_DATA_COMMPROBLEM;
    else
    {	if (!inshowtime) returnvalue = C_CONF_DATA_REJ;
        else returnvalue = C_CONF_DATA_ACK;
    }
    return returnvalue;
}

unsigned int NT_ADSL_get_FarEndLineOperData(unsigned int lineNum, T_NTadslFarEndLineOperData*	p_OperData)
{	bool commerror=FALSE;
    unsigned int returnvalue;
    int i;

    /* check whether modem in showtime  */
    unsigned char status[RECTRLE_BUFFSIZE];
    bool inshowtime=FALSE;
    if (ixAdslUtilCtrleRead(lineNum,status,CTEAD_MODEM_STATUS,1))
    {	inshowtime = (status[0]==MODEM_STATUS_UP);
    }
    else commerror=TRUE;

    /* read	NT far end operational data  */
    if (!commerror)
    {	/* relCapacityOccupation  */
        unsigned char getdata[RECTRLE_BUFFSIZE];
        if (ixAdslUtilCtrleRead(lineNum,getdata,CTEAD_NT_FAR_RELCAPACITYOCC_UP_LSB,2))
        {	unsigned short lsb,msb;
            lsb = (unsigned	short)((unsigned char)getdata[0]);
            msb = (unsigned	short)((unsigned char)getdata[1]);
            p_OperData->relCapacityOccupationUpstr = inshowtime? msb<<8 | lsb:-1;
        }
        else commerror =TRUE;
    }

    if (!commerror)
    {	/* noiseMarginUpstr  */
        unsigned char getdata[RECTRLE_BUFFSIZE];
        if (ixAdslUtilCtrleRead(lineNum,getdata,CTEAD_NT_FAR_NOISEMARGIN_UP,1))
        {	p_OperData->noiseMarginUpstr = inshowtime?(signed char)getdata[0]:-1;
        }
        else commerror =TRUE;
    }

    if (!commerror)
    {	/* outputpowerdownstream  */
        unsigned char getdata[RECTRLE_BUFFSIZE];
        if (ixAdslUtilCtrleRead(lineNum,getdata,CTEAD_NT_FAR_OUTPUTPOWER_DN,1))
        {	p_OperData->outputPowerDnstr = inshowtime?(signed char)getdata[0]:-1;
        }
        else commerror =TRUE;
    }

    if (!commerror)
    {	/* attenuationupstr  */
        unsigned char getdata[RECTRLE_BUFFSIZE];
        if (ixAdslUtilCtrleRead(lineNum,getdata,CTEAD_NT_FAR_ATTENUATION_UP,1))
        {	p_OperData->attenuationUpstr = inshowtime?(unsigned char)getdata[0]:-1;
        }
        else commerror =TRUE;
    }

    if (!commerror)
    {	/* carrierLoad	*/
        unsigned char getdata[RECTRLE_BUFFSIZE];
        if (ixAdslUtilCtrleRead(lineNum,getdata,CTEAD_NT_FAR_CARRIER_LOAD,64))
        {	for ( i=0;i<=63;i++)
        p_OperData->carrierLoad[i] = inshowtime?(unsigned char)getdata[i]:-1;
        }
        else commerror =TRUE;
    }
    if (!commerror)
    {	/* carrierLoad	*/
        unsigned char getdata[RECTRLE_BUFFSIZE];
        if (ixAdslUtilCtrleRead(lineNum,getdata,CTEAD_NT_FAR_CARRIER_LOAD+64,64))
        {	for ( i=0;i<=63;i++)
        p_OperData->carrierLoad[i+64] =	inshowtime?(unsigned char)getdata[i]:-1;
        }
        else commerror =TRUE;
    }

    /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_REJ if	comms ok but modem   */
    /* not in showtime, C_CONF_DATA_COMMPROBLEM if comms not ok  */
    /*unsigned int returnvalue*/
    if (commerror) returnvalue = C_CONF_DATA_COMMPROBLEM;
    else
    {	if (!inshowtime) returnvalue = C_CONF_DATA_REJ;
        else returnvalue = C_CONF_DATA_ACK;
    }
    return returnvalue;
}

#if CPU == XSCALE
unsigned int ADSL_get_ChannelOperData(unsigned int lineNum, T_adslChannelOperData* p_OperData)
{	bool commerror=FALSE; unsigned int returnvalue;

    /* check whether modem in showtime  */
    unsigned char status[RECTRLE_BUFFSIZE];
    bool inshowtime=FALSE;
    if (ixAdslUtilCtrleRead(lineNum,status,CTEAD_MODEM_STATUS,1))
    {	inshowtime = (status[0]==MODEM_STATUS_UP);
    }
    else commerror=TRUE;

    /* read	channel	data  */
    if (!commerror)
    {	unsigned char getdata[RECTRLE_BUFFSIZE];
        if (ixAdslUtilCtrleRead(lineNum,getdata,CTEAD_NEAR_INT_CHANNELDATA_LSB,8))
        {	unsigned short lsb,msb;
            lsb = (unsigned	short)((unsigned char)getdata[0]);
            msb = (unsigned	short)((unsigned char)getdata[1]);
            p_OperData->actualBitrateNearInt = msb<<8 | lsb;
            lsb = (unsigned	short)((unsigned char)getdata[2]);
            msb = (unsigned	short)((unsigned char)getdata[3]);
            p_OperData->actualBitrateNearFast = msb<<8 | lsb;
            lsb = (unsigned	short)((unsigned char)getdata[4]);
            msb = (unsigned	short)((unsigned char)getdata[5]);
            p_OperData->actualBitrateFarInt	= inshowtime? msb<<8 | lsb:-1;
            lsb = (unsigned	short)((unsigned char)getdata[6]);
            msb = (unsigned	short)((unsigned char)getdata[7]);
            p_OperData->actualBitrateFarFast = inshowtime? msb<<8 |	lsb:-1;
        }
        else commerror =TRUE;
    }

    /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_REJ if	comms ok but modem   */
    /* not in showtime, C_CONF_DATA_COMMPROBLEM if comms not ok  */
    /*unsigned int returnvalue*/
    if (commerror) returnvalue = C_CONF_DATA_COMMPROBLEM;
    else
    {	if (!inshowtime) returnvalue = C_CONF_DATA_REJ;
        else returnvalue = C_CONF_DATA_ACK;
    }
    return returnvalue;
}
#endif

#if CPU == XSCALE
unsigned int ADSL_def_counter_poll(unsigned int lineNum, T_adsl_def_counter_set*p_counter_set)
{	bool commerror=FALSE; unsigned int returnvalue;

    /* check whether modem in showtime  */
    unsigned char status[RECTRLE_BUFFSIZE];
    bool inshowtime=FALSE;
    if (ixAdslUtilCtrleRead(lineNum,status,CTEAD_MODEM_STATUS,1))
    {	inshowtime = (status[0]==MODEM_STATUS_UP);
    }
    else commerror=TRUE;

    /* read	performance data (counters)  */
    if (!commerror)
    {	unsigned char getdata[RECTRLE_BUFFSIZE];
        if (ixAdslUtilCtrleRead(lineNum,getdata,CTEAD_NEAR_FEC_NI_LSB,24))
        {	unsigned short lsb,msb;
            lsb = (unsigned	short)((unsigned char)getdata[0]);
            msb = (unsigned	short)((unsigned char)getdata[1]);
            p_counter_set->near_end.FecNotInterleaved = msb<<8 | lsb;
            lsb = (unsigned	short)((unsigned char)getdata[2]);
            msb = (unsigned	short)((unsigned char)getdata[3]);
            p_counter_set->near_end.FecInterleaved = msb<<8	| lsb;
            lsb = (unsigned	short)((unsigned char)getdata[4]);
            msb = (unsigned	short)((unsigned char)getdata[5]);
            p_counter_set->near_end.CrcNotInterleaved = msb<<8 | lsb;
            lsb = (unsigned	short)((unsigned char)getdata[6]);
            msb = (unsigned	short)((unsigned char)getdata[7]);
            p_counter_set->near_end.CrcInterleaved = msb<<8	| lsb;
            lsb = (unsigned	short)((unsigned char)getdata[8]);
            msb = (unsigned	short)((unsigned char)getdata[9]);
            p_counter_set->near_end.HecNotInterleaved = msb<<8 | lsb;
            lsb = (unsigned	short)((unsigned char)getdata[10]);
            msb = (unsigned	short)((unsigned char)getdata[11]);
            p_counter_set->near_end.HecInterleaved = msb<<8	| lsb;

            lsb = (unsigned	short)((unsigned char)getdata[12]);
            msb = (unsigned	short)((unsigned char)getdata[13]);
            p_counter_set->far_end.FecNotInterleaved = inshowtime? msb<<8 |	lsb:-1;
            lsb = (unsigned	short)((unsigned char)getdata[14]);
            msb = (unsigned	short)((unsigned char)getdata[15]);
            p_counter_set->far_end.FecInterleaved =	inshowtime?msb<<8 | lsb: -1;
            lsb = (unsigned	short)((unsigned char)getdata[16]);
            msb = (unsigned	short)((unsigned char)getdata[17]);
            p_counter_set->far_end.CrcNotInterleaved = inshowtime?msb<<8 | lsb: -1;
            lsb = (unsigned	short)((unsigned char)getdata[18]);
            msb = (unsigned	short)((unsigned char)getdata[19]);
            p_counter_set->far_end.CrcInterleaved =	inshowtime?msb<<8 | lsb:-1;
            lsb = (unsigned	short)((unsigned char)getdata[20]);
            msb = (unsigned	short)((unsigned char)getdata[21]);
            p_counter_set->far_end.HecNotInterleaved = inshowtime?msb<<8 | lsb:-1;
            lsb = (unsigned	short)((unsigned char)getdata[22]);
            msb = (unsigned	short)((unsigned char)getdata[23]);
            p_counter_set->far_end.HecInterleaved =	inshowtime?msb<<8 | lsb: -1;
        }
        else commerror =TRUE;
    }

    /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_REJ if	comms ok but modem   */
    /* not in showtime, C_CONF_DATA_COMMPROBLEM if comms not ok  */
    /*unsigned int returnvalue*/
    if (commerror) returnvalue = C_CONF_DATA_COMMPROBLEM;
    else
    {	if (!inshowtime) returnvalue = C_CONF_DATA_REJ;
        else returnvalue = C_CONF_DATA_ACK;
    }
    return returnvalue;
}
#endif

#if CPU == XSCALE
unsigned int ADSL_def_bitmap_poll(unsigned int lineNum,T_adsl_def_bitmap_set*p_bitmap_set)
{	bool commerror=FALSE; unsigned int returnvalue;

    /* check whether modem in showtime  */
    unsigned char status[RECTRLE_BUFFSIZE];
    bool inshowtime=FALSE;
    if (ixAdslUtilCtrleRead(lineNum,status,CTEAD_MODEM_STATUS,1))
    {	inshowtime = (status[0]==MODEM_STATUS_UP);
    }
    else commerror=TRUE;

    /* read	bitmaps	 */
    if (!commerror)
    {	unsigned char getdata[RECTRLE_BUFFSIZE];
        if (ixAdslUtilCtrleRead(lineNum,getdata,CTEAD_NEAR_BITMAP_STATUS,4))
        {	p_bitmap_set->near_end.status =	(unsigned char)getdata[0];
            p_bitmap_set->far_end.status = inshowtime? (unsigned char)getdata[1]: -1;
            p_bitmap_set->near_end.change =	(unsigned char)getdata[2];
            p_bitmap_set->far_end.change = inshowtime? (unsigned char)getdata[3]: -1;
        }
        else commerror =TRUE;
    }

    /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_REJ if	comms ok but modem   */
    /* not in showtime, C_CONF_DATA_COMMPROBLEM if comms not ok  */
    /*unsigned int returnvalue*/
    if (commerror) returnvalue = C_CONF_DATA_COMMPROBLEM;
    else
    {	if (!inshowtime) returnvalue = C_CONF_DATA_REJ;
        else returnvalue = C_CONF_DATA_ACK;
    }
    return returnvalue;
}
#endif

unsigned int ADSL_get_BER_data(unsigned int lineNum, T_BERdata_set*p_ber_set)
{	bool commerror=FALSE; unsigned int returnvalue;

    /* read	ber fast  */
    if (!commerror)
    {	unsigned char getdata[RECTRLE_BUFFSIZE];
        if (ixAdslUtilCtrleRead(lineNum,getdata,CTEAD_ACC_BER_FAST,10))
        {	unsigned int lsb,l2b,m3b,msb;
            lsb = (unsigned	short)((unsigned char)getdata[0]);
            msb = (unsigned	short)((unsigned char)getdata[1]);
            p_ber_set->berFast.AccumulatedBer = msb<<8 | lsb;
            lsb = (unsigned	int)((unsigned char)getdata[2]);
            l2b = (unsigned	int)((unsigned char)getdata[3]);
            m3b = (unsigned	int)((unsigned char)getdata[4]);
            msb = (unsigned	int)((unsigned char)getdata[5]);
            p_ber_set->berFast.NsecValidBer	= msb<<24 | m3b<<16 | l2b<<8 | lsb;
            lsb = (unsigned	int)((unsigned char)getdata[6]);
            l2b = (unsigned	int)((unsigned char)getdata[7]);
            m3b = (unsigned	int)((unsigned char)getdata[8]);
            msb = (unsigned	int)((unsigned char)getdata[9]);
            p_ber_set->berFast.NsecInvalidBer = msb<<24 | m3b<<16 |	l2b<<8 | lsb;
        }
        else commerror =TRUE;
    }

    /* read	ber interleaved	 */
    if (!commerror)
    {	unsigned char getdata[RECTRLE_BUFFSIZE];
        if (ixAdslUtilCtrleRead(lineNum,getdata,CTEAD_ACC_BER_IL,10))
        {	unsigned int lsb,l2b,m3b,msb;
            lsb = (unsigned	short)((unsigned char)getdata[0]);
            msb = (unsigned	short)((unsigned char)getdata[1]);
            p_ber_set->berInterleaved.AccumulatedBer = msb<<8 | lsb;
            lsb = (unsigned	int)((unsigned char)getdata[2]);
            l2b = (unsigned	int)((unsigned char)getdata[3]);
            m3b = (unsigned	int)((unsigned char)getdata[4]);
            msb = (unsigned	int)((unsigned char)getdata[5]);
            p_ber_set->berInterleaved.NsecValidBer = msb<<24 | m3b<<16 | l2b<<8 | lsb;
            lsb = (unsigned	int)((unsigned char)getdata[6]);
            l2b = (unsigned	int)((unsigned char)getdata[7]);
            m3b = (unsigned	int)((unsigned char)getdata[8]);
            msb = (unsigned	int)((unsigned char)getdata[9]);
            p_ber_set->berInterleaved.NsecInvalidBer = msb<<24 | m3b<<16 | l2b<<8 |	lsb;
        }
        else commerror =TRUE;
    }

    /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_REJ if	comms ok but modem   */
    /* not in showtime, C_CONF_DATA_COMMPROBLEM if comms not ok  */
    /*unsigned int returnvalue*/
    if (commerror) returnvalue = C_CONF_DATA_COMMPROBLEM;
    else returnvalue = C_CONF_DATA_ACK;

    return returnvalue;
}


unsigned int LT_ADSL_enableModemLine(unsigned int lineNum, T_adsl_ConfigurationData* configData)
{	bool commerror=FALSE; unsigned int returnvalue;

    /* check whether modemstatus is	down or	up  */
    unsigned char status[RECTRLE_BUFFSIZE];
    bool isdown=FALSE;

    if (ixAdslUtilCtrleRead(lineNum,status,CTEAD_MODEM_STATUS,1))
    {	isdown = (status[0]==MODEM_STATUS_DOWN || status[0]==MODEM_STATUS_UP);
    }
    else commerror=TRUE;

    /* write config	data if	modem is down or up  */
    if (isdown)
    {
        if ( LT_ADSL_downloadModemConfig(lineNum, configData) )
        {
                /*
                * Error Failed to open line.
                */
                DEBUG_CMD(DEBUG_LOG("ADSL: Configuration download failed\n"));
                commerror=TRUE;
        }

        /* the adsl open command  */
        if (!commerror)
        {	unsigned char putdata[1];
            putdata[0]=ADSL_OPEN;

            if (!ixAdslUtilCtrleWrite(lineNum,putdata,(unsigned int)TXCOMMANDREG,1))
                commerror =TRUE;
        }
    }

    /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_REJ if	comms ok but modem   */
    /* was not down	or up, C_CONF_DATA_COMMPROBLEM if comms	not ok	*/
    /*unsigned int returnvalue*/
    if (commerror) returnvalue = C_CONF_DATA_COMMPROBLEM;
    else
    {	if (!isdown) returnvalue = C_CONF_DATA_REJ;
        else returnvalue = C_CONF_DATA_ACK;
    }
    return returnvalue;
}

unsigned int LT_ADSL_enableModemLine_glite(unsigned int lineNum, T_adsl_ConfigurationData* configData)
{	bool commerror=FALSE;
    unsigned int returnvalue;

    /* check whether modemstatus is	down or	up  */
    unsigned char status[RECTRLE_BUFFSIZE];
    bool isdown=FALSE;
    if (ixAdslUtilCtrleRead(lineNum,status,CTEAD_MODEM_STATUS,1))
    {	isdown = (status[0]==MODEM_STATUS_DOWN || status[0]==MODEM_STATUS_UP);
    }
    else commerror=TRUE;

    /* write config	data if	modem is down or up  */
    if (isdown)
    {
        if ( LT_ADSL_downloadModemConfig(lineNum, configData) )
        {
                /*
                * Error Failed to open line.
                */
                DEBUG_CMD(DEBUG_LOG("ADSL: Configuration download failed\n"));
                commerror=TRUE;
        }
        /* the adsl open command  */
        if (!commerror)
        {	unsigned char putdata[1];
            putdata[0]=ADSL_OPEN_GLITE;
            if (!ixAdslUtilCtrleWrite(lineNum,putdata,(unsigned int)TXCOMMANDREG,1))
                commerror =TRUE;
        }
    }

    /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_REJ if	comms ok but modem   */
    /* was not down	or up, C_CONF_DATA_COMMPROBLEM if comms	not ok	*/
    /*unsigned int returnvalue*/
    if (commerror) returnvalue = C_CONF_DATA_COMMPROBLEM;
    else
    {	if (!isdown) returnvalue = C_CONF_DATA_REJ;
        else returnvalue = C_CONF_DATA_ACK;
    }
    return returnvalue;
}

unsigned int LT_ADSL_enableModemLine_gdmt(unsigned int lineNum, T_adsl_ConfigurationData* configData)
{	bool commerror=FALSE;
    unsigned int returnvalue;

    /* check whether modemstatus is	down or	up  */
    unsigned char status[RECTRLE_BUFFSIZE];
    bool isdown=FALSE;
    if (ixAdslUtilCtrleRead(lineNum,status,CTEAD_MODEM_STATUS,1))
    {	isdown = (status[0]==MODEM_STATUS_DOWN || status[0]==MODEM_STATUS_UP);
    }
    else commerror=TRUE;

    /* write config	data if	modem is down or up  */
    if (isdown)
    {
        if ( LT_ADSL_downloadModemConfig(lineNum, configData) )
        {
                /*
                * Error Failed to open line.
                */
                DEBUG_CMD(DEBUG_LOG("ADSL: Configuration download failed\n"));
                commerror=TRUE;
        }

        /* the adsl open command  */
        if (!commerror)
        {	unsigned char putdata[1];
            putdata[0]=ADSL_OPEN_GDMT;
            if (!ixAdslUtilCtrleWrite(lineNum,putdata,(unsigned int)TXCOMMANDREG,1))
                commerror =TRUE;
        }
    }

    /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_REJ if	comms ok but modem   */
    /* was not down	or up, C_CONF_DATA_COMMPROBLEM if comms	not ok	*/
    /*unsigned int returnvalue*/
    if (commerror) returnvalue = C_CONF_DATA_COMMPROBLEM;
    else
    {	if (!isdown) returnvalue = C_CONF_DATA_REJ;
        else returnvalue = C_CONF_DATA_ACK;
    }
    return returnvalue;
}

unsigned int LT_ADSL_enableModemLine_ansi(unsigned int lineNum, T_adsl_ConfigurationData* configData)
{
    bool commerror=FALSE;
    unsigned int returnvalue;

    /* check whether modemstatus is	down or	up  */
    unsigned char status[RECTRLE_BUFFSIZE];
    bool isdown=FALSE;

    if (ixAdslUtilCtrleRead(lineNum,status,CTEAD_MODEM_STATUS,1))
    {	isdown = (status[0]==MODEM_STATUS_DOWN || status[0]==MODEM_STATUS_UP);
    }
    else commerror=TRUE;

    /* write config	data if	modem is down or up  */
    if (isdown)
    {
        if ( LT_ADSL_downloadModemConfig(lineNum, configData) )
        {
                /*
                * Error Failed to open line.
                */
                DEBUG_CMD(DEBUG_LOG("ADSL: Configuration download failed\n"));
                commerror=TRUE;
        }

        /* the adsl open command  */
        if (!commerror)
        {	unsigned char putdata[1];
            putdata[0]=ADSL_OPEN_ANSI;
            if (!ixAdslUtilCtrleWrite(lineNum,putdata,(unsigned int)TXCOMMANDREG,1))
                commerror =TRUE;
        }
    }

    /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_REJ if	comms ok but modem   */
    /* was not down	or up, C_CONF_DATA_COMMPROBLEM if comms	not ok	*/
    /*unsigned int returnvalue*/
    if (commerror) returnvalue = C_CONF_DATA_COMMPROBLEM;
    else
    {	if (!isdown) returnvalue = C_CONF_DATA_REJ;
        else returnvalue = C_CONF_DATA_ACK;
    }
    return returnvalue;
}

unsigned int LT_ADSL_enableModemLine_multimode(unsigned int lineNum, T_adsl_ConfigurationData* configData)
{	bool commerror=FALSE;
    unsigned int returnvalue;

    /* check whether modemstatus is	down or	up  */
    unsigned char status[RECTRLE_BUFFSIZE];
    bool isdown=FALSE;
    if (ixAdslUtilCtrleRead(lineNum,status,CTEAD_MODEM_STATUS,1))
    {	isdown = (status[0]==MODEM_STATUS_DOWN || status[0]==MODEM_STATUS_UP);
    }
    else commerror=TRUE;

    /* write config	data if	modem is down or up  */
    if (isdown)
    {
        if ( LT_ADSL_downloadModemConfig(lineNum, configData) )
        {
                /*
                * Error Failed to open line.
                */
                DEBUG_CMD(DEBUG_LOG("ADSL: Configuration download failed\n"));
                commerror=TRUE;
        }

        /* the adsl open command  */
        if (!commerror)
        {	unsigned char putdata[1];
            putdata[0]=ADSL_OPEN_MULTIMODE;
            if (!ixAdslUtilCtrleWrite(lineNum,putdata,(unsigned int)TXCOMMANDREG,1))
                commerror =TRUE;
        }
    }

    /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_REJ if	comms ok but modem   */
    /* was not down	or up, C_CONF_DATA_COMMPROBLEM if comms	not ok	*/
    /*unsigned int returnvalue*/
    if (commerror) returnvalue = C_CONF_DATA_COMMPROBLEM;
    else
    {	if (!isdown) returnvalue = C_CONF_DATA_REJ;
        else returnvalue = C_CONF_DATA_ACK;
    }
    return returnvalue;
}

unsigned int LT_ADSL_enableModemLine_loopback(unsigned int lineNum, T_adsl_ConfigurationData* configData)
{
    bool commerror=FALSE;
    unsigned int returnvalue;

    /* check whether modemstatus is	down or	up  */
    unsigned char status[RECTRLE_BUFFSIZE];
    bool isdown=FALSE;

    if (ixAdslUtilCtrleRead(lineNum,status,CTEAD_MODEM_STATUS,1))
    {
    isdown = (status[0]==MODEM_STATUS_DOWN || status[0]==MODEM_STATUS_UP);
    }
    else commerror=TRUE;

    /* write config	data if	modem is down or up  */
    if (isdown)
    {
        if (!commerror)	/* utopia parameters  */
        {	unsigned char putdata[3];
            putdata[0]=(char)configData->UtopiaLevel;
            putdata[1]=(char)configData->UtopiaAddrFast;
            putdata[2]=(char)configData->UtopiaAddrInterl;
            if (!ixAdslUtilCtrleWrite(lineNum,putdata,CTEAD_UTOPIA_LEVEL,3))
                commerror =TRUE;
        }
        if (!commerror)	/* functionality code  */
        {	unsigned char putdata[1];
            putdata[0]=(char)configData->FunctionCode1FD;
            if (!ixAdslUtilCtrleWrite(lineNum,putdata,CTEAD_FUNC_FIELD_1FD,1))
                commerror =TRUE;
        }



    /* the adsl open_loopback command  */
    if (!commerror)
    {
        unsigned char putdata[1];
        putdata[0]=ADSL_OPEN_LOOPBACK;
        if (!ixAdslUtilCtrleWrite(lineNum,putdata,(unsigned int)TXCOMMANDREG,1))
        {
        commerror = TRUE;
        }
    }
    }

    /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_REJ if
    * comms ok but modem was not down	or up, C_CONF_DATA_COMMPROBLEM
    * if comms	not ok	*/
    if (commerror)
    {
    returnvalue = C_CONF_DATA_COMMPROBLEM;
    }
    else
    {
    if (!isdown) returnvalue = C_CONF_DATA_REJ;
    else returnvalue = C_CONF_DATA_ACK;
    }
    return returnvalue;
}


unsigned int ADSL_disableModemLine(unsigned int lineNum)
{	bool commerror=FALSE; unsigned int returnvalue;

    /* check modem status (down,initializing,showtime)  */
    unsigned char status[RECTRLE_BUFFSIZE];
    bool isdown=FALSE,inshowtime=FALSE,waitforinit=FALSE,init=FALSE;
    if (ixAdslUtilCtrleRead(lineNum,status,CTEAD_MODEM_STATUS,1))
    {	isdown = (status[0]==MODEM_STATUS_DOWN);
        inshowtime = (status[0]==MODEM_STATUS_UP);
        waitforinit = (status[0]==MODEM_STATUS_WAIT_FOR_INIT);
        init = (status[0]==MODEM_STATUS_INITIALIZING);
    }
    else commerror=TRUE;

    if (inshowtime || waitforinit || init)
    {	/* the adsl close command  */
        if (!commerror)
        {	unsigned char putdata[1];
            putdata[0]=ADSL_CLOSE;
            if (!ixAdslUtilCtrleWrite(lineNum,putdata,(unsigned int)TXCOMMANDREG,1))
                commerror =TRUE;
        }
    }

    /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_REJ if	comms ok but modem   */
    /* was in an unknown state, C_CONF_DATA_COMMPROBLEM if comms not ok,  */
    /* C_CONF_DATA_ALREADY_DOWN when modem was down	 */
    /*unsigned int returnvalue*/
    if (commerror) returnvalue = C_CONF_DATA_COMMPROBLEM;
    else if	(inshowtime||waitforinit||init)	returnvalue = C_CONF_DATA_ACK;
        else if (isdown) returnvalue =	C_CONF_DATA_ALREADY_DOWN;
            else returnvalue = C_CONF_DATA_REJ;/* (other state than	inshowtime||waitforinit||init||down)  */
    return returnvalue;
}

unsigned int ADSL_disableModemLine_orderly_shutdown(unsigned int lineNum)
{	bool commerror=FALSE; unsigned int returnvalue;

    /* check modem status (down,initializing,showtime)  */
    unsigned char status[RECTRLE_BUFFSIZE];
    bool isdown=FALSE,inshowtime=FALSE,waitforinit=FALSE,init=FALSE;
    if (ixAdslUtilCtrleRead(lineNum,status,CTEAD_MODEM_STATUS,1))
    {	isdown = (status[0]==MODEM_STATUS_DOWN);
        inshowtime = (status[0]==MODEM_STATUS_UP);
        waitforinit = (status[0]==MODEM_STATUS_WAIT_FOR_INIT);
        init = (status[0]==MODEM_STATUS_INITIALIZING);
    }
    else commerror=TRUE;

    if (inshowtime || waitforinit/*|| init*/)
    {	/* the adsl close command  */
        if (!commerror)
        {	unsigned char putdata[1];
            putdata[0]=ADSL_CLOSE_ORDERLY_SHUTDOWN;
            if (!ixAdslUtilCtrleWrite(lineNum,putdata,(unsigned int)TXCOMMANDREG,1))
                commerror =TRUE;
        }
    }

    /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_REJ if	comms ok but modem   */
    /* was in an unknown state, C_CONF_DATA_COMMPROBLEM if comms not ok,  */
    /* C_CONF_DATA_ALREADY_DOWN when modem was down	 */
    /*unsigned int returnvalue*/
    if (commerror) returnvalue = C_CONF_DATA_COMMPROBLEM;
    else if	(inshowtime||waitforinit/*||init*/) returnvalue	= C_CONF_DATA_ACK;
        else if (isdown) returnvalue =	C_CONF_DATA_ALREADY_DOWN;
            else returnvalue = C_CONF_DATA_REJ;/* (other state than	inshowtime||waitforinit||init||down)  */
    return returnvalue;
}

unsigned int NT_ADSL_dyingGasp(unsigned int lineNum)
{	bool commerror=FALSE; unsigned int returnvalue;

    /* the adsl dying gasp command	*/
    unsigned char putdata[1];
    putdata[0]=ADSL_SEND_DYING_GASP;
    if (!ixAdslUtilCtrleWrite(lineNum,putdata,(unsigned int)TXCOMMANDREG,1))
        commerror =TRUE;

    /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_COMMPROBLEM if	comms not ok  */
    /*unsigned int returnvalue*/
    if (commerror) returnvalue = C_CONF_DATA_COMMPROBLEM;
    else returnvalue = C_CONF_DATA_ACK;
    return returnvalue;
}

unsigned int ADSL_write_functionality_code_1FD(unsigned int lineNum, char ch)
{bool commerror=FALSE; unsigned	int returnvalue;

    unsigned char putdata[1];
    putdata[0]=ch;
    if (!ixAdslUtilCtrleWrite(lineNum,putdata,CTEAD_FUNC_FIELD_1FD,1))
        commerror =TRUE;

    /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_COMMPROBLEM if	comms not ok  */
    /*unsigned int returnvalue*/
    if (commerror) returnvalue = C_CONF_DATA_COMMPROBLEM;
    else returnvalue = C_CONF_DATA_ACK;
    return returnvalue;
}

unsigned int ADSL_SelfTest_Long(unsigned int lineNum)
{	bool commerror=FALSE; unsigned int returnvalue;

    /* check whether modemstatus is	down  */
    unsigned char status[RECTRLE_BUFFSIZE];
    bool isdown=FALSE;
    if (ixAdslUtilCtrleRead(lineNum,status,CTEAD_MODEM_STATUS,1))
    {	isdown = (status[0]==MODEM_STATUS_DOWN);
    }
    else commerror=TRUE;

    if (isdown)
    {	/* the selftest	long command  */
        if (!commerror)
        {	unsigned char putdata[1];
            putdata[0]=ADSL_SELFTEST_LONG;
            if (!ixAdslUtilCtrleWrite(lineNum,putdata,(unsigned int)TXCOMMANDREG,1))
                commerror =TRUE;
        }
    }

    /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_REJ if	comms ok but modem   */
    /* was not down, C_CONF_DATA_COMMPROBLEM if comms not ok  */
    /*unsigned int returnvalue*/
    if (commerror) returnvalue = C_CONF_DATA_COMMPROBLEM;
    else
    {	if (!isdown) returnvalue = C_CONF_DATA_REJ;
        else returnvalue = C_CONF_DATA_ACK;
    }
    return returnvalue;
}

unsigned int ADSL_SelfTest_Short(unsigned int lineNum)
{	bool commerror=FALSE; unsigned int returnvalue;

    /* check whether modemstatus is	down  */
    unsigned char status[RECTRLE_BUFFSIZE];
    bool isdown=FALSE;
    if (ixAdslUtilCtrleRead(lineNum,status,CTEAD_MODEM_STATUS,1))
    {	isdown = (status[0]==MODEM_STATUS_DOWN);
    }
    else commerror=TRUE;

    if (isdown)
    {	/* the selftest	long command  */
        if (!commerror)
        {	unsigned char putdata[1];
            putdata[0]=ADSL_SELFTEST_SHORT;
            if (!ixAdslUtilCtrleWrite(lineNum,putdata,(unsigned int)TXCOMMANDREG,1))
                commerror =TRUE;
        }
    }

    /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_REJ if	comms ok but modem   */
    /* was not down, C_CONF_DATA_COMMPROBLEM if comms not ok  */
    /*unsigned int returnvalue*/
    if (commerror) returnvalue = C_CONF_DATA_COMMPROBLEM;
    else
    {	if (!isdown) returnvalue = C_CONF_DATA_REJ;
        else returnvalue = C_CONF_DATA_ACK;
    }
    return returnvalue;
}


unsigned int ADSL_SelfTest_SendTones(unsigned int lineNum)
{	bool commerror=FALSE; unsigned int returnvalue;

    /* check whether modemstatus is	down  */
    unsigned char status[RECTRLE_BUFFSIZE];
    bool isdown=FALSE;
    if (ixAdslUtilCtrleRead(lineNum,status,CTEAD_MODEM_STATUS,1))
    {	isdown = (status[0]==MODEM_STATUS_DOWN);
    }
    else commerror=TRUE;

    if (isdown)
    {	/* the selftest	long command  */
        if (!commerror)
        {	unsigned char putdata[1];
            putdata[0]=ADSL_SELFTEST_SEND_TONES;
            if (!ixAdslUtilCtrleWrite(lineNum,putdata,(unsigned int)TXCOMMANDREG,1))
                commerror =TRUE;
        }
    }

    /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_REJ if	comms ok but modem   */
    /* was not down, C_CONF_DATA_COMMPROBLEM if comms not ok  */
    /*unsigned int returnvalue*/
    if (commerror) returnvalue = C_CONF_DATA_COMMPROBLEM;
    else
    {	if (!isdown) returnvalue = C_CONF_DATA_REJ;
        else returnvalue = C_CONF_DATA_ACK;
    }
    return returnvalue;
}


unsigned char
ADSL_setFunctionCode(unsigned int lineNum, unsigned char code)
{
unsigned char rc;
            unsigned char putdata[1];
            putdata[0]=(char)code;
            rc = ixAdslUtilCtrleWrite(lineNum,putdata,CTEAD_FUNC_FIELD_1FD,1);

    return rc;
}

unsigned int ADSL_getModemState(unsigned int lineNum, unsigned char *retStatus)
{
unsigned char status[2];

if (ixAdslUtilCtrleRead(lineNum,status,CTEAD_MODEM_STATUS,1))
{
    *retStatus = status[0];
    return(FALSE);
}


return(C_CONF_DATA_COMMPROBLEM);

}

unsigned int ADSL_getRxCommand( unsigned int lineNum, char unsigned *rxCommand )
{
unsigned char status[2];

if (ixAdslUtilCtrleRead(lineNum,status,RXCOMMANDREG,1))
{
    *rxCommand = status[0];
    return(FALSE);
}

/* Check for Async responces to connands and discard */
return(1);


}

unsigned int ADSL_CloseLine(unsigned int lineNum)
{

    bool commerror=FALSE; unsigned int returnvalue;
        unsigned char putdata[1];
        putdata[0]=ADSL_CLOSE;

        if (!ixAdslUtilCtrleWrite(lineNum,putdata,(unsigned int)TXCOMMANDREG,1))
            commerror =TRUE;

        /* return C_CONF_DATA_ACK if all ok, C_CONF_DATA_REJ if comms ok but modem   */
        /* was in an unknown state, C_CONF_DATA_COMMPROBLEM if comms not ok,  */
        /* C_CONF_DATA_ALREADY_DOWN when modem was down  */
        /*unsigned int returnvalue*/
        if (commerror) returnvalue = C_CONF_DATA_COMMPROBLEM;
        else returnvalue = C_CONF_DATA_ACK;/* (other state than inshowtime||waitforinit||init||down)  */

        return returnvalue;

}

unsigned int LT_ADSL_downloadModemConfig(unsigned int lineNum, T_adsl_ConfigurationData* configData)
{	bool commerror=FALSE;
    unsigned int returnvalue;
    int i;

    /* check whether modemstatus is	down or	up  */
    unsigned char status[RECTRLE_BUFFSIZE];
    bool isdown=FALSE;
    if (ixAdslUtilCtrleRead(lineNum,status,CTEAD_MODEM_STATUS,1))
    {	isdown = (status[0]==MODEM_STATUS_DOWN || status[0]==MODEM_STATUS_UP);
    }
    else commerror=TRUE;

    /* write config	data if	modem is down or up  */
    if (isdown)
    {	if (!commerror)	/* noisemargin	*/
        {	unsigned char putdata[6];
            putdata[0]=(char)configData->MinNoiseMarginDnstr;
            putdata[1]=(char)configData->MinNoiseMarginUpstr;
            putdata[2]=(char)configData->MaxAddNoiseMarginDnstr;
            putdata[3]=(char)configData->MaxAddNoiseMarginUpstr;
            putdata[4]=(char)configData->TargetNoiseMarginDnstr;
            putdata[5]=(char)configData->TargetNoiseMarginUpstr;
            if (!ixAdslUtilCtrleWrite(lineNum,putdata,CTEAD_LT_MINNOISEMARGIN_DN,6))
                commerror =TRUE;
        }

        if (!commerror)	/* power spectral density  */
        {	unsigned char putdata[1];
            putdata[0]=(char)configData->MaxPSDDnstr;
            if (!ixAdslUtilCtrleWrite(lineNum,putdata,CTEAD_LT_MAXPSD_DN,1))
                commerror =TRUE;
        }

        if (!commerror)	/* aggregate power level  */
        {	unsigned char putdata[2];
            putdata[0]=(char)configData->MaxAggrPwrLvlDnstr;
            putdata[1]=(char)configData->MaxAggrPwrLvlUpstr;
            if (!ixAdslUtilCtrleWrite(lineNum,putdata,CTEAD_LT_MAXAGGPOWERLEVEL_DN,2))
                commerror =TRUE;
        }

        if (!commerror)	/* carrier mask	 */
        {	unsigned char putdata[32];
            for ( i=0;i<=31;i++)
                putdata[i]=(char)configData->CarrierMask[i];
            if (!ixAdslUtilCtrleWrite(lineNum,putdata,CTEAD_LT_CARRIERMASK,32))
                commerror =TRUE;
        }

        if (!commerror)	/* rate	adaptation mode	 */
        {	unsigned char putdata[2];
            putdata[0]=(char)configData->RAModeDnstr;
            putdata[1]=(char)configData->RAModeUpstr;
            if (!ixAdslUtilCtrleWrite(lineNum,putdata,CTEAD_LT_RAMODE_DN,2))
                commerror =TRUE;
        }

        if (!commerror)	/* bitrate  */
        {	unsigned char putdata[24];
            putdata[0]=(char)(configData->InterlPlannedBitrateDnstr	% 256);
            putdata[1]=(char)(configData->InterlPlannedBitrateDnstr	/ 256);
            putdata[2]=(char)(configData->InterlPlannedBitrateUpstr	% 256);
            putdata[3]=(char)(configData->InterlPlannedBitrateUpstr	/ 256);
            putdata[4]=(char)(configData->FastPlannedBitrateDnstr %	256);
            putdata[5]=(char)(configData->FastPlannedBitrateDnstr /	256);
            putdata[6]=(char)(configData->FastPlannedBitrateUpstr %	256);
            putdata[7]=(char)(configData->FastPlannedBitrateUpstr /	256);
            putdata[8]=(char)(configData->InterlMinBitrateDnstr % 256);
            putdata[9]=(char)(configData->InterlMinBitrateDnstr / 256);
            putdata[10]=(char)(configData->InterlMinBitrateUpstr % 256);
            putdata[11]=(char)(configData->InterlMinBitrateUpstr / 256);
            putdata[12]=(char)(configData->FastMinBitrateDnstr % 256);
            putdata[13]=(char)(configData->FastMinBitrateDnstr / 256);
            putdata[14]=(char)(configData->FastMinBitrateUpstr % 256);
            putdata[15]=(char)(configData->FastMinBitrateUpstr / 256);
            putdata[16]=(char)(configData->InterlMaxBitrateDnstr % 256);
            putdata[17]=(char)(configData->InterlMaxBitrateDnstr / 256);
            putdata[18]=(char)(configData->InterlMaxBitrateUpstr % 256);
            putdata[19]=(char)(configData->InterlMaxBitrateUpstr / 256);
            putdata[20]=(char)(configData->FastMaxBitrateDnstr % 256);
            putdata[21]=(char)(configData->FastMaxBitrateDnstr / 256);
            putdata[22]=(char)(configData->FastMaxBitrateUpstr % 256);
            putdata[23]=(char)(configData->FastMaxBitrateUpstr / 256);
            if (!ixAdslUtilCtrleWrite(lineNum,putdata,CTEAD_LT_INTERL_PLANNED_BITRATE_DN,24))
                commerror =TRUE;
        }

        if (!commerror)	/* rate	adaptation ratio  */
        {   
            unsigned char putdata[2];
            putdata[0]=(char)configData->RARatioDnstr;
            putdata[1]=(char)configData->RARatioUpstr;
            if (!ixAdslUtilCtrleWrite(lineNum,putdata,CTEAD_LT_RARATIO_DN,2))
                commerror =TRUE;
        }

        if (!commerror)	/* rate	adaptation shift margin	 */
        {
            unsigned char putdata[4];
            putdata[0]=(char)configData->RADownshiftMarginDnstr;
            putdata[1]=(char)configData->RADownshiftMarginUpstr;
            putdata[2]=(char)configData->RAUpshiftMarginDnstr;
            putdata[3]=(char)configData->RAUpshiftMarginUpstr;
            if (!ixAdslUtilCtrleWrite(lineNum,putdata,CTEAD_LT_RADOWNSHIFTMARGIN_DN,4))
                commerror =TRUE;
        }

        if (!commerror)	/* rate	adaptation shift interval  */
        {
            unsigned char putdata[8];
            putdata[0]=(char)configData->RADownshiftIntervalDnstr;
            putdata[1]=(char)configData->RADownshiftIntervalUpstr;
            putdata[2]=(char)-1;/* dummy value  */
            putdata[3]=(char)-1;/* dummy value  */
            putdata[4]=(char)(configData->RAUpshiftIntervalDnstr % 256);
            putdata[5]=(char)(configData->RAUpshiftIntervalDnstr / 256);
            putdata[6]=(char)(configData->RAUpshiftIntervalUpstr % 256);
            putdata[7]=(char)(configData->RAUpshiftIntervalUpstr / 256);
            if (!ixAdslUtilCtrleWrite(lineNum,putdata,CTEAD_LT_RADOWNSHIFTINTERVAL_DN,8))
                commerror =TRUE;
        }

        if (!commerror)	/* max interleaved channel delay  */
        {
            unsigned char putdata[2];
            putdata[0]=(char)configData->InterlMaxDelayDnstr;
            putdata[1]=(char)configData->InterlMaxDelayUpstr;
            if (!ixAdslUtilCtrleWrite(lineNum,putdata,CTEAD_LT_INTERL_MAXDELAY_DN,2))
                commerror =TRUE;
        }
        if (!commerror)	/* utopia parameters  */
        {
            unsigned char putdata[3];
            putdata[0]=(char)configData->UtopiaLevel;
            putdata[1]=(char)configData->UtopiaAddrFast;
            putdata[2]=(char)configData->UtopiaAddrInterl;
            if (!ixAdslUtilCtrleWrite(lineNum,putdata,CTEAD_UTOPIA_LEVEL,3))
                commerror =TRUE;
        }

        if (!commerror)	/* functionality code  */
        {
            unsigned char putdata[1];
            putdata[0]=(char)configData->FunctionCode1FD;
            if (!ixAdslUtilCtrleWrite(lineNum,putdata,CTEAD_FUNC_FIELD_1FD,1))
                commerror =TRUE;
        }

        if (!commerror)	/* vendorID  */
        {	unsigned char putdata[8];
            putdata[0]=(char)configData->itu_vendorid_countrycode;
            putdata[1]=(char)configData->itu_vendorid_reserved;
            putdata[2]=(char)configData->itu_vendorid_vendorcode_1;
            putdata[3]=(char)configData->itu_vendorid_vendorcode_2;
            putdata[4]=(char)configData->itu_vendorid_vendorcode_3;
            putdata[5]=(char)configData->itu_vendorid_vendorcode_4;
            putdata[6]=(char)configData->itu_vendorid_vendorspecific_1;
            putdata[7]=(char)configData->itu_vendorid_vendorspecific_2;
            if (!ixAdslUtilCtrleWrite(lineNum,putdata,CTEAD_ITU_VENDORID_COUNTRYCODE,8))
                commerror =TRUE;
        }

    }
    returnvalue=commerror;
    return returnvalue;
}
