#ifndef TIMERQUEUE_H
#define TIMERQUEUE_H

#include <sys/time.h>

class TimerQueue;
typedef void TimerQueueCallback(TimerQueue *timerqueue, void *cookie);

/**
 * Implements a queue of events which should be triggered
 * at a particular time.
 */
class TimerQueue {
public:
  TimerQueue();
  ~TimerQueue();

  /**
   * Add a callback to the timer queue.
   * 
   * 'callback' will be invoked after 'ms' milliseconds (from now).
   * 'cookie' will be passwed to 'callback'
   */
  void add(unsigned long ms, TimerQueueCallback *callback, void *cookie);

  /**
   * For all events which have expired, invoke the 'callback' function
   * and remove from the queue.
   */
  void check_events();

  /**
   * Get the timeout represented by the nearest event.
   *
   * This means tv = (nearest - now).
   * If the event time has already passed for the nearest event,
   * returns, 'tv' is set to 0.
   * 
   * Returns 0 if at least one entry is in the queue, or 1 if the
   * queue is empty (and tv is unchanged);
   */
  int get_timeout(struct timeval& tv);

private:
  struct TimerEvent {
    struct TimerQueue::TimerEvent *next;
    struct timeval tv;
    TimerQueueCallback *callback;
    void *cookie;
  };

  /* Singly linked list of events with the nearest first.
   * The first entry is a dummy entry
   */
  TimerEvent events;
};

#endif
