#ifndef SERVER_CONFIG_H
#define SERVER_CONFIG_H

#include "config.h"

#if defined(__cplusplus)
extern "C" {
#endif

/**
 * Returns the global config handle.
 * 
 * If server_load_config() has not been successfully called, returns 0.
 *
 * @return the global config handle.
 */
config_t *server_global_config();

/**
 * Loads the server config from the given file.
 * 
 * No semantic parsing is done.
 * Use server_get_global_config() to access the config handle.
 * Prints a message to stderr if a syntax error occurred.
 *
 * @param filename the config file to load
 * @return 0 if OK, or non-zero of the configuration syntax was incorrect
 */
int server_load_config(const char *filename);

/**
 * Frees the previously loaded config.
 */
void server_free_config();

/**
 * Helper function to get a value from the config.
 *
 * @param type section type
 * @param section section name
 * @param param parameter name
 * @param required non-zero if a fatal error should be thrown on a missing parameter
 * @return the value of the given parameter in the given section
 */
const char *config_get_string(const char *type, const char *section, const char *param, int required);

/**
 * Helper function to get an integer value from the config.
 *
 * Throws a fatal error and exits if either the parameter is missing
 * or if the value does not parse as an integer.
 * The standard 0x and 0 prefixes may be used for hex and octal.
 *
 * @param type section type
 * @param section section name
 * @param param parameter name
 * @return the integer value of the given parameter in the given section
 */
int config_get_int(const char *type, const char *section, const char *param);

/**
 * Helper function to get an integer value from the config.
 *
 * Throws a fatal error and exits if either the parameter exists but
 * does not parse as an integer.
 * The standard 0x and 0 prefixes may be used for hex and octal.
 * If the parameter is missing, returns the provided default value.
 *
 * @param type section type
 * @param section section name
 * @param param parameter name
 * @param default_value the value to return if the parameter is missing
 * @return the integer value of the given parameter in the given section
 */
int config_get_optional_int(const char *type, const char *section, const char *param, int default_value);

#if defined(__cplusplus)
}
#endif

#endif
