#include <stdio.h>
#include <stdlib.h>

#include "server_config.h"

/*#define DEBUG*/

static config_t *global_config;

/**
 * Gives this module a handle to the global config.
 *
 * This module only uses the "sasl" sections.
 */
config_t *server_global_config()
{
  return(global_config);
}

/**
 * Reads the config from the given filename.
 *
 * Returns 0 if OK, or -1 if error.
 */
int server_load_config(const char *filename)
{
  global_config = config_alloc();

  if (config_read(filename, global_config) != 0) {
    fprintf(stderr, "%s\n", global_config->error_string);

    server_free_config();
    return(-1);
  }

  if (global_config->line == 0) {
    fprintf(stderr, "Warning: File %s did not exist or is empty\n", filename);
  }
  return(0);
}

void server_free_config()
{
  config_free(global_config);
  global_config = 0;
}

const char *config_get_string(const char *type, const char *section, const char *param, int required)
{
  config_section_t *s = config_section_find(global_config->sections, type, section);
  if (s) {
    config_param_t *p = config_param_find(s->vars, param);
    if (p) {
#ifdef DEBUG
      printf("server_config: [%s %s] %s => %s\n", type, section, param, p->value);
#endif
      return(p->value);
    }
  }
  if (!required) {
#ifdef DEBUG
    printf("server_config: [%s %s] %s => <NOT FOUND>\n", type, section, param);
#endif
    return(0);
  }
  fprintf(stderr, "Required config parameter '%s' not found in section [%s %s] in file %s\n",
    param, type, section, global_config->filename);
  exit(10);
}

static int get_int_value(const char *type, const char *section, const char *param, int required, int default_value)
{
  const char *value = config_get_string(type, section, param, required);
  if (value != 0) {
    char *pt;
    int intvalue = strtol(value, &pt, 0);
    if (pt == value || *pt) {
      fprintf(stderr, "Parameter '%s' in section [%s %s]: Not a valid integer: %s\n",
        param, type, section, value);
      exit(10);
    }
    return(intvalue);
  }
  return(default_value);
}

int config_get_int(const char *type, const char *section, const char *param)
{
  return(get_int_value(type, section, param, 1, 0));
}

int config_get_optional_int(const char *type, const char *section, const char *param, int default_value)
{
  return(get_int_value(type, section, param, 0, default_value));
}
