#include <ctype.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <syslog.h>
#include <assert.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <unistd.h>
#include <glob.h>

#include "config.h"
#include "server_config.h"
#include "gets_alloc.h"
#include "timerqueue.h"

#define DEFAULT_INTERVAL 60
#define DEFAULT_MAXSIZE  8000

#ifdef EMBED
#define DEFAULT_CONFIG_DIR "/etc/config/"
#else
#define DEFAULT_CONFIG_DIR ""
#endif

static const char *version = "reportd v0.1";

/*#define DEBUG*/

static void setup_targets(TimerQueue *timerqueue);

class OutputEntry {
public:
    OutputEntry();

    /**
     * Read the parameters in this section and
     * configure this output entry accordingly.
     */
    int config(config_section_t *section);

    /**
     * Schedule this entry on the timer queue after the configured interval.
     */
    int schedule(TimerQueue *timerqueue);

    /**
     * Schedule this entry on the timer queue for the first time.
     */
    int schedule_first(TimerQueue *timerqueue);

    /**
     * Perform this action.
     */
    int action();
private:

    static void callback(TimerQueue *timerqueue, void *cookie);
    int sendOutput(FILE *fh, const char *prefix);

    enum OutputType {
        TYPE_NONE,
        TYPE_FILES,
        TYPE_CMD
    };
    const char *name;
    OutputType type;
    int interval;
    int priority;
    const char *what;
    int enabled;
    int maxlines;
    int initial;
};

int main(int argc, char *argv[])
{

    char *config_filename = DEFAULT_CONFIG_DIR "reportd.conf";

    /* REVISIT: Use getopt() */
    if (argc > 2 && strcmp(argv[1], "-f") == 0) {
        config_filename = argv[2];
        argc -= 2;
        argv += 2;
    }

    openlog("reportd", 0, LOG_DAEMON);

    syslog(LOG_NOTICE, "%s starting with %s", version, config_filename);

    if (server_load_config(config_filename) != 0) {
        return 1;
    }

    const char *target = config_get_string("reportd", "global", "target", 1);
    if (strcmp(target, "syslog") != 0) {
       fprintf(stderr, "Only supported target is syslog\n");
       exit(10);
    }

    TimerQueue *timerqueue = new TimerQueue();

    setup_targets(timerqueue);

    struct timeval timeout;

    while (timerqueue->get_timeout(timeout) == 0) {
#ifdef DEBUG
        printf("Waiting for timeout in %d seconds\n", (int)timeout.tv_sec);
#endif

        select(0, 0, 0, 0, &timeout);

#ifdef DEBUG
        printf("Checking events\n");
#endif
        timerqueue->check_events();
    }
#ifdef DEBUG
    printf("Exiting because nothing to do\n");
#endif

    delete timerqueue;

    server_free_config();

    return 0;
}

static void setup_targets(TimerQueue *timerqueue)
{
    config_section_t *section = config_section_find(server_global_config()->sections, "output", 0);
    while (section) {
        OutputEntry *entry = new OutputEntry();

        entry->config(section);

        // Run everything immediately
        entry->schedule_first(timerqueue);

        section = config_section_find(section->next, "output", 0);
    }
}

OutputEntry::OutputEntry()
{
    type = TYPE_NONE;
    what = 0;
    name = 0;
    priority = LOG_INFO;
    enabled = 0;

    /*global_maxsize = config_get_optional_int("reportd", "global", "maxsize", DEFAULT_MAXSIZE);*/
    interval = config_get_optional_int("reportd", "global", "interval", DEFAULT_INTERVAL);
    maxlines = config_get_optional_int("reportd", "global", "maxlines", 0);
    initial = config_get_optional_int("reportd", "global", "initial", 0);
}

int OutputEntry::config(config_section_t *section)
{
    config_param_t *p;

    enabled = 1;

    p = config_param_find(section->vars, "type");
    if (!p) {
        fprintf(stderr, "Output entry %s has no type\n", section->name);
        exit(10);
    }
    name = section->name;

    if (strcmp(p->value, "files") == 0) {
        type = TYPE_FILES;
        p = config_param_find(section->vars, "files");
        if (!p) {
            fprintf(stderr, "Output entry %s has no files entry\n", section->name);
            exit(10);
        }
        what = p->value;
    }
    else if (strcmp(p->value, "cmd") == 0) {
        type = TYPE_CMD;
        p = config_param_find(section->vars, "cmd");
        if (!p) {
            fprintf(stderr, "Output entry %s has no cmd entry\n", section->name);
            exit(10);
        }
        what = p->value;
    }
    else {
        fprintf(stderr, "Output entry %s has invalid type: %s\n", section->name, p->value);
        exit(10);
    }

    p = config_param_find(section->vars, "interval");
    if (p) {
        interval = atoi(p->value);
    }

    p = config_param_find(section->vars, "priority");
    if (p) {
        priority = atoi(p->value);
    }

    p = config_param_find(section->vars, "disabled");
    if (p) {
        enabled = !atoi(p->value);
    }

    p = config_param_find(section->vars, "maxlines");
    if (p) {
        maxlines = atoi(p->value);
    }

    return(0);
}

int OutputEntry::schedule(TimerQueue *timerqueue)
{
    if (enabled) {
        timerqueue->add(interval * 1000, OutputEntry::callback, this);
    }
    return(0);
}

int OutputEntry::schedule_first(TimerQueue *timerqueue)
{
    if (enabled) {
        timerqueue->add(initial * 1000, OutputEntry::callback, this);
    }
    return(0);
}

int OutputEntry::action()
{
#ifdef DEBUG
    printf("OutputEntry::action() name=%s what=%s type=%d\n", name, what, type);
#endif

    switch (type) {
        case TYPE_FILES:
            {
                glob_t globinfo;

                if (glob(what, 0, 0, &globinfo) != 0) {
                    fprintf(stderr, "Output action %s failed list files '%s'\n", name, what);
                    enabled = 0;
                    return(-1);
                }

                for (size_t i = 0; i < globinfo.gl_pathc; i++) {
                    FILE *fh = fopen(globinfo.gl_pathv[i], "r");
                    if (fh == 0) {
                        fprintf(stderr, "Output action %s failed open '%s' for reading\n", name, globinfo.gl_pathv[i]);
                    }
                    else {
                        sendOutput(fh, globinfo.gl_pathv[i]);

                        fclose(fh);
                    }
                }
                globfree(&globinfo);
                return(0);
            }

        case TYPE_CMD:
            {
                FILE *fh = popen(what, "r");

                if (!fh) {
                    fprintf(stderr, "Output action %s failed to run command '%s'\n", name, what);
                    return(-1);
                }
                sendOutput(fh, what);

                fclose(fh);

                /* Reap children */
                int status;
                while (waitpid(0, &status, WNOHANG) > 0) {
                }
                return(0);
            }

        default:
            ;
    }
    return(0);
}


int OutputEntry::sendOutput(FILE *fh, const char *prefix)
{
#if 0
    char *buf = new char[global_maxsize + 1];
    int pos = 0;

    pos = snprintf(buf, global_maxsize + 1, "[%s ]", prefix ? prefix : name);

    /* This is where we will go back and insert the continuation mark */
    int cont = pos - 2;
    int first = 1;
    
    char *pt;
    while ((pt = gets_alloc(fh)) != 0) {
        size_t remaining = global_maxsize - pos;
        if (strlen(pt) >= remaining) {
            buf[pos] = 0;
            buf[cont] = '*';
            syslog(priority, "%s", buf);
            
            pos = snprintf(buf, global_maxsize + 1, "[%s ]", prefix ? prefix : name);
            first = 1;
        }
        pos += snprintf(buf + pos, global_maxsize + 1 - pos, "%s%s", first ? "" : "\n", pt);
        first = 0;
    }

    if (first) {
        pos += snprintf(buf + pos, global_maxsize + 1 - pos, "No output");
    }
    buf[pos] = 0;
    buf[cont] = ' ';
    syslog(priority, "%s", buf);

    delete [] buf;
#else
    const char *pref = prefix ? prefix : name;
    const char *pt;
    int lines = 0;
    while ((pt = gets_alloc(fh)) != 0) {
        syslog(priority, "[%s] %s", pref, pt);
        if (++lines >= maxlines && maxlines) {
            gets_alloc_done();
            break;
        }
    }

    if (!lines) {
        syslog(priority, "[%s] No output", pref);
    }
#endif

    return(0);
}


void OutputEntry::callback(TimerQueue *timerqueue, void *cookie)
{
    OutputEntry *entry = (OutputEntry *)cookie;
    entry->action();
    entry->schedule(timerqueue);
}
