#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>
#include <assert.h>

#include "config.h"

/*
 * Allocate, free config.
 * Add, find, delete sections.
 * Add, find, delete parameters.
 */
/*#define DEBUG*/

int config_param_add_value(config_section_t * s, const char *name, const char *value)
{
	config_param_t *p;
	config_param_t *q = 0;
	
	/* Quick check for duplicates, also locate last entry */
	for (p=s->vars; p != NULL; p=p->next) {
		if (strcasecmp(p->name, name) == 0)  {
			return(-1);
		}
		else {
			q = p;
		}
	}

	p = malloc(sizeof(*p));
	assert(p);
	p->name = strdup(name);

	/* For the value, we'll be a bit clever and specially cache
	 * a zero length string.
	 */
	if (value == NULL) {
		p->value = NULL;
	}
	else if (*value == '\0') {
		p->value = "";
	}
	else {
		p->value = strdup(value);
	}

	/* Finally link the sucker in (at the end of the list for compatibility) */
	p->next = NULL;
	if (s->vars == NULL) {
		s->vars = p;
	}
	else {
		q->next = p;	
	}
	return(0);
}

void config_param_set_value(config_section_t * s, const char *name, const char *value)
{
	config_param_t *p = config_param_find(s->vars, name);

	if (p) {
		if (value) {
			config_param_set(p, value);
		}
		else {
			config_param_delete(s, p);
		}
	}
	else if (value) {
		config_param_add_value(s, name, value);
	}
}

void config_param_set(config_param_t *p, const char *value)
{
	if (p->value && *p->value) {
		free(p->value);
	}
	if (!value) {
		p->value = 0;
	}
	else if (!*value) {
		p->value = "";
	}
	else {
		p->value = strdup(value);
	}
}

/**
 * Frees up the memory associated with 'p'.
 * Does not "fix up" the linked list.
 */
static void param_free(config_param_t * p)
{
	/* Check for the name being a private name in which case we must free it */
	free(p->name);

	/* Free the value string if appropriate */
	if (p->value && *p->value) {
		free(p->value);
	}
	free(p);
}

/* Dispose an entire parameter list
 */
static void param_free_all(config_param_t * p)
{
	while (p) {
		config_param_t *q = p->next;
		param_free(p);
		p = q;
	}
}

/* Dispose a single parameter from a section
 */
void config_param_delete(config_section_t *s, config_param_t * p)
{
	if (p == s->vars) {
		s->vars = p->next;
		param_free(p);
	}
	else {
		config_param_t * q;
	
		/* Find the param in the list */
		for (q=s->vars; q; q=q->next) {
			if (q->next == p) {
				q->next = p->next;
				param_free(p);
				return;
			}
		}
		assert("ipseconf_param_delete() param not found in section" == 0);
	}
}

config_param_t *config_param_find(config_param_t *p, const char *name)
{
	for(; p; p=p->next) {
		if (strcasecmp(name, p->name) == 0) {
			return p;
		}
	}
	return(0);
}

config_section_t *config_section_add(config_t *conf, const char *type, const char *name)
{
	config_section_t * section;
	config_section_t * sp;
	const char *p;

#ifdef DEBUG
	printf("section_create(name=%s, type=%s)\n", name, type);
#endif

	if (!isalpha(name[0])) {
		return(0);
	}
	for (p=name+1; *p != '\0'; p++) {
		if (!isalnum(*p) && strchr("._-", *p) == NULL) {
			return(0);
		}
	}

	/* Create the new section record */
	section = malloc(sizeof(*section));
	assert(section);
	section->type = strdup(type);
	section->name = strdup(name);
	section->vars = NULL;
	section->next = NULL;
	if (conf->sections) {
		/* Add it at the end of the list */
		for (sp=conf->sections; sp->next != NULL; sp = sp->next) {
		}
		sp->next = section;
	}
	else {
		conf->sections = section;
	}

	return section;
}

/* Free the memory allocated for a section.
 * Does not "fix up" the linked list.
 */
static void section_free(config_section_t *s)
{
	assert(s);
	free(s->name);
	free(s->type);
	param_free_all(s->vars);
	free(s);
}

config_section_t *config_section_find(config_section_t *section, const char *type, const char *name)
{
	config_section_t *q;

	/* Find the section in the list */
	for (q=section; q; q=q->next) {
		if (strcmp(q->type, type) == 0 && (!name || strcasecmp(q->name, name) == 0)) {
			break;
		}
	}
	return(q);
}

int config_section_delete(config_t *conf, config_section_t *s)
{
	assert(s);

	if (s == conf->sections) {
		conf->sections = conf->sections->next;
		section_free(s);
		return(0);
	}
	else {
		config_section_t *q;
	
		/* Find the section in the list */
		for (q=conf->sections; q; q=q->next) {
			if (q->next == s) {
				q->next = s->next;
				section_free(s);
				return(0);
			}
		}
		assert("ipseconf_section_delete() section not found" == 0);
		return(1);
	}
}

config_t *config_alloc(void)
{
	config_t *conf = malloc(sizeof(*conf));
	memset(conf, 0, sizeof(*conf));

	return(conf);
}

/* Dispose of the currently loaded structures */
void config_free(config_t *conf)
{
	if (conf) {
		while (conf->sections) {
			config_section_t *p = conf->sections->next;
			section_free(conf->sections);
			conf->sections = p;
		}
		conf->sections = 0;

		if (conf->error_string) {
			free(conf->error_string);
			conf->error_string = 0;
		}

		free(conf);
	}
}
