#ifndef BEEPPROXY_CONFIG_H
#define BEEPPROXY_CONFIG_H

#include <stdio.h>

/* Provides functions to read and write "sectionalised" configuration files.
 */

/* Define a name/value pair which is used to cache parameter values */
typedef struct config_param_s {
	struct config_param_s	*next;		/* Ref to next parameter (allocated) */
	char			*name;		/* Name of parameter (allocated) */
	char			*value;		/* Value of parameter (allocated) */
} config_param_t;

/* This structure defines a section.  Basically, a section contains
 * a type, a name and some variables.
 */
typedef struct config_section_s {
	struct config_section_s	*next;		/* Ref to next section (allocated) */
	char			*type;		/* Type of section e.g. "conn", "config" (allocated) */
	config_param_t	*vars;		        /* Params associated with section (allocated) */
	char			*name;		/* Name of section (allocated) */
} config_section_t;

typedef struct config_s {
	config_section_t *sections;             /* List of sections in this config */
	const char *filename;                   /* Most recently parsed file */
	int line;                               /* Last line parsed in file */
	int error;                              /* 0 if parsed OK, or < 0 if error */
	char *error_string;                     /* Contains an error explanation if error != 0 */
} config_t;

#if defined(__cplusplus)
extern "C" {
#endif

/**
 * Loads configuration from the given filename into 'conf'
 * Returns 0 if OK, or non-zero if error.
 */
int config_read(const char *filename, config_t *conf);

/**
 * Writes the (presumably modified) configuration to the given file.
 * Returns 0 if OK, or -1 if unable to write.
 */
int config_write(const char *filename, config_t *conf);

/**
 * Like config_write(), but writes to a filehandle.
 */
void config_write_to_fh(FILE *fh, config_t *conf);

/**
 * Allocates an empty config structure.
 * Use config_free() to free.
 */
config_t *config_alloc(void);

/**
 * Frees previously loaded configuration.
 */
void config_free(config_t *conf);

/**
 * Delete the given section.
 * Returns 0 if OK, or 1 if not found.
 */
int config_section_delete(config_t *conf, config_section_t *section);

/**
 * Creates a new section of the given name and type
 * and adds it into the list.
 * 
 * Returns the newly added section.
 */
config_section_t *config_section_add(config_t *conf, const char *type, const char *name);

/**
 * Searches for and returns the section with the given name and type, starting
 * with (including) the given section. Use section->next to skip the "current" section.
 *
 * 'name' may be 0 to accept any section of the given type.
 *
 * Returns 0 if not found.
 */
config_section_t *config_section_find(config_section_t *section, const char *type, const char *name);

/**
 * Delete the given parameter from the given section.
 */
void config_param_delete(config_section_t *section, config_param_t *param);

/**
 * Add the name=value parameter to the given section.
 * Returns 0 if OK, or -1 if the parameter already exists.
 */
int config_param_add_value(config_section_t *section, const char *name, const char *value);

/**
 * Set the name=value parameter in the given section, replacing any existing entry if it exists.
 * If 'value' is 0, deletes the parameter.
 */
void config_param_set_value(config_section_t *section, const char *name, const char *value);

/**
 * Set the value of the given parameter.
 */
void config_param_set(config_param_t *param, const char *value);

/**
 * Returns the parameter with given name, starting the search from 'p'.
 * Use section->vars to search an entire section.
 *
 * Returns 0 if not found.
 */
config_param_t *config_param_find(config_param_t *p, const char *name);

#if defined(__cplusplus)
}
#endif

#endif
