/**
 * @file IxOsCacheMMU.c
 *
 * @brief The file contains implementation for the OS memory alocation
 * functions
 *
 * @version $Revision: 1.1.1.1 $
 * 
 * @par
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * @par
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * @par
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * @par
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * @par
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * @par
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * @par
 * -- End Intel Copyright Notice --
 */

#ifdef __vxworks

#include <vxWorks.h>
#include <sysLib.h>
#include "ixp425.h"

#elif defined(__linux)

#include <linux/kernel.h>
#include <linux/slab.h>
#include "ixp425.h"

#else /* POSIX */

#include <stdlib.h>

#endif

#include <stdlib.h>
#include "IxOsCacheMMU.h"

#ifdef __vxworks

#ifdef IX_ACC_CACHE_ENABLED

/* 
 * Allocate on a cache line boundary (null pointers are
 * not affected by this operation). This operation is NOT cache safe.
 */
void *ixOsServCacheDmaAlloc(UINT32 size)
{
    return memalign(IX_XSCALE_CACHE_LINE_SIZE,
		    size + IX_XSCALE_CACHE_LINE_SIZE);
}

void ixOsServCacheDmaFree(void *ptr, UINT32 size)
{
    free(ptr);
}

#else /* IX_ACC_CACHE_ENABLED */

/* 
 * Allocate on a cache line boundary (null pointers are
 * not affected by this operation). This operation is NOT cache safe.
 */
void *ixOsServCacheDmaAlloc(UINT32 size)
{
    void *address;
    void **returnAddress;

    /* get an area of memory */
    address = cacheDmaMalloc(IX_XSCALE_CACHE_LINE_SIZE + 
			    sizeof(void *) +
			    size +
			    IX_XSCALE_CACHE_LINE_SIZE);
    
    if (address == NULL) return (void *)NULL;

    /* cacheDmaMalloc does not always return an aligned part of memory */
    returnAddress = (void **)ROUND_UP(((UINT32)address), 
				      IX_XSCALE_CACHE_LINE_SIZE);
    
    /* aligned to the next aligned part of memory */
    returnAddress += IX_XSCALE_CACHE_LINE_SIZE / sizeof(UINT32);

    /* store the pointer in the word before the memory 
    * area returned to the user
    */
    returnAddress[- 1] = address;
    return (void *)returnAddress;
}

void ixOsServCacheDmaFree(void *ptr, UINT32 size)
{
    free(((void **)ptr)[-1]);
}

#endif /* IX_ACC_CACHE_ENABLED */

#elif defined(__linux)

/* 
 * Allocate on a cache line boundary (null pointers are
 * not affected by this operation). This operation is NOT cache safe.
 */
void *ixOsServCacheDmaAlloc(UINT32 size)
{
    struct page *page;
    unsigned long order;

    size = PAGE_ALIGN(size);
    order = get_order(size);
    page = alloc_pages(GFP_KERNEL, order);
    if (!page)
    {
	printk("%s: Failed to allocate pages\n", __FUNCTION__);
	return NULL;
    }
    return page_address(page);
}

void ixOsServCacheDmaFree(void *ptr, UINT32 size)
{
    unsigned long order;

    size = PAGE_ALIGN(size);
    order = get_order(size);
    free_pages((unsigned int)ptr, order);
}

#else /* POSIX */
/* 
 * Allocate on a cache line boundary (null pointers are
 * not affected by this operation). This operation is NOT cache safe.
 */
void *ixOsServCacheDmaAlloc(UINT32 size)
{
    return memalign(IX_XSCALE_CACHE_LINE_SIZE,
		    size + IX_XSCALE_CACHE_LINE_SIZE);
}

void ixOsServCacheDmaFree(void *ptr, UINT32 size)
{
    free(ptr);
}

#endif

/**
 * @}
 */
